/*
 * Amanda, The Advanced Maryland Automatic Network Disk Archiver
 * Copyright (c) 1991, 1994 University of Maryland
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of U.M. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  U.M. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * U.M. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL U.M.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: James da Silva, Systems Design and Analysis Group
 *			   Computer Science Department
 *			   University of Maryland at College Park
 */
/*
 * chg-generic.sh - generic tape changer script
 */
#include "config.h"
#include "options.h"

pname="chg-generic"
ourconf="chg-generic.conf"
ourstate="chg-generic.state"

PATH=LIBEXEC_DIR
PATH=$PATH:/usr/bin:/usr/ucb:/usr/sbin:/bin:/sbin
export PATH

/* read in some config parameters */

if [ \! -f $ourconf ]; then
	echo "<none> $pname: $ourconf does not exist"
	exit 2
fi

firstslot=`awk '$1 == "firstslot" {print $2}' $ourconf 2>/dev/null`
if [ "$firstslot" = "" ]; then
	echo "<none> $pname: firstslot not specified in $ourconf"
	exit 2
fi

lastslot=`awk '$1 == "lastslot" {print $2}' $ourconf 2>/dev/null`
if [ "$lastslot" = "" ]; then
	echo "<none> $pname: lastslot not specified in $ourconf"
	exit 2
fi

nslots=`expr $lastslot - $firstslot + 1`

gravity=`awk '$1 == "gravity" {print $2}' $ourconf 2>/dev/null`
if [ "$gravity" = "" ]; then
	echo "<none> $pname: gravity not specified in $ourconf"
	exit 2
fi

needeject=`awk '$1 == "needeject" {print $2}' $ourconf 2>/dev/null`
if [ "$needeject" = "" ]; then
	echo "<none> $pname: needeject not specified in $ourconf"
	exit 2
fi


/* read in state: only curslot and curloaded at the present time */

curslot=`awk '$1 == "curslot" {print $2}' $ourstate 2>/dev/null`
if [ "$curslot" = "" ]; then
	curslot=$firstslot
fi

curloaded=`awk '$1 == "curloaded" {print $2}' $ourstate 2>/dev/null`
if [ "$curloaded" = "" ]; then
	curloaded=0
fi


/* process the command-line */

/* control vars to avoid code duplication: not all shells have functions! */
usage=0
checkgravity=0
ejectslot=0
loadslot=0
slotempty=0

if [ $# -ge 1 ]; then command=$1; else command="-usage"; fi

case "$command" in

-info) # return basic information about changer

	backwards=`expr 1 - $gravity`
	echo $curslot $nslots $backwards
	exit 0
	;;

-reset) # reset changer

	checkgravity=0
	loadslot=1
	newslot=$firstslot

	/* XXX put changer-specific reset here, if applicable */
	;;

-eject) # eject tape if loaded

	checkgravity=0
	loadslot=0
	newslot=$curslot
	ejectslot=1

	if [ $curloaded -eq 0 ]; then
		echo $curslot "slot already empty"
		exit 1
	fi
	;;

-slot)	# change to slot

	checkgravity=1
	loadslot=1

	slotparm=$2
	case "$slotparm" in
	[0-9]*)	
		newslot=$slotparm
		if [ \( $newslot -gt $lastslot \) -o \
		     \( $newslot -lt $firstslot \) ]; then
			echo $newslot "no slot $newslot: legal range is" \
				"$firstslot ... $lastslot"
			exit 1
		fi;;
	current)
		newslot=$curslot;;
	first)
		newslot=$firstslot;;
	last)
		newslot=$lastslot;;
	next)
		newslot=`expr $curslot + 1`
		if [ $newslot -gt $lastslot ]; then
			newslot=$firstslot
		fi;;
	prev)
		newslot=`expr $curslot - 1`
		if [ $newslot -lt $firstslot ]; then
			newslot=$lastslot
		fi;;
	*)
		echo "<none> bad slot name \"$slotparm\""
		exit 1
		;;
	esac
	;;
*)
	usage=1
	;;
esac


if [ $usage -eq 1 ]; then
	echo "<none> usage: $pname {-reset | -slot [<slot-number>|current|next|prev]}"
	exit 2
fi


/* check for legal move */

if [ \( $checkgravity -eq 1 \) -a \( $gravity -ne 0 \) ]; then
	if [ \( $newslot -lt $curslot \) -o \( "$slotparm" = "prev" \) ]
	then
		echo "$newslot cannot go backwards in gravity stacker"
		exit 1
	fi
fi

/* get tape device name */

device=`awk '$1 == "slot" && $2 == '$newslot' {print $3}' $ourconf 2>/dev/null`
if [ "$device" = "" ]; then
	echo "$newslot $pname: slot $newslot device not specified in $ourconf"
	exit 2
fi

/* check if load needs an eject first */

if [ \( $needeject -eq 1 \) -a \( $loadslot -eq 1 \) -a \
     \( $curloaded -eq 1 \) -a \( $newslot -ne $curslot \) ]; then
	ejectslot=1
fi


if [ $ejectslot -eq 1 ]; then	/* eject the tape from the drive */

	/* XXX put changer-specific load command here, if applicable */

	curloaded=0		/* unless something goes wrong */
	slotempty=0

#ifndef TAPEIO_TEST
	/* generically, first check that the device is there */

	if [ ! -c $device ]; then
		echo "$newslot $device: not a device file"
		exit 2
	fi

	/* second, try to unload the device */
	mt -f $device offline >/dev/null 2>&1
	if [ $? -ne 0 ]; then
		/* 
		 * XXX if the changer-specific eject command can distinguish
		 * betweeen "slot empty" and more serious errors, return 1
		 * for the first case, 2 for the second case.  Generically,
		 * we just presume an error signifies an empty slot.
		 */
		slotempty=1
	fi
#else
	/* ignore this, it's some debug scaffolding that I use */
	if [ ! -f $device/file000 ]; then
		slotempty=1
	fi
#endif

fi

if [ $loadslot -eq 1 ]; then	/* load the tape from the slot */

	/* XXX put changer-specific load command here, if applicable */

	curloaded=1		/* unless something goes wrong */
	slotempty=0

#ifndef TAPEIO_TEST
	/* generically, first check that the device is there */

	if [ ! -c $device ]; then
		echo "$newslot $device: not a device file"
		exit 2
	fi

	/* second, try to rewind the device */
	mt -f $device rewind >/dev/null 2>&1
	if [ $? -ne 0 ]; then
		/* 
		 * XXX if the changer-specific load command can distinguish
		 * betweeen "slot empty" and more serious errors, return 1
		 * for the first case, 2 for the second case.  Generically,
		 * we just presume an error signifies an empty slot.
		 */
		slotempty=1
		curloaded=0
	fi
#else
	/* ignore this, it's some debug scaffolding that I use */
	if [ ! -f $device/file000 ]; then
		slotempty=1
		curloaded=0
	fi
#endif
fi

/* update state */

echo "# generic-changer state cache: DO NOT EDIT!" >  $ourstate
echo curslot $newslot 				   >> $ourstate
echo curloaded $curloaded			   >> $ourstate

/* return slot info */

if [ $slotempty -eq 1 ]; then
	echo "$newslot slot is empty"
	exit 1
fi

echo $newslot $device

exit 0
