/* char.c -- low level character i/o for monitor
 * copyright (c) 1984  American Information Systems Corporation
 *  Dock Williams
 *  November, 1983
 *
 * low level routines for character i/o
 * this level of i/o is replaced by stdio for debugging
 * all other files contain ifdeffing to turn off or on the use of stdio
 *
 */

#include "monitor.h"
#include "ascii.h"

#ifdef DB32

#include "3200config.h"


    char
inchar()

/* inchar()
 *  out:    character read from console
 *
 *  inchar, get a character from the console via uart0
 *  asumes uart is programmed and ready to go
 *  does polled input io, read status of chip until a character is received
 *  then read in the received chracter check for special characters 
 *  and return it
 */
{
	register char c;

#ifdef REV_2A /****************** CPU REVISION-2A **********************/
	/* wait until the RXRDY bit is set */
	while ( !(*((char *)(UART0 + U_STAT)) & U_RXRDY) ) ;

	/* read in the received character */
	switch( c = *((char *)(UART0 + U_IDATA)) ) {
#endif /************************* CPU REVISION-2A **********************/
	    case XON:
		cflags &= ~CF_OUTHLD;
		break;
	    case XOFF:
		cflags |= CF_OUTHLD;
		break;
	    case ESCAPE:
		cflags |= CF_ESCHIT;
		cflags &= ~CF_OUTHLD;
		break;
	}
	return(c);
}



    char
getchar()

/* getchar()
 *  out:    return a charcter after some checking
 *
 *  same as inchar but it does not return special characters 
 */
{
    register char c;

    while(1) {
	c = inchar();
	switch( c ) {
	    case XON:
	    case XOFF:
		continue;
	    case ESCAPE:
	    default:
		return(c);
	}
    }
}


quitchk()
/* quitchk()
 *
 *  check for special conditions like receiving control characters
 *  if in output hold, get characters until we get XON
 *  if we get escape, quit everything
 */
{

#ifdef REV_2A /****************** CPU REVISION-2A **********************/
	if ( *((char *)(UART0 + U_STAT)) & U_RXRDY ) 
#endif /************************* CPU REVISION-2A **********************/

		inchar();

	while(cflags & CF_OUTHLD) {
	    inchar();
	    if (cflags & CF_ESCHIT) return(1);
	}
	if (cflags & CF_ESCHIT) return(1);
	else return(0);
}



putchar(c)
    char c;
/* putchar(c)
 *  in: c       character to print on the console
 *
 *  putchar, put a character out to the console via uart 0
 *  assumes uart is programmed and ready to go
 *  does polled output io, read status until the chip is
 *  able to take a charcter, then ship it out
 *  check for input characters before outputting
 *  if there is input, read it in and check for special characters
 */
{
	/* if a character has been received */
	/* read it in and check the received character */

#ifdef REV_2A /****************** CPU REVISION-2A **********************/
	if(*((char *)(UART0 + U_STAT)) & U_RXRDY) inchar();
#endif /************************* CPU REVISION-2A **********************/

	if(cflags & CF_OUTHLD) {
	    if (quitchk()) return;
	}

	/* otherwise just ship out the received character */
	/* if newline echo cr lf */
	if (c == '\n') putchar('\r');

	/* for tabs echo a single character */
	if (c == '\t') c = ' ';

#ifdef REV_2A /****************** CPU REVISION-2A **********************/
	/* wait until the TXRDY bit is set */
	while( !( *((char *)(UART0 + U_STAT)) & U_TXRDY) ) ;

	/* now ouput the argument character */
	*((char *)(UART0 + U_ODATA)) = c;
#endif /************************* CPU REVISION-2A **********************/
}



    char
inc_aux()

/* inc_aux()
 *  out:    character read from auxilliary port
 *
 *  inc_aux, get a character from the console via uart1
 *  asumes uart is programmed and ready to go
 *  used exclusively by the loader
 */
{
#ifdef REV_2A /****************** CPU REVISION-2A **********************/
	/* wait until the RXRDY bit is set */
	while ( !(*((char *)(UART1 + U_STAT)) & U_RXRDY) )
		if (quitchk()) return('\0');
			
	/* read and return the received character */
	return ( (*((char *)(UART1 + U_IDATA))) & 0x00ff );
#endif /************************* CPU REVISION-2A **********************/
}

outquick(c)
/* outquick(c)
 *  in:	    character to ship out auxilliary port
 *  out:    character at auxilliary port
 *
 *  asumes uart is programmed and ready to go
 *  used exclusively by the loader to echo status information
 *  does not poll the txready status bit of the uart
 *  but simply outputs the input character
 *  this was done so that incoming characters coming in on 
 *  another port could be read in a polled environment without loss
 *  because there is the possibiliy of waiting too long to output 
 *  a character in this routine
 */
char c;
{
#ifdef REV_2A /****************** CPU REVISION-2A **********************/
	*((char *)(UART0 + U_ODATA)) = c;
#endif /************************* CPU REVISION-2A **********************/
}



#else

#include <stdio.h>

inputchar()
{
	int c, i;
	char chr;

	while( (i = read(0, &chr, 1)) != -1) {
		c = chr;
		if (i == 0 || c == '\032') quitrestore();
		return(c);
	}
}


outputchar(c)
    char c;
{
	if(c=='\n'){
		putchar('\r');
		fflush(stdout);
	}
	putchar(c);
	fflush(stdout);
}

quitchk()
/*
 * check for special conditions like receiving control characters
 * if in output hold, get characters until we get XON
 * if we get escape, quit everything
 */
{
	while(cflags & CF_OUTHLD) inputchar();
	if (cflags & CF_ESCHIT) return(1);
	return(0);
}
