/*******************************************************************************
* localinput.cpp: Local streams
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: localinput.cpp,v 1.11 2002/10/01 08:09:11 jpsaman Exp $
*
* Authors: Arnaud de Bossoreille de Ribou <bozo@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "../../core/defs.h"

#include "../../core/core.h"

#include "../../mpeg/mpeg.h"
#include "../../mpeg/ts.h"
#include "../../mpeg/rtp.h"
#include "../../server/program.h"
#include "../../server/buffer.h"
#include "../../server/output.h"
#include "../../server/channel.h"
#include "../../server/broadcast.h"
#include "../../server/request.h"
#include "../../server/input.h"
#include "../../server/tsstreamer.h"

#include "../../mpeg/reader.h"
#include "../../mpeg/converter.h"

#include "localinput.h"


//------------------------------------------------------------------------------
// Library declaration
//------------------------------------------------------------------------------
#ifdef __PLUGIN__
GENERATE_LIB_ARGS(C_LocalInputModule, handle);
#endif


//------------------------------------------------------------------------------
// Builtin declaration
//------------------------------------------------------------------------------
#ifdef __BUILTIN__
C_Module* NewBuiltin_localinput(handle hLog)
{
  return new C_LocalInputModule(hLog);
}
#endif


/*******************************************************************************
* C_LocalInput class
********************************************************************************
*
*******************************************************************************/

//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_LocalInput::C_LocalInput(C_Module* pModule,
                           const C_String& strName) : C_Input(pModule,
                                                              strName)
{
}


//------------------------------------------------------------------------------
// Destructor
//------------------------------------------------------------------------------
C_LocalInput::~C_LocalInput()
{
}


//------------------------------------------------------------------------------
// Initialization
//------------------------------------------------------------------------------
void C_LocalInput::OnInit()
{
  C_Application* pApp = C_Application::GetApp();
  ASSERT(pApp);

  // Get the path to the config file
  C_String strSettingName = GetName() + ".ConfigPath";
  m_strConfigPath = pApp->GetSetting(strSettingName, "");

  // Load settings
  m_cSettings.Load(m_strConfigPath + "/input.cfg", false);

  m_strFilesPath = m_cSettings.GetSetting("Input.FilesPath", "");
  if(    m_strFilesPath.Length()
      && m_strFilesPath[m_strFilesPath.Length() -1] != '/')
    m_strFilesPath += "/";

  // Build the program list
  C_String strPgrmCount = m_cSettings.GetSetting("Input.ProgramCount", "0");
  unsigned int uiPgrmCount = strPgrmCount.ToInt();

  if(uiPgrmCount)
  {
    for(unsigned int ui = 1; ui <= uiPgrmCount; ui++)
    {
      C_String strName = m_cSettings.GetSetting(C_String(ui) + ".Name",
                                                C_String("Pgrm") + ui);
      C_String* pStr = new C_String(strName);
      m_vProgramNames.Add(pStr);
    }
  }
  else
  {
    Log(m_hLog, LOG_WARN,
        "The \"ProgramCount\" variable isn't specified in the file \"" +
        m_strConfigPath + "/input.cfg\"");
  }
}


//------------------------------------------------------------------------------
// Destruction
//------------------------------------------------------------------------------
void C_LocalInput::OnDestroy()
{
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_List<C_Program> C_LocalInput::OnGetAvailablePgrms()
{
  C_List<C_Program> cPgrmList;

  for(unsigned int ui = 0; ui < m_vProgramNames.Size(); ui++)
  {
    C_Program* pProgram = new C_Program(/*ui + 1,*/ m_vProgramNames[ui]);
    ASSERT(pProgram);
    cPgrmList.PushEnd(pProgram);
  }

  return cPgrmList;
}


//------------------------------------------------------------------------------
// Start the reception of the given program
//------------------------------------------------------------------------------
void C_LocalInput::OnStartStreaming(C_Broadcast* pBroadcast)
{
  ASSERT(pBroadcast);

  // We choose a TS packet buffer able to store up to 3s of stream at 8 Mbits/s
  C_NetList* pTsProvider = new C_NetList(3*3*797);

  // The netlist must be at least of the size of the Reader buffer +
  // the size of the output buffer + 2 to be sure that there are always free
  // packets in it
  const C_Channel* pChannel = pBroadcast->GetChannel();
  ASSERT(pTsProvider->Capacity() - pChannel->GetBuffCapacity() - 2 > 0);
  unsigned int uiSize = pTsProvider->Capacity() -pChannel->GetBuffCapacity()-2;
  C_SyncFifo* pBuffer;

  // Get the type of the program
  unsigned int uiId =
        m_vProgramNames.Find(pBroadcast->GetProgram()->GetName()) + 1;
  ASSERT(uiId > 0);
  C_String strType = m_cSettings.GetSetting(C_String(uiId) + ".Type",
                                            "Mpeg2-TS").ToLower();

  C_String strReaderType;
  C_String strConverterType;

  // Specific behaviour depends on the type
  if(strType == "mpeg1-ps")             // Mpeg 1 - Program Stream
  {
    // Update the size of the buffer and create it
    uiSize -= 2;
    pBuffer = new C_SyncFifo(uiSize);
    // Reader configuration
    strReaderType = "file";
    C_String strFile = m_strFilesPath +
                       m_cSettings.GetSetting(C_String(uiId)+".FileName", "");
    pBroadcast->SetOption("filename", strFile);

    // Converter configuration
    strConverterType = "ps2ts";
    pBroadcast->SetOption("mpegversion", "1");
    pBroadcast->SetOption("preparse", "1");
  }
  else if(strType == "mpeg2-ps")        // Mpeg 2 - Program Stream
  {
    // Update the size of the buffer and create it
    uiSize -= 2;
    pBuffer = new C_SyncFifo(uiSize);
    // Reader configuration
    strReaderType = "file";
    C_String strFile = m_strFilesPath +
                       m_cSettings.GetSetting(C_String(uiId)+".FileName", "");
    pBroadcast->SetOption("filename", strFile);

    // Converter configuration
    strConverterType = "ps2ts";
    pBroadcast->SetOption("mpegversion", "2");
    pBroadcast->SetOption("preparse", "1");
  }
  else if(strType == "mpeg2-ts")        // Mpeg 2 - Transport Stream
  {
    // Update the size of the buffer and create it
    pBuffer = new C_SyncFifo(uiSize);
    // Reader configuration
    strReaderType = "file";
    C_String strFile = m_strFilesPath +
                       m_cSettings.GetSetting(C_String(uiId)+".FileName", "");
    pBroadcast->SetOption("filename", strFile);

    // Converter configuration
    strConverterType = "ts2ts";
  }
  else if(strType == "dvd")             // DVD device (Mpeg2 PS)
  {
    // Update the size of the buffer and create it
    uiSize -= 2;
    pBuffer = new C_SyncFifo(uiSize);
    // Reader configuration
    strReaderType = "dvd";
    C_String strDevice = m_cSettings.GetSetting(C_String(uiId)+".Device", "");
    pBroadcast->SetOption("device", strDevice);

    // Converter configuration
    strConverterType = "ps2ts";
    pBroadcast->SetOption("mpegversion", "2");
    pBroadcast->SetOption("preparse", "0");
  }
  else
  {
    delete pTsProvider;
    throw E_Exception(GEN_ERR, "Unsupported or unknown type : " + strType);
  }

  C_MpegReader* pReader;

  // Create the converter
  C_MpegReaderModule* pReaderModule = (C_MpegReaderModule*)
                                C_Application::GetModuleManager()
                                        ->GetModule("mpegreader",
                                                    strReaderType);

  if(pReaderModule)
  {
    pReader = pReaderModule->NewMpegReader(pBroadcast);
    ASSERT(pReader);
  }
  else
  {
    throw E_Exception(GEN_ERR,
                      "Module mpegreader:" + strConverterType +
                      " not present");
  }

  C_MpegConverter* pConverter;

  // Create the converter
  C_MpegConverterModule* pConverterModule = (C_MpegConverterModule*)
                                C_Application::GetModuleManager()
                                        ->GetModule("mpegconverter",
                                                    strConverterType);

  if(pConverterModule)
  {
    C_MpegConverterConfig cConfig;
    cConfig.m_hLog = m_hLog;
    cConfig.m_pBroadcast = pBroadcast;
    cConfig.m_pReader = pReader;
    cConfig.m_pTsProvider = pTsProvider;
    cConfig.m_pHandler = pBuffer;
    cConfig.m_iInitFill = pBuffer->Capacity();
    cConfig.m_pEventHandler = m_pEventHandler;
    pConverter = pConverterModule->NewMpegConverter(cConfig);
    ASSERT(pConverter);
  }
  else
  {
    throw E_Exception(GEN_ERR,
                      "Module mpegconverter:" + strConverterType +
                      " not present");
  }

  pReader->SetConverter(pConverter);

  // Create the streamer
  C_TsStreamer* pStreamer = new C_TsStreamer(m_hLog, pBroadcast,
                                             pTsProvider, pBuffer,
                                             m_pEventHandler, true, true);
  ASSERT(pStreamer);

  m_cConverters.Add(pBroadcast, pConverter);
  m_cStreamers.Add(pBroadcast, pStreamer);

  try
  {
    pConverter->Create();
    pStreamer->Create();
  }
  catch(E_Exception e)
  {
    pConverter->Stop();
    pStreamer->Stop();

    // Unregister the 2 thread and delete them
    m_cConverters.Delete(pBroadcast);
    m_cStreamers.Delete(pBroadcast);

    throw E_Exception(GEN_ERR, "unable to start streaming of program "+
                      pBroadcast->GetProgram()->GetName(), e);
  }
}


//------------------------------------------------------------------------------
// Resume the reception of the given program
//------------------------------------------------------------------------------
void C_LocalInput::OnResumeStreaming(C_Broadcast* pBroadcast)
{
  ASSERT(pBroadcast);

  // Find the converter that receive the pgrm
  C_MpegConverter* pConverter = m_cConverters.Get(pBroadcast);
  ASSERT(pConverter);

  // Resume the thread
  try
  {
    pConverter->Resume();
  }
  catch(E_Exception e)
  {
    throw E_Exception(GEN_ERR, "Unable to resume streaming of program "+
                      pBroadcast->GetProgram()->GetName(), e);
  }
}


//------------------------------------------------------------------------------
// Suspend the reception of the given program
//------------------------------------------------------------------------------
void C_LocalInput::OnSuspendStreaming(C_Broadcast* pBroadcast)
{
  ASSERT(pBroadcast);

  // Find the converter that receive the pgrm
  C_MpegConverter* pConverter = m_cConverters.Get(pBroadcast);
  ASSERT(pConverter);

  C_String strPgrmName = pBroadcast->GetProgram()->GetName();

  // Suspend the thread
  try
  {
    pConverter->Suspend();
  }
  catch(E_Exception e)
  {
    throw E_Exception(GEN_ERR, "Unable to resume streaming of program "+
                      pBroadcast->GetProgram()->GetName(), e);
  }
}


//------------------------------------------------------------------------------
// Stop the reception of the given program
//------------------------------------------------------------------------------
void C_LocalInput::OnStopStreaming(C_Broadcast* pBroadcast)
{
  ASSERT(pBroadcast);

  // Find the reader and the streamer that receive the pgrm
  C_MpegConverter* pConverter = m_cConverters.Remove(pBroadcast);
  ASSERT(pConverter);
  C_TsStreamer* pStreamer = m_cStreamers.Remove(pBroadcast);
  ASSERT(pStreamer);

  // Stop the threads
  try
  {
    pConverter->Stop();
    delete pConverter;
    pStreamer->Stop();
    delete pStreamer;
  }
  catch(E_Exception e)
  {
    throw E_Exception(GEN_ERR, "Unable to stop streaming of program "+
                      pBroadcast->GetProgram()->GetName(), e);
  }
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_LocalInput::OnUpdateProgram(C_String strProgram,
                                     C_String strFileName, C_String strType)
{
  C_String strPgrmCount = m_cSettings.GetSetting("Input.ProgramCount", "0");
  unsigned int uiPgrmCount = strPgrmCount.ToInt();

  unsigned int uiId = m_vProgramNames.Find(strProgram) + 1;
  if (uiId>0)
  {
    Log(m_hLog, LOG_WARN, "The program already exists in the list for this input");
  }

  if(uiPgrmCount)
  {
      uiPgrmCount++;
      C_String strKey = C_String(uiPgrmCount)+ ".Name";
      m_cSettings.Update(strKey.ToLower(),strProgram);
      strKey = C_String(uiPgrmCount)+ ".FileName";
      m_cSettings.Update(strKey.ToLower(), strFileName);
      strKey = C_String(uiPgrmCount)+ ".Type";
      m_cSettings.Update(strKey.ToLower(), strType);
      strKey = C_String("Input.ProgramCount");
      m_cSettings.Update(strKey.ToLower(), C_String(uiPgrmCount));

      // add to m_vProgramNames
      C_String strName = m_cSettings.GetSetting(C_String(uiPgrmCount) + ".Name",
                                                C_String("Pgrm") + uiPgrmCount);
      C_String* pStr = new C_String(strName);
      m_vProgramNames.Add(pStr);
  }
  else
  {
    throw E_Exception(GEN_ERR,
        "The \"ProgramCount\" variable isn't specified in the file \"" +
        m_strConfigPath + "/input.cfg\"");
  }
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_LocalInput::OnDeleteProgram(C_String strProgram)
{
  // Get the type of the program
  unsigned int uiId = m_vProgramNames.Find(strProgram) + 1;
  if (uiId>0)
  {
    // Delete settings from m_cSettings
    C_String strPgrmCount = m_cSettings.GetSetting("Input.ProgramCount", "0");
    unsigned int uiPgrmCount = strPgrmCount.ToInt();

    if(uiPgrmCount)
    {
      C_String strDelKey = C_String(uiId)+ ".Name";
      m_cSettings.Delete(strDelKey.ToLower());
      if (m_cSettings.GetSetting(C_String(uiId)+".FileName", "") != "" )
      {
        strDelKey = C_String(uiId)+ ".FileName";
        m_cSettings.Delete(strDelKey.ToLower());
      }
      if (m_cSettings.GetSetting(C_String(uiId)+".Device", "") != "" )
      {
        strDelKey = C_String(uiId)+ ".Device";
        m_cSettings.Delete(strDelKey.ToLower());
      }
      strDelKey = C_String(uiId)+ ".Type";
      m_cSettings.Delete(strDelKey.ToLower());

      // Move all other settings one item down
      for(unsigned int ui = uiId; ui <= uiPgrmCount; ui++)
      {
        C_String strProgram = m_cSettings.GetSetting(C_String(ui+1) + ".Name",C_String("Pgrm") + ui+1);
        C_String strFileName = m_cSettings.GetSetting(C_String(ui+1)+".FileName", "");
        C_String strType = m_cSettings.GetSetting(C_String(ui+1) + ".Type", "Mpeg2-TS");
        C_String strDevice = m_cSettings.GetSetting(C_String(ui+1)+".Device", "");

        // update settings
        C_String strKey = C_String(ui)+ ".Name";
        m_cSettings.Update(strKey.ToLower(),strProgram);
        strKey = C_String(ui)+ ".FileName";
        if (strFileName != "")
        {
          m_cSettings.Update(strKey.ToLower(), strFileName);
        }
        else
          m_cSettings.Update(strKey.ToLower(), "");
        strKey = C_String(ui)+ ".Device";
        if (strDevice != "")
        {
          m_cSettings.Update(strKey.ToLower(), strDevice);
        }
        else
          m_cSettings.Update(strKey.ToLower(), "");
        strKey = C_String(ui)+ ".Type";
        m_cSettings.Update(strKey.ToLower(), strType);
      }
      // Lower program count
      uiPgrmCount--;
      strDelKey = C_String("Input.ProgramCount");
      m_cSettings.Update(strDelKey.ToLower(), C_String(uiPgrmCount));
      m_vProgramNames.Delete(uiId);

      // ** DEBUG **
      Log(m_hLog, LOG_NOTE, "Debugging inputs m_cSettings list.");
      C_String strPgrmCount = m_cSettings.GetSetting("Input.ProgramCount", "0");
      unsigned int uiPgrmCount = strPgrmCount.ToInt();
      for(unsigned int ui = 1; ui <= uiPgrmCount; ui++)
      {
        C_String strProgram = m_cSettings.GetSetting(C_String(ui) + ".Name",C_String("Pgrm") + ui);
        C_String strFileName = m_cSettings.GetSetting(C_String(ui)+".FileName", "");
        C_String strDevice = m_cSettings.GetSetting(C_String(ui)+".Device", "");
        C_String strType = m_cSettings.GetSetting(C_String(ui) + ".Type", "Mpeg2-TS").ToLower();
        Log(m_hLog, LOG_NOTE, "Program: \"" + strProgram +"\"");
        Log(m_hLog, LOG_NOTE, "  FileName: \"" + strFileName+"\"");
        Log(m_hLog, LOG_NOTE, "  Device: \"" + strDevice+"\"");
        Log(m_hLog, LOG_NOTE, "  Type: \"" + strType+"\"");
      }
    }
    else
    {
      throw E_Exception(GEN_ERR,"Deleting program information from input failed");
    }
  }
  else
  {
    throw E_Exception(GEN_ERR, "The program is not known by this input.");
  }
}
