/*******************************************************************************
* output.cpp: output streams
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: output.cpp,v 1.1 2001/10/06 21:23:37 bozo Exp $
*
* Authors: Benoit Steiner <benny@via.ecp.fr>
*          Arnaud de Bossoreille de Ribou <bozo@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "../core/defs.h"

#include "config.h"
#include "../core/core.h"
#include "../mpeg/mpeg.h"
#include "../mpeg/ts.h"
#include "buffer.h"
#include "output.h"



//******************************************************************************
//
//******************************************************************************
//
//******************************************************************************
E_Output::E_Output(const C_String& strMsg, const E_Exception& e) :
                                                 E_Exception(GEN_ERR, strMsg, e)
{
}



//******************************************************************************
//
//******************************************************************************
//
//******************************************************************************

//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
C_Output::C_Output(unsigned int iBuffSize) : m_cTsBuff(iBuffSize)
{
  // Byte loss couter initialisation
  m_iByteLost = 0;

  ZERO(m_pTsProvider);
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
C_Output::~C_Output()
{
  ZERO(m_pTsProvider);

  ASSERT(m_cTsBuff.Size() == 0);
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_Output::Init(C_NetList* pTsProvider)
{
  ASSERT(pTsProvider);
  m_pTsProvider = pTsProvider;

  ASSERT(m_cTsBuff.Size() == 0);
  OnInit();
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_Output::Close()
{
  Flush();
  OnClose();
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// Bla Bla sur buffer
//------------------------------------------------------------------------------
void C_Output::Send(C_TsPacket* pPacket)
{
  ASSERT(pPacket);

  // Append the packet to the buffer for later write to the output
  int iRc = m_cTsBuff.Push(pPacket);
  ASSERT(!iRc);

  if(m_cTsBuff.Size() == m_cTsBuff.Capacity())
  {
    // Send the buffered packets to the device and give them back to the
    // netlist
    WriteToPort();
  }
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_Output::Flush()
{
  WriteToPort();
}



//******************************************************************************
//
//******************************************************************************
//
//******************************************************************************

//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
C_NetOutput::C_NetOutput(const C_String& strInterface,
                         const C_String& strSrcHost, const C_String& strSrcPort,
                         const C_String& strDstHost, const C_String& strDstPort)
                : C_Output(TS_IN_ETHER), m_cSocketBuff(TS_IN_ETHER),
                  m_strInterface(strInterface),
                  m_strSrcHost(strSrcHost), m_strSrcPort(strSrcPort),
                  m_strDstHost(strDstHost), m_strDstPort(strDstPort)
{
  // Init the buffer
  for(int iIndex = 0; iIndex < TS_IN_ETHER; iIndex++)
    m_cSocketBuff.SetSlotSize(iIndex, TS_PACKET_LEN);
  m_cOutputInetAddr.Build(strDstHost, strDstPort);
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
C_NetOutput::~C_NetOutput()
{
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
/*C_Output* C_NetOutput::Clone() const
{
  return new C_NetOutput(*this);
}*/


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_NetOutput::OnInit()
{
  try
  {
    // Open the socket
    m_cSocket.Open(SOCK_DGRAM);

    // Allow the socket to send broadcast packets
    int iOptVal = 1;
    m_cSocket.SetOption(SOL_SOCKET, SO_BROADCAST, &iOptVal, sizeof(iOptVal));

    // Allow to use the ip/port couple more than once at a time to be
    // able to send several streams to a same client using the same port
    m_cSocket.SetOption(SOL_SOCKET, SO_REUSEADDR, &iOptVal, sizeof(iOptVal));

    // Try to increase the size of the socket output buffer to 1/2MB (8Mb/s
    // during 1/2s) to avoid packet loss
    iOptVal = 524288;
    for(;;)
    {
      try
      {
        m_cSocket.SetOption(SOL_SOCKET, SO_SNDBUF, &iOptVal, sizeof(iOptVal));
        break;
      }
      catch(E_Exception e)
      {
        iOptVal = iOptVal / 2;

        if(iOptVal <= 524288/16)
          throw E_Output("Unable to allocate output buffer", e);
      }
    }

#ifdef HAVE_SO_BINDTODEVICE
    // If an interface is specified then bind to it
    // (Very useful when several interfaces are connected to the same subnet)
    if(m_strInterface != "")
    {
      struct ifreq sInterface;
      strncpy(sInterface.ifr_ifrn.ifrn_name,
              m_strInterface.GetString(), IFNAMSIZ);
      m_cSocket.SetOption(SOL_SOCKET, SO_BINDTODEVICE, (char *)&sInterface,
                          sizeof(sInterface));
    }
#endif

    // Bind it to the local address
    m_cSocket.Bind(m_strSrcHost, m_strSrcPort);

#ifndef BUGGY_VLC
    // Connect it
    m_cSocket.Connect(m_strDstHost, m_strDstPort);
#endif
  }
  catch(E_Exception e)
  {
    throw E_Output("Output initialisation failed", e);
  }
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_NetOutput::OnClose()
{
  try
  {
    m_cSocket.Close();
  }
  catch(E_Exception e)
  {
    throw E_Output("Output termination failed", e);
  }
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_NetOutput::WriteToPort()
{
  ASSERT(m_pTsProvider);

  unsigned int iPacketNumber = m_cTsBuff.Size();
  
  if(iPacketNumber > 0)
  {
    // To avoid problems with the socket buff
    ASSERT(iPacketNumber <= TS_IN_ETHER);

#ifdef BUGGY_VLC
    // TS concatenation
    for(unsigned int iIndex = 0; iIndex < iPacketNumber; iIndex++)
      memcpy(m_ByteBuff + TS_PACKET_LEN * iIndex, m_cTsBuff[iIndex],
             TS_PACKET_LEN);

    // Send the data that were stored in the buffer
    int iRc = m_cSocket.WriteTo(m_cOutputInetAddr,
                                m_ByteBuff, TS_PACKET_LEN * iPacketNumber);

    if(iRc != TS_IN_ETHER * TS_PACKET_LEN)
      m_iByteLost += TS_IN_ETHER * TS_PACKET_LEN - iRc;
#else
    // Fill in the socketbuff
    for(unsigned int iIndex = 0; iIndex < iPacketNumber; iIndex++)
      m_cSocketBuff.SetSlotBuff(iIndex, m_cTsBuff[iIndex]);
  
    // Send the data that were stored in the buffer
    try
    {
      int iRc = m_cSocket.Send(m_cSocketBuff, iPacketNumber);

      if(iRc != TS_IN_ETHER * TS_PACKET_LEN)
        m_iByteLost += TS_IN_ETHER * TS_PACKET_LEN - iRc;
    }
    catch(E_Exception e)
    {
      throw E_Output("Connection lost", e);
    }
#endif

    // Free the now unused packets
    C_TsPacket* pPacket;
    for(unsigned int i = 0; i < iPacketNumber; i++)
    {
      // Pop the packet from the buffer
      pPacket = m_cTsBuff.Pop();
      ASSERT(pPacket);
      // And release it
      m_pTsProvider->ReleasePacket(pPacket);
    }
  }
}


