/*****************************************************************************
 * socket.c: VideoLAN Channel Server network facilities
 *****************************************************************************
 * Copyright (C) 2001 VideoLAN
 * $Id: socket.c,v 1.6 2002/10/12 19:45:31 marcari Exp $
 *
 * Authors: Marc Ariberti <marcari@via.ecp.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <iostream>

#include "vlcs.h"
#include "logger.h"
#include "config.h"
#include "socket.h"

C_Socket::C_Socket(C_Logger &logger) : i_handle(0), logger(logger)
{
}

void C_Socket::Close( int i_ret )
{
    close( i_handle );
}

/**
 * C_IPv4Socket class implementation
 */

C_IPv4Socket::C_IPv4Socket(C_Logger &logger, int port, char * bind_addr = NULL)
    throw (int) : C_Socket(logger) 
{
    int i_opt;
    struct sockaddr_in sa_server;
    struct in_addr s_bind_addr;

    this->logger = logger;
    
    i_handle = socket( AF_INET, SOCK_DGRAM, 0 );

    if( i_handle < 0 )
    {
        logger << C_Logger::ALERT << "Cannot open socket (" << strerror(errno) 
                << "), exiting" << C_Logger::END;
        throw( -1 );
    }

    if( setsockopt( i_handle, SOL_SOCKET, SO_REUSEADDR, &i_opt, sizeof(i_opt) )
          < 0 )
    {
        logger << C_Logger::ALERT << "Cannot setsockopt (" << strerror(errno) 
                << "), exiting" << C_Logger::END;
        throw( -1 );
    }

    
    sa_server.sin_family = AF_INET;

    if ( bind_addr != NULL )
    {
        inet_aton(bind_addr, &s_bind_addr);
        sa_server.sin_addr.s_addr = s_bind_addr.s_addr;
    } 
    else 
    {
        sa_server.sin_addr.s_addr = INADDR_ANY;
    }
    sa_server.sin_port = ntohs( VLCS_PORT );
    if( bind( i_handle, (struct sockaddr *)&sa_server, sizeof(sa_server) ) < 0 )
    {
        logger << C_Logger::ALERT << "Cannot bind (" << strerror(errno) << "), exiting"
                << C_Logger::END;
        throw( -1 );
    }
}

int C_IPv4Socket::Receive( char * p_buffer, int buffer_len )
{
    socklen_t i_dummy = sizeof( struct sockaddr_in );
    
    return recvfrom( i_handle, p_buffer, buffer_len, 0,
                        (struct sockaddr *)&m_sa_client, &i_dummy);
}

void C_IPv4Socket::Reply( char * psz_message )
{
    time_t tm;
    time(&tm);

    /* TODO: remove this ugly kludge */
    logger << C_Logger::INFO << C_Config::RemoveLF( asctime( localtime( &tm ))) << " : "
            << Ntop( ) << " => " << psz_message << C_Logger::END;
    for( ; ; )
    {
        if( sendto( i_handle, psz_message, strlen( psz_message ) + 1, 0,
                    (struct sockaddr *)&m_sa_client,
                    sizeof( struct sockaddr_in ) ) < 0 )
        {
            if( errno == EINTR )
            {
                continue;
            }
            logger << C_Logger::ALERT << "Cannot sendto : " << strerror(errno) << C_Logger::END;
        }
        break;
    }    
}

const char * C_IPv4Socket::Ntop( void )
{
    return inet_ntop(AF_INET, &(m_sa_client.sin_addr), str, sizeof(str));
}
/**
 * C_IPv6Socket class implementation
 */

C_IPv6Socket::C_IPv6Socket(C_Logger &logger, int port, char * bind_addr = NULL)
    throw (int) : C_Socket(logger)
{
    int i_opt;
    struct sockaddr_in6 sa_server;
    struct in6_addr s_bind_addr;

    this->logger = logger;
    
    i_handle = socket( AF_INET6, SOCK_DGRAM, 0 );

    if( i_handle < 0 )
    {
        logger << C_Logger::ALERT << "Cannot open socket (" << strerror(errno) 
                << "), exiting" << C_Logger::END;
        throw( -1 );
    }

    if( setsockopt( i_handle, SOL_SOCKET, SO_REUSEADDR, &i_opt, sizeof(i_opt) )
          < 0 )
    {
        logger << C_Logger::ALERT << "Cannot setsockopt (" << strerror(errno) 
                << "), exiting" << C_Logger::END;
        throw( -1 );
    }

    
    sa_server.sin6_family = AF_INET6;

    if ( bind_addr != NULL )
    {
        inet_pton(AF_INET6, bind_addr, &s_bind_addr);
        sa_server.sin6_addr.s6_addr = s_bind_addr.s6_addr;
    } 
    else 
    {
        sa_server.sin6_addr = in6addr_any;
    }
    sa_server.sin6_port = ntohs( VLCS_PORT );
    if( bind( i_handle, (struct sockaddr *)&sa_server, sizeof(sa_server) ) < 0 )
    {
        logger << C_Logger::ALERT << "Cannot bind (" << strerror(errno) << "), exiting"
                << C_Logger::END;
        throw( -1 );
    }
}

int C_IPv6Socket::Receive( char * p_buffer, int buffer_len )
{
    socklen_t i_dummy = sizeof( struct sockaddr_in6 );
    
    return recvfrom( i_handle, p_buffer, buffer_len, 0,
                        (struct sockaddr *)&m_sa_client, &i_dummy);
}

void C_IPv6Socket::Reply( char * psz_message )
{
    time_t tm;
    time(&tm);

    /* TODO: remove this ugly kludge */
    logger << C_Logger::INFO << C_Config::RemoveLF( asctime( localtime( &tm ))) << " : "
            << Ntop( ) << " => " << psz_message << C_Logger::END;

    for( ; ; )
    {
        if( sendto( i_handle, psz_message, strlen( psz_message ) + 1, 0,
                    (struct sockaddr *)&m_sa_client,
                    sizeof( struct sockaddr_in6 ) ) < 0 )
        {
            if( errno == EINTR )
            {
                continue;
            }
            //logger->Alert( "Cannot sendto (%s)\n", strerror(errno) );
        }
        break;
    }    
}

const char * C_IPv6Socket::Ntop( void )
{
    return inet_ntop(AF_INET6, &(m_sa_client.sin6_addr), str, sizeof(str));
}
