/* #define RANDOMSTATE /* define to get random memory contents, registers, etc on startup */

#include <stdio.h>
#include <setjmp.h>
#include <signal.h>
#include <varargs.h>

#include <signaltype.h>

/* extern XXX */ char **argvec;

extern int errno;


#include "pdp11.h"
#include "driverint.h"
#include "instrs.h"
#define NEED_NAME
#define NEED_FUNCTIONS
#define NEED_OPFORMATS
#define NEED_DEFS
#define NEED_ALT_DEFS
#include "instrtbl.h"
#include "pdp11-proto.h"
static void dohalt();
static void odt();

extern DRIVER *driverlist[];

unsigned char core[CORESIZE];

static word genregs[2][6];
word *regset;
word sps[4];
word *spp;
word pc;
word *regs[8];
word psw;
word cpu_error;
word pirq;
word maint;
word fps;
word fec;
word fea;
static dfword facc[6];
pagedesc pdesc[4][2][8];
word mmr0;
word mmr1;
word mmr2;
word mmr3;
word cache_control;
word hitmiss;
word mem_sys_err;
word stack_limit;
word micro_break;
word console_switch;
ubamap ubam[8];
int pswc_cm;
int pswc_pri;
int pending;
int halted;
int dma_inhibit;

enum odt_state_{
  ODT_PROMPT = 1,
  ODT_R,
  ODT_OPENLOC,
  ODT_OPENREG,
  ODT_OPENINST,
  ODT_LOC,
  ODT_REG,
  ODT_NEWVAL_LOC,
  ODT_NEWVAL_REG,
  ODT_PROMPT_PREV_REG,
  ODT_PROMPT_PREV_LOC,
  } ;
typedef enum odt_state_ ODT_STATE;

static int notraps;
static jmp_buf notraps_jmp;
static jmp_buf trap_jmp;
static jmp_buf odtdone;
static struct sigvec oldsigio;

static ODT_STATE odt_state;
static unsigned long int odt_xaddr;
static unsigned long int odt_xaddrinc;
static word odt_xregno;
static word odt_newvalue;
static int odt_singlestep;

#define DEF_UNKNOWN (N_INSTR_DEF+1)
#define DEF_UNUSED (N_INSTR_DEF+2)

short int deftbl[65536];

static DRIVER *drivers;
DRIVER *drivermask[8192];
static INTRQ *intrq_head[8];
static INTRQ **intrq_tail[8];
int intrq_pri;

char *initialfn;

void bugchk(va_alist)
va_dcl
{
 va_list ap;
 char *fmt;

 va_start(ap);
 fmt = va_arg(ap,char *);
 fprintf(stderr,"%s: INTERNAL BUG: ",argvec[0]);
 vfprintf(stderr,fmt,ap);
 fprintf(stderr,"\n");
 va_end(ap);
 abort();
}

void busreset()
{
 DRIVER *d;

 for (d=drivers;d;d=d->link) if (d->busreset) (*d->busreset)(d);
}

void checkpsw()
{
 int i;

 psw &= ~PSW_MBZ;
 i = (psw & PSW_CM) >> PSW_CM_SHIFT;
 if (i != pswc_cm)
  { clearpac();
    pswc_cm = i;
  }
 pswc_pri = (psw & PSW_PRI) >> PSW_PRI_SHIFT;
 regset = &genregs[(psw&PSW_RS)?1:0][0];
 spp = &sps[pswc_cm];
 for (i=0;i<6;i++) regs[i] = &regset[i];
 regs[6] = spp;
 regs[7] = &pc;
}

static void adddriver(d)
DRIVER *d;
{
 int i;

 d->link = drivers;
 drivers = d;
 for (i=0;i<8192;i++) d->iomask[i] = 0;
 (*d->init)(d,&d->iomask[0]);
 for (i=0;i<8192;i++)
  { if (d->iomask[i])
     { if (drivermask[i])
	{ fprintf(stderr,"%o claimed by both %s driver and %s driver\n",0760000|i,drivermask[i]->name,d->name);
	}
       drivermask[i] = d;
     }
  }
}

int idcmp(idcp1,idcp2)
char *idcp1;
char *idcp2;
{
 return( (0xffff&(int)((INSTR_DEF *)idcp1)->value) - (0xffff&(int)((INSTR_DEF *)idcp2)->value) );
}

void initinstrs()
{
 int i;
 char *idcp[N_INSTR_DEF];
 INSTR_DEF idarr[N_INSTR_DEF];

 for (i=0;i<N_INSTR_DEF;i++) idcp[i] = (char *) &instr_defs[i];
 heapsort(&idcp[0],N_INSTR_DEF,idcmp);
 for (i=0;i<N_INSTR_DEF;i++) idarr[i] = * (INSTR_DEF *) idcp[i];
 for (i=0;i<N_INSTR_DEF;i++) instr_defs[i] = idarr[i];
}

static void initdrivers()
{
 int i;

 drivers = 0;
 for (i=0;i<8192;i++) drivermask[i] = 0;
 for (i=0;driverlist[i];i++) adddriver(driverlist[i]);
 for (i=0;i<8;i++)
  { intrq_head[i] = 0;
    intrq_tail[i] = &intrq_head[i];
  }
}

static void initmachine()
{
 int i;

#ifdef RANDOMSTATE
 for (i=0;i<sizeof(core)/sizeof(int);i++)
  { ((int *)&core[0])[i] = mth$int_random(); /* XXX assumes core is int-aligned */
  }
 for (i=0;i<6;i++)
  { genregs[0][i] = mth$int_random();
    genregs[1][i] = mth$int_random();
  }
 for (i=0;i<4;i++) sps[i] = mth$int_random();
#endif
 psw = (MODE_K << PSW_CM_SHIFT) | (MODE_K << PSW_PM_SHIFT) | (7 << PSW_PRI_SHIFT);
 cpu_error = 0;
 pirq = 0;
 maint = (1 << MAINT_BOOTADDR_SHIFT) | MAINT_FPAAVAIL | MAINT_MODULEID_KDJ11A | MAINT_POWEROPT_ODT | MAINT_DCOK;
 mmr0 = 0;
 mmr3 = 0;
 cache_control = 0;
 hitmiss = 0;
 mem_sys_err = 0;
 checkpsw();
 for (i=0;i<65536;i++) deftbl[i] = DEF_UNKNOWN;
 dma_inhibit = 0;
 pending = 0;
 clearpac();
 busreset();
}

void loadcore(fn)
char *fn;
{
 FILE *f;
 unsigned long int addr;
 unsigned long int count;
 int value;

 f = fopen(fn,"r");
 if (f == 0)
  { fprintf(stderr,"%s: can't open %s for reading\n",argvec[0],fn);
    exit(1);
  }
 while (1)
  { if (fscanf(f,"%lx%lx",&addr,&count) != 2) break;
    for (;count>0;count--)
     { if (fscanf(f,"%x",&value) != 1) break;
       if (addr < CORESIZE) core[addr] = value & 0xff;
       addr ++;
     }
  }
 fclose(f);
}

void unimpl()
{
 trapto(010);
}

int get_deftbl(inst)
word inst;
{
 int l;
 int m;
 int h;

 if (deftbl[inst] != DEF_UNKNOWN) return(deftbl[inst]);
 l = 0;
 h = N_INSTR_DEF - 1;
 while (h-l > 1)
  { m = (h + l) / 2;
    if (inst >= instr_defs[m].value)
     { l = m;
     }
    else
     { h = m;
     }
  }
 if ((inst & instr_defs[l].mask) != instr_defs[l].value) l = DEF_UNUSED;
 deftbl[inst] = l;
 return(l);
} 

static int checkinterrupt()
{
 INTRQ *irq;

 while (intrq_pri > pswc_pri)
  { irq = intrq_head[intrq_pri];
    if (irq == 0)
     { intrq_tail[intrq_pri] = &intrq_head[intrq_pri];
       intrq_pri --;
       continue;
     }
    if ((irq->driver->intchk == 0) || (*irq->driver->intchk)(irq)) return(1);
    if (halted)
     { printf("\r\n[dismissing level %d interrupt from `%s']",intrq_pri,irq->driver->name);
     }
    irq->flags &= ~IRQ_F_ACTIVE;
    intrq_head[intrq_pri] = irq->pri_link;
  }
 return(0);
}

static void takeinterrupt()
{
 INTRQ *irq;
 word oldpc;
 word oldpsw;

 irq = intrq_head[intrq_pri];
 if (halted)
  { printf("\r\n[taking interrupt: driver `%s' pri %d vec %o]",irq->driver->name,intrq_pri,irq->vec);
  }
 intrq_head[intrq_pri] = irq->pri_link;
 if (intrq_head[intrq_pri] == 0)
  { intrq_tail[intrq_pri] = &intrq_head[intrq_pri];
    for (intrq_pri--;(intrq_pri>0)&&(intrq_head[intrq_pri]==0);intrq_pri--) ;
  }
 if (irq->driver->intack) (*irq->driver->intack)(irq);
 oldpc = pc;
 oldpsw = psw;
 psw = MODE_K << PSW_CM_SHIFT;
 checkpsw();
 pc = fetchword(irq->vec,MMAN_DSPACE);
 /* this makes more sense to me, but apparently real machines use the uncommented line following....
 psw = (fetchword(irq->vec+2,MMAN_DSPACE) & ~(PSW_MBZ|PSW_PRI)) | ((irq->pri << PSW_PRI_SHIFT) & PSW_PRI);
 */
 psw = fetchword(irq->vec+2,MMAN_DSPACE) & ~PSW_MBZ;
 checkpsw();
 sp_push(oldpsw);
 sp_push(oldpc);
 irq->flags &= ~IRQ_F_ACTIVE;
}

static void do_pending()
{
 register int pend;

 pend = pending;
 pending = 0;
 if (pend & PEND_YSTACK)
  { cpu_error |= CPUERR_Y_STACK;
    trapto(4);
  }
 if (pend & PEND_TRACE)
  { trapto(014);
  }
}

static void step()
{
 register word iaddr;
 register word inst;
 register int dno;
 register DRIVER *d;
 static int last_pri = 7;

 /*dumpstate();*/
 if (pswc_pri > last_pri) last_pri = pswc_pri;
 if (intrq_pri > last_pri)
  { if (checkinterrupt()) takeinterrupt();
    return;
  }
 last_pri = pswc_pri;
 for (d=drivers;d;d=d->link)
  { register void (*f)();
    f = d->tick;
    if (f) (*f)(d);
  }
 iaddr = pc;
 if (! MMR_FROZEN)
  { mmr1 = 0;
    mmr2 = iaddr;
  }
 pc += 2;
 inst = fetchword(iaddr,MMAN_ISPACE);
 if (psw & PSW_T) pending |= PEND_TRACE;
 dno = (deftbl[inst] == DEF_UNKNOWN) ? get_deftbl(inst) : deftbl[inst];
 if (dno == DEF_UNUSED) unimpl();
 (*instr_defs[dno].fxn)(inst);
 if (pending) do_pending();
}

static void run()
{
 halted = 0;
 notraps = 0;
 switch (maint & MAINT_POWEROPT)
  { case MAINT_POWEROPT_TRAP24:
       pc = fetchword(024,MMAN_DSPACE);
       psw = fetchword(026,MMAN_DSPACE) & ~PSW_MBZ;
       break;
    case MAINT_POWEROPT_ODT:
       odt();
       break;
    case MAINT_POWEROPT_173000:
       pc = 0173000;
       psw = PSW_PRI;
       break;
    case MAINT_POWEROPT_USERBOOT:
       pc = maint & MAINT_BOOTADDR;
       psw = PSW_PRI;
       break;
  }
 checkpsw();
 while (setjmp(trap_jmp))
  { if (halted) dohalt();
  }
 while (1)
  { step();
    if (halted) dohalt();
  }
}

static void dohalt()
{
 if (maint & MAINT_HALTOPT)
  { sps[MODE_K] = 4;
    trapto(04);
    halted = 0;
  }
 else
  { odt();
  }
}

void trapto(trapv)
int trapv;
{
 static int doing_trap = 0;
 word newpc;
 word newpsw;
 word oldpc;
 word oldpsw;

 if (notraps) longjmp(notraps_jmp,1);
 oldpc = pc;
 oldpsw = psw;
 doing_trap ++;
 switch (doing_trap)
  { case 1:
       psw = MODE_K << PSW_CM_SHIFT;
       checkpsw();
       pc = fetchword(trapv,MMAN_DSPACE);
       newpsw = fetchword(trapv+2,MMAN_DSPACE);
       psw = newpsw & ~PSW_MBZ;
       checkpsw();
       sp_push(oldpsw);
       sp_push(oldpc);
       break;
    case 2:
       psw = MODE_K << PSW_CM_SHIFT;
       checkpsw();
       pc = fetchword(04,MMAN_DSPACE);
       newpsw = fetchword(06,MMAN_DSPACE);
       psw = newpsw & ~PSW_MBZ;
       checkpsw();
       storeword(2,MMAN_DSPACE,oldpsw);
       storeword(0,MMAN_DSPACE,oldpc);
       sp = 0;
       break;
    default:
       mmr0 &= ~MMR0_MM_ENB;
       pc = core[4] | (core[5] << 8);
       newpsw = core[6] | (core[7] << 8);
       psw = (newpsw & ~(PSW_MBZ|PSW_CM|PSW_PRI)) | (MODE_K << PSW_CM_SHIFT) | (7 << PSW_PRI_SHIFT);
       checkpsw();
       core[0] = oldpc & 0xff;
       core[1] = (oldpc >> 8) & 0xff;
       core[2] = oldpsw & 0xff;
       core[3] = (oldpsw >> 8) & 0xff;
       sp = 0;
       break;
  }
 doing_trap --;
 longjmp(trap_jmp,1);
}

void interrupt(d,vec,pri)
DRIVER *d;
int vec;
int pri;
{
 INTRQ *irq;

 irq = &d->irq[pri];
 if (irq->flags & IRQ_F_ACTIVE) return;
 irq->driver = d;
 irq->vec = vec;
 irq->pri = pri;
 irq->pri_link = 0;
 irq->flags |= IRQ_F_ACTIVE;
 *intrq_tail[pri] = irq;
 intrq_tail[pri] = &irq->pri_link;
 if (pri > intrq_pri) intrq_pri = pri;
}

void main(ac,av)
int ac;
char **av;
{
 int skip;
 int errs;
 int argno;

 argvec = av; /* XXX */
 if (0)
  {
usage:;
    fprintf(stderr,"Usage: %s initial-core-file\n",argvec[0]);
    exit(1);
  }
 skip = 0;
 errs = 0;
 argno = 0;
 for (ac--,av++;ac;ac--,av++)
  { if (skip > 0)
     { skip --;
       continue;
     }
    if (**av == '-')
     { for (++*av;**av;++*av)
	{ if (0)
	   {
needarg:;
	     fprintf(stderr,"%s: -%c needs another argument\n",argvec[0],**av);
	     errs ++;
	     continue;
	   }
	  switch (**av)
	   { default:
		fprintf(stderr,"%s: bad flag -%c\n",argvec[0],**av);
		errs ++;
		break;
	   }
	}
     }
    else
     { switch (argno++)
	{ default:
	     fprintf(stderr,"%s: extra argument %s\n",argvec[0],*av);
	     errs ++;
	     break;
	  case 0:
	     initialfn = *av;
	     break;
	}
     }
  }
 if (errs) goto usage;
 if (! initialfn) goto usage;
 initinstrs();
 initdrivers();
 initmachine();
 loadcore(initialfn);
 run();
}

static void dumpstate()
{
 printf("\r\n");
 printf("r0=%06o r2=%06o r4=%06o sp=%06o\r\n",r0,r2,r4,r6);
 printf("r1=%06o r3=%06o r5=%06o pc=%06o\r\n",r1,r3,r5,r7);
 printf("psw=%06o: cm=%c pm=%c rs=%d pri=%d t=%d n=%d z=%d v=%d c=%d\r\n",
	psw,
	"ks?u"[(psw&PSW_CM)>>PSW_CM_SHIFT],
	"ks?u"[(psw&PSW_PM)>>PSW_PM_SHIFT],
	(psw&PSW_RS) ? 1 : 0,
	(psw&PSW_PRI) >> PSW_PRI_SHIFT,
	(psw&PSW_T) ? 1 : 0,
	(psw&PSW_N) ? 1 : 0,
	(psw&PSW_Z) ? 1 : 0,
	(psw&PSW_V) ? 1 : 0,
	(psw&PSW_C) ? 1 : 0 );
}

#define state      odt_state
#define xaddr      odt_xaddr
#define xaddrinc   odt_xaddrinc
#define xregno     odt_xregno
#define newvalue   odt_newvalue
#define singlestep odt_singlestep

static void Gcmd(addr)
word addr;
{
 mmr0 = 0;
 mmr3 = 0;
 pirq = 0;
 fps = 0;
 cpu_error = 0;
 /* set ccr<8> (flush), clear mem_sys_err */
 psw = 0;
 pc = addr;
 halted = 0;
 longjmp(odtdone,1);
} 

static void Pcmd()
{
 halted = 0;
 longjmp(odtdone,1);
}

static Scmd()
{
 halted = 1;
 singlestep = 1;
 longjmp(odtdone,1);
}

static signaltype nullh()
{
}

static void setsigio()
{
 struct sigvec sv;

 sv.sv_handler = nullh;
 sv.sv_mask = 0;
 sv.sv_flags = 0;
 sigvec(SIGIO,&sv,&oldsigio);
}

static void resetsigio()
{
 sigvec(SIGIO,&oldsigio,(struct sigvec *)0);
}

static void flushin()
{
 char ch;

 while (read(0,&ch,1) == 1) ;
}

static int get()
{
 char ch;

 fflush(stdout);
 while (1)
  { if (read(0,&ch,1) == 1) return(ch);
    sigpause(0);
  }
}

static void put(ch)
int ch;
{
/*
 char chc;

 chc = ch;
 write(1,&chc,1);
*/
 putchar(ch);
}

static int puts(s)
/* #define CONST const */
#define CONST
CONST char *s;				/* To keep damn gcc & <stdio.h> happy */
{
 for (;*s;s++) put(*s);
}

static void putq()
{
 puts("?\r\n@");
}

static void putx8(val)
unsigned long int val;
{
 int i;

 val &= 0xffffffff;
 for (i=28;i>=0;i-=4) put("0123456789abcdef"[(val>>i)&0xf]);
}

static void put8(val)
unsigned long int val;
{
 int i;

 val &= 017777777;
 for (i=21;i>=0;i-=3) put('0'+((val>>i)&7));
}

static void put6(val)
word val;
{
 int i;

 val &= 0xffff;
 for (i=15;i>=0;i-=3) put('0'+((val>>i)&7));
}

static void put3(val)
word val;
{
 put('0'+((val>>6)&3));
 put('0'+((val>>3)&7));
 put('0'+(val&7));
}

static void put2(val)
word val;
{
 put('0'+((val>>3)&7));
 put('0'+(val&7));
}

static void odt()
{
 setsigio();
 if (setjmp(odtdone))
  { fflush(stdout);
    resetsigio();
    notraps = 0;
    return;
  }
 flushin();
 cons__flush();
 state = ODT_PROMPT;
 notraps = 1;
 while (setjmp(notraps_jmp))
  { char obuf[64];
    sprintf(&obuf[0],"(trap from %d)",state);
    puts(&obuf[0]);
    state = ODT_PROMPT;
  }
 puts("\r\n");
 if (singlestep)
  { singlestep = 0;
    if (intrq_pri > pswc_pri)
     { puts("[possible level ");
       put('0'+intrq_pri);
       puts(" interrupt pending]\r\n");
     }
    disas(pc);
  }
 else
  { put6(pc);
  }
 puts("\r\n@");
 fflush(stdout);
 while (1)
  { int ch;
    ch = 0x7f & get();
    if (ch == 3)
     { cons__reset();
       exit(0);
     }
    if (ch != 012) put(ch);
    switch (state)
     { case ODT_PROMPT:
       case ODT_PROMPT_PREV_REG:
       case ODT_PROMPT_PREV_LOC:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		state = ODT_LOC;
		xaddr = ch - '0';
		break;
	     case 'r': case 'R': case '$':
		state = ODT_R;
		xregno = 0;
		break;
	     case 'g': case 'G':
		Gcmd((word)0);
		break;
	     case 'p': case 'P':
		Pcmd();
		break;
	     case 's': case 'S':
		Scmd();
		break;
	     case 'Q':
		cons__reset();
		exit(0);
		break;
	     case 'Z':
		kill(getpid(),SIGTSTP);
		break;
	     case '/':
		switch (state)
		 { case ODT_PROMPT:
		      putq();
		      break;
		   case ODT_PROMPT_PREV_REG:
		      xregno &= 0377;
		      if (xregno != 077) xregno &= 7;
		      state = ODT_OPENREG;
		      put6((xregno>7)?psw:*regs[xregno]);
		      put(' ');
		      break;
		   case ODT_PROMPT_PREV_LOC:
		      xaddr &= 017777776;
		      state = ODT_OPENLOC;
		      put6(fetchphys(xaddr));
		      put(' ');
		      break;
		 }
		break;
	     default:
		state = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_R:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		state = ODT_REG;
		xregno = (xregno << 3) | (ch - '0');
		break;
	     case 's': case 'S':
		state = ODT_REG;
		xregno = 077;
		break;
	     default:
		state = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_OPENLOC:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		state = ODT_NEWVAL_LOC;
		newvalue = ch - '0';
		break;
	     case '\r':
		state = ODT_PROMPT_PREV_LOC;
		puts("\n@");
		break;
	     case '\n':
		xaddr = (xaddr & ~0xffff) | ((xaddr + 2) & 0xfffe);
		state = ODT_OPENLOC;
		puts("\r\n@");
		put8(xaddr,8);
		put('/');
		put6(fetchphys(xaddr));
		put(' ');
		break;
	     default:
		state = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_OPENREG:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		state = ODT_NEWVAL_REG;
		newvalue = ch - '0';
		break;
	     case '\r':
		state = ODT_PROMPT_PREV_REG;
		puts("\n@");
		break;
	     case '\n':
		if (xregno > 7)
		 { state = ODT_PROMPT_PREV_REG;
		   puts("\r\n@");
		 }
		else
		 { xregno = (xregno + 1) & 7;
		   state = ODT_OPENREG;
		   puts("\r\n@R");
		   put('0'+xregno);
		   put('/');
		   put6(*regs[xregno]);
		   put(' ');
		 }
		break;
	     default:
		state = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_OPENINST:
	  switch (ch)
	   { case '\r':
		state = ODT_PROMPT_PREV_LOC;
		puts("\n@");
		break;
	     case '\n':
		xaddr += xaddrinc;
		puts("\r\n");
		xaddrinc = 2;
		xaddrinc = disas((word)(xaddr&0xfffe));
		put(' ');
		break;
	     default:
		state = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_LOC:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		xaddr = (xaddr << 3) | (ch - '0');
		break;
	     case '/':
		xaddr &= 017777776;
		state = ODT_OPENLOC;
		put6(fetchphys(xaddr));
		put(' ');
		break;
	     case 'g': case 'G':
		Gcmd((word)(xaddr&0xffff));
		break;
	     case 's': case 'S':
		pc = xaddr & 0xffff;
		Scmd();
		break;
	     case 'i':
		puts("\r\n");
		state = ODT_OPENINST;
		xaddrinc = 2;
		xaddrinc = disas((word)(xaddr&0xfffe));
		put(' ');
		break;
	     default:
		state = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_REG:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		xregno = (xregno << 3) | (ch - '0');
		break;
	     case 's': case 'S':
		xregno = 077;
		break;
	     case '/':
		xregno &= 0377;
		if (xregno != 077) xregno &= 7;
		state = ODT_OPENREG;
		put6((xregno>7)?psw:*regs[xregno]);
		put(' ');
		break;
	     default:
		state = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_NEWVAL_LOC:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		newvalue = (newvalue << 3) | (ch - '0');
		break;
	     case '\r':
		storephys(xaddr,newvalue);
		state = ODT_PROMPT_PREV_LOC;
		puts("\n@");
		break;
	     case '\n':
		storephys(xaddr,newvalue);
		xaddr = (xaddr & ~0xffff) | ((xaddr + 2) & 0xfffe);
		state = ODT_OPENLOC;
		puts("\r\n@");
		put8(xaddr,8);
		put('/');
		put6(fetchphys(xaddr));
		put(' ');
		break;
	     default:
		state = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
       case ODT_NEWVAL_REG:
	  switch (ch)
	   { case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7':
		newvalue = (newvalue << 3) | (ch - '0');
		break;
	     case '\r':
		if (xregno > 7)
		 { psw = (psw & PSW_T) | (newvalue & ~(PSW_T|PSW_MBZ));
		 }
		else
		 { *regs[xregno] = newvalue;
		 }
		state = ODT_PROMPT_PREV_REG;
		puts("\n@");
		break;
	     case '\n':
		if (xregno > 7)
		 { psw = (psw & PSW_T) | (newvalue & ~(PSW_T|PSW_MBZ));
		   state = ODT_PROMPT_PREV_REG;
		   puts("\r\n@");
		 }
		else
		 { *regs[xregno] = newvalue;
		   xregno = (xregno + 1) & 7;
		   state = ODT_OPENREG;
		   puts("\r\n@R");
		   put('0'+xregno);
		   put('/');
		   put6(*regs[xregno]);
		   put(' ');
		 }
		break;
	     default:
		state = ODT_PROMPT;
		putq();
		break;
	   }
	  break;
     }
  }
}

/*#include "disas-common"*/
/* This file is used (via #include) in pdp11.c and disas11.c */
/* The necessary stuff is assumed to have been set up. */

int adjust_adjacent(dtx,opfwant)
int dtx;
INSTR_OPS opfwant;
{
 int dtx2;

 for (dtx2=dtx;(dtx2>=0)&&(instr_defs[dtx2].value==instr_defs[dtx].value);dtx2--) if (instr_defs[dtx2].ops_format == opfwant) return(dtx2);
 for (dtx2=dtx;(dtx2<N_INSTR_DEF)&&(instr_defs[dtx2].value==instr_defs[dtx].value);dtx2--) if (instr_defs[dtx2].ops_format == opfwant) return(dtx2);
 bugchk("adjust_adjacent can't find it");
}

int disas(addr)
word addr;
{
 word inst;
 int dtx;
 int i;
 word pctemp;
 word a;
 int nw;

 put6(addr);
 puts(": ");
 inst = fetchword(addr,MMAN_ISPACE);
 dtx = get_deftbl(inst);
 pctemp = (addr + 2) & 0xffff;
 if (dtx == DEF_UNUSED)
  { puts(".word   ");
    put6(inst);
  }
 else
  { switch (instr_defs[dtx].ops_format)
     { case OPS_F0:
	  if (fps & FPS_FD) dtx = adjust_adjacent(dtx,OPS_D0);
	  break;
       case OPS_FA6_F0:
	  if (fps & FPS_FD) dtx = adjust_adjacent(dtx,OPS_FA6_D0);
	  break;
       case OPS_F0_FA6:
	  if (fps & FPS_FD) dtx = adjust_adjacent(dtx,OPS_D0_FA6);
	  break;
       case OPS_D0:
	  if (! (fps & FPS_FD)) dtx = adjust_adjacent(dtx,OPS_F0);
	  break;
       case OPS_FA6_D0:
	  if (! (fps & FPS_FD)) dtx = adjust_adjacent(dtx,OPS_FA6_F0);
	  break;
       case OPS_D0_FA6:
	  if (! (fps & FPS_FD)) dtx = adjust_adjacent(dtx,OPS_F0_FA6);
	  break;
     }
    puts(instr_defs[dtx].name);
    for (i=strlen(instr_defs[dtx].name);i<8;i++) put(' ');
    nw = 0;
    switch (instr_defs[dtx].ops_format)
     { case OPS_NONE:
	  break;
       case OPS_G0:
	  nw += disas_g(inst,&pctemp);
	  break;
       case OPS_R0:
	  nw += disas_r(inst);
	  break;
       case OPS_O30:
	  put('0'+(inst&7));
	  nw ++;
	  break;
       case OPS_B80:
	  put6(pctemp+((inst&0xff)<<1)+((inst&0x80)?0xfe00:0));
	  nw += 6;
	  break;
       case OPS_R6_G0:
	  nw += disas_r(inst>>6);
	  put(',');
	  nw ++;
	  nw += disas_g(inst,&pctemp);
	  break;
       case OPS_G0_R6:
	  nw += disas_g(inst,&pctemp);
	  put(',');
	  nw ++;
	  nw += disas_r(inst>>6);
	  break;
       case OPS_G6_G0:
	  nw += disas_g(inst>>6,&pctemp);
	  put(',');
	  nw ++;
	  nw += disas_g(inst,&pctemp);
	  break;
       case OPS_R6_B60:
	  nw += disas_r(inst>>6);
	  put(',');
	  nw ++;
	  put6(pctemp-((inst&077)<<1));
	  nw += 6;
	  break;
       case OPS_O80:
	  put3(inst);
	  nw += 3;
	  break;
       case OPS_O60:
	  put2(inst);
	  nw += 2;
	  break;
       case OPS_F0:
       case OPS_D0:
	  nw += disas_f(inst,&pctemp);
	  break;
       case OPS_FA6_F0:
       case OPS_FA6_D0:
	  nw += disas_fr((inst>>6)&3);
	  put(',');
	  nw ++;
	  nw += disas_f(inst,&pctemp);
	  break;
       case OPS_F0_FA6:
       case OPS_D0_FA6:
	  nw += disas_f(inst,&pctemp);
	  put(',');
	  nw ++;
	  nw += disas_fr((inst>>6)&3);
	  break;
       case OPS_FA6_G0:
	  nw += disas_fr((inst>>6)&3);
	  put(',');
	  nw ++;
	  nw += disas_g(inst,&pctemp);
	  break;
       case OPS_G0_FA6:
	  nw += disas_g(inst,&pctemp);
	  put(',');
	  nw ++;
	  nw += disas_fr((inst>>6)&3);
	  break;
     }
    put(' ');
    nw ++;
    for (;nw<24;nw++) put(' ');
    for (;nw%8;nw++) put(' ');
    for (a=addr;a!=pctemp;a=(a+2)&0xffff)
     { put((a==addr)?'[':' ');
       put6(fetchword(a,MMAN_ISPACE));
     }
    put(']');
  }
 return((pctemp-addr)&0xffff);
}

int disas_r(rno)
int rno;
{
 static char *rnames[] = { "r0", "r1", "r2", "r3", "r4", "r5", "sp", "pc" };

 puts(rnames[rno&7]);
 return(strlen(rnames[rno&7]));
}

int disas_fr(rno)
int rno;
{
 static char *rnames[] = { "f0", "f1", "f2", "f3", "f4", "f5", "?f6?", "?f7?" };

 puts(rnames[rno&7]);
 return(strlen(rnames[rno&7]));
}

int disas_g(op,pcp)
int op;
word *pcp;
{
 int n;

 n = 0;
 switch (op & 070)
  { case 000:
       n += disas_r(op);
       break;
    case 010:
       put('(');
       n ++;
       n += disas_r(op);
       put(')');
       n ++;
       break;
    case 020:
       if ((op & 7) == 7)
	{ put('#');
	  n ++;
	  put6(fetchword(*pcp,MMAN_ISPACE));
	  n += 6;
	  *pcp = (*pcp + 2) & 0xffff;
	}
       else
	{ put('(');
	  n ++;
	  n += disas_r(op);
	  puts(")+");
	  n += 2;
	}
       break;
    case 030:
       if ((op & 7) == 7)
	{ puts("@#");
	  n += 2;
	  put6(fetchword(*pcp,MMAN_ISPACE));
	  n += 6;
	  *pcp = (*pcp + 2) & 0xffff;
	}
       else
	{ puts("@(");
	  n += 2;
	  n += disas_r(op);
	  puts(")+");
	  n += 2;
	}
       break;
    case 040:
       puts("-(");
       n += 2;
       n += disas_r(op);
       put(')');
       n ++;
       break;
    case 050:
       puts("@-(");
       n += 3;
       n += disas_r(op);
       put(')');
       n ++;
       break;
    case 060:
       if ((op & 7) == 7)
	{ put6(fetchword(*pcp,MMAN_ISPACE)+2+*pcp);
	  n += 6;
	  *pcp = (*pcp + 2) & 0xffff;
	}
       else
	{ put6(fetchword(*pcp,MMAN_ISPACE));
	  n += 6;
	  *pcp = (*pcp + 2) & 0xffff;
	  put('(');
	  n ++;
	  n += disas_r(op);
	  put(')');
	  n ++;
	}
       break;
    case 070:
       if ((op & 7) == 7)
	{ put('@');
	  n ++;
	  put6(fetchword(*pcp,MMAN_ISPACE)+2+*pcp);
	  n += 6;
	  *pcp = (*pcp + 2) & 0xffff;
	}
       else
	{ put('@');
	  n ++;
	  put6(fetchword(*pcp,MMAN_ISPACE));
	  n += 6;
	  *pcp = (*pcp + 2) & 0xffff;
	  put('(');
	  n ++;
	  n += disas_r(op);
	  put(')');
	  n ++;
	}
       break;
  }
 return(n);
}

int disas_f(op,pcp)
int op;
word *pcp;
{
 int n;

 n = 0;
 switch (op & 070)
  { case 000:
       n += disas_fr(op);
       break;
    case 010:
       put('(');
       n ++;
       n += disas_r(op);
       put(')');
       n ++;
       break;
    case 020:
       if ((op & 7) == 7)
	{ puts("#0x");
	  n += 3;
	  if (fps & FPS_FD)
	   { unsigned long int fl;
	     unsigned long int fh;
	     fl = 0xffff && (unsigned long int) fetchword(*pcp,MMAN_ISPACE);
	     *pcp = (*pcp + 2) & 0xffff;
	     fl |= (0xffff && (unsigned long int) fetchword(*pcp,MMAN_ISPACE)) << 16;
	     *pcp = (*pcp + 2) & 0xffff;
	     fh = 0xffff && (unsigned long int) fetchword(*pcp,MMAN_ISPACE);
	     *pcp = (*pcp + 2) & 0xffff;
	     fh |= (0xffff && (unsigned long int) fetchword(*pcp,MMAN_ISPACE)) << 16;
	     *pcp = (*pcp + 2) & 0xffff;
	     putx8(fh);
	     putx8(fl);
	     n += 16;
	   }
	  else
	   { unsigned long int f;
	     f = 0xffff && (unsigned long int) fetchword(*pcp,MMAN_ISPACE);
	     *pcp = (*pcp + 2) & 0xffff;
	     f |= (0xffff && (unsigned long int) fetchword(*pcp,MMAN_ISPACE)) << 16;
	     *pcp = (*pcp + 2) & 0xffff;
	     putx8(f);
	     n += 8;
	   }
	}
       else
	{ put('(');
	  n ++;
	  n += disas_r(op);
	  puts(")+");
	  n += 2;
	}
       break;
    case 030:
       if ((op & 7) == 7)
	{ puts("@#");
	  n += 2;
	  put6(fetchword(*pcp,MMAN_ISPACE));
	  n += 6;
	  *pcp = (*pcp + 2) & 0xffff;
	}
       else
	{ puts("@(");
	  n += 2;
	  n += disas_r(op);
	  puts(")+");
	  n += 2;
	}
       break;
    case 040:
       puts("-(");
       n += 2;
       n += disas_r(op);
       put(')');
       n ++;
       break;
    case 050:
       puts("@-(");
       n += 3;
       n += disas_r(op);
       put(')');
       n ++;
       break;
    case 060:
       if ((op & 7) == 7)
	{ put6(fetchword(*pcp,MMAN_ISPACE)+2+*pcp);
	  n += 6;
	  *pcp = (*pcp + 2) & 0xffff;
	}
       else
	{ put6(fetchword(*pcp,MMAN_ISPACE));
	  n += 6;
	  *pcp = (*pcp + 2) & 0xffff;
	  put('(');
	  n ++;
	  n += disas_r(op);
	  put(')');
	  n ++;
	}
       break;
    case 070:
       if ((op & 7) == 7)
	{ put('@');
	  n ++;
	  put6(fetchword(*pcp,MMAN_ISPACE)+2+*pcp);
	  n += 6;
	  *pcp = (*pcp + 2) & 0xffff;
	}
       else
	{ put('@');
	  n ++;
	  put6(fetchword(*pcp,MMAN_ISPACE));
	  n += 6;
	  *pcp = (*pcp + 2) & 0xffff;
	  put('(');
	  n ++;
	  n += disas_r(op);
	  put(')');
	  n ++;
	}
       break;
  }
 return(n);
}
