/* p11 - pdp11 emulator; Copyright (C) 1994 Hartmut Brandt, Joerg Micheel 
 * see the file LICENSE for further information */

# include "proc.h"

/*
 * processor internal registers and generic device support
 */

fd_set	fd_devices;
IODev	*async_devices[FD_DEVICES];
void	*async_argp[FD_DEVICES];
pid_t	async_pids[FD_DEVICES];

Timeout	timeouts[MAXTIMEOUT];
int	ntimeouts;

static void	onsigio();
static void	onalarm();
static void	kill_children(void);
static void	ltc(void *);

/*
 * Processor internal registers
 */
enum {
	Reg_PSW		= 017777776,
	Reg_PIRQ	= 017777772,
	Reg_CPUErr	= 017777766,
	Reg_LTC		= 017777546,
	Reg_Maint	= 017777750,
	Reg_CSW		= 017777570,	/* console switch */
};

static void	proc_ctrl_complete(IODev *);
static void	proc_reset(IODev *);
static ushort	proc_fetch(IODev *, unsigned);
static void	proc_store(IODev *, unsigned, int, ushort);
static ushort	proc_vector(IODev *);
static void	proc_info(IODev *);

static void 	proc_setreq(IODev *);
static void	do_pwrfail(void *);

static struct idata {
	int	timerreq;
	int	tinterval;
	int	trunning;
	ushort	csw;
	int	pwrfail;
} idata = {
	0,
	20,
	0,
	0,
	0,
};


IOOps kdj_ops = {
	0,
	0,
	proc_ctrl_complete,
	proc_reset,
	proc_fetch,
	proc_store,
	proc_vector,
	0,			/* dma 		*/
	0,			/* async	*/
	proc_info,
};

IODev internals = {
	0,		/* reqpri 	*/
	0,		/* next 	*/
	&idata,		/* data		*/
	"KDJ11A",	/* name 	*/
	&kdj_ops,	/* ioops	*/
	0,		/* unit		*/
};

void
catch_io_sigs(void)
{
	SIGNAL(SIGIO, onsigio);
	SIGNAL(SIGALRM, onalarm);
}


static void
proc_ctrl_complete(IODev *dev)
{
	ATEXIT(kill_children, NULL);

	/*
	 * Load power-up values (assume proc is zeroed)
	 */
	proc.pri = 7;

	proc.reg[7] = startloc;
	proc.reg[6] = 01000;
	proc.maint = 025;

	/*
	 * Install in iopage
	 */
	proc.iopage[IOP(Reg_PSW   )] = dev;
	proc.iopage[IOP(Reg_CPUErr)] = dev;
	proc.iopage[IOP(Reg_PIRQ  )] = dev;
	proc.iopage[IOP(Reg_LTC   )] = dev;
	proc.iopage[IOP(Reg_Maint )] = dev;
	proc.iopage[IOP(Reg_CSW   )] = dev;

	register_timer(20, ltc, &proc);
	start_timer();
}

void
start_timer(void)
{
	struct itimerval tval;

	if(!idata.trunning) {
		tval.it_interval.tv_usec = tval.it_value.tv_usec = 1000 * idata.tinterval;
		tval.it_interval.tv_sec = tval.it_value.tv_sec = 0;
		setitimer(ITIMER_REAL, &tval, 0);
		idata.trunning = 1;
	}
}

int
stop_timer(void)
{
	struct itimerval tval;

	if(idata.trunning) {
		tval.it_interval.tv_usec = tval.it_value.tv_usec = 0;
		tval.it_interval.tv_sec = tval.it_value.tv_sec = 0;
		setitimer(ITIMER_REAL, &tval, 0);
		idata.trunning = 0;
		return 1;
	} else
		return 0;
}

static void	
onalarm()
{
	Timeout *t;

	for(t = timeouts; t < &timeouts[ntimeouts]; t++)
		if(--t->curr == 0) {
			t->func(t->arg);
			t->curr = t->time;
		}
}

static void
ltc(void *d)
{
	if(proc.ltc & 0100) {
		idata.timerreq = 1;
		proc_setreq(&internals);
	}
}

/*
 * reset signal
 */
static void
proc_reset(IODev *dev)
{
	proc.pirq = 0;
	proc.ltc = 0;
	dev->reqpri = 0;
	((struct idata *)dev->data)->timerreq = 0;
}

/*
 * fetch value from register
 */
ushort
proc_fetch(IODev *dev, unsigned e)
{
	ushort v;

	switch(e) {
	case Reg_PSW:
		v = ((proc.c&1)<<0) | ((proc.v&1)<<1) | ((proc.z&1)<<2) | ((proc.n&1)<<3)
		  | (proc.t<<4) | (proc.pri << 5) | (proc.regs << 11)
		  | (proc.prevmode << 12) | (proc.curmode << 14);
		break;

	case Reg_CPUErr:	v = proc.cpuerr; 	break;
	case Reg_PIRQ:		v = proc.pirq;		break;
	case Reg_LTC:		v = proc.ltc;		break;
	case Reg_Maint:		v = proc.maint;		break; 
	case Reg_CSW:		v = 0;			break;

	default:
		Warning("proc_fetch(%u)", e);
		Trap4(020);
	}	

	return v;
}

/*
 * load internal registers, do with care
 */
static void
proc_store(IODev *dev, unsigned pa, int mode, ushort v)
{
	int i;
	ushort *dp;
	unsigned *ap, *lp;
	struct idata *ip;
	ushort old;

	switch(pa) {

	case Reg_PSW:
		/*
		 * set all, expect unused  and T-bit
		 */
		if(!(mode & M_Low)) {
			int newmode = (v >> 14) & 3, regs = (v >> 11) & 1;
			switchmode(newmode, regs);
			proc.curmode = newmode;
			proc.prevmode = (v >> 12) & 3;
			proc.regs = regs;
		}
		if(!(mode & M_High)) {
			proc.c = (v >> 0) & 1;
			proc.v = (v >> 1) & 1;
			proc.z = (v >> 2) & 1;
			proc.n = (v >> 3) & 1;
			proc.t = (v >> 4) & 1;
			proc.pri = (v >> 5) & 7;
		}
		break;

	case Reg_CPUErr:
		/*
		 * cleared by write
		 */
		proc.cpuerr = 0; 	
		break;


	case Reg_PIRQ:
		/*
		 * high byte writeable
		 */
		if(mode & M_Low)
			break;
		proc.pirq = v & 0177000;

		proc_setreq(dev);
		break;

	case Reg_LTC:
		/*
		 * Bit 0100 only
		 */
		if(mode & M_High)
			break;
		proc.ltc = v & 0100;
		break;

	case Reg_Maint:
		/*
		 * Not writeable
		 */
		break;

	case Reg_CSW:
		ip = dev->data;
		old = ip->csw;
		if(mode & M_High)
			SHB(ip->csw, v);
		else if(mode & M_Low)
			SLB(ip->csw, v);
		else
			ip->csw = v;
		if(old != ip->csw)
			printf("CSW: %06ho\n", ip->csw);
		break;

	default:
		Warning("proc_store(%u)", pa);
		Trap4(020);
	}	
}

/*
 * return vector
 */
static ushort
proc_vector(IODev *dev)
{
	int pirq = (proc.pirq >> 1) & 7;
	struct idata *id = (struct idata *)dev->data;

	if(id->pwrfail) {
		id->pwrfail = 0;
		proc_setreq(dev);
		return 024;
	}
	if(pirq >= 6)
		return 0240;
	if(id->timerreq) {
		id->timerreq = 0;
		proc_setreq(dev);
		return 0100;
	}
	if(pirq > 0)
		return 0240;
	Warning("bad call to proc_vector");
	return 4;
}

/*
 * set interrupt requests
 */
static void
proc_setreq(IODev *dev)
{
	int i, req, pri;

	dev->reqpri = pri = 0;

	if(((struct idata *)dev->data)->pwrfail)
		pri = NMIPRI;

	if(((struct idata *)dev->data)->timerreq)
		pri = 6;

	for(i = 0100000, req = 7; i >= 01000; i >>= 1, req--)
		if(proc.pirq & i) {
			proc.pirq = (proc.pirq & 0177000) | (req * 042);
			if(req > pri)
				pri = req;
			break;
		}

	if(pri)
		IRQ(dev, pri);
}

/*
 * initiate power fail timer
 */
void
proc_pwrfail()
{
	if(idata.pwrfail)
		return;		/* already power off */
	idata.pwrfail = 1;
	proc.maint &= ~1;	/* switch off PWOK */
	proc_setreq(&internals);
	register_timer(1000, (void (*)(void *))pwrfail_exit, &proc);
}

void
pwrfail_exit(Proc *procp)
{
/*	printf("power fail exit\n"); */
	exit(0);
}

/*
 * start an asynchronous device handler
 */
int
register_handler(char *prog, char *argv[], IODev *dev, void *argp)
{
	int	sd[2];
	int	pid;
	int	fd;

	if( socketpair( PF_UNIX, SOCK_STREAM, 0, sd ) < 0 )
		panic( "socketpair: %s", strerror(errno) );

	switch( pid = fork() ) {
	case -1:
		/* failure */
		panic( "fork: %s", strerror(errno) );
	case 0:
		/* child */
		close( 0 );
		dup( sd[0] );

		for(fd = getdtablesize() - 1; fd > 0; fd--)
			close(fd);
		execv(prog, argv);
		panic( "exec: %s - %s", prog, strerror(errno) );
	default:
		/* parent */
		close( sd[0] );
		async_devices[sd[1]] = dev;
		async_argp[sd[1]] = argp;
		async_pids[sd[1]] = pid;
		FD_SET(sd[1], &fd_devices);
	}

	if(fcntl(sd[1], F_SETFL, O_NONBLOCK))
		panic( "fcntl(O_NONBLOCK): %s", strerror(errno) );

	set_fd_async(sd[1]);

	return sd[1];
}

/*
 * on exit kill all registered children
 * (but be honest and send a SIGTERM)
 */
static void
kill_children(void)
{
	int i;

	signal(SIGCHLD, SIG_IGN);
	signal(SIGPIPE, SIG_IGN);
	signal(SIGIO, SIG_IGN);
	for(i = 0; i < FD_DEVICES; i++)
		if(async_pids[i])
			kill(async_pids[i], SIGTERM);
}


static void 
onsigio()
{
	fd_set	set = fd_devices;
	int	fd;
	static struct timeval tv;

	if(select(FD_DEVICES, &set, 0, 0, &tv) > 0)
		for(fd = 0; fd < FD_DEVICES; fd++)
			if(FD_ISSET(fd, &set)) {
				(*async_devices[fd]->ioops->async)
					(async_devices[fd], async_argp[fd]);
			}
}

static void
proc_info(IODev *dev)
{
	struct idata *id = (struct idata *)dev->data;

	printf("KDJ11A - internal registers\n");
	printf("LTC     %08o: %s   %s   interval:%d ms  %s%s\n", Reg_LTC, 
		id->trunning ? "running" : "stopped",
		id->timerreq ? "req. pending" : "no req.",
		id->tinterval,
		(proc.ltc & 0100) ? "int-enabled  " : "int-disabled  ",
		id->pwrfail ? "pwrfail-init " : "");
	printf("PSW     %08o: cur=%d prev=%d reg=%d pri=%d %c%c%c%c%c\n", Reg_PSW,
		proc.curmode, proc.prevmode, proc.regs, proc.pri,
		".T"[proc.t], ".N"[proc.n], ".Z"[proc.z], ".V"[proc.v], ".C"[proc.c]);
	printf("PIRQ    %08o: %c%c%c%c%c%c%c %o %o\n", Reg_PIRQ,
		".7"[(proc.pirq & 0100000) != 0],
		".6"[(proc.pirq & 0040000) != 0],
		".5"[(proc.pirq & 0020000) != 0],
		".4"[(proc.pirq & 0010000) != 0],
		".3"[(proc.pirq & 0004000) != 0],
		".2"[(proc.pirq & 0002000) != 0],
		".1"[(proc.pirq & 0001000) != 0],
		(proc.pirq >> 5) & 7,
		(proc.pirq >> 1) & 7);
	printf("CPUErr  %08o: %s%s%s%s%s%s\n", Reg_CPUErr,
		(proc.cpuerr & 0200) ? "HALT " : "",
		(proc.cpuerr & 0100) ? "ADDR " : "",
		(proc.cpuerr & 0040) ? "NXM " : "",
		(proc.cpuerr & 0020) ? "IOTO " : "",
		(proc.cpuerr & 0010) ? "YELL " : "",
		(proc.cpuerr & 0004) ? "RED " : "");
	printf("Maint   %08o: %06o\n", Reg_Maint, proc.maint);
}

int
register_timer(unsigned p, void (*f)(void *), void *a)
{
	if(ntimeouts == MAXTIMEOUT)
		panic("out of timeouts");
	timeouts[ntimeouts].func = f;
	timeouts[ntimeouts].time = timeouts[ntimeouts].curr = p / idata.tinterval;
	timeouts[ntimeouts].arg = a;
	return ntimeouts++;
}
