#include <stdio.h>
#include "schart.h"

#define CORNER          '+'
#define VERT_LINE       '|'
#define HORIZ_LINE      '-'

struct  page    *firstpg = NULL;        /* ptr to first page image */
struct  page    *freepgs = NULL;        /* free page image anchor */

struct  xref    xrefhdr;        /* xref header node */

char    lines[NLINES][VERT_SPACE][NODE_WIDTH + 1];
char    box[BOX_HEIGHT][NODE_WIDTH + 1];
char    space[BOX_HEIGHT][NODE_WIDTH + 1];

/* schart - make a structure chart from a tree
*  returns:  nothing.
*/
schart (p)
	struct  node    *p;     /* the root of the tree */
{
	struct  page    *getpage();
	struct  node    *savebro;

	initlines(lines, box, space);
	while (p != NULL) {
                firstpg = getpage();
                firstpg->p_prow = 1;
                firstpg->p_pcol = 1;
		savebro = p->n_bro;
                p->n_bro = NULL;
		chart(p, firstpg, 0, SON);
		prchart(firstpg);
                p->n_bro = savebro;
		p = savebro;
	}
        prxref();
        fclose(stdout);
}


/* chart - create the chart
*  Preforms a pre-order traversal of the tree.
*  As each node is encountered, it is placed in a page structure
*  for later output on printer pages.
*  returns:  nothing.
*/
chart(p, pg, lev, relation)
	struct  node    *p;     /* current node */
	struct  page    *pg;    /* current page image */
	int     lev;            /* current level of the tree */
	char    relation;       /* relationship with previous node */
{
	struct  page    *placenode();

	pg = placenode(p, pg, lev, relation);
	lev++;
	if (p->n_son)
		chart(p->n_son, pg, lev, SON);
	if (p->n_bro) {
		lev--;
		chart(p->n_bro, pg, lev, BRO);
		lev++;
	}
	lev--;
}

/* placenode - place a node in a page structure
*  Check if the node moved off the bottom or off the right edge
*  of the current page image.  If so, change to a new page image.
*  If the previous node was a brother, change its connecting line.
*  If the previous node was a parent, check if the display block
*  for the current node is occupied.  If it is occupied, then adjust
*  it to the right.  Finally, place the current node.
*  returns:  the page image the current node was placed in.
*/
struct  page    *
placenode(p, pg, lev, relation)
	struct  node    *p;     /* current node */
	struct  page    *pg;    /* current page image */
	int     lev;            /* current level of the tree */
	char    relation;       /* relationship with previous node */
{
	int     row;            /* logical node row within page image */
	int     col;            /* logical node col within page image */
	struct  display *disp;  /* display block for this node */
	struct  page    *getrtpg();
	struct  page    *getdwnpg();

	row = lev % NROW;
	if (relation == BRO) {
		while (pg->p_col[row] >= NCOL)
		        pg = getrtpg(pg);

		col = pg->p_col[row];
		disp = (col > 0)? &pg->p_disp[row][col - 1]:
		                  &pg->p_left->p_disp[row][NCOL - 1];
		disp->d_line = (disp->d_line == SONONLY)?
			FIRSTSON:
			MIDBRO;
	}

	else {
	        if (row == 0 && lev != 0)
		        pg = getdwnpg(pg);

		col = (row > 0)?  pg->p_col[row - 1] - 1:
                      (lev != 0)? pg->p_up->p_col[NROW - 1] - 1:
				  0;
		if (col < pg->p_col[row])
			col = adjust(&pg, row, col);
	}

	disp = &pg->p_disp[row][col];
	disp->d_node = p;
	disp->d_line = (relation == SON) ? SONONLY:
					   LASTBRO;
	pg->p_col[row] = col + 1;
        xref(p, pg, row, col);
	return(pg);
}

/* adjust - adjust a node and its direct ancestors to the right
*  Find the first empty display block that is to the right of
*  and on the same row as the one in conflict.
*  Then, adjust all direct ancestors over to the new column.
*  returns:  logical node column where the current node can be placed.
*            also returns a pointer to the page image for the current
*            node.
*/
adjust(page, row, col)
	struct  page    **page; /* page where we have conflict */
	int     row;            /* row of conflict */
	int     col;            /* column of conflict */
{
	struct  page    *pg;    /* current page image */
	struct  page    *npg;   /* new page image after adjustment */
	struct  page    *tpg;   /* temp page image ptr */
	struct  page    *getrtpg();
	struct  display *disp;  /* display block in conflict */
	struct  display *ndisp; /* new display block after adjustment */
	int     ncol;           /* column of new display block */
	int     c;              /* temp column pointer */

	pg = npg = *page;
	while (npg->p_col[row] >= NCOL)
		npg = getrtpg(npg);
	*page = npg;
	ncol = npg->p_col[row];
	do {
		if (--row < 0) {
			row = NROW - 1;
			pg = pg->p_up;
			tpg = pg;
			while (npg->p_up == NULL)
				tpg = getrtpg(tpg);
			npg = npg->p_up;
		}
		disp = &pg->p_disp[row][col];
		ndisp = &npg->p_disp[row][ncol];
		ndisp->d_node = disp->d_node;
		ndisp->d_line = disp->d_line;
		disp->d_node = NULL;
		disp->d_line = NOLINE;
                for (tpg = pg; tpg != npg; tpg = tpg->p_right)
			tpg->p_col[row] = NCOL;
		npg->p_col[row] = ncol + 1;
	} while (ndisp->d_line != LASTBRO);
/*
*  Stretch the brother line to the "lastbro" node that was just adjusted
*/
        if (pg == npg)
                for (c = col; c < ncol; c++)
                        pg->p_disp[row][c].d_line = BROONLY;
        else {
                for (c = col; c < NCOL; c++)
                        pg->p_disp[row][c].d_line = BROONLY;
                for (tpg = pg->p_right; tpg != npg; tpg = tpg->p_right)
                        for (c = 0; c < NCOL; c++)
                                tpg->p_disp[row][c].d_line = BROONLY;
                for (c = 0; c < ncol; c++)
                        npg->p_disp[row][c].d_line = BROONLY;
	}
	return(ncol);
}

/* getrtpg - get right page
*  If a page to the right already exists, return a pointer to it.
*  Otherwise, get another page and link it to the surrounding pages.
*  returns: a pointer to the page to the right
*/
struct page *
getrtpg(pg)
	struct  page    *pg;    /* current page */
{
	struct  page    *tpg;   /* temp page pointer */
	struct  page    *getpage();

	if (pg->p_right != NULL)
		return(pg->p_right);
	tpg = getpage();
	tpg->p_prow = pg->p_prow;
	tpg->p_pcol = pg->p_pcol + 1;
	pg->p_right = tpg;
	tpg->p_left = pg;
	if (pg->p_down && (pg = pg->p_down->p_right)) {
		pg->p_up = tpg;
		tpg->p_down = pg;
	}
	return(tpg);
}

/* getdwnpg - get down page
*  If a page on the downside already exists, return a pointer to it.
*  Otherwise, get another page and link it to the surrounding pages.
*  returns:  a pointer to the page on the downside.
*/
struct page *
getdwnpg(pg)
	struct  page    *pg;    /* current page */
{
	struct  page    *tpg;   /* temp page pointer */
	struct  page    *getpage();

	if (pg->p_down != NULL)
		return(pg->p_down);
	tpg = getpage();
	tpg->p_prow = pg->p_prow + 1;
	tpg->p_pcol = pg->p_pcol;
	pg->p_down = tpg;
	tpg->p_up = pg;
	if (pg->p_left && (pg = pg->p_left->p_down)) {
		pg->p_right = tpg;
		tpg->p_left = pg;
	}
	return(tpg);
}


/* getpage - get a page image
*  If there is a page image on the free page image list use it.
*  Otherwise, get storage from the system.
*  In any case, initialize the page.
*  returns:  a pointer to the page image.
*/
struct page *
getpage()
{
	struct  page    *pg;    /* the page image */
	int     row;            /* row index */
	int     col;            /* column index */
	char    *malloc();

	if (freepgs != NULL) {
		pg = freepgs;
		freepgs = freepgs->p_right;
	}
	else
		pg = (struct page *) malloc(sizeof(*pg));
/*
*  Initialize the page
*/
	pg->p_right = NULL;
	pg->p_left  = NULL;
	pg->p_up    = NULL;
	pg->p_down  = NULL;
	for (row = 0; row < NROW; row++) {
		for (col = 0; col < NCOL; col++) {
			pg->p_disp[row][col].d_node = NULL;
			pg->p_disp[row][col].d_line = NOLINE;
		}
		pg->p_col[row] = 0;
	}
	return(pg);
}

/* prchart - print the chart
*  This is a pre-order tree traversal.  However, the page images are
*  not exactly linked together in a tree form.  As each page image
*  is encountered, it is printed.  Then, all of its brother and son
*  images are printed.  Lastly, the page image is freed.  Since the
*  page images are connected in a matrix form instead of a binary tree,
*  the page image is removed from the matrix structure before being
*  freed.
*/
prchart(pg)
	struct  page    *pg;    /* the first page image */
{
        prpage(pg);
        if (pg->p_right)
                prchart(pg->p_right);
        if (pg->p_down)
                prchart(pg->p_down);
	if (pg->p_left)
		pg->p_left->p_right = NULL;
	if (pg->p_up)
		pg->p_up->p_down = NULL;
        pg->p_right = freepgs;
        freepgs = pg;
}

/* prpage - print a page image
*  This is a brute force routine that contains lots of ugly code.
*  Page images are printed one logical node row at a time.
*  The output is pieced together literally out of small building
*  blocks.  The blocks are the various types of connecting lines,
*  node boxes, and just plain fillers.
*/
prpage(pg)
	struct  page    *pg;    /* page to print */
{
	int     row;            /* row index */
	int     orow;           /* output row index */
	int     col;            /* column index */
	int     ecol;           /* end column */
	int     coloff;         /* column offset within box */
	int     len;            /* length of node's name */
	int     i;
	char    outrow[NODE_HEIGHT][PAGE_WIDTH + 1]; /* logical node row buffer */
	char    *c;             /* char ptr */
	char    *b;             /* ptr within box */
	struct  display *disp;  /* display block */

	for (row = 0; row < NROW; row++) {
		for (orow = 0; orow < NODE_HEIGHT; orow++)
			outrow[orow][0] = EOS;
		ecol = pg->p_col[row];
		for (col = 0; col < ecol; col++) {
			disp = &pg->p_disp[row][col];
			for (orow = 0; orow < VERT_SPACE; orow++)
				strcat(outrow[orow], lines[disp->d_line][orow]);

			if (disp->d_line > BROONLY) {
				c = &box[1][(BOX_WIDTH - FNLEN)/2];
				for (i = 0; i <= FNLEN; i++)
					*c++ = BLANK;
				len = strlen(disp->d_node->n_name);
				coloff = (BOX_WIDTH - FNLEN)/2 + (FNLEN - len)/2;
				b = &box[1][coloff];
				c = disp->d_node->n_name;
				while (*c)
					*b++ = *c++;
				if (disp->d_node->n_comp == FALSE)
			                *b = ASTERISK;
				c = box[0];
			}
			else
				c = space[0];

			for (orow = VERT_SPACE; orow < NODE_HEIGHT; orow++) {
				strcat(outrow[orow], c);
				c = c + NODE_WIDTH + 1;
			}
		}
/*
*  Put in page number and output the logical node row
*/
		if (row == 0) {
			i = col*NODE_WIDTH;
			c = &outrow[0][i];
			for (; i < PAGE_WIDTH + 1; i++)
				*c++ = BLANK;
			sprintf(&outrow[0][PAGE_WIDTH - 14], "Page %d.%d",
				pg->p_prow, pg->p_pcol);
		}
                for (orow = 0; orow < NODE_HEIGHT; orow++)
                        printf("%s\n", outrow[orow]);
	}
}


initlines(lines, box, space)
	char    lines[][VERT_SPACE][NODE_WIDTH + 1];
	char    box[BOX_HEIGHT][NODE_WIDTH + 1];
	char    space[BOX_HEIGHT][NODE_WIDTH + 1];
{
	int     row;
	int     col;
	int     i;

	for (i = 0; i < NLINES; i++) {
		for (row = 0; row < VERT_SPACE; row++) {
			for (col = 0; col <= NODE_WIDTH; col++)
				lines[i][row][col] = BLANK;
			lines[i][row][NODE_WIDTH] = EOS;
		}
		switch(i) {
case NOLINE:            break;

case FIRSTSON:
case SONONLY:           col = BOX_WIDTH/2;
			for (row = 0; row < VERT_SPACE; row++)
				lines[i][row][col] = VERT_LINE;

			if (i == FIRSTSON) {
				row = VERT_SPACE/2;
				lines[i][row][col] = CORNER;
				while (++col < NODE_WIDTH)
					lines[i][row][col] = HORIZ_LINE;
			}
		        break;
case MIDBRO:
case BROONLY:           row = VERT_SPACE/2;
			for (col = 0; col < NODE_WIDTH; col++)
				lines[i][row][col] = HORIZ_LINE;
			if (i == MIDBRO) {
				col = BOX_WIDTH/2;
				lines[i][row][col] = CORNER;
				while (++row < VERT_SPACE)
					lines[i][row][col] = VERT_LINE;
			}
		        break;

case LASTBRO:           row = VERT_SPACE/2;
			for (col = 0; col < BOX_WIDTH/2; col++)
				lines[i][row][col] = HORIZ_LINE;
			lines[i][row][col] = CORNER;
			while (++row < VERT_SPACE)
				lines[i][row][col] = VERT_LINE;
		        break;
		}
	}

	for (row = 0; row < BOX_HEIGHT; row++) {
		for (col = 0; col < NODE_WIDTH; col++) {
			box[row][col] = BLANK;
			space[row][col] = BLANK;
		}
		box[row][NODE_WIDTH] = EOS;
		space[row][NODE_WIDTH] = EOS;
	}
	for (col = 1; col < BOX_WIDTH - 1; col++) {
		box[0][col] = HORIZ_LINE;
		box[BOX_HEIGHT - 1][col] = HORIZ_LINE;
	}
	for (row = 1; row < BOX_HEIGHT - 1; row++) {
		box[row][0] = VERT_LINE;
		box[row][BOX_WIDTH - 1] = VERT_LINE;
	}
	box[0][0] = CORNER;
	box[0][BOX_WIDTH - 1] = CORNER;
	box[BOX_HEIGHT - 1][0] = CORNER;
	box[BOX_HEIGHT - 1][BOX_WIDTH - 1] = CORNER;
}


/* xref - put a node in the cross reference map.
*  Xref nodes are kept in a sorted linked list.  Duplicate nodes are
*  kept in a linked list anchored by the first node.
*  Returns:  nothing.
*/
xref(p, pg, row, col)
        struct  node    *p;     /* current node */
        struct  page    *pg;    /* current page image */
	int     row;            /* logical node row on page image */
	int     col;            /* logical node column on page image */
{
        struct  xref    *x;     /* ptr to existing xref nodes */
        struct  xref    *y;     /* ptr to the new xref node */
	struct  xref    *getxref();
	int     cmp;            /* return code from strcmp() */

	y = getxref(p, pg, row, col);
	for (x = &xrefhdr; x->x_next; x = x->x_next) {
		cmp = strcmp(p->n_name, x->x_next->x_name);
		if (cmp < 0) {
			y->x_next = x->x_next;
			x->x_next = y;
			return;
		}
		else if (cmp == 0) {
			for (x = x->x_next; x->x_same; x = x->x_same)
				;
			x->x_same = y;
			return;
		}
	}
	x->x_next = y;
}


/* getxref - get and initialize an xref node.
*  Returns:  a pointer to the xref node.
*/
struct xref *
getxref(p, pg, row, col)
        struct  node    *p;     /* current node */
        struct  page    *pg;    /* current page image */
	int     row;            /* logical node row on page image */
	int     col;            /* logical node column on page image */
{
	static  struct  xref    *freexref = NULL;
	static  struct  xref    *lastxref = NULL;
	struct  xref    *x;     /* new xref node */
	char    *malloc();

        if (freexref >= lastxref) {
                freexref = (struct xref *) malloc(100*sizeof(*x));
		if ((int) freexref == -1) {
			fprintf(stderr, "Out of storage\n");
			exit(1);
		}
                lastxref = freexref + 100;
        }
        x = freexref++;
	x->x_name = p->n_name;
	x->x_tree = firstpg->p_disp[0][0].d_node->n_name;
        x->x_prow = pg->p_prow;
        x->x_pcol = pg->p_pcol;
        x->x_nrow = row + 1;
        x->x_ncol = col + 1;
        x->x_same = NULL;
        x->x_next = NULL;
        return(x);
}


/* Print the cross reference map.
*  Returns:  nothing.
*/
prxref()
{
        struct  xref    *x;     /* ptr down x_next chain */
        struct  xref    *y;     /* ptr down x_same chain */

        printf("\nfunction   tree       page   node   node\n");
        printf(  "name       name        no    row    col\n\n");

        for (x = xrefhdr.x_next; x; x = x->x_next) {
                printf("%-10s %-8s %2d.%-2d %5.3d %6.3d\n",
			x->x_name, x->x_tree, x->x_prow, x->x_pcol,
			x->x_nrow, x->x_ncol);
                for (y = x->x_same; y; y = y->x_same)
                        printf("%\t   %-8s %2d.%-2d %5.3d %6.3d\n",
                        y->x_tree, y->x_prow, y->x_pcol,
			y->x_nrow, y->x_ncol);
        }
}
