/***************************************************************************
                    kommanderwidget.h - Text widget core functionality 
                             -------------------
    copyright          : (C) 2002-2003 Marc Britton <consume@optusnet.com.au>
                         (C) 2004      Michal Rudolf <mrudolf@kdewebdwev.org>
    
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 
#ifndef _HAVE_KOMMANDERWIDGET_H_
#define _HAVE_KOMMANDERWIDGET_H_

/* KDE INCLUDES */
#include <tdeprocess.h>

#include "kommander_export.h"
/* QT INCLUDES */
#include <tqmap.h>
#include <tqobject.h>
#include <tqpair.h>
#include <tqstring.h>
#include <tqstringlist.h>

class ParserData;

class KOMMANDER_EXPORT KommanderWidget
{
  friend class MyProcess;
public:
  KommanderWidget(TQObject *);
  virtual ~KommanderWidget();

  //List of available states. Most widgets have only one state, but f. e. radiobutton has both 
  // 'checked' and 'unchecked'
  virtual TQStringList states() const;
  virtual TQStringList displayStates() const;
  virtual TQString currentState() const = 0;

  virtual bool isKommanderWidget() const = 0;

  // Associated script
  virtual void setAssociatedText(const TQStringList& a_associations);
  virtual TQStringList associatedText() const;
  virtual bool hasAssociatedText();

  // Execute default script, expanding all @macros.
  virtual TQString evalAssociatedText();
  // Execute given script, expanding all @macros.
  virtual TQString evalAssociatedText(const TQString&);
  // Evaluate given Kommander function using given args.
  virtual TQString evalFunction(const TQString& function, const TQStringList& args);
  // Parse and evaluate function for given widget, converting it to appropriate DCOP call.
  virtual TQString evalWidgetFunction(const TQString& identifier, const TQString& s, int& pos);
  // Evaluate given array function using given args.
  virtual TQString evalArrayFunction(const TQString&, const TQStringList&);
  // Parse and evaluate given execBegin..execEnd block.
  virtual TQString evalExecBlock(const TQStringList&, const TQString& s, int& pos);
  // Parse and evaluate given forEach..end block.
  virtual TQString evalForEachBlock(const TQStringList&, const TQString& s, int& pos);
  // Parse and evaluate given for..end block.
  virtual TQString evalForBlock(const TQStringList&, const TQString& s, int& pos);
  // Parse and evaluate given switch..case..end block.
  virtual TQString evalSwitchBlock(const TQStringList&, const TQString& s, int& pos);
  // Parse and evaluate given if..endif block.
  virtual TQString evalIfBlock(const TQStringList&, const TQString& s, int& pos);
  // Population text. It will become widgetText after populate() is called
  virtual TQString populationText() const;
  virtual void setPopulationText(const TQString&);
  virtual void populate() = 0;

  // Handles all widget-specific DCOP calls 
  virtual TQString handleDCOP(int function, const TQStringList& args = TQStringList());
  // Checks if appropriate function is supported by widget. By default all functions
  // are reported as supported: use this to allow recognizing incorrect function calls.
  virtual bool isFunctionSupported(int function);
  // Checks if the function is common widget function (i. e. supported by all widgets)
  virtual bool isCommonFunction(int function);
  // Checks if the string is a valid widget name)
  virtual bool isWidget(const TQString& a_name) const;
  // Returns widget from name 
  virtual KommanderWidget* widgetByName(const TQString& a_name) const;
  // Returns current widget name;
  virtual TQString widgetName() const;
  // Returns filename associated with the dialog
  virtual TQString fileName();

  TQObject* object() { return m_thisObject;}


  // Recognizes editor vs executor mode
  static bool inEditor;
  // Prints errors in message boxes, not in stderr
  static bool showErrors;
  // Default parser
  static bool useInternalParser;
  // Return global variable value
  TQString global(const TQString& variableName);
  // Set global variable value
  void setGlobal(const TQString& variableName, const TQString& value);

protected:
  virtual void setStates(const TQStringList& a_states);
  virtual void setDisplayStates(const TQStringList& a_displayStates);

  // Execute DCOP query and return its result or null on failure
  // Only TQString and int are now handled
  TQString DCOPQuery(const TQStringList& args);
  TQString localDCOPQuery(const TQString function, const TQStringList& args = TQStringList());
  TQString localDCOPQuery(const TQString function, const TQString& arg1, 
     const TQString& arg2, const TQString& arg3 = TQString(),
     const TQString& arg4 = TQString());
  // Execute given command, return its result
  TQString execCommand(const TQString& a_command, const TQString& a_shell = TQString()) const;
  // Find and run dialog (with optional parameters)
  TQString runDialog(const TQString& a_dialog, const TQString& a_params = TQString());
  // Display error message a_error; display current class name if no other is given
  void printError(const TQString& a_error) const;
  // Auxiliary functions for parser
  // Find matching brackets starting from current position
  TQString parseBrackets(const TQString& s, int& from, bool& ok) const;
  // Return identifier: the longest string of letters and numbers starting from i
  TQString parseIdentifier(const TQString& s, int& from) const;
  // Parse arguments for given function. Returns list of arguments without quotations
  TQStringList parseArgs(const TQString& s, bool &ok);
  // Remove quotes from given identifier
  TQString parseQuotes(const TQString& s) const;
  // Parse function
  TQStringList parseFunction(const TQString& group, const TQString& function,
    const TQString& s, int& from, bool& ok);
  // Detect and return block boundary
  int parseBlockBoundary(const TQString& s, int from, const TQStringList& args) const;

  // Parse given identifier as widget name
  KommanderWidget* parseWidget(const TQString& name) const;
  // Return parent dialog of this widget
  TQWidget* parentDialog() const;
  TQString substituteVariable(TQString text, TQString variable, TQString value) const;

  ParserData* internalParserData() const;

  TQObject *m_thisObject;
  TQStringList m_states;
  TQStringList m_displayStates;
  TQStringList m_associatedText;
  TQString m_populationText;

  // Internal parser data
  static ParserData* m_parserData;
};


#define ESCCHAR '@'

#endif
