/*
   Copyright (C) 2002 by Roberto Raggi <roberto@kdevelop.org>
   Copyright (C) 2003 Oliver Kellogg <okellogg@users.sourceforge.net>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   version 2, License as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "problemreporter.h"
#include "adasupportpart.h"
#include "kdevpartcontroller.h"
#include "kdevmainwindow.h"
#include "configproblemreporter.h"
#include "backgroundparser.h"

#include <tqfileinfo.h>

#include <tdeversion.h>
#include <tdeparts/part.h>
#include <tdetexteditor/editinterface.h>
#include <tdetexteditor/document.h>
#include <tdetexteditor/markinterface.h>

#include <tdetexteditor/markinterfaceextension.h>

#include <kdebug.h>
#include <tdelocale.h>
#include <kstatusbar.h>
#include <kurl.h>
#include <tdeapplication.h>
#include <kiconloader.h>

#include <tdeconfig.h>

#include <tqtimer.h>
#include <tqregexp.h>
#include <tqvbox.h>
#include <tqwhatsthis.h>
#include <kdialogbase.h>


class ProblemItem: public TQListViewItem{
public:
	ProblemItem( TQListView* parent, const TQString& level, const TQString& problem,
				 const TQString& file, const TQString& line, const TQString& column  )
		: TQListViewItem( parent, level, problem, file, line, column ) {}

	ProblemItem( TQListViewItem* parent, const TQString& level, const TQString& problem,
				 const TQString& file, const TQString& line, const TQString& column  )
		: TQListViewItem( parent, level, problem, file, line, column ) {}

	int compare( TQListViewItem* item, int column, bool ascending ) const {
		if( column == 3 || column == 4 ){
			int a = text( column ).toInt();
			int b = item->text( column ).toInt();
			if( a == b )
				return 0;
			return( a > b ? -1 : 1 );
		}
		return TQListViewItem::compare( item, column, ascending );
	}

};

ProblemReporter::ProblemReporter( AdaSupportPart* part, TQWidget* parent, const char* name )
    : TQListView( parent, name ),
      m_adaSupport( part ),
      m_editor( 0 ),
      m_document( 0 ),
	  m_markIface( 0 ),
      m_bgParser( 0 )
{
    TQWhatsThis::add(this, i18n("<b>Problem reporter</b><p>This window shows errors reported by a language parser."));

    addColumn( i18n("Level") );
    addColumn( i18n("Problem") );
    addColumn( i18n("File") );
    addColumn( i18n("Line") );
    //addColumn( i18n("Column") );
    setAllColumnsShowFocus( true );

    m_timer = new TQTimer( this );

    connect( part->partController(), TQ_SIGNAL(activePartChanged(KParts::Part*)),
             this, TQ_SLOT(slotActivePartChanged(KParts::Part*)) );
    connect( part->partController(), TQ_SIGNAL(partAdded(KParts::Part*)),
             this, TQ_SLOT(slotPartAdded(KParts::Part*)) );
    connect( part->partController(), TQ_SIGNAL(partRemoved(KParts::Part*)),
             this, TQ_SLOT(slotPartRemoved(KParts::Part*)) );

    connect( m_timer, TQ_SIGNAL(timeout()), this, TQ_SLOT(reparse()) );

    connect( this, TQ_SIGNAL(doubleClicked(TQListViewItem*)),
             this, TQ_SLOT(slotSelected(TQListViewItem*)) );
    connect( this, TQ_SIGNAL(returnPressed(TQListViewItem*)),
             this, TQ_SLOT(slotSelected(TQListViewItem*)) );

    configure();
}

ProblemReporter::~ProblemReporter()
{
    if( m_bgParser ) {
        m_bgParser->wait();
    }

    delete( m_bgParser );
    m_bgParser = 0;
}

void ProblemReporter::slotActivePartChanged( KParts::Part* part )
{
    if( !part )
        return;
    
    if( m_editor )
	reparse();

    m_document = dynamic_cast<KTextEditor::Document*>( part );
    if( m_document ){
        m_filename = m_document->url().path();
    }

    m_editor = dynamic_cast<KTextEditor::EditInterface*>( part );
    if( m_editor )
        connect( m_document, TQ_SIGNAL(textChanged()), this, TQ_SLOT(slotTextChanged()) );

	m_markIface = dynamic_cast<KTextEditor::MarkInterface*>( part );

	m_timer->changeInterval( m_delay );
}

void ProblemReporter::slotTextChanged()
{
    if( m_active )
        m_timer->changeInterval( m_delay );
}

void ProblemReporter::reparse()
{
    kdDebug() << "ProblemReporter::reparse()" << endl;

    if( !m_editor )
        return;

    m_timer->stop();

    if( m_bgParser ) {
        if( m_bgParser->running() ) {
            m_timer->changeInterval( m_delay );
            return;
        }

        delete( m_bgParser );
        m_bgParser = 0;
    }

    TQListViewItem* current = firstChild();
    while( current ){
        TQListViewItem* i = current;
        current = current->nextSibling();

        if( i->text(2) == m_filename )
            delete( i );
    }

	if( m_markIface ){
		TQPtrList<KTextEditor::Mark> marks = m_markIface->marks();
		TQPtrListIterator<KTextEditor::Mark> it( marks );
		while( it.current() ){
			m_markIface->removeMark( it.current()->line, KTextEditor::MarkInterface::markType07 );
			++it;
		}
	}

/* Temporarily deactivated (crashes)*/
    if (!m_adaSupport->fileExtensions ().contains (TQFileInfo (m_filename).extension ()))
    {
        m_bgParser = new BackgroundParser( this, m_editor->text(), m_filename );
	m_bgParser->start();
    }
 /**/
}

void ProblemReporter::slotSelected( TQListViewItem* item )
{
    KURL url( item->text(2) );
    int line = item->text( 3 ).toInt();
    // int column = item->text( 4 ).toInt();
    m_adaSupport->partController()->editDocument( url, line-1 );
}

void ProblemReporter::reportError( TQString message,
                                   TQString filename,
                                   int line, int column )
{
	if( m_markIface ){
		m_markIface->addMark( line-1, KTextEditor::MarkInterface::markType07 );
	}

    new ProblemItem( this,
                       "error",
                       message.replace( TQRegExp("\n"), "" ),
                       filename,
                       TQString::number( line ),
                       TQString::number( column ) );
}

void ProblemReporter::reportWarning( TQString message,
                                     TQString filename,
                                     int line, int column )
{
    new ProblemItem( this,
                       "warning",
                       message.replace( TQRegExp("\n"), "" ),
                       filename,
                       TQString::number( line ),
                       TQString::number( column ) );
}

void ProblemReporter::reportMessage( TQString message,
                                     TQString filename,
                                     int line, int column )
{
    new TQListViewItem( this,
                       "message",
                       message.replace( TQRegExp("\n"), "" ),
                       filename,
                       TQString::number( line ),
                       TQString::number( column ) );
}

void ProblemReporter::configure()
{
    kdDebug() << "ProblemReporter::configure()" << endl;
    TDEConfig* config = tdeApp->config();
    config->setGroup( "General Options" );
    m_active = config->readBoolEntry( "EnableAdaBgParser", true );
    m_delay = config->readNumEntry( "BgParserDelay", 500 );
}

void ProblemReporter::configWidget( KDialogBase* dlg )
{
    kdDebug() << "ProblemReporter::configWidget()" << endl;
    TQVBox *vbox = dlg->addVBoxPage(i18n("Ada Parsing"), i18n("Ada Parsing"),  BarIcon( "text-x-src", TDEIcon::SizeMedium ));
    ConfigureProblemReporter* w = new ConfigureProblemReporter( vbox );
    connect(dlg, TQ_SIGNAL(okClicked()), w, TQ_SLOT(accept()));
    connect(dlg, TQ_SIGNAL(okClicked()), this, TQ_SLOT(configure()));
}

void ProblemReporter::slotPartAdded( KParts::Part* part )
{
	KTextEditor::MarkInterfaceExtension* iface = dynamic_cast<KTextEditor::MarkInterfaceExtension*>( part );
	
	if( !iface )
		return;
		
	iface->setPixmap( KTextEditor::MarkInterface::markType07, SmallIcon("process-stop") );
}

void ProblemReporter::slotPartRemoved( KParts::Part* part )
{
    kdDebug() << "ProblemReporter::slotPartRemoved()" << endl;
    if( part == m_document ){
        m_document = 0;
        m_editor = 0;
        m_timer->stop();
    }
}

#include "problemreporter.moc"
