/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.kipi-plugins.org
 *
 * Date        : 2006-10-12
 * Description : IPTC caption settings page.
 *
 * Copyright (C) 2006-2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// QT includes.

#include <tqlayout.h>
#include <tqhgroupbox.h>
#include <tqgroupbox.h>
#include <tqlabel.h>
#include <tqwhatsthis.h>
#include <tqvalidator.h>
#include <tqcheckbox.h>

// KDE includes.

#include <tdelocale.h>
#include <kdialog.h>
#include <klineedit.h>
#include <ktextedit.h>
#include <tdeapplication.h>
#include <tdeaboutdata.h>
#include <kseparator.h>
#include <kactivelabel.h>

// LibKExiv2 includes. 

#include <libkexiv2/kexiv2.h>

// Local includes.

#include "iptccaption.h"
#include "iptccaption.moc"

namespace KIPIMetadataEditPlugin
{

class IPTCCaptionPriv
{
public:

    IPTCCaptionPriv()
    {
        captionEdit             = 0;
        writerEdit              = 0;
        headlineEdit            = 0;
        specialInstructionEdit  = 0;
        captionCheck            = 0;
        specialInstructionCheck = 0;
        writerCheck             = 0;
        headlineCheck           = 0;
        syncJFIFCommentCheck    = 0;
        syncHOSTCommentCheck    = 0;
        syncEXIFCommentCheck    = 0;
    }

    TQCheckBox *captionCheck;
    TQCheckBox *specialInstructionCheck;
    TQCheckBox *writerCheck;
    TQCheckBox *headlineCheck;
    TQCheckBox *syncJFIFCommentCheck;
    TQCheckBox *syncHOSTCommentCheck;
    TQCheckBox *syncEXIFCommentCheck;

    KTextEdit *captionEdit;
    KTextEdit *specialInstructionEdit;

    KLineEdit *writerEdit;
    KLineEdit *headlineEdit;
};

IPTCCaption::IPTCCaption(TQWidget* parent)
           : TQWidget(parent)
{
    d = new IPTCCaptionPriv;
    TQVBoxLayout *vlay = new TQVBoxLayout( parent, 0, KDialog::spacingHint() );

    // IPTC only accept printable Ascii char.
    TQRegExp asciiRx("[\x20-\x7F]+$");
    TQValidator *asciiValidator = new TQRegExpValidator(asciiRx, this);
    
    // --------------------------------------------------------

    d->captionCheck         = new TQCheckBox(i18n("Caption:"), parent);
    d->captionEdit          = new KTextEdit(parent);
    d->syncJFIFCommentCheck = new TQCheckBox(i18n("Sync JFIF Comment section"), parent);
    d->syncHOSTCommentCheck = new TQCheckBox(i18n("Sync caption entered through %1")
                                            .arg(tdeApp->aboutData()->appName()), 
                                            parent);
    d->syncEXIFCommentCheck = new TQCheckBox(i18n("Sync EXIF Comment"), parent);
    KSeparator *line        = new KSeparator(Horizontal, parent);

/*    d->captionEdit->setValidator(asciiValidator);
    d->captionEdit->setMaxLength(2000);*/
    vlay->addWidget(d->captionCheck);
    vlay->addWidget(d->captionEdit);
    vlay->addWidget(d->syncJFIFCommentCheck);
    vlay->addWidget(d->syncHOSTCommentCheck);
    vlay->addWidget(d->syncEXIFCommentCheck);
    vlay->addWidget(line);
    TQWhatsThis::add(d->captionEdit, i18n("<p>Enter the content description. This field is limited "
                                         "to 2000 ASCII characters."));

    // --------------------------------------------------------

    d->writerCheck = new TQCheckBox(i18n("Caption Writer:"), parent);
    d->writerEdit  = new KLineEdit(parent);
    d->writerEdit->setValidator(asciiValidator);
    d->writerEdit->setMaxLength(32);
    vlay->addWidget(d->writerCheck);
    vlay->addWidget(d->writerEdit);
    TQWhatsThis::add(d->writerEdit, i18n("<p>Enter the name of the caption author. This field is limited "
                                        "to 32 ASCII characters."));
        
    // --------------------------------------------------------

    d->headlineCheck = new TQCheckBox(i18n("Headline:"), parent);
    d->headlineEdit  = new KLineEdit(parent);
    d->headlineEdit->setValidator(asciiValidator);
    d->headlineEdit->setMaxLength(256);
    vlay->addWidget(d->headlineCheck);
    vlay->addWidget(d->headlineEdit);
    TQWhatsThis::add(d->headlineEdit, i18n("<p>Enter here the content synopsis. This field is limited "
                                          "to 256 ASCII characters."));

    // --------------------------------------------------------

    d->specialInstructionCheck = new TQCheckBox(i18n("Special Instructions:"), parent);
    d->specialInstructionEdit  = new KTextEdit(parent);
/*    d->specialInstructionEdit->setValidator(asciiValidator);
    d->specialInstructionEdit->setMaxLength(256);*/
    vlay->addWidget(d->specialInstructionCheck);
    vlay->addWidget(d->specialInstructionEdit);
    TQWhatsThis::add(d->specialInstructionEdit, i18n("<p>Enter the editorial usage instructions. "
                                                    "This field is limited to 256 ASCII characters."));

    // --------------------------------------------------------

    KActiveLabel *note = new KActiveLabel(i18n("<b>Note: "
                 "<b><a href='http://en.wikipedia.org/wiki/IPTC'>IPTC</a></b> "
                 "text tags only support the printable "
                 "<b><a href='http://en.wikipedia.org/wiki/Ascii'>ASCII</a></b> "
                 "characters set and limit strings size. "
                 "Use contextual help for details.</b>"), parent);
    vlay->addWidget(note);
    vlay->addStretch();

    // --------------------------------------------------------
                                     
    connect(d->captionCheck, TQ_SIGNAL(toggled(bool)),
            d->captionEdit, TQ_SLOT(setEnabled(bool)));

    connect(d->captionCheck, TQ_SIGNAL(toggled(bool)),
            d->syncJFIFCommentCheck, TQ_SLOT(setEnabled(bool)));

    connect(d->captionCheck, TQ_SIGNAL(toggled(bool)),
            d->syncHOSTCommentCheck, TQ_SLOT(setEnabled(bool)));

    connect(d->captionCheck, TQ_SIGNAL(toggled(bool)),
            d->syncEXIFCommentCheck, TQ_SLOT(setEnabled(bool)));

    connect(d->writerCheck, TQ_SIGNAL(toggled(bool)),
            d->writerEdit, TQ_SLOT(setEnabled(bool)));

    connect(d->headlineCheck, TQ_SIGNAL(toggled(bool)),
            d->headlineEdit, TQ_SLOT(setEnabled(bool)));
    
    connect(d->specialInstructionCheck, TQ_SIGNAL(toggled(bool)),
            d->specialInstructionEdit, TQ_SLOT(setEnabled(bool)));

    // --------------------------------------------------------

    connect(d->captionCheck, TQ_SIGNAL(toggled(bool)),
            this, TQ_SIGNAL(signalModified()));

    connect(d->writerCheck, TQ_SIGNAL(toggled(bool)),
            this, TQ_SIGNAL(signalModified()));

    connect(d->headlineCheck, TQ_SIGNAL(toggled(bool)),
            this, TQ_SIGNAL(signalModified()));
    
    connect(d->specialInstructionCheck, TQ_SIGNAL(toggled(bool)),
            this, TQ_SIGNAL(signalModified()));

    // --------------------------------------------------------

    connect(d->captionEdit, TQ_SIGNAL(textChanged()),
            this, TQ_SIGNAL(signalModified()));
    
    connect(d->specialInstructionEdit, TQ_SIGNAL(textChanged()),
            this, TQ_SIGNAL(signalModified()));

    connect(d->writerEdit, TQ_SIGNAL(textChanged(const TQString &)),
            this, TQ_SIGNAL(signalModified()));

    connect(d->headlineEdit, TQ_SIGNAL(textChanged(const TQString &)),
            this, TQ_SIGNAL(signalModified()));
}

IPTCCaption::~IPTCCaption()
{
    delete d;
}

bool IPTCCaption::syncJFIFCommentIsChecked()
{
    return d->syncJFIFCommentCheck->isChecked();
}

bool IPTCCaption::syncHOSTCommentIsChecked()
{
    return d->syncHOSTCommentCheck->isChecked();
}

bool IPTCCaption::syncEXIFCommentIsChecked()
{
    return d->syncEXIFCommentCheck->isChecked();
}

TQString IPTCCaption::getIPTCCaption()
{
    return d->captionEdit->text();
}

void IPTCCaption::setCheckedSyncJFIFComment(bool c)
{
    d->syncJFIFCommentCheck->setChecked(c);
}

void IPTCCaption::setCheckedSyncHOSTComment(bool c)
{
    d->syncHOSTCommentCheck->setChecked(c);
}

void IPTCCaption::setCheckedSyncEXIFComment(bool c)
{
    d->syncEXIFCommentCheck->setChecked(c);
}

void IPTCCaption::readMetadata(TQByteArray& iptcData)
{
    blockSignals(true);
    KExiv2Iface::KExiv2 exiv2Iface;
    exiv2Iface.setIptc(iptcData);
    TQString data;

    d->captionEdit->clear();
    d->captionCheck->setChecked(false);
    data = exiv2Iface.getIptcTagString("Iptc.Application2.Caption", false);    
    if (!data.isNull())
    {
        d->captionEdit->setText(data);
        d->captionCheck->setChecked(true);
    }
    d->captionEdit->setEnabled(d->captionCheck->isChecked());
    d->syncJFIFCommentCheck->setEnabled(d->captionCheck->isChecked());
    d->syncHOSTCommentCheck->setEnabled(d->captionCheck->isChecked());
    d->syncEXIFCommentCheck->setEnabled(d->captionCheck->isChecked());

    d->writerEdit->clear();
    d->writerCheck->setChecked(false);
    data = exiv2Iface.getIptcTagString("Iptc.Application2.Writer", false);    
    if (!data.isNull())
    {
        d->writerEdit->setText(data);
        d->writerCheck->setChecked(true);
    }
    d->writerEdit->setEnabled(d->writerCheck->isChecked());

    d->headlineEdit->clear();
    d->headlineCheck->setChecked(false);
    data = exiv2Iface.getIptcTagString("Iptc.Application2.Headline", false);    
    if (!data.isNull())
    {
        d->headlineEdit->setText(data);
        d->headlineCheck->setChecked(true);
    }
    d->headlineEdit->setEnabled(d->headlineCheck->isChecked());

    d->specialInstructionEdit->clear();
    d->specialInstructionCheck->setChecked(false);
    data = exiv2Iface.getIptcTagString("Iptc.Application2.SpecialInstructions", false);    
    if (!data.isNull())
    {
        d->specialInstructionEdit->setText(data);
        d->specialInstructionCheck->setChecked(true);
    }
    d->specialInstructionEdit->setEnabled(d->specialInstructionCheck->isChecked());

    blockSignals(false);
}

void IPTCCaption::applyMetadata(TQByteArray& exifData, TQByteArray& iptcData)
{
    KExiv2Iface::KExiv2 exiv2Iface;
    exiv2Iface.setExif(exifData);
    exiv2Iface.setIptc(iptcData);

    if (d->captionCheck->isChecked())
    {
        exiv2Iface.setIptcTagString("Iptc.Application2.Caption", d->captionEdit->text());

        if (syncEXIFCommentIsChecked())
            exiv2Iface.setExifComment(d->captionEdit->text());

        if (syncJFIFCommentIsChecked())
            exiv2Iface.setComments(d->captionEdit->text().utf8());
    }
    else
        exiv2Iface.removeIptcTag("Iptc.Application2.Caption");

    if (d->writerCheck->isChecked())
        exiv2Iface.setIptcTagString("Iptc.Application2.Writer", d->writerEdit->text());
    else
        exiv2Iface.removeIptcTag("Iptc.Application2.Writer");

    if (d->headlineCheck->isChecked())
        exiv2Iface.setIptcTagString("Iptc.Application2.Headline", d->headlineEdit->text());
    else
        exiv2Iface.removeIptcTag("Iptc.Application2.Headline");

    if (d->specialInstructionCheck->isChecked())
        exiv2Iface.setIptcTagString("Iptc.Application2.SpecialInstructions", d->specialInstructionEdit->text());
    else
        exiv2Iface.removeIptcTag("Iptc.Application2.SpecialInstructions");

    exifData = exiv2Iface.getExif();
    iptcData = exiv2Iface.getIptc();
}

}  // namespace KIPIMetadataEditPlugin

