/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "prefsdlg.h"

#include "config.h"

#include "glade_support.h" /* for gettext */
#include "glade_interface.h"
#include "encoding.h"
#include "main.h" /* TODO: remove dependency. */
#include "bindprops.h"

typedef struct PreferencesDialogData_ {
    GtkWidget *encoding_none, *encoding_specified,
              *encoding_optionmenu;
    GtkWidget *unix_radiobutton, *windows_radiobutton;
} PreferencesDialogData;

static void preferences_dialog_start_encoding_changed(gpointer user_data, ...)
{
    const PreferencesDialogData *data = (PreferencesDialogData *)user_data;
    const gchar *encoding;
    gboolean specified;

    specified = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(data->encoding_specified));
    gtk_widget_set_sensitive(data->encoding_optionmenu, specified);
    if(specified) {
        gint enc_num;
        
        enc_num = gtk_option_menu_get_history(GTK_OPTION_MENU(data->encoding_optionmenu));
        encoding = encoding_names[enc_num];
    } else
        encoding = NULL;
    g_object_set(gtk_settings_get_default(),
                 N_("start-encoding"), encoding, NULL);
}

static void preferences_dialog_start_term_activate(GtkButton *button,
                                                   gpointer user_data)
{
    if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)) )
        g_object_set(gtk_settings_get_default(),
                     N_("start-encoding-term"), (gchar*)user_data, NULL);
}

static void preferences_dialog_update_start_encoding(
        const PreferencesDialogData *dlg_data)
{
    gboolean specified;
    gchar *encoding;
    GtkWidget *radio_item;

    g_object_get(G_OBJECT(gtk_settings_get_default()),
                 N_("start-encoding"), &encoding, NULL);
    radio_item = encoding ? dlg_data->encoding_specified :
                            dlg_data->encoding_none;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio_item), TRUE);
    gtk_widget_set_sensitive(dlg_data->encoding_optionmenu, encoding != NULL);
    if(encoding) {
        gint i;
        
        for(i=0; i<num_encodings; ++i)
            if( !strcmp(encoding, encoding_names[i]) ) {
                gtk_option_menu_set_history(
                        GTK_OPTION_MENU(dlg_data->encoding_optionmenu), i);
                break;
            }
    }
    g_free(encoding);
}

static void preferences_dialog_update_start_term(
        const PreferencesDialogData *dlg_data)
{
    gchar *term;
    GtkWidget *radio_item;

    g_object_get(G_OBJECT(gtk_settings_get_default()),
                 N_("start-encoding-term"), &term, NULL);
    radio_item = !strcmp(term, "0A") ? dlg_data->unix_radiobutton :
                            dlg_data->windows_radiobutton;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio_item), TRUE);
    g_free(term);
}

static void preferences_dialog_start_encoding_notify(GObject *settings,
                                                     GParamSpec *pspec,
                                                     gpointer user_data)
{
    preferences_dialog_update_start_encoding((PreferencesDialogData*)user_data);
}

static void preferences_dialog_start_term_notify(GObject *settings,
                                                 GParamSpec *pspec,
                                                 gpointer user_data)
{
    preferences_dialog_update_start_term((PreferencesDialogData*)user_data);
}

void call_preferences_dialog(GtkWidget *mainwin)
{
    const gchar *const ask_encoding_widgets[] = {
        "ask_encoding0", "ask_encoding1", "ask_encoding2"
    };
    PreferencesDialogData *dlg_data;
    GObject *settings;
    static GtkWidget *dlg = NULL;
    GtkWidget *wrap_text_check, *autoindent_check, *save_strip_space_check;
    GtkWidget *encoding_menu, *bom_check, *ask_encoding_radio;
    int i;
    guint ask_encoding_value;

    if(dlg) {
        gtk_window_present(GTK_WINDOW(dlg));
        return;
    }
    dlg = create_preferences_dialog();
    dlg_data = g_malloc(sizeof(PreferencesDialogData));
    g_object_set_data_full(G_OBJECT(dlg), N_("data"), dlg_data, g_free);
    gtk_window_set_transient_for(GTK_WINDOW(dlg), GTK_WINDOW(mainwin));
    my_disable_window_resize(GTK_WINDOW(dlg));

    wrap_text_check        = lookup_widget(dlg, N_("wrap_text_check"));
    autoindent_check       = lookup_widget(dlg, N_("autoindent_check"));
    save_strip_space_check = lookup_widget(dlg, N_("save_strip_space_check"));

    bom_check              = lookup_widget(dlg, N_("bom_checkbutton"));
    dlg_data->unix_radiobutton    = lookup_widget(dlg, N_("unix_radiobutton"));
    dlg_data->windows_radiobutton = lookup_widget(dlg, N_("windows_radiobutton"));
    dlg_data->encoding_none       = lookup_widget(dlg, N_("encoding_none"));
    dlg_data->encoding_specified  = lookup_widget(dlg, N_("encoding_specified"));
    dlg_data->encoding_optionmenu = lookup_widget(dlg, N_("encoding_optionmenu"));

    encoding_menu = gtk_menu_new();
    gtk_option_menu_set_menu(GTK_OPTION_MENU(dlg_data->encoding_optionmenu),
                             encoding_menu);
    for(i=0; i<num_encodings; ++i) {
        GtkWidget *item;
        
        item = gtk_menu_item_new_with_label(encoding_names[i]);
        gtk_widget_show(item);
        gtk_menu_shell_append(GTK_MENU_SHELL(encoding_menu), item);
    }        

    settings = G_OBJECT(gtk_settings_get_default());

    preferences_dialog_update_start_encoding(dlg_data);
    preferences_dialog_update_start_term    (dlg_data);
    g_object_get(G_OBJECT(gtk_settings_get_default()),
                 N_("always-ask-encoding"), &ask_encoding_value, NULL);
    ask_encoding_radio = lookup_widget(dlg, ask_encoding_widgets[ask_encoding_value]);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ask_encoding_radio), TRUE);    
    g_signal_connect(settings, N_("notify::start-encoding"),
                     G_CALLBACK(preferences_dialog_start_encoding_notify), dlg_data);
    g_signal_connect(settings, N_("notify::start-encoding-term"),
                     G_CALLBACK(preferences_dialog_start_term_notify), dlg_data);

    my_glib_bind_properties2(settings,                   N_("wrap-text"),
                             G_OBJECT(wrap_text_check),  N_("active"),
                             TRUE);
    my_glib_bind_properties2(settings,                   N_("autoindent"),
                             G_OBJECT(autoindent_check), N_("active"),
                             TRUE);
    my_glib_bind_properties2(settings,                   N_("strip-space-at-save"),
                             G_OBJECT(save_strip_space_check), N_("active"),
                             TRUE);
    my_glib_bind_properties2(settings,                     N_("start-encoding-bom"),
                             G_OBJECT(bom_check),          N_("active"),
                             TRUE);

    g_signal_connect_swapped(G_OBJECT(dlg_data->encoding_specified), N_("toggled"),
                             G_CALLBACK(preferences_dialog_start_encoding_changed), dlg_data);
    g_signal_connect_swapped(G_OBJECT(dlg_data->encoding_optionmenu), N_("changed"),
                             G_CALLBACK(preferences_dialog_start_encoding_changed), dlg_data);

    g_signal_connect_after(G_OBJECT(dlg_data->unix_radiobutton), N_("toggled"),
                           G_CALLBACK(preferences_dialog_start_term_activate), N_("0A"));
    g_signal_connect_after(G_OBJECT(dlg_data->windows_radiobutton), N_("toggled"),
                           G_CALLBACK(preferences_dialog_start_term_activate), N_("0D0A"));

    gtk_widget_show(GTK_WIDGET(dlg));
}
