/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#if defined(GDK_WINDOWING_X11) && CONF_ICONV_IS_SAFE
#  define X11_AND_ICONV
#endif

#include <stdio.h>
#include <gtk/gtk.h>
#ifdef X11_AND_ICONV
#  include <X11/Xlib.h>
#  include <X11/Xutil.h>
#  include <gdk/gdkx.h>
#endif /* X11_AND_ICONV */
#if 0/*defined(HAVE_GNOME)*/ /* gnome-history.h disappered in Gnome2. Where it is? */
#  include <gnome-history.h>
#endif
#include "glade_support.h"
#ifdef G_PLATFORM_WIN32
#  include <windows.h>
#endif
#ifdef GDK_WINDOWING_WIN32
#  include <gdk/gdkwin32.h>
#endif
#include <gdk/gdkkeysyms.h>
#include <glade_interface.h>

#include "mainwin.h"

#include "main.h"
#include "bindprops.h"
#include "encoding.h"
#include "prefs.h"
#include "dnd.h"

guint16 mainwin_windows_count = 0;

MainwinData *mainwin_get_data(GtkWidget *mainwin)
{
    return g_object_get_data(G_OBJECT(mainwin), N_("data"));
}

MainwinData *mainwin_get_data_by_widget(GtkWidget *widget)
{
    return mainwin_get_data(my_gtk_widget_get_toplevel(widget));
}

static void mainwin_set_title_label(MainwinData *win_data, const gchar *str)
{
    GtkRequisition label_requisition;
#if CONF_UNICODE_TITLE <= 1
    gboolean is_displayable = FALSE;
#elif CONF_UNICODE_TITLE >= 2
    gboolean is_displayable = TRUE;
#endif /* CONF_UNICODE_TITLE value */
    gchar *file_title, *full_title;
#if CONF_UNICODE_TITLE > 0 && CONF_UNICODE_TITLE < 3
    gboolean known_displayable, only_ascii = TRUE;
    const gchar *ptr;
#endif /* CONF_UNICODE_TITLE value */

    /* FIXME: (When label expands?) it sometimes jumps to the center of the previous position */
    /* during a little part of second before going to correct place (Linux GTK 2.0.6). */
    if(!str) str = _("(Untitled)");
    gtk_label_set_text(GTK_LABEL(win_data->title_label), str);
    gtk_widget_size_request(win_data->title_label, &label_requisition);
    gtk_layout_set_size(GTK_LAYOUT(win_data->title_label_box),
                        label_requisition.width, label_requisition.height);
    gtk_tooltips_set_tip(GTK_TOOLTIPS(file_tips), win_data->title_event_box, str, NULL);

#if CONF_UNICODE_TITLE > 0 && CONF_UNICODE_TITLE < 3
    for(ptr=str; *ptr; ++ptr)
        if((unsigned char)*ptr >= 128) {
            only_ascii = FALSE;
            break;
        }
    if(only_ascii)
        is_displayable = TRUE;
    else {
#  ifdef X11_AND_ICONV
        XTextProperty prop;

        if( GTK_WIDGET_REALIZED(win_data->mainwin)) {
            GdkWindow *window = GTK_WIDGET(win_data->mainwin)->window;

            if( XGetTextProperty(GDK_WINDOW_XDISPLAY(window),
                                 GDK_WINDOW_XID(window),
                                 &prop,
                                 gdk_x11_get_xatom_by_name(N_("WM_LOCALE_NAME"))) )
            {
                /* With iconv only check whether the conversion is possible. */
                /* Actual conversion is accomplished by X server. */
                GIConv ic;
                unsigned char *tmp, *window_enc, *outbuf;
                gsize bytes_read, bytes_written, str_len = strlen(str);

                tmp = strchr(prop.value, N_('.'));
                window_enc = tmp ? tmp+1 : prop.value;
                outbuf = g_convert(str, str_len,
                                   window_enc, N_("UTF-8"),
                                   &bytes_read, &bytes_written,
                                   NULL);
                is_displayable = outbuf && bytesread == str_len;
                g_free(outbuf);
                XFree(prop.value);
            }
        }
#  elsif defined(GDK_WINDOWING_WIN32)
        if(is_windows_nt())
            is_displayable = TRUE;
        else {
            gsize bytes_written;

            file_title = g_locale_from_utf8(str, -1, NULL, &bytes_written, NULL);
            is_displayable = file_title != NULL;
            g_free(file_title);
        }
#  endif /* windowing system */
    }
#endif /* CONF_UNICODE_TITLE value */
    /* TODO: display it partially recoded or transliterated if possible. */
    file_title = (char*)(is_displayable ? str : _("<undisplayable Unicode>"));
    full_title = g_strdup_printf(_("%s - %s"), file_title, g_get_prgname());
#ifdef GDK_WINDOWING_WIN32
    if(is_windows_nt()) {
        WCHAR *wide_char_title;

        wide_char_title = (WCHAR*)g_utf8_to_utf16(full_title, -1,
                                                  NULL, NULL,
                                                  NULL);
        if(GTK_WIDGET_REALIZED(win_data->mainwin)) /* FIXME: Here and in similar places. */
            SetWindowTextW(GDK_WINDOW_HWND(GTK_WIDGET(win_data->mainwin)->window), wide_char_title);
        g_free(wide_char_title);
    } else
        gtk_window_set_title(GTK_WINDOW(win_data->mainwin), full_title); /* Doesn't work right, seems a bug in GTK 2.2.1.1 */
#else
    gtk_window_set_title(GTK_WINDOW(win_data->mainwin), full_title);
#endif /* GDK_WINDOWING_WIN32 */
    g_free(full_title);
#ifdef GDK_WINDOWING_X11 /* On other windowing systems set_icon_name is bad. */
    if(GTK_WIDGET_REALIZED(win_data->mainwin))
        gdk_window_set_icon_name(GTK_WIDGET(win_data->mainwin)->window, file_title);
#endif /* GDK_WINDOWING_X11 */
}

static void calculate_line_height(MainwinData *win_data)
{
    GtkStyle *style;
    PangoFontMetrics *metrics;
    gint space;

    style = gtk_widget_get_style(win_data->thetext);
    metrics = pango_context_get_metrics(gtk_widget_get_pango_context(win_data->thetext),
                                        style->font_desc,
                                        NULL);
    /* TODO: Cache line height. */
    win_data->the_line_height = PANGO_PIXELS( pango_font_metrics_get_ascent (metrics) +
                                              pango_font_metrics_get_descent(metrics) );
    pango_font_metrics_unref(metrics);
    /*memset(&space, 0, sizeof(GValue));*/
    /*g_value_init(&space, G_TYPE_INT);*/
    g_object_get(G_OBJECT(win_data->thetext), N_("pixels-above-lines"), &space, NULL);
    win_data->the_line_height += space;
    g_object_get(G_OBJECT(win_data->thetext), N_("pixels-below-lines"), &space, NULL);
    win_data->the_line_height += space;
    /*g_value_unset(&space);*/
}

static gint on_key_press(GtkWidget *widget, GdkEventKey *event, gpointer user_data)
{
    switch(event->keyval) {
        case GDK_Return:
            if(preferences_get_autoindent()) {
                GtkTextBuffer *buffer;
                GtkTextMark *mark;
                GtkTextIter iter, start_iter;
                guint line;
                gchar *line_beg, *spaces_end, *str;
                int i;

                buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(widget));
                mark = gtk_text_buffer_get_insert(buffer);
                gtk_text_buffer_get_iter_at_mark(buffer, &iter, mark);
                line = gtk_text_iter_get_line(&iter);
                gtk_text_buffer_get_iter_at_line(buffer, &start_iter, line);
                line_beg = gtk_text_buffer_get_text(buffer, &start_iter, &iter, TRUE);
                spaces_end = line_beg;
                while(*spaces_end==N_(' ') || *spaces_end==N_('\t'))
                    ++spaces_end;
                str = (gchar*)g_malloc( 1+(spaces_end-line_beg)+1 );
                str[0] = N_('\n');
                for(i=0; i<spaces_end-line_beg; ++i)
                    str[i+1] = line_beg[i];
                str[spaces_end-line_beg+1] = N_('\0');
                gtk_text_buffer_insert_interactive_at_cursor(buffer, str, -1,
                        gtk_text_view_get_editable(GTK_TEXT_VIEW(widget)));
                g_free(str);
                g_free(line_beg);
                return TRUE;
            } else
                return FALSE;
        default:
            return FALSE;
    }
}

static void mainwin_set_text_wrap(MainwinData *win_data, gboolean wrap)
{
    gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(win_data->thetext),
                                wrap ? GTK_WRAP_WORD : GTK_WRAP_NONE);
    gtk_text_view_set_pixels_below_lines(GTK_TEXT_VIEW(win_data->thetext),
                                         wrap ? 5 : 0);
    calculate_line_height(win_data);
}

static void mainwin_text_wrap_notify(GObject *settings,
                                     GParamSpec *pspec,
                                     gpointer user_data)
{
    gboolean wrap;

    g_object_get(G_OBJECT(settings), N_("wrap-text"), &wrap, NULL);
    mainwin_set_text_wrap((MainwinData*)user_data, wrap);
}

void my_text_view_upper_or_lower_case(GtkTextBuffer *buffer, gboolean need_upper)
{
    GtkTextIter start, end;
    GtkTextMark *mark;
    gchar *str, *new_str;

    gtk_text_buffer_get_selection_bounds(buffer, &start, &end);
    str = gtk_text_buffer_get_text(buffer, &start, &end, TRUE);
    mark = gtk_text_buffer_create_mark(buffer, NULL, &start, FALSE);
    gtk_text_buffer_begin_user_action(buffer);
    gtk_text_buffer_delete(buffer, &start, &end);
    gtk_text_buffer_get_iter_at_mark(buffer, &start, mark);
    gtk_text_buffer_delete_mark(buffer, mark);
    new_str = need_upper ? g_utf8_strup  (str, -1) : g_utf8_strdown(str, -1);
    gtk_text_buffer_insert(buffer, &start, new_str, -1);
    gtk_text_buffer_end_user_action(buffer);
    g_free(str);
    g_free(new_str);
}

static void on_mark_set(GtkTextBuffer *textbuffer,
                        GtkTextIter *iter,
                        GtkTextMark *mark,
                        gpointer user_data)
{
    const MainwinData *win_data;
    gboolean non_empty;
    GtkTextIter iter1, iter2;
    gint line1, line2;
    gchar buf[9+1+9+1];

    win_data = (MainwinData*)user_data;
    non_empty = gtk_text_buffer_get_selection_bounds(textbuffer, &iter1, &iter2);

    gtk_widget_set_sensitive(GTK_WIDGET(win_data->copy_menu_item), non_empty);
    gtk_widget_set_sensitive(GTK_WIDGET(win_data->copy_button),    non_empty);
    gtk_widget_set_sensitive(GTK_WIDGET(win_data->cut_menu_item),  non_empty);
    gtk_widget_set_sensitive(GTK_WIDGET(win_data->cut_button),     non_empty);
    gtk_widget_set_sensitive(GTK_WIDGET(win_data->make_uppercase_menu_item), non_empty);
    gtk_widget_set_sensitive(GTK_WIDGET(win_data->make_lowercase_menu_item), non_empty);

    line1 = gtk_text_iter_get_line(&iter1);
    line2 = gtk_text_iter_get_line(&iter2);
    if(line1 == line2 ||
       (line2 == line1+1 && gtk_text_iter_get_line_offset(&iter2) == 0))
    {
        if(non_empty)
            sprintf(buf, N_("Line %d"), (int)line1+1);
        else
            sprintf(buf, N_("%d:%d"),
                    (int)line1+1, (int)gtk_text_iter_get_line_offset(&iter1)+1);
        gtk_label_set_text(GTK_LABEL(win_data->coord_label), buf);
    } else
        gtk_label_set_text(GTK_LABEL(win_data->coord_label), NULL);
}

void my_text_view_scroll_lines(const MainwinData *win_data, gdouble lines)
{
    GtkWidget *scroll;
    GtkAdjustment *adj;

    scroll = gtk_widget_get_parent(win_data->thetext);
    adj = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(scroll));
    gtk_adjustment_set_value(adj,
                             MIN(adj->upper - adj->page_size,
                                 gtk_adjustment_get_value(adj)
                                 + lines*win_data->the_line_height));
}

/* GTK+ 2.0.6 doesn't wrap D&D into {begin,end}_user_action. I think it is a bug.
   TODO: Check in higher GTK versions and report bug if needed. */

static void on_drag_begin(GtkWidget *widget, GdkDragContext *drag_context, gpointer user_data)
{
    GtkTextBuffer *buffer;

    buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(widget));
    gtk_text_buffer_begin_user_action(buffer);
}

static void on_drag_end(GtkWidget *widget, GdkDragContext *drag_context, gpointer user_data)
{
    GtkTextBuffer *buffer;

    buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(widget));
    gtk_text_buffer_end_user_action(buffer);
}

static void on_undo_move(GtkTextView *view)
{
    GtkTextBuffer *buffer;
    GtkTextMark *mark;

    buffer = GTK_TEXT_BUFFER(gtk_text_view_get_buffer(view));
    mark = gtk_text_buffer_get_mark(buffer, N_("insert"));
    gtk_text_view_scroll_mark_onscreen(view, mark);
}

static void mainwin_on_format_changed(FileDocument *document, gpointer format, gpointer user_data)
{
    const MainwinData *win_data = (MainwinData*)user_data;
    const TextFileFormat *fmt = (TextFileFormat*)format;

    if(!fmt || fmt->encoding == -1)
        gtk_label_set_text(GTK_LABEL(win_data->encoding_label), N_(" - "));
    else {
        const gchar *base;
        gchar *full;

        base = encoding_names[fmt->encoding];
        if(!strcmp(fmt->line_term, "\n"))
            full = g_strconcat(base, N_("/unix"), NULL);
        else if(!strcmp(fmt->line_term, "\r\n"))
            full = g_strconcat(base, N_("/win"), NULL);
        else if(!strcmp(fmt->line_term, "\n\r"))
            full = g_strconcat(base, N_("/mac"), NULL);
        else /* needed to suppress GCC's warning */
            full = g_strconcat(base, N_("/??"), NULL);
        gtk_label_set_text(GTK_LABEL(win_data->encoding_label), full);
        g_free(full);
    }
}

void mainwin_set_text_font(MainwinData *win_data, PangoFontDescription *font)
{
    gtk_widget_modify_font(win_data->thetext, font);
    calculate_line_height(win_data);
}

void mainwin_set_text_font_by_name(MainwinData *win_data, const gchar *fontname)
{
    if(fontname) {
        PangoFontDescription *font;

        font = pango_font_description_from_string(fontname);
        mainwin_set_text_font(win_data, font);
        pango_font_description_free(font);
    }
}

static void mainwin_text_font_notify(GObject *settings,
                                     GParamSpec *pspec,
                                     gpointer user_data)
{
    const gchar *fontname;

    g_object_get(G_OBJECT(settings), N_("text_font"), &fontname, NULL);
    mainwin_set_text_font_by_name((MainwinData*)user_data, fontname);
}

void mainwin_goto_line_dialog(const MainwinData *win_data)
{
    GtkWidget *dlg, *input;
    gint max;
    GtkTextIter iter;

    max = MAX(1, gtk_text_buffer_get_line_count(win_data->thebuffer));
    if(max>1) {
        dlg = gtk_dialog_new_with_buttons(_("Go to line"),
                                          GTK_WINDOW(win_data->mainwin),
                                          GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT,
                                          GTK_STOCK_OK,     GTK_RESPONSE_ACCEPT,
                                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                          NULL);
        input = gtk_spin_button_new_with_range(1.0, max, 1.0);
        gtk_entry_set_activates_default(GTK_ENTRY(input), TRUE);
        gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(input), FALSE);
        gtk_spin_button_set_snap_to_ticks(GTK_SPIN_BUTTON(input), TRUE);
        gtk_text_buffer_get_selection_bounds(win_data->thebuffer, &iter, NULL);
        gtk_spin_button_set_value(GTK_SPIN_BUTTON(input),
                                  gtk_text_iter_get_line(&iter)+1);
        gtk_dialog_set_default_response(GTK_DIALOG(dlg), GTK_RESPONSE_ACCEPT);
    } else {
        dlg = gtk_dialog_new_with_buttons(_("Go to line"),
                                          GTK_WINDOW(win_data->mainwin),
                                          GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT,
                                          GTK_STOCK_CLOSE, GTK_RESPONSE_CANCEL,
                                          NULL);
        input = gtk_label_new(_("There are only one line 1 in this text!"));
        gtk_dialog_set_default_response(GTK_DIALOG(dlg), GTK_RESPONSE_CANCEL);
    }
    my_disable_window_resize(GTK_WINDOW(dlg));
    gtk_widget_show(input);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dlg)->vbox), input);
    if(max>1) gtk_widget_grab_focus(input);

    if(gtk_dialog_run(GTK_DIALOG(dlg)) == GTK_RESPONSE_ACCEPT) {
        gint line;

        line = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(input));
        gtk_text_buffer_get_iter_at_line(win_data->thebuffer, &iter, line-1);
        gtk_text_buffer_place_cursor(win_data->thebuffer, &iter);
        gtk_text_view_scroll_mark_onscreen(GTK_TEXT_VIEW(win_data->thetext),
                                           gtk_text_buffer_get_insert(win_data->thebuffer));

    }
    gtk_widget_destroy(dlg);
}

static void my_file_document_format_destroy(gpointer format)
{
    g_free(format);
}

static gpointer my_file_document_format_copy(gpointer format)
{
    return g_memdup(format, sizeof(TextFileFormat));
}

static gboolean my_file_format_cb(TextFileFormat *format, gpointer user_data)
{
    const MainwinData *win_data = (MainwinData*)user_data;
    SelectEncodingDialog se_dlg;

    /* The check for format->encoding != -1 should be eliminated. */
    if(format && format->encoding != -1) return TRUE;
    create_select_encoding_dialog(win_data->mainwin,
                                  _("Choose encoding of the file to open"),
                                  &se_dlg, FALSE, format);
    if(gtk_dialog_run(GTK_DIALOG(se_dlg.dialog)) == GTK_RESPONSE_ACCEPT) {
        select_encoding_dialog_get_encoding(&se_dlg, format);
        gtk_widget_destroy(se_dlg.dialog);
    } else {
        gtk_widget_destroy(se_dlg.dialog);
        return FALSE;
    }
#if 0
    if((*enc&~MY_ENC_ATTR) == MY_ENC_LOCALE) {
        gchar *locale_name;

        locale_name = g_get_codeset();
        ic = g_iconv_open(N_("UTF-8"), locale_name);
        g_free(locale_name);
    }
#endif /* 0 */
    return TRUE;
}

gboolean mainwin_ask_encoding(MainwinData *win_data)
{
    SelectEncodingDialog se_dlg;
    TextFileFormat fmt;

    fmt = default_text_file_format;
    create_select_encoding_dialog(win_data->mainwin,
                                  _("Choose encoding"),
                                  &se_dlg, TRUE, &fmt);
    if(gtk_dialog_run(GTK_DIALOG(se_dlg.dialog)) == GTK_RESPONSE_ACCEPT) {
        select_encoding_dialog_get_encoding(&se_dlg, &fmt);
        file_document_set_format(win_data->file_document, &fmt);
        gtk_widget_destroy(se_dlg.dialog);
        return TRUE;
    } else {
        gtk_widget_destroy(se_dlg.dialog);
        return FALSE;
    }
}

static void strip_trailing_whitespace(GtkTextBuffer *buffer)
{
    gint line_count, i;

    line_count = gtk_text_buffer_get_line_count(buffer);
    for(i=0; i<line_count; ++i) {
        GtkTextIter line_start, line_preend, line_end;
        GtkTextIter iter, iter2;
        gboolean at_cr_lf = TRUE;

        gtk_text_buffer_get_iter_at_line(buffer, &line_start, i);
        if(i == line_count-1)
            gtk_text_buffer_get_end_iter(buffer, &line_end);
        else {
            gtk_text_buffer_get_iter_at_line(buffer, &line_end, i+1);
            gtk_text_iter_backward_char(&line_end);
        }
        line_preend = line_end; /* for the case that will be not assigned below */
        iter = line_end;
        while( !gtk_text_iter_equal(&iter, &line_start) ) {
            gunichar c;

            iter2 = iter;
            gtk_text_iter_backward_char(&iter2);
            c = gtk_text_iter_get_char(&iter2);
            if(at_cr_lf) {
                /* TODO: What about G_UNICODE_PARAGRAPH_SEPARATOR? */
                if(g_unichar_type(c) == G_UNICODE_LINE_SEPARATOR)
                    continue;
                at_cr_lf = FALSE;
                line_preend = iter;
            }
            if(!g_unichar_isspace(c)) break;
            iter = iter2;
        }
        /* iter points to start of trailing space */
        gtk_text_buffer_delete(buffer, &iter, &line_preend);
    }
}

static gboolean on_mainwin_clear(gpointer doc_data, gpointer *format, GError **error)
{
    MainwinData *win_data = (MainwinData*)doc_data;

    gtk_text_buffer_set_text(GTK_TEXT_BUFFER(win_data->thebuffer), N_(""), 0);
    text_undo_data_clear(win_data->undo_data);
    return TRUE;
}

static gboolean on_mainwin_save(gpointer doc_data,
                                const gchar *file_name,
                                gpointer *format,
                                GError **error)
{
    MainwinData *win_data = (MainwinData*)doc_data;

    if( (*format && ((TextFileFormat*)*format)->encoding != -1) || mainwin_ask_encoding(win_data) ) {
        if( preferences_get_boolean(N_("strip-space-at-save")) )
            strip_trailing_whitespace(win_data->thebuffer);
        if(!*format) *format = g_memdup(&default_text_file_format, sizeof default_text_file_format);
        return text_buffer_save(win_data->thebuffer,
                                file_name,
                                *format,
                                error);
    } else
        return FALSE;
}

static gboolean on_mainwin_load(gpointer doc_data,
                                const gchar *file_name,
                                gpointer *format,
                                GError **error)
{
    MainwinData *win_data = (MainwinData*)doc_data;
    gboolean res;

    if(!*format) *format = g_memdup(&default_text_file_format, sizeof default_text_file_format);
    res = text_buffer_load(win_data->thebuffer,
                           file_name,
                           (TextFileFormat**)format,
                           my_file_format_cb, win_data,
                           TRUE,
                           error);
    if(res) {
        GtkTextIter start;

        file_document_set_format(win_data->file_document, *format);
        text_undo_data_clear(win_data->undo_data);
        gtk_text_buffer_get_start_iter(win_data->thebuffer, &start);
        gtk_text_buffer_place_cursor(win_data->thebuffer, &start);
        gtk_text_view_scroll_mark_onscreen(
                GTK_TEXT_VIEW(win_data->thetext),
                gtk_text_buffer_get_mark(GTK_TEXT_BUFFER(win_data->thebuffer),
                                         N_("insert")));
    }
    return res;
}

static void on_file_name_notify(GObject    *obj,
                                GParamSpec *pspec,
                                gpointer    data)
{
    MainwinData *win_data = (MainwinData*)data;
    const gchar *str;

    g_object_get(obj, N_("file-name"), &str, NULL);
    mainwin_set_title_label(win_data, str);
    gtk_widget_set_sensitive(GTK_WIDGET(win_data->revert_menu_item), str != NULL);
}

static void on_save_or_load(FileDocument *document,
                            const gchar *file_name,
                            gpointer format,
                            gboolean save,
                            gpointer user_data)
{
    gchar *format_str;
    
    g_assert(file_name);

    format_str = text_file_format_to_str(format);
    preferences_history_add(file_name, format_str);
    g_free(format_str);

    /* TODO: This probably should go to document.c... */
#if 0
    gnome_history_recently_used(file_name,
                                N_("text/plain"),
                                g_set_prgname(),
                                _("Load text file"));
#endif /* 0 */
}

static void on_document_modified_changed(GObject *settings,
                                         GParamSpec *pspec,
                                         gpointer user_data)
{
    MainwinData *win_data = (MainwinData*)user_data;
    gboolean modified;

    modified = file_document_get_modified(win_data->file_document);
    gtk_text_buffer_set_modified(win_data->thebuffer, modified);
    if(modified) /* There are no "mapped" property. */
        gtk_widget_map  (win_data->modified_label);
    else
        gtk_widget_unmap(win_data->modified_label);
}

static void on_textbuffer_modified_changed(GtkTextBuffer *textbuffer,
                                           gpointer user_data)
{
    MainwinData *win_data = (MainwinData*)user_data;

    file_document_set_modified(win_data->file_document,
                               gtk_text_buffer_get_modified(win_data->thebuffer));
}

void my_report_error(gpointer data, GError *error)
{
    MainwinData *win_data = (MainwinData*)data;

    my_g_error_message(win_data->mainwin, error);
}

struct HistoryItemData {
    gchar *file_name;
    gchar *format_str;
};

static void history_item_free(gpointer data, GObject *where_the_object_was)
{
    struct HistoryItemData *item_data = (struct HistoryItemData*)data;
    
    g_free(item_data->file_name);
    g_free(item_data->format_str);
    g_free(item_data);
}

static void on_history_menu_item(GtkMenuItem *item, gpointer data)
{
    const MainwinData *win_data;
    const struct HistoryItemData *item_data = (struct HistoryItemData*)data;
    TextFileFormat *format;

    win_data = mainwin_get_data_by_widget(GTK_WIDGET(item));
    format = text_file_format_from_str(item_data->format_str);
    file_document_do_load_file(win_data->file_document,
                               item_data->file_name,
                               (gpointer*)&format);
    g_free(format);
}

static void mainwin_on_file_menu_show(GtkWidget *menu, gpointer user_data)
{
    const MainwinData *win_data = (MainwinData*)user_data;
    GList *childs, *iter;
    gint pos, count, i;

    /* Remove old history items */
    pos = 0;
    iter = childs = gtk_container_get_children(GTK_CONTAINER(menu));
    while(iter->data != win_data->before_history_item) {
        iter = g_list_next(iter);
        ++pos;
    }
    iter = g_list_next(iter);
    while(iter->data != win_data->after_history_item) {
        gtk_widget_destroy(GTK_WIDGET(iter->data));
        iter = g_list_next(iter);
    }
    g_list_free(childs);

    /* Insert history items */
    count = preferences_history_count();
    for(i=0; i<count; ++i) {
        gchar *text, *ptr, *format_str;
        GtkWidget *item;
        struct HistoryItemData *data;

        ++pos;
        data = g_malloc(sizeof(struct HistoryItemData));
        data->file_name = preferences_history_get(i);
        if(!data->file_name) { /* sanity check for broken INI file */
            g_free(data);
            break;
        }
        data->format_str = preferences_history_get_format(i);
        ptr = text = g_malloc(3+strlen(data->file_name)+1);
        if(i!=9) {
            *ptr++ = N_('_');
            *ptr++ = (char)((i+1)%10+'0');
        }
        *ptr++ = N_(' ');
        strcpy(ptr, data->file_name);
        item = gtk_menu_item_new_with_mnemonic(text);
        g_free(text);
        g_object_weak_ref(G_OBJECT(item), history_item_free, data);
        g_signal_connect(G_OBJECT(item), N_("activate"),
                         G_CALLBACK(on_history_menu_item), data);
        gtk_widget_show(item);
        gtk_menu_shell_insert(GTK_MENU_SHELL(menu), item, pos);
    }
}

#ifdef LIMITED_VERSION
static void mainwin_on_help_menu_show(GtkWidget *menu, gpointer user_data)
{
    const MainwinData *win_data = (MainwinData*)user_data;

    (is_registered_version() ? gtk_widget_hide : gtk_widget_show)
        (GTK_WIDGET(win_data->unlock_menu_item));
}
#endif /* LIMITED_VERSION */

GtkWidget *mainwin_new(void)
{
    MainwinData *win_data;

    GtkWidget *mainwin;
    GtkWidget *undo_button, *redo_button,
              *undo_menu_item, *redo_menu_item,
              *text_wrap_menu_item, *autoindent_menu_item;
    GtkWidget *file_menu;
#ifdef LIMITED_VERSION
    GtkWidget *help_menu;
#endif
    GtkTooltips *tooltips;
    TextFileFormat file_format;
    gchar *line_term_hex, *encoding;
    int i;

    win_data = g_malloc(sizeof(MainwinData));
    mainwin = win_data->mainwin = create_mainwin();

    g_object_set_data_full(G_OBJECT(mainwin), N_("data"), win_data, g_free);

    gtk_window_set_title(GTK_WINDOW(mainwin), g_get_prgname());

    win_data->title_label     = lookup_widget(mainwin, N_("title_label"));
    win_data->title_label_box = lookup_widget(mainwin, N_("title_label_box"));
    win_data->title_ellipses  = lookup_widget(mainwin, N_("title_ellipses"));
    win_data->title_event_box = lookup_widget(mainwin, N_("title_event_box"));
    win_data->thetext         = lookup_widget(mainwin, N_("thetext"));
    win_data->modified_label  = lookup_widget(mainwin, N_("modified_label"));
    win_data->coord_label     = lookup_widget(mainwin, N_("coord_label"));
    win_data->encoding_label  = lookup_widget(mainwin, N_("encoding_label"));

    /* TODO: exclude unnecessary stuff here. */
    file_menu = lookup_widget(mainwin, N_("file_menu_menu"));
#ifdef LIMITED_VERSION
    help_menu = lookup_widget(mainwin, N_("help_menu_menu"));
#endif
    win_data->before_history_item = (GtkMenuItem*)lookup_widget(mainwin, N_("before_history_item"));
    win_data->after_history_item  = (GtkMenuItem*)lookup_widget(mainwin, N_("after_history_item" ));
    win_data->cut_button  = lookup_widget(mainwin, N_("cut_button" ));
    win_data->copy_button = lookup_widget(mainwin, N_("copy_button"));

    win_data->make_uppercase_menu_item = (GtkMenuItem*)lookup_widget(mainwin, N_("make_uppercase_menu_item"));
    win_data->make_lowercase_menu_item = (GtkMenuItem*)lookup_widget(mainwin, N_("make_lowercase_menu_item"));
    win_data->copy_menu_item           = (GtkMenuItem*)lookup_widget(mainwin, N_("copy_menu_item"));
    win_data->cut_menu_item            = (GtkMenuItem*)lookup_widget(mainwin, N_("cut_menu_item"));
    win_data->revert_menu_item         = (GtkMenuItem*)lookup_widget(win_data->mainwin, N_("revert_item"));
    win_data->unlock_menu_item         = (GtkMenuItem*)lookup_widget(win_data->mainwin, N_("unlock_item"));

    undo_menu_item = lookup_widget(mainwin, N_("undo_menu_item"));
    redo_menu_item = lookup_widget(mainwin, N_("redo_menu_item"));
    undo_button    = lookup_widget(mainwin, N_("undo_button"));
    redo_button    = lookup_widget(mainwin, N_("redo_button"));
    text_wrap_menu_item    = lookup_widget(mainwin, N_("text_wrap_menu_item"));
    autoindent_menu_item   = lookup_widget(mainwin, N_("autoindent_menu_item"));

    gtk_widget_set_size_request(win_data->title_label, -1, -1);
    /*g_object_set_property(G_OBJECT(thetext), N_("cursor-color"), red);*/

    /* block */ {
        GtkStyle *style;
        PangoFontMetrics *metrics;
        int width;

        style = gtk_widget_get_style(win_data->coord_label);
        metrics = pango_context_get_metrics(gtk_widget_get_pango_context(win_data->coord_label),
                                            style->font_desc,
                                            NULL);
        width = (3+2)*pango_font_metrics_get_approximate_digit_width(metrics) +
                      pango_font_metrics_get_approximate_char_width (metrics);
        width = width*5/4;
        pango_font_metrics_unref(metrics);
        gtk_widget_set_size_request(win_data->coord_label, width/PANGO_SCALE, -1);
    }

    win_data->thebuffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(win_data->thetext));
    win_data->undo_data = text_undo_data_new(win_data->thebuffer);
    g_signal_connect_swapped(G_OBJECT(win_data->undo_data), N_("moved"),
                             G_CALLBACK(on_undo_move), win_data->thetext);

    win_data->file_document = file_document_new(mainwin,
                                                win_data,
                                                on_mainwin_clear,
                                                on_mainwin_save,
                                                on_mainwin_load,
                                                my_report_error);
    /*win_data->file_format = default_text_file_format;*/
    file_document_set_format_operations(win_data->file_document,
                                        &my_file_document_format_destroy,
                                        &my_file_document_format_copy);
    g_signal_connect(G_OBJECT(win_data->file_document), N_("format-changed"),
                     G_CALLBACK(mainwin_on_format_changed), win_data);
    /* TODO: Setting BOM etc. defaults rely on file_format.encoding != -1 */
    file_format.encoding = -1;
    file_format.bom_known = TRUE; /* FIXME */
    g_object_get(G_OBJECT(gtk_settings_get_default()),
                 N_("start-encoding-bom"), &file_format.bom, NULL);
    g_object_get(G_OBJECT(gtk_settings_get_default()),
                 N_("start-encoding"), &encoding, NULL);
    if(encoding)
        for(i=0; i<num_encodings; ++i)
            if( !strcmp(encoding, encoding_names[i]) ) {
                file_format.encoding = i;
                if(i >= num_encodings_with_bom)
                    file_format.bom = FALSE; /* FIXME */
                break;
            }
    g_free(encoding);
    g_object_get(G_OBJECT(gtk_settings_get_default()),
                 N_("start-encoding-term"), &line_term_hex, NULL);
    file_format.line_term = decode_line_term(line_term_hex);
    g_free(line_term_hex);
    file_document_set_format(win_data->file_document, &file_format);

    /* GtkTextBuffer doesn't have "modified" property :-( */
    g_signal_connect(G_OBJECT(win_data->file_document), N_("notify::modified"),
                     G_CALLBACK(on_document_modified_changed), win_data);
    g_signal_connect(G_OBJECT(win_data->thebuffer), N_("modified-changed"),
                     G_CALLBACK(on_textbuffer_modified_changed), win_data);


    g_signal_connect(G_OBJECT(win_data->thebuffer), N_("mark-set"),
                     G_CALLBACK(on_mark_set), win_data);
    g_signal_connect(G_OBJECT(file_menu), N_("show"),
                     G_CALLBACK(mainwin_on_file_menu_show), win_data);
#ifdef LIMITED_VERSION
    g_signal_connect(G_OBJECT(help_menu), N_("show"),
                     G_CALLBACK(mainwin_on_help_menu_show), win_data);
#endif
    g_signal_connect(G_OBJECT(win_data->file_document), N_("notify::file-name"),
                     G_CALLBACK(on_file_name_notify), win_data);
    g_signal_connect(G_OBJECT(win_data->file_document), N_("save-or-load"),
                     G_CALLBACK(on_save_or_load), NULL);

    my_glib_bind_properties2_one_way(G_OBJECT(win_data->undo_data), N_("can-undo"),
                                     G_OBJECT(undo_button), N_("sensitive"),
                                     TRUE);
    my_glib_bind_properties2_one_way(G_OBJECT(win_data->undo_data), N_("can-redo"),
                                     G_OBJECT(redo_button), N_("sensitive"),
                                     TRUE);
    my_glib_bind_properties2_one_way(G_OBJECT(win_data->undo_data), N_("can-undo"),
                                     G_OBJECT(undo_menu_item), N_("sensitive"),
                                     TRUE);
    my_glib_bind_properties2_one_way(G_OBJECT(win_data->undo_data), N_("can-redo"),
                                     G_OBJECT(redo_menu_item), N_("sensitive"),
                                     TRUE);

    my_glib_bind_properties2(G_OBJECT(gtk_settings_get_default()), N_("wrap-text"),
                             G_OBJECT(text_wrap_menu_item), N_("active"), TRUE);
    my_glib_bind_properties2(G_OBJECT(gtk_settings_get_default()), N_("autoindent"),
                             G_OBJECT(autoindent_menu_item), N_("active"), TRUE);

    gtk_widget_set_sensitive(GTK_WIDGET(win_data->make_uppercase_menu_item), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(win_data->make_lowercase_menu_item), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(win_data->copy_menu_item), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(win_data->cut_menu_item ), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(win_data->copy_button   ), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(win_data->cut_button    ), FALSE);

    /* Glade 2.0.0 set wrong tooltips for toolbar, so do it manually: */
    tooltips = (GtkTooltips*)lookup_widget(mainwin, N_("tooltips"));
    gtk_tooltips_set_tip(tooltips,
                         lookup_widget(mainwin, N_("new_file_button")),
                         _("New"), NULL);
    gtk_tooltips_set_tip(tooltips,
                         lookup_widget(mainwin, N_("open_file_button")),
                         _("Open"), NULL);
    gtk_tooltips_set_tip(tooltips,
                          lookup_widget(mainwin, N_("save_button")),
                          _("Save"), NULL);
    gtk_tooltips_set_tip(tooltips,
                         lookup_widget(mainwin, N_("save_as_button")),
                         _("Save as"), NULL);
    gtk_tooltips_set_tip(tooltips,
                         lookup_widget(mainwin, N_("cut_button")),
                         _("Cut"), NULL);
    gtk_tooltips_set_tip(tooltips,
                         lookup_widget(mainwin, N_("copy_button")),
                         _("Copy"), NULL);
    gtk_tooltips_set_tip(tooltips,
                         lookup_widget(mainwin, N_("paste_button")),
                         _("Paste"), NULL);
    gtk_tooltips_set_tip(tooltips,
                         lookup_widget(mainwin, N_("undo_button")),
                         _("Undo"), NULL);
    gtk_tooltips_set_tip(tooltips,
                         lookup_widget(mainwin, N_("redo_button")),
                         _("Redo"), NULL);
    gtk_tooltips_set_tip(tooltips,
                         lookup_widget(mainwin, N_("find_button")),
                         _("Find"), NULL);
    gtk_tooltips_set_tip(tooltips,
                         lookup_widget(mainwin, N_("find_next_button")),
                         _("Find next"), NULL);
    gtk_tooltips_set_tip(tooltips,
                         lookup_widget(mainwin, N_("replace_button")),
                         _("Replace"), NULL);

    g_signal_connect(G_OBJECT(win_data->thetext), N_("key-press-event"),
                     G_CALLBACK(on_key_press), NULL);
    g_signal_connect(G_OBJECT(win_data->thetext), N_("drag-begin"),
                     G_CALLBACK(on_drag_begin), NULL);
    g_signal_connect(G_OBJECT(win_data->thetext), N_("drag-end"),
                     G_CALLBACK(on_drag_end), NULL);

    /* They are disconnected in "destroy" handler. */
    win_data->wrap_text_sig =
        g_signal_connect(G_OBJECT(gtk_settings_get_default()), N_("notify::wrap-text"),
                         G_CALLBACK(mainwin_text_wrap_notify), win_data);
    win_data->text_font_sig =
        g_signal_connect(G_OBJECT(gtk_settings_get_default()), N_("notify::text-font"),
                         G_CALLBACK(mainwin_text_font_notify), win_data);
    mainwin_set_text_wrap(win_data, preferences_get_boolean(N_("wrap-text")));
    /* block */ {
        gchar *str;
        g_object_get(gtk_settings_get_default(), N_("text-font"), &str, NULL);
        mainwin_set_text_font_by_name(win_data, str);
    }

    initialize_search(&win_data->search_data, win_data->thetext);
    setup_dnd(win_data->mainwin);
    /*setup_dnd(win_data->thetext);*/

    gtk_widget_grab_focus(win_data->thetext);

    /* This doesn't flash because gtk_widget_show_all() only queries showing, not does it. */
    gtk_widget_hide(GTK_WIDGET(win_data->unlock_menu_item));
    gtk_widget_show(mainwin);
    gtk_widget_unmap(win_data->title_ellipses);
    gtk_widget_unmap(win_data->modified_label);
    /*gtk_widget_hide(mainwin);*/ /* In Gtk 2.2.1-4 causes a window with no decorations. */

    ++mainwin_windows_count;
    return mainwin;
}

/* FIXME: about file format here... */
void mainwin_insert_file_at_cursor_dialog(MainwinData *win_data)
{
    gint response;
    GtkWidget *file_dlg;

    file_dlg = gtk_file_selection_new(_("Insert File at Cursor")); /* TODO: the common dialog */
    gtk_window_set_transient_for(GTK_WINDOW(file_dlg), GTK_WINDOW(win_data->mainwin));
    gtk_window_set_destroy_with_parent(GTK_WINDOW(file_dlg), TRUE);
    gtk_file_selection_show_fileop_buttons(GTK_FILE_SELECTION(file_dlg));

    response = gtk_dialog_run(GTK_DIALOG(file_dlg));
    gtk_widget_hide(file_dlg);
    if(response == GTK_RESPONSE_OK) {
        const char *filename;
        TextFileFormat format;

        filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(file_dlg));
        format = default_text_file_format;
        text_buffer_insert_file_at_cursor(win_data->mainwin,
                                          win_data->thebuffer,
                                          filename,
                                          &format,
                                          my_file_format_cb, win_data,
                                          TRUE);
    }
}

void mainwin_destroy(MainwinData *win_data)
{
    /* Without this when closing the mainwin sometimes the signal comes after the widgets are destroyed. */
    /* TODO: do it by automatical means provided by Glib. */
    g_signal_handlers_disconnect_by_func(win_data->thebuffer, G_CALLBACK(on_mark_set), NULL);
    g_signal_handler_disconnect(G_OBJECT(gtk_settings_get_default()),
                                win_data->wrap_text_sig);
    g_signal_handler_disconnect(G_OBJECT(gtk_settings_get_default()),
                                win_data->text_font_sig);
    destroy_search(&win_data->search_data);
    --mainwin_windows_count;
    if(!mainwin_windows_count) gtk_main_quit();
}
