/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "inisettings.h"

#include <stdlib.h>
#include <errno.h>
#include <gtk/gtk.h>

#define N_(s) s

IniSettingsFile *ini_settings_new(const char                   *file_name,
                                  IniSettingsFileErrorCallback  error_callback,
                                  gpointer                      user_data)
{
    IniSettingsFile *ini_file;

    ini_file = g_malloc(sizeof(IniSettingsFile));

    ini_file->file_name = g_strdup(file_name);
    ini_file->dict = NULL;
    ini_file->loading = FALSE;
    ini_file->freeze_write_count = ini_file->freeze_load_count = 0;
    ini_file->keys_table  = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
    ini_file->sects_table = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
    ini_file->error_callback = error_callback;
    ini_file->user_data = user_data;
    
    ini_settings_load(ini_file); /* TODO: necessary? */
    
    return ini_file;
}

void ini_settings_destroy(IniSettingsFile *ini_file)
{
    g_free(ini_file->file_name);
    iniparser_freedict(ini_file->dict);
    g_hash_table_destroy(ini_file->keys_table);
    g_hash_table_destroy(ini_file->sects_table);
}

/* FIXME: Write atomically. */
void ini_settings_write(IniSettingsFile *ini_file)
{
    FILE *file;

    file = fopen(ini_file->file_name, "w");
    if(file) {
        iniparser_dump_ini(ini_file->dict, file);
        if(!fclose(file) && ini_file->error_callback)
            (*ini_file->error_callback)(errno, ini_file->user_data);
    } else if(ini_file->error_callback)
        (*ini_file->error_callback)(errno, ini_file->user_data);
}

static void ini_settings_update_entry(IniSettingsFile *ini_file, const gchar *name)
{
    GtkSettings *settings;
    gchar *nick;
    GValue gvalue/*, old_gvalue*/;
    GtkSettingsValue svalue;
    GType type;
    GParamSpec *pspec;
    
    settings = gtk_settings_get_default();

    pspec = g_object_class_find_property(gtk_type_class(GTK_TYPE_SETTINGS), name);
    nick = (gchar*)g_param_spec_get_nick(pspec);

    if( !iniparser_find_entry(ini_file->dict, (char*)nick) ) return;

    type = G_PARAM_SPEC_VALUE_TYPE(pspec);
    memset(&gvalue, 0, sizeof(GValue));
    g_value_init(&gvalue, type);
#if 0
    memset(&old_gvalue, 0, sizeof(GValue));
    g_value_init(&old_gvalue, type);
#endif
    switch(type) {
        case G_TYPE_STRING:
            g_value_set_string(&gvalue, iniparser_getstring(ini_file->dict, nick, NULL));
            break;
        case G_TYPE_INT:
            g_value_set_int(&gvalue, iniparser_getint(ini_file->dict, nick, 0));
            break;
        case G_TYPE_UINT:
            g_value_set_uint(&gvalue, (guint)iniparser_getint(ini_file->dict, nick, 0));
            break;
        /*case G_TYPE_LONG:*/
        /*case G_TYPE_ULONG:*/
        case G_TYPE_DOUBLE:
            g_value_set_double(&gvalue, iniparser_getdouble(ini_file->dict, nick, 0.0));
            break;
        case G_TYPE_FLOAT:
            g_value_set_float(&gvalue, (float)iniparser_getdouble(ini_file->dict, nick, 0.0));
            break;
        case G_TYPE_BOOLEAN:
            g_value_set_boolean(&gvalue, iniparser_getboolean(ini_file->dict, nick, 0));
            break;
        case G_TYPE_CHAR:
            g_value_set_char(&gvalue, iniparser_getstring(ini_file->dict, nick, '\0')[0]);
            break;
        case G_TYPE_UCHAR:
            g_value_set_uchar(&gvalue, (guchar)iniparser_getstring(ini_file->dict, nick, '\0')[0]);
            break;
#if 0 /* TODO, there are G_TYPE_UNICHAR */
        case G_TYPE_UNICHAR: {
            const gchar *str;
            
            str = iniparser_getstring(ini_file->dict, nick, NULL);
            g_value_set_unichar(&gvalue, strtoul(str, NULL, 16));
            break;
        }
#endif /* 0 */
        default:
            g_assert_not_reached();
            break;
    }
#if 0 /* TODO: Doesn't work so now for startup values. */
    g_object_get_property(G_OBJECT(settings), name, &old_gvalue);
    if(!g_param_values_cmp(pspec, &gvalue, &old_gvalue))
#endif
        g_object_set_property(G_OBJECT(settings), name, &gvalue);
    g_value_unset(&gvalue);
#if 0
    g_value_unset(&old_gvalue);
#endif
}

static void ini_settings_load_iteration(gpointer key, gpointer value, gpointer user_data)
{
    ini_settings_update_entry((IniSettingsFile*)user_data, (char*)key);
}

static void ini_settings_sect_iteration(gpointer key, gpointer value, gpointer user_data)
{
    dictionary_set((dictionary*)user_data, (char*)key, NULL);
}

void ini_settings_load(IniSettingsFile *ini_file)
{
    if(ini_file->freeze_load_count) return;

    if(ini_file->dict) iniparser_freedict(ini_file->dict);
    ini_file->dict = iniparser_load(ini_file->file_name);
    if(!ini_file->dict) ini_file->dict = dictionary_new(0);
    g_hash_table_foreach(ini_file->sects_table,
                         ini_settings_sect_iteration,
                         ini_file->dict);

    ini_file->loading = TRUE;
    g_object_freeze_notify(G_OBJECT(gtk_settings_get_default()));
    g_hash_table_foreach(ini_file->keys_table,
                         ini_settings_load_iteration,
                         ini_file);
    /* This moment writing is disabled. */
    g_object_thaw_notify(G_OBJECT(gtk_settings_get_default()));
    ini_file->loading = FALSE;
}

static void ini_settings_property_cb(GObject    *object,
                                     GParamSpec *pspec,
                                     gpointer    data)
{
    IniSettingsFile *ini_file;
    GValue value;
    gchar *name, *nick;
    GType type;

    ini_file = (IniSettingsFile*)data;

    if(ini_file->loading) return;

    name = (gchar*)g_param_spec_get_name(pspec);
    nick = (gchar*)g_param_spec_get_nick(pspec);
    type = G_PARAM_SPEC_VALUE_TYPE(pspec);
    memset(&value, 0, sizeof(GValue));
    g_value_init(&value, type);
    g_object_get_property(object, name, &value);
    switch(type) {
        case G_TYPE_STRING:
            iniparser_setstr(ini_file->dict, nick, (char*)g_value_get_string(&value));
            break;
        case G_TYPE_INT:
            dictionary_setint(ini_file->dict, nick, g_value_get_int(&value));
            break;
        case G_TYPE_UINT:
            dictionary_setint(ini_file->dict, nick, (int)g_value_get_uint(&value));
            break;
        /*case G_TYPE_LONG:*/
        /*case G_TYPE_ULONG:*/
        case G_TYPE_DOUBLE:
            dictionary_setdouble(ini_file->dict, nick, g_value_get_double(&value));
            break;
        case G_TYPE_FLOAT:
            dictionary_setdouble(ini_file->dict, nick, (double)g_value_get_float(&value));
            break;
        case G_TYPE_BOOLEAN:
            iniparser_setstr(ini_file->dict,
                             nick,
                             g_value_get_boolean(&value) ? N_("true") : N_("false"));
            break;
        case G_TYPE_CHAR: {
            gchar str[2];
            
            str[0] = g_value_get_char(&value);
            str[1] = '\0';
            iniparser_setstr(ini_file->dict, nick, str);
            break;
        }
        case G_TYPE_UCHAR: {
            guchar str[2];
            
            str[0] = g_value_get_uchar(&value);
            str[1] = '\0';
            iniparser_setstr(ini_file->dict,nick, str);
            break;
        }
#if 0 /* TODO, there are G_TYPE_UNICHAR */
        case G_TYPE_UNICHAR: {
            gunichar uc;
            char str[9]; /* hex str */
            
            uc = g_value_get_unichar(&value);
            for(int i=8; i>=0; --i, uc /= 16) {
                char h;
                
                h = uc % 16;
                str[i] = h<10 : '0'+h : 'a'-(char)10+h;
            }
            str[8] = '\0';
            iniparser_setstr(ini_file->dict, nick, str);
            break;
        }
#endif /* 0 */
        default:
            g_assert_not_reached();
            break;
    }
    g_value_unset(&value);
    
    /* TODO: Fight with freeze/thaw notify producing multiple savings in a row */
    if(!ini_file->freeze_write_count) ini_settings_write(ini_file);
}

void ini_settings_install_property(IniSettingsFile *ini_file, GParamSpec *pspec)
{
    gchar *notify_signal_name, *sect;
    const gchar *name, *nick, *colon;
    GObject *settings;
    
    g_param_spec_ref(pspec);
    gtk_settings_install_property(pspec);
    name = g_param_spec_get_name(pspec);
    nick = g_param_spec_get_nick(pspec);
    settings = G_OBJECT(gtk_settings_get_default());
    notify_signal_name = g_strconcat(N_("notify::"), name, NULL);
    g_signal_connect(settings, notify_signal_name,
                     G_CALLBACK(ini_settings_property_cb), ini_file);
    g_free(notify_signal_name);
    g_object_freeze_notify(settings);
    colon = strchr(nick, ':');
    g_assert(colon);
    sect = g_strndup(nick, colon-nick);
    g_hash_table_insert(ini_file->sects_table, g_strdup(sect), NULL);
    g_hash_table_insert(ini_file->keys_table , g_strdup(name), NULL);
    dictionary_set(ini_file->dict, sect, NULL);
    ini_settings_update_entry(ini_file, name);
    g_param_spec_unref(pspec);
    g_object_thaw_notify(settings);
}

void ini_settings_freeze_write(IniSettingsFile *ini_file)
{
    ++ini_file->freeze_write_count;
}

void ini_settings_thaw_write_no_write(IniSettingsFile *ini_file)
{
    g_return_if_fail(ini_file->freeze_write_count > 0);
    
    --ini_file->freeze_write_count;
}

void ini_settings_thaw_write(IniSettingsFile *ini_file)
{
    g_return_if_fail(ini_file->freeze_write_count > 0);
    
    --ini_file->freeze_write_count;
    if(!ini_file->freeze_write_count) 
        ini_settings_write(ini_file);
}

void ini_settings_freeze_load(IniSettingsFile *ini_file)
{
    if(!ini_file->freeze_load_count) 
        ini_settings_load(ini_file);
    ++ini_file->freeze_load_count;
}

void ini_settings_freeze_load_no_load(IniSettingsFile *ini_file)
{
    ++ini_file->freeze_load_count;
}

void ini_settings_thaw_load(IniSettingsFile *ini_file)
{
    g_return_if_fail(ini_file->freeze_load_count > 0);
    
    --ini_file->freeze_load_count;
}
