/*
 * (C) 2016 by Laura Garcia <nevola@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */

#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <arpa/inet.h>
#include <errno.h>
#include <linux/netfilter/nf_tables.h>

#include "internal.h"
#include <libmnl/libmnl.h>
#include <libnftnl/expr.h>
#include <libnftnl/rule.h>

struct nftnl_expr_hash {
	enum nft_registers	sreg;
	enum nft_registers	dreg;
	unsigned int		len;
	unsigned int		modulus;
	unsigned int		seed;
	unsigned int		offset;
};

static int
nftnl_expr_hash_set(struct nftnl_expr *e, uint16_t type,
		    const void *data, uint32_t data_len)
{
	struct nftnl_expr_hash *hash = nftnl_expr_data(e);

	switch (type) {
	case NFTNL_EXPR_HASH_SREG:
		hash->sreg = *((uint32_t *)data);
		break;
	case NFTNL_EXPR_HASH_DREG:
		hash->dreg = *((uint32_t *)data);
		break;
	case NFTNL_EXPR_HASH_LEN:
		hash->len = *((uint32_t *)data);
		break;
	case NFTNL_EXPR_HASH_MODULUS:
		hash->modulus = *((uint32_t *)data);
		break;
	case NFTNL_EXPR_HASH_SEED:
		hash->seed = *((uint32_t *)data);
		break;
	case NFTNL_EXPR_HASH_OFFSET:
		hash->offset = *((uint32_t *)data);
		break;
	default:
		return -1;
	}
	return 0;
}

static const void *
nftnl_expr_hash_get(const struct nftnl_expr *e, uint16_t type,
		    uint32_t *data_len)
{
	struct nftnl_expr_hash *hash = nftnl_expr_data(e);

	switch (type) {
	case NFTNL_EXPR_HASH_SREG:
		*data_len = sizeof(hash->sreg);
		return &hash->sreg;
	case NFTNL_EXPR_HASH_DREG:
		*data_len = sizeof(hash->dreg);
		return &hash->dreg;
	case NFTNL_EXPR_HASH_LEN:
		*data_len = sizeof(hash->len);
		return &hash->len;
	case NFTNL_EXPR_HASH_MODULUS:
		*data_len = sizeof(hash->modulus);
		return &hash->modulus;
	case NFTNL_EXPR_HASH_SEED:
		*data_len = sizeof(hash->seed);
		return &hash->seed;
	case NFTNL_EXPR_HASH_OFFSET:
		*data_len = sizeof(hash->offset);
		return &hash->offset;
	}
	return NULL;
}

static int nftnl_expr_hash_cb(const struct nlattr *attr, void *data)
{
	const struct nlattr **tb = data;
	int type = mnl_attr_get_type(attr);

	if (mnl_attr_type_valid(attr, NFTA_HASH_MAX) < 0)
		return MNL_CB_OK;

	switch (type) {
	case NFTA_HASH_SREG:
	case NFTA_HASH_DREG:
	case NFTA_HASH_LEN:
	case NFTA_HASH_MODULUS:
	case NFTA_HASH_SEED:
	case NFTA_HASH_OFFSET:
		if (mnl_attr_validate(attr, MNL_TYPE_U32) < 0)
			abi_breakage();
		break;
	}

	tb[type] = attr;
	return MNL_CB_OK;
}

static void
nftnl_expr_hash_build(struct nlmsghdr *nlh, const struct nftnl_expr *e)
{
	struct nftnl_expr_hash *hash = nftnl_expr_data(e);

	if (e->flags & (1 << NFTNL_EXPR_HASH_SREG))
		mnl_attr_put_u32(nlh, NFTA_HASH_SREG, htonl(hash->sreg));
	if (e->flags & (1 << NFTNL_EXPR_HASH_DREG))
		mnl_attr_put_u32(nlh, NFTA_HASH_DREG, htonl(hash->dreg));
	if (e->flags & (1 << NFTNL_EXPR_HASH_LEN))
		mnl_attr_put_u32(nlh, NFTA_HASH_LEN, htonl(hash->len));
	if (e->flags & (1 << NFTNL_EXPR_HASH_MODULUS))
		mnl_attr_put_u32(nlh, NFTA_HASH_MODULUS, htonl(hash->modulus));
	if (e->flags & (1 << NFTNL_EXPR_HASH_SEED))
		mnl_attr_put_u32(nlh, NFTA_HASH_SEED, htonl(hash->seed));
	if (e->flags & (1 << NFTNL_EXPR_HASH_OFFSET))
		mnl_attr_put_u32(nlh, NFTA_HASH_OFFSET, htonl(hash->offset));
}

static int
nftnl_expr_hash_parse(struct nftnl_expr *e, struct nlattr *attr)
{
	struct nftnl_expr_hash *hash = nftnl_expr_data(e);
	struct nlattr *tb[NFTA_HASH_MAX+1] = {};
	int ret = 0;

	if (mnl_attr_parse_nested(attr, nftnl_expr_hash_cb, tb) < 0)
		return -1;

	if (tb[NFTA_HASH_SREG]) {
		hash->sreg = ntohl(mnl_attr_get_u32(tb[NFTA_HASH_SREG]));
		e->flags |= (1 << NFTNL_EXPR_HASH_SREG);
	}
	if (tb[NFTA_HASH_DREG]) {
		hash->dreg = ntohl(mnl_attr_get_u32(tb[NFTA_HASH_DREG]));
		e->flags |= (1 << NFTNL_EXPR_HASH_DREG);
	}
	if (tb[NFTA_HASH_LEN]) {
		hash->len = ntohl(mnl_attr_get_u32(tb[NFTA_HASH_LEN]));
		e->flags |= (1 << NFTNL_EXPR_HASH_LEN);
	}
	if (tb[NFTA_HASH_MODULUS]) {
		hash->modulus = ntohl(mnl_attr_get_u32(tb[NFTA_HASH_MODULUS]));
		e->flags |= (1 << NFTNL_EXPR_HASH_MODULUS);
	}
	if (tb[NFTA_HASH_SEED]) {
		hash->seed = ntohl(mnl_attr_get_u32(tb[NFTA_HASH_SEED]));
		e->flags |= (1 << NFTNL_EXPR_HASH_SEED);
	}
	if (tb[NFTA_HASH_OFFSET]) {
		hash->offset = ntohl(mnl_attr_get_u32(tb[NFTA_HASH_OFFSET]));
		e->flags |= (1 << NFTNL_EXPR_HASH_OFFSET);
	}

	return ret;
}

static int nftnl_expr_hash_json_parse(struct nftnl_expr *e, json_t *root,
				      struct nftnl_parse_err *err)
{
#ifdef JSON_PARSING
	uint32_t sreg, dreg, len, modulus, seed, offset;

	if (nftnl_jansson_parse_reg(root, "sreg", NFTNL_TYPE_U32,
				    &sreg, err) == 0)
		nftnl_expr_set_u32(e, NFTNL_EXPR_HASH_SREG, sreg);

	if (nftnl_jansson_parse_reg(root, "dreg", NFTNL_TYPE_U32,
				    &dreg, err) == 0)
		nftnl_expr_set_u32(e, NFTNL_EXPR_HASH_DREG, dreg);

	if (nftnl_jansson_parse_val(root, "len", NFTNL_TYPE_U32,
				    &len, err) == 0)
		nftnl_expr_set_u32(e, NFTNL_EXPR_HASH_LEN, len);

	if (nftnl_jansson_parse_val(root, "modulus", NFTNL_TYPE_U32,
				    &modulus, err) == 0)
		nftnl_expr_set_u32(e, NFTNL_EXPR_HASH_MODULUS, modulus);

	if (nftnl_jansson_parse_val(root, "seed", NFTNL_TYPE_U32,
				    &seed, err) == 0)
		nftnl_expr_set_u32(e, NFTNL_EXPR_HASH_SEED, seed);

	if (nftnl_jansson_parse_val(root, "offset", NFTNL_TYPE_U32,
				    &offset, err) == 0)
		nftnl_expr_set_u32(e, NFTNL_EXPR_HASH_OFFSET, offset);

	return 0;
#else
	errno = EOPNOTSUPP;
	return -1;
#endif
}

static int
nftnl_expr_hash_snprintf_default(char *buf, size_t size,
				 const struct nftnl_expr *e)
{
	struct nftnl_expr_hash *hash = nftnl_expr_data(e);
	int len = size, offset = 0, ret;

	ret = snprintf(buf, len, "reg %u = jhash(reg %u, %u, 0x%x) %% mod %u ",
		       hash->dreg, hash->sreg, hash->len, hash->seed,
		       hash->modulus);
	SNPRINTF_BUFFER_SIZE(ret, size, len, offset);

	if (hash->offset) {
		ret = snprintf(buf + offset, len, "offset %u ", hash->offset);
		SNPRINTF_BUFFER_SIZE(ret, size, len, offset);
	}

	return offset;
}

static int nftnl_expr_hash_export(char *buf, size_t size,
				  const struct nftnl_expr *e, int type)
{
	struct nftnl_expr_hash *hash = nftnl_expr_data(e);

	NFTNL_BUF_INIT(b, buf, size);

	if (e->flags & (1 << NFTNL_EXPR_HASH_SREG))
		nftnl_buf_u32(&b, type, hash->sreg, SREG);
	if (e->flags & (1 << NFTNL_EXPR_HASH_DREG))
		nftnl_buf_u32(&b, type, hash->dreg, DREG);
	if (e->flags & (1 << NFTNL_EXPR_HASH_LEN))
		nftnl_buf_u32(&b, type, hash->len, LEN);
	if (e->flags & (1 << NFTNL_EXPR_HASH_MODULUS))
		nftnl_buf_u32(&b, type, hash->modulus, MODULUS);
	if (e->flags & (1 << NFTNL_EXPR_HASH_SEED))
		nftnl_buf_u32(&b, type, hash->seed, SEED);
	if (e->flags & (1 << NFTNL_EXPR_HASH_OFFSET))
		nftnl_buf_u32(&b, type, hash->offset, OFFSET);

	return nftnl_buf_done(&b);
}

static int
nftnl_expr_hash_snprintf(char *buf, size_t len, uint32_t type,
			 uint32_t flags, const struct nftnl_expr *e)
{
	switch (type) {
	case NFTNL_OUTPUT_DEFAULT:
		return nftnl_expr_hash_snprintf_default(buf, len, e);
	case NFTNL_OUTPUT_XML:
	case NFTNL_OUTPUT_JSON:
		return nftnl_expr_hash_export(buf, len, e, type);
	default:
		break;
	}
	return -1;
}

static bool nftnl_expr_hash_cmp(const struct nftnl_expr *e1,
				const struct nftnl_expr *e2)
{
       struct nftnl_expr_hash *h1 = nftnl_expr_data(e1);
       struct nftnl_expr_hash *h2 = nftnl_expr_data(e2);
       bool eq = true;

       if (e1->flags & (1 << NFTNL_EXPR_HASH_SREG))
               eq &= (h1->sreg == h2->sreg);
       if (e1->flags & (1 << NFTNL_EXPR_HASH_DREG))
               eq &= (h1->dreg == h2->dreg);
       if (e1->flags & (1 << NFTNL_EXPR_HASH_LEN))
               eq &= (h1->len == h2->len);
       if (e1->flags & (1 << NFTNL_EXPR_HASH_MODULUS))
               eq &= (h1->modulus == h2->modulus);
       if (e1->flags & (1 << NFTNL_EXPR_HASH_SEED))
               eq &= (h1->seed == h2->seed);
	if (e1->flags & (1 << NFTNL_EXPR_HASH_OFFSET))
		eq &= (h1->offset == h2->offset);

       return eq;
}

struct expr_ops expr_ops_hash = {
	.name		= "hash",
	.alloc_len	= sizeof(struct nftnl_expr_hash),
	.max_attr	= NFTA_HASH_MAX,
	.cmp		= nftnl_expr_hash_cmp,
	.set		= nftnl_expr_hash_set,
	.get		= nftnl_expr_hash_get,
	.parse		= nftnl_expr_hash_parse,
	.build		= nftnl_expr_hash_build,
	.snprintf	= nftnl_expr_hash_snprintf,
	.json_parse	= nftnl_expr_hash_json_parse,
};
