// flit.cpp -- Flit is a "tray" with clock, sound, and battery applets

#define APP_VER "1.5.2" // Last update 2025-03-28
#define PROG    "flit"
// Version 1.5.2 - M. Losh - Repainting before running wireless operations
// Version 1.5.1 - M. Losh - Remove custom screen scaling code now that FLTK 1.4 has that built-in
// Version 1.5.0 - M. Losh - Revise battery power reading, allow longer 'go wireless' commands 
// Version 1.4.5 - M. Losh - Added a capability to defer external processes (want to give time for menu to close, but it does not work yet)
// Version 1.4.4 - M. Losh - Check that wireless tools is installed if the wireless meter is enabled  
// Version 1.4.3 - M. Losh - Make double-click not move frame so easily  
// Version 1.4.2 - M. Losh - Double-click in wireless applet invokes manage connections; conf for max 'expected' quality  
// Version 1.4.1 - M. Losh - Made wireless commands persistent/editable  
// Version 1.4.0 - M. Losh - Added WiFi signal meter applet    

// Version 1.3.2 Added: saved_volume variable by Rich
//				Updated: get_config and save_config to allow saving and restoring
//						 volume level between sessions.
//				Rewrote: ALSA and newvol Audio level calculations to round instead of truncate.
//						 Old versions are commented out with //RR, new versions in following line.
//				Updated: Lowered volume stepsize to 3% from 5% in do_sound_adj to fit better in
//						 ALSAs 5 bit (0-31) volume level range. 

// Version 1.3.0 - M. Losh - Released with changes to default values of custom colors
//                           and to the "About" text    

/* Version pre-1.3.0  Jakob Bysewski <jbysewski@googlemail.com> 
 *              * ALSA is now working with USB soundcards without a hardware mixer using an alsa virtual softvol device
 *              * flit prevents itself from starting multiple instances using a pidfile
 *              * flit can be made dockable
 *              * flit can be made to appear on all desktops (experimental - to be tested)
 *              * flit got command line parameters using argp
 *              * flit can be passed an optional config file to be used
 *              * support for custom fg and bg colors in config
 */
// 

// Version 1.2.2 Fix ALSA bug when selected parameter is Mono

// Version 1.2.1 Fix OSS setsound bug and diag message for OSS setup

// Version 1.2.0 Rounded up version number for public release

// Version 1.1.7 Made size scaling work with floating-point factor

// Version 1.1.6 Support size scaling, fixed hide/show bug

// Version 1.1.5 Compile-time option for OLPC hardware battery support

// Version 1.1.0 ALSA sound control support

// Version 1.0.0 Let's call this major level done!  No real code changes.

// Version 0.9.9 Handle case where battery rate is "unknown" (or some other non-numeric)
//               by allowing calc to proceed using derived rate (as introduced in ver 0.9.3/0.9.4)

// Version 0.9.8 Made the automatic pop-open of menu with Ctrl optional, 
//               Start in unfocused input state

// Version 0.9.7 Removed Esc keybinding to open menu, added '%' to batt. icon

// Version 0.9.6 Changed format of % statements in tooltips and diag. output
//               Changed Alt+(key) bindings to Ctrl+(key)

// Version 0.9.5 Prevent unhiding sound control if no sound system available,
//               Alt key automatically opens menu (cheap way to improve key-binding)
//               Surpress charge/discharge time estimates for 3 minutes to build up data

// Version 0.9.4 Fixed scaling factors used in effective charging/discharging rate

// Version 0.9.3 Added effective charging/discharging rate calculation in cases 
//               when ACPI does not provide one.

// Version 0.9.2 added support for keyboard shortcuts to open right-click menu
//               and to control sound using arror keys, or +/- keys, or through menu.
//               Also, the Pause key will mute/unmute sound.

// Version 0.9.1 added support for all (hopefully) likely OSS volume parameter types
//               and a new .flit.conf param: oss_control_name to favor a specific
//               OSS control parameter, (e.g. pcm, vol, or vmix0-outvol)
//               Note: vmix controls will not work unless the vmix device is fully 
//               "attached" to your hardware.

// Version 0.9.0 added support for MIXT_STEREODB OSS parameter type

// Version 0.8.9 rearranged some code to make transparent style work better.

// Version 0.8.8 switches to direct OSS API (ioctl() stuff) for sound management.
//      Hopefully, it will support a wider variety of sound hardware.
//      Also, this version peeks under the actual intended location
//      of the flit window to determine the "transparent" color.


#include <stdio.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <time.h>
#include <sys/types.h>
#include <dirent.h>
#include <errno.h>
#include <argp.h>

#include <FL/Fl.H>
#include <FL/fl_ask.H>
#include <FL/Fl_Box.H>
#include <FL/Fl_Window.H>
#include <FL/Fl_Menu_Item.H>
#include <FL/Fl_Tooltip.H>
//#include <FL/Enumerations.H>
#include <FL/fl_draw.H>
#include <FL/Fl_Help_Dialog.H>
#include <FL/x.H>
#include <X11/Xlib.h>
#include <FL/Fl_SVG_Image.H>
#include "soundcard.h"
#include "pidfile.h"

typedef long long int uint64_t; 

// UNCOMMENT TO SUPPORT OLPC-1 BATTERY MONITOR
//#define OLPC

// UNCOMMENT TO PRINT DIAGNOSTIC MESSAGES
//#define DIAG

// This adds a bit of gloss to the icons
#define VANITY 1

#ifndef MIN
#define MIN(a, b)   (((a) < (b)) ? (a):(b))
#endif

#define METER_W     28
#define SOUND_W     28
#define CLOCK_W     64
#define SIZE_H      20
#define SIZE_W      (METER_W + METER_W + SOUND_W + CLOCK_W)

#define SECONDS_PER_HOUR                3600
#define SAMPLES_PER_HOUR                720
#define MIN_SAMPLES_BEFORE_ESTIMATION   6
#define MAX_SAMPLES 100

#define BATTERY_BASE_DIR "/sys/class/power_supply/BAT0"
#define PATH_TO_FLIT_HELP "/usr/local/share/doc/flit/flit_help.htm"

// argp stuff -----------------------------------------------------------------
#define PROGNAME_VERSION PROG " " APP_VER

const char *argp_program_version = PROGNAME_VERSION;
const char *argp_program_bug_address = "<mlockmoore@gmail.com>";

// Volume level saved in config file.
int saved_volume = -1;

// This structure is used by main to communicate with parse_opt
struct arguments
{
    bool on_all_desktops;       // should the application appear on all desktops
    bool dock;                  // should the application be docked
    bool multiple_instances;    // allow multiple instances
    char *config_file;          // alternative config file to use
};

/*
 * Options. Field 1 in ARGP
 * Order of fields: {NAME, KEY, ARG, FLAGS, DOC}
 */
static struct argp_option options[] =
{
    {"alldesktops", 'a', 0,             0, "Make flit appear on all desktops (experimental)"},
    {"dock",        'd', 0,             0, "Make flit dock by setting _NET_WM_WINDOW_TYPE_DOCK"},
    {"multiple",    'm', 0,             0, "Allow multiple instances of flit"},
    {"config",      'c', "CONFIGFILE",  0, "Use CONFIGFILE instead of ~/.flit.conf"},
    {0}
};

/*
 * Parser. Field 2 in ARGP
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
    struct arguments *arguments = (struct arguments*)state->input;

    switch(key)
    {
        case 'a':
            arguments->on_all_desktops = true;            
            break;
        case 'd':
            arguments->dock = true;
            break;
        case 'm':
            arguments->multiple_instances = true;
            break;
        case 'c':
            arguments->config_file = arg;
            break;
        default:
            return ARGP_ERR_UNKNOWN;
    }

    return 0;
}

/*
 * ARGS_DOC. Field 3 in ARGP
 * Only needed for additional parameters
 */
static char args_doc[] = "";


/*
 * DOC. Field 4 in ARGP
 */
static char doc[] =
"flit -- A 'systray' with clock, sound, wireless, and battery applets";

/*
 * The ARGP structure
 */
static struct argp argp = {options, parse_opt, args_doc, doc};

// end argp stuff -------------------------------------------------------------

const char About_text[] = 
"Flit version %s\n"
"copyright 2010 - 2025 by Michael A. Losh and Jakob Bysewski\n"
"\n"
"Flit is an applet 'tray', with clock, sound control, wireless monitor,\n" 
"and battery monitor.\n"
"Flit is free software: you can redistribute it and/or\n"
"modify it under the terms of the GNU General Public License\n"
"as published by the Free Software Foundation, version 3.\n"
"\n"
"Flit is distributed in the hope that it will be useful,\n"
"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n"
"See http://www.gnu.org/licenses/ for more details.\n"
"Send bug reports to: mlockmore@@gmail.com";


char Wireless_Enable_cmd[256]  = "sudo /usr/local/sbin/iwconfig %s txpower auto; sudo /sbin/ifconfig %s up";
char Wireless_Disable_cmd[256] = "sudo /sbin/ifconfig %s down; sudo iwconfig %s txpower off";
char Wireless_Online_cmd[256]  = "sudo /usr/local/sbin/iwconfig %s essid any; sleep 5; sudo /sbin/udhcpc -i %s\"";
char Wireless_Manage_cmd[256]  = "editor ${HOME}/.flit.conf";

char WirelessDevName[64] = "";

int Base_t_sec;
int Running = 1;
enum {  MI_ABOUT, MI_HELP,  MI_CLOCK, MI_SHOW24HR, 
        MI_SOUND, MI_SOUND_MUTE, MI_SOUND_UP, MI_SOUND_DOWN, 
        MI_WIRELESS, MI_WIRELESS_ONLINE, MI_WIRELESS_DISABLE, MI_WIRELESS_CONNECTIONS, 
        MI_BATTERY, 
        MI_HOTKEY_TOGGLE, MI_NORMAL_STYLE, MI_INVERSE_STYLE, MI_TRANSPARENT_STYLE, MI_CUSTOM_STYLE, 
        MI_SAVE_CONFIG, MI_QUIT};

static void MenuCB(Fl_Widget* window_p, void *userdata);
Fl_Menu_Item right_click_menu[24] = {
    {"&About",                   FL_CTRL+'a', MenuCB, (void*)MI_ABOUT},
    {"&Help",                    FL_CTRL+'h', MenuCB, (void*)MI_HELP,FL_MENU_DIVIDER},
    {"Show/hide cloc&k",         FL_CTRL+'k', MenuCB, (void*)MI_CLOCK},
    {"&Toggle 12/24 hr. disp.",  FL_CTRL+'t', MenuCB, (void*)MI_SHOW24HR, FL_MENU_DIVIDER},
    {"Show/hide soun&d",         FL_CTRL+'d', MenuCB, (void*)MI_SOUND},
    {"&Mute/unmute",             FL_CTRL+'m', MenuCB, (void*)MI_SOUND_MUTE},
    {"&Up the sound",            FL_CTRL+'u', MenuCB, (void*)MI_SOUND_UP},
    {"&Lower the sound",         FL_CTRL+'l', MenuCB, (void*)MI_SOUND_DOWN, FL_MENU_DIVIDER},
    {"Show/hide &wireless",      FL_CTRL+'w', MenuCB, (void*)MI_WIRELESS},
    {"Go &online",               FL_CTRL+'o', MenuCB, (void*)MI_WIRELESS_ONLINE},
    {"Disa&ble wireless",        FL_CTRL+'b', MenuCB, (void*)MI_WIRELESS_DISABLE},
    {"Mana&ge connections",      FL_CTRL+'g', MenuCB, (void*)MI_WIRELESS_CONNECTIONS, FL_MENU_DIVIDER},
    {"Show/hide &battery meter", FL_CTRL+'b', MenuCB, (void*)MI_BATTERY, FL_MENU_DIVIDER},
    {"Toggle Ctrl key menu activation", FL_CTRL+'y', MenuCB, (void*)MI_HOTKEY_TOGGLE},
    {"&Normal style",            FL_CTRL+'n', MenuCB, (void*)MI_NORMAL_STYLE},  
    {"&Inverse style",           FL_CTRL+'i', MenuCB, (void*)MI_INVERSE_STYLE}, 
    {"Trans&parent style",       FL_CTRL+'p', MenuCB, (void*)MI_TRANSPARENT_STYLE},    
    {"&Custom style",            FL_CTRL+'c', MenuCB, (void*)MI_CUSTOM_STYLE, FL_MENU_DIVIDER},    
    {"&Save configuration",      FL_CTRL+'s', MenuCB, (void*)MI_SAVE_CONFIG, FL_MENU_DIVIDER},  
    {"&Quit",                    FL_CTRL+'q', MenuCB, (void*)MI_QUIT},
    {0}
};

uchar  Fg_r;    // Normal forground color, typically black
uchar  Fg_g;
uchar  Fg_b;

uchar  Bg_r;    // Normal background color, often gray
uchar  Bg_g;
uchar  Bg_b;

uchar C_Fg_r = 0x00;    // Custom foreground color from config file
uchar C_Fg_g = 0x00;
uchar C_Fg_b = 0x1f;

uchar C_Bg_r = 0x58;    // Custom background color from config file
uchar C_Bg_g = 0x7d;
uchar C_Bg_b = 0xaa;

uchar  RW_r = 0x40; // X11 root window background color
uchar  RW_g = 0x50;
uchar  RW_b = 0x80;

enum {BATT_STATE_UNK, BATT_STATE_CHARGED, BATT_STATE_CHARGING, BATT_STATE_DISCHARGING};
enum {NORMAL_STYLE = 0, INVERSE_STYLE = 1, TRANSPARENT_STYLE = 2, CUSTOM_STYLE = 3};
enum {LOC_SE, LOC_SW, LOC_NW, LOC_NE, LOC_CUST};

#ifdef OLPC
enum {BATT_SYSTEM_ACPI, BATT_SYSTEM_OLPC};
int BatterySystemType = BATT_SYSTEM_ACPI;

#define OLPC_BATT_CAP_DEVFILE   "/sys/devices/platform/olpc-battery.0/power_supply/olpc-battery/capacity" 
//#define OLPC_BATT_CAP_DEVFILE   "/home/tc/projects/flit/capacity" 

#define OLPC_BATT_AC_DEVFILE   "/sys/devices/platform/olpc-battery.0/power_supply/olpc-ac/online"
//#define OLPC_BATT_AC_DEVFILE   "/home/tc/projects/flit/online"
#endif


int get_simple_devfile_string(const char* devfile_path, char* buf, int buflen)
{
    int fd;
    ssize_t bytes_read;
#ifdef DEBUG
    char buffer[256];
#endif
    fd = open(devfile_path, O_RDONLY);
    if (fd == -1) {
#ifdef DEBUG
		sprintf(buffer, "Error opening device file '%s'", devfile_path);
        perror(buffer);
#endif
        return 0;
    }

    bytes_read = read(fd, buf, buflen-1);
    if (bytes_read == -1) {
#ifdef DEBUG
        sprintf(buffer, "Error reading from device file '%s'", devfile_path);
        perror(buffer);
#endif
        close(fd);
        return 0;
    }

    buf[bytes_read] = '\0'; // Null-terminate the buffer

    //printf("Data read from device: %s\n", buffer);

    close(fd);
    return 1;
}

int get_simple_devfile_uint64(const char* devfile_path, uint64_t& val)
{

	char buf[32];
	uint64_t v;
	if (get_simple_devfile_string(devfile_path, buf, 32)) {
		if (1 == sscanf(buf, "%lld", &v)) {
			val = v;
			return 1;
		}
	}
	return 0;
}

#define BATT_INFO_BASEPATH  "/sys/class/power_supply/BAT0/"

#define BATT_PERCENT_DEVFILE      BATT_INFO_BASEPATH "capacity"     // % of feasible charge

#define BATT_CHARGE_FULL_DEVFILE  BATT_INFO_BASEPATH "charge_full"  // microamp hours
#define BATT_CHARGE_NOW_DEVFILE   BATT_INFO_BASEPATH "charge_now"   // microamp hours
#define BATT_CURRENT_NOW_DEVFILE  BATT_INFO_BASEPATH "current_now"  // microamps

#define BATT_ENERGY_FULL_DEVFILE  BATT_INFO_BASEPATH "energy_full"  // microwatt hours
#define BATT_ENERGY_NOW_DEVFILE   BATT_INFO_BASEPATH "energy_now"   // microwatt hours
#define BATT_POWER_NOW_DEVFILE    BATT_INFO_BASEPATH "power_now"    // microwatts
 
#define BATT_VOLTAGE_NOW_DEVFILE  BATT_INFO_BASEPATH "voltage_now"  // microvolts
#define BATT_STATUS_DEVFILE       BATT_INFO_BASEPATH "status"		// Charging, Full, Discharging

int get_batt_percentage(const char* batt_percent_devfile_path)
{
	uint64_t val;
	if (get_simple_devfile_uint64(batt_percent_devfile_path, val)) {
		int pcnt = (int)val;
		return pcnt;
	}
	return 0;
}

/*
float get_batt_current_uAmps(const char* batt_info_base_devfile_path)
{
	static int have_current_devfile = 1; // assume it does
	char fullpath[256];
	uint64_t uAmps = 0;
	if (have_current_devfile) {
		strcpy(fullpath, batt_info_base_devfile_path);
		strcat(fullpath, BATT_CURRENT_NOW_DEVFILE);
		if (get_simple_devfile_uint64(fullpath, uAmps)) {
			printf("Battery discharging at %1.3f amps draw rate\n", (float)uAmps / 1.0e6 );
			return (float)uAmps;
		}
		else {
			have_current_devfile = 0; // remember so we don't try again
		}
	}
	// getting current directly didn't work... try to get power and voltage and then calculate current
	uint64_t uWatts = 0;
	strcpy(fullpath, batt_info_base_devfile_path);
	strcat(fullpath, BATT_POWER_NOW_DEVFILE);
	if (get_simple_devfile_uint64(fullpath, uWatts)) {
		printf("Battery discharging at %lld microwatts power draw rate\n", uWatts);
		uint64_t uVolts = 0;
		strcpy(fullpath, batt_info_base_devfile_path);
		strcat(fullpath, BATT_VOLTAGE_NOW_DEVFILE);
		if (get_simple_devfile_uint64(fullpath, uVolts)) {
			float volts = (float)uVolts / (float)1.0e6;
			float amps = (float)uWatts / (float)uVolts;
			printf("Battery voltage is %1.3f V, so current must be %1.3f Amps\n", volts, amps);
			return amps * 1.0e6; // convert to microamps
		}
	}
	return 0.0; // couldn't figure out a current
}
*/
class Flit_Batt_Meter : public Fl_Box {
    int level;
    int toggle;

  public:
    int show_plug;
        
    Flit_Batt_Meter(int x, int y, int w, int h, const char * l = 0) :
        Fl_Box(x, y, w, h, l), level(0), toggle(0), show_plug(1)
        {};
    int value(void) { return level;};
    void value(int n) {
        level = n; 
        toggle = !toggle;
        redraw(); 
        }
    void  draw() {
        fl_color(FL_FOREGROUND_COLOR);
        if (show_plug) {
            label("");
        }
        Fl_Box::draw();
        if (show_plug) {
            fl_rectf(4, 4 , 10, 2);
            fl_polygon(12,3, 15,2, 15,7, 12,6);
            fl_polygon(15,2, 17,2, 17,7, 15,7);
            fl_line(17, 3, 19, 3);
            fl_line(17, 6, 19, 6);
        }
        if (level < 13  && toggle % 2 == 0 && !show_plug) {
            fl_color(FL_RED);   // outline in red every other update
        }
        fl_rectf(2, 10, 22, 8);
        fl_line (24, 12, 24, 15);
        fl_color(FL_BLUE);
        fl_rectf(3, 11, 20, 6);
        fl_line (23, 13, 23, 14);
#ifdef VANITY
        // A concession to vanity: a bit of highlight
        fl_color(0x80, 0x80, 0xFF); // light blue
        fl_line(4, 12, 22, 12);
#endif      
        if (level > 50) {
            fl_color(FL_GREEN);
        }
        else if (level > 35) {
            fl_color(FL_YELLOW);
        }
        else if (level > 20) {
            fl_color(fl_rgb_color(0xFF, 0xC0, 8));
        }
        else {
            fl_color(FL_RED);
        }
        int charge_w = (level + 2) / 5;     // + 2 does a little rounding
        fl_rectf(3, 11, charge_w, 6);

#ifdef VANITY
        if (level > 40) {
            fl_color(0x80, 0xFF, 0x80); // light green
        }
        else if (level > 20) {
            fl_color(0xFF, 0xFF, 0x80); // light yellow
        }
        else {
            fl_color(0xFF, 0x80, 0x80); // light red

        }
        fl_line(3, 12, (2 + charge_w), 12);
#endif
    }

};

class Wireless_Meter : public Fl_Box {
	int signal; // percentage!
	int ap;
	int connected;
    char tip_txt[16];
  public:
	Wireless_Meter(int x, int y, int w, int h);
	void set_ap(int new_ap);
	void set_signal_percent(int new_signal);
	void set_connected(int new_con);
	//void set_zoom(float _z);
	void draw(void);
	void rectf(int _x, int _y, int _w, int _h);
};

int LowContrastColor  = 8;
int HighContrastColor = FL_BLACK;
int LowSigColor		  = FL_RED;
int MediumSigColor	  = FL_YELLOW;
int HighSigColor	  = FL_GREEN;

Wireless_Meter::Wireless_Meter(int x, int y, int w, int h) 
 : Fl_Box(x, y, w, h, 0) 
{
	ap = 0;
	connected = 0;
}
void Wireless_Meter::set_ap(int new_ap)  
{ 
	ap = new_ap;  redraw(); 
}

void Wireless_Meter::set_signal_percent(int new_signal)  
{
	signal = new_signal;  redraw(); 
}

void Wireless_Meter::set_connected(int new_con)
{
	connected = new_con;  redraw(); 
}

void Wireless_Meter::rectf(int _x, int _y, int _w, int _h)
{
	// draw relative to the widget's (X, Y) location
	fl_rectf(x() + _x, y() + _y, _w, _h);
}

void Wireless_Meter::draw(void)
{
	
	// erase background
	fl_color(FL_BACKGROUND_COLOR);
	rectf( 0, 0, 28, 20);
	
	//int signal = 37; // testing override
	Fl_Color clr = FL_BLACK;
	if      (signal > 90) clr = FL_GREEN;
	else if (signal > 65) clr = FL_YELLOW;
	else if (signal > 40) clr = fl_rgb_color(0xFF, 0xC0, 0x04); // orange
	else                clr = FL_RED;

	Fl_Box::draw();

	// horiz. signal bar
    fl_color(FL_FOREGROUND_COLOR);
	rectf( 3, 16, 22, 4);
	// signal fills box from bottom up	
    fl_color(clr);
    int bw = (20 * signal)/100;  // bar width
	rectf( 4, 17, bw, 2);

	// WiFi icon
    fl_color(FL_FOREGROUND_COLOR);
	fl_line_style(FL_SOLID | FL_CAP_ROUND, 2.5);

	fl_begin_line();
	fl_curve((x()+4.5),5,  (x()+8),1,  (x()+19),1,  (x()+22.5),5);
	fl_end_line();

	fl_begin_line();
	fl_curve((x()+7),7.5,  (x()+11),4.5,  (x()+16),4.5,  (x()+20),7.5);
	fl_end_line();

	fl_begin_line();
	fl_curve((x()+9.5),10,  (x()+11),8,  (x()+16),8,  (x()+17.5),10);
	fl_end_line();
	
	fl_line_style(FL_SOLID, 1.0);
	fl_begin_polygon();
	fl_circle((x()+14.0), 12.5, 2.0);
	fl_end_polygon();
	
	if (connected) {
		sprintf(tip_txt, "%d %%", signal);
	}
	else if (ap) {
		sprintf(tip_txt, "not connected");
	}
	else {
		sprintf(tip_txt, "not enabled");		
	}
	tooltip(tip_txt);
     
};

class Flit_Sound_Control : public Fl_Box {
    int level;
    char tip_txt[16];
    public:
    int show_plug;
        
    Flit_Sound_Control(int x, int y, int w, int h, const char * l = 0) :
        Fl_Box(x, y, w, h, l), level(91)
        {};
    int value(void) { return level;};
    void value(int n) { 
        level = n; 
        redraw(); 
        sprintf(tip_txt, "Vol. %d", value());
        tooltip(tip_txt);
#ifdef DIAG
        printf("Sound ctrl. new value is %d\n", level);
#endif
    };
    void  draw() {      
        Fl_Box::draw();
        
        fl_color(0xA0, 0xA0, 0xB0); // Medium blue-gray
        fl_rectf(x() + 5, 7, 3, 6); // coil color
        fl_polygon(x()+8,6, x()+12,2, x()+12,18, x()+8,14 );    // horn color
        
        fl_color(FL_FOREGROUND_COLOR);
        fl_rect(x() + 4, 6, 4, 8);  // coil outline
        fl_loop(x()+8,6, x()+12,2, x()+12,18, x()+8,14 );       // horn outline
        fl_loop(x()+13,7, x()+14,9, x()+14,11, x()+13,13 );     // dome outline
        
        fl_color(0x40, 0x40, 0x40); // dark gray
        fl_loop(x()+13,7, x()+14,9, x()+14,11, x()+13,13 );     // dome outline

#ifdef VANITY
        // A concession to vanity: a bit of highlight
        fl_color(0xE0, 0xE0, 0xE0); // light gray
        fl_line(x()+5, 9, x()+7, 9);
        fl_line(x()+9, 7, x()+11, 5); 
#endif       
        if (level < 1 ) {
			fl_line_style(FL_SOLID, 2.0);
            fl_color(FL_RED);   
//            fl_arc(x()+4, 2, 16, 16, 0.0, 359.9);
            fl_arc(x()+5, 3, 14, 14, 0.0, 359.9);
            fl_line(x()+7, 5, x()+16, 14);
//            fl_line(x()+8, 5, x()+17, 14);
        }

		fl_line_style(FL_SOLID, 4.0);
		fl_color(FL_FOREGROUND_COLOR);

		fl_arc(x()+1, 0, 20, 20, 0.0, 45.0);   // sound level arc backing
		fl_arc(x()+1, 0, 20, 20, 315.0, 359.9);

		// some tick marks
		fl_line_style(FL_SOLID, 5.0);
		for (int i = 0; i <= 5; i++) {
			int st = i * 18;
			int en = st + 4;
			if (st >= 45.0) 
				st -= 45.0;
			else  
				st += 313.0;
			if (en >= 45.0) 
				en -= 45.0;
			else
				en += 313.0;
			if (st > en)   st = 0.0;
			fl_arc(x()+0, -1, 22, 22, st, en);
		}

		fl_line_style(FL_SOLID, 2.5);
		fl_color(fl_rgb_color(8, 0xFF, 0xFF));  // cyan color
		
		float deg = (float)level * 90.0 / 100.0;
		if (deg < 45.0) {
			fl_arc(x()+1, 0, 20, 20, 315.0, 315.0 + deg);
		}
		else {
			fl_arc(x()+1, 0, 20, 20, 315.0, 359.9);
			fl_arc(x()+1, 0, 20, 20, 0.0, (deg - 45.0));
		}
				
/*
        fl_color(Bg_r + Fg_r / 2, Bg_g + Fg_g / 2, Bg_b + Fg_b / 2);
        if (level > 0) {
            fl_color(FL_FOREGROUND_COLOR);
            fl_line(x() + 16, 15, x() + 18, 17);
        }
        if (level >= 10) {
            fl_line(x() + 18, 17, x() + 19, 19);
        }
        if (level >= 20) {
            fl_line(x() + 17, 13, x() + 19, 14);
        }
        if (level >= 30) {
            fl_line(x() + 19, 14, x() + 21, 15);
        }       
        if (level >= 40) {
            fl_line(x() + 17, 11, x() + 19, 11);
        }
        if (level >= 50) {
            fl_line(x() + 19, 11, x() + 22, 11);
        }
        if (level >= 60) {
            fl_line(x() + 17, 9, x() + 19, 8);
        }
        if (level >= 70) {
            fl_line(x() + 19, 8, x() + 21, 7);
        }
        if (level >= 80) {
            fl_line(x() + 16, 6, x() + 18, 4);
        }
        if (level >= 90) {
            fl_line(x() + 18, 4, x() + 20, 2);
        }
*/
    }
};

enum { SOUND_TYPE_OSS, SOUND_TYPE_ALSA};

char DeferredCmdStr[1024] = {0};
int  CmdIsDeferred = 1;

void defer(char* cmdstr)
{
    strncpy(DeferredCmdStr, cmdstr, 1023);
    CmdIsDeferred = 1;
}

class Flit_Frame : public Fl_Window {
    protected:
        int     size_w;
        int     size_h;
        int     win_loc;
        int     style;
        int     win_x;
        int     win_y;
        int     dragging;
        char    configfilename[256];
        const char *configfile_option;

        int     show_sound;
        int     sound_type;
        Flit_Sound_Control* sound_p;
        int     prev_sound_vol;
        int     mixer_fd;
        int     mix_ctrl;
        int     mix_min;
        int     mix_max;
        char    sound_control_name[64];
        char    alsa_amixer_path[128];
    
        int     show_clock;
        Fl_Box* clock_p;
        int     show24hr;
        char    timestr[16];
        char    ampm[4];
        time_t  last_update_time;
        
        
        int     show_battery;
        Flit_Batt_Meter* meter_p;
        int     monAmps;
        int     sampleCount;
        int     sample;
        float   hist[MAX_SAMPLES];
/*        float   capacity_mAh;
        float   charge_mAh;
        float   orig_charge_mAh;
        float   avgcur_mA; */
        float   charge_pct;
        float   remaining_hrs;
        int     batt_state;
        char    meter_str[8];
        char    meter_tip[64];
        char    time_tip[64];
        char    batt_state_filename[128];
        
        int		show_wireless;
        int		wireless_installed;
        int		have_warned_for_no_wireless;
        Wireless_Meter* wireless_p;
        char    buf[1024];
        int     max_expected_link_quality;

    public:
        int     menu_hotkey_activation;
        void set_color(Fl_Color c) {
            if (meter_p)    meter_p->labelcolor(c);
            if (sound_p)    sound_p->labelcolor(c);
            if (clock_p)    clock_p->labelcolor(c);
        }
        
        void set_boxtype(Fl_Boxtype boxtype) {
            if (meter_p)    meter_p->box(boxtype);
            if (sound_p)    sound_p->box(boxtype);
            if (clock_p)    clock_p->box(boxtype);
            if (wireless_p) wireless_p->box(boxtype);
            redraw();
        }
        
        Flit_Frame(const char *title = 0, const char *configfile_option = 0) :
          Fl_Window(10,  10,  100,  20, title),
          size_w(0),
          size_h(SIZE_H),
          win_loc(LOC_SE),
          style(NORMAL_STYLE),
          win_x(10),
          win_y(10),
          dragging(0),
          configfile_option(configfile_option),
          show_sound(1),
          sound_type(SOUND_TYPE_OSS),
          sound_p(NULL),
          prev_sound_vol(-1),
          mixer_fd(0),
          mix_ctrl(-1),
          mix_min(0),
          mix_max(100),
          show_clock(1),
          clock_p(NULL),
          show24hr(0),
          last_update_time(0),
          show_battery(1),
          meter_p(NULL),
          sampleCount(0),
          sample(0),
/*          samples(0),
          capacity_mAh(999999.0),
          charge_mAh(999999.0),
          orig_charge_mAh(99999.0),
          avgcur_mA(1),
          charge_pct(100.0), */
          remaining_hrs(9999.0),
          batt_state(BATT_STATE_UNK),
          show_wireless(1),
          wireless_installed(-1),
          have_warned_for_no_wireless(0),
          wireless_p(NULL),
          max_expected_link_quality(-1),
          menu_hotkey_activation(1)
        {
            Base_t_sec = time(NULL);
            strcpy(sound_control_name, "autosel");
        };
        
        void get_time(void) {
          time_t now        = time(NULL);
          struct tm* timep  = localtime(&now);
          if (show24hr) {
              strcpy(ampm, "");
          }
          else {
              if (timep->tm_hour>=12) {
                  strcpy(ampm, " PM");
              }
              else {
                  strcpy(ampm, " AM");
              }
          }
          int hour = timep->tm_hour;
          if (!show24hr) {
              hour = hour % 12;
              if (hour < 1) hour = 12;
          }
          
          sprintf(timestr, "%d:%02d%s", hour, timep->tm_min, ampm);
          clock_p->label(timestr);
          strcpy(time_tip, ctime(&now));
          clock_p->tooltip(time_tip);
          clock_p->redraw();
        };
        
        void get_wifi_dev_name(void) {
			int device = 0;
			FILE* iwconfig_fp = popen("sudo iwconfig", "r");
			if (iwconfig_fp) {
				char devname[64];
				char word2[64];
				char word3[64];
				while (!device && fgets(buf, 1024, iwconfig_fp)) {
					if (3 == sscanf(buf, "%s %s %s",  devname, word2, word3)) {
						if (!strcmp(word2, "IEEE") && strstr(word3, "802.11")) {
							device = 1;
							strcpy(WirelessDevName, devname);
							printf("Flit assumes WiFi device is %s\n", WirelessDevName);
						}
					}
				}
				pclose(iwconfig_fp);
			}
			if (!device) {
				fprintf(stderr, "iwconfig utility not available... ensure wireless_tools is installed!\n");
			}
		}
		
		int wireless_available(void) {
			if (wireless_installed <= 0) {
				int res = system("which iwconfig > /dev/null");
				wireless_installed = !res;	// 'which' returns 1 if not found, 0 if found
				if (!wireless_installed && !have_warned_for_no_wireless) {
					fl_alert("Flit could not locate the wireless_tools utilities.\n"
							 "Please install before showing the wireless monitor.");
					have_warned_for_no_wireless = 1;
				} 
			}
			return wireless_installed;
		}
		
		void get_wireless_info(void) {
			char cmd[256];
			sprintf(cmd, "iwconfig %s", WirelessDevName);
			FILE* cmd_fp = popen(cmd, "r");
			char* p;
			int associated = 0;
			if (cmd_fp) {
				int qual, possible, sig;
				while(fgets(buf, 256, cmd_fp)) {
					//printf("%s: %s\n", __func__, buf);
					p = strstr(buf, "Access Point:");
					if (p != NULL) {
						if (strstr(p, "Not-Associated")) {
							wireless_p->set_ap(0);
							wireless_p->set_signal_percent(0);
							wireless_p->set_connected(0);
							break;
						}
						else {
							wireless_p->set_ap(1);
							associated = 1;
						}
					}
					else if (3 == sscanf(buf, " Link Quality=%d/%d  Signal level=%d", 
                                 &qual, &possible, &sig)) {
						//printf("%s: >>>> associated %d, qual %d, poss %d, percent %d\n", __func__, associated, qual, possible, qual * 100 / possible);
						if (associated) {
                            int percent;
                            if (max_expected_link_quality > 0) {
                                percent = qual * 100 / max_expected_link_quality;
                            }
                            else {
                                percent = qual * 100 / possible;
                            }
							wireless_p->set_signal_percent(percent);
						}
						break;
					}
				}
				pclose(cmd_fp);
			}
			if (associated) {
				sprintf(cmd, "/sbin/ifconfig %s | grep \"inet addr\" > /dev/null", WirelessDevName);
				wireless_p->set_connected(0 == system(cmd));
			}
			wireless_p->redraw();
		}
        
        void get_sound_info(void) {
            int c;
            int success = 0;
            int autosel = 0;    // Whether Flit should pick a control
            oss_mixext info;
            mix_ctrl = 0;
            autosel = 0;
            if (!strcmp(sound_control_name, "autosel") ) {
printf("Will autoselect sound control parameter\n");
                autosel = 1;
            }
            if(!mixer_fd) {
                mixer_fd = open("/dev/mixer", O_RDWR);
                if (-1 != mixer_fd) {
                    oss_mixerinfo mi;
                    mi.dev = -1;
                    if (-1 != ioctl(mixer_fd, SNDCTL_MIXERINFO, &mi)) {
#ifdef DIAG                     
                        printf("Mixer device is '%s', priority %d\n", mi.name, mi.priority);    
#endif              
                    }
                    else {
                        perror("SNDCTL_MIXERINFO ioctl()");
                    }
                    
                    int mix_ctls = 0; 
                    if (-1 != ioctl(mixer_fd, SNDCTL_MIX_NREXT, &mix_ctls)) {
#ifdef DIAG                     
                        printf("There are %d controls in the mixer\n", mix_ctls);
#endif              
                    }       
                    for (c = 0 ; c < mix_ctls ; c++ ) {
                        info.dev = 0;
                        info.ctrl = c;
                        if (-1 != ioctl(mixer_fd, SNDCTL_MIX_EXTINFO, &info)) {
#ifdef DIAG                     
                            printf("--->\t%d\tConsidering volume control item: '%s', type %d, flags 0x%X\n", 
                                c, info.extname, info.type, info.flags);
#endif              
                            if (    !success 
                                &&  (   !strcmp(info.extname, sound_control_name)  // found the specified OSS control?
                                    ||  (   autosel == 1
                                        &&  (   (info.flags & MIXF_MAINVOL)             // OSS thinks it can control volume
                                            ||  (info.flags & MIXF_PCMVOL)              // ''
                                            )
                                        )
                                    )
                                ) 
                            {
                                success = 1;
                                mix_ctrl = c;   
                                mix_max = info.maxvalue;
                                mix_min = info.minvalue;
#ifdef DIAG                     
                                printf("===>\t%d\tSelected volume control item: '%s'\n", c, info.extname);
                                printf("===>\t%d\tmin %d, max %d,\n", c, mix_min, mix_max);
#else                               
                                break; // quit after the first one that seems to match
#endif              
                            }
#ifdef DIAG                     
                            else { 
                                printf("\t\t\t(not selected)\n");
                            }
#endif              
                        }
                        else 
                            perror("SNDCTL_MIX_EXTINFO ioctl()");
                        
                    } // end for
                    
                }
#ifdef DIAG
                else {
                    perror("/dev/mixer open()");
                    printf("\n");
                }
#endif                
            }
            
            if (!success) {
                // Check for ALSA sound
#ifdef DIAG
                printf("Looking for ALSA sound...\n");
#endif
                char potential_sound_control[48];
                FILE* which_pf = 0;
                *alsa_amixer_path = '\0';
                *potential_sound_control = '\0';
                which_pf = popen("which amixer", "r");
                if(fgets(alsa_amixer_path, 127, which_pf)) {
#ifdef DIAG
                    printf("AMIXER path is %s\n", alsa_amixer_path);
#endif
                }
                pclose(which_pf);

                if (strlen(alsa_amixer_path) > 1) {
                    sound_type = SOUND_TYPE_ALSA;
                                                
                    FILE* amixerpf = NULL;
                    char cmd[128];
                    char buf[512];
                    char *s;
                    int need_autosel = autosel;
                    int need_lims = 1;
                    //int newval;
                    int minval = -1;
                    int maxval = -1;
                    if (need_autosel) {
                        //sprintf(cmd, "amixer", sound_control_name);
                        strcpy(cmd, "amixer scontents");
                    }
                    else {
                        sprintf(cmd, "amixer sget %s", sound_control_name);
                    }
                    amixerpf = popen(cmd, "r");
                    if (amixerpf) {
                        while(!feof(amixerpf) && fgets(buf, 511, amixerpf)) {
                            // look for limits
                            s = strstr(buf, "Limits:");
                            if (need_lims && s && (2 == sscanf(s, "Limits: Playback %d - %d", &minval, &maxval) || 2 == sscanf(s, "Limits: %d - %d", &minval, &maxval))) {
                                mix_min = minval;
                                mix_max = maxval;
#ifdef DIAG
                                printf("Sound volume ranges from %d to %d\n", mix_min, mix_max);
#endif
                                success = 1;
                                need_lims = 0;

                                continue; 
                            }
                            s = strstr(buf, "Simple mixer control");
                            if (need_autosel && s && 1 == sscanf(s, "Simple mixer control '%[^']", potential_sound_control)) {
#ifdef DIAG
                                printf("Considering potential sound control '%s'\n",  potential_sound_control);
#endif
                                continue;
                            }

                            //s = strstr(buf, "pvolume");
                            s = strstr(buf, "volume");
                            if (need_autosel && s && strlen(potential_sound_control) > 0) {
                                strcpy(sound_control_name, potential_sound_control);
#ifdef DIAG
                                printf("Accepted sound countrol '%s'\n", sound_control_name);
#endif
                                need_autosel = 0;
                                continue;
                            }
                            else {
                                *potential_sound_control = '\0'; // blank any current candidate out
                                continue;
                            }

                        }
                        pclose(amixerpf);
                    }
                }
                else {
#ifdef DIAG
                    printf("ALSA amixer not found!\n");
#endif
                }
            }
                
            if (!success) {
                printf("No sound found!\n");
                show_sound = 0;
                mixer_fd = 0;
            }
        }

        void get_sound_level(void) {
            oss_mixext info;
            oss_mixer_value vr;
            int pct;
            int rawval;
            if (sound_type == SOUND_TYPE_OSS && mixer_fd && mix_ctrl) {
                info.dev = 0;
                info.ctrl = mix_ctrl;               
                if (-1 != ioctl(mixer_fd, SNDCTL_MIX_EXTINFO, &info)) {
                    vr.dev=info.dev;
                    vr.ctrl=info.ctrl;
                    vr.timestamp=info.timestamp;
                    if (-1 != ioctl(mixer_fd, SNDCTL_MIX_READ, &vr)) {
                        switch(info.type) {
                            case MIXT_MONOSLIDER:
                            case MIXT_STEREOSLIDER:
                                rawval = vr.value & 0xFF;
                                break;
                            case MIXT_MONOSLIDER16:
                            case MIXT_STEREOSLIDER16:
                                rawval = vr.value & 0x7FFF;
                                break;                              
                            case MIXT_MUTE:
                                rawval = (vr.value) ? mix_max : mix_min;
                                break;
                            case MIXT_SLIDER:   // 32-bit
                            default:            // Maybe MIXT_VALUE, MIXT_HEXVALUE, MIXT_MONODB, MIXT_STEREODB
                                rawval = vr.value;
                                break;
                        }
                                
                        pct = (int)(100.0 * (float)(rawval - mix_min) / (float)mix_max);                        
#ifdef DIAG                     
                        printf("Read 0x%08X from control, raw volume is %d or %d %%\n", 
                            vr.value, rawval, pct);
#endif              
                        sound_p->value(pct);
                    }
                    else 
                        perror("SNDCTL_MIX_READ ioctl()");
                }
                else 
                    perror("SNDCTL_MIX_EXTINFO ioctl()");
            }
            else if (sound_type == SOUND_TYPE_ALSA) {
                FILE* amixerpf = NULL;
                char buf[512];
                char *s;
                int lvol = -1;
                int rvol = -1;
                int v;
                char cmd[128];
                sprintf(cmd, "amixer sget %s", sound_control_name);
                amixerpf = popen(cmd, "r");
                if (amixerpf) {
                    while(!feof(amixerpf) && fgets(buf, 511, amixerpf)) {
                        //s = strstr(buf, "Front Left: Playback");
                        s = strstr(buf, "Front Left:");
                        if (s && (1 == sscanf(s, "Front Left: Playback %d", & v) || 1 == sscanf(s, "Front Left: %d", & v))) {
                            lvol = v;
                            //continue;
                        }
                        //s = strstr(buf, "Front Right: Playback");
                        s = strstr(buf, "Front Right:");
                        if (s && (1 == sscanf(s, "Front Right: Playback %d", & v) || 1 == sscanf(s, "Front Right: %d", & v))) {
                            rvol = v;
                        }
                        if (lvol >= 0 && rvol >= 0 && (mix_max > mix_min)) {
//RR                            pct = (int)(((float)(rvol + lvol)) * 100.0 / (2.0 * (float)(mix_max - mix_min)));
						pct = (int)(roundf(((rvol + lvol) * (100.0 / ((float)(mix_max - mix_min) * 2.0)))));
#ifdef DIAG
                            printf("Sound level is %d %%\n", pct);
#endif
                            sound_p->value(pct);
                            break;
                        }
                    }
                    pclose(amixerpf);
                }
            }
            else {
#ifdef DIAG  
                printf("No sound device or suitable control\n");
#endif              
            }
        };
        
        void set_sound_level(int vol) {
            oss_mixext info;
            oss_mixer_value vr;
            int newvol;
            int intended;
//RR            newvol = (int)((float)(vol * mix_max) / 100.0) - mix_min;
            newvol = (int)(roundf((vol * ((float)(mix_max - mix_min) / 100.0))));
#ifdef DIAG
            printf("vol=%d newvol=%d mix_max=%d mix_min=%d\n", vol, newvol, mix_max, mix_min);
#endif
            if (sound_type == SOUND_TYPE_OSS && mixer_fd && mix_ctrl) {
                info.dev = 0;
                info.ctrl = mix_ctrl;               
                if (-1 != ioctl(mixer_fd, SNDCTL_MIX_EXTINFO, &info)) {
                    vr.dev=info.dev;
                    vr.ctrl=info.ctrl;
                    vr.timestamp=info.timestamp;
                    switch(info.type) {
                        case MIXT_MONOSLIDER:
                        case MIXT_STEREOSLIDER:
                            vr.value = (newvol << 8) | newvol;
                            break;
                        case MIXT_MONOSLIDER16:
                        case MIXT_STEREOSLIDER16:
                            vr.value = (newvol << 16) | newvol;
                            break;                              
                        case MIXT_MUTE:
                            vr.value = (newvol) ? mix_max : mix_min;
                            break;
                        case MIXT_SLIDER:   // 32-bit
                        default:            // Maybe MIXT_VALUE, MIXT_HEXVALUE, MIXT_MONODB, MIXT_STEREODB
                            vr.value = newvol;
                            break;
                    }
                    intended = vr.value;        
                    if (-1 != ioctl(mixer_fd, SNDCTL_MIX_WRITE, &vr)) {
#ifdef DIAG                     
                        printf("Written value is 0x%08X, intended value 0x%08X (for %d %%)\n", 
                                vr.value, intended, vol);
#endif              
                        if (vr.value == intended) {
                            sound_p->value(vol);
                        }
                        else {
                            // find out what the value is, actually
                            get_sound_level();
                        }
                    }
                    else 
                        perror("SNDCTL_MIX_WRITE ioctl()");
                }
                else 
                    perror("SNDCTL_MIX_EXTINFO ioctl()");
            }
            else if (sound_type == SOUND_TYPE_ALSA) {
                FILE* amixerpf = NULL;
                char cmd[128];
                char buf[512];
                char *s;
                //int minval, maxval;
                int lvol = -1;
                int rvol = -1;
                int v;
                int actual_pct;
                sprintf(cmd, "amixer sset %s %d", sound_control_name, newvol);
#ifdef DIAG
                printf("********************\nCmd for intended %d pct: '%s'\n", vol, cmd);
#endif
                amixerpf = popen(cmd, "r");
                if (amixerpf) {
                    while(!feof(amixerpf) && fgets(buf, 511, amixerpf)) {
                        s = strstr(buf, "Mono: Playback");
                        if (s && 1 == sscanf(s, "Mono: Playback %d", & v)) {
                            lvol = v;
                            rvol = v;
                            continue;
                        }
                        //s = strstr(buf, "Front Left: Playback");
                        s = strstr(buf, "Front Left:");
                        if (s && (1 == sscanf(s, "Front Left: Playback %d", & v) || 1 == sscanf(s, "Front Left: %d", & v))) {
                            lvol = v;
                            continue;
                        }
                        //s = strstr(buf, "Front Right: Playback");
                        s = strstr(buf, "Front Right:");
                        if (s && (1 == sscanf(s, "Front Right: Playback %d", & v) || 1 == sscanf(s, "Front Right: %d", & v))) {
                            rvol = v;
                            continue;
                        }
                    }
                    if (lvol >= 0 && rvol >= 0 && (mix_max > mix_min)) {
                        actual_pct = (rvol + lvol) * 100.0 / (2.0 * (mix_max - mix_min));
#ifdef DIAG
                        printf("Sound level after change is actually %d %%\n", actual_pct);
#endif
                        sound_p->value(actual_pct);
                    }
                    pclose(amixerpf);
                }
               
            }
            else {

#ifdef DIAG                     
                printf("No sound device or suitable control\n");
#endif              
            }
        };

        void get_battery_info(void) {
			// determine whether we have have amps & amp hours "charge" info or should use 
			//   power flow and overall energy
			
			char buf[256];
			
			if (   get_simple_devfile_string(BATT_CURRENT_NOW_DEVFILE, buf, 256)
				&& get_simple_devfile_string(BATT_CHARGE_NOW_DEVFILE, buf, 256) ) 
			{
				monAmps = 1;
				printf("Battery monitoring in terms of amps and amp hours stored\n");
			}
			else if (	get_simple_devfile_string(BATT_POWER_NOW_DEVFILE, buf, 256)
					 && get_simple_devfile_string(BATT_ENERGY_NOW_DEVFILE, buf, 256) )
			{
				monAmps = 0;
				printf("Battery monitoring in terms of Watts power draw and Watt hours stored\n");
			} 
            else {
                meter_p->label("\nAC");
                meter_p->show_plug = 1;
            }
        };

        void update_battery_meter(void) {
//                int hours = (int)floor(remaining_hrs);
                int mins  = (int)round(remaining_hrs * 60.0);
                sprintf(meter_str, "%1.f%%\n-",charge_pct);
                if (batt_state == BATT_STATE_CHARGED) {
                    strcpy(meter_tip, "Full");
                }
                else if (sampleCount < MIN_SAMPLES_BEFORE_ESTIMATION) {
                    sprintf(meter_tip, "%1.f %%", charge_pct);
                }
                else if (remaining_hrs > 1.0) {
                    sprintf(meter_tip, "%1.f %%, ~ %1.1f hrs %s", 
                            charge_pct, remaining_hrs,
                            (batt_state == BATT_STATE_CHARGING) ? "to charge":"remaining"
                           );
                }
                else {
                    sprintf(meter_tip, "%1.f %%, ~ %d minute%s %s", 
                            charge_pct, mins, (mins==1) ? "":"s", 
                            (batt_state == BATT_STATE_CHARGING) ? "to charge":"remaining"
                           );
                }
                meter_p->label(meter_str);
                meter_p->tooltip(meter_tip);
                meter_p->value((int)floor(charge_pct));
                meter_p->redraw();
                return;
        }

        void get_battery_state(void) {
			charge_pct = get_batt_percentage(BATT_PERCENT_DEVFILE);
			char statestr[32];
			if (get_simple_devfile_string(BATT_STATUS_DEVFILE, statestr, 32)) {
				int old_batt_state = batt_state;
				if      (!strncmp(statestr, "Discharging", 11))  batt_state = BATT_STATE_DISCHARGING;
				else if (!strncmp(statestr, "Charging", 8))      batt_state = BATT_STATE_CHARGING;
				else if (!strncmp(statestr, "Full", 4))          batt_state = BATT_STATE_CHARGED;
				else 					   				         batt_state = BATT_STATE_UNK;
				if (old_batt_state != batt_state) {
					sample      = 0;
					sampleCount = 0;
				}
			}
			else {
				sample      = 0;
				sampleCount = 0;
				batt_state  = BATT_STATE_UNK;
				meter_p->label("??");
				return;
			}
			uint64_t AC_online = 0;
			get_simple_devfile_uint64("/sys/class/power_supply/AC/online", AC_online);
#ifdef DIAG			
			printf( "Flit battery state code is %d, AC is %sonline\n", batt_state, (AC_online) ? "" : "NOT ");
#endif
			uint64_t rate = 0;
			if (get_simple_devfile_uint64((monAmps) ? BATT_CURRENT_NOW_DEVFILE : BATT_POWER_NOW_DEVFILE, rate)) 
			{ 
				hist[sample] = (float)rate;
				sampleCount++;
				if (sampleCount > MAX_SAMPLES) {
					sampleCount = MAX_SAMPLES;
					sample = 0; // return to first slot (to support calc of a moving average of MAX_SAMPLES)
				}
				else {
					sample++; // bump to next slot
				}
				uint64_t store = 0; 
				if(get_simple_devfile_uint64((monAmps) ? BATT_CHARGE_NOW_DEVFILE : BATT_ENERGY_NOW_DEVFILE, store)) 
				{
					float storeAmount = (float)store;
					float avgRate = 0.0;
					for (int i = 0; i < sampleCount; i++) {
						avgRate += hist[i];  // tally
					}
					avgRate = avgRate / (float)sampleCount; // take average
					if (batt_state == BATT_STATE_DISCHARGING && avgRate > 1.0) {
						remaining_hrs = storeAmount / avgRate;
						meter_p->show_plug = 0;
					}
					else if (batt_state == BATT_STATE_CHARGING && avgRate > 1.0) {
						uint64_t full = 0; 
						if(get_simple_devfile_uint64((monAmps) ? BATT_CHARGE_FULL_DEVFILE : BATT_ENERGY_FULL_DEVFILE, full)) {
							float fullAmount = (float)full;
							remaining_hrs = (fullAmount - storeAmount) / avgRate;
#ifdef DIAG			
							printf("Full amount is %1.3f, current amount is %1.3f, gap is %1.3f; rate is %1.3f, time is %1.3f\n",
									fullAmount/1.0e6, storeAmount/1.0e6, (fullAmount - storeAmount)/1.0e6, avgRate/1.0e6, remaining_hrs);
#endif
						}
						meter_p->show_plug = 1;
					}
					else if (AC_online > 0) {
						meter_p->show_plug = 1;
						remaining_hrs = 9999.0;
					}
					else {
						remaining_hrs = 9999.0;
					}
#ifdef DIAG			
					if ((sample % 4) == 1) {
						if (monAmps) {
							printf("%s: sample %d (%d total); current %1.3f Amps inst, %1.3f Amps avg; holding %1.3f Amp Hrs; est. %1.2f hours remaining\n",
									statestr, sample, sampleCount, rate / 1.0e6, avgRate / 1.0e6, storeAmount / 1.0e6, remaining_hrs);
						}
						else {
							printf("%s: sample %d (%d total); power %1.3f Watts inst, %1.3f Watts avg; holding %1.3f Watt Hrs; est. %1.2f hours remaining\n",
									statestr, sample, sampleCount, rate / 1.0e6, avgRate / 1.0e6, storeAmount / 1.0e6, remaining_hrs);
						}
					}
#endif
				}
			}
			meter_p->show_plug = (batt_state == BATT_STATE_DISCHARGING)  ?  0 : 1; 
			update_battery_meter();
            meter_p->redraw();
        };
        
        void toggle_show24hr(void) {
          show24hr = !show24hr;
          get_time();
          redraw();
        };
        
        void do_wireless_manage(void) {
            char cmd[256];
			sprintf(cmd, Wireless_Manage_cmd, WirelessDevName, WirelessDevName);
			printf("trying command: '%s'\n", cmd);
			defer(cmd);
			get_wireless_info();
			update();
        }
        
        void do_menu(void) {
            const Fl_Menu_Item *m = right_click_menu->popup(Fl::event_x(), Fl::event_y() - 20, "Flit", 0, 0);
            if ( m ) m->do_callback(this, m->user_data());
        };
        
        // -1 for quieter, +1 for louder, 0 for toggle
        void do_sound_adj(int direction) {
            int vol;
            vol = sound_p->value();
            if (direction < 0) {
                prev_sound_vol = vol;
                vol -= 3;
                if (vol < 0) vol = 0;
            }
            else if (direction > 0) {
                prev_sound_vol = vol;
                vol += 3;
                if (vol > 100) vol = 100;
            }
            else {
                // direction is 0, so toggle
                if (sound_p->value() > 0) {
                    prev_sound_vol = vol;
                    vol = 0;
                }
                else {
                    vol = prev_sound_vol;
                }
            }
            set_sound_level(vol);
            sound_p->redraw();
            Fl_Tooltip::delay(0);       // force tooltip on
            Fl_Tooltip::exit(sound_p);
            Fl_Tooltip::enter(sound_p);
            Fl_Tooltip::delay(1);
        }
        
        int handle(int e) {
            static int off_x = 0;
            static int off_y = 0;
            int key;
            switch ( e ) {
            case FL_KEYDOWN:
                key = Fl::event_key();
                if (menu_hotkey_activation && (key == FL_Control_L || key == FL_Control_R)) {
                    do_menu();
                    return 1;
                }
                break;
            case FL_KEYUP:
                key = Fl::event_key();
                if (show_sound &&  (key == FL_Up || key == '=')) {  // '=' is also + key                
                    do_sound_adj(+1);
                    return 1;
                }
                else if (show_sound && (key == FL_Down || key == '-')) {
                    do_sound_adj(-1);
                    return 1;
                }
                else if (show_sound && key == FL_Pause) {
                    do_sound_adj(0);
                    return 1;
                }
                break;
            case FL_PUSH:
                if (Fl::event_button() == FL_LEFT_MOUSE) {
                    if (    show_sound && Fl::event_clicks()
                        &&  Fl::event_x() > sound_p->x() 
                        &&  Fl::event_x() < (sound_p->x() + SOUND_W)) 
                    {
						dragging = 0;
                        do_sound_adj(0); // zero means toggle   
                        return 1;
                    }
                    else if (   show_wireless && Fl::event_clicks()
                             && Fl::event_x() > wireless_p->x() 
                             && Fl::event_x() < (wireless_p->x() + METER_W)) 
                    {
						dragging = 0;
                        do_wireless_manage();   
                        return 1;
                    }
                    else {
						dragging = 1;
                        off_x = Fl::event_x();
                        off_y = Fl::event_y();
                    }
                }
                else if (Fl::event_button() == FL_RIGHT_MOUSE ) {
                    do_menu();
                    return(1);          // (tells caller we handled this event)
                }   
                break;
            case FL_DRAG:
                if ( dragging && Fl::event_button() == FL_LEFT_MOUSE ) {
                    win_x = Fl::event_x_root() - off_x;
                    win_y = Fl::event_y_root() - off_y;
                    position(win_x, win_y);
                    win_loc = LOC_CUST;
                    redraw();
                    return 1;
                } 
                break; 
            case FL_MOUSEWHEEL:
                if (show_sound && Fl::event_x() > sound_p->x() 
                    &&  Fl::event_x() < (sound_p->x() + SOUND_W)) {
                    do_sound_adj(-1 * Fl::event_dy());
                    return 1;
                }
            default:
                break;
            }
            return(Fl_Window::handle(e));
        };
        void update(void) {
            time_t now      = time(NULL);
            if (now - last_update_time >= 5) {
                if (show_clock)     get_time();
                if (show_sound)     get_sound_level();
                if (show_wireless)  {
					if (wireless_available()) {
						get_wireless_info();
					}
					else {
						show_wireless = 0;
					}
				}
                if (show_battery)   get_battery_state();
                last_update_time = now;
				arrange();
            }
            if (show_sound && prev_sound_vol < 0) {
                prev_sound_vol = sound_p->value();
            }
            redraw();
        }
        void get_config() {
            char confline[512];
            char s[64];
            int n;
            float f;
            int xpos, ypos;
            int r,g,b;

            if(configfile_option) {
                strcpy(configfilename, configfile_option);
            }
            else {
                char* env_home_p = getenv("HOME");
                sprintf(configfilename, "%s/.flit.conf", env_home_p);
            }

            FILE* cfgf = NULL;
            
            cfgf = fopen(configfilename, "r");
            if (cfgf) {
                while (fgets(confline, 500, cfgf)) {
                    if (1 == sscanf(confline, "show_clock = %d\n", &n)) {
#ifdef DIAG     
                        printf("show_clock is %d\n", n);
#endif
                        show_clock = n;
                    }
                    else if (1 == sscanf(confline, "show_sound = %d\n", &n)) {
#ifdef DIAG     
                        printf("show_sound is %d\n", n);
#endif
                        show_sound = n;
                    }
                    else if (1 == sscanf(confline, "show_wireless = %d\n", &n)) {
#ifdef DIAG     
                        printf("show_wireless is %d\n", n);
#endif
                        show_wireless = n;
                    }               
                    else if (1 == sscanf(confline, "show_battery = %d\n", &n)) {
#ifdef DIAG     
                        printf("show_battery is %d\n", n);
#endif
                        show_battery = n;
                    }               
                    else if (1 == sscanf(confline, "menu_hotkey_activation = %d\n", &n)) {
#ifdef DIAG     
                        printf("menu_hotkey_activation is %d\n", n);
#endif
                        menu_hotkey_activation = n;
                    }
                    else if (1 == sscanf(confline, "style = %d\n", &n)) {
#ifdef DIAG     
                        printf("style is %d\n", n);
#endif
                        style = n;
                    }
                    else if (1 == sscanf(confline, "zoom = %f\n", &f)) {
#ifdef DIAG     
#endif
                        printf("zoom is %1.3f\n", f);
                        Fl::screen_scale(f); 
                    }
                    else if (1 == sscanf(confline, "sound_control_name = %s\n", s)) {
                        if (strlen(s) > 0 && strlen(s) < 63) {
                            strcpy(sound_control_name, s);
                        }
#ifdef DIAG     
                        printf("sound_control_name is %s\n", sound_control_name);
#endif
                        
                    }
                    else if (1 == sscanf(confline, "sound_volume_level = %d\n", &n)) {
#ifdef DIAG     
                        printf("volume is %d\n", n);
#endif
                        if((n >= 0) && (n <= 100))
							saved_volume = n;
                    }
                    else if (1 == sscanf(confline, "show24hr = %d\n", &n)) {
#ifdef DIAG     
                        printf("show24hr is %d\n", n);
#endif
                        show24hr = n;
                    }
                    else if (1 == sscanf(confline, "location = %s\n", s)) {
#ifdef DIAG     
                        printf("Location spec is %s\n", s);
#endif
                        if (!strcmp(s, "se")) {
                            win_loc = LOC_SE;
                        }
                        else if (!strcmp(s, "sw")) {
                            win_loc = LOC_SW;
                        }
                        else if (!strcmp(s, "nw")) {
                            win_loc = LOC_NW;
                        }
                        else if (!strcmp(s, "ne")) {
                            win_loc = LOC_NE;
                        }
                        else {
                            if (2 == sscanf(s, "%d,%d", &xpos, &ypos)) {
                                win_x = xpos;
                                win_y = ypos;
                                win_loc = LOC_CUST;
                            }
                        } 
                    } // end location   
            else if(1 == sscanf(confline, "custom_fg = %s\n", s)) {
                if(3 == sscanf(s, "%d,%d,%d", &r, &g, &b)) {
                    C_Fg_r = r;
                    C_Fg_g = g;
                    C_Fg_b = b;
                }
            }
            else if(1 == sscanf(confline, "custom_bg = %s\n", s)) {
                if(3 == sscanf(s, "%d,%d,%d", &r, &g, &b)) {
                    C_Bg_r = r;
                    C_Bg_g = g;
                    C_Bg_b = b;
                }
            }
            else if (strstr(confline, "Wireless_Enable_cmd")) {
				char * d = Wireless_Enable_cmd; 
				char * c = strstr(confline, "="); c++;
				while (isspace(*c)) c++;
				while (*c && *c != '\n')  *d++ = *c++;
				*d = '\0';
			}
            else if (strstr(confline, "Wireless_Disable_cmd")) {
				char * d = Wireless_Disable_cmd; 
				char * c = strstr(confline, "="); c++;
				while (isspace(*c)) c++;
				while (*c && *c != '\n')  *d++ = *c++;
				*d = '\0';
			}
            else if (strstr(confline, "Wireless_Online_cmd")) {
				char * d = Wireless_Online_cmd; 
				char * c = strstr(confline, "="); c++;
				while (isspace(*c)) c++;
				while (*c && *c != '\n')  *d++ = *c++;
				*d = '\0';
			}
            else if (strstr(confline, "Wireless_Manage_cmd")) {
				char * d = Wireless_Manage_cmd; 
				char * c = strstr(confline, "="); c++;
				while (isspace(*c)) c++;
				while (*c && *c != '\n')  *d++ = *c++;
				*d = '\0';
			} //                    fprintf(cfgf, "max_expected_link_quality = %d\n", max_expected_link_quality);
            else if (strstr(confline, "max_expected_link_quality")) {
				char * c = strstr(confline, "="); c++;
				while (isspace(*c)) c++;
                max_expected_link_quality = -1;
                int max_qual = -1;
                if (1 == sscanf(c, "%i", &max_qual)) {
                    max_expected_link_quality = max_qual;
                }
			}

                } // end while more lines.
            }
            else {
#ifdef DIAG
                printf("No config file found: using defaults.\n");
#endif              
            }
        };  // end get_config()

        void arrange(void) {
            size_h = SIZE_H;
            size_w = 0;
            begin();
            if (show_battery) {
                if (!meter_p) {
                    meter_p = new Flit_Batt_Meter(0, 0, METER_W, SIZE_H);
                    get_battery_info();
                    meter_p->labelsize(8);
                }
                meter_p->position(size_w, 0);
                size_w += METER_W;
            }
            else if (meter_p) { 
                meter_p->position(size_w, 0);
            }

            if (show_wireless) {
                if (!wireless_p) {
                    wireless_p = new Wireless_Meter(0, 0, METER_W, SIZE_H);
                    get_wireless_info();
                    wireless_p->labelsize(8);
                }
                wireless_p->position(size_w, 0);
                size_w += METER_W;
            }
            else if (wireless_p) { 
                wireless_p->position(size_w, 0);
            }

            if (show_sound) {
                if (!sound_p) {
                    sound_p = new Flit_Sound_Control(0, 0, SOUND_W, SIZE_H);
                    sound_p->labelsize(8);
                }
                sound_p->position(size_w, 0);
                size_w += SOUND_W;
            }
            else if (sound_p) {           
                sound_p->position(0, 0);
            }

            if (show_clock) {   
                if (!clock_p) {
                    clock_p = new Fl_Box(0, 0, CLOCK_W, SIZE_H);
                    get_time();
                    clock_p->labelsize(12);
                }
                clock_p->position(size_w, 0);
                size_w += CLOCK_W;
            }
            else if (clock_p) {
                clock_p->position(0, 0);
            }
            
            end();
            
            switch(win_loc) {
                case LOC_SW:
                    win_x = 0;
                    win_y = Fl::h() - size_h;
                    break;
                case LOC_NW:
                    win_x = 0;
                    win_y = 0;
                    break;
                case LOC_NE:
                    win_x = Fl::w() - size_w; 
                    win_y = 0;
                    break;
                case LOC_SE:
                    win_x = Fl::w() - size_w; 
                    win_y = Fl::h() - size_h;
                    break;
            }
            resize(win_x, win_y, size_w, size_h);
            //printf("win_x %d, win_y %d, size_w %d, size_h %d\n", win_x, win_y, size_w, size_h);
            
            redraw();
        };
        
        void toggle_show_clock(void) {
            if (show_clock && (show_wireless + show_battery + show_sound)) // can't disable clock unless one other is showing
            {
                show_clock = 0;
                if (clock_p) clock_p->hide();
            }
            else if (!show_clock) { // can always re-enable clock
                show_clock = 1;
                if (clock_p) clock_p->show();
            }
            arrange();
            set_style(style);
        };
        
        void toggle_show_sound(void) {
            if (show_sound && (show_wireless + show_clock + show_battery)) // can't disable sound ctrl unless one other is showing
            {
                show_sound = 0;
                mixer_fd = 0;
                if (sound_p) sound_p->hide();
            }
            else if (!show_sound) { // can always re-enable sound
                show_sound = 1;
                if (sound_p) sound_p->show();
                get_sound_info();   // checks for availability
                if (!show_sound) {
                    if (sound_p) sound_p->hide();
                    fl_alert("No sound system (e.g. OSS) found.\n");
                }
            }
            arrange();
            set_style(style);
        };
        
        void toggle_show_wireless(void) {
            if (show_wireless && (show_battery + show_sound + show_clock)) // can't disable wireless meter unless one other is showing
            {
                show_wireless = 0;
            }
            else if (!show_wireless) {   // can always re-enable battery
                show_wireless = 1;
            }
            arrange();
            if (wireless_p)   show_wireless ? wireless_p->show() : wireless_p->hide();
            set_style(style);
        };
        
        void toggle_show_batt(void) {
            if (show_battery && (show_wireless + show_sound + show_clock)) // can't disable battery meter unless one other is showing
            {
                show_battery = 0;
                if (meter_p) meter_p->hide();
            }
            else if (!show_battery) {   // can always re-enable battery
                show_battery = 1;
                if (meter_p) meter_p->show();
            }
            arrange();
            set_style(style);
        };
        
        void set_style(int newstyle) {
            int bg_r, bg_g, bg_b, fg_r, fg_g, fg_b;
            style = newstyle;
            int most;
            switch(style) {
                case NORMAL_STYLE:
                    Fl::set_color(FL_BACKGROUND_COLOR, Bg_r, Bg_g, Bg_b);
                    Fl::set_color(FL_FOREGROUND_COLOR, Fg_r, Fg_g, Fg_b);
                    set_boxtype(FL_THIN_DOWN_BOX);
                    break;
                case INVERSE_STYLE:
                    Fl::set_color(FL_BACKGROUND_COLOR, Fg_r, Fg_g, Fg_b);
                    Fl::set_color(FL_FOREGROUND_COLOR, Bg_r, Bg_g, Bg_b);
                    set_boxtype(FL_THIN_DOWN_BOX);
                    break;
                case TRANSPARENT_STYLE:
					get_background_color();
					
                    bg_r = MIN(0xFF, RW_r + 8);
                    bg_g = MIN(0xFF, RW_g + 8);
                    bg_b = MIN(0xFF, RW_b + 8);
					most = bg_r;
					if (bg_g > most) most = bg_g;
					if (bg_b > most) most = bg_b;
					
					if (most > bg_r)  bg_r = (7 * bg_r + 3 * most) / 10;
					if (most > bg_g)  bg_g = (7 * bg_g + 3 * most) / 10;
					if (most > bg_b)  bg_b = (7 * bg_b + 3 * most) / 10;
                   
                    fg_r = Fg_r;
                    fg_g = Fg_g;
                    fg_b = Fg_b;
//                    printf("(bg_r + bg_g + bg_b) is 0x%02X\n", (bg_r + bg_g + bg_b));
                    if ((bg_r + bg_g + bg_b) < 0xA0) {
                        // we have a fairly dark background, so foreground color
                        // needs to be fairly light
                        fg_r = fg_g = fg_b = 0xC0;
//                        printf("Dark background! \n");
                    }
                    Fl::set_color(FL_BACKGROUND_COLOR, bg_r, bg_g, bg_b);
                    Fl::set_color(FL_FOREGROUND_COLOR, fg_r, fg_g, fg_b);
                    set_boxtype(FL_FLAT_BOX);
                    break;
                case CUSTOM_STYLE:
                    Fl::set_color(FL_BACKGROUND_COLOR, C_Bg_r, C_Bg_g, C_Bg_b);
                    Fl::set_color(FL_FOREGROUND_COLOR, C_Fg_r, C_Fg_g, C_Fg_b);
                    set_boxtype(FL_FLAT_BOX);
                    break;
            }
            redraw();
        };

        void get_background_color(void) {
            Window rootw;
            XImage * xip = NULL;

            rootw = DefaultRootWindow(fl_display);
            // Take a peek at the X11 root window, so we can find out it's color
            float sf = Fl::screen_scale(0);
            int sx = (int)((float)win_x * sf) - 8;
            int sy = (int)((float)win_y * sf) + 4;
            xip = XGetImage(fl_display, rootw, sx, sy, 1, 1, -1, ZPixmap); // 1x1 pixel screen grab
            if (xip) {
                RW_r = (uchar)xip->data[2]; // This may only work with true-color color modes
                RW_g = (uchar)xip->data[1];
                RW_b = (uchar)xip->data[0];
#ifdef DIAG     
                printf("Root RGB colors are %02X:%02X:%02x at X %d, Y %d\n", RW_r, RW_g, RW_b, sx, sy);
#endif
            }

        }
        
        void configure(void) {
			//show_wireless = 1;	// hardcode for now...
            get_config();
            clear_border();
            box(FL_NO_BOX);
            if (show_sound) get_sound_info();
            arrange();
            if (show_wireless) get_wifi_dev_name();
            get_background_color();
            Fl::get_color(FL_FOREGROUND_COLOR, Fg_r, Fg_g, Fg_b);
            Fl::get_color(FL_BACKGROUND_COLOR, Bg_r, Bg_g, Bg_b);
            set_style(style);
            
        };
        
        void save_config(void) {
			float Z = Fl::screen_scale(0);
            const char *loc_tag[] = {"se", "sw", "nw", "ne"};
            FILE* cfgf = NULL;
            cfgf = fopen(configfilename, "w");
            if (cfgf) {
                fprintf(cfgf, "# Configuration file for Flit \n\n");
                fprintf(cfgf, "# Note: this is a machine-generated file; if you change any\n");
                fprintf(cfgf, "#       content, preserve spelling, capitalization, and spacing!\n\n");
                fprintf(cfgf, "# Enable (1) or disable (0) individual applets\n");
                fprintf(cfgf, "show_clock = %d\n", show_clock);
                fprintf(cfgf, "show_sound = %d\n", show_sound);
                fprintf(cfgf, "show_wireless = %d\n", show_wireless);
                fprintf(cfgf, "show_battery = %d\n", show_battery);
                fprintf(cfgf, "\n");
                fprintf(cfgf, "# Show clock in 24 hour format (1 means yes, 0 means no - use AM/PM)\n");
                fprintf(cfgf, "show24hr = %d\n", show24hr);
                fprintf(cfgf, "\n");
                fprintf(cfgf, "# OSS/ALSA sound control name to use, autosel means 'try to pick for me'\n");
                fprintf(cfgf, "sound_control_name = %s\n", sound_control_name);
                fprintf(cfgf, "\n");
                fprintf(cfgf, "# OSS/ALSA sound volume level (0 to 100)\n");
                fprintf(cfgf, "sound_volume_level = %d\n", sound_p->value());
                fprintf(cfgf, "\n");
                fprintf(cfgf, "# Wireless command patterns, the wireless device name like wlan0 will be supplied twice, which you can reference by %%s in the patterns below\n");
                fprintf(cfgf, "Wireless_Enable_cmd = %s\n", Wireless_Enable_cmd);
                fprintf(cfgf, "Wireless_Disable_cmd = %s\n", Wireless_Disable_cmd);
                fprintf(cfgf, "Wireless_Online_cmd = %s\n", Wireless_Online_cmd);
                fprintf(cfgf, "Wireless_Manage_cmd = %s\n", Wireless_Manage_cmd);
                fprintf(cfgf, "\n");
                fprintf(cfgf, "# Wireless expected top link quality value, auto means use fraction as reported by iwconfig\n");
                if (max_expected_link_quality > 0) {
                    fprintf(cfgf, "max_expected_link_quality = %d\n", max_expected_link_quality);
                }
                else {
                    fprintf(cfgf, "max_expected_link_quality = auto\n");
                }
                fprintf(cfgf, "\n");
                fprintf(cfgf, "# Automatically pop up right-click menu when Ctrl key is first pressed (1 = do, 0 = don't)\n");
                fprintf(cfgf, "menu_hotkey_activation = %d\n", menu_hotkey_activation);
                fprintf(cfgf, "\n");
                fprintf(cfgf, "# Color style: 0 is normal, 1 is inverted, 2 is 'transparent' (based on root window color), 3 is custom style\n");
                fprintf(cfgf, "style = %d\n", style);
                fprintf(cfgf, "\n");
                fprintf(cfgf, "# Zoom size: 1.0 is normal, 1.50 is %%150, etc.\n");
                fprintf(cfgf, "zoom = %1.2f\n", Z);
                fprintf(cfgf, "\n");
                fprintf(cfgf, "# Location: se (Southeast, i.e. lower right), sw, nw, ne, or a custom X,Y like this example:\n");
                fprintf(cfgf, "#location = %d,%d\n", win_x, win_y);
                if (win_loc == LOC_CUST) {
                    fprintf(cfgf, "location = %d,%d\n", win_x, win_y);
                }
                else {
                    fprintf(cfgf, "location = %s\n", loc_tag[win_loc]);
                }
                fprintf(cfgf, "\n");
                fprintf(cfgf, "# Custom foreground and background color color (r,g,b) 0-255:\n");
                fprintf(cfgf, "custom_fg = %u,%u,%u\n", C_Fg_r, C_Fg_g, C_Fg_b);
                fprintf(cfgf, "custom_bg = %u,%u,%u\n", C_Bg_r, C_Bg_g, C_Bg_b);
                fprintf(cfgf, "\n");
                
                fclose(cfgf);
            }
        }
        
};

static void MenuCB(Fl_Widget* window_p, void *userdata) 
{
    Fl_Help_Dialog hd;
    long choice = (long)userdata;
    Flit_Frame* mainwnd_p = (Flit_Frame *)window_p;
    char cmd[512];
    switch (choice) {
        case MI_ABOUT:
            fl_message(About_text, APP_VER) ; //, argp_program_bug_address);
            break;
        case MI_HELP:
            hd.load(PATH_TO_FLIT_HELP);
            hd.textsize(14);
            hd.show();
            while (hd.visible()) {
                Fl::wait(1);
            } 
            break;
        case MI_CLOCK:
            mainwnd_p->toggle_show_clock();
            break;
        case MI_SHOW24HR:
            mainwnd_p->toggle_show24hr();
            break;
        case MI_WIRELESS:
			mainwnd_p->get_wireless_info();
            mainwnd_p->toggle_show_wireless();
            break;
        case MI_WIRELESS_ONLINE:
			mainwnd_p->update();
			Fl::wait(0.1);
			sprintf(cmd, Wireless_Enable_cmd, WirelessDevName, WirelessDevName);
			printf("trying command: '%s'\n", cmd);
			system(cmd);
			mainwnd_p->get_wireless_info();
			mainwnd_p->update();
			sprintf(cmd, Wireless_Online_cmd, WirelessDevName, WirelessDevName);
			printf("trying command: '%s'\n", cmd);
			defer(cmd);
			break;
		case MI_WIRELESS_DISABLE:
			sprintf(cmd, Wireless_Disable_cmd, WirelessDevName, WirelessDevName);
			printf("trying command: '%s'\n", cmd);
			defer(cmd);
			mainwnd_p->get_wireless_info();
			mainwnd_p->update();
			break;
		case MI_WIRELESS_CONNECTIONS:
			sprintf(cmd, Wireless_Enable_cmd, WirelessDevName, WirelessDevName);
			printf("trying command: '%s'\n", cmd);
			system(cmd);
			mainwnd_p->get_wireless_info();
			mainwnd_p->update();
            mainwnd_p->do_wireless_manage();
			break;
        case MI_BATTERY:
            mainwnd_p->toggle_show_batt();
            break;
        case MI_SOUND:
            mainwnd_p->toggle_show_sound();
            break;
        case MI_SOUND_MUTE:
            mainwnd_p->do_sound_adj(0); // Toggle sound on/off
            break;
        case MI_SOUND_UP:
            mainwnd_p->do_sound_adj(+1);
            break;
        case MI_SOUND_DOWN:
            mainwnd_p->do_sound_adj(-1);
            break;
        case MI_HOTKEY_TOGGLE:
            mainwnd_p->menu_hotkey_activation = !(mainwnd_p->menu_hotkey_activation);
            break;
        case MI_NORMAL_STYLE:
            mainwnd_p->set_style(NORMAL_STYLE);
            break;
        case MI_INVERSE_STYLE:
            mainwnd_p->set_style(INVERSE_STYLE);
            break;
        case MI_TRANSPARENT_STYLE:
            mainwnd_p->set_style(TRANSPARENT_STYLE);
            break;
        case MI_CUSTOM_STYLE:
            mainwnd_p->set_style(CUSTOM_STYLE);
            break;
        case MI_SAVE_CONFIG:
            mainwnd_p->save_config();
            break;
        case MI_QUIT:
            Running = 0;
            break;
            
        default:
            break;
    }

}
    
extern Display *fl_display;
extern Window fl_window;

void make_window_dock()
{
    Atom dock, win_type;
    win_type = XInternAtom(fl_display,"_NET_WM_WINDOW_TYPE",False);
    dock = XInternAtom(fl_display,"_NET_WM_WINDOW_TYPE_DOCK",False);
    XChangeProperty(fl_display, fl_window,  win_type, XA_ATOM, 32,  PropModeReplace,
          (unsigned char *) &dock, 1);
}

void show_window_on_all_desktops()
{
    // Need to test which method really works
    
    //Atom desktop, win_type;
    //win_type = XInternAtom(fl_display,"_NET_WM_WINDOW_TYPE",False);
    //desktop = XInternAtom(fl_display,"_NET_WM_DESKTOP",False);
    //XChangeProperty(fl_display, fl_window,  win_type, XA_ATOM, 32,  PropModeReplace,
    //        (unsigned char *) &desktop, 0xFFFFFFFF);

    XClientMessageEvent xev;
    xev.type = ClientMessage;
    xev.window = fl_window;
    xev.message_type = XInternAtom(fl_display, "_NET_WM_DESKTOP", False); 
    xev.format = 32;

    /* Force on all desktops! */
    xev.data.l[0] = 0xFFFFFFFF;

    XSendEvent(
        fl_display,
    fl_window, 
    False, 
    SubstructureNotifyMask|SubstructureRedirectMask, 
    (XEvent *) &xev);
}

void DoDeferredCmd(Flit_Frame* mainwnd_p)
{
    CmdIsDeferred = 0;
    fl_cursor(FL_CURSOR_WAIT);
    system(DeferredCmdStr);
    mainwnd_p->get_wireless_info();
    fl_cursor(FL_CURSOR_DEFAULT);
}

int main(int argc, char** argv)
{
	int batt_percent = get_batt_percentage(BATT_PERCENT_DEVFILE);
	printf("Battery now at %d%%\n", batt_percent);
	
    struct arguments arguments;
    // Set argument defaults
    arguments.on_all_desktops       = false;
    arguments.dock                  = false;
    arguments.multiple_instances    = false;
    arguments.config_file           = NULL;

    // Parse our command line arguments
    argp_parse(&argp, argc, argv, 0, 0, &arguments);

    if(!arguments.multiple_instances) {
        if(is_already_running(PROG)) {
            printf("flit is already running - exiting\n");
            return 0;
        }
        create_pidfile(PROG);        
    }

    fl_open_display();
    
    float sf = Fl::screen_scale(0);  // maybe check for which screen flit is on?
    printf("Default scaling factor from the window manager is %1.3f\n", sf);

    fl_message_font(fl_font(), 12);
    Flit_Frame MainWnd("Flit", arguments.config_file);
    MainWnd.configure();
    MainWnd.update();
    MainWnd.show();
    
    // Needed for docking to work
    MainWnd.make_current();

    if(arguments.on_all_desktops) {
        show_window_on_all_desktops();
    }

    if(arguments.dock) {
        make_window_dock();    
    }
    
	if(saved_volume >= 0)
	{
		MainWnd.set_sound_level(saved_volume);
#ifdef DIAG
		printf("saved_volume=%d\n", saved_volume);
#endif
	}

    //XSetInputFocus(fl_display, PointerRoot, RevertToPointerRoot, CurrentTime);
    while(Running) {
        Fl::wait(1.0);
        if (CmdIsDeferred) {
            DoDeferredCmd(&MainWnd);
        }
        MainWnd.update();
    }    
    //MainWnd.save_config();
    
    if(!arguments.multiple_instances) {
        delete_pidfile(PROG);
    }

    return 0; // Fl::run();
}        
