#!/bin/bash

extract_changelog() {
  local spec_file="$1"
  if [ -f "$spec_file" ]; then
    sed -n '/%changelog/,$p' "$spec_file" | sed '1d'
  fi
}

if ! command -v yq &>/dev/null; then
  echo "yq not found. Please install yq to proceed."
  exit 1
fi

RECIPES_DIR="pkg/recipes"
SPEC_OUTPUT_DIR="pkg/fedora-spec"
mkdir -p $SPEC_OUTPUT_DIR

for RECIPE_PATH in "$RECIPES_DIR"/*/; do
  RECIPE_NAME=$(basename "$RECIPE_PATH")
  RECIPE_FILE="${RECIPE_PATH}recipe.yml"

  if [ ! -f "$RECIPE_FILE" ]; then
    echo "No recipe.yml found in $RECIPE_PATH, skipping."
    continue
  fi

  RECIPE_VERSION=$(yq eval '.metadata.version // "0.0.1"' "$RECIPE_FILE")
  RECIPE_RELEASE=1
  PKG_LICENSE=$(yq eval '.metadata.license // "UNKNOWN"' "$RECIPE_FILE")
  PKG_DESCRIPTION=$(yq eval '.metadata.description // "No description available."' "$RECIPE_FILE")
  MAINTAINER=$(yq eval '.metadata.maintainer // "Unknown Maintainer"' "$RECIPE_FILE")
  SOURCE_URL="https://github.com/ilya-zlobintsev/LACT/archive/refs/tags/v${RECIPE_VERSION}.tar.gz"

  LACT_COMMIT_HASH=$(git rev-parse --short v$RECIPE_VERSION)

  MAKE_COMMAND_RAW=$(yq eval '.build.steps[0]' "$RECIPE_FILE" | grep -oP '(make.+)')
  MAKE_COMMAND=${MAKE_COMMAND_RAW::-1}

  # Collect Fedora-specific dependencies safely
  PKG_DEPENDS=$(yq eval '.metadata.depends | with_entries(select(.key | contains("fedora"))) | .[] | join(" ")' "$RECIPE_FILE" | xargs)
  PKG_BUILD_DEPENDS=$(yq eval '.metadata.build_depends | with_entries(select(.key | contains("fedora"))) | .[] | join(" ")' "$RECIPE_FILE" | xargs)

  # Include dependencies from the 'all' key if they exist
  ALL_DEPENDS=$(yq eval 'select(.metadata.depends.all != null) | .metadata.depends.all | join(" ")' "$RECIPE_FILE" | xargs)
  ALL_BUILD_DEPENDS=$(yq eval 'select(.metadata.build_depends.all != null) | .metadata.build_depends.all | join(" ")' "$RECIPE_FILE" | xargs)

  PKG_DEPENDS="${PKG_DEPENDS} ${ALL_DEPENDS}"
  PKG_BUILD_DEPENDS="${PKG_BUILD_DEPENDS} ${ALL_BUILD_DEPENDS}"

  # Trim any leading or trailing whitespace
  PKG_DEPENDS=$(echo "$PKG_DEPENDS" | xargs)
  PKG_BUILD_DEPENDS=$(echo "$PKG_BUILD_DEPENDS" | xargs)

  # Generate the spec file
  SPEC_FILE="${SPEC_OUTPUT_DIR}/${RECIPE_NAME}.spec"

  EXISTING_CHANGELOG=""
  if [ -f "$SPEC_FILE" ]; then
    EXISTING_CHANGELOG=$(extract_changelog "$SPEC_FILE")
  fi
  # For proper date formatting consistent with other builds.
  export LC_ALL=c
  cat <<EOF >"$SPEC_FILE"
Name:           $RECIPE_NAME
Version:        $RECIPE_VERSION
Release:        $RECIPE_RELEASE
Summary:        $PKG_DESCRIPTION
License:        $PKG_LICENSE
URL:            https://github.com/ilya-zlobintsev/LACT
Source0:        $SOURCE_URL

BuildRoot:      %{_tmppath}/%{name}-%{version}-%{release}-root-%(%{__id_u} -n)
BuildRequires:  rust cargo $PKG_BUILD_DEPENDS
Requires:       $PKG_DEPENDS

%description
$PKG_DESCRIPTION

%prep
%setup -q -n LACT-%{version}

%build
VERGEN_GIT_SHA=$LACT_COMMIT_HASH $MAKE_COMMAND %{?_smp_mflags}

%install
rm -rf %{buildroot}
make install PREFIX=/usr DESTDIR=%{buildroot}

%files
%defattr(-,root,root,-)
%license LICENSE
%doc README.md
/usr/bin/lact
/usr/lib/systemd/system/lactd.service
/usr/share/applications/io.github.ilya_zlobintsev.LACT.desktop
/usr/share/icons/hicolor/scalable/apps/io.github.ilya_zlobintsev.LACT.svg
/usr/share/pixmaps/io.github.ilya_zlobintsev.LACT.png
/usr/share/metainfo/io.github.ilya_zlobintsev.LACT.metainfo.xml

%changelog
* $(date +"%a %b %d %Y") - $MAINTAINER - $GH_RELEASE_NAME - $GH_RELEASE_TAG
- Autogenerated from CI, please see $GH_RELEASE_URL for detailed changelog.
${EXISTING_CHANGELOG}
EOF

  echo "Spec file created at $SPEC_FILE"
  cat "$SPEC_FILE"
done
