#!/bin/sh

alias cp='busybox cp -a'
alias date='busybox date'
alias mkdir='busybox mkdir -p'
alias mv='busybox mv'
alias rm='busybox rm -rf'

DATE=`date '+%Y/%m/%d'`

PROGRAM="fltk"
PROGRAMVERSION="1.3.8"
EXTENSIONNAME="$PROGRAM-1.3"
SRCDIR="$PROGRAM-$PROGRAMVERSION"
SRCPACKAGE="$SRCDIR-source.tar.bz2"
PROGRAMINSTALL="$PWD/pkg-dev/usr/local"
URL="https://www.fltk.org/pub/$PROGRAM/$PROGRAMVERSION/$SRCPACKAGE"

TOOLCHAIN="compiletc sstrip submitqc intltool"
BUILD_DEPS="Xorg-7.7-dev libjpeg-turbo-dev"
PACKAGE_DEPS="libXext libjpeg-turbo libpng libXcursor"

FLTO="-flto"
OPTIMIZE="-Os"
GDEBUG="No"
# Uncomment the next line to compile a version that can be run under gdb.
#GDEBUG="Debug"

if [ "$GDEBUG" == "Debug" ]
then
	OPTIMIZE="-O0 -g"
	FLTO=""
fi


PROCESSOR_TYPE=`uname -m`
echo "$PROCESSOR_TYPE detected."

case "$PROCESSOR_TYPE" in
        i686)
        CFLAGS="$FLTO -march=i486 -mtune=i686 $OPTIMIZE -pipe -Wall -Wextra -fno-plt"
        CXXFLAGS="$FLTO -march=i486 -mtune=i686 $OPTIMIZE -pipe -Wall -Wextra -fno-exceptions -fno-rtti"
        LDFLAGS="-Wl,-T/usr/local/lib/ldscripts/elf_i386.xbn"
        ;;

        x86_64)
        CFLAGS="$FLTO -mtune=generic $OPTIMIZE -pipe -Wall -Wextra -fno-plt"
        CXXFLAGS="$FLTO -mtune=generic $OPTIMIZE -pipe -Wall -Wextra -fno-exceptions -fno-rtti"
        LDFLAGS="-Wl,-T/usr/local/lib/ldscripts/elf_x86_64.xbn"
        ;;

        armv*)
        CFLAGS="-march=armv6zk -mtune=arm1176jzf-s -mfpu=vfp $OPTIMIZE -pipe -Wall -Wextra"
        CXXFLAGS="-march=armv6zk -mtune=arm1176jzf-s -mfpu=vfp $OPTIMIZE -pipe -Wall -Wextra -fno-exceptions -fno-rtti"
        LDFLAGS="-Wl,-O1"
        ;;

        *)
        echo "$PROCESSOR_TYPE: Unknown processor type. Please add an entry for it in this script."
        exit
        ;;
esac

export CFLAGS CXXFLAGS LDFLAGS

# Download packages required for compiling, filter out success messages.
tce-load -w "$TOOLCHAIN $BUILD_DEPS" 2>&1 | grep -v "already downloaded"

# Install packages required for compiling, filter out success messages.
tce-load -i "$TOOLCHAIN $BUILD_DEPS" 2>&1 | grep -v "already installed"

# Remove output from previous build.
rm pkg
rm pkg-dev
rm "$PROCESSOR_TYPE"
rm "$EXTENSIONNAME".tcz*
rm "$EXTENSIONNAME"-dev.tcz*
rm "$SRCDIR"

# Create the destination directories for packaging.
mkdir "pkg/usr/local/lib"

# Create directory for savingthe build for the current architecture.
mkdir "$PROCESSOR_TYPE"

# Download source code and patch if not present.
[ -e "$SRCPACKAGE" ] || wget $URL
[ -e "fltk-no-printing.patch" ] || wget http://tinycorelinux.net/10.x/x86/tcz/src/fltk/fltk-no-printing.patch

# Unpack source code package.
tar -xf "$SRCPACKAGE"

# Go into the source code directory.
cd "$SRCDIR"

# Apply patches.
patch -Np1 -i ../fltk-no-printing.patch
patch -Np1 -i ../fltk-ansi-patch-1.3.x.patch

# Copy header file for decoding ANSI escape sequences.
cp ../AnsiMode.H src/

./configure \
--prefix="$PROGRAMINSTALL" \
--localstatedir=/var \
--enable-shared \
--disable-gl \
--disable-xinerama \
--disable-xft \
--disable-xfixes \
--enable-xcursor \
--disable-xrender

make

make install

cd ..
# Make sets  prefix  equal to  /home/tc/BuildTCZs/Fltk/pkg-dev/usr/local.
# Find line beginning with  prefix=$PWD  and replace that line
# with  prefix=/usr/local  in  fltk-config.
sed -i 's|^prefix='"$PWD"'.*|prefix=/usr/local|' "$PROGRAMINSTALL/bin/$PROGRAM"-config

# Separate and strip the fltk stuff.
sudo mv pkg-dev/usr/local/lib/*.so* pkg/usr/local/lib/
sudo strip --strip-unneeded pkg/usr/local/lib/*

#--------------- Package program ---------------#
# Create .tcz, .md5.txt, and .list files for program.
mksquashfs pkg "$EXTENSIONNAME".tcz -noappend
md5sum "$EXTENSIONNAME".tcz > "$EXTENSIONNAME".tcz.md5.txt
cd pkg
# Find files including their path starting from the current directory,
# cut the leading dot, sort it, and save it to the .list file.
find . -not -type d | cut -c 2- | sort > ../"$EXTENSIONNAME".tcz.list
cd ..

# Create dependency file.
for EXT in $PACKAGE_DEPS
do
	echo "$EXT".tcz >> "$EXTENSIONNAME".tcz.dep
done


#------------- Package program-dev -------------#
# Create .tcz, .md5.txt, and .list files for program.
mksquashfs pkg-dev "$EXTENSIONNAME"-dev.tcz -noappend
md5sum "$EXTENSIONNAME"-dev.tcz > "$EXTENSIONNAME"-dev.tcz.md5.txt
cd pkg-dev
# Find files including their path starting from the current directory,
# cut the leading dot, sort it, and save it to the .list file.
find . -not -type d | cut -c 2- | sort > ../"$EXTENSIONNAME"-dev.tcz.list
cd ..

# Create dependency file.
echo "$EXTENSIONNAME".tcz > "$EXTENSIONNAME"-dev.tcz.dep
for EXT in $BUILD_DEPS
do
	echo "$EXT".tcz >> "$EXTENSIONNAME"-dev.tcz.dep
done

# Check the newly created extension files for errors.
submitqc --libs > submitqc.txt 2>&1

# Create archive of the source package used for this build. Save a copy to  $PROCESSOR_TYPE  directory.
busybox tar -czf "$PROGRAM"-source-v"$PROGRAMVERSION".tar.gz "$SRCPACKAGE" "$0" AnsiMode.H* *.patch "ansi-patch-1.3.x--r10037.cxx"
cp "$PROGRAM"-source-v"$PROGRAMVERSION".tar.gz "$PROCESSOR_TYPE"

# Save copies in the  $PROCESSOR_TYPE  directory in case we want
# to compile for another architecture.
cp "$EXTENSIONNAME"*.tcz* "$PROCESSOR_TYPE"

# Save a copy of the build script used to create this version.
cp "$0" "$PROCESSOR_TYPE"

# Save a copy of the submitqc report.
cp submitqc.txt "$PROCESSOR_TYPE"

echo "Check $PACKAGEDIR/submitqc.txt for any errors."
echo "Send $PROGRAM-$PROCESSOR_TYPE.tar.gz.bfe to: tcesubmit@gmail.com"
