/*
 *  This file is part of seq42/seq32/sequencer64.
 *
 *  seq32 is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  seq32 is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with seq24; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * \file          lfownd.cpp
 *
 *  This module declares/defines the base class for the LFO window of the
 *  application.
 *
 * \library       sequencer64 application
 * \author        Seq42 team; modifications by Chris Ahlstrom
 * \date          2016-07-30
 * \updates       2018-10-30
 * \license       GNU GPLv2 or above
 *
 *  Created on: 22 mar 2013
 *      Author: mattias
 *
 *  The LFO window provides a way to apply various kinds of low-frequency
 *  oscillators to the existing events.
 *
 *  This window provides the following parametric controls:
 *
 *      -   Value.  Provides the "mean" value of the LFO, a kind of "DC
 *          offset" for the waveform.  Ranges from 0 to
 *          127, and defaults to 64.
 *      -   Range.  Provides the range about the mean, a kind of "depth of
 *          modulation" value.  Ranges from 0 to 127, and defaults to 64.
 *      -   Speed.  Provides the number of periods of oscillation per pattern.
 *          So the frequency of the oscillation depends on this parameter and
 *          the legnth of the pattern.  To get the same frequency for a longer
 *          pattern, this parameter would have to be increased.
 *      -   Phase. Provides the phase shift of the oscillation, range from 0
 *          to 1, which represents a true phase shift of 0 to 360 degrees.  We
 *          shoulda made it "radians" :-D.
 *      -   Type.  Indicates the type of waveform for the low-frequency
 *          oscillation.
 *          -@  Sine.
 *          -#  Ramp-up Sawtooth.
 *          -#  Decay Sawtooth.
 *          -#  Triangle.
 *
 *  Note that a certain amount of playing with the sliders in this window is
 *  necessary to completely understand what it does.
 */

#include <string>
#include <sigc++/slot.h>
#include <gtkmm/box.h>
#include <gtkmm/scale.h>
#include <gtkmm/label.h>

#include "calculations.hpp"             /* seq64::wave_type_t enumeration   */
#include "lfownd.hpp"
#include "seqdata.hpp"
#include "seqedit.hpp"
#include "sequence.hpp"

/*
 * Do not document the namespace; it breaks Doxygen.
 */

namespace seq64
{

/**
 *  Constructs the LFO window.
 *
 * \param p
 *      The performance object, which holds parameters necessary for
 *      manipulating events.
 *
 * \param seq
 *      The sequence/pattern that is to be affected by the LFO window.  It
 *      holds the actual MIDI events being modified.
 *
 * \param sdata
 *      The data pane/panel of the pattern editor window representing the
 *      sequence.  We need to tell it to redraw.
 */

lfownd::lfownd (perform & p, sequence & seq, seqdata & sdata)
 :
    gui_window_gtk2 (p),    // , 500, 400),
    m_seq           (seq),
    m_seqdata       (sdata),
    m_hbox          (manage(new Gtk::HBox(true, 8))),          // (false, 2))),
    m_scale_value   (manage(new Gtk::VScale(0, 127, 0.1))),
    m_scale_range   (manage(new Gtk::VScale(0, 127, 0.1))),
    m_scale_speed   (manage(new Gtk::VScale(0, 16,  0.01))),
    m_scale_phase   (manage(new Gtk::VScale(0, 1,   0.01))),
    m_scale_wave    (manage(new Gtk::VScale(1, 5,   1))),
    m_wave_name     (manage(new Gtk::Label("Sine"))),
    m_value         (0.0),
    m_range         (0.0),
    m_speed         (0.0),
    m_phase         (0.0),
    m_wave          (WAVE_SINE)
{
    std::string title = "Sequencer64 - LFO Editor - ";
    title.append(m_seq.name());
    set_title(title);
    set_size_request(400, 300); // set_size_request(150, 200);
    m_scale_value->set_tooltip_text
    (
        "Value: a kind of DC offset for the data value. Range: 0 to 127."
    );
    m_scale_range->set_tooltip_text
    (
        "Range: controls the depth of modulation. Range: 0 to 127."
    );
    m_scale_speed->set_tooltip_text
    (
        "Speed: the number of periods per pattern (divided by beat width, "
        "normally 4).  For long patterns, this parameter needs to be set "
        "high in some cases.  Also subject to an 'anti-aliasing' effect in "
        "some parts of the range, especially for short patterns. "
        "Try it.  For short patterns, try a value of 1."
    );
    m_scale_phase->set_tooltip_text
    (
        "Phase: phase shift in a beat width (quarter note). "
        "A value of 1 is a phase shift of 360 degrees."
    );
    m_scale_wave->set_tooltip_text
    (
        "Wave type: 1 = sine; 2 = ramp sawtooth; 3 = decay sawtooth; "
        "4 = triangle."
    );

    m_scale_value->set_value(64);
    m_scale_range->set_value(64);
    m_scale_speed->set_value(0);
    m_scale_phase->set_value(0);
    m_scale_wave->set_value(1);

    m_scale_value->signal_value_changed().connect
    (
        sigc::mem_fun(*this, &lfownd::scale_lfo_change)
    );
    m_scale_range->signal_value_changed().connect
    (
        sigc::mem_fun( *this, &lfownd::scale_lfo_change)
    );
    m_scale_speed->signal_value_changed().connect
    (
        sigc::mem_fun( *this, &lfownd::scale_lfo_change)
    );
    m_scale_phase->signal_value_changed().connect
    (
        sigc::mem_fun( *this, &lfownd::scale_lfo_change)
    );
    m_scale_wave->signal_value_changed().connect
    (
        sigc::mem_fun( *this, &lfownd::scale_lfo_change)
    );
    Gtk::VBox * vbox1 = manage(new Gtk::VBox(false, 2));
    Gtk::VBox * vbox2 = manage(new Gtk::VBox(false, 2));
    Gtk::VBox * vbox3 = manage(new Gtk::VBox(false, 2));
    Gtk::VBox * vbox4 = manage(new Gtk::VBox(false, 2));
    Gtk::VBox * vbox5 = manage(new Gtk::VBox(false, 2));
    Gtk::Label * label1 = manage(new Gtk::Label("DC Value"));
    Gtk::Label * label2 = manage(new Gtk::Label("Mod Range"));
    Gtk::Label * label3 = manage(new Gtk::Label("Periods"));
    Gtk::Label * label4 = manage(new Gtk::Label("Phase Shift"));
    Gtk::Label * label5 = manage(new Gtk::Label("Waveform"));
    m_wave_name->set_width_chars(12);
    vbox1->pack_start(*label1,  false, false, 16);           // 8
    vbox1->pack_start(*m_scale_value,  true, true, 0);
    vbox1->pack_start(*manage(new Gtk::Label(" ")), false, false, 0);   ////
    vbox2->pack_start(*label2,  false, false, 16);
    vbox2->pack_start(*m_scale_range,  true, true, 0);
    vbox2->pack_start(*manage(new Gtk::Label(" ")), false, false, 0);   ////
    vbox3->pack_start(*label3,  false, false, 16);
    vbox3->pack_start(*m_scale_speed,  true, true, 0);
    vbox3->pack_start(*manage(new Gtk::Label(" ")), false, false, 0);   ////
    vbox4->pack_start(*label4,  false, false, 16);
    vbox4->pack_start(*m_scale_phase,  true, true, 0);
    vbox4->pack_start(*manage(new Gtk::Label(" ")), false, false, 0);   ////
    vbox5->pack_start(*label5,  false, false, 16);
    vbox5->pack_start(*m_scale_wave, true, true, 0);
    vbox5->pack_start(*m_wave_name, false, false, 8);       // 0
    vbox5->pack_start(*manage(new Gtk::Label(" ")), false, false, 0);
    m_hbox->pack_start(*vbox1);
    m_hbox->pack_start(*vbox2);
    m_hbox->pack_start(*vbox3);
    m_hbox->pack_start(*vbox4);
    m_hbox->pack_start(*vbox5, true, true, 4);
    add(*m_hbox);
}

/**
 *  Provides a rote destructor.
 */

lfownd::~lfownd ()
{
    // no code
}

/**
 *  Toggles the visibility of the LFO window.
 */

void
lfownd::toggle_visible ()
{
    if (is_visible())
    {
        hide();
    }
    else
    {
        show_all();
        raise();
    }
}

/**
 *  Changes the scaling provided by this window.  Changes take place right
 *  away in this callback.
 */

void
lfownd::scale_lfo_change ()
{
    int wtype = int(m_scale_wave->get_value());
    m_value = m_scale_value->get_value();
    m_range = m_scale_range->get_value();
    m_speed = m_scale_speed->get_value();
    m_phase = m_scale_phase->get_value();
    m_wave = wave_type_t(wtype);
    m_wave_name->set_text(wave_type_name(wave_type_t(wtype)));
    m_seq.change_event_data_lfo
    (
        m_value, m_range, m_speed, m_phase, m_wave,
        m_seqdata.m_status, m_seqdata.m_cc, true
    );
    m_seqdata.update_pixmap();
    m_seqdata.draw_pixmap_on_window();
}

/**
 *  Undoes the LFO changes if there is undo available.
 *
 * \return
 *      Always returns true.
 */

bool
lfownd::on_focus_out_event (GdkEventFocus * /* p0 */)
{
    if (m_seq.get_hold_undo())
    {
        m_seq.push_undo(true);
        m_seq.set_hold_undo(false);
    }
    return true;
}

}           /* namespace seq64 */

/*
 * lfownd.cpp
 *
 * vim: sw=4 ts=4 wm=4 et ft=cpp
 */

