/* PicFormat was written by and is Copyright (C) Richard A. Rost  March 17,2021.
 * 
 * This program allows you to do some simple image manipulations from
 * the command line, including resizing, rotating, and format changes
 * such as jpg to png.
 * 
 * My Copyright terms are simple. You may copy, modify, and use this
 * program as you see fit provided you meet the following terms:
 * 
 * 1. You leave my name as the author and do not take credit for my work.
 * 2. While I hope this program will be useful it is offered WITHOUT ANY
 *    WARRANTY and you agree not to hold me liable for any damages.
 * 3. You leave this header in it's entirety at the top of all source
 *    files for this program and any forks. You may append notes, list
 *    changes, and add the authors of any changes to the end of this header.
 * 4. You do not collect any fee for this program as is or modified.
 * 5. You do not collect any fee for any changes you make to this program.
 * 6. You do not include or package it with any software for which any fee
 *    is collected.
 * 7. You do not include it on media (CD, DVD, etc.) which is sold or any
 *    fee is collected for the cost of the media, shipping, handling, etc.
 * 8. Items 4, 5, 6, and 7 apply to the source code as well as the executable.
 * 9. If you modify this program, you must make the full source code with a
 *    functioning compile script available in one of the following ways.
 *   A. Packaged with the executable when you distribute it.
 *   B. As a separate package available from where you distribute the
 *      executable. If distributed on a CD a web link to the source package
 *      is acceptable.
 * 
 * 
 * This program was written using the Geany 1.23 fast and lightweight IDE.
 * Tabs are set to 4.
 * -----------------------End of original header----------------------------
 */


/******************************************************************/
/* Changelog
 * 3/17/2021 v0.10 Rich
 * Initial release.
 * 
 */
/******************************************************************/


#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>		// close
#include <fcntl.h>		// open O_RDONLY
#include <errno.h>
#include <string.h>		// memset strerror strstr
#include <dirent.h>		// opendir readdir struct dirent 
#include <sys/ioctl.h>
#include <ctype.h>		// isdigit
#include <sys/types.h>
#include <sys/stat.h>

#include <Imlib2.h>

/******************************************************************/
//	Defines

#define COPYRIGHT "Copyright Richard A. Rost Mar 17,2021"
#define PROGRAM "PicFormat"
#define VERSION "version 0.20"

#define OptInput	1 << 0
#define OptOutput	1 << 1
#define OptWidth	1 << 2
#define OptHeight	1 << 3
#define OptPercent	1 << 4
#define OptRotate	1 << 5
/******************************************************************/

/******************************************************************/
//	Global variables


// Used to override errno when calling Bail().
//static int NoSuchDevice=19;
static int InvalidArgument=22;

/******************************************************************/

// Used by Check4ImlibError to tramslate IMLIB_LOAD_ERROR_ enums
// back to errno codes.
const int imlibloaderrors2errno[]=
{
	0,				//"",
	ENOENT,			//"FILE_DOES_NOT_EXIST",
	EISDIR,			//"FILE_IS_DIRECTORY",
	EACCES,			//"PERMISSION_DENIED_TO_READ",
	-1,				//"NO_LOADER_FOR_FILE_FORMAT",
	ENAMETOOLONG,	//"PATH_TOO_LONG",
	ENOENT,			//"PATH_COMPONENT_NON_EXISTANT",
	ENOTDIR,		//"PATH_COMPONENT_NOT_DIRECTORY",
	EFAULT,			//"PATH_POINTS_OUTSIDE_ADDRESS_SPACE",
	ELOOP,			//"TOO_MANY_SYMBOLIC_LINKS",
	ENOMEM,			//"OUT_OF_MEMORY",
	EMFILE,			//"OUT_OF_FILE_DESCRIPTORS",
	EACCES,			//"PERMISSION_DENIED_TO_WRITE",
	ENOSPC,			//"OUT_OF_DISK_SPACE",
	-1				//"UNKNOWN"
};


struct ImageData
{
	char *file;						// Filename of image.
	Imlib_Image image;				// Image handle.
	Imlib_Load_Error fileerror;		// Error code when loading or saving file
	int width;						// Image width in pixels.
	int height;						// Image height in pixels.
	int percent;					// Percent to increases/decrease image dimensions.
	int AspectWidth;				// Aspect width and height reduced to their least common denominator
	int AspectHeight;				//
	int Rotation;					// CW 0=0  1=90  2=180  3=270  degrees
	char format[8];					// Image format (png, jpg, bmp, etc.) must be lower case.
};


/******************************************************************/
//	Prototypes

static void			Bail(int linenumber, int *ErrNum);
static int			CalcRatio(int B, int C, int D);
static void			Check4ImlibError(int io, struct ImageData *data);
static void			GetImageSize(Imlib_Image image, int *Width, int *Height);
static int			IsNumber(char *string);
static int			LeastCommonDivisor(int Numerator, int Denominator);
static void			LoadImageFile(struct ImageData *src);
static void			ReduceFraction(int *Width, int *Height);
static void			SaveImageFile(struct ImageData *dest);
static int			String2Long(char *string, int caller);
static void			Usage(void);

/******************************************************************/


/******************************************************************/
/* Print an error message and the line number of the source file
 * that triggered it. *ErrNum can optionally be used to force a
 * specific error, or pass in NULL to leave its default value.
 */
static void Bail(int linenumber, int *ErrNum)
{
	if(ErrNum)
		errno=*ErrNum;
	printf("Error line: %d\n%s\n", linenumber, strerror(errno));
	exit(1);
}
/******************************************************************/

/******************************************************************/
/* Calculate the unknown value for A/B==C/D. To calculate for A, you
 * would call  A=CalcRatio(B, C, D);. To calculate for B, you would
 * call B=CalcRatio(A, D, C);. The * 2 doubles the numerator
 * effectively changing a 0.5 remainder to 1.0 after dividing. The
 * + 1 performs rounding, then / 2 truncates the remainder again.
 */
static int CalcRatio(int B, int C, int D)
{
	int A;

	A=((((B * C) * 2) / D) + 1) / 2;
	
	return(A);
}
/******************************************************************/

/******************************************************************/
/* Imlib2 converts errno to enumerated imlib2 error numbers.
 * Unfortunately, they don't provide error strings to go with their
 * error numbers. So we convert most of their errors back to errno
 * so we can use strerror to provide error messages.
 */
static void Check4ImlibError(int io, struct ImageData *data)
{
	// Return if no error.
	if(! data->fileerror) return;

	// Print source of error (input, output) and filename.
	printf("-%c %s: ", io, data->file);

	// Decode imlib2 errors.
	switch(data->fileerror)
	{	// Imlib specific error.
		case IMLIB_LOAD_ERROR_NO_LOADER_FOR_FILE_FORMAT:
		printf("Unknown file format\n");
		break;

		// Imlib specific error.
		case IMLIB_LOAD_ERROR_UNKNOWN:
		printf("Unknown error\n");
		break;

		// Translate back to errno errors.
		default:
		printf("%s\n", strerror(imlibloaderrors2errno[data->fileerror]));
		break;
	}

	exit(1);
}
/******************************************************************/

/******************************************************************/
static void GetImageSize(Imlib_Image image, int *Width, int *Height)
{
	imlib_context_set_image(image);
	*Width=imlib_image_get_width();
	*Height=imlib_image_get_height();
	return;
}
/******************************************************************/

/******************************************************************/
/* Tests if a string forms a valid whole number.
 * A leading plus or minus sign is legal. Decimal points are not.
 * Returns 1 if string is valid, and 0 if not.
 */
static int IsNumber(char *string)
{
	char *num=string;

	// A leading sign character is legal, so skip it if present.
	if((*num == '+') || (*num == '-'))
		num++;

	while(*num)
	{
		if(isdigit(*num) == 0)
			return(0);
		num++;
	}

	return(1);	// Valid number.
}
/******************************************************************/

/******************************************************************/
/* Used for reducing a fraction to its lowest common denominator.
 * Called by ReduceFraction
 */
static int LeastCommonDivisor(int Numerator, int Denominator)
{
	int remainder;
	
	while(Numerator != 0)
	{
		remainder=Denominator % Numerator;
		Denominator=Numerator;
		Numerator=remainder;
	}
	return(Denominator);
}
/******************************************************************/

/******************************************************************/
/* Loads the file pointed to file (full path and filename) and
 * returns a pointer to an image if it succeeds. Wildcards and
 * globbing of filenames are not supported.
 */
static void LoadImageFile(struct ImageData *src)
{

	src->image=imlib_load_image_with_error_return(src->file, &src->fileerror);
	if(src->image)
	{
		imlib_context_set_image(src->image);
	}
	return;
}
/******************************************************************/

/******************************************************************/
/* Reduce a fraction to its lowest common denominator.
 */
static void ReduceFraction(int *Width, int *Height)
{
	int LCD;
	
	LCD=LeastCommonDivisor(*Width, *Height);
	*Width/=LCD;
	*Height/=LCD;
	return;
}
/******************************************************************/

/******************************************************************/
/* Saves the file pointed to file (full path and filename).
 * Wildcards and globbing of filenames are not supported.
 */
static void SaveImageFile(struct ImageData *dest)
{

	imlib_image_set_format(dest->format);
	imlib_save_image_with_error_return(dest->file, &dest->fileerror);
	return;
}
/******************************************************************/

/******************************************************************/
/* Takes a whole number in the form of a string and returns an int.
 * A leading plus or minus sign is legal. Decimal points are not.
 */
static int String2Long(char *string, int caller)
{
	int i;

	if(IsNumber(string) == 0)
	{
		printf("%s  invalid number.\n", string);
		Bail(caller, &InvalidArgument);
	}

	errno=0;
	i=strtol(string, NULL, 10);
	if(errno)
		Bail(caller, NULL);

	return(i);
}
/******************************************************************/

/******************************************************************/
/* Help message.
 */
static void Usage(void)
{

	printf("\n%s %s %s %s\n%s\n"
			"\n%s resizes, rotates, and changes formats (i.e. .jpg to .png)\n\n"
			"Usage: %s [-i FILE] [-o FILE] [-w N] [-h N] [-p N] [-r N]\n\n"
			"\t-i FILE\tInput.\n"
			"\t-o FILE\tOutput.\n"
			"\t-w N\tOutput width.\n"
			"\t-h N\tOutput height.\n"
			"\t-p N\tPercent, 110=10%% bigger, 50=1/2 size.\n"
			"\t-r N\tRotate image CW 90, 180, or 270 degrees.\n"
			"\nNotes\n"
			"\tN is an unsigned integer.\n"
			"\tOnly -w OR -h OR -p is needed if resizing.\n"
			"\tIf rotating 90 or 270, -w and -h are swapped.\n\n"
			, PROGRAM, VERSION, __DATE__, __TIME__, COPYRIGHT
			, PROGRAM, PROGRAM);
	exit(1);
}
/******************************************************************/


/****************************** Main ******************************/
int main(int argc, char *argv[])
{
	int i;
	int opt;
	int options=0;
	char *dot, *ptr;
	struct ImageData src= { 0 };
	struct ImageData dest= { 0 };

	if(argc < 2)
	{
		// Running program without arguments returns help message.
		Usage();
	}

	while ((opt=getopt(argc, argv, "-:i:o:w:h:p:r:")) != -1) 
	{
		switch(opt)
		{
			case 'i':	// Input file.
			src.file=optarg;
			options|=OptInput;
			break;

			case 'o':	// Output file.
			dest.file=optarg;
			ptr=dest.file;
			for(dot=NULL; *ptr; ptr++)						// Check if the filename has an extension.
				if(*ptr == '.')
					dot=ptr;								// Remember location of last dot in filename.
			if((dot == NULL) || (*(dot + 1) == '\0'))		// Error if no extension or filename ends in dot.
			{
				printf("%s  invalid extension.\n", dest.file);
				return(1);
			}
			dot++;
			ptr=dest.format;
			for(i=0; (i < 7) && (*dot); i++, dot++, ptr++)	// Copy up to 7 characters of filenames extension.
				*ptr=tolower(*dot);							// Set format command is case sensitive in imlib2.
			options|=OptOutput;
			break;

			case 'w':	// Width for output file.
			dest.width=String2Long(optarg, __LINE__);
			if(dest.width <= 0) Bail(__LINE__, &InvalidArgument);
			options|=OptWidth;
			break;

			case 'h':	// Height for output file.
			dest.height=String2Long(optarg, __LINE__);
			if(dest.height <= 0) Bail(__LINE__, &InvalidArgument);
			options|=OptHeight;
			break;

			case 'p':	// Percent to increases/decrease image dimensions.
			dest.percent=String2Long(optarg, __LINE__);
			if(dest.percent <= 0) Bail(__LINE__, &InvalidArgument);
			options|=OptPercent;
			break;

			case 'r':	// Rotate image CW 90, 180, or 270 degrees.
			i=String2Long(optarg, __LINE__);
			switch(i)
			{	// These case statements are supposed to fall through, so suppress warning.
				#pragma GCC diagnostic ignored "-Wimplicit-fallthrough"
				case 270:	dest.Rotation++;
				case 180:	dest.Rotation++;
				case 90:	dest.Rotation++;
				break;
				#pragma GCC diagnostic warning "-Wimplicit-fallthrough"

				default:
				printf("-r %d  invalid value.\n", i);
				return(1);
				break;
			}
			options|=OptRotate;
			break;

/*			#pragma GCC diagnostic ignored "-Wimplicit-fallthrough"
			case '?':	 printf("Unknown option: %c\n", optopt);
			case ':':	 printf("Missing arg for %c\n", optopt);
			case 1:		 printf("Non-option arg: %s\n", optarg);
			#pragma GCC diagnostic warning "-Wimplicit-fallthrough"
*/
			Usage();	// Errors return help message.
			break;
		}
	}

	if(!(options & OptInput))
	{
		printf("Missing input file.\n");
		return(1);
	}
	LoadImageFile(&src);
	// Check4ImlibError exits if fileerror is nonzero, returns otherwise.
	Check4ImlibError('i', &src);
	// Get the format from the loader, not the file name extension.
	ptr=imlib_image_format();
	for(i=0; (i < 7) && (*ptr); i++, ptr++)	// Copy up to 7 characters of image format type.
		src.format[i]=*ptr;
	GetImageSize(src.image, &src.width, &src.height);
	src.AspectWidth=src.width;
	src.AspectHeight=src.height;
	ReduceFraction(&src.AspectWidth, &src.AspectHeight);
	printf("Input=%s  Fmt=%s  W=%d  H=%d  AR=%d:%d\n", src.file, src.format, src.width, src.height, src.AspectWidth, src.AspectHeight);
	if(options == OptInput)
	{	// Only input file given. Exit after printing filename. image format, width, height, and aspect ratio.
		// Free the src.image.
		imlib_free_image();
		return(0);
	}
	if(!(options & OptOutput))
	{	// Options in addition to input file were given, but output file omitted.
		printf("Missing output file.\n");
		// Free the src.image.
		imlib_free_image();
		return(1);
	}
	// Change image size by an integer percentage.
	if(options & OptPercent)
	{
		// We apply percentage to the width. No special reason.
		// Apply percentage.
		dest.width=(dest.percent * src.width) / 100;
		// Set the OptWidth and clear the OptHeight flags. This forces
		// the following switch statement to follow the case OptWidth: path.
		options|=OptWidth;
		options&=~(OptHeight);
	}

	switch(options & (OptWidth | OptHeight))
	{
		case 0:				// Neither width nor height are specified. Keep original size.
		dest.width=src.width;
		dest.height=src.height;
		break;

		case OptWidth:		// Only width is specified. Calculate height to maintain aspect ratio.
		dest.height=CalcRatio(dest.width, src.height, src.width);
		break;

		case OptHeight:		// Only height is specified. Calculate width to maintain aspect ratio.
		dest.width=CalcRatio(dest.height, src.width, src.height);
		break;

		default:			// Both width and height are specified. Lets hope they got it right.
		break;
	}
	dest.AspectWidth=dest.width;
	dest.AspectHeight=dest.height;
	ReduceFraction(&dest.AspectWidth, &dest.AspectHeight);
	printf("Output=%s  Fmt=%s  W=%d  H=%d  AR=%d:%d  Rotate=%d\n", dest.file, dest.format, dest.width, dest.height, dest.AspectWidth, dest.AspectHeight, (dest.Rotation * 90));
	dest.image=imlib_create_cropped_scaled_image(0, 0, src.width, src.height, dest.width, dest.height);
	// Free the src.image.
	imlib_free_image();
	if(dest.image == NULL)
		Bail(__LINE__, &InvalidArgument);
	imlib_context_set_image(dest.image);
	// If dest.Rotation is zero, no rotation is performed.
	imlib_image_orientate(dest.Rotation);
	SaveImageFile(&dest);
	// Check4ImlibError exits if fileerror is nonzero, returns otherwise.
	// So free the dest.image first.
	imlib_free_image();
	Check4ImlibError('o', &dest);
	return(0);
}
