/* AutoCursor was written by and is Copyright (C) Richard A. Rost  January 25,2021.
 * 
 * This program allows you to automatically hide/unhide the mouse cursor
 * based on events from /dev/input/eventN. For example, you could have
 * keyboard and touch screen events hide the mouse cursosr, and have
 * mouse events make it visible (unhide) again.
 * 
 * My Copyright terms are simple. You may copy, modify, and use this
 * program as you see fit provided you meet the following terms:
 * 
 * 1. You leave my name as the author and do not take credit for my work.
 * 2. While I hope this program will be useful it is offered WITHOUT ANY
 *    WARRANTY and you agree not to hold me liable for any damages.
 * 3. You leave this header in it's entirety at the top of all source
 *    files for this program and any forks. You may append notes, list
 *    changes, and add the authors of any changes to the end of this header.
 * 4. You do not collect any fee for this program as is or modified.
 * 5. You do not collect any fee for any changes you make to this program.
 * 6. You do not include or package it with any software for which any fee
 *    is collected.
 * 7. You do not include it on media (CD, DVD, etc.) which is sold or any
 *    fee is collected for the cost of the media, shipping, handling, etc.
 * 8. Items 4, 5, 6, and 7 apply to the source code as well as the executable.
 * 9. If you modify this program, you must make the full source code with a
 *    functioning compile script available in one of the following ways.
 *   A. Packaged with the executable when you distribute it.
 *   B. As a separate package available from where you distribute the
 *      executable. If distributed on a CD a web link to the source package
 *      is acceptable.
 * 
 * 
 * This program was written using the Geany 1.23 fast and lightweight IDE.
 * Tabs are set to 4.
 * -----------------------End of original header----------------------------
 */


/******************************************************************/
/* Changelog
 * 1/31/2021 v0.20 Rich
 * Omit the quotation marks from the search term pointed to
 * by  description  in LoadSettings().
 * 
 * 1/31/2021 v0.30 Rich
 * Added option to append instead of overwrite when saving settings.
 * Added umask to SaveSettings() so you don't need to be root to edit settings file.
 * 
 * 2/1/2021 v0.31 Rich
 * Cosmetic fixes to usage message.
 * 
 */
/******************************************************************/


#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>		// close
#include <fcntl.h>		// open O_RDONLY
#include <errno.h>
#include <string.h>		// memset strerror strstr
#include <dirent.h>		// opendir readdir struct dirent 
#include <sys/ioctl.h>
#include <linux/input.h>
#include <ctype.h>		// isdigit
#include <sys/types.h>
#include <sys/stat.h>


#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/Xfixes.h>

/******************************************************************/
//	Defines

#define COPYRIGHT "Copyright Richard A. Rost Jan 25,2021"
#define PROGRAM "AutoCursor"
#define VERSION "version 0.31"
#define EVENTstr "event"
#define EVENTstrlen (sizeof(EVENTstr) - 1)
#define devdir "/dev/input/"
/******************************************************************/

/******************************************************************/
//	Structures

struct Events
{
	char description[1024];		// Description of device associated with eventN.
	char showhide;				// Set to S to show cursor, H to hide it.
	int fd;						// File descriptor opened for this event.
};
/******************************************************************/

/******************************************************************/
//	Global variables

static struct Events		*eventlist;				// 
static int					eventmax;
static char					path[1024];

// Used to override errno when calling Bail().
static int NoSuchDevice=19;
static int InvalidArgument=22;

// Used for showing/hiding cursor.
static Window rootwindow;		// The desktop window ID
static int rootwidth;			// The desktop width in pixels
static int rootheight;			// The desktop height in pixels
static int screen;
static Display *display;		//
/******************************************************************/


/******************************************************************/
//	Prototypes

static void			AddFD(int *fd, int *fdmax, int index);
static void			Bail(int linenumber, int *ErrNum);
static void			BuildEventlist(void);
static void			Cleanup(void);
static int			InitFD_set(fd_set *fdset);
static int			IsNumber(char *string);
static void 		GetScreenParams(char *DisplayName);
static void			LoadSettings(char *fullname, int *fdmax);
static void			SaveSettings(char *fullname, int append);
static void			ShowDeviceInfo(void);
static int			String2Long(char *string, int caller);
static void			Usage(void);
static void			Zfree(void **Pointer);
static void			Zmalloc(void **Pointer, int size, int caller);

/******************************************************************/


/******************************************************************/
/* Opens a file descriptor in /dev/input/eventN where N is the index
 * and saves it to fd. If fdmax is passed in, it gets updated if fd
 * contains a larger number. Passing in NULL skips that part.
 */
static void AddFD(int *fd, int *fdmax, int index)
{
	sprintf(path,"%s%s%d", devdir, EVENTstr, index);
	if((*fd=open(path, O_RDONLY | O_NONBLOCK)) == -1)
	{
		printf("Error, can't open %s\n", path);
		Bail(__LINE__, NULL);
	}
	
	if(fdmax)
		if(*fd > *fdmax)
			*fdmax=*fd;

	return;
}
/******************************************************************/

/******************************************************************/
/* Print an error message and the line number of the source file
 * that triggered it. *ErrNum can optionally be used to force a
 * specific error, or pass in NULL to leave its default value.
 */
static void Bail(int linenumber, int *ErrNum)
{
	Cleanup();
	if(ErrNum)
		errno=*ErrNum;
	printf("Error line: %d\n%s\n", linenumber, strerror(errno));
	exit(1);
}
/******************************************************************/

/******************************************************************/
/* Scans /dev/input/event* and builds an array of struct Events.
 * The index into the array represents the number of the event.
 * If there are any gaps in event numbers, those entries contain zeros.
 */
static void BuildEventlist(void)
{
	int i;
	int fd;
	char name[512];
	DIR *directory;
	struct dirent *entity;

	eventmax=0;
	directory=opendir(devdir);
	if(directory)
	{	// First we find the highest event number.
		while((entity=readdir(directory)) != NULL)
		{	// Only return entries starting with EVENTstr.
			if(strncmp(EVENTstr, entity->d_name, EVENTstrlen) == 0)
			{	// Convert the ASCII number at the end of EVENTstr to an int.
				i=String2Long(entity->d_name + EVENTstrlen, __LINE__);
				if(i > eventmax)
					eventmax=i;
			}
		}
	}
	else
	{
		Bail(__LINE__, NULL);
	}
	// Now that we know the highest event number, we can allocate RAM.
	Zmalloc((void **)&eventlist, sizeof(struct Events) * (eventmax + 1), __LINE__);
	// And finally we populate the event list.
	rewinddir(directory);
	while((entity=readdir(directory)) != NULL)
	{	// Only return entries starting with EVENTstr.
		if(strncmp(EVENTstr, entity->d_name, EVENTstrlen) == 0)
		{	// Convert the ASCII number at the end of EVENTstr to an int.
			i=String2Long(entity->d_name + EVENTstrlen, __LINE__);
			// Open event file.
			AddFD(&fd, NULL, i);
			// Retrieve description.
			ioctl(fd, EVIOCGNAME(sizeof(name)), name);
			close(fd);
			// Save event # and description, i.e   2:      "Logitech USB-PS/2 Optical Mouse"
			sprintf(eventlist[i].description, "%d:\t\"%s\"", i, name);
			// Mark this slot as having a valid entry. If there are any
			// gaps in the event numbering, those slots will still contain
			// zero, and we will be able to avoid trying to use them.
			eventlist[i].showhide='v';
		}
	}
	closedir(directory);

	return;
}
/******************************************************************/

/******************************************************************/
/* Free allocated RAM and close open file descriptors.
 */
static void Cleanup(void)
{
	int i;

	if(eventlist)
	{
		for(i=0; i <= eventmax; i++)
			if(eventlist[i].fd)
				close(eventlist[i].fd);
		Zfree((void **)&eventlist);
	}

	return;
}
/******************************************************************/

/******************************************************************/
/* Fetch some common parameters required by X11 functions
 */
static void GetScreenParams(char *DisplayName)
{
	if((display=XOpenDisplay(DisplayName)))
	{
		screen=DefaultScreen(display);
		rootwindow=RootWindow(display,screen);		// XID of the root window
		rootwidth=DisplayWidth(display,screen);		// Screen width in pixels
		rootheight=DisplayHeight(display,screen);	// Screen height in pixels
		return;
	}
	Bail(__LINE__, NULL);
}
/******************************************************************/

/******************************************************************/
/* Load a file descriptor set for select() from the eventlist array.
 * Array entries that have showhide set to s or h have open file
 * descriptors in fd. If savesettings is set, it means fdset was
 * updated and there are settings to be saved if the -S argument
 * was invoked when the program was started.
 */
static int InitFD_set(fd_set *fdset)
{
	int i;
	int savesettings=0;

	FD_ZERO(fdset);

	for(i=0; i <= eventmax; i++)
	{
		switch(eventlist[i].showhide) 
		{
			case 's':
			case 'h':
			FD_SET(eventlist[i].fd, fdset);
			savesettings=1;
			break;
		}
	}

	return(savesettings);
}
/******************************************************************/

/******************************************************************/
/* Tests if a string forms a valid whole number.
 * A leading plus or minus sign is legal. Decimal points are not.
 * Returns 1 if string is valid, and 0 if not.
 */
static int IsNumber(char *string)
{
	char *num=string;

	// A leading sign character is legal, so skip it if present.
	if((*num == '+') || (*num == '-'))
		num++;

	while(*num)
	{
		if(isdigit(*num) == 0)
			return(0);
		num++;
	}

	return(1);	// Valid number.
}
/******************************************************************/

/******************************************************************/
/* Loads event settings from a file. Settings are saved as
 * Action:Description.
 * 
 * This means show the cursor when this mouse generates an event:
 * s:"Logitech USB-PS/2 Optical Mouse"
 * 
 * This means hide the cursor when this keyboard generates an event:
 * h:"DELL DELL USB Keyboard"
 * 
 * Events are saved by description because event numbers can change.
 * Plugging in a mouse could cause a touchpads or touchscreens event
 * number to be different on the next boot. Save your settings with
 * all affected peripherals attached. If a description can't be found
 * when loading settings, it just gets skipped.
 */
static void LoadSettings(char *fullname, int *fdmax)
{
	int i;
	FILE *config;
	char *description;
	char buffer[1024];

	if((config=fopen(fullname, "r")) == NULL) // Open read only.
	{
		Cleanup();
		Bail(__LINE__, NULL);
	}

	while(fgets(buffer, sizeof(buffer), config))
	{	// The first quotation mark +1 is the beginning of the description.
		description=strstr(buffer, "\"") + 1;
		// Remove the the lastquote.
		description[strlen(description) - 2]='\0';

		for(i=0; i <= eventmax; i++)
		{
			switch(eventlist[i].showhide) 
			{
				case 'v':	// Search each valid entry for a matching description.
				if(strstr(eventlist[i].description, description))
				{	// buffer[0] contains s or h.
					eventlist[i].showhide=buffer[0];
					// Open a file descriptor for this event.
					AddFD(&eventlist[i].fd, fdmax, i);
				}
				break;
			}
		}
	}

	fclose(config);

	return;
}
/******************************************************************/

/******************************************************************/
/* Saves event settings to a file. Settings are saved as
 * Action:Description.
 * 
 * This means show the cursor when this mouse generates an event:
 * s:"Logitech USB-PS/2 Optical Mouse"
 * 
 * This means hide the cursor when this keyboard generates an event:
 * h:"DELL DELL USB Keyboard"
 * 
 * Events are saved by description because event numbers can change.
 * Plugging in a mouse could cause a touchpads or touchscreens event
 * number to be different on the next boot. Save your settings with
 * all affected peripherals attached. If a description can't be found
 * when loading settings, it just gets skipped.
 */
static void SaveSettings(char *fullname, int append)
{
	int i;
	FILE *config;
	char *description;
	char mode[]="w";		// Open write only, create, truncate.

	if(append)
		mode[0]='a';			// Open write only, create, append.

	// umask defaults to 022 (User Group Other). That means write permission
	// bits are cleared for Group and Other. Setting umask to 0 means anyone
	// can edit the file even though the program gets run by root.
	umask(0);

	if((config=fopen(fullname, mode)) == NULL)
	{
		Cleanup();
		Bail(__LINE__, NULL);
	}

	for(i=0; i <= eventmax; i++)
	{
		switch(eventlist[i].showhide) 
		{
			case 's':
			case 'h':
			description=strstr(eventlist[i].description, "\"");
			fprintf(config, "%c:%s\n", eventlist[i].showhide, description);
			break;
		}
	}

	fclose(config);

	return;
}
/******************************************************************/

/******************************************************************/
/* Prints out a list of event numbers and desriptions like this:
 * 
 * Event#  Input device name
 * 0:      "Power Button"
 * 1:      "Power Button"
 * 2:      "AT Translated Set 2 keyboard"
 * 3:      "ImExPS/2 Logitech Explorer Mouse"
 * 4:      "PC Speaker"
 */

static void ShowDeviceInfo(void)
{
	int i;

	printf("Event#\tInput device name\n");

	for(i=0; i <= eventmax; i++)
	{
		if(eventlist[i].showhide)
		{
			printf("%s\n", eventlist[i].description);
		}
	}

	return;
}
/******************************************************************/

/******************************************************************/
/* Takes a whole number in the form of a string and returns an int.
 * A leading plus or minus sign is legal. Decimal points are not.
 */
static int String2Long(char *string, int caller)
{
	int i;

	if(IsNumber(string) == 0)
	{
		printf("%s is not a valid number.\n", string);
		Bail(caller, &InvalidArgument);
	}

	errno=0;
	i=strtol(string, NULL, 10);
	if(errno)
		Bail(caller, NULL);

	return(i);
}
/******************************************************************/

/******************************************************************/
/* Help message.
 */
static void Usage(void)
{

	printf("\n%s %s %s %s\n%s\n"
			"\n%s controls cursor visibility based on device\n"
			"events, such as mouse, keyboard, touch screen, etc.\n\n"
			"Usage: %s [-sN] [-hN] [-i] [-e] [-S file] [-A] [-L file]\n\n"
			"\t-s N\tShow cursor when event N occurs.\n"
			"\t-h N\tHide cursor when event N occurs.\n"
			"\t-i\tInitial state of cursor is hidden.\n"
			"\t-e\tDisplay list of available events.\n"
			"\t-S file\tSave event settings to file.\n"
			"\t-A\tAppend instead of overwrite when -S is used.\n"
			"\t-L file\tLoad event settings from file.\n"
			"\nExample: %s -i -s5 -h1 -h3 -s2 -S ~/.cursor.conf\n\n"
			"Starts with cursor hidden.\n"
			"Shows cursor when events 5 or 2 occur.\n"
			"Hides cursor when events 1 or 3 occur.\n"
			"Saves the event settings (but not -i) to a file.\n\n"
			, PROGRAM, VERSION, __DATE__, __TIME__, COPYRIGHT
			, PROGRAM, PROGRAM, PROGRAM);
	return;
}
/******************************************************************/

/******************************************************************/
/* Checks to see the pointer is not NULL, frees memeory, and sets
 * pointer equal to NULL
 */
static void Zfree(void **Pointer)
{
	if(*Pointer == NULL)
		return;
	free(*Pointer);
	*Pointer=NULL;
	return;
}
/******************************************************************/

/******************************************************************/
/* Mallocs a block of memory and zeros it out. If the malloc request
 * fails, the program aborts and prints the line number the Zmalloc
 * call originated from. Otherwise a pointer to valid memory is
 * returned.
 */
static void Zmalloc(void **Pointer, int size, int caller)
{
	if(*Pointer != NULL) Bail(caller, NULL);

	if((*Pointer=malloc(size)) == NULL) Bail(caller, NULL);

	memset(*Pointer, 0, size);
	return;
}
/******************************************************************/


/****************************** Main ******************************/
int main(int argc, char *argv[])
{
	int i;
	int changevisibility;			// Flag to indicate cursor visibility needs to be toggled.
	int savesettings=0;				// Flag to indicate settings are available to save.
	int append=0;					// Flag to tell SaveSettings() to append instead of overwrite.
	char *loadfile=NULL;			// If set, points to filename to load from.
	char *savefile=NULL;			// If set, points to filename to save to.
	int fdmax=0;					// Highest file descriptor select() should monitor.
	int opt;
	char buffer[4096];
	char showhide='s';				// Current cursor state. s=showing, h=hidden.
	fd_set Mcursor, Dcursor;		// Master cursor events list, Detected cursor events list.

	if(argc < 2)
	{	// Running program without arguments returns help message.
		Usage();
		return(0);
	}

	GetScreenParams(NULL);
	BuildEventlist();


	while ((opt=getopt(argc, argv, "-:is:h:eS:L:A")) != -1) 
	{
		switch(opt) 
		{
			case 'i':	// Initial state of cursor, hide.
			showhide='h';
			break;

			case 's':	// Add an event number to show the cursor.
			case 'h':	// Add an event number to hide the cursor.
			i=String2Long(optarg, __LINE__);
			if((i < 0) || (i > eventmax) || (eventlist[i].showhide == 0))
			{
				Bail(__LINE__, &NoSuchDevice);
			}
			AddFD(&eventlist[i].fd, &fdmax, i);
			eventlist[i].showhide=opt;
			break;

			case 'e':	// Print event list with descriptions.
			ShowDeviceInfo();
			Cleanup();
			return(0);
			break;

			case 'S':	// Save settings to a file.
			savefile=optarg;
			break;

			case 'L':	// Load settings from a file.
			loadfile=optarg;
			break;

			case 'A':	// Save settings to a file.
			append=1;	// Flag to tell SaveSettings() to append instead of overwrite.
			break;

			case '?':	// printf("Unknown option: %c\n", optopt);
			case ':':	// printf("Missing arg for %c\n", optopt);
			case 1:		// printf("Non-option arg: %s\n", optarg);
			Usage();	// Errors return help message.
			Cleanup();
			return(0);
			break;
		}
	}

	savesettings=InitFD_set(&Mcursor);

	if((savesettings) && (savefile))
		SaveSettings(savefile, append);

	if(loadfile)
	{	// Since eventlist, fdmax, and Mcursor were initialized, we start over.
		Cleanup();
		BuildEventlist();
		LoadSettings(loadfile, &fdmax);
		InitFD_set(&Mcursor);
	}

	if(showhide == 'h')
	{	// Gets set by -i.
		XFixesHideCursor(display, rootwindow);
		XFlush(display);
	}


	while(1)
	{
		Dcursor=Mcursor;
		// Wait for events to be generated.
		select((fdmax + 1), &Dcursor, NULL, NULL, NULL);
		changevisibility=0;
		for(i=0; i <= eventmax; i++)
		{	// Find which file descriptor generated the event.
			if(FD_ISSET(eventlist[i].fd, &Dcursor))
			{	// Read and discard all events. 
				while(read(eventlist[i].fd, buffer, sizeof(buffer) - 1) > 0);
				if(eventlist[i].showhide != showhide)
					changevisibility=1;
			}
		}

		if(changevisibility)
		{
			switch(showhide) 
			{
				case 's':	// Cursor is visible, hide it.
				XFixesHideCursor(display, rootwindow);
				showhide = 'h';
				break;

				case 'h':	// Cursor is hidden, show it.
				XFixesShowCursor(display, rootwindow);
				showhide = 's';
				break;
			}
			XFlush(display);
		}
	}

	Cleanup();
	return(0);
}
/******************************************************************/
