;;; ucs-utils.el --- Utilities for Unicode characters
;;
;; Copyright (c) 2012-2015 Roland Walker
;;
;; Author: Roland Walker <walker@pobox.com>
;; Homepage: http://github.com/rolandwalker/ucs-utils
;; URL: http://raw.githubusercontent.com/rolandwalker/ucs-utils/master/ucs-utils.el
;; Package-Version: 0.8.4
;; Package-Revision: v0.8.4-0-gcbfd42f822bf
;; Last-Updated: 26 Aug 2015
;; EmacsWiki: UcsUtils
;; Package-Requires: ((persistent-soft "0.8.8") (pcache "0.2.3") (list-utils "0.4.2"))
;; Keywords: i18n, extensions
;;
;; Simplified BSD License
;;
;;; Commentary:
;;
;; Quickstart
;;
;;      (require 'ucs-utils)
;;
;;      (ucs-utils-char "Middle Dot"         ; character to return
;;                      ?.                   ; fallback if unavailable
;;                      'char-displayable-p) ; test for character to pass
;;
;;      (ucs-utils-first-existing-char '("White Bullet"
;;                                       "Bullet Operator"
;;                                       "Circled Bullet"
;;                                       "Middle Dot"
;;                                       ?.) 'cdp)
;;
;;      (ucs-utils-string "Horizontal Ellipsis" '[["..."]])
;;
;; Explanation
;;
;; This library provides utilities for manipulating Unicode
;; characters, with integrated ability to return fallback characters
;; when Unicode display is not possible.
;;
;; Some ambiguities in Emacs' built-in Unicode data are resolved, and
;; character support is updated to Unicode 8.0.
;;
;; There are three interactive commands:
;;
;;     `ucs-utils-ucs-insert'       ; `ucs-insert' workalike using ido
;;     `ucs-utils-eval'             ; the inverse of `ucs-insert'
;;     `ucs-utils-install-aliases'  ; install shorter aliases
;;
;; The other functions are only useful from other Lisp code:
;;
;;     `ucs-utils-char'
;;     `ucs-utils-first-existing-char'
;;     `ucs-utils-vector'
;;     `ucs-utils-string'
;;     `ucs-utils-intact-string'
;;     `ucs-utils-pretty-name'
;;     `ucs-utils-read-char-by-name'
;;     `ucs-utils-subst-char-in-region'
;;
;; To use ucs-utils, place the ucs-utils.el library somewhere Emacs
;; can find it, and add the following to your ~/.emacs file:
;;
;;     (require 'ucs-utils)
;;
;; and optionally
;;
;;     (ucs-install-aliases)
;;
;; See Also
;;
;;     M-x customize-group RET ucs-utils RET
;;     http://en.wikipedia.org/wiki/Universal_Character_Set
;;
;; Notes
;;
;; Compatibility and Requirements
;;
;;     GNU Emacs version 25.1-devel     : not tested
;;     GNU Emacs version 24.5           : not tested
;;     GNU Emacs version 24.4           : yes
;;     GNU Emacs version 24.3           : yes
;;     GNU Emacs version 23.3           : yes (*)
;;     GNU Emacs version 22.3 and lower : no
;;
;;     (*) For full Emacs 23.x support, the library ucs-utils-6.0-delta.el
;;         should also be installed.
;;
;;     Uses if present: persistent-soft.el (Recommended)
;;
;; Bugs
;;
;; TODO
;;
;;     Accept synonyms on inputs? at least Tab would be nice.
;;     There is an official list of aliases at
;;
;;         http://www.unicode.org/Public/8.0.0/ucd/NameAliases.txt
;;
;;     generated names for CJK blocks added in Unicode 6.2
;;         CJK Unified Ideographs
;;         CJK Unified Ideographs Extension A
;;         CJK Unified Ideographs Extension C
;;
;;     support alternate naming schemes for CJK ideographs
;;
;;     support helm or other choosers which are able to cope
;;     with the entire set of character names, including
;;     CJK ideographs
;;
;;     spin out older portions of ucs-utils-names-corrections
;;     which are not needed in recent Emacs releases (as with
;;     ucs-utils-6.0-delta.el)
;;
;;     Namespace cache keys as with font-utils and unicode-utils.
;;
;;     Separate test run without persistent-soft.el
;;
;;; License
;;
;; Simplified BSD License:
;;
;; Redistribution and use in source and binary forms, with or
;; without modification, are permitted provided that the following
;; conditions are met:
;;
;;   1. Redistributions of source code must retain the above
;;      copyright notice, this list of conditions and the following
;;      disclaimer.
;;
;;   2. Redistributions in binary form must reproduce the above
;;      copyright notice, this list of conditions and the following
;;      disclaimer in the documentation and/or other materials
;;      provided with the distribution.
;;
;; This software is provided by Roland Walker "AS IS" and any express
;; or implied warranties, including, but not limited to, the implied
;; warranties of merchantability and fitness for a particular
;; purpose are disclaimed.  In no event shall Roland Walker or
;; contributors be liable for any direct, indirect, incidental,
;; special, exemplary, or consequential damages (including, but not
;; limited to, procurement of substitute goods or services; loss of
;; use, data, or profits; or business interruption) however caused
;; and on any theory of liability, whether in contract, strict
;; liability, or tort (including negligence or otherwise) arising in
;; any way out of the use of this software, even if advised of the
;; possibility of such damage.
;;
;; The views and conclusions contained in the software and
;; documentation are those of the authors and should not be
;; interpreted as representing official policies, either expressed
;; or implied, of Roland Walker.
;;
;; No rights are claimed over data created by the Unicode
;; Consortium, which are included here under the terms of
;; the Unicode Terms of Use.
;;
;;; Code:
;;

;;; requirements

(eval-and-compile
  ;; for callf, callf2, assert, loop, gensym, incf, remove-if
  (require 'cl))

(autoload 'pp                        "pp"              "Output the pretty-printed representation of OBJECT, any Lisp object.")
(autoload 'pp-display-expression     "pp"              "Prettify and display EXPRESSION in an appropriate way, depending on length.")

(autoload 'persistent-soft-store             "persistent-soft" "Under SYMBOL, store VALUE in the LOCATION persistent data store."    )
(autoload 'persistent-soft-fetch             "persistent-soft" "Return the value for SYMBOL in the LOCATION persistent data store."  )
(autoload 'persistent-soft-exists-p          "persistent-soft" "Return t if SYMBOL exists in the LOCATION persistent data store."    )
(autoload 'persistent-soft-flush             "persistent-soft" "Flush data for the LOCATION data store to disk."                     )
(autoload 'persistent-soft-location-readable "persistent-soft" "Return non-nil if LOCATION is a readable persistent-soft data store.")
(autoload 'persistent-soft-location-destroy  "persistent-soft" "Destroy LOCATION (a persistent-soft data store)."                    )

(autoload 'make-tconc                        "list-utils"      "Create a tconc data structure."                                      )
(autoload 'tconc-list                        "list-utils"      "Efficiently append LIST to TC."                                      )
(autoload 'list-utils-uniq                   "list-utils"      "Return a uniquified copy of LIST, preserving order."                 )

;;; declarations

(declare-function ucs-utils-orig-read-char-by-name "ucs-utils.el")
(declare-function tconc "list-utils.el")

;;; customizable variables

;;;###autoload
(defgroup ucs-utils nil
  "Utilities for Unicode characters."
  :version "0.8.4"
  :link '(emacs-commentary-link :tag "Commentary" "ucs-utils")
  :link '(url-link :tag "GitHub" "http://github.com/rolandwalker/ucs-utils")
  :link '(url-link :tag "EmacsWiki" "http://emacswiki.org/emacs/UcsUtils")
  :prefix "ucs-utils-"
  :group 'i18n
  :group 'extensions)

(defcustom ucs-utils-trade-memory-for-speed t
  "At the cost of memory, speed up checks for UCS characters.

The speedup is greater than an order of magnitude.  The memory
cost is roughly 3 Megabytes."
  :type 'boolean
  :group 'ucs-utils)

(defcustom ucs-utils-use-persistent-storage "ucs-utils"
  "Use persistent disk storage when available.

This speeds some operations between sessions.

Internally, this value is a string which is used for the filename
of the persistent data store."
  :type '(choice
          (const :tag "Yes"  "ucs-utils")
          (const :tag "No"   nil))
  :group 'ucs-utils)

(defcustom ucs-utils-hide-numbered-cjk-ideographs t
  "Hide numbered CJK ideographs from ido-style completion.

This is needed for performance reasons in most cases."
  :type 'boolean
  :group 'ucs-utils)

;;; variables

(defvar ucs-utils-names nil "Alist of cached (CHAR-NAME . CHAR-CODE) pairs, like `ucs-names'.")
(defvar ucs-utils-names-hash nil "A hashed copy of the `ucs-utils-names' alist.")
(defvar ucs-utils-all-prettified-names nil "List of all UCS names, prettified.")
(defvar persistent-soft-storage-expiration (* 60 60 24 30) "Number of seconds to keep on-disk storage.")

(defvar ucs-utils-names-deletions
  '(
    ;; erroneously returned by `get-char-code-property'
    ("CJK COMPATIBILITY IDEOGRAPH-FA6E"                          . #xFA6E)
    ("CJK COMPATIBILITY IDEOGRAPH-FA6F"                          . #xFA6F))
  "Invalid character names.")

(defvar ucs-utils-names-corrections
  '(
    ;; ambiguities: name collisions in various versions of the standard,
    ;; for which the latest is preferred
    ("LATIN CAPITAL LETTER YOGH"                                 . #x021C)
    ("LATIN SMALL LETTER YOGH"                                   . #x021D)
    ("CYRILLIC CAPITAL LETTER E"                                 . #x042D)
    ("CYRILLIC CAPITAL LETTER I"                                 . #x0418)
    ("CYRILLIC SMALL LETTER I"                                   . #x0438)
    ("CYRILLIC SMALL LETTER E"                                   . #x044D)
    ("DOUBLE VERTICAL BAR"                                       . #x23F8)
    ("GEORGIAN SMALL LETTER AN"                                  . #x2D00)
    ("GEORGIAN SMALL LETTER BAN"                                 . #x2D01)
    ("GEORGIAN SMALL LETTER GAN"                                 . #x2D02)
    ("GEORGIAN SMALL LETTER DON"                                 . #x2D03)
    ("GEORGIAN SMALL LETTER EN"                                  . #x2D04)
    ("GEORGIAN SMALL LETTER VIN"                                 . #x2D05)
    ("GEORGIAN SMALL LETTER ZEN"                                 . #x2D06)
    ("GEORGIAN SMALL LETTER TAN"                                 . #x2D07)
    ("GEORGIAN SMALL LETTER IN"                                  . #x2D08)
    ("GEORGIAN SMALL LETTER KAN"                                 . #x2D09)
    ("GEORGIAN SMALL LETTER LAS"                                 . #x2D0A)
    ("GEORGIAN SMALL LETTER MAN"                                 . #x2D0B)
    ("GEORGIAN SMALL LETTER NAR"                                 . #x2D0C)
    ("GEORGIAN SMALL LETTER ON"                                  . #x2D0D)
    ("GEORGIAN SMALL LETTER PAR"                                 . #x2D0E)
    ("GEORGIAN SMALL LETTER ZHAR"                                . #x2D0F)
    ("GEORGIAN SMALL LETTER RAE"                                 . #x2D10)
    ("GEORGIAN SMALL LETTER SAN"                                 . #x2D11)
    ("GEORGIAN SMALL LETTER TAR"                                 . #x2D12)
    ("GEORGIAN SMALL LETTER UN"                                  . #x2D13)
    ("GEORGIAN SMALL LETTER PHAR"                                . #x2D14)
    ("GEORGIAN SMALL LETTER KHAR"                                . #x2D15)
    ("GEORGIAN SMALL LETTER GHAN"                                . #x2D16)
    ("GEORGIAN SMALL LETTER QAR"                                 . #x2D17)
    ("GEORGIAN SMALL LETTER SHIN"                                . #x2D18)
    ("GEORGIAN SMALL LETTER CHIN"                                . #x2D19)
    ("GEORGIAN SMALL LETTER CAN"                                 . #x2D1A)
    ("GEORGIAN SMALL LETTER JIL"                                 . #x2D1B)
    ("GEORGIAN SMALL LETTER CIL"                                 . #x2D1C)
    ("GEORGIAN SMALL LETTER CHAR"                                . #x2D1D)
    ("GEORGIAN SMALL LETTER XAN"                                 . #x2D1E)
    ("GEORGIAN SMALL LETTER JHAN"                                . #x2D1F)
    ("GEORGIAN SMALL LETTER HAE"                                 . #x2D20)
    ("GEORGIAN SMALL LETTER HE"                                  . #x2D21)
    ("GEORGIAN SMALL LETTER HIE"                                 . #x2D22)
    ("GEORGIAN SMALL LETTER WE"                                  . #x2D23)
    ("GEORGIAN SMALL LETTER HAR"                                 . #x2D24)
    ("GEORGIAN SMALL LETTER HOE"                                 . #x2D25)
    ("HANGUL LETTER KIYEOK"                                      . #x3131)
    ("HANGUL LETTER PIEUP"                                       . #x3142)
    ("HANGUL LETTER CIEUC"                                       . #x3148)
    ("PARENTHESIZED HANGUL KIYEOK"                               . #x3200)
    ("PARENTHESIZED HANGUL PIEUP"                                . #x3205)
    ("PARENTHESIZED HANGUL CIEUC"                                . #x3208)
    ("CIRCLED HANGUL KIYEOK"                                     . #x3260)
    ("CIRCLED HANGUL PIEUP"                                      . #x3265)
    ("CIRCLED HANGUL CIEUC"                                      . #x3268)
    ("HALFWIDTH HANGUL LETTER KIYEOK"                            . #xFFA1)
    ("HALFWIDTH HANGUL LETTER PIEUP"                             . #xFFB2)
    ("HALFWIDTH HANGUL LETTER CIEUC"                             . #xFFB8)
    ("SQUARED MV"                                                . #x1F14B)
    ("BELL"                                                      . #x1F514)
    ;; this name is completely made up, to deal with #x1F514 BELL
    ("ALERT (BEL)"                                               . #x0007)
    ;; other ambiguities, such as a glitch in some Emacs release
    ("TIBETAN LETTER -A"                                         . #x0F60)
    ("TIBETAN SUBJOINED LETTER -A"                               . #x0FB0)
    ;; Unicode 6.0 to 6.1 delta
    ("ARMENIAN DRAM SIGN"                                        . #x058F)
    ("ARABIC SIGN SAMVAT"                                        . #x0604)
    ("ARABIC LETTER BEH WITH SMALL V BELOW"                      . #x08A0)
    ("ARABIC LETTER JEEM WITH TWO DOTS ABOVE"                    . #x08A2)
    ("ARABIC LETTER TAH WITH TWO DOTS ABOVE"                     . #x08A3)
    ("ARABIC LETTER FEH WITH DOT BELOW AND THREE DOTS ABOVE"     . #x08A4)
    ("ARABIC LETTER QAF WITH DOT BELOW"                          . #x08A5)
    ("ARABIC LETTER LAM WITH DOUBLE BAR"                         . #x08A6)
    ("ARABIC LETTER MEEM WITH THREE DOTS ABOVE"                  . #x08A7)
    ("ARABIC LETTER YEH WITH TWO DOTS BELOW AND HAMZA ABOVE"     . #x08A8)
    ("ARABIC LETTER YEH WITH TWO DOTS BELOW AND DOT ABOVE"       . #x08A9)
    ("ARABIC LETTER REH WITH LOOP"                               . #x08AA)
    ("ARABIC LETTER WAW WITH DOT WITHIN"                         . #x08AB)
    ("ARABIC LETTER ROHINGYA YEH"                                . #x08AC)
    ("ARABIC CURLY FATHA"                                        . #x08E4)
    ("ARABIC CURLY DAMMA"                                        . #x08E5)
    ("ARABIC CURLY KASRA"                                        . #x08E6)
    ("ARABIC CURLY FATHATAN"                                     . #x08E7)
    ("ARABIC CURLY DAMMATAN"                                     . #x08E8)
    ("ARABIC CURLY KASRATAN"                                     . #x08E9)
    ("ARABIC TONE ONE DOT ABOVE"                                 . #x08EA)
    ("ARABIC TONE TWO DOTS ABOVE"                                . #x08EB)
    ("ARABIC TONE LOOP ABOVE"                                    . #x08EC)
    ("ARABIC TONE ONE DOT BELOW"                                 . #x08ED)
    ("ARABIC TONE TWO DOTS BELOW"                                . #x08EE)
    ("ARABIC TONE LOOP BELOW"                                    . #x08EF)
    ("ARABIC OPEN FATHATAN"                                      . #x08F0)
    ("ARABIC OPEN DAMMATAN"                                      . #x08F1)
    ("ARABIC OPEN KASRATAN"                                      . #x08F2)
    ("ARABIC SMALL HIGH WAW"                                     . #x08F3)
    ("ARABIC FATHA WITH RING"                                    . #x08F4)
    ("ARABIC FATHA WITH DOT ABOVE"                               . #x08F5)
    ("ARABIC KASRA WITH DOT BELOW"                               . #x08F6)
    ("ARABIC LEFT ARROWHEAD ABOVE"                               . #x08F7)
    ("ARABIC RIGHT ARROWHEAD ABOVE"                              . #x08F8)
    ("ARABIC LEFT ARROWHEAD BELOW"                               . #x08F9)
    ("ARABIC RIGHT ARROWHEAD BELOW"                              . #x08FA)
    ("ARABIC DOUBLE RIGHT ARROWHEAD ABOVE"                       . #x08FB)
    ("ARABIC DOUBLE RIGHT ARROWHEAD ABOVE WITH DOT"              . #x08FC)
    ("ARABIC RIGHT ARROWHEAD ABOVE WITH DOT"                     . #x08FD)
    ("ARABIC DAMMA WITH DOT"                                     . #x08FE)
    ("GUJARATI ABBREVIATION SIGN"                                . #x0AF0)
    ("LAO LETTER KHMU GO"                                        . #x0EDE)
    ("LAO LETTER KHMU NYO"                                       . #x0EDF)
    ("GEORGIAN CAPITAL LETTER YN"                                . #x10C7)
    ("GEORGIAN CAPITAL LETTER AEN"                               . #x10CD)
    ("GEORGIAN LETTER AEN"                                       . #x10FD)
    ("GEORGIAN LETTER HARD SIGN"                                 . #x10FE)
    ("GEORGIAN LETTER LABIAL SIGN"                               . #x10FF)
    ("SUNDANESE SIGN VIRAMA"                                     . #x1BAB)
    ("SUNDANESE CONSONANT SIGN PASANGAN MA"                      . #x1BAC)
    ("SUNDANESE CONSONANT SIGN PASANGAN WA"                      . #x1BAD)
    ("SUNDANESE AVAGRAHA"                                        . #x1BBA)
    ("SUNDANESE LETTER REU"                                      . #x1BBB)
    ("SUNDANESE LETTER LEU"                                      . #x1BBC)
    ("SUNDANESE LETTER BHA"                                      . #x1BBD)
    ("SUNDANESE LETTER FINAL K"                                  . #x1BBE)
    ("SUNDANESE LETTER FINAL M"                                  . #x1BBF)
    ("SUNDANESE PUNCTUATION BINDU SURYA"                         . #x1CC0)
    ("SUNDANESE PUNCTUATION BINDU PANGLONG"                      . #x1CC1)
    ("SUNDANESE PUNCTUATION BINDU PURNAMA"                       . #x1CC2)
    ("SUNDANESE PUNCTUATION BINDU CAKRA"                         . #x1CC3)
    ("SUNDANESE PUNCTUATION BINDU LEU SATANGA"                   . #x1CC4)
    ("SUNDANESE PUNCTUATION BINDU KA SATANGA"                    . #x1CC5)
    ("SUNDANESE PUNCTUATION BINDU DA SATANGA"                    . #x1CC6)
    ("SUNDANESE PUNCTUATION BINDU BA SATANGA"                    . #x1CC7)
    ("VEDIC SIGN ROTATED ARDHAVISARGA"                           . #x1CF3)
    ("VEDIC TONE CANDRA ABOVE"                                   . #x1CF4)
    ("VEDIC SIGN JIHVAMULIYA"                                    . #x1CF5)
    ("VEDIC SIGN UPADHMANIYA"                                    . #x1CF6)
    ("MATHEMATICAL RISING DIAGONAL"                              . #x27CB)
    ("MATHEMATICAL FALLING DIAGONAL"                             . #x27CD)
    ("COPTIC CAPITAL LETTER BOHAIRIC KHEI"                       . #x2CF2)
    ("COPTIC SMALL LETTER BOHAIRIC KHEI"                         . #x2CF3)
    ("GEORGIAN SMALL LETTER YN"                                  . #x2D27)
    ("GEORGIAN SMALL LETTER AEN"                                 . #x2D2D)
    ("TIFINAGH LETTER YE"                                        . #x2D66)
    ("TIFINAGH LETTER YO"                                        . #x2D67)
    ("TURNED COMMA"                                              . #x2E32)
    ("RAISED DOT"                                                . #x2E33)
    ("RAISED COMMA"                                              . #x2E34)
    ("TURNED SEMICOLON"                                          . #x2E35)
    ("DAGGER WITH LEFT GUARD"                                    . #x2E36)
    ("DAGGER WITH RIGHT GUARD"                                   . #x2E37)
    ("TURNED DAGGER"                                             . #x2E38)
    ("TOP HALF SECTION SIGN"                                     . #x2E39)
    ("TWO-EM DASH"                                               . #x2E3A)
    ("THREE-EM DASH"                                             . #x2E3B)
    ("CJK IDEOGRAPH, LAST"                                       . #x9FCC)
    ("COMBINING CYRILLIC LETTER UKRAINIAN IE"                    . #xA674)
    ("COMBINING CYRILLIC LETTER I"                               . #xA675)
    ("COMBINING CYRILLIC LETTER YI"                              . #xA676)
    ("COMBINING CYRILLIC LETTER U"                               . #xA677)
    ("COMBINING CYRILLIC LETTER HARD SIGN"                       . #xA678)
    ("COMBINING CYRILLIC LETTER YERU"                            . #xA679)
    ("COMBINING CYRILLIC LETTER SOFT SIGN"                       . #xA67A)
    ("COMBINING CYRILLIC LETTER OMEGA"                           . #xA67B)
    ("COMBINING CYRILLIC LETTER IOTIFIED E"                      . #xA69F)
    ("LATIN CAPITAL LETTER C WITH BAR"                           . #xA792)
    ("LATIN SMALL LETTER C WITH BAR"                             . #xA793)
    ("LATIN CAPITAL LETTER H WITH HOOK"                          . #xA7AA)
    ("MODIFIER LETTER CAPITAL H WITH STROKE"                     . #xA7F8)
    ("MODIFIER LETTER SMALL LIGATURE OE"                         . #xA7F9)
    ("MEETEI MAYEK LETTER E"                                     . #xAAE0)
    ("MEETEI MAYEK LETTER O"                                     . #xAAE1)
    ("MEETEI MAYEK LETTER CHA"                                   . #xAAE2)
    ("MEETEI MAYEK LETTER NYA"                                   . #xAAE3)
    ("MEETEI MAYEK LETTER TTA"                                   . #xAAE4)
    ("MEETEI MAYEK LETTER TTHA"                                  . #xAAE5)
    ("MEETEI MAYEK LETTER DDA"                                   . #xAAE6)
    ("MEETEI MAYEK LETTER DDHA"                                  . #xAAE7)
    ("MEETEI MAYEK LETTER NNA"                                   . #xAAE8)
    ("MEETEI MAYEK LETTER SHA"                                   . #xAAE9)
    ("MEETEI MAYEK LETTER SSA"                                   . #xAAEA)
    ("MEETEI MAYEK VOWEL SIGN II"                                . #xAAEB)
    ("MEETEI MAYEK VOWEL SIGN UU"                                . #xAAEC)
    ("MEETEI MAYEK VOWEL SIGN AAI"                               . #xAAED)
    ("MEETEI MAYEK VOWEL SIGN AU"                                . #xAAEE)
    ("MEETEI MAYEK VOWEL SIGN AAU"                               . #xAAEF)
    ("MEETEI MAYEK CHEIKHAN"                                     . #xAAF0)
    ("MEETEI MAYEK AHANG KHUDAM"                                 . #xAAF1)
    ("MEETEI MAYEK ANJI"                                         . #xAAF2)
    ("MEETEI MAYEK SYLLABLE REPETITION MARK"                     . #xAAF3)
    ("MEETEI MAYEK WORD REPETITION MARK"                         . #xAAF4)
    ("MEETEI MAYEK VOWEL SIGN VISARGA"                           . #xAAF5)
    ("MEETEI MAYEK VIRAMA"                                       . #xAAF6)
    ("CJK COMPATIBILITY IDEOGRAPH-FA2E"                          . #xFA2E)
    ("CJK COMPATIBILITY IDEOGRAPH-FA2F"                          . #xFA2F)
    ("MEROITIC HIEROGLYPHIC LETTER A"                            . #x10980)
    ("MEROITIC HIEROGLYPHIC LETTER E"                            . #x10981)
    ("MEROITIC HIEROGLYPHIC LETTER I"                            . #x10982)
    ("MEROITIC HIEROGLYPHIC LETTER O"                            . #x10983)
    ("MEROITIC HIEROGLYPHIC LETTER YA"                           . #x10984)
    ("MEROITIC HIEROGLYPHIC LETTER WA"                           . #x10985)
    ("MEROITIC HIEROGLYPHIC LETTER BA"                           . #x10986)
    ("MEROITIC HIEROGLYPHIC LETTER BA-2"                         . #x10987)
    ("MEROITIC HIEROGLYPHIC LETTER PA"                           . #x10988)
    ("MEROITIC HIEROGLYPHIC LETTER MA"                           . #x10989)
    ("MEROITIC HIEROGLYPHIC LETTER NA"                           . #x1098A)
    ("MEROITIC HIEROGLYPHIC LETTER NA-2"                         . #x1098B)
    ("MEROITIC HIEROGLYPHIC LETTER NE"                           . #x1098C)
    ("MEROITIC HIEROGLYPHIC LETTER NE-2"                         . #x1098D)
    ("MEROITIC HIEROGLYPHIC LETTER RA"                           . #x1098E)
    ("MEROITIC HIEROGLYPHIC LETTER RA-2"                         . #x1098F)
    ("MEROITIC HIEROGLYPHIC LETTER LA"                           . #x10990)
    ("MEROITIC HIEROGLYPHIC LETTER KHA"                          . #x10991)
    ("MEROITIC HIEROGLYPHIC LETTER HHA"                          . #x10992)
    ("MEROITIC HIEROGLYPHIC LETTER SA"                           . #x10993)
    ("MEROITIC HIEROGLYPHIC LETTER SA-2"                         . #x10994)
    ("MEROITIC HIEROGLYPHIC LETTER SE"                           . #x10995)
    ("MEROITIC HIEROGLYPHIC LETTER KA"                           . #x10996)
    ("MEROITIC HIEROGLYPHIC LETTER QA"                           . #x10997)
    ("MEROITIC HIEROGLYPHIC LETTER TA"                           . #x10998)
    ("MEROITIC HIEROGLYPHIC LETTER TA-2"                         . #x10999)
    ("MEROITIC HIEROGLYPHIC LETTER TE"                           . #x1099A)
    ("MEROITIC HIEROGLYPHIC LETTER TE-2"                         . #x1099B)
    ("MEROITIC HIEROGLYPHIC LETTER TO"                           . #x1099C)
    ("MEROITIC HIEROGLYPHIC LETTER DA"                           . #x1099D)
    ("MEROITIC HIEROGLYPHIC SYMBOL VIDJ"                         . #x1099E)
    ("MEROITIC HIEROGLYPHIC SYMBOL VIDJ-2"                       . #x1099F)
    ("MEROITIC CURSIVE LETTER A"                                 . #x109A0)
    ("MEROITIC CURSIVE LETTER E"                                 . #x109A1)
    ("MEROITIC CURSIVE LETTER I"                                 . #x109A2)
    ("MEROITIC CURSIVE LETTER O"                                 . #x109A3)
    ("MEROITIC CURSIVE LETTER YA"                                . #x109A4)
    ("MEROITIC CURSIVE LETTER WA"                                . #x109A5)
    ("MEROITIC CURSIVE LETTER BA"                                . #x109A6)
    ("MEROITIC CURSIVE LETTER PA"                                . #x109A7)
    ("MEROITIC CURSIVE LETTER MA"                                . #x109A8)
    ("MEROITIC CURSIVE LETTER NA"                                . #x109A9)
    ("MEROITIC CURSIVE LETTER NE"                                . #x109AA)
    ("MEROITIC CURSIVE LETTER RA"                                . #x109AB)
    ("MEROITIC CURSIVE LETTER LA"                                . #x109AC)
    ("MEROITIC CURSIVE LETTER KHA"                               . #x109AD)
    ("MEROITIC CURSIVE LETTER HHA"                               . #x109AE)
    ("MEROITIC CURSIVE LETTER SA"                                . #x109AF)
    ("MEROITIC CURSIVE LETTER ARCHAIC SA"                        . #x109B0)
    ("MEROITIC CURSIVE LETTER SE"                                . #x109B1)
    ("MEROITIC CURSIVE LETTER KA"                                . #x109B2)
    ("MEROITIC CURSIVE LETTER QA"                                . #x109B3)
    ("MEROITIC CURSIVE LETTER TA"                                . #x109B4)
    ("MEROITIC CURSIVE LETTER TE"                                . #x109B5)
    ("MEROITIC CURSIVE LETTER TO"                                . #x109B6)
    ("MEROITIC CURSIVE LETTER DA"                                . #x109B7)
    ("MEROITIC CURSIVE LOGOGRAM RMT"                             . #x109BE)
    ("MEROITIC CURSIVE LOGOGRAM IMN"                             . #x109BF)
    ("SORA SOMPENG LETTER SAH"                                   . #x110D0)
    ("SORA SOMPENG LETTER TAH"                                   . #x110D1)
    ("SORA SOMPENG LETTER BAH"                                   . #x110D2)
    ("SORA SOMPENG LETTER CAH"                                   . #x110D3)
    ("SORA SOMPENG LETTER DAH"                                   . #x110D4)
    ("SORA SOMPENG LETTER GAH"                                   . #x110D5)
    ("SORA SOMPENG LETTER MAH"                                   . #x110D6)
    ("SORA SOMPENG LETTER NGAH"                                  . #x110D7)
    ("SORA SOMPENG LETTER LAH"                                   . #x110D8)
    ("SORA SOMPENG LETTER NAH"                                   . #x110D9)
    ("SORA SOMPENG LETTER VAH"                                   . #x110DA)
    ("SORA SOMPENG LETTER PAH"                                   . #x110DB)
    ("SORA SOMPENG LETTER YAH"                                   . #x110DC)
    ("SORA SOMPENG LETTER RAH"                                   . #x110DD)
    ("SORA SOMPENG LETTER HAH"                                   . #x110DE)
    ("SORA SOMPENG LETTER KAH"                                   . #x110DF)
    ("SORA SOMPENG LETTER JAH"                                   . #x110E0)
    ("SORA SOMPENG LETTER NYAH"                                  . #x110E1)
    ("SORA SOMPENG LETTER AH"                                    . #x110E2)
    ("SORA SOMPENG LETTER EEH"                                   . #x110E3)
    ("SORA SOMPENG LETTER IH"                                    . #x110E4)
    ("SORA SOMPENG LETTER UH"                                    . #x110E5)
    ("SORA SOMPENG LETTER OH"                                    . #x110E6)
    ("SORA SOMPENG LETTER EH"                                    . #x110E7)
    ("SORA SOMPENG LETTER MAE"                                   . #x110E8)
    ("SORA SOMPENG DIGIT ZERO"                                   . #x110F0)
    ("SORA SOMPENG DIGIT ONE"                                    . #x110F1)
    ("SORA SOMPENG DIGIT TWO"                                    . #x110F2)
    ("SORA SOMPENG DIGIT THREE"                                  . #x110F3)
    ("SORA SOMPENG DIGIT FOUR"                                   . #x110F4)
    ("SORA SOMPENG DIGIT FIVE"                                   . #x110F5)
    ("SORA SOMPENG DIGIT SIX"                                    . #x110F6)
    ("SORA SOMPENG DIGIT SEVEN"                                  . #x110F7)
    ("SORA SOMPENG DIGIT EIGHT"                                  . #x110F8)
    ("SORA SOMPENG DIGIT NINE"                                   . #x110F9)
    ("CHAKMA SIGN CANDRABINDU"                                   . #x11100)
    ("CHAKMA SIGN ANUSVARA"                                      . #x11101)
    ("CHAKMA SIGN VISARGA"                                       . #x11102)
    ("CHAKMA LETTER AA"                                          . #x11103)
    ("CHAKMA LETTER I"                                           . #x11104)
    ("CHAKMA LETTER U"                                           . #x11105)
    ("CHAKMA LETTER E"                                           . #x11106)
    ("CHAKMA LETTER KAA"                                         . #x11107)
    ("CHAKMA LETTER KHAA"                                        . #x11108)
    ("CHAKMA LETTER GAA"                                         . #x11109)
    ("CHAKMA LETTER GHAA"                                        . #x1110A)
    ("CHAKMA LETTER NGAA"                                        . #x1110B)
    ("CHAKMA LETTER CAA"                                         . #x1110C)
    ("CHAKMA LETTER CHAA"                                        . #x1110D)
    ("CHAKMA LETTER JAA"                                         . #x1110E)
    ("CHAKMA LETTER JHAA"                                        . #x1110F)
    ("CHAKMA LETTER NYAA"                                        . #x11110)
    ("CHAKMA LETTER TTAA"                                        . #x11111)
    ("CHAKMA LETTER TTHAA"                                       . #x11112)
    ("CHAKMA LETTER DDAA"                                        . #x11113)
    ("CHAKMA LETTER DDHAA"                                       . #x11114)
    ("CHAKMA LETTER NNAA"                                        . #x11115)
    ("CHAKMA LETTER TAA"                                         . #x11116)
    ("CHAKMA LETTER THAA"                                        . #x11117)
    ("CHAKMA LETTER DAA"                                         . #x11118)
    ("CHAKMA LETTER DHAA"                                        . #x11119)
    ("CHAKMA LETTER NAA"                                         . #x1111A)
    ("CHAKMA LETTER PAA"                                         . #x1111B)
    ("CHAKMA LETTER PHAA"                                        . #x1111C)
    ("CHAKMA LETTER BAA"                                         . #x1111D)
    ("CHAKMA LETTER BHAA"                                        . #x1111E)
    ("CHAKMA LETTER MAA"                                         . #x1111F)
    ("CHAKMA LETTER YYAA"                                        . #x11120)
    ("CHAKMA LETTER YAA"                                         . #x11121)
    ("CHAKMA LETTER RAA"                                         . #x11122)
    ("CHAKMA LETTER LAA"                                         . #x11123)
    ("CHAKMA LETTER WAA"                                         . #x11124)
    ("CHAKMA LETTER SAA"                                         . #x11125)
    ("CHAKMA LETTER HAA"                                         . #x11126)
    ("CHAKMA VOWEL SIGN A"                                       . #x11127)
    ("CHAKMA VOWEL SIGN I"                                       . #x11128)
    ("CHAKMA VOWEL SIGN II"                                      . #x11129)
    ("CHAKMA VOWEL SIGN U"                                       . #x1112A)
    ("CHAKMA VOWEL SIGN UU"                                      . #x1112B)
    ("CHAKMA VOWEL SIGN E"                                       . #x1112C)
    ("CHAKMA VOWEL SIGN AI"                                      . #x1112D)
    ("CHAKMA VOWEL SIGN O"                                       . #x1112E)
    ("CHAKMA VOWEL SIGN AU"                                      . #x1112F)
    ("CHAKMA VOWEL SIGN OI"                                      . #x11130)
    ("CHAKMA O MARK"                                             . #x11131)
    ("CHAKMA AU MARK"                                            . #x11132)
    ("CHAKMA VIRAMA"                                             . #x11133)
    ("CHAKMA MAAYYAA"                                            . #x11134)
    ("CHAKMA DIGIT ZERO"                                         . #x11136)
    ("CHAKMA DIGIT ONE"                                          . #x11137)
    ("CHAKMA DIGIT TWO"                                          . #x11138)
    ("CHAKMA DIGIT THREE"                                        . #x11139)
    ("CHAKMA DIGIT FOUR"                                         . #x1113A)
    ("CHAKMA DIGIT FIVE"                                         . #x1113B)
    ("CHAKMA DIGIT SIX"                                          . #x1113C)
    ("CHAKMA DIGIT SEVEN"                                        . #x1113D)
    ("CHAKMA DIGIT EIGHT"                                        . #x1113E)
    ("CHAKMA DIGIT NINE"                                         . #x1113F)
    ("CHAKMA SECTION MARK"                                       . #x11140)
    ("CHAKMA DANDA"                                              . #x11141)
    ("CHAKMA DOUBLE DANDA"                                       . #x11142)
    ("CHAKMA QUESTION MARK"                                      . #x11143)
    ("SHARADA SIGN CANDRABINDU"                                  . #x11180)
    ("SHARADA SIGN ANUSVARA"                                     . #x11181)
    ("SHARADA SIGN VISARGA"                                      . #x11182)
    ("SHARADA LETTER A"                                          . #x11183)
    ("SHARADA LETTER AA"                                         . #x11184)
    ("SHARADA LETTER I"                                          . #x11185)
    ("SHARADA LETTER II"                                         . #x11186)
    ("SHARADA LETTER U"                                          . #x11187)
    ("SHARADA LETTER UU"                                         . #x11188)
    ("SHARADA LETTER VOCALIC R"                                  . #x11189)
    ("SHARADA LETTER VOCALIC RR"                                 . #x1118A)
    ("SHARADA LETTER VOCALIC L"                                  . #x1118B)
    ("SHARADA LETTER VOCALIC LL"                                 . #x1118C)
    ("SHARADA LETTER E"                                          . #x1118D)
    ("SHARADA LETTER AI"                                         . #x1118E)
    ("SHARADA LETTER O"                                          . #x1118F)
    ("SHARADA LETTER AU"                                         . #x11190)
    ("SHARADA LETTER KA"                                         . #x11191)
    ("SHARADA LETTER KHA"                                        . #x11192)
    ("SHARADA LETTER GA"                                         . #x11193)
    ("SHARADA LETTER GHA"                                        . #x11194)
    ("SHARADA LETTER NGA"                                        . #x11195)
    ("SHARADA LETTER CA"                                         . #x11196)
    ("SHARADA LETTER CHA"                                        . #x11197)
    ("SHARADA LETTER JA"                                         . #x11198)
    ("SHARADA LETTER JHA"                                        . #x11199)
    ("SHARADA LETTER NYA"                                        . #x1119A)
    ("SHARADA LETTER TTA"                                        . #x1119B)
    ("SHARADA LETTER TTHA"                                       . #x1119C)
    ("SHARADA LETTER DDA"                                        . #x1119D)
    ("SHARADA LETTER DDHA"                                       . #x1119E)
    ("SHARADA LETTER NNA"                                        . #x1119F)
    ("SHARADA LETTER TA"                                         . #x111A0)
    ("SHARADA LETTER THA"                                        . #x111A1)
    ("SHARADA LETTER DA"                                         . #x111A2)
    ("SHARADA LETTER DHA"                                        . #x111A3)
    ("SHARADA LETTER NA"                                         . #x111A4)
    ("SHARADA LETTER PA"                                         . #x111A5)
    ("SHARADA LETTER PHA"                                        . #x111A6)
    ("SHARADA LETTER BA"                                         . #x111A7)
    ("SHARADA LETTER BHA"                                        . #x111A8)
    ("SHARADA LETTER MA"                                         . #x111A9)
    ("SHARADA LETTER YA"                                         . #x111AA)
    ("SHARADA LETTER RA"                                         . #x111AB)
    ("SHARADA LETTER LA"                                         . #x111AC)
    ("SHARADA LETTER LLA"                                        . #x111AD)
    ("SHARADA LETTER VA"                                         . #x111AE)
    ("SHARADA LETTER SHA"                                        . #x111AF)
    ("SHARADA LETTER SSA"                                        . #x111B0)
    ("SHARADA LETTER SA"                                         . #x111B1)
    ("SHARADA LETTER HA"                                         . #x111B2)
    ("SHARADA VOWEL SIGN AA"                                     . #x111B3)
    ("SHARADA VOWEL SIGN I"                                      . #x111B4)
    ("SHARADA VOWEL SIGN II"                                     . #x111B5)
    ("SHARADA VOWEL SIGN U"                                      . #x111B6)
    ("SHARADA VOWEL SIGN UU"                                     . #x111B7)
    ("SHARADA VOWEL SIGN VOCALIC R"                              . #x111B8)
    ("SHARADA VOWEL SIGN VOCALIC RR"                             . #x111B9)
    ("SHARADA VOWEL SIGN VOCALIC L"                              . #x111BA)
    ("SHARADA VOWEL SIGN VOCALIC LL"                             . #x111BB)
    ("SHARADA VOWEL SIGN E"                                      . #x111BC)
    ("SHARADA VOWEL SIGN AI"                                     . #x111BD)
    ("SHARADA VOWEL SIGN O"                                      . #x111BE)
    ("SHARADA VOWEL SIGN AU"                                     . #x111BF)
    ("SHARADA SIGN VIRAMA"                                       . #x111C0)
    ("SHARADA SIGN AVAGRAHA"                                     . #x111C1)
    ("SHARADA SIGN JIHVAMULIYA"                                  . #x111C2)
    ("SHARADA SIGN UPADHMANIYA"                                  . #x111C3)
    ("SHARADA OM"                                                . #x111C4)
    ("SHARADA DANDA"                                             . #x111C5)
    ("SHARADA DOUBLE DANDA"                                      . #x111C6)
    ("SHARADA ABBREVIATION SIGN"                                 . #x111C7)
    ("SHARADA SEPARATOR"                                         . #x111C8)
    ("SHARADA DIGIT ZERO"                                        . #x111D0)
    ("SHARADA DIGIT ONE"                                         . #x111D1)
    ("SHARADA DIGIT TWO"                                         . #x111D2)
    ("SHARADA DIGIT THREE"                                       . #x111D3)
    ("SHARADA DIGIT FOUR"                                        . #x111D4)
    ("SHARADA DIGIT FIVE"                                        . #x111D5)
    ("SHARADA DIGIT SIX"                                         . #x111D6)
    ("SHARADA DIGIT SEVEN"                                       . #x111D7)
    ("SHARADA DIGIT EIGHT"                                       . #x111D8)
    ("SHARADA DIGIT NINE"                                        . #x111D9)
    ("TAKRI LETTER A"                                            . #x11680)
    ("TAKRI LETTER AA"                                           . #x11681)
    ("TAKRI LETTER I"                                            . #x11682)
    ("TAKRI LETTER II"                                           . #x11683)
    ("TAKRI LETTER U"                                            . #x11684)
    ("TAKRI LETTER UU"                                           . #x11685)
    ("TAKRI LETTER E"                                            . #x11686)
    ("TAKRI LETTER AI"                                           . #x11687)
    ("TAKRI LETTER O"                                            . #x11688)
    ("TAKRI LETTER AU"                                           . #x11689)
    ("TAKRI LETTER KA"                                           . #x1168A)
    ("TAKRI LETTER KHA"                                          . #x1168B)
    ("TAKRI LETTER GA"                                           . #x1168C)
    ("TAKRI LETTER GHA"                                          . #x1168D)
    ("TAKRI LETTER NGA"                                          . #x1168E)
    ("TAKRI LETTER CA"                                           . #x1168F)
    ("TAKRI LETTER CHA"                                          . #x11690)
    ("TAKRI LETTER JA"                                           . #x11691)
    ("TAKRI LETTER JHA"                                          . #x11692)
    ("TAKRI LETTER NYA"                                          . #x11693)
    ("TAKRI LETTER TTA"                                          . #x11694)
    ("TAKRI LETTER TTHA"                                         . #x11695)
    ("TAKRI LETTER DDA"                                          . #x11696)
    ("TAKRI LETTER DDHA"                                         . #x11697)
    ("TAKRI LETTER NNA"                                          . #x11698)
    ("TAKRI LETTER TA"                                           . #x11699)
    ("TAKRI LETTER THA"                                          . #x1169A)
    ("TAKRI LETTER DA"                                           . #x1169B)
    ("TAKRI LETTER DHA"                                          . #x1169C)
    ("TAKRI LETTER NA"                                           . #x1169D)
    ("TAKRI LETTER PA"                                           . #x1169E)
    ("TAKRI LETTER PHA"                                          . #x1169F)
    ("TAKRI LETTER BA"                                           . #x116A0)
    ("TAKRI LETTER BHA"                                          . #x116A1)
    ("TAKRI LETTER MA"                                           . #x116A2)
    ("TAKRI LETTER YA"                                           . #x116A3)
    ("TAKRI LETTER RA"                                           . #x116A4)
    ("TAKRI LETTER LA"                                           . #x116A5)
    ("TAKRI LETTER VA"                                           . #x116A6)
    ("TAKRI LETTER SHA"                                          . #x116A7)
    ("TAKRI LETTER SA"                                           . #x116A8)
    ("TAKRI LETTER HA"                                           . #x116A9)
    ("TAKRI LETTER RRA"                                          . #x116AA)
    ("TAKRI SIGN ANUSVARA"                                       . #x116AB)
    ("TAKRI SIGN VISARGA"                                        . #x116AC)
    ("TAKRI VOWEL SIGN AA"                                       . #x116AD)
    ("TAKRI VOWEL SIGN I"                                        . #x116AE)
    ("TAKRI VOWEL SIGN II"                                       . #x116AF)
    ("TAKRI VOWEL SIGN U"                                        . #x116B0)
    ("TAKRI VOWEL SIGN UU"                                       . #x116B1)
    ("TAKRI VOWEL SIGN E"                                        . #x116B2)
    ("TAKRI VOWEL SIGN AI"                                       . #x116B3)
    ("TAKRI VOWEL SIGN O"                                        . #x116B4)
    ("TAKRI VOWEL SIGN AU"                                       . #x116B5)
    ("TAKRI SIGN VIRAMA"                                         . #x116B6)
    ("TAKRI SIGN NUKTA"                                          . #x116B7)
    ("TAKRI DIGIT ZERO"                                          . #x116C0)
    ("TAKRI DIGIT ONE"                                           . #x116C1)
    ("TAKRI DIGIT TWO"                                           . #x116C2)
    ("TAKRI DIGIT THREE"                                         . #x116C3)
    ("TAKRI DIGIT FOUR"                                          . #x116C4)
    ("TAKRI DIGIT FIVE"                                          . #x116C5)
    ("TAKRI DIGIT SIX"                                           . #x116C6)
    ("TAKRI DIGIT SEVEN"                                         . #x116C7)
    ("TAKRI DIGIT EIGHT"                                         . #x116C8)
    ("TAKRI DIGIT NINE"                                          . #x116C9)
    ("MIAO LETTER PA"                                            . #x16F00)
    ("MIAO LETTER BA"                                            . #x16F01)
    ("MIAO LETTER YI PA"                                         . #x16F02)
    ("MIAO LETTER PLA"                                           . #x16F03)
    ("MIAO LETTER MA"                                            . #x16F04)
    ("MIAO LETTER MHA"                                           . #x16F05)
    ("MIAO LETTER ARCHAIC MA"                                    . #x16F06)
    ("MIAO LETTER FA"                                            . #x16F07)
    ("MIAO LETTER VA"                                            . #x16F08)
    ("MIAO LETTER VFA"                                           . #x16F09)
    ("MIAO LETTER TA"                                            . #x16F0A)
    ("MIAO LETTER DA"                                            . #x16F0B)
    ("MIAO LETTER YI TTA"                                        . #x16F0C)
    ("MIAO LETTER YI TA"                                         . #x16F0D)
    ("MIAO LETTER TTA"                                           . #x16F0E)
    ("MIAO LETTER DDA"                                           . #x16F0F)
    ("MIAO LETTER NA"                                            . #x16F10)
    ("MIAO LETTER NHA"                                           . #x16F11)
    ("MIAO LETTER YI NNA"                                        . #x16F12)
    ("MIAO LETTER ARCHAIC NA"                                    . #x16F13)
    ("MIAO LETTER NNA"                                           . #x16F14)
    ("MIAO LETTER NNHA"                                          . #x16F15)
    ("MIAO LETTER LA"                                            . #x16F16)
    ("MIAO LETTER LYA"                                           . #x16F17)
    ("MIAO LETTER LHA"                                           . #x16F18)
    ("MIAO LETTER LHYA"                                          . #x16F19)
    ("MIAO LETTER TLHA"                                          . #x16F1A)
    ("MIAO LETTER DLHA"                                          . #x16F1B)
    ("MIAO LETTER TLHYA"                                         . #x16F1C)
    ("MIAO LETTER DLHYA"                                         . #x16F1D)
    ("MIAO LETTER KA"                                            . #x16F1E)
    ("MIAO LETTER GA"                                            . #x16F1F)
    ("MIAO LETTER YI KA"                                         . #x16F20)
    ("MIAO LETTER QA"                                            . #x16F21)
    ("MIAO LETTER QGA"                                           . #x16F22)
    ("MIAO LETTER NGA"                                           . #x16F23)
    ("MIAO LETTER NGHA"                                          . #x16F24)
    ("MIAO LETTER ARCHAIC NGA"                                   . #x16F25)
    ("MIAO LETTER HA"                                            . #x16F26)
    ("MIAO LETTER XA"                                            . #x16F27)
    ("MIAO LETTER GHA"                                           . #x16F28)
    ("MIAO LETTER GHHA"                                          . #x16F29)
    ("MIAO LETTER TSSA"                                          . #x16F2A)
    ("MIAO LETTER DZZA"                                          . #x16F2B)
    ("MIAO LETTER NYA"                                           . #x16F2C)
    ("MIAO LETTER NYHA"                                          . #x16F2D)
    ("MIAO LETTER TSHA"                                          . #x16F2E)
    ("MIAO LETTER DZHA"                                          . #x16F2F)
    ("MIAO LETTER YI TSHA"                                       . #x16F30)
    ("MIAO LETTER YI DZHA"                                       . #x16F31)
    ("MIAO LETTER REFORMED TSHA"                                 . #x16F32)
    ("MIAO LETTER SHA"                                           . #x16F33)
    ("MIAO LETTER SSA"                                           . #x16F34)
    ("MIAO LETTER ZHA"                                           . #x16F35)
    ("MIAO LETTER ZSHA"                                          . #x16F36)
    ("MIAO LETTER TSA"                                           . #x16F37)
    ("MIAO LETTER DZA"                                           . #x16F38)
    ("MIAO LETTER YI TSA"                                        . #x16F39)
    ("MIAO LETTER SA"                                            . #x16F3A)
    ("MIAO LETTER ZA"                                            . #x16F3B)
    ("MIAO LETTER ZSA"                                           . #x16F3C)
    ("MIAO LETTER ZZA"                                           . #x16F3D)
    ("MIAO LETTER ZZSA"                                          . #x16F3E)
    ("MIAO LETTER ARCHAIC ZZA"                                   . #x16F3F)
    ("MIAO LETTER ZZYA"                                          . #x16F40)
    ("MIAO LETTER ZZSYA"                                         . #x16F41)
    ("MIAO LETTER WA"                                            . #x16F42)
    ("MIAO LETTER AH"                                            . #x16F43)
    ("MIAO LETTER HHA"                                           . #x16F44)
    ("MIAO LETTER NASALIZATION"                                  . #x16F50)
    ("MIAO SIGN ASPIRATION"                                      . #x16F51)
    ("MIAO SIGN REFORMED VOICING"                                . #x16F52)
    ("MIAO SIGN REFORMED ASPIRATION"                             . #x16F53)
    ("MIAO VOWEL SIGN A"                                         . #x16F54)
    ("MIAO VOWEL SIGN AA"                                        . #x16F55)
    ("MIAO VOWEL SIGN AHH"                                       . #x16F56)
    ("MIAO VOWEL SIGN AN"                                        . #x16F57)
    ("MIAO VOWEL SIGN ANG"                                       . #x16F58)
    ("MIAO VOWEL SIGN O"                                         . #x16F59)
    ("MIAO VOWEL SIGN OO"                                        . #x16F5A)
    ("MIAO VOWEL SIGN WO"                                        . #x16F5B)
    ("MIAO VOWEL SIGN W"                                         . #x16F5C)
    ("MIAO VOWEL SIGN E"                                         . #x16F5D)
    ("MIAO VOWEL SIGN EN"                                        . #x16F5E)
    ("MIAO VOWEL SIGN ENG"                                       . #x16F5F)
    ("MIAO VOWEL SIGN OEY"                                       . #x16F60)
    ("MIAO VOWEL SIGN I"                                         . #x16F61)
    ("MIAO VOWEL SIGN IA"                                        . #x16F62)
    ("MIAO VOWEL SIGN IAN"                                       . #x16F63)
    ("MIAO VOWEL SIGN IANG"                                      . #x16F64)
    ("MIAO VOWEL SIGN IO"                                        . #x16F65)
    ("MIAO VOWEL SIGN IE"                                        . #x16F66)
    ("MIAO VOWEL SIGN II"                                        . #x16F67)
    ("MIAO VOWEL SIGN IU"                                        . #x16F68)
    ("MIAO VOWEL SIGN ING"                                       . #x16F69)
    ("MIAO VOWEL SIGN U"                                         . #x16F6A)
    ("MIAO VOWEL SIGN UA"                                        . #x16F6B)
    ("MIAO VOWEL SIGN UAN"                                       . #x16F6C)
    ("MIAO VOWEL SIGN UANG"                                      . #x16F6D)
    ("MIAO VOWEL SIGN UU"                                        . #x16F6E)
    ("MIAO VOWEL SIGN UEI"                                       . #x16F6F)
    ("MIAO VOWEL SIGN UNG"                                       . #x16F70)
    ("MIAO VOWEL SIGN Y"                                         . #x16F71)
    ("MIAO VOWEL SIGN YI"                                        . #x16F72)
    ("MIAO VOWEL SIGN AE"                                        . #x16F73)
    ("MIAO VOWEL SIGN AEE"                                       . #x16F74)
    ("MIAO VOWEL SIGN ERR"                                       . #x16F75)
    ("MIAO VOWEL SIGN ROUNDED ERR"                               . #x16F76)
    ("MIAO VOWEL SIGN ER"                                        . #x16F77)
    ("MIAO VOWEL SIGN ROUNDED ER"                                . #x16F78)
    ("MIAO VOWEL SIGN AI"                                        . #x16F79)
    ("MIAO VOWEL SIGN EI"                                        . #x16F7A)
    ("MIAO VOWEL SIGN AU"                                        . #x16F7B)
    ("MIAO VOWEL SIGN OU"                                        . #x16F7C)
    ("MIAO VOWEL SIGN N"                                         . #x16F7D)
    ("MIAO VOWEL SIGN NG"                                        . #x16F7E)
    ("MIAO TONE RIGHT"                                           . #x16F8F)
    ("MIAO TONE TOP RIGHT"                                       . #x16F90)
    ("MIAO TONE ABOVE"                                           . #x16F91)
    ("MIAO TONE BELOW"                                           . #x16F92)
    ("MIAO LETTER TONE-2"                                        . #x16F93)
    ("MIAO LETTER TONE-3"                                        . #x16F94)
    ("MIAO LETTER TONE-4"                                        . #x16F95)
    ("MIAO LETTER TONE-5"                                        . #x16F96)
    ("MIAO LETTER TONE-6"                                        . #x16F97)
    ("MIAO LETTER TONE-7"                                        . #x16F98)
    ("MIAO LETTER TONE-8"                                        . #x16F99)
    ("MIAO LETTER REFORMED TONE-1"                               . #x16F9A)
    ("MIAO LETTER REFORMED TONE-2"                               . #x16F9B)
    ("MIAO LETTER REFORMED TONE-4"                               . #x16F9C)
    ("MIAO LETTER REFORMED TONE-5"                               . #x16F9D)
    ("MIAO LETTER REFORMED TONE-6"                               . #x16F9E)
    ("MIAO LETTER REFORMED TONE-8"                               . #x16F9F)
    ("ARABIC MATHEMATICAL ALEF"                                  . #x1EE00)
    ("ARABIC MATHEMATICAL BEH"                                   . #x1EE01)
    ("ARABIC MATHEMATICAL JEEM"                                  . #x1EE02)
    ("ARABIC MATHEMATICAL DAL"                                   . #x1EE03)
    ("ARABIC MATHEMATICAL WAW"                                   . #x1EE05)
    ("ARABIC MATHEMATICAL ZAIN"                                  . #x1EE06)
    ("ARABIC MATHEMATICAL HAH"                                   . #x1EE07)
    ("ARABIC MATHEMATICAL TAH"                                   . #x1EE08)
    ("ARABIC MATHEMATICAL YEH"                                   . #x1EE09)
    ("ARABIC MATHEMATICAL KAF"                                   . #x1EE0A)
    ("ARABIC MATHEMATICAL LAM"                                   . #x1EE0B)
    ("ARABIC MATHEMATICAL MEEM"                                  . #x1EE0C)
    ("ARABIC MATHEMATICAL NOON"                                  . #x1EE0D)
    ("ARABIC MATHEMATICAL SEEN"                                  . #x1EE0E)
    ("ARABIC MATHEMATICAL AIN"                                   . #x1EE0F)
    ("ARABIC MATHEMATICAL FEH"                                   . #x1EE10)
    ("ARABIC MATHEMATICAL SAD"                                   . #x1EE11)
    ("ARABIC MATHEMATICAL QAF"                                   . #x1EE12)
    ("ARABIC MATHEMATICAL REH"                                   . #x1EE13)
    ("ARABIC MATHEMATICAL SHEEN"                                 . #x1EE14)
    ("ARABIC MATHEMATICAL TEH"                                   . #x1EE15)
    ("ARABIC MATHEMATICAL THEH"                                  . #x1EE16)
    ("ARABIC MATHEMATICAL KHAH"                                  . #x1EE17)
    ("ARABIC MATHEMATICAL THAL"                                  . #x1EE18)
    ("ARABIC MATHEMATICAL DAD"                                   . #x1EE19)
    ("ARABIC MATHEMATICAL ZAH"                                   . #x1EE1A)
    ("ARABIC MATHEMATICAL GHAIN"                                 . #x1EE1B)
    ("ARABIC MATHEMATICAL DOTLESS BEH"                           . #x1EE1C)
    ("ARABIC MATHEMATICAL DOTLESS NOON"                          . #x1EE1D)
    ("ARABIC MATHEMATICAL DOTLESS FEH"                           . #x1EE1E)
    ("ARABIC MATHEMATICAL DOTLESS QAF"                           . #x1EE1F)
    ("ARABIC MATHEMATICAL INITIAL BEH"                           . #x1EE21)
    ("ARABIC MATHEMATICAL INITIAL JEEM"                          . #x1EE22)
    ("ARABIC MATHEMATICAL INITIAL HEH"                           . #x1EE24)
    ("ARABIC MATHEMATICAL INITIAL HAH"                           . #x1EE27)
    ("ARABIC MATHEMATICAL INITIAL YEH"                           . #x1EE29)
    ("ARABIC MATHEMATICAL INITIAL KAF"                           . #x1EE2A)
    ("ARABIC MATHEMATICAL INITIAL LAM"                           . #x1EE2B)
    ("ARABIC MATHEMATICAL INITIAL MEEM"                          . #x1EE2C)
    ("ARABIC MATHEMATICAL INITIAL NOON"                          . #x1EE2D)
    ("ARABIC MATHEMATICAL INITIAL SEEN"                          . #x1EE2E)
    ("ARABIC MATHEMATICAL INITIAL AIN"                           . #x1EE2F)
    ("ARABIC MATHEMATICAL INITIAL FEH"                           . #x1EE30)
    ("ARABIC MATHEMATICAL INITIAL SAD"                           . #x1EE31)
    ("ARABIC MATHEMATICAL INITIAL QAF"                           . #x1EE32)
    ("ARABIC MATHEMATICAL INITIAL SHEEN"                         . #x1EE34)
    ("ARABIC MATHEMATICAL INITIAL TEH"                           . #x1EE35)
    ("ARABIC MATHEMATICAL INITIAL THEH"                          . #x1EE36)
    ("ARABIC MATHEMATICAL INITIAL KHAH"                          . #x1EE37)
    ("ARABIC MATHEMATICAL INITIAL DAD"                           . #x1EE39)
    ("ARABIC MATHEMATICAL INITIAL GHAIN"                         . #x1EE3B)
    ("ARABIC MATHEMATICAL TAILED JEEM"                           . #x1EE42)
    ("ARABIC MATHEMATICAL TAILED HAH"                            . #x1EE47)
    ("ARABIC MATHEMATICAL TAILED YEH"                            . #x1EE49)
    ("ARABIC MATHEMATICAL TAILED LAM"                            . #x1EE4B)
    ("ARABIC MATHEMATICAL TAILED NOON"                           . #x1EE4D)
    ("ARABIC MATHEMATICAL TAILED SEEN"                           . #x1EE4E)
    ("ARABIC MATHEMATICAL TAILED AIN"                            . #x1EE4F)
    ("ARABIC MATHEMATICAL TAILED SAD"                            . #x1EE51)
    ("ARABIC MATHEMATICAL TAILED QAF"                            . #x1EE52)
    ("ARABIC MATHEMATICAL TAILED SHEEN"                          . #x1EE54)
    ("ARABIC MATHEMATICAL TAILED KHAH"                           . #x1EE57)
    ("ARABIC MATHEMATICAL TAILED DAD"                            . #x1EE59)
    ("ARABIC MATHEMATICAL TAILED GHAIN"                          . #x1EE5B)
    ("ARABIC MATHEMATICAL TAILED DOTLESS NOON"                   . #x1EE5D)
    ("ARABIC MATHEMATICAL TAILED DOTLESS QAF"                    . #x1EE5F)
    ("ARABIC MATHEMATICAL STRETCHED BEH"                         . #x1EE61)
    ("ARABIC MATHEMATICAL STRETCHED JEEM"                        . #x1EE62)
    ("ARABIC MATHEMATICAL STRETCHED HEH"                         . #x1EE64)
    ("ARABIC MATHEMATICAL STRETCHED HAH"                         . #x1EE67)
    ("ARABIC MATHEMATICAL STRETCHED TAH"                         . #x1EE68)
    ("ARABIC MATHEMATICAL STRETCHED YEH"                         . #x1EE69)
    ("ARABIC MATHEMATICAL STRETCHED KAF"                         . #x1EE6A)
    ("ARABIC MATHEMATICAL STRETCHED MEEM"                        . #x1EE6C)
    ("ARABIC MATHEMATICAL STRETCHED NOON"                        . #x1EE6D)
    ("ARABIC MATHEMATICAL STRETCHED SEEN"                        . #x1EE6E)
    ("ARABIC MATHEMATICAL STRETCHED AIN"                         . #x1EE6F)
    ("ARABIC MATHEMATICAL STRETCHED FEH"                         . #x1EE70)
    ("ARABIC MATHEMATICAL STRETCHED SAD"                         . #x1EE71)
    ("ARABIC MATHEMATICAL STRETCHED QAF"                         . #x1EE72)
    ("ARABIC MATHEMATICAL STRETCHED SHEEN"                       . #x1EE74)
    ("ARABIC MATHEMATICAL STRETCHED TEH"                         . #x1EE75)
    ("ARABIC MATHEMATICAL STRETCHED THEH"                        . #x1EE76)
    ("ARABIC MATHEMATICAL STRETCHED KHAH"                        . #x1EE77)
    ("ARABIC MATHEMATICAL STRETCHED DAD"                         . #x1EE79)
    ("ARABIC MATHEMATICAL STRETCHED ZAH"                         . #x1EE7A)
    ("ARABIC MATHEMATICAL STRETCHED GHAIN"                       . #x1EE7B)
    ("ARABIC MATHEMATICAL STRETCHED DOTLESS BEH"                 . #x1EE7C)
    ("ARABIC MATHEMATICAL STRETCHED DOTLESS FEH"                 . #x1EE7E)
    ("ARABIC MATHEMATICAL LOOPED ALEF"                           . #x1EE80)
    ("ARABIC MATHEMATICAL LOOPED BEH"                            . #x1EE81)
    ("ARABIC MATHEMATICAL LOOPED JEEM"                           . #x1EE82)
    ("ARABIC MATHEMATICAL LOOPED DAL"                            . #x1EE83)
    ("ARABIC MATHEMATICAL LOOPED HEH"                            . #x1EE84)
    ("ARABIC MATHEMATICAL LOOPED WAW"                            . #x1EE85)
    ("ARABIC MATHEMATICAL LOOPED ZAIN"                           . #x1EE86)
    ("ARABIC MATHEMATICAL LOOPED HAH"                            . #x1EE87)
    ("ARABIC MATHEMATICAL LOOPED TAH"                            . #x1EE88)
    ("ARABIC MATHEMATICAL LOOPED YEH"                            . #x1EE89)
    ("ARABIC MATHEMATICAL LOOPED LAM"                            . #x1EE8B)
    ("ARABIC MATHEMATICAL LOOPED MEEM"                           . #x1EE8C)
    ("ARABIC MATHEMATICAL LOOPED NOON"                           . #x1EE8D)
    ("ARABIC MATHEMATICAL LOOPED SEEN"                           . #x1EE8E)
    ("ARABIC MATHEMATICAL LOOPED AIN"                            . #x1EE8F)
    ("ARABIC MATHEMATICAL LOOPED FEH"                            . #x1EE90)
    ("ARABIC MATHEMATICAL LOOPED SAD"                            . #x1EE91)
    ("ARABIC MATHEMATICAL LOOPED QAF"                            . #x1EE92)
    ("ARABIC MATHEMATICAL LOOPED REH"                            . #x1EE93)
    ("ARABIC MATHEMATICAL LOOPED SHEEN"                          . #x1EE94)
    ("ARABIC MATHEMATICAL LOOPED TEH"                            . #x1EE95)
    ("ARABIC MATHEMATICAL LOOPED THEH"                           . #x1EE96)
    ("ARABIC MATHEMATICAL LOOPED KHAH"                           . #x1EE97)
    ("ARABIC MATHEMATICAL LOOPED THAL"                           . #x1EE98)
    ("ARABIC MATHEMATICAL LOOPED DAD"                            . #x1EE99)
    ("ARABIC MATHEMATICAL LOOPED ZAH"                            . #x1EE9A)
    ("ARABIC MATHEMATICAL LOOPED GHAIN"                          . #x1EE9B)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK BEH"                     . #x1EEA1)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK JEEM"                    . #x1EEA2)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK DAL"                     . #x1EEA3)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK WAW"                     . #x1EEA5)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK ZAIN"                    . #x1EEA6)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK HAH"                     . #x1EEA7)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK TAH"                     . #x1EEA8)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK YEH"                     . #x1EEA9)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK LAM"                     . #x1EEAB)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK MEEM"                    . #x1EEAC)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK NOON"                    . #x1EEAD)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK SEEN"                    . #x1EEAE)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK AIN"                     . #x1EEAF)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK FEH"                     . #x1EEB0)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK SAD"                     . #x1EEB1)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK QAF"                     . #x1EEB2)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK REH"                     . #x1EEB3)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK SHEEN"                   . #x1EEB4)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK TEH"                     . #x1EEB5)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK THEH"                    . #x1EEB6)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK KHAH"                    . #x1EEB7)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK THAL"                    . #x1EEB8)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK DAD"                     . #x1EEB9)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK ZAH"                     . #x1EEBA)
    ("ARABIC MATHEMATICAL DOUBLE-STRUCK GHAIN"                   . #x1EEBB)
    ("ARABIC MATHEMATICAL OPERATOR MEEM WITH HAH WITH TATWEEL"   . #x1EEF0)
    ("ARABIC MATHEMATICAL OPERATOR HAH WITH DAL"                 . #x1EEF1)
    ("RAISED MC SIGN"                                            . #x1F16A)
    ("RAISED MD SIGN"                                            . #x1F16B)
    ("CIRCLED CROSS POMMEE"                                      . #x1F540)
    ("CROSS POMMEE WITH HALF-CIRCLE BELOW"                       . #x1F541)
    ("CROSS POMMEE"                                              . #x1F542)
    ("NOTCHED LEFT SEMICIRCLE WITH THREE DOTS"                   . #x1F543)
    ("GRINNING FACE"                                             . #x1F600)
    ("EXPRESSIONLESS FACE"                                       . #x1F611)
    ("CONFUSED FACE"                                             . #x1F615)
    ("KISSING FACE"                                              . #x1F617)
    ("KISSING FACE WITH SMILING EYES"                            . #x1F619)
    ("FACE WITH STUCK-OUT TONGUE"                                . #x1F61B)
    ("WORRIED FACE"                                              . #x1F61F)
    ("FROWNING FACE WITH OPEN MOUTH"                             . #x1F626)
    ("ANGUISHED FACE"                                            . #x1F627)
    ("GRIMACING FACE"                                            . #x1F62C)
    ("FACE WITH OPEN MOUTH"                                      . #x1F62E)
    ("HUSHED FACE"                                               . #x1F62F)
    ("SLEEPING FACE"                                             . #x1F634)
    ;; Unicode 6.1 to 6.2 delta
    ("TURKISH LIRA SIGN"                                         . #x20BA)
    ;; Unicode 6.2 to 6.3 delta
    ("ARABIC LETTER MARK"                                        . #x061C)
    ("LEFT-TO-RIGHT ISOLATE"                                     . #x2066)
    ("RIGHT-TO-LEFT ISOLATE"                                     . #x2067)
    ("FIRST STRONG ISOLATE"                                      . #x2068)
    ("POP DIRECTIONAL ISOLATE"                                   . #x2069)
    ;; Unicode 6.3 to 7.0 delta
    ("GREEK CAPITAL LETTER YOT"                                                    . #x037F)
    ("CYRILLIC CAPITAL LETTER EN WITH LEFT HOOK"                                   . #x0528)
    ("CYRILLIC SMALL LETTER EN WITH LEFT HOOK"                                     . #x0529)
    ("CYRILLIC CAPITAL LETTER DZZHE"                                               . #x052A)
    ("CYRILLIC SMALL LETTER DZZHE"                                                 . #x052B)
    ("CYRILLIC CAPITAL LETTER DCHE"                                                . #x052C)
    ("CYRILLIC SMALL LETTER DCHE"                                                  . #x052D)
    ("CYRILLIC CAPITAL LETTER EL WITH DESCENDER"                                   . #x052E)
    ("CYRILLIC SMALL LETTER EL WITH DESCENDER"                                     . #x052F)
    ("RIGHT-FACING ARMENIAN ETERNITY SIGN"                                         . #x058D)
    ("LEFT-FACING ARMENIAN ETERNITY SIGN"                                          . #x058E)
    ("ARABIC NUMBER MARK ABOVE"                                                    . #x0605)
    ("ARABIC LETTER BEH WITH HAMZA ABOVE"                                          . #x08A1)
    ("ARABIC LETTER LOW ALEF"                                                      . #x08AD)
    ("ARABIC LETTER DAL WITH THREE DOTS BELOW"                                     . #x08AE)
    ("ARABIC LETTER SAD WITH THREE DOTS BELOW"                                     . #x08AF)
    ("ARABIC LETTER GAF WITH INVERTED STROKE"                                      . #x08B0)
    ("ARABIC LETTER STRAIGHT WAW"                                                  . #x08B1)
    ("ARABIC LETTER ZAIN WITH INVERTED V ABOVE"                                    . #x08B2)
    ("ARABIC MARK SIDEWAYS NOON GHUNNA"                                            . #x08FF)
    ("DEVANAGARI LETTER MARWARI DDA"                                               . #x0978)
    ("BENGALI ANJI"                                                                . #x0980)
    ("TELUGU SIGN COMBINING CANDRABINDU ABOVE"                                     . #x0C00)
    ("TELUGU LETTER LLLA"                                                          . #x0C34)
    ("KANNADA SIGN CANDRABINDU"                                                    . #x0C81)
    ("MALAYALAM SIGN CANDRABINDU"                                                  . #x0D01)
    ("SINHALA LITH DIGIT ZERO"                                                     . #x0DE6)
    ("SINHALA LITH DIGIT ONE"                                                      . #x0DE7)
    ("SINHALA LITH DIGIT TWO"                                                      . #x0DE8)
    ("SINHALA LITH DIGIT THREE"                                                    . #x0DE9)
    ("SINHALA LITH DIGIT FOUR"                                                     . #x0DEA)
    ("SINHALA LITH DIGIT FIVE"                                                     . #x0DEB)
    ("SINHALA LITH DIGIT SIX"                                                      . #x0DEC)
    ("SINHALA LITH DIGIT SEVEN"                                                    . #x0DED)
    ("SINHALA LITH DIGIT EIGHT"                                                    . #x0DEE)
    ("SINHALA LITH DIGIT NINE"                                                     . #x0DEF)
    ("RUNIC LETTER K"                                                              . #x16F1)
    ("RUNIC LETTER SH"                                                             . #x16F2)
    ("RUNIC LETTER OO"                                                             . #x16F3)
    ("RUNIC LETTER FRANKS CASKET OS"                                               . #x16F4)
    ("RUNIC LETTER FRANKS CASKET IS"                                               . #x16F5)
    ("RUNIC LETTER FRANKS CASKET EH"                                               . #x16F6)
    ("RUNIC LETTER FRANKS CASKET AC"                                               . #x16F7)
    ("RUNIC LETTER FRANKS CASKET AESC"                                             . #x16F8)
    ("LIMBU LETTER GYAN"                                                           . #x191D)
    ("LIMBU LETTER TRA"                                                            . #x191E)
    ("COMBINING DOUBLED CIRCUMFLEX ACCENT"                                         . #x1AB0)
    ("COMBINING DIAERESIS-RING"                                                    . #x1AB1)
    ("COMBINING INFINITY"                                                          . #x1AB2)
    ("COMBINING DOWNWARDS ARROW"                                                   . #x1AB3)
    ("COMBINING TRIPLE DOT"                                                        . #x1AB4)
    ("COMBINING X-X BELOW"                                                         . #x1AB5)
    ("COMBINING WIGGLY LINE BELOW"                                                 . #x1AB6)
    ("COMBINING OPEN MARK BELOW"                                                   . #x1AB7)
    ("COMBINING DOUBLE OPEN MARK BELOW"                                            . #x1AB8)
    ("COMBINING LIGHT CENTRALIZATION STROKE BELOW"                                 . #x1AB9)
    ("COMBINING STRONG CENTRALIZATION STROKE BELOW"                                . #x1ABA)
    ("COMBINING PARENTHESES ABOVE"                                                 . #x1ABB)
    ("COMBINING DOUBLE PARENTHESES ABOVE"                                          . #x1ABC)
    ("COMBINING PARENTHESES BELOW"                                                 . #x1ABD)
    ("COMBINING PARENTHESES OVERLAY"                                               . #x1ABE)
    ("VEDIC TONE RING ABOVE"                                                       . #x1CF8)
    ("VEDIC TONE DOUBLE RING ABOVE"                                                . #x1CF9)
    ("COMBINING LATIN SMALL LETTER ALPHA"                                          . #x1DE7)
    ("COMBINING LATIN SMALL LETTER B"                                              . #x1DE8)
    ("COMBINING LATIN SMALL LETTER BETA"                                           . #x1DE9)
    ("COMBINING LATIN SMALL LETTER SCHWA"                                          . #x1DEA)
    ("COMBINING LATIN SMALL LETTER F"                                              . #x1DEB)
    ("COMBINING LATIN SMALL LETTER L WITH DOUBLE MIDDLE TILDE"                     . #x1DEC)
    ("COMBINING LATIN SMALL LETTER O WITH LIGHT CENTRALIZATION STROKE"             . #x1DED)
    ("COMBINING LATIN SMALL LETTER P"                                              . #x1DEE)
    ("COMBINING LATIN SMALL LETTER ESH"                                            . #x1DEF)
    ("COMBINING LATIN SMALL LETTER U WITH LIGHT CENTRALIZATION STROKE"             . #x1DF0)
    ("COMBINING LATIN SMALL LETTER W"                                              . #x1DF1)
    ("COMBINING LATIN SMALL LETTER A WITH DIAERESIS"                               . #x1DF2)
    ("COMBINING LATIN SMALL LETTER O WITH DIAERESIS"                               . #x1DF3)
    ("COMBINING LATIN SMALL LETTER U WITH DIAERESIS"                               . #x1DF4)
    ("COMBINING UP TACK ABOVE"                                                     . #x1DF5)
    ("NORDIC MARK SIGN"                                                            . #x20BB)
    ("MANAT SIGN"                                                                  . #x20BC)
    ("RUBLE SIGN"                                                                  . #x20BD)
    ("BLACK MEDIUM LEFT-POINTING TRIANGLE"                                         . #x23F4)
    ("BLACK MEDIUM RIGHT-POINTING TRIANGLE"                                        . #x23F5)
    ("BLACK MEDIUM UP-POINTING TRIANGLE"                                           . #x23F6)
    ("BLACK MEDIUM DOWN-POINTING TRIANGLE"                                         . #x23F7)
    ("DOUBLE VERTICAL BAR"                                                         . #x23F8)
    ("BLACK SQUARE FOR STOP"                                                       . #x23F9)
    ("BLACK CIRCLE FOR RECORD"                                                     . #x23FA)
    ("BLACK SAFETY SCISSORS"                                                       . #x2700)
    ("DOWNWARDS TRIANGLE-HEADED ZIGZAG ARROW"                                      . #x2B4D)
    ("SHORT SLANTED NORTH ARROW"                                                   . #x2B4E)
    ("SHORT BACKSLANTED SOUTH ARROW"                                               . #x2B4F)
    ("SLANTED NORTH ARROW WITH HOOKED HEAD"                                        . #x2B5A)
    ("BACKSLANTED SOUTH ARROW WITH HOOKED TAIL"                                    . #x2B5B)
    ("SLANTED NORTH ARROW WITH HORIZONTAL TAIL"                                    . #x2B5C)
    ("BACKSLANTED SOUTH ARROW WITH HORIZONTAL TAIL"                                . #x2B5D)
    ("BENT ARROW POINTING DOWNWARDS THEN NORTH EAST"                               . #x2B5E)
    ("SHORT BENT ARROW POINTING DOWNWARDS THEN NORTH EAST"                         . #x2B5F)
    ("LEFTWARDS TRIANGLE-HEADED ARROW"                                             . #x2B60)
    ("UPWARDS TRIANGLE-HEADED ARROW"                                               . #x2B61)
    ("RIGHTWARDS TRIANGLE-HEADED ARROW"                                            . #x2B62)
    ("DOWNWARDS TRIANGLE-HEADED ARROW"                                             . #x2B63)
    ("LEFT RIGHT TRIANGLE-HEADED ARROW"                                            . #x2B64)
    ("UP DOWN TRIANGLE-HEADED ARROW"                                               . #x2B65)
    ("NORTH WEST TRIANGLE-HEADED ARROW"                                            . #x2B66)
    ("NORTH EAST TRIANGLE-HEADED ARROW"                                            . #x2B67)
    ("SOUTH EAST TRIANGLE-HEADED ARROW"                                            . #x2B68)
    ("SOUTH WEST TRIANGLE-HEADED ARROW"                                            . #x2B69)
    ("LEFTWARDS TRIANGLE-HEADED DASHED ARROW"                                      . #x2B6A)
    ("UPWARDS TRIANGLE-HEADED DASHED ARROW"                                        . #x2B6B)
    ("RIGHTWARDS TRIANGLE-HEADED DASHED ARROW"                                     . #x2B6C)
    ("DOWNWARDS TRIANGLE-HEADED DASHED ARROW"                                      . #x2B6D)
    ("CLOCKWISE TRIANGLE-HEADED OPEN CIRCLE ARROW"                                 . #x2B6E)
    ("ANTICLOCKWISE TRIANGLE-HEADED OPEN CIRCLE ARROW"                             . #x2B6F)
    ("LEFTWARDS TRIANGLE-HEADED ARROW TO BAR"                                      . #x2B70)
    ("UPWARDS TRIANGLE-HEADED ARROW TO BAR"                                        . #x2B71)
    ("RIGHTWARDS TRIANGLE-HEADED ARROW TO BAR"                                     . #x2B72)
    ("DOWNWARDS TRIANGLE-HEADED ARROW TO BAR"                                      . #x2B73)
    ("NORTH WEST TRIANGLE-HEADED ARROW TO BAR"                                     . #x2B76)
    ("NORTH EAST TRIANGLE-HEADED ARROW TO BAR"                                     . #x2B77)
    ("SOUTH EAST TRIANGLE-HEADED ARROW TO BAR"                                     . #x2B78)
    ("SOUTH WEST TRIANGLE-HEADED ARROW TO BAR"                                     . #x2B79)
    ("LEFTWARDS TRIANGLE-HEADED ARROW WITH DOUBLE HORIZONTAL STROKE"               . #x2B7A)
    ("UPWARDS TRIANGLE-HEADED ARROW WITH DOUBLE HORIZONTAL STROKE"                 . #x2B7B)
    ("RIGHTWARDS TRIANGLE-HEADED ARROW WITH DOUBLE HORIZONTAL STROKE"              . #x2B7C)
    ("DOWNWARDS TRIANGLE-HEADED ARROW WITH DOUBLE HORIZONTAL STROKE"               . #x2B7D)
    ("HORIZONTAL TAB KEY"                                                          . #x2B7E)
    ("VERTICAL TAB KEY"                                                            . #x2B7F)
    ("LEFTWARDS TRIANGLE-HEADED ARROW OVER RIGHTWARDS TRIANGLE-HEADED ARROW"       . #x2B80)
    ("UPWARDS TRIANGLE-HEADED ARROW LEFTWARDS OF DOWNWARDS TRIANGLE-HEADED ARROW"  . #x2B81)
    ("RIGHTWARDS TRIANGLE-HEADED ARROW OVER LEFTWARDS TRIANGLE-HEADED ARROW"       . #x2B82)
    ("DOWNWARDS TRIANGLE-HEADED ARROW LEFTWARDS OF UPWARDS TRIANGLE-HEADED ARROW"  . #x2B83)
    ("LEFTWARDS TRIANGLE-HEADED PAIRED ARROWS"                                     . #x2B84)
    ("UPWARDS TRIANGLE-HEADED PAIRED ARROWS"                                       . #x2B85)
    ("RIGHTWARDS TRIANGLE-HEADED PAIRED ARROWS"                                    . #x2B86)
    ("DOWNWARDS TRIANGLE-HEADED PAIRED ARROWS"                                     . #x2B87)
    ("LEFTWARDS BLACK CIRCLED WHITE ARROW"                                         . #x2B88)
    ("UPWARDS BLACK CIRCLED WHITE ARROW"                                           . #x2B89)
    ("RIGHTWARDS BLACK CIRCLED WHITE ARROW"                                        . #x2B8A)
    ("DOWNWARDS BLACK CIRCLED WHITE ARROW"                                         . #x2B8B)
    ("ANTICLOCKWISE TRIANGLE-HEADED RIGHT U-SHAPED ARROW"                          . #x2B8C)
    ("ANTICLOCKWISE TRIANGLE-HEADED BOTTOM U-SHAPED ARROW"                         . #x2B8D)
    ("ANTICLOCKWISE TRIANGLE-HEADED LEFT U-SHAPED ARROW"                           . #x2B8E)
    ("ANTICLOCKWISE TRIANGLE-HEADED TOP U-SHAPED ARROW"                            . #x2B8F)
    ("RETURN LEFT"                                                                 . #x2B90)
    ("RETURN RIGHT"                                                                . #x2B91)
    ("NEWLINE LEFT"                                                                . #x2B92)
    ("NEWLINE RIGHT"                                                               . #x2B93)
    ("FOUR CORNER ARROWS CIRCLING ANTICLOCKWISE"                                   . #x2B94)
    ("RIGHTWARDS BLACK ARROW"                                                      . #x2B95)
    ("THREE-D TOP-LIGHTED LEFTWARDS EQUILATERAL ARROWHEAD"                         . #x2B98)
    ("THREE-D RIGHT-LIGHTED UPWARDS EQUILATERAL ARROWHEAD"                         . #x2B99)
    ("THREE-D TOP-LIGHTED RIGHTWARDS EQUILATERAL ARROWHEAD"                        . #x2B9A)
    ("THREE-D LEFT-LIGHTED DOWNWARDS EQUILATERAL ARROWHEAD"                        . #x2B9B)
    ("BLACK LEFTWARDS EQUILATERAL ARROWHEAD"                                       . #x2B9C)
    ("BLACK UPWARDS EQUILATERAL ARROWHEAD"                                         . #x2B9D)
    ("BLACK RIGHTWARDS EQUILATERAL ARROWHEAD"                                      . #x2B9E)
    ("BLACK DOWNWARDS EQUILATERAL ARROWHEAD"                                       . #x2B9F)
    ("DOWNWARDS TRIANGLE-HEADED ARROW WITH LONG TIP LEFTWARDS"                     . #x2BA0)
    ("DOWNWARDS TRIANGLE-HEADED ARROW WITH LONG TIP RIGHTWARDS"                    . #x2BA1)
    ("UPWARDS TRIANGLE-HEADED ARROW WITH LONG TIP LEFTWARDS"                       . #x2BA2)
    ("UPWARDS TRIANGLE-HEADED ARROW WITH LONG TIP RIGHTWARDS"                      . #x2BA3)
    ("LEFTWARDS TRIANGLE-HEADED ARROW WITH LONG TIP UPWARDS"                       . #x2BA4)
    ("RIGHTWARDS TRIANGLE-HEADED ARROW WITH LONG TIP UPWARDS"                      . #x2BA5)
    ("LEFTWARDS TRIANGLE-HEADED ARROW WITH LONG TIP DOWNWARDS"                     . #x2BA6)
    ("RIGHTWARDS TRIANGLE-HEADED ARROW WITH LONG TIP DOWNWARDS"                    . #x2BA7)
    ("BLACK CURVED DOWNWARDS AND LEFTWARDS ARROW"                                  . #x2BA8)
    ("BLACK CURVED DOWNWARDS AND RIGHTWARDS ARROW"                                 . #x2BA9)
    ("BLACK CURVED UPWARDS AND LEFTWARDS ARROW"                                    . #x2BAA)
    ("BLACK CURVED UPWARDS AND RIGHTWARDS ARROW"                                   . #x2BAB)
    ("BLACK CURVED LEFTWARDS AND UPWARDS ARROW"                                    . #x2BAC)
    ("BLACK CURVED RIGHTWARDS AND UPWARDS ARROW"                                   . #x2BAD)
    ("BLACK CURVED LEFTWARDS AND DOWNWARDS ARROW"                                  . #x2BAE)
    ("BLACK CURVED RIGHTWARDS AND DOWNWARDS ARROW"                                 . #x2BAF)
    ("RIBBON ARROW DOWN LEFT"                                                      . #x2BB0)
    ("RIBBON ARROW DOWN RIGHT"                                                     . #x2BB1)
    ("RIBBON ARROW UP LEFT"                                                        . #x2BB2)
    ("RIBBON ARROW UP RIGHT"                                                       . #x2BB3)
    ("RIBBON ARROW LEFT UP"                                                        . #x2BB4)
    ("RIBBON ARROW RIGHT UP"                                                       . #x2BB5)
    ("RIBBON ARROW LEFT DOWN"                                                      . #x2BB6)
    ("RIBBON ARROW RIGHT DOWN"                                                     . #x2BB7)
    ("UPWARDS WHITE ARROW FROM BAR WITH HORIZONTAL BAR"                            . #x2BB8)
    ("UP ARROWHEAD IN A RECTANGLE BOX"                                             . #x2BB9)
    ("BALLOT BOX WITH LIGHT X"                                                     . #x2BBD)
    ("CIRCLED X"                                                                   . #x2BBE)
    ("CIRCLED BOLD X"                                                              . #x2BBF)
    ("BLACK SQUARE CENTRED"                                                        . #x2BC0)
    ("BLACK DIAMOND CENTRED"                                                       . #x2BC1)
    ("TURNED BLACK PENTAGON"                                                       . #x2BC2)
    ("HORIZONTAL BLACK OCTAGON"                                                    . #x2BC3)
    ("BLACK OCTAGON"                                                               . #x2BC4)
    ("BLACK MEDIUM UP-POINTING TRIANGLE CENTRED"                                   . #x2BC5)
    ("BLACK MEDIUM DOWN-POINTING TRIANGLE CENTRED"                                 . #x2BC6)
    ("BLACK MEDIUM LEFT-POINTING TRIANGLE CENTRED"                                 . #x2BC7)
    ("BLACK MEDIUM RIGHT-POINTING TRIANGLE CENTRED"                                . #x2BC8)
    ("TOP HALF BLACK CIRCLE"                                                       . #x2BCA)
    ("BOTTOM HALF BLACK CIRCLE"                                                    . #x2BCB)
    ("LIGHT FOUR POINTED BLACK CUSP"                                               . #x2BCC)
    ("ROTATED LIGHT FOUR POINTED BLACK CUSP"                                       . #x2BCD)
    ("WHITE FOUR POINTED CUSP"                                                     . #x2BCE)
    ("ROTATED WHITE FOUR POINTED CUSP"                                             . #x2BCF)
    ("SQUARE POSITION INDICATOR"                                                   . #x2BD0)
    ("UNCERTAINTY SIGN"                                                            . #x2BD1)
    ("STENOGRAPHIC FULL STOP"                                                      . #x2E3C)
    ("VERTICAL SIX DOTS"                                                           . #x2E3D)
    ("WIGGLY VERTICAL LINE"                                                        . #x2E3E)
    ("CAPITULUM"                                                                   . #x2E3F)
    ("DOUBLE HYPHEN"                                                               . #x2E40)
    ("REVERSED COMMA"                                                              . #x2E41)
    ("DOUBLE LOW-REVERSED-9 QUOTATION MARK"                                        . #x2E42)
    ("CYRILLIC CAPITAL LETTER DOUBLE O"                                            . #xA698)
    ("CYRILLIC SMALL LETTER DOUBLE O"                                              . #xA699)
    ("CYRILLIC CAPITAL LETTER CROSSED O"                                           . #xA69A)
    ("CYRILLIC SMALL LETTER CROSSED O"                                             . #xA69B)
    ("MODIFIER LETTER CYRILLIC HARD SIGN"                                          . #xA69C)
    ("MODIFIER LETTER CYRILLIC SOFT SIGN"                                          . #xA69D)
    ("LATIN SMALL LETTER C WITH PALATAL HOOK"                                      . #xA794)
    ("LATIN SMALL LETTER H WITH PALATAL HOOK"                                      . #xA795)
    ("LATIN CAPITAL LETTER B WITH FLOURISH"                                        . #xA796)
    ("LATIN SMALL LETTER B WITH FLOURISH"                                          . #xA797)
    ("LATIN CAPITAL LETTER F WITH STROKE"                                          . #xA798)
    ("LATIN SMALL LETTER F WITH STROKE"                                            . #xA799)
    ("LATIN CAPITAL LETTER VOLAPUK AE"                                             . #xA79A)
    ("LATIN SMALL LETTER VOLAPUK AE"                                               . #xA79B)
    ("LATIN CAPITAL LETTER VOLAPUK OE"                                             . #xA79C)
    ("LATIN SMALL LETTER VOLAPUK OE"                                               . #xA79D)
    ("LATIN CAPITAL LETTER VOLAPUK UE"                                             . #xA79E)
    ("LATIN SMALL LETTER VOLAPUK UE"                                               . #xA79F)
    ("LATIN CAPITAL LETTER REVERSED OPEN E"                                        . #xA7AB)
    ("LATIN CAPITAL LETTER SCRIPT G"                                               . #xA7AC)
    ("LATIN CAPITAL LETTER L WITH BELT"                                            . #xA7AD)
    ("LATIN CAPITAL LETTER TURNED K"                                               . #xA7B0)
    ("LATIN CAPITAL LETTER TURNED T"                                               . #xA7B1)
    ("LATIN EPIGRAPHIC LETTER SIDEWAYS I"                                          . #xA7F7)
    ("MYANMAR LETTER SHAN GHA"                                                     . #xA9E0)
    ("MYANMAR LETTER SHAN CHA"                                                     . #xA9E1)
    ("MYANMAR LETTER SHAN JHA"                                                     . #xA9E2)
    ("MYANMAR LETTER SHAN NNA"                                                     . #xA9E3)
    ("MYANMAR LETTER SHAN BHA"                                                     . #xA9E4)
    ("MYANMAR SIGN SHAN SAW"                                                       . #xA9E5)
    ("MYANMAR MODIFIER LETTER SHAN REDUPLICATION"                                  . #xA9E6)
    ("MYANMAR LETTER TAI LAING NYA"                                                . #xA9E7)
    ("MYANMAR LETTER TAI LAING FA"                                                 . #xA9E8)
    ("MYANMAR LETTER TAI LAING GA"                                                 . #xA9E9)
    ("MYANMAR LETTER TAI LAING GHA"                                                . #xA9EA)
    ("MYANMAR LETTER TAI LAING JA"                                                 . #xA9EB)
    ("MYANMAR LETTER TAI LAING JHA"                                                . #xA9EC)
    ("MYANMAR LETTER TAI LAING DDA"                                                . #xA9ED)
    ("MYANMAR LETTER TAI LAING DDHA"                                               . #xA9EE)
    ("MYANMAR LETTER TAI LAING NNA"                                                . #xA9EF)
    ("MYANMAR TAI LAING DIGIT ZERO"                                                . #xA9F0)
    ("MYANMAR TAI LAING DIGIT ONE"                                                 . #xA9F1)
    ("MYANMAR TAI LAING DIGIT TWO"                                                 . #xA9F2)
    ("MYANMAR TAI LAING DIGIT THREE"                                               . #xA9F3)
    ("MYANMAR TAI LAING DIGIT FOUR"                                                . #xA9F4)
    ("MYANMAR TAI LAING DIGIT FIVE"                                                . #xA9F5)
    ("MYANMAR TAI LAING DIGIT SIX"                                                 . #xA9F6)
    ("MYANMAR TAI LAING DIGIT SEVEN"                                               . #xA9F7)
    ("MYANMAR TAI LAING DIGIT EIGHT"                                               . #xA9F8)
    ("MYANMAR TAI LAING DIGIT NINE"                                                . #xA9F9)
    ("MYANMAR LETTER TAI LAING LLA"                                                . #xA9FA)
    ("MYANMAR LETTER TAI LAING DA"                                                 . #xA9FB)
    ("MYANMAR LETTER TAI LAING DHA"                                                . #xA9FC)
    ("MYANMAR LETTER TAI LAING BA"                                                 . #xA9FD)
    ("MYANMAR LETTER TAI LAING BHA"                                                . #xA9FE)
    ("MYANMAR SIGN TAI LAING TONE-2"                                               . #xAA7C)
    ("MYANMAR SIGN TAI LAING TONE-5"                                               . #xAA7D)
    ("MYANMAR LETTER SHWE PALAUNG CHA"                                             . #xAA7E)
    ("MYANMAR LETTER SHWE PALAUNG SHA"                                             . #xAA7F)
    ("LATIN SMALL LETTER BARRED ALPHA"                                             . #xAB30)
    ("LATIN SMALL LETTER A REVERSED-SCHWA"                                         . #xAB31)
    ("LATIN SMALL LETTER BLACKLETTER E"                                            . #xAB32)
    ("LATIN SMALL LETTER BARRED E"                                                 . #xAB33)
    ("LATIN SMALL LETTER E WITH FLOURISH"                                          . #xAB34)
    ("LATIN SMALL LETTER LENIS F"                                                  . #xAB35)
    ("LATIN SMALL LETTER SCRIPT G WITH CROSSED-TAIL"                               . #xAB36)
    ("LATIN SMALL LETTER L WITH INVERTED LAZY S"                                   . #xAB37)
    ("LATIN SMALL LETTER L WITH DOUBLE MIDDLE TILDE"                               . #xAB38)
    ("LATIN SMALL LETTER L WITH MIDDLE RING"                                       . #xAB39)
    ("LATIN SMALL LETTER M WITH CROSSED-TAIL"                                      . #xAB3A)
    ("LATIN SMALL LETTER N WITH CROSSED-TAIL"                                      . #xAB3B)
    ("LATIN SMALL LETTER ENG WITH CROSSED-TAIL"                                    . #xAB3C)
    ("LATIN SMALL LETTER BLACKLETTER O"                                            . #xAB3D)
    ("LATIN SMALL LETTER BLACKLETTER O WITH STROKE"                                . #xAB3E)
    ("LATIN SMALL LETTER OPEN O WITH STROKE"                                       . #xAB3F)
    ("LATIN SMALL LETTER INVERTED OE"                                              . #xAB40)
    ("LATIN SMALL LETTER TURNED OE WITH STROKE"                                    . #xAB41)
    ("LATIN SMALL LETTER TURNED OE WITH HORIZONTAL STROKE"                         . #xAB42)
    ("LATIN SMALL LETTER TURNED O OPEN-O"                                          . #xAB43)
    ("LATIN SMALL LETTER TURNED O OPEN-O WITH STROKE"                              . #xAB44)
    ("LATIN SMALL LETTER STIRRUP R"                                                . #xAB45)
    ("LATIN LETTER SMALL CAPITAL R WITH RIGHT LEG"                                 . #xAB46)
    ("LATIN SMALL LETTER R WITHOUT HANDLE"                                         . #xAB47)
    ("LATIN SMALL LETTER DOUBLE R"                                                 . #xAB48)
    ("LATIN SMALL LETTER R WITH CROSSED-TAIL"                                      . #xAB49)
    ("LATIN SMALL LETTER DOUBLE R WITH CROSSED-TAIL"                               . #xAB4A)
    ("LATIN SMALL LETTER SCRIPT R"                                                 . #xAB4B)
    ("LATIN SMALL LETTER SCRIPT R WITH RING"                                       . #xAB4C)
    ("LATIN SMALL LETTER BASELINE ESH"                                             . #xAB4D)
    ("LATIN SMALL LETTER U WITH SHORT RIGHT LEG"                                   . #xAB4E)
    ("LATIN SMALL LETTER U BAR WITH SHORT RIGHT LEG"                               . #xAB4F)
    ("LATIN SMALL LETTER UI"                                                       . #xAB50)
    ("LATIN SMALL LETTER TURNED UI"                                                . #xAB51)
    ("LATIN SMALL LETTER U WITH LEFT HOOK"                                         . #xAB52)
    ("LATIN SMALL LETTER CHI"                                                      . #xAB53)
    ("LATIN SMALL LETTER CHI WITH LOW RIGHT RING"                                  . #xAB54)
    ("LATIN SMALL LETTER CHI WITH LOW LEFT SERIF"                                  . #xAB55)
    ("LATIN SMALL LETTER X WITH LOW RIGHT RING"                                    . #xAB56)
    ("LATIN SMALL LETTER X WITH LONG LEFT LEG"                                     . #xAB57)
    ("LATIN SMALL LETTER X WITH LONG LEFT LEG AND LOW RIGHT RING"                  . #xAB58)
    ("LATIN SMALL LETTER X WITH LONG LEFT LEG WITH SERIF"                          . #xAB59)
    ("LATIN SMALL LETTER Y WITH SHORT RIGHT LEG"                                   . #xAB5A)
    ("MODIFIER BREVE WITH INVERTED BREVE"                                          . #xAB5B)
    ("MODIFIER LETTER SMALL HENG"                                                  . #xAB5C)
    ("MODIFIER LETTER SMALL L WITH INVERTED LAZY S"                                . #xAB5D)
    ("MODIFIER LETTER SMALL L WITH MIDDLE TILDE"                                   . #xAB5E)
    ("MODIFIER LETTER SMALL U WITH LEFT HOOK"                                      . #xAB5F)
    ("LATIN SMALL LETTER INVERTED ALPHA"                                           . #xAB64)
    ("GREEK LETTER SMALL CAPITAL OMEGA"                                            . #xAB65)
    ("COMBINING LIGATURE LEFT HALF BELOW"                                          . #xFE27)
    ("COMBINING LIGATURE RIGHT HALF BELOW"                                         . #xFE28)
    ("COMBINING TILDE LEFT HALF BELOW"                                             . #xFE29)
    ("COMBINING TILDE RIGHT HALF BELOW"                                            . #xFE2A)
    ("COMBINING MACRON LEFT HALF BELOW"                                            . #xFE2B)
    ("COMBINING MACRON RIGHT HALF BELOW"                                           . #xFE2C)
    ("COMBINING CONJOINING MACRON BELOW"                                           . #xFE2D)
    ("GREEK ONE QUARTER SIGN"                                                      . #x1018B)
    ("GREEK SINUSOID SIGN"                                                         . #x1018C)
    ("GREEK SYMBOL TAU RHO"                                                        . #x101A0)
    ("COPTIC EPACT THOUSANDS MARK"                                                 . #x102E0)
    ("COPTIC EPACT DIGIT ONE"                                                      . #x102E1)
    ("COPTIC EPACT DIGIT TWO"                                                      . #x102E2)
    ("COPTIC EPACT DIGIT THREE"                                                    . #x102E3)
    ("COPTIC EPACT DIGIT FOUR"                                                     . #x102E4)
    ("COPTIC EPACT DIGIT FIVE"                                                     . #x102E5)
    ("COPTIC EPACT DIGIT SIX"                                                      . #x102E6)
    ("COPTIC EPACT DIGIT SEVEN"                                                    . #x102E7)
    ("COPTIC EPACT DIGIT EIGHT"                                                    . #x102E8)
    ("COPTIC EPACT DIGIT NINE"                                                     . #x102E9)
    ("COPTIC EPACT NUMBER TEN"                                                     . #x102EA)
    ("COPTIC EPACT NUMBER TWENTY"                                                  . #x102EB)
    ("COPTIC EPACT NUMBER THIRTY"                                                  . #x102EC)
    ("COPTIC EPACT NUMBER FORTY"                                                   . #x102ED)
    ("COPTIC EPACT NUMBER FIFTY"                                                   . #x102EE)
    ("COPTIC EPACT NUMBER SIXTY"                                                   . #x102EF)
    ("COPTIC EPACT NUMBER SEVENTY"                                                 . #x102F0)
    ("COPTIC EPACT NUMBER EIGHTY"                                                  . #x102F1)
    ("COPTIC EPACT NUMBER NINETY"                                                  . #x102F2)
    ("COPTIC EPACT NUMBER ONE HUNDRED"                                             . #x102F3)
    ("COPTIC EPACT NUMBER TWO HUNDRED"                                             . #x102F4)
    ("COPTIC EPACT NUMBER THREE HUNDRED"                                           . #x102F5)
    ("COPTIC EPACT NUMBER FOUR HUNDRED"                                            . #x102F6)
    ("COPTIC EPACT NUMBER FIVE HUNDRED"                                            . #x102F7)
    ("COPTIC EPACT NUMBER SIX HUNDRED"                                             . #x102F8)
    ("COPTIC EPACT NUMBER SEVEN HUNDRED"                                           . #x102F9)
    ("COPTIC EPACT NUMBER EIGHT HUNDRED"                                           . #x102FA)
    ("COPTIC EPACT NUMBER NINE HUNDRED"                                            . #x102FB)
    ("OLD ITALIC LETTER ESS"                                                       . #x1031F)
    ("OLD PERMIC LETTER AN"                                                        . #x10350)
    ("OLD PERMIC LETTER BUR"                                                       . #x10351)
    ("OLD PERMIC LETTER GAI"                                                       . #x10352)
    ("OLD PERMIC LETTER DOI"                                                       . #x10353)
    ("OLD PERMIC LETTER E"                                                         . #x10354)
    ("OLD PERMIC LETTER ZHOI"                                                      . #x10355)
    ("OLD PERMIC LETTER DZHOI"                                                     . #x10356)
    ("OLD PERMIC LETTER ZATA"                                                      . #x10357)
    ("OLD PERMIC LETTER DZITA"                                                     . #x10358)
    ("OLD PERMIC LETTER I"                                                         . #x10359)
    ("OLD PERMIC LETTER KOKE"                                                      . #x1035A)
    ("OLD PERMIC LETTER LEI"                                                       . #x1035B)
    ("OLD PERMIC LETTER MENOE"                                                     . #x1035C)
    ("OLD PERMIC LETTER NENOE"                                                     . #x1035D)
    ("OLD PERMIC LETTER VOOI"                                                      . #x1035E)
    ("OLD PERMIC LETTER PEEI"                                                      . #x1035F)
    ("OLD PERMIC LETTER REI"                                                       . #x10360)
    ("OLD PERMIC LETTER SII"                                                       . #x10361)
    ("OLD PERMIC LETTER TAI"                                                       . #x10362)
    ("OLD PERMIC LETTER U"                                                         . #x10363)
    ("OLD PERMIC LETTER CHERY"                                                     . #x10364)
    ("OLD PERMIC LETTER SHOOI"                                                     . #x10365)
    ("OLD PERMIC LETTER SHCHOOI"                                                   . #x10366)
    ("OLD PERMIC LETTER YRY"                                                       . #x10367)
    ("OLD PERMIC LETTER YERU"                                                      . #x10368)
    ("OLD PERMIC LETTER O"                                                         . #x10369)
    ("OLD PERMIC LETTER OO"                                                        . #x1036A)
    ("OLD PERMIC LETTER EF"                                                        . #x1036B)
    ("OLD PERMIC LETTER HA"                                                        . #x1036C)
    ("OLD PERMIC LETTER TSIU"                                                      . #x1036D)
    ("OLD PERMIC LETTER VER"                                                       . #x1036E)
    ("OLD PERMIC LETTER YER"                                                       . #x1036F)
    ("OLD PERMIC LETTER YERI"                                                      . #x10370)
    ("OLD PERMIC LETTER YAT"                                                       . #x10371)
    ("OLD PERMIC LETTER IE"                                                        . #x10372)
    ("OLD PERMIC LETTER YU"                                                        . #x10373)
    ("OLD PERMIC LETTER YA"                                                        . #x10374)
    ("OLD PERMIC LETTER IA"                                                        . #x10375)
    ("COMBINING OLD PERMIC LETTER AN"                                              . #x10376)
    ("COMBINING OLD PERMIC LETTER DOI"                                             . #x10377)
    ("COMBINING OLD PERMIC LETTER ZATA"                                            . #x10378)
    ("COMBINING OLD PERMIC LETTER NENOE"                                           . #x10379)
    ("COMBINING OLD PERMIC LETTER SII"                                             . #x1037A)
    ("ELBASAN LETTER A"                                                            . #x10500)
    ("ELBASAN LETTER BE"                                                           . #x10501)
    ("ELBASAN LETTER CE"                                                           . #x10502)
    ("ELBASAN LETTER CHE"                                                          . #x10503)
    ("ELBASAN LETTER DE"                                                           . #x10504)
    ("ELBASAN LETTER NDE"                                                          . #x10505)
    ("ELBASAN LETTER DHE"                                                          . #x10506)
    ("ELBASAN LETTER EI"                                                           . #x10507)
    ("ELBASAN LETTER E"                                                            . #x10508)
    ("ELBASAN LETTER FE"                                                           . #x10509)
    ("ELBASAN LETTER GE"                                                           . #x1050A)
    ("ELBASAN LETTER GJE"                                                          . #x1050B)
    ("ELBASAN LETTER HE"                                                           . #x1050C)
    ("ELBASAN LETTER I"                                                            . #x1050D)
    ("ELBASAN LETTER JE"                                                           . #x1050E)
    ("ELBASAN LETTER KE"                                                           . #x1050F)
    ("ELBASAN LETTER LE"                                                           . #x10510)
    ("ELBASAN LETTER LLE"                                                          . #x10511)
    ("ELBASAN LETTER ME"                                                           . #x10512)
    ("ELBASAN LETTER NE"                                                           . #x10513)
    ("ELBASAN LETTER NA"                                                           . #x10514)
    ("ELBASAN LETTER NJE"                                                          . #x10515)
    ("ELBASAN LETTER O"                                                            . #x10516)
    ("ELBASAN LETTER PE"                                                           . #x10517)
    ("ELBASAN LETTER QE"                                                           . #x10518)
    ("ELBASAN LETTER RE"                                                           . #x10519)
    ("ELBASAN LETTER RRE"                                                          . #x1051A)
    ("ELBASAN LETTER SE"                                                           . #x1051B)
    ("ELBASAN LETTER SHE"                                                          . #x1051C)
    ("ELBASAN LETTER TE"                                                           . #x1051D)
    ("ELBASAN LETTER THE"                                                          . #x1051E)
    ("ELBASAN LETTER U"                                                            . #x1051F)
    ("ELBASAN LETTER VE"                                                           . #x10520)
    ("ELBASAN LETTER XE"                                                           . #x10521)
    ("ELBASAN LETTER Y"                                                            . #x10522)
    ("ELBASAN LETTER ZE"                                                           . #x10523)
    ("ELBASAN LETTER ZHE"                                                          . #x10524)
    ("ELBASAN LETTER GHE"                                                          . #x10525)
    ("ELBASAN LETTER GHAMMA"                                                       . #x10526)
    ("ELBASAN LETTER KHE"                                                          . #x10527)
    ("CAUCASIAN ALBANIAN LETTER ALT"                                               . #x10530)
    ("CAUCASIAN ALBANIAN LETTER BET"                                               . #x10531)
    ("CAUCASIAN ALBANIAN LETTER GIM"                                               . #x10532)
    ("CAUCASIAN ALBANIAN LETTER DAT"                                               . #x10533)
    ("CAUCASIAN ALBANIAN LETTER EB"                                                . #x10534)
    ("CAUCASIAN ALBANIAN LETTER ZARL"                                              . #x10535)
    ("CAUCASIAN ALBANIAN LETTER EYN"                                               . #x10536)
    ("CAUCASIAN ALBANIAN LETTER ZHIL"                                              . #x10537)
    ("CAUCASIAN ALBANIAN LETTER TAS"                                               . #x10538)
    ("CAUCASIAN ALBANIAN LETTER CHA"                                               . #x10539)
    ("CAUCASIAN ALBANIAN LETTER YOWD"                                              . #x1053A)
    ("CAUCASIAN ALBANIAN LETTER ZHA"                                               . #x1053B)
    ("CAUCASIAN ALBANIAN LETTER IRB"                                               . #x1053C)
    ("CAUCASIAN ALBANIAN LETTER SHA"                                               . #x1053D)
    ("CAUCASIAN ALBANIAN LETTER LAN"                                               . #x1053E)
    ("CAUCASIAN ALBANIAN LETTER INYA"                                              . #x1053F)
    ("CAUCASIAN ALBANIAN LETTER XEYN"                                              . #x10540)
    ("CAUCASIAN ALBANIAN LETTER DYAN"                                              . #x10541)
    ("CAUCASIAN ALBANIAN LETTER CAR"                                               . #x10542)
    ("CAUCASIAN ALBANIAN LETTER JHOX"                                              . #x10543)
    ("CAUCASIAN ALBANIAN LETTER KAR"                                               . #x10544)
    ("CAUCASIAN ALBANIAN LETTER LYIT"                                              . #x10545)
    ("CAUCASIAN ALBANIAN LETTER HEYT"                                              . #x10546)
    ("CAUCASIAN ALBANIAN LETTER QAY"                                               . #x10547)
    ("CAUCASIAN ALBANIAN LETTER AOR"                                               . #x10548)
    ("CAUCASIAN ALBANIAN LETTER CHOY"                                              . #x10549)
    ("CAUCASIAN ALBANIAN LETTER CHI"                                               . #x1054A)
    ("CAUCASIAN ALBANIAN LETTER CYAY"                                              . #x1054B)
    ("CAUCASIAN ALBANIAN LETTER MAQ"                                               . #x1054C)
    ("CAUCASIAN ALBANIAN LETTER QAR"                                               . #x1054D)
    ("CAUCASIAN ALBANIAN LETTER NOWC"                                              . #x1054E)
    ("CAUCASIAN ALBANIAN LETTER DZYAY"                                             . #x1054F)
    ("CAUCASIAN ALBANIAN LETTER SHAK"                                              . #x10550)
    ("CAUCASIAN ALBANIAN LETTER JAYN"                                              . #x10551)
    ("CAUCASIAN ALBANIAN LETTER ON"                                                . #x10552)
    ("CAUCASIAN ALBANIAN LETTER TYAY"                                              . #x10553)
    ("CAUCASIAN ALBANIAN LETTER FAM"                                               . #x10554)
    ("CAUCASIAN ALBANIAN LETTER DZAY"                                              . #x10555)
    ("CAUCASIAN ALBANIAN LETTER CHAT"                                              . #x10556)
    ("CAUCASIAN ALBANIAN LETTER PEN"                                               . #x10557)
    ("CAUCASIAN ALBANIAN LETTER GHEYS"                                             . #x10558)
    ("CAUCASIAN ALBANIAN LETTER RAT"                                               . #x10559)
    ("CAUCASIAN ALBANIAN LETTER SEYK"                                              . #x1055A)
    ("CAUCASIAN ALBANIAN LETTER VEYZ"                                              . #x1055B)
    ("CAUCASIAN ALBANIAN LETTER TIWR"                                              . #x1055C)
    ("CAUCASIAN ALBANIAN LETTER SHOY"                                              . #x1055D)
    ("CAUCASIAN ALBANIAN LETTER IWN"                                               . #x1055E)
    ("CAUCASIAN ALBANIAN LETTER CYAW"                                              . #x1055F)
    ("CAUCASIAN ALBANIAN LETTER CAYN"                                              . #x10560)
    ("CAUCASIAN ALBANIAN LETTER YAYD"                                              . #x10561)
    ("CAUCASIAN ALBANIAN LETTER PIWR"                                              . #x10562)
    ("CAUCASIAN ALBANIAN LETTER KIW"                                               . #x10563)
    ("CAUCASIAN ALBANIAN CITATION MARK"                                            . #x1056F)
    ("LINEAR A SIGN AB001"                                                         . #x10600)
    ("LINEAR A SIGN AB002"                                                         . #x10601)
    ("LINEAR A SIGN AB003"                                                         . #x10602)
    ("LINEAR A SIGN AB004"                                                         . #x10603)
    ("LINEAR A SIGN AB005"                                                         . #x10604)
    ("LINEAR A SIGN AB006"                                                         . #x10605)
    ("LINEAR A SIGN AB007"                                                         . #x10606)
    ("LINEAR A SIGN AB008"                                                         . #x10607)
    ("LINEAR A SIGN AB009"                                                         . #x10608)
    ("LINEAR A SIGN AB010"                                                         . #x10609)
    ("LINEAR A SIGN AB011"                                                         . #x1060A)
    ("LINEAR A SIGN AB013"                                                         . #x1060B)
    ("LINEAR A SIGN AB016"                                                         . #x1060C)
    ("LINEAR A SIGN AB017"                                                         . #x1060D)
    ("LINEAR A SIGN AB020"                                                         . #x1060E)
    ("LINEAR A SIGN AB021"                                                         . #x1060F)
    ("LINEAR A SIGN AB021F"                                                        . #x10610)
    ("LINEAR A SIGN AB021M"                                                        . #x10611)
    ("LINEAR A SIGN AB022"                                                         . #x10612)
    ("LINEAR A SIGN AB022F"                                                        . #x10613)
    ("LINEAR A SIGN AB022M"                                                        . #x10614)
    ("LINEAR A SIGN AB023"                                                         . #x10615)
    ("LINEAR A SIGN AB023M"                                                        . #x10616)
    ("LINEAR A SIGN AB024"                                                         . #x10617)
    ("LINEAR A SIGN AB026"                                                         . #x10618)
    ("LINEAR A SIGN AB027"                                                         . #x10619)
    ("LINEAR A SIGN AB028"                                                         . #x1061A)
    ("LINEAR A SIGN A028B"                                                         . #x1061B)
    ("LINEAR A SIGN AB029"                                                         . #x1061C)
    ("LINEAR A SIGN AB030"                                                         . #x1061D)
    ("LINEAR A SIGN AB031"                                                         . #x1061E)
    ("LINEAR A SIGN AB034"                                                         . #x1061F)
    ("LINEAR A SIGN AB037"                                                         . #x10620)
    ("LINEAR A SIGN AB038"                                                         . #x10621)
    ("LINEAR A SIGN AB039"                                                         . #x10622)
    ("LINEAR A SIGN AB040"                                                         . #x10623)
    ("LINEAR A SIGN AB041"                                                         . #x10624)
    ("LINEAR A SIGN AB044"                                                         . #x10625)
    ("LINEAR A SIGN AB045"                                                         . #x10626)
    ("LINEAR A SIGN AB046"                                                         . #x10627)
    ("LINEAR A SIGN AB047"                                                         . #x10628)
    ("LINEAR A SIGN AB048"                                                         . #x10629)
    ("LINEAR A SIGN AB049"                                                         . #x1062A)
    ("LINEAR A SIGN AB050"                                                         . #x1062B)
    ("LINEAR A SIGN AB051"                                                         . #x1062C)
    ("LINEAR A SIGN AB053"                                                         . #x1062D)
    ("LINEAR A SIGN AB054"                                                         . #x1062E)
    ("LINEAR A SIGN AB055"                                                         . #x1062F)
    ("LINEAR A SIGN AB056"                                                         . #x10630)
    ("LINEAR A SIGN AB057"                                                         . #x10631)
    ("LINEAR A SIGN AB058"                                                         . #x10632)
    ("LINEAR A SIGN AB059"                                                         . #x10633)
    ("LINEAR A SIGN AB060"                                                         . #x10634)
    ("LINEAR A SIGN AB061"                                                         . #x10635)
    ("LINEAR A SIGN AB065"                                                         . #x10636)
    ("LINEAR A SIGN AB066"                                                         . #x10637)
    ("LINEAR A SIGN AB067"                                                         . #x10638)
    ("LINEAR A SIGN AB069"                                                         . #x10639)
    ("LINEAR A SIGN AB070"                                                         . #x1063A)
    ("LINEAR A SIGN AB073"                                                         . #x1063B)
    ("LINEAR A SIGN AB074"                                                         . #x1063C)
    ("LINEAR A SIGN AB076"                                                         . #x1063D)
    ("LINEAR A SIGN AB077"                                                         . #x1063E)
    ("LINEAR A SIGN AB078"                                                         . #x1063F)
    ("LINEAR A SIGN AB079"                                                         . #x10640)
    ("LINEAR A SIGN AB080"                                                         . #x10641)
    ("LINEAR A SIGN AB081"                                                         . #x10642)
    ("LINEAR A SIGN AB082"                                                         . #x10643)
    ("LINEAR A SIGN AB085"                                                         . #x10644)
    ("LINEAR A SIGN AB086"                                                         . #x10645)
    ("LINEAR A SIGN AB087"                                                         . #x10646)
    ("LINEAR A SIGN A100-102"                                                      . #x10647)
    ("LINEAR A SIGN AB118"                                                         . #x10648)
    ("LINEAR A SIGN AB120"                                                         . #x10649)
    ("LINEAR A SIGN A120B"                                                         . #x1064A)
    ("LINEAR A SIGN AB122"                                                         . #x1064B)
    ("LINEAR A SIGN AB123"                                                         . #x1064C)
    ("LINEAR A SIGN AB131A"                                                        . #x1064D)
    ("LINEAR A SIGN AB131B"                                                        . #x1064E)
    ("LINEAR A SIGN A131C"                                                         . #x1064F)
    ("LINEAR A SIGN AB164"                                                         . #x10650)
    ("LINEAR A SIGN AB171"                                                         . #x10651)
    ("LINEAR A SIGN AB180"                                                         . #x10652)
    ("LINEAR A SIGN AB188"                                                         . #x10653)
    ("LINEAR A SIGN AB191"                                                         . #x10654)
    ("LINEAR A SIGN A301"                                                          . #x10655)
    ("LINEAR A SIGN A302"                                                          . #x10656)
    ("LINEAR A SIGN A303"                                                          . #x10657)
    ("LINEAR A SIGN A304"                                                          . #x10658)
    ("LINEAR A SIGN A305"                                                          . #x10659)
    ("LINEAR A SIGN A306"                                                          . #x1065A)
    ("LINEAR A SIGN A307"                                                          . #x1065B)
    ("LINEAR A SIGN A308"                                                          . #x1065C)
    ("LINEAR A SIGN A309A"                                                         . #x1065D)
    ("LINEAR A SIGN A309B"                                                         . #x1065E)
    ("LINEAR A SIGN A309C"                                                         . #x1065F)
    ("LINEAR A SIGN A310"                                                          . #x10660)
    ("LINEAR A SIGN A311"                                                          . #x10661)
    ("LINEAR A SIGN A312"                                                          . #x10662)
    ("LINEAR A SIGN A313A"                                                         . #x10663)
    ("LINEAR A SIGN A313B"                                                         . #x10664)
    ("LINEAR A SIGN A313C"                                                         . #x10665)
    ("LINEAR A SIGN A314"                                                          . #x10666)
    ("LINEAR A SIGN A315"                                                          . #x10667)
    ("LINEAR A SIGN A316"                                                          . #x10668)
    ("LINEAR A SIGN A317"                                                          . #x10669)
    ("LINEAR A SIGN A318"                                                          . #x1066A)
    ("LINEAR A SIGN A319"                                                          . #x1066B)
    ("LINEAR A SIGN A320"                                                          . #x1066C)
    ("LINEAR A SIGN A321"                                                          . #x1066D)
    ("LINEAR A SIGN A322"                                                          . #x1066E)
    ("LINEAR A SIGN A323"                                                          . #x1066F)
    ("LINEAR A SIGN A324"                                                          . #x10670)
    ("LINEAR A SIGN A325"                                                          . #x10671)
    ("LINEAR A SIGN A326"                                                          . #x10672)
    ("LINEAR A SIGN A327"                                                          . #x10673)
    ("LINEAR A SIGN A328"                                                          . #x10674)
    ("LINEAR A SIGN A329"                                                          . #x10675)
    ("LINEAR A SIGN A330"                                                          . #x10676)
    ("LINEAR A SIGN A331"                                                          . #x10677)
    ("LINEAR A SIGN A332"                                                          . #x10678)
    ("LINEAR A SIGN A333"                                                          . #x10679)
    ("LINEAR A SIGN A334"                                                          . #x1067A)
    ("LINEAR A SIGN A335"                                                          . #x1067B)
    ("LINEAR A SIGN A336"                                                          . #x1067C)
    ("LINEAR A SIGN A337"                                                          . #x1067D)
    ("LINEAR A SIGN A338"                                                          . #x1067E)
    ("LINEAR A SIGN A339"                                                          . #x1067F)
    ("LINEAR A SIGN A340"                                                          . #x10680)
    ("LINEAR A SIGN A341"                                                          . #x10681)
    ("LINEAR A SIGN A342"                                                          . #x10682)
    ("LINEAR A SIGN A343"                                                          . #x10683)
    ("LINEAR A SIGN A344"                                                          . #x10684)
    ("LINEAR A SIGN A345"                                                          . #x10685)
    ("LINEAR A SIGN A346"                                                          . #x10686)
    ("LINEAR A SIGN A347"                                                          . #x10687)
    ("LINEAR A SIGN A348"                                                          . #x10688)
    ("LINEAR A SIGN A349"                                                          . #x10689)
    ("LINEAR A SIGN A350"                                                          . #x1068A)
    ("LINEAR A SIGN A351"                                                          . #x1068B)
    ("LINEAR A SIGN A352"                                                          . #x1068C)
    ("LINEAR A SIGN A353"                                                          . #x1068D)
    ("LINEAR A SIGN A354"                                                          . #x1068E)
    ("LINEAR A SIGN A355"                                                          . #x1068F)
    ("LINEAR A SIGN A356"                                                          . #x10690)
    ("LINEAR A SIGN A357"                                                          . #x10691)
    ("LINEAR A SIGN A358"                                                          . #x10692)
    ("LINEAR A SIGN A359"                                                          . #x10693)
    ("LINEAR A SIGN A360"                                                          . #x10694)
    ("LINEAR A SIGN A361"                                                          . #x10695)
    ("LINEAR A SIGN A362"                                                          . #x10696)
    ("LINEAR A SIGN A363"                                                          . #x10697)
    ("LINEAR A SIGN A364"                                                          . #x10698)
    ("LINEAR A SIGN A365"                                                          . #x10699)
    ("LINEAR A SIGN A366"                                                          . #x1069A)
    ("LINEAR A SIGN A367"                                                          . #x1069B)
    ("LINEAR A SIGN A368"                                                          . #x1069C)
    ("LINEAR A SIGN A369"                                                          . #x1069D)
    ("LINEAR A SIGN A370"                                                          . #x1069E)
    ("LINEAR A SIGN A371"                                                          . #x1069F)
    ("LINEAR A SIGN A400-VAS"                                                      . #x106A0)
    ("LINEAR A SIGN A401-VAS"                                                      . #x106A1)
    ("LINEAR A SIGN A402-VAS"                                                      . #x106A2)
    ("LINEAR A SIGN A403-VAS"                                                      . #x106A3)
    ("LINEAR A SIGN A404-VAS"                                                      . #x106A4)
    ("LINEAR A SIGN A405-VAS"                                                      . #x106A5)
    ("LINEAR A SIGN A406-VAS"                                                      . #x106A6)
    ("LINEAR A SIGN A407-VAS"                                                      . #x106A7)
    ("LINEAR A SIGN A408-VAS"                                                      . #x106A8)
    ("LINEAR A SIGN A409-VAS"                                                      . #x106A9)
    ("LINEAR A SIGN A410-VAS"                                                      . #x106AA)
    ("LINEAR A SIGN A411-VAS"                                                      . #x106AB)
    ("LINEAR A SIGN A412-VAS"                                                      . #x106AC)
    ("LINEAR A SIGN A413-VAS"                                                      . #x106AD)
    ("LINEAR A SIGN A414-VAS"                                                      . #x106AE)
    ("LINEAR A SIGN A415-VAS"                                                      . #x106AF)
    ("LINEAR A SIGN A416-VAS"                                                      . #x106B0)
    ("LINEAR A SIGN A417-VAS"                                                      . #x106B1)
    ("LINEAR A SIGN A418-VAS"                                                      . #x106B2)
    ("LINEAR A SIGN A501"                                                          . #x106B3)
    ("LINEAR A SIGN A502"                                                          . #x106B4)
    ("LINEAR A SIGN A503"                                                          . #x106B5)
    ("LINEAR A SIGN A504"                                                          . #x106B6)
    ("LINEAR A SIGN A505"                                                          . #x106B7)
    ("LINEAR A SIGN A506"                                                          . #x106B8)
    ("LINEAR A SIGN A508"                                                          . #x106B9)
    ("LINEAR A SIGN A509"                                                          . #x106BA)
    ("LINEAR A SIGN A510"                                                          . #x106BB)
    ("LINEAR A SIGN A511"                                                          . #x106BC)
    ("LINEAR A SIGN A512"                                                          . #x106BD)
    ("LINEAR A SIGN A513"                                                          . #x106BE)
    ("LINEAR A SIGN A515"                                                          . #x106BF)
    ("LINEAR A SIGN A516"                                                          . #x106C0)
    ("LINEAR A SIGN A520"                                                          . #x106C1)
    ("LINEAR A SIGN A521"                                                          . #x106C2)
    ("LINEAR A SIGN A523"                                                          . #x106C3)
    ("LINEAR A SIGN A524"                                                          . #x106C4)
    ("LINEAR A SIGN A525"                                                          . #x106C5)
    ("LINEAR A SIGN A526"                                                          . #x106C6)
    ("LINEAR A SIGN A527"                                                          . #x106C7)
    ("LINEAR A SIGN A528"                                                          . #x106C8)
    ("LINEAR A SIGN A529"                                                          . #x106C9)
    ("LINEAR A SIGN A530"                                                          . #x106CA)
    ("LINEAR A SIGN A531"                                                          . #x106CB)
    ("LINEAR A SIGN A532"                                                          . #x106CC)
    ("LINEAR A SIGN A534"                                                          . #x106CD)
    ("LINEAR A SIGN A535"                                                          . #x106CE)
    ("LINEAR A SIGN A536"                                                          . #x106CF)
    ("LINEAR A SIGN A537"                                                          . #x106D0)
    ("LINEAR A SIGN A538"                                                          . #x106D1)
    ("LINEAR A SIGN A539"                                                          . #x106D2)
    ("LINEAR A SIGN A540"                                                          . #x106D3)
    ("LINEAR A SIGN A541"                                                          . #x106D4)
    ("LINEAR A SIGN A542"                                                          . #x106D5)
    ("LINEAR A SIGN A545"                                                          . #x106D6)
    ("LINEAR A SIGN A547"                                                          . #x106D7)
    ("LINEAR A SIGN A548"                                                          . #x106D8)
    ("LINEAR A SIGN A549"                                                          . #x106D9)
    ("LINEAR A SIGN A550"                                                          . #x106DA)
    ("LINEAR A SIGN A551"                                                          . #x106DB)
    ("LINEAR A SIGN A552"                                                          . #x106DC)
    ("LINEAR A SIGN A553"                                                          . #x106DD)
    ("LINEAR A SIGN A554"                                                          . #x106DE)
    ("LINEAR A SIGN A555"                                                          . #x106DF)
    ("LINEAR A SIGN A556"                                                          . #x106E0)
    ("LINEAR A SIGN A557"                                                          . #x106E1)
    ("LINEAR A SIGN A559"                                                          . #x106E2)
    ("LINEAR A SIGN A563"                                                          . #x106E3)
    ("LINEAR A SIGN A564"                                                          . #x106E4)
    ("LINEAR A SIGN A565"                                                          . #x106E5)
    ("LINEAR A SIGN A566"                                                          . #x106E6)
    ("LINEAR A SIGN A568"                                                          . #x106E7)
    ("LINEAR A SIGN A569"                                                          . #x106E8)
    ("LINEAR A SIGN A570"                                                          . #x106E9)
    ("LINEAR A SIGN A571"                                                          . #x106EA)
    ("LINEAR A SIGN A572"                                                          . #x106EB)
    ("LINEAR A SIGN A573"                                                          . #x106EC)
    ("LINEAR A SIGN A574"                                                          . #x106ED)
    ("LINEAR A SIGN A575"                                                          . #x106EE)
    ("LINEAR A SIGN A576"                                                          . #x106EF)
    ("LINEAR A SIGN A577"                                                          . #x106F0)
    ("LINEAR A SIGN A578"                                                          . #x106F1)
    ("LINEAR A SIGN A579"                                                          . #x106F2)
    ("LINEAR A SIGN A580"                                                          . #x106F3)
    ("LINEAR A SIGN A581"                                                          . #x106F4)
    ("LINEAR A SIGN A582"                                                          . #x106F5)
    ("LINEAR A SIGN A583"                                                          . #x106F6)
    ("LINEAR A SIGN A584"                                                          . #x106F7)
    ("LINEAR A SIGN A585"                                                          . #x106F8)
    ("LINEAR A SIGN A586"                                                          . #x106F9)
    ("LINEAR A SIGN A587"                                                          . #x106FA)
    ("LINEAR A SIGN A588"                                                          . #x106FB)
    ("LINEAR A SIGN A589"                                                          . #x106FC)
    ("LINEAR A SIGN A591"                                                          . #x106FD)
    ("LINEAR A SIGN A592"                                                          . #x106FE)
    ("LINEAR A SIGN A594"                                                          . #x106FF)
    ("LINEAR A SIGN A595"                                                          . #x10700)
    ("LINEAR A SIGN A596"                                                          . #x10701)
    ("LINEAR A SIGN A598"                                                          . #x10702)
    ("LINEAR A SIGN A600"                                                          . #x10703)
    ("LINEAR A SIGN A601"                                                          . #x10704)
    ("LINEAR A SIGN A602"                                                          . #x10705)
    ("LINEAR A SIGN A603"                                                          . #x10706)
    ("LINEAR A SIGN A604"                                                          . #x10707)
    ("LINEAR A SIGN A606"                                                          . #x10708)
    ("LINEAR A SIGN A608"                                                          . #x10709)
    ("LINEAR A SIGN A609"                                                          . #x1070A)
    ("LINEAR A SIGN A610"                                                          . #x1070B)
    ("LINEAR A SIGN A611"                                                          . #x1070C)
    ("LINEAR A SIGN A612"                                                          . #x1070D)
    ("LINEAR A SIGN A613"                                                          . #x1070E)
    ("LINEAR A SIGN A614"                                                          . #x1070F)
    ("LINEAR A SIGN A615"                                                          . #x10710)
    ("LINEAR A SIGN A616"                                                          . #x10711)
    ("LINEAR A SIGN A617"                                                          . #x10712)
    ("LINEAR A SIGN A618"                                                          . #x10713)
    ("LINEAR A SIGN A619"                                                          . #x10714)
    ("LINEAR A SIGN A620"                                                          . #x10715)
    ("LINEAR A SIGN A621"                                                          . #x10716)
    ("LINEAR A SIGN A622"                                                          . #x10717)
    ("LINEAR A SIGN A623"                                                          . #x10718)
    ("LINEAR A SIGN A624"                                                          . #x10719)
    ("LINEAR A SIGN A626"                                                          . #x1071A)
    ("LINEAR A SIGN A627"                                                          . #x1071B)
    ("LINEAR A SIGN A628"                                                          . #x1071C)
    ("LINEAR A SIGN A629"                                                          . #x1071D)
    ("LINEAR A SIGN A634"                                                          . #x1071E)
    ("LINEAR A SIGN A637"                                                          . #x1071F)
    ("LINEAR A SIGN A638"                                                          . #x10720)
    ("LINEAR A SIGN A640"                                                          . #x10721)
    ("LINEAR A SIGN A642"                                                          . #x10722)
    ("LINEAR A SIGN A643"                                                          . #x10723)
    ("LINEAR A SIGN A644"                                                          . #x10724)
    ("LINEAR A SIGN A645"                                                          . #x10725)
    ("LINEAR A SIGN A646"                                                          . #x10726)
    ("LINEAR A SIGN A648"                                                          . #x10727)
    ("LINEAR A SIGN A649"                                                          . #x10728)
    ("LINEAR A SIGN A651"                                                          . #x10729)
    ("LINEAR A SIGN A652"                                                          . #x1072A)
    ("LINEAR A SIGN A653"                                                          . #x1072B)
    ("LINEAR A SIGN A654"                                                          . #x1072C)
    ("LINEAR A SIGN A655"                                                          . #x1072D)
    ("LINEAR A SIGN A656"                                                          . #x1072E)
    ("LINEAR A SIGN A657"                                                          . #x1072F)
    ("LINEAR A SIGN A658"                                                          . #x10730)
    ("LINEAR A SIGN A659"                                                          . #x10731)
    ("LINEAR A SIGN A660"                                                          . #x10732)
    ("LINEAR A SIGN A661"                                                          . #x10733)
    ("LINEAR A SIGN A662"                                                          . #x10734)
    ("LINEAR A SIGN A663"                                                          . #x10735)
    ("LINEAR A SIGN A664"                                                          . #x10736)
    ("LINEAR A SIGN A701 A"                                                        . #x10740)
    ("LINEAR A SIGN A702 B"                                                        . #x10741)
    ("LINEAR A SIGN A703 D"                                                        . #x10742)
    ("LINEAR A SIGN A704 E"                                                        . #x10743)
    ("LINEAR A SIGN A705 F"                                                        . #x10744)
    ("LINEAR A SIGN A706 H"                                                        . #x10745)
    ("LINEAR A SIGN A707 J"                                                        . #x10746)
    ("LINEAR A SIGN A708 K"                                                        . #x10747)
    ("LINEAR A SIGN A709 L"                                                        . #x10748)
    ("LINEAR A SIGN A709-2 L2"                                                     . #x10749)
    ("LINEAR A SIGN A709-3 L3"                                                     . #x1074A)
    ("LINEAR A SIGN A709-4 L4"                                                     . #x1074B)
    ("LINEAR A SIGN A709-6 L6"                                                     . #x1074C)
    ("LINEAR A SIGN A710 W"                                                        . #x1074D)
    ("LINEAR A SIGN A711 X"                                                        . #x1074E)
    ("LINEAR A SIGN A712 Y"                                                        . #x1074F)
    ("LINEAR A SIGN A713 OMEGA"                                                    . #x10750)
    ("LINEAR A SIGN A714 ABB"                                                      . #x10751)
    ("LINEAR A SIGN A715 BB"                                                       . #x10752)
    ("LINEAR A SIGN A717 DD"                                                       . #x10753)
    ("LINEAR A SIGN A726 EYYY"                                                     . #x10754)
    ("LINEAR A SIGN A732 JE"                                                       . #x10755)
    ("LINEAR A SIGN A800"                                                          . #x10760)
    ("LINEAR A SIGN A801"                                                          . #x10761)
    ("LINEAR A SIGN A802"                                                          . #x10762)
    ("LINEAR A SIGN A803"                                                          . #x10763)
    ("LINEAR A SIGN A804"                                                          . #x10764)
    ("LINEAR A SIGN A805"                                                          . #x10765)
    ("LINEAR A SIGN A806"                                                          . #x10766)
    ("LINEAR A SIGN A807"                                                          . #x10767)
    ("PALMYRENE LETTER ALEPH"                                                      . #x10860)
    ("PALMYRENE LETTER BETH"                                                       . #x10861)
    ("PALMYRENE LETTER GIMEL"                                                      . #x10862)
    ("PALMYRENE LETTER DALETH"                                                     . #x10863)
    ("PALMYRENE LETTER HE"                                                         . #x10864)
    ("PALMYRENE LETTER WAW"                                                        . #x10865)
    ("PALMYRENE LETTER ZAYIN"                                                      . #x10866)
    ("PALMYRENE LETTER HETH"                                                       . #x10867)
    ("PALMYRENE LETTER TETH"                                                       . #x10868)
    ("PALMYRENE LETTER YODH"                                                       . #x10869)
    ("PALMYRENE LETTER KAPH"                                                       . #x1086A)
    ("PALMYRENE LETTER LAMEDH"                                                     . #x1086B)
    ("PALMYRENE LETTER MEM"                                                        . #x1086C)
    ("PALMYRENE LETTER FINAL NUN"                                                  . #x1086D)
    ("PALMYRENE LETTER NUN"                                                        . #x1086E)
    ("PALMYRENE LETTER SAMEKH"                                                     . #x1086F)
    ("PALMYRENE LETTER AYIN"                                                       . #x10870)
    ("PALMYRENE LETTER PE"                                                         . #x10871)
    ("PALMYRENE LETTER SADHE"                                                      . #x10872)
    ("PALMYRENE LETTER QOPH"                                                       . #x10873)
    ("PALMYRENE LETTER RESH"                                                       . #x10874)
    ("PALMYRENE LETTER SHIN"                                                       . #x10875)
    ("PALMYRENE LETTER TAW"                                                        . #x10876)
    ("PALMYRENE LEFT-POINTING FLEURON"                                             . #x10877)
    ("PALMYRENE RIGHT-POINTING FLEURON"                                            . #x10878)
    ("PALMYRENE NUMBER ONE"                                                        . #x10879)
    ("PALMYRENE NUMBER TWO"                                                        . #x1087A)
    ("PALMYRENE NUMBER THREE"                                                      . #x1087B)
    ("PALMYRENE NUMBER FOUR"                                                       . #x1087C)
    ("PALMYRENE NUMBER FIVE"                                                       . #x1087D)
    ("PALMYRENE NUMBER TEN"                                                        . #x1087E)
    ("PALMYRENE NUMBER TWENTY"                                                     . #x1087F)
    ("NABATAEAN LETTER FINAL ALEPH"                                                . #x10880)
    ("NABATAEAN LETTER ALEPH"                                                      . #x10881)
    ("NABATAEAN LETTER FINAL BETH"                                                 . #x10882)
    ("NABATAEAN LETTER BETH"                                                       . #x10883)
    ("NABATAEAN LETTER GIMEL"                                                      . #x10884)
    ("NABATAEAN LETTER DALETH"                                                     . #x10885)
    ("NABATAEAN LETTER FINAL HE"                                                   . #x10886)
    ("NABATAEAN LETTER HE"                                                         . #x10887)
    ("NABATAEAN LETTER WAW"                                                        . #x10888)
    ("NABATAEAN LETTER ZAYIN"                                                      . #x10889)
    ("NABATAEAN LETTER HETH"                                                       . #x1088A)
    ("NABATAEAN LETTER TETH"                                                       . #x1088B)
    ("NABATAEAN LETTER FINAL YODH"                                                 . #x1088C)
    ("NABATAEAN LETTER YODH"                                                       . #x1088D)
    ("NABATAEAN LETTER FINAL KAPH"                                                 . #x1088E)
    ("NABATAEAN LETTER KAPH"                                                       . #x1088F)
    ("NABATAEAN LETTER FINAL LAMEDH"                                               . #x10890)
    ("NABATAEAN LETTER LAMEDH"                                                     . #x10891)
    ("NABATAEAN LETTER FINAL MEM"                                                  . #x10892)
    ("NABATAEAN LETTER MEM"                                                        . #x10893)
    ("NABATAEAN LETTER FINAL NUN"                                                  . #x10894)
    ("NABATAEAN LETTER NUN"                                                        . #x10895)
    ("NABATAEAN LETTER SAMEKH"                                                     . #x10896)
    ("NABATAEAN LETTER AYIN"                                                       . #x10897)
    ("NABATAEAN LETTER PE"                                                         . #x10898)
    ("NABATAEAN LETTER SADHE"                                                      . #x10899)
    ("NABATAEAN LETTER QOPH"                                                       . #x1089A)
    ("NABATAEAN LETTER RESH"                                                       . #x1089B)
    ("NABATAEAN LETTER FINAL SHIN"                                                 . #x1089C)
    ("NABATAEAN LETTER SHIN"                                                       . #x1089D)
    ("NABATAEAN LETTER TAW"                                                        . #x1089E)
    ("NABATAEAN NUMBER ONE"                                                        . #x108A7)
    ("NABATAEAN NUMBER TWO"                                                        . #x108A8)
    ("NABATAEAN NUMBER THREE"                                                      . #x108A9)
    ("NABATAEAN NUMBER FOUR"                                                       . #x108AA)
    ("NABATAEAN CRUCIFORM NUMBER FOUR"                                             . #x108AB)
    ("NABATAEAN NUMBER FIVE"                                                       . #x108AC)
    ("NABATAEAN NUMBER TEN"                                                        . #x108AD)
    ("NABATAEAN NUMBER TWENTY"                                                     . #x108AE)
    ("NABATAEAN NUMBER ONE HUNDRED"                                                . #x108AF)
    ("OLD NORTH ARABIAN LETTER HEH"                                                . #x10A80)
    ("OLD NORTH ARABIAN LETTER LAM"                                                . #x10A81)
    ("OLD NORTH ARABIAN LETTER HAH"                                                . #x10A82)
    ("OLD NORTH ARABIAN LETTER MEEM"                                               . #x10A83)
    ("OLD NORTH ARABIAN LETTER QAF"                                                . #x10A84)
    ("OLD NORTH ARABIAN LETTER WAW"                                                . #x10A85)
    ("OLD NORTH ARABIAN LETTER ES-2"                                               . #x10A86)
    ("OLD NORTH ARABIAN LETTER REH"                                                . #x10A87)
    ("OLD NORTH ARABIAN LETTER BEH"                                                . #x10A88)
    ("OLD NORTH ARABIAN LETTER TEH"                                                . #x10A89)
    ("OLD NORTH ARABIAN LETTER ES-1"                                               . #x10A8A)
    ("OLD NORTH ARABIAN LETTER KAF"                                                . #x10A8B)
    ("OLD NORTH ARABIAN LETTER NOON"                                               . #x10A8C)
    ("OLD NORTH ARABIAN LETTER KHAH"                                               . #x10A8D)
    ("OLD NORTH ARABIAN LETTER SAD"                                                . #x10A8E)
    ("OLD NORTH ARABIAN LETTER ES-3"                                               . #x10A8F)
    ("OLD NORTH ARABIAN LETTER FEH"                                                . #x10A90)
    ("OLD NORTH ARABIAN LETTER ALEF"                                               . #x10A91)
    ("OLD NORTH ARABIAN LETTER AIN"                                                . #x10A92)
    ("OLD NORTH ARABIAN LETTER DAD"                                                . #x10A93)
    ("OLD NORTH ARABIAN LETTER GEEM"                                               . #x10A94)
    ("OLD NORTH ARABIAN LETTER DAL"                                                . #x10A95)
    ("OLD NORTH ARABIAN LETTER GHAIN"                                              . #x10A96)
    ("OLD NORTH ARABIAN LETTER TAH"                                                . #x10A97)
    ("OLD NORTH ARABIAN LETTER ZAIN"                                               . #x10A98)
    ("OLD NORTH ARABIAN LETTER THAL"                                               . #x10A99)
    ("OLD NORTH ARABIAN LETTER YEH"                                                . #x10A9A)
    ("OLD NORTH ARABIAN LETTER THEH"                                               . #x10A9B)
    ("OLD NORTH ARABIAN LETTER ZAH"                                                . #x10A9C)
    ("OLD NORTH ARABIAN NUMBER ONE"                                                . #x10A9D)
    ("OLD NORTH ARABIAN NUMBER TEN"                                                . #x10A9E)
    ("OLD NORTH ARABIAN NUMBER TWENTY"                                             . #x10A9F)
    ("MANICHAEAN LETTER ALEPH"                                                     . #x10AC0)
    ("MANICHAEAN LETTER BETH"                                                      . #x10AC1)
    ("MANICHAEAN LETTER BHETH"                                                     . #x10AC2)
    ("MANICHAEAN LETTER GIMEL"                                                     . #x10AC3)
    ("MANICHAEAN LETTER GHIMEL"                                                    . #x10AC4)
    ("MANICHAEAN LETTER DALETH"                                                    . #x10AC5)
    ("MANICHAEAN LETTER HE"                                                        . #x10AC6)
    ("MANICHAEAN LETTER WAW"                                                       . #x10AC7)
    ("MANICHAEAN SIGN UD"                                                          . #x10AC8)
    ("MANICHAEAN LETTER ZAYIN"                                                     . #x10AC9)
    ("MANICHAEAN LETTER ZHAYIN"                                                    . #x10ACA)
    ("MANICHAEAN LETTER JAYIN"                                                     . #x10ACB)
    ("MANICHAEAN LETTER JHAYIN"                                                    . #x10ACC)
    ("MANICHAEAN LETTER HETH"                                                      . #x10ACD)
    ("MANICHAEAN LETTER TETH"                                                      . #x10ACE)
    ("MANICHAEAN LETTER YODH"                                                      . #x10ACF)
    ("MANICHAEAN LETTER KAPH"                                                      . #x10AD0)
    ("MANICHAEAN LETTER XAPH"                                                      . #x10AD1)
    ("MANICHAEAN LETTER KHAPH"                                                     . #x10AD2)
    ("MANICHAEAN LETTER LAMEDH"                                                    . #x10AD3)
    ("MANICHAEAN LETTER DHAMEDH"                                                   . #x10AD4)
    ("MANICHAEAN LETTER THAMEDH"                                                   . #x10AD5)
    ("MANICHAEAN LETTER MEM"                                                       . #x10AD6)
    ("MANICHAEAN LETTER NUN"                                                       . #x10AD7)
    ("MANICHAEAN LETTER SAMEKH"                                                    . #x10AD8)
    ("MANICHAEAN LETTER AYIN"                                                      . #x10AD9)
    ("MANICHAEAN LETTER AAYIN"                                                     . #x10ADA)
    ("MANICHAEAN LETTER PE"                                                        . #x10ADB)
    ("MANICHAEAN LETTER FE"                                                        . #x10ADC)
    ("MANICHAEAN LETTER SADHE"                                                     . #x10ADD)
    ("MANICHAEAN LETTER QOPH"                                                      . #x10ADE)
    ("MANICHAEAN LETTER XOPH"                                                      . #x10ADF)
    ("MANICHAEAN LETTER QHOPH"                                                     . #x10AE0)
    ("MANICHAEAN LETTER RESH"                                                      . #x10AE1)
    ("MANICHAEAN LETTER SHIN"                                                      . #x10AE2)
    ("MANICHAEAN LETTER SSHIN"                                                     . #x10AE3)
    ("MANICHAEAN LETTER TAW"                                                       . #x10AE4)
    ("MANICHAEAN ABBREVIATION MARK ABOVE"                                          . #x10AE5)
    ("MANICHAEAN ABBREVIATION MARK BELOW"                                          . #x10AE6)
    ("MANICHAEAN NUMBER ONE"                                                       . #x10AEB)
    ("MANICHAEAN NUMBER FIVE"                                                      . #x10AEC)
    ("MANICHAEAN NUMBER TEN"                                                       . #x10AED)
    ("MANICHAEAN NUMBER TWENTY"                                                    . #x10AEE)
    ("MANICHAEAN NUMBER ONE HUNDRED"                                               . #x10AEF)
    ("MANICHAEAN PUNCTUATION STAR"                                                 . #x10AF0)
    ("MANICHAEAN PUNCTUATION FLEURON"                                              . #x10AF1)
    ("MANICHAEAN PUNCTUATION DOUBLE DOT WITHIN DOT"                                . #x10AF2)
    ("MANICHAEAN PUNCTUATION DOT WITHIN DOT"                                       . #x10AF3)
    ("MANICHAEAN PUNCTUATION DOT"                                                  . #x10AF4)
    ("MANICHAEAN PUNCTUATION TWO DOTS"                                             . #x10AF5)
    ("MANICHAEAN PUNCTUATION LINE FILLER"                                          . #x10AF6)
    ("PSALTER PAHLAVI LETTER ALEPH"                                                . #x10B80)
    ("PSALTER PAHLAVI LETTER BETH"                                                 . #x10B81)
    ("PSALTER PAHLAVI LETTER GIMEL"                                                . #x10B82)
    ("PSALTER PAHLAVI LETTER DALETH"                                               . #x10B83)
    ("PSALTER PAHLAVI LETTER HE"                                                   . #x10B84)
    ("PSALTER PAHLAVI LETTER WAW-AYIN-RESH"                                        . #x10B85)
    ("PSALTER PAHLAVI LETTER ZAYIN"                                                . #x10B86)
    ("PSALTER PAHLAVI LETTER HETH"                                                 . #x10B87)
    ("PSALTER PAHLAVI LETTER YODH"                                                 . #x10B88)
    ("PSALTER PAHLAVI LETTER KAPH"                                                 . #x10B89)
    ("PSALTER PAHLAVI LETTER LAMEDH"                                               . #x10B8A)
    ("PSALTER PAHLAVI LETTER MEM-QOPH"                                             . #x10B8B)
    ("PSALTER PAHLAVI LETTER NUN"                                                  . #x10B8C)
    ("PSALTER PAHLAVI LETTER SAMEKH"                                               . #x10B8D)
    ("PSALTER PAHLAVI LETTER PE"                                                   . #x10B8E)
    ("PSALTER PAHLAVI LETTER SADHE"                                                . #x10B8F)
    ("PSALTER PAHLAVI LETTER SHIN"                                                 . #x10B90)
    ("PSALTER PAHLAVI LETTER TAW"                                                  . #x10B91)
    ("PSALTER PAHLAVI SECTION MARK"                                                . #x10B99)
    ("PSALTER PAHLAVI TURNED SECTION MARK"                                         . #x10B9A)
    ("PSALTER PAHLAVI FOUR DOTS WITH CROSS"                                        . #x10B9B)
    ("PSALTER PAHLAVI FOUR DOTS WITH DOT"                                          . #x10B9C)
    ("PSALTER PAHLAVI NUMBER ONE"                                                  . #x10BA9)
    ("PSALTER PAHLAVI NUMBER TWO"                                                  . #x10BAA)
    ("PSALTER PAHLAVI NUMBER THREE"                                                . #x10BAB)
    ("PSALTER PAHLAVI NUMBER FOUR"                                                 . #x10BAC)
    ("PSALTER PAHLAVI NUMBER TEN"                                                  . #x10BAD)
    ("PSALTER PAHLAVI NUMBER TWENTY"                                               . #x10BAE)
    ("PSALTER PAHLAVI NUMBER ONE HUNDRED"                                          . #x10BAF)
    ("BRAHMI NUMBER JOINER"                                                        . #x1107F)
    ("MAHAJANI LETTER A"                                                           . #x11150)
    ("MAHAJANI LETTER I"                                                           . #x11151)
    ("MAHAJANI LETTER U"                                                           . #x11152)
    ("MAHAJANI LETTER E"                                                           . #x11153)
    ("MAHAJANI LETTER O"                                                           . #x11154)
    ("MAHAJANI LETTER KA"                                                          . #x11155)
    ("MAHAJANI LETTER KHA"                                                         . #x11156)
    ("MAHAJANI LETTER GA"                                                          . #x11157)
    ("MAHAJANI LETTER GHA"                                                         . #x11158)
    ("MAHAJANI LETTER CA"                                                          . #x11159)
    ("MAHAJANI LETTER CHA"                                                         . #x1115A)
    ("MAHAJANI LETTER JA"                                                          . #x1115B)
    ("MAHAJANI LETTER JHA"                                                         . #x1115C)
    ("MAHAJANI LETTER NYA"                                                         . #x1115D)
    ("MAHAJANI LETTER TTA"                                                         . #x1115E)
    ("MAHAJANI LETTER TTHA"                                                        . #x1115F)
    ("MAHAJANI LETTER DDA"                                                         . #x11160)
    ("MAHAJANI LETTER DDHA"                                                        . #x11161)
    ("MAHAJANI LETTER NNA"                                                         . #x11162)
    ("MAHAJANI LETTER TA"                                                          . #x11163)
    ("MAHAJANI LETTER THA"                                                         . #x11164)
    ("MAHAJANI LETTER DA"                                                          . #x11165)
    ("MAHAJANI LETTER DHA"                                                         . #x11166)
    ("MAHAJANI LETTER NA"                                                          . #x11167)
    ("MAHAJANI LETTER PA"                                                          . #x11168)
    ("MAHAJANI LETTER PHA"                                                         . #x11169)
    ("MAHAJANI LETTER BA"                                                          . #x1116A)
    ("MAHAJANI LETTER BHA"                                                         . #x1116B)
    ("MAHAJANI LETTER MA"                                                          . #x1116C)
    ("MAHAJANI LETTER RA"                                                          . #x1116D)
    ("MAHAJANI LETTER LA"                                                          . #x1116E)
    ("MAHAJANI LETTER VA"                                                          . #x1116F)
    ("MAHAJANI LETTER SA"                                                          . #x11170)
    ("MAHAJANI LETTER HA"                                                          . #x11171)
    ("MAHAJANI LETTER RRA"                                                         . #x11172)
    ("MAHAJANI SIGN NUKTA"                                                         . #x11173)
    ("MAHAJANI ABBREVIATION SIGN"                                                  . #x11174)
    ("MAHAJANI SECTION MARK"                                                       . #x11175)
    ("MAHAJANI LIGATURE SHRI"                                                      . #x11176)
    ("SHARADA SUTRA MARK"                                                          . #x111CD)
    ("SHARADA EKAM"                                                                . #x111DA)
    ("SINHALA ARCHAIC DIGIT ONE"                                                   . #x111E1)
    ("SINHALA ARCHAIC DIGIT TWO"                                                   . #x111E2)
    ("SINHALA ARCHAIC DIGIT THREE"                                                 . #x111E3)
    ("SINHALA ARCHAIC DIGIT FOUR"                                                  . #x111E4)
    ("SINHALA ARCHAIC DIGIT FIVE"                                                  . #x111E5)
    ("SINHALA ARCHAIC DIGIT SIX"                                                   . #x111E6)
    ("SINHALA ARCHAIC DIGIT SEVEN"                                                 . #x111E7)
    ("SINHALA ARCHAIC DIGIT EIGHT"                                                 . #x111E8)
    ("SINHALA ARCHAIC DIGIT NINE"                                                  . #x111E9)
    ("SINHALA ARCHAIC NUMBER TEN"                                                  . #x111EA)
    ("SINHALA ARCHAIC NUMBER TWENTY"                                               . #x111EB)
    ("SINHALA ARCHAIC NUMBER THIRTY"                                               . #x111EC)
    ("SINHALA ARCHAIC NUMBER FORTY"                                                . #x111ED)
    ("SINHALA ARCHAIC NUMBER FIFTY"                                                . #x111EE)
    ("SINHALA ARCHAIC NUMBER SIXTY"                                                . #x111EF)
    ("SINHALA ARCHAIC NUMBER SEVENTY"                                              . #x111F0)
    ("SINHALA ARCHAIC NUMBER EIGHTY"                                               . #x111F1)
    ("SINHALA ARCHAIC NUMBER NINETY"                                               . #x111F2)
    ("SINHALA ARCHAIC NUMBER ONE HUNDRED"                                          . #x111F3)
    ("SINHALA ARCHAIC NUMBER ONE THOUSAND"                                         . #x111F4)
    ("KHOJKI LETTER A"                                                             . #x11200)
    ("KHOJKI LETTER AA"                                                            . #x11201)
    ("KHOJKI LETTER I"                                                             . #x11202)
    ("KHOJKI LETTER U"                                                             . #x11203)
    ("KHOJKI LETTER E"                                                             . #x11204)
    ("KHOJKI LETTER AI"                                                            . #x11205)
    ("KHOJKI LETTER O"                                                             . #x11206)
    ("KHOJKI LETTER AU"                                                            . #x11207)
    ("KHOJKI LETTER KA"                                                            . #x11208)
    ("KHOJKI LETTER KHA"                                                           . #x11209)
    ("KHOJKI LETTER GA"                                                            . #x1120A)
    ("KHOJKI LETTER GGA"                                                           . #x1120B)
    ("KHOJKI LETTER GHA"                                                           . #x1120C)
    ("KHOJKI LETTER NGA"                                                           . #x1120D)
    ("KHOJKI LETTER CA"                                                            . #x1120E)
    ("KHOJKI LETTER CHA"                                                           . #x1120F)
    ("KHOJKI LETTER JA"                                                            . #x11210)
    ("KHOJKI LETTER JJA"                                                           . #x11211)
    ("KHOJKI LETTER NYA"                                                           . #x11213)
    ("KHOJKI LETTER TTA"                                                           . #x11214)
    ("KHOJKI LETTER TTHA"                                                          . #x11215)
    ("KHOJKI LETTER DDA"                                                           . #x11216)
    ("KHOJKI LETTER DDHA"                                                          . #x11217)
    ("KHOJKI LETTER NNA"                                                           . #x11218)
    ("KHOJKI LETTER TA"                                                            . #x11219)
    ("KHOJKI LETTER THA"                                                           . #x1121A)
    ("KHOJKI LETTER DA"                                                            . #x1121B)
    ("KHOJKI LETTER DDDA"                                                          . #x1121C)
    ("KHOJKI LETTER DHA"                                                           . #x1121D)
    ("KHOJKI LETTER NA"                                                            . #x1121E)
    ("KHOJKI LETTER PA"                                                            . #x1121F)
    ("KHOJKI LETTER PHA"                                                           . #x11220)
    ("KHOJKI LETTER BA"                                                            . #x11221)
    ("KHOJKI LETTER BBA"                                                           . #x11222)
    ("KHOJKI LETTER BHA"                                                           . #x11223)
    ("KHOJKI LETTER MA"                                                            . #x11224)
    ("KHOJKI LETTER YA"                                                            . #x11225)
    ("KHOJKI LETTER RA"                                                            . #x11226)
    ("KHOJKI LETTER LA"                                                            . #x11227)
    ("KHOJKI LETTER VA"                                                            . #x11228)
    ("KHOJKI LETTER SA"                                                            . #x11229)
    ("KHOJKI LETTER HA"                                                            . #x1122A)
    ("KHOJKI LETTER LLA"                                                           . #x1122B)
    ("KHOJKI VOWEL SIGN AA"                                                        . #x1122C)
    ("KHOJKI VOWEL SIGN I"                                                         . #x1122D)
    ("KHOJKI VOWEL SIGN II"                                                        . #x1122E)
    ("KHOJKI VOWEL SIGN U"                                                         . #x1122F)
    ("KHOJKI VOWEL SIGN E"                                                         . #x11230)
    ("KHOJKI VOWEL SIGN AI"                                                        . #x11231)
    ("KHOJKI VOWEL SIGN O"                                                         . #x11232)
    ("KHOJKI VOWEL SIGN AU"                                                        . #x11233)
    ("KHOJKI SIGN ANUSVARA"                                                        . #x11234)
    ("KHOJKI SIGN VIRAMA"                                                          . #x11235)
    ("KHOJKI SIGN NUKTA"                                                           . #x11236)
    ("KHOJKI SIGN SHADDA"                                                          . #x11237)
    ("KHOJKI DANDA"                                                                . #x11238)
    ("KHOJKI DOUBLE DANDA"                                                         . #x11239)
    ("KHOJKI WORD SEPARATOR"                                                       . #x1123A)
    ("KHOJKI SECTION MARK"                                                         . #x1123B)
    ("KHOJKI DOUBLE SECTION MARK"                                                  . #x1123C)
    ("KHOJKI ABBREVIATION SIGN"                                                    . #x1123D)
    ("KHUDAWADI LETTER A"                                                          . #x112B0)
    ("KHUDAWADI LETTER AA"                                                         . #x112B1)
    ("KHUDAWADI LETTER I"                                                          . #x112B2)
    ("KHUDAWADI LETTER II"                                                         . #x112B3)
    ("KHUDAWADI LETTER U"                                                          . #x112B4)
    ("KHUDAWADI LETTER UU"                                                         . #x112B5)
    ("KHUDAWADI LETTER E"                                                          . #x112B6)
    ("KHUDAWADI LETTER AI"                                                         . #x112B7)
    ("KHUDAWADI LETTER O"                                                          . #x112B8)
    ("KHUDAWADI LETTER AU"                                                         . #x112B9)
    ("KHUDAWADI LETTER KA"                                                         . #x112BA)
    ("KHUDAWADI LETTER KHA"                                                        . #x112BB)
    ("KHUDAWADI LETTER GA"                                                         . #x112BC)
    ("KHUDAWADI LETTER GGA"                                                        . #x112BD)
    ("KHUDAWADI LETTER GHA"                                                        . #x112BE)
    ("KHUDAWADI LETTER NGA"                                                        . #x112BF)
    ("KHUDAWADI LETTER CA"                                                         . #x112C0)
    ("KHUDAWADI LETTER CHA"                                                        . #x112C1)
    ("KHUDAWADI LETTER JA"                                                         . #x112C2)
    ("KHUDAWADI LETTER JJA"                                                        . #x112C3)
    ("KHUDAWADI LETTER JHA"                                                        . #x112C4)
    ("KHUDAWADI LETTER NYA"                                                        . #x112C5)
    ("KHUDAWADI LETTER TTA"                                                        . #x112C6)
    ("KHUDAWADI LETTER TTHA"                                                       . #x112C7)
    ("KHUDAWADI LETTER DDA"                                                        . #x112C8)
    ("KHUDAWADI LETTER DDDA"                                                       . #x112C9)
    ("KHUDAWADI LETTER RRA"                                                        . #x112CA)
    ("KHUDAWADI LETTER DDHA"                                                       . #x112CB)
    ("KHUDAWADI LETTER NNA"                                                        . #x112CC)
    ("KHUDAWADI LETTER TA"                                                         . #x112CD)
    ("KHUDAWADI LETTER THA"                                                        . #x112CE)
    ("KHUDAWADI LETTER DA"                                                         . #x112CF)
    ("KHUDAWADI LETTER DHA"                                                        . #x112D0)
    ("KHUDAWADI LETTER NA"                                                         . #x112D1)
    ("KHUDAWADI LETTER PA"                                                         . #x112D2)
    ("KHUDAWADI LETTER PHA"                                                        . #x112D3)
    ("KHUDAWADI LETTER BA"                                                         . #x112D4)
    ("KHUDAWADI LETTER BBA"                                                        . #x112D5)
    ("KHUDAWADI LETTER BHA"                                                        . #x112D6)
    ("KHUDAWADI LETTER MA"                                                         . #x112D7)
    ("KHUDAWADI LETTER YA"                                                         . #x112D8)
    ("KHUDAWADI LETTER RA"                                                         . #x112D9)
    ("KHUDAWADI LETTER LA"                                                         . #x112DA)
    ("KHUDAWADI LETTER VA"                                                         . #x112DB)
    ("KHUDAWADI LETTER SHA"                                                        . #x112DC)
    ("KHUDAWADI LETTER SA"                                                         . #x112DD)
    ("KHUDAWADI LETTER HA"                                                         . #x112DE)
    ("KHUDAWADI SIGN ANUSVARA"                                                     . #x112DF)
    ("KHUDAWADI VOWEL SIGN AA"                                                     . #x112E0)
    ("KHUDAWADI VOWEL SIGN I"                                                      . #x112E1)
    ("KHUDAWADI VOWEL SIGN II"                                                     . #x112E2)
    ("KHUDAWADI VOWEL SIGN U"                                                      . #x112E3)
    ("KHUDAWADI VOWEL SIGN UU"                                                     . #x112E4)
    ("KHUDAWADI VOWEL SIGN E"                                                      . #x112E5)
    ("KHUDAWADI VOWEL SIGN AI"                                                     . #x112E6)
    ("KHUDAWADI VOWEL SIGN O"                                                      . #x112E7)
    ("KHUDAWADI VOWEL SIGN AU"                                                     . #x112E8)
    ("KHUDAWADI SIGN NUKTA"                                                        . #x112E9)
    ("KHUDAWADI SIGN VIRAMA"                                                       . #x112EA)
    ("KHUDAWADI DIGIT ZERO"                                                        . #x112F0)
    ("KHUDAWADI DIGIT ONE"                                                         . #x112F1)
    ("KHUDAWADI DIGIT TWO"                                                         . #x112F2)
    ("KHUDAWADI DIGIT THREE"                                                       . #x112F3)
    ("KHUDAWADI DIGIT FOUR"                                                        . #x112F4)
    ("KHUDAWADI DIGIT FIVE"                                                        . #x112F5)
    ("KHUDAWADI DIGIT SIX"                                                         . #x112F6)
    ("KHUDAWADI DIGIT SEVEN"                                                       . #x112F7)
    ("KHUDAWADI DIGIT EIGHT"                                                       . #x112F8)
    ("KHUDAWADI DIGIT NINE"                                                        . #x112F9)
    ("GRANTHA SIGN CANDRABINDU"                                                    . #x11301)
    ("GRANTHA SIGN ANUSVARA"                                                       . #x11302)
    ("GRANTHA SIGN VISARGA"                                                        . #x11303)
    ("GRANTHA LETTER A"                                                            . #x11305)
    ("GRANTHA LETTER AA"                                                           . #x11306)
    ("GRANTHA LETTER I"                                                            . #x11307)
    ("GRANTHA LETTER II"                                                           . #x11308)
    ("GRANTHA LETTER U"                                                            . #x11309)
    ("GRANTHA LETTER UU"                                                           . #x1130A)
    ("GRANTHA LETTER VOCALIC R"                                                    . #x1130B)
    ("GRANTHA LETTER VOCALIC L"                                                    . #x1130C)
    ("GRANTHA LETTER EE"                                                           . #x1130F)
    ("GRANTHA LETTER AI"                                                           . #x11310)
    ("GRANTHA LETTER OO"                                                           . #x11313)
    ("GRANTHA LETTER AU"                                                           . #x11314)
    ("GRANTHA LETTER KA"                                                           . #x11315)
    ("GRANTHA LETTER KHA"                                                          . #x11316)
    ("GRANTHA LETTER GA"                                                           . #x11317)
    ("GRANTHA LETTER GHA"                                                          . #x11318)
    ("GRANTHA LETTER NGA"                                                          . #x11319)
    ("GRANTHA LETTER CA"                                                           . #x1131A)
    ("GRANTHA LETTER CHA"                                                          . #x1131B)
    ("GRANTHA LETTER JA"                                                           . #x1131C)
    ("GRANTHA LETTER JHA"                                                          . #x1131D)
    ("GRANTHA LETTER NYA"                                                          . #x1131E)
    ("GRANTHA LETTER TTA"                                                          . #x1131F)
    ("GRANTHA LETTER TTHA"                                                         . #x11320)
    ("GRANTHA LETTER DDA"                                                          . #x11321)
    ("GRANTHA LETTER DDHA"                                                         . #x11322)
    ("GRANTHA LETTER NNA"                                                          . #x11323)
    ("GRANTHA LETTER TA"                                                           . #x11324)
    ("GRANTHA LETTER THA"                                                          . #x11325)
    ("GRANTHA LETTER DA"                                                           . #x11326)
    ("GRANTHA LETTER DHA"                                                          . #x11327)
    ("GRANTHA LETTER NA"                                                           . #x11328)
    ("GRANTHA LETTER PA"                                                           . #x1132A)
    ("GRANTHA LETTER PHA"                                                          . #x1132B)
    ("GRANTHA LETTER BA"                                                           . #x1132C)
    ("GRANTHA LETTER BHA"                                                          . #x1132D)
    ("GRANTHA LETTER MA"                                                           . #x1132E)
    ("GRANTHA LETTER YA"                                                           . #x1132F)
    ("GRANTHA LETTER RA"                                                           . #x11330)
    ("GRANTHA LETTER LA"                                                           . #x11332)
    ("GRANTHA LETTER LLA"                                                          . #x11333)
    ("GRANTHA LETTER VA"                                                           . #x11335)
    ("GRANTHA LETTER SHA"                                                          . #x11336)
    ("GRANTHA LETTER SSA"                                                          . #x11337)
    ("GRANTHA LETTER SA"                                                           . #x11338)
    ("GRANTHA LETTER HA"                                                           . #x11339)
    ("GRANTHA SIGN NUKTA"                                                          . #x1133C)
    ("GRANTHA SIGN AVAGRAHA"                                                       . #x1133D)
    ("GRANTHA VOWEL SIGN AA"                                                       . #x1133E)
    ("GRANTHA VOWEL SIGN I"                                                        . #x1133F)
    ("GRANTHA VOWEL SIGN II"                                                       . #x11340)
    ("GRANTHA VOWEL SIGN U"                                                        . #x11341)
    ("GRANTHA VOWEL SIGN UU"                                                       . #x11342)
    ("GRANTHA VOWEL SIGN VOCALIC R"                                                . #x11343)
    ("GRANTHA VOWEL SIGN VOCALIC RR"                                               . #x11344)
    ("GRANTHA VOWEL SIGN EE"                                                       . #x11347)
    ("GRANTHA VOWEL SIGN AI"                                                       . #x11348)
    ("GRANTHA VOWEL SIGN OO"                                                       . #x1134B)
    ("GRANTHA VOWEL SIGN AU"                                                       . #x1134C)
    ("GRANTHA SIGN VIRAMA"                                                         . #x1134D)
    ("GRANTHA AU LENGTH MARK"                                                      . #x11357)
    ("GRANTHA SIGN PLUTA"                                                          . #x1135D)
    ("GRANTHA LETTER VEDIC ANUSVARA"                                               . #x1135E)
    ("GRANTHA LETTER VEDIC DOUBLE ANUSVARA"                                        . #x1135F)
    ("GRANTHA LETTER VOCALIC RR"                                                   . #x11360)
    ("GRANTHA LETTER VOCALIC LL"                                                   . #x11361)
    ("GRANTHA VOWEL SIGN VOCALIC L"                                                . #x11362)
    ("GRANTHA VOWEL SIGN VOCALIC LL"                                               . #x11363)
    ("COMBINING GRANTHA DIGIT ZERO"                                                . #x11366)
    ("COMBINING GRANTHA DIGIT ONE"                                                 . #x11367)
    ("COMBINING GRANTHA DIGIT TWO"                                                 . #x11368)
    ("COMBINING GRANTHA DIGIT THREE"                                               . #x11369)
    ("COMBINING GRANTHA DIGIT FOUR"                                                . #x1136A)
    ("COMBINING GRANTHA DIGIT FIVE"                                                . #x1136B)
    ("COMBINING GRANTHA DIGIT SIX"                                                 . #x1136C)
    ("COMBINING GRANTHA LETTER A"                                                  . #x11370)
    ("COMBINING GRANTHA LETTER KA"                                                 . #x11371)
    ("COMBINING GRANTHA LETTER NA"                                                 . #x11372)
    ("COMBINING GRANTHA LETTER VI"                                                 . #x11373)
    ("COMBINING GRANTHA LETTER PA"                                                 . #x11374)
    ("TIRHUTA ANJI"                                                                . #x11480)
    ("TIRHUTA LETTER A"                                                            . #x11481)
    ("TIRHUTA LETTER AA"                                                           . #x11482)
    ("TIRHUTA LETTER I"                                                            . #x11483)
    ("TIRHUTA LETTER II"                                                           . #x11484)
    ("TIRHUTA LETTER U"                                                            . #x11485)
    ("TIRHUTA LETTER UU"                                                           . #x11486)
    ("TIRHUTA LETTER VOCALIC R"                                                    . #x11487)
    ("TIRHUTA LETTER VOCALIC RR"                                                   . #x11488)
    ("TIRHUTA LETTER VOCALIC L"                                                    . #x11489)
    ("TIRHUTA LETTER VOCALIC LL"                                                   . #x1148A)
    ("TIRHUTA LETTER E"                                                            . #x1148B)
    ("TIRHUTA LETTER AI"                                                           . #x1148C)
    ("TIRHUTA LETTER O"                                                            . #x1148D)
    ("TIRHUTA LETTER AU"                                                           . #x1148E)
    ("TIRHUTA LETTER KA"                                                           . #x1148F)
    ("TIRHUTA LETTER KHA"                                                          . #x11490)
    ("TIRHUTA LETTER GA"                                                           . #x11491)
    ("TIRHUTA LETTER GHA"                                                          . #x11492)
    ("TIRHUTA LETTER NGA"                                                          . #x11493)
    ("TIRHUTA LETTER CA"                                                           . #x11494)
    ("TIRHUTA LETTER CHA"                                                          . #x11495)
    ("TIRHUTA LETTER JA"                                                           . #x11496)
    ("TIRHUTA LETTER JHA"                                                          . #x11497)
    ("TIRHUTA LETTER NYA"                                                          . #x11498)
    ("TIRHUTA LETTER TTA"                                                          . #x11499)
    ("TIRHUTA LETTER TTHA"                                                         . #x1149A)
    ("TIRHUTA LETTER DDA"                                                          . #x1149B)
    ("TIRHUTA LETTER DDHA"                                                         . #x1149C)
    ("TIRHUTA LETTER NNA"                                                          . #x1149D)
    ("TIRHUTA LETTER TA"                                                           . #x1149E)
    ("TIRHUTA LETTER THA"                                                          . #x1149F)
    ("TIRHUTA LETTER DA"                                                           . #x114A0)
    ("TIRHUTA LETTER DHA"                                                          . #x114A1)
    ("TIRHUTA LETTER NA"                                                           . #x114A2)
    ("TIRHUTA LETTER PA"                                                           . #x114A3)
    ("TIRHUTA LETTER PHA"                                                          . #x114A4)
    ("TIRHUTA LETTER BA"                                                           . #x114A5)
    ("TIRHUTA LETTER BHA"                                                          . #x114A6)
    ("TIRHUTA LETTER MA"                                                           . #x114A7)
    ("TIRHUTA LETTER YA"                                                           . #x114A8)
    ("TIRHUTA LETTER RA"                                                           . #x114A9)
    ("TIRHUTA LETTER LA"                                                           . #x114AA)
    ("TIRHUTA LETTER VA"                                                           . #x114AB)
    ("TIRHUTA LETTER SHA"                                                          . #x114AC)
    ("TIRHUTA LETTER SSA"                                                          . #x114AD)
    ("TIRHUTA LETTER SA"                                                           . #x114AE)
    ("TIRHUTA LETTER HA"                                                           . #x114AF)
    ("TIRHUTA VOWEL SIGN AA"                                                       . #x114B0)
    ("TIRHUTA VOWEL SIGN I"                                                        . #x114B1)
    ("TIRHUTA VOWEL SIGN II"                                                       . #x114B2)
    ("TIRHUTA VOWEL SIGN U"                                                        . #x114B3)
    ("TIRHUTA VOWEL SIGN UU"                                                       . #x114B4)
    ("TIRHUTA VOWEL SIGN VOCALIC R"                                                . #x114B5)
    ("TIRHUTA VOWEL SIGN VOCALIC RR"                                               . #x114B6)
    ("TIRHUTA VOWEL SIGN VOCALIC L"                                                . #x114B7)
    ("TIRHUTA VOWEL SIGN VOCALIC LL"                                               . #x114B8)
    ("TIRHUTA VOWEL SIGN E"                                                        . #x114B9)
    ("TIRHUTA VOWEL SIGN SHORT E"                                                  . #x114BA)
    ("TIRHUTA VOWEL SIGN AI"                                                       . #x114BB)
    ("TIRHUTA VOWEL SIGN O"                                                        . #x114BC)
    ("TIRHUTA VOWEL SIGN SHORT O"                                                  . #x114BD)
    ("TIRHUTA VOWEL SIGN AU"                                                       . #x114BE)
    ("TIRHUTA SIGN CANDRABINDU"                                                    . #x114BF)
    ("TIRHUTA SIGN ANUSVARA"                                                       . #x114C0)
    ("TIRHUTA SIGN VISARGA"                                                        . #x114C1)
    ("TIRHUTA SIGN VIRAMA"                                                         . #x114C2)
    ("TIRHUTA SIGN NUKTA"                                                          . #x114C3)
    ("TIRHUTA SIGN AVAGRAHA"                                                       . #x114C4)
    ("TIRHUTA GVANG"                                                               . #x114C5)
    ("TIRHUTA ABBREVIATION SIGN"                                                   . #x114C6)
    ("TIRHUTA OM"                                                                  . #x114C7)
    ("TIRHUTA DIGIT ZERO"                                                          . #x114D0)
    ("TIRHUTA DIGIT ONE"                                                           . #x114D1)
    ("TIRHUTA DIGIT TWO"                                                           . #x114D2)
    ("TIRHUTA DIGIT THREE"                                                         . #x114D3)
    ("TIRHUTA DIGIT FOUR"                                                          . #x114D4)
    ("TIRHUTA DIGIT FIVE"                                                          . #x114D5)
    ("TIRHUTA DIGIT SIX"                                                           . #x114D6)
    ("TIRHUTA DIGIT SEVEN"                                                         . #x114D7)
    ("TIRHUTA DIGIT EIGHT"                                                         . #x114D8)
    ("TIRHUTA DIGIT NINE"                                                          . #x114D9)
    ("SIDDHAM LETTER A"                                                            . #x11580)
    ("SIDDHAM LETTER AA"                                                           . #x11581)
    ("SIDDHAM LETTER I"                                                            . #x11582)
    ("SIDDHAM LETTER II"                                                           . #x11583)
    ("SIDDHAM LETTER U"                                                            . #x11584)
    ("SIDDHAM LETTER UU"                                                           . #x11585)
    ("SIDDHAM LETTER VOCALIC R"                                                    . #x11586)
    ("SIDDHAM LETTER VOCALIC RR"                                                   . #x11587)
    ("SIDDHAM LETTER VOCALIC L"                                                    . #x11588)
    ("SIDDHAM LETTER VOCALIC LL"                                                   . #x11589)
    ("SIDDHAM LETTER E"                                                            . #x1158A)
    ("SIDDHAM LETTER AI"                                                           . #x1158B)
    ("SIDDHAM LETTER O"                                                            . #x1158C)
    ("SIDDHAM LETTER AU"                                                           . #x1158D)
    ("SIDDHAM LETTER KA"                                                           . #x1158E)
    ("SIDDHAM LETTER KHA"                                                          . #x1158F)
    ("SIDDHAM LETTER GA"                                                           . #x11590)
    ("SIDDHAM LETTER GHA"                                                          . #x11591)
    ("SIDDHAM LETTER NGA"                                                          . #x11592)
    ("SIDDHAM LETTER CA"                                                           . #x11593)
    ("SIDDHAM LETTER CHA"                                                          . #x11594)
    ("SIDDHAM LETTER JA"                                                           . #x11595)
    ("SIDDHAM LETTER JHA"                                                          . #x11596)
    ("SIDDHAM LETTER NYA"                                                          . #x11597)
    ("SIDDHAM LETTER TTA"                                                          . #x11598)
    ("SIDDHAM LETTER TTHA"                                                         . #x11599)
    ("SIDDHAM LETTER DDA"                                                          . #x1159A)
    ("SIDDHAM LETTER DDHA"                                                         . #x1159B)
    ("SIDDHAM LETTER NNA"                                                          . #x1159C)
    ("SIDDHAM LETTER TA"                                                           . #x1159D)
    ("SIDDHAM LETTER THA"                                                          . #x1159E)
    ("SIDDHAM LETTER DA"                                                           . #x1159F)
    ("SIDDHAM LETTER DHA"                                                          . #x115A0)
    ("SIDDHAM LETTER NA"                                                           . #x115A1)
    ("SIDDHAM LETTER PA"                                                           . #x115A2)
    ("SIDDHAM LETTER PHA"                                                          . #x115A3)
    ("SIDDHAM LETTER BA"                                                           . #x115A4)
    ("SIDDHAM LETTER BHA"                                                          . #x115A5)
    ("SIDDHAM LETTER MA"                                                           . #x115A6)
    ("SIDDHAM LETTER YA"                                                           . #x115A7)
    ("SIDDHAM LETTER RA"                                                           . #x115A8)
    ("SIDDHAM LETTER LA"                                                           . #x115A9)
    ("SIDDHAM LETTER VA"                                                           . #x115AA)
    ("SIDDHAM LETTER SHA"                                                          . #x115AB)
    ("SIDDHAM LETTER SSA"                                                          . #x115AC)
    ("SIDDHAM LETTER SA"                                                           . #x115AD)
    ("SIDDHAM LETTER HA"                                                           . #x115AE)
    ("SIDDHAM VOWEL SIGN AA"                                                       . #x115AF)
    ("SIDDHAM VOWEL SIGN I"                                                        . #x115B0)
    ("SIDDHAM VOWEL SIGN II"                                                       . #x115B1)
    ("SIDDHAM VOWEL SIGN U"                                                        . #x115B2)
    ("SIDDHAM VOWEL SIGN UU"                                                       . #x115B3)
    ("SIDDHAM VOWEL SIGN VOCALIC R"                                                . #x115B4)
    ("SIDDHAM VOWEL SIGN VOCALIC RR"                                               . #x115B5)
    ("SIDDHAM VOWEL SIGN E"                                                        . #x115B8)
    ("SIDDHAM VOWEL SIGN AI"                                                       . #x115B9)
    ("SIDDHAM VOWEL SIGN O"                                                        . #x115BA)
    ("SIDDHAM VOWEL SIGN AU"                                                       . #x115BB)
    ("SIDDHAM SIGN CANDRABINDU"                                                    . #x115BC)
    ("SIDDHAM SIGN ANUSVARA"                                                       . #x115BD)
    ("SIDDHAM SIGN VISARGA"                                                        . #x115BE)
    ("SIDDHAM SIGN VIRAMA"                                                         . #x115BF)
    ("SIDDHAM SIGN NUKTA"                                                          . #x115C0)
    ("SIDDHAM SIGN SIDDHAM"                                                        . #x115C1)
    ("SIDDHAM DANDA"                                                               . #x115C2)
    ("SIDDHAM DOUBLE DANDA"                                                        . #x115C3)
    ("SIDDHAM SEPARATOR DOT"                                                       . #x115C4)
    ("SIDDHAM SEPARATOR BAR"                                                       . #x115C5)
    ("SIDDHAM REPETITION MARK-1"                                                   . #x115C6)
    ("SIDDHAM REPETITION MARK-2"                                                   . #x115C7)
    ("SIDDHAM REPETITION MARK-3"                                                   . #x115C8)
    ("SIDDHAM END OF TEXT MARK"                                                    . #x115C9)
    ("MODI LETTER A"                                                               . #x11600)
    ("MODI LETTER AA"                                                              . #x11601)
    ("MODI LETTER I"                                                               . #x11602)
    ("MODI LETTER II"                                                              . #x11603)
    ("MODI LETTER U"                                                               . #x11604)
    ("MODI LETTER UU"                                                              . #x11605)
    ("MODI LETTER VOCALIC R"                                                       . #x11606)
    ("MODI LETTER VOCALIC RR"                                                      . #x11607)
    ("MODI LETTER VOCALIC L"                                                       . #x11608)
    ("MODI LETTER VOCALIC LL"                                                      . #x11609)
    ("MODI LETTER E"                                                               . #x1160A)
    ("MODI LETTER AI"                                                              . #x1160B)
    ("MODI LETTER O"                                                               . #x1160C)
    ("MODI LETTER AU"                                                              . #x1160D)
    ("MODI LETTER KA"                                                              . #x1160E)
    ("MODI LETTER KHA"                                                             . #x1160F)
    ("MODI LETTER GA"                                                              . #x11610)
    ("MODI LETTER GHA"                                                             . #x11611)
    ("MODI LETTER NGA"                                                             . #x11612)
    ("MODI LETTER CA"                                                              . #x11613)
    ("MODI LETTER CHA"                                                             . #x11614)
    ("MODI LETTER JA"                                                              . #x11615)
    ("MODI LETTER JHA"                                                             . #x11616)
    ("MODI LETTER NYA"                                                             . #x11617)
    ("MODI LETTER TTA"                                                             . #x11618)
    ("MODI LETTER TTHA"                                                            . #x11619)
    ("MODI LETTER DDA"                                                             . #x1161A)
    ("MODI LETTER DDHA"                                                            . #x1161B)
    ("MODI LETTER NNA"                                                             . #x1161C)
    ("MODI LETTER TA"                                                              . #x1161D)
    ("MODI LETTER THA"                                                             . #x1161E)
    ("MODI LETTER DA"                                                              . #x1161F)
    ("MODI LETTER DHA"                                                             . #x11620)
    ("MODI LETTER NA"                                                              . #x11621)
    ("MODI LETTER PA"                                                              . #x11622)
    ("MODI LETTER PHA"                                                             . #x11623)
    ("MODI LETTER BA"                                                              . #x11624)
    ("MODI LETTER BHA"                                                             . #x11625)
    ("MODI LETTER MA"                                                              . #x11626)
    ("MODI LETTER YA"                                                              . #x11627)
    ("MODI LETTER RA"                                                              . #x11628)
    ("MODI LETTER LA"                                                              . #x11629)
    ("MODI LETTER VA"                                                              . #x1162A)
    ("MODI LETTER SHA"                                                             . #x1162B)
    ("MODI LETTER SSA"                                                             . #x1162C)
    ("MODI LETTER SA"                                                              . #x1162D)
    ("MODI LETTER HA"                                                              . #x1162E)
    ("MODI LETTER LLA"                                                             . #x1162F)
    ("MODI VOWEL SIGN AA"                                                          . #x11630)
    ("MODI VOWEL SIGN I"                                                           . #x11631)
    ("MODI VOWEL SIGN II"                                                          . #x11632)
    ("MODI VOWEL SIGN U"                                                           . #x11633)
    ("MODI VOWEL SIGN UU"                                                          . #x11634)
    ("MODI VOWEL SIGN VOCALIC R"                                                   . #x11635)
    ("MODI VOWEL SIGN VOCALIC RR"                                                  . #x11636)
    ("MODI VOWEL SIGN VOCALIC L"                                                   . #x11637)
    ("MODI VOWEL SIGN VOCALIC LL"                                                  . #x11638)
    ("MODI VOWEL SIGN E"                                                           . #x11639)
    ("MODI VOWEL SIGN AI"                                                          . #x1163A)
    ("MODI VOWEL SIGN O"                                                           . #x1163B)
    ("MODI VOWEL SIGN AU"                                                          . #x1163C)
    ("MODI SIGN ANUSVARA"                                                          . #x1163D)
    ("MODI SIGN VISARGA"                                                           . #x1163E)
    ("MODI SIGN VIRAMA"                                                            . #x1163F)
    ("MODI SIGN ARDHACANDRA"                                                       . #x11640)
    ("MODI DANDA"                                                                  . #x11641)
    ("MODI DOUBLE DANDA"                                                           . #x11642)
    ("MODI ABBREVIATION SIGN"                                                      . #x11643)
    ("MODI SIGN HUVA"                                                              . #x11644)
    ("MODI DIGIT ZERO"                                                             . #x11650)
    ("MODI DIGIT ONE"                                                              . #x11651)
    ("MODI DIGIT TWO"                                                              . #x11652)
    ("MODI DIGIT THREE"                                                            . #x11653)
    ("MODI DIGIT FOUR"                                                             . #x11654)
    ("MODI DIGIT FIVE"                                                             . #x11655)
    ("MODI DIGIT SIX"                                                              . #x11656)
    ("MODI DIGIT SEVEN"                                                            . #x11657)
    ("MODI DIGIT EIGHT"                                                            . #x11658)
    ("MODI DIGIT NINE"                                                             . #x11659)
    ("WARANG CITI CAPITAL LETTER NGAA"                                             . #x118A0)
    ("WARANG CITI CAPITAL LETTER A"                                                . #x118A1)
    ("WARANG CITI CAPITAL LETTER WI"                                               . #x118A2)
    ("WARANG CITI CAPITAL LETTER YU"                                               . #x118A3)
    ("WARANG CITI CAPITAL LETTER YA"                                               . #x118A4)
    ("WARANG CITI CAPITAL LETTER YO"                                               . #x118A5)
    ("WARANG CITI CAPITAL LETTER II"                                               . #x118A6)
    ("WARANG CITI CAPITAL LETTER UU"                                               . #x118A7)
    ("WARANG CITI CAPITAL LETTER E"                                                . #x118A8)
    ("WARANG CITI CAPITAL LETTER O"                                                . #x118A9)
    ("WARANG CITI CAPITAL LETTER ANG"                                              . #x118AA)
    ("WARANG CITI CAPITAL LETTER GA"                                               . #x118AB)
    ("WARANG CITI CAPITAL LETTER KO"                                               . #x118AC)
    ("WARANG CITI CAPITAL LETTER ENY"                                              . #x118AD)
    ("WARANG CITI CAPITAL LETTER YUJ"                                              . #x118AE)
    ("WARANG CITI CAPITAL LETTER UC"                                               . #x118AF)
    ("WARANG CITI CAPITAL LETTER ENN"                                              . #x118B0)
    ("WARANG CITI CAPITAL LETTER ODD"                                              . #x118B1)
    ("WARANG CITI CAPITAL LETTER TTE"                                              . #x118B2)
    ("WARANG CITI CAPITAL LETTER NUNG"                                             . #x118B3)
    ("WARANG CITI CAPITAL LETTER DA"                                               . #x118B4)
    ("WARANG CITI CAPITAL LETTER AT"                                               . #x118B5)
    ("WARANG CITI CAPITAL LETTER AM"                                               . #x118B6)
    ("WARANG CITI CAPITAL LETTER BU"                                               . #x118B7)
    ("WARANG CITI CAPITAL LETTER PU"                                               . #x118B8)
    ("WARANG CITI CAPITAL LETTER HIYO"                                             . #x118B9)
    ("WARANG CITI CAPITAL LETTER HOLO"                                             . #x118BA)
    ("WARANG CITI CAPITAL LETTER HORR"                                             . #x118BB)
    ("WARANG CITI CAPITAL LETTER HAR"                                              . #x118BC)
    ("WARANG CITI CAPITAL LETTER SSUU"                                             . #x118BD)
    ("WARANG CITI CAPITAL LETTER SII"                                              . #x118BE)
    ("WARANG CITI CAPITAL LETTER VIYO"                                             . #x118BF)
    ("WARANG CITI SMALL LETTER NGAA"                                               . #x118C0)
    ("WARANG CITI SMALL LETTER A"                                                  . #x118C1)
    ("WARANG CITI SMALL LETTER WI"                                                 . #x118C2)
    ("WARANG CITI SMALL LETTER YU"                                                 . #x118C3)
    ("WARANG CITI SMALL LETTER YA"                                                 . #x118C4)
    ("WARANG CITI SMALL LETTER YO"                                                 . #x118C5)
    ("WARANG CITI SMALL LETTER II"                                                 . #x118C6)
    ("WARANG CITI SMALL LETTER UU"                                                 . #x118C7)
    ("WARANG CITI SMALL LETTER E"                                                  . #x118C8)
    ("WARANG CITI SMALL LETTER O"                                                  . #x118C9)
    ("WARANG CITI SMALL LETTER ANG"                                                . #x118CA)
    ("WARANG CITI SMALL LETTER GA"                                                 . #x118CB)
    ("WARANG CITI SMALL LETTER KO"                                                 . #x118CC)
    ("WARANG CITI SMALL LETTER ENY"                                                . #x118CD)
    ("WARANG CITI SMALL LETTER YUJ"                                                . #x118CE)
    ("WARANG CITI SMALL LETTER UC"                                                 . #x118CF)
    ("WARANG CITI SMALL LETTER ENN"                                                . #x118D0)
    ("WARANG CITI SMALL LETTER ODD"                                                . #x118D1)
    ("WARANG CITI SMALL LETTER TTE"                                                . #x118D2)
    ("WARANG CITI SMALL LETTER NUNG"                                               . #x118D3)
    ("WARANG CITI SMALL LETTER DA"                                                 . #x118D4)
    ("WARANG CITI SMALL LETTER AT"                                                 . #x118D5)
    ("WARANG CITI SMALL LETTER AM"                                                 . #x118D6)
    ("WARANG CITI SMALL LETTER BU"                                                 . #x118D7)
    ("WARANG CITI SMALL LETTER PU"                                                 . #x118D8)
    ("WARANG CITI SMALL LETTER HIYO"                                               . #x118D9)
    ("WARANG CITI SMALL LETTER HOLO"                                               . #x118DA)
    ("WARANG CITI SMALL LETTER HORR"                                               . #x118DB)
    ("WARANG CITI SMALL LETTER HAR"                                                . #x118DC)
    ("WARANG CITI SMALL LETTER SSUU"                                               . #x118DD)
    ("WARANG CITI SMALL LETTER SII"                                                . #x118DE)
    ("WARANG CITI SMALL LETTER VIYO"                                               . #x118DF)
    ("WARANG CITI DIGIT ZERO"                                                      . #x118E0)
    ("WARANG CITI DIGIT ONE"                                                       . #x118E1)
    ("WARANG CITI DIGIT TWO"                                                       . #x118E2)
    ("WARANG CITI DIGIT THREE"                                                     . #x118E3)
    ("WARANG CITI DIGIT FOUR"                                                      . #x118E4)
    ("WARANG CITI DIGIT FIVE"                                                      . #x118E5)
    ("WARANG CITI DIGIT SIX"                                                       . #x118E6)
    ("WARANG CITI DIGIT SEVEN"                                                     . #x118E7)
    ("WARANG CITI DIGIT EIGHT"                                                     . #x118E8)
    ("WARANG CITI DIGIT NINE"                                                      . #x118E9)
    ("WARANG CITI NUMBER TEN"                                                      . #x118EA)
    ("WARANG CITI NUMBER TWENTY"                                                   . #x118EB)
    ("WARANG CITI NUMBER THIRTY"                                                   . #x118EC)
    ("WARANG CITI NUMBER FORTY"                                                    . #x118ED)
    ("WARANG CITI NUMBER FIFTY"                                                    . #x118EE)
    ("WARANG CITI NUMBER SIXTY"                                                    . #x118EF)
    ("WARANG CITI NUMBER SEVENTY"                                                  . #x118F0)
    ("WARANG CITI NUMBER EIGHTY"                                                   . #x118F1)
    ("WARANG CITI NUMBER NINETY"                                                   . #x118F2)
    ("WARANG CITI OM"                                                              . #x118FF)
    ("PAU CIN HAU LETTER PA"                                                       . #x11AC0)
    ("PAU CIN HAU LETTER KA"                                                       . #x11AC1)
    ("PAU CIN HAU LETTER LA"                                                       . #x11AC2)
    ("PAU CIN HAU LETTER MA"                                                       . #x11AC3)
    ("PAU CIN HAU LETTER DA"                                                       . #x11AC4)
    ("PAU CIN HAU LETTER ZA"                                                       . #x11AC5)
    ("PAU CIN HAU LETTER VA"                                                       . #x11AC6)
    ("PAU CIN HAU LETTER NGA"                                                      . #x11AC7)
    ("PAU CIN HAU LETTER HA"                                                       . #x11AC8)
    ("PAU CIN HAU LETTER GA"                                                       . #x11AC9)
    ("PAU CIN HAU LETTER KHA"                                                      . #x11ACA)
    ("PAU CIN HAU LETTER SA"                                                       . #x11ACB)
    ("PAU CIN HAU LETTER BA"                                                       . #x11ACC)
    ("PAU CIN HAU LETTER CA"                                                       . #x11ACD)
    ("PAU CIN HAU LETTER TA"                                                       . #x11ACE)
    ("PAU CIN HAU LETTER THA"                                                      . #x11ACF)
    ("PAU CIN HAU LETTER NA"                                                       . #x11AD0)
    ("PAU CIN HAU LETTER PHA"                                                      . #x11AD1)
    ("PAU CIN HAU LETTER RA"                                                       . #x11AD2)
    ("PAU CIN HAU LETTER FA"                                                       . #x11AD3)
    ("PAU CIN HAU LETTER CHA"                                                      . #x11AD4)
    ("PAU CIN HAU LETTER A"                                                        . #x11AD5)
    ("PAU CIN HAU LETTER E"                                                        . #x11AD6)
    ("PAU CIN HAU LETTER I"                                                        . #x11AD7)
    ("PAU CIN HAU LETTER O"                                                        . #x11AD8)
    ("PAU CIN HAU LETTER U"                                                        . #x11AD9)
    ("PAU CIN HAU LETTER UA"                                                       . #x11ADA)
    ("PAU CIN HAU LETTER IA"                                                       . #x11ADB)
    ("PAU CIN HAU LETTER FINAL P"                                                  . #x11ADC)
    ("PAU CIN HAU LETTER FINAL K"                                                  . #x11ADD)
    ("PAU CIN HAU LETTER FINAL T"                                                  . #x11ADE)
    ("PAU CIN HAU LETTER FINAL M"                                                  . #x11ADF)
    ("PAU CIN HAU LETTER FINAL N"                                                  . #x11AE0)
    ("PAU CIN HAU LETTER FINAL L"                                                  . #x11AE1)
    ("PAU CIN HAU LETTER FINAL W"                                                  . #x11AE2)
    ("PAU CIN HAU LETTER FINAL NG"                                                 . #x11AE3)
    ("PAU CIN HAU LETTER FINAL Y"                                                  . #x11AE4)
    ("PAU CIN HAU RISING TONE LONG"                                                . #x11AE5)
    ("PAU CIN HAU RISING TONE"                                                     . #x11AE6)
    ("PAU CIN HAU SANDHI GLOTTAL STOP"                                             . #x11AE7)
    ("PAU CIN HAU RISING TONE LONG FINAL"                                          . #x11AE8)
    ("PAU CIN HAU RISING TONE FINAL"                                               . #x11AE9)
    ("PAU CIN HAU SANDHI GLOTTAL STOP FINAL"                                       . #x11AEA)
    ("PAU CIN HAU SANDHI TONE LONG"                                                . #x11AEB)
    ("PAU CIN HAU SANDHI TONE"                                                     . #x11AEC)
    ("PAU CIN HAU SANDHI TONE LONG FINAL"                                          . #x11AED)
    ("PAU CIN HAU SANDHI TONE FINAL"                                               . #x11AEE)
    ("PAU CIN HAU MID-LEVEL TONE"                                                  . #x11AEF)
    ("PAU CIN HAU GLOTTAL STOP VARIANT"                                            . #x11AF0)
    ("PAU CIN HAU MID-LEVEL TONE LONG FINAL"                                       . #x11AF1)
    ("PAU CIN HAU MID-LEVEL TONE FINAL"                                            . #x11AF2)
    ("PAU CIN HAU LOW-FALLING TONE LONG"                                           . #x11AF3)
    ("PAU CIN HAU LOW-FALLING TONE"                                                . #x11AF4)
    ("PAU CIN HAU GLOTTAL STOP"                                                    . #x11AF5)
    ("PAU CIN HAU LOW-FALLING TONE LONG FINAL"                                     . #x11AF6)
    ("PAU CIN HAU LOW-FALLING TONE FINAL"                                          . #x11AF7)
    ("PAU CIN HAU GLOTTAL STOP FINAL"                                              . #x11AF8)
    ("CUNEIFORM SIGN KAP ELAMITE"                                                  . #x1236F)
    ("CUNEIFORM SIGN AB TIMES NUN"                                                 . #x12370)
    ("CUNEIFORM SIGN AB2 TIMES A"                                                  . #x12371)
    ("CUNEIFORM SIGN AMAR TIMES KUG"                                               . #x12372)
    ("CUNEIFORM SIGN DAG KISIM5 TIMES U2 PLUS MASH"                                . #x12373)
    ("CUNEIFORM SIGN DAG3"                                                         . #x12374)
    ("CUNEIFORM SIGN DISH PLUS SHU"                                                . #x12375)
    ("CUNEIFORM SIGN DUB TIMES SHE"                                                . #x12376)
    ("CUNEIFORM SIGN EZEN TIMES GUD"                                               . #x12377)
    ("CUNEIFORM SIGN EZEN TIMES SHE"                                               . #x12378)
    ("CUNEIFORM SIGN GA2 TIMES AN PLUS KAK PLUS A"                                 . #x12379)
    ("CUNEIFORM SIGN GA2 TIMES ASH2"                                               . #x1237A)
    ("CUNEIFORM SIGN GE22"                                                         . #x1237B)
    ("CUNEIFORM SIGN GIG"                                                          . #x1237C)
    ("CUNEIFORM SIGN HUSH"                                                         . #x1237D)
    ("CUNEIFORM SIGN KA TIMES ANSHE"                                               . #x1237E)
    ("CUNEIFORM SIGN KA TIMES ASH3"                                                . #x1237F)
    ("CUNEIFORM SIGN KA TIMES GISH"                                                . #x12380)
    ("CUNEIFORM SIGN KA TIMES GUD"                                                 . #x12381)
    ("CUNEIFORM SIGN KA TIMES HI TIMES ASH2"                                       . #x12382)
    ("CUNEIFORM SIGN KA TIMES LUM"                                                 . #x12383)
    ("CUNEIFORM SIGN KA TIMES PA"                                                  . #x12384)
    ("CUNEIFORM SIGN KA TIMES SHUL"                                                . #x12385)
    ("CUNEIFORM SIGN KA TIMES TU"                                                  . #x12386)
    ("CUNEIFORM SIGN KA TIMES UR2"                                                 . #x12387)
    ("CUNEIFORM SIGN LAGAB TIMES GI"                                               . #x12388)
    ("CUNEIFORM SIGN LU2 SHESHIG TIMES BAD"                                        . #x12389)
    ("CUNEIFORM SIGN LU2 TIMES ESH2 PLUS LAL"                                      . #x1238A)
    ("CUNEIFORM SIGN LU2 TIMES SHU"                                                . #x1238B)
    ("CUNEIFORM SIGN MESH"                                                         . #x1238C)
    ("CUNEIFORM SIGN MUSH3 TIMES ZA"                                               . #x1238D)
    ("CUNEIFORM SIGN NA4"                                                          . #x1238E)
    ("CUNEIFORM SIGN NIN"                                                          . #x1238F)
    ("CUNEIFORM SIGN NIN9"                                                         . #x12390)
    ("CUNEIFORM SIGN NINDA2 TIMES BAL"                                             . #x12391)
    ("CUNEIFORM SIGN NINDA2 TIMES GI"                                              . #x12392)
    ("CUNEIFORM SIGN NU11 ROTATED NINETY DEGREES"                                  . #x12393)
    ("CUNEIFORM SIGN PESH2 ASTERISK"                                               . #x12394)
    ("CUNEIFORM SIGN PIR2"                                                         . #x12395)
    ("CUNEIFORM SIGN SAG TIMES IGI GUNU"                                           . #x12396)
    ("CUNEIFORM SIGN TI2"                                                          . #x12397)
    ("CUNEIFORM SIGN UM TIMES ME"                                                  . #x12398)
    ("CUNEIFORM NUMERIC SIGN ONE QUARTER GUR"                                      . #x12463)
    ("CUNEIFORM NUMERIC SIGN ONE HALF GUR"                                         . #x12464)
    ("CUNEIFORM NUMERIC SIGN ELAMITE ONE THIRD"                                    . #x12465)
    ("CUNEIFORM NUMERIC SIGN ELAMITE TWO THIRDS"                                   . #x12466)
    ("CUNEIFORM NUMERIC SIGN ELAMITE FORTY"                                        . #x12467)
    ("CUNEIFORM NUMERIC SIGN ELAMITE FIFTY"                                        . #x12468)
    ("CUNEIFORM NUMERIC SIGN FOUR U VARIANT FORM"                                  . #x12469)
    ("CUNEIFORM NUMERIC SIGN FIVE U VARIANT FORM"                                  . #x1246A)
    ("CUNEIFORM NUMERIC SIGN SIX U VARIANT FORM"                                   . #x1246B)
    ("CUNEIFORM NUMERIC SIGN SEVEN U VARIANT FORM"                                 . #x1246C)
    ("CUNEIFORM NUMERIC SIGN EIGHT U VARIANT FORM"                                 . #x1246D)
    ("CUNEIFORM NUMERIC SIGN NINE U VARIANT FORM"                                  . #x1246E)
    ("CUNEIFORM PUNCTUATION SIGN DIAGONAL QUADCOLON"                               . #x12474)
    ("MRO LETTER TA"                                                               . #x16A40)
    ("MRO LETTER NGI"                                                              . #x16A41)
    ("MRO LETTER YO"                                                               . #x16A42)
    ("MRO LETTER MIM"                                                              . #x16A43)
    ("MRO LETTER BA"                                                               . #x16A44)
    ("MRO LETTER DA"                                                               . #x16A45)
    ("MRO LETTER A"                                                                . #x16A46)
    ("MRO LETTER PHI"                                                              . #x16A47)
    ("MRO LETTER KHAI"                                                             . #x16A48)
    ("MRO LETTER HAO"                                                              . #x16A49)
    ("MRO LETTER DAI"                                                              . #x16A4A)
    ("MRO LETTER CHU"                                                              . #x16A4B)
    ("MRO LETTER KEAAE"                                                            . #x16A4C)
    ("MRO LETTER OL"                                                               . #x16A4D)
    ("MRO LETTER MAEM"                                                             . #x16A4E)
    ("MRO LETTER NIN"                                                              . #x16A4F)
    ("MRO LETTER PA"                                                               . #x16A50)
    ("MRO LETTER OO"                                                               . #x16A51)
    ("MRO LETTER O"                                                                . #x16A52)
    ("MRO LETTER RO"                                                               . #x16A53)
    ("MRO LETTER SHI"                                                              . #x16A54)
    ("MRO LETTER THEA"                                                             . #x16A55)
    ("MRO LETTER EA"                                                               . #x16A56)
    ("MRO LETTER WA"                                                               . #x16A57)
    ("MRO LETTER E"                                                                . #x16A58)
    ("MRO LETTER KO"                                                               . #x16A59)
    ("MRO LETTER LAN"                                                              . #x16A5A)
    ("MRO LETTER LA"                                                               . #x16A5B)
    ("MRO LETTER HAI"                                                              . #x16A5C)
    ("MRO LETTER RI"                                                               . #x16A5D)
    ("MRO LETTER TEK"                                                              . #x16A5E)
    ("MRO DIGIT ZERO"                                                              . #x16A60)
    ("MRO DIGIT ONE"                                                               . #x16A61)
    ("MRO DIGIT TWO"                                                               . #x16A62)
    ("MRO DIGIT THREE"                                                             . #x16A63)
    ("MRO DIGIT FOUR"                                                              . #x16A64)
    ("MRO DIGIT FIVE"                                                              . #x16A65)
    ("MRO DIGIT SIX"                                                               . #x16A66)
    ("MRO DIGIT SEVEN"                                                             . #x16A67)
    ("MRO DIGIT EIGHT"                                                             . #x16A68)
    ("MRO DIGIT NINE"                                                              . #x16A69)
    ("MRO DANDA"                                                                   . #x16A6E)
    ("MRO DOUBLE DANDA"                                                            . #x16A6F)
    ("BASSA VAH LETTER ENNI"                                                       . #x16AD0)
    ("BASSA VAH LETTER KA"                                                         . #x16AD1)
    ("BASSA VAH LETTER SE"                                                         . #x16AD2)
    ("BASSA VAH LETTER FA"                                                         . #x16AD3)
    ("BASSA VAH LETTER MBE"                                                        . #x16AD4)
    ("BASSA VAH LETTER YIE"                                                        . #x16AD5)
    ("BASSA VAH LETTER GAH"                                                        . #x16AD6)
    ("BASSA VAH LETTER DHII"                                                       . #x16AD7)
    ("BASSA VAH LETTER KPAH"                                                       . #x16AD8)
    ("BASSA VAH LETTER JO"                                                         . #x16AD9)
    ("BASSA VAH LETTER HWAH"                                                       . #x16ADA)
    ("BASSA VAH LETTER WA"                                                         . #x16ADB)
    ("BASSA VAH LETTER ZO"                                                         . #x16ADC)
    ("BASSA VAH LETTER GBU"                                                        . #x16ADD)
    ("BASSA VAH LETTER DO"                                                         . #x16ADE)
    ("BASSA VAH LETTER CE"                                                         . #x16ADF)
    ("BASSA VAH LETTER UWU"                                                        . #x16AE0)
    ("BASSA VAH LETTER TO"                                                         . #x16AE1)
    ("BASSA VAH LETTER BA"                                                         . #x16AE2)
    ("BASSA VAH LETTER VU"                                                         . #x16AE3)
    ("BASSA VAH LETTER YEIN"                                                       . #x16AE4)
    ("BASSA VAH LETTER PA"                                                         . #x16AE5)
    ("BASSA VAH LETTER WADDA"                                                      . #x16AE6)
    ("BASSA VAH LETTER A"                                                          . #x16AE7)
    ("BASSA VAH LETTER O"                                                          . #x16AE8)
    ("BASSA VAH LETTER OO"                                                         . #x16AE9)
    ("BASSA VAH LETTER U"                                                          . #x16AEA)
    ("BASSA VAH LETTER EE"                                                         . #x16AEB)
    ("BASSA VAH LETTER E"                                                          . #x16AEC)
    ("BASSA VAH LETTER I"                                                          . #x16AED)
    ("BASSA VAH COMBINING HIGH TONE"                                               . #x16AF0)
    ("BASSA VAH COMBINING LOW TONE"                                                . #x16AF1)
    ("BASSA VAH COMBINING MID TONE"                                                . #x16AF2)
    ("BASSA VAH COMBINING LOW-MID TONE"                                            . #x16AF3)
    ("BASSA VAH COMBINING HIGH-LOW TONE"                                           . #x16AF4)
    ("BASSA VAH FULL STOP"                                                         . #x16AF5)
    ("PAHAWH HMONG VOWEL KEEB"                                                     . #x16B00)
    ("PAHAWH HMONG VOWEL KEEV"                                                     . #x16B01)
    ("PAHAWH HMONG VOWEL KIB"                                                      . #x16B02)
    ("PAHAWH HMONG VOWEL KIV"                                                      . #x16B03)
    ("PAHAWH HMONG VOWEL KAUB"                                                     . #x16B04)
    ("PAHAWH HMONG VOWEL KAUV"                                                     . #x16B05)
    ("PAHAWH HMONG VOWEL KUB"                                                      . #x16B06)
    ("PAHAWH HMONG VOWEL KUV"                                                      . #x16B07)
    ("PAHAWH HMONG VOWEL KEB"                                                      . #x16B08)
    ("PAHAWH HMONG VOWEL KEV"                                                      . #x16B09)
    ("PAHAWH HMONG VOWEL KAIB"                                                     . #x16B0A)
    ("PAHAWH HMONG VOWEL KAIV"                                                     . #x16B0B)
    ("PAHAWH HMONG VOWEL KOOB"                                                     . #x16B0C)
    ("PAHAWH HMONG VOWEL KOOV"                                                     . #x16B0D)
    ("PAHAWH HMONG VOWEL KAWB"                                                     . #x16B0E)
    ("PAHAWH HMONG VOWEL KAWV"                                                     . #x16B0F)
    ("PAHAWH HMONG VOWEL KUAB"                                                     . #x16B10)
    ("PAHAWH HMONG VOWEL KUAV"                                                     . #x16B11)
    ("PAHAWH HMONG VOWEL KOB"                                                      . #x16B12)
    ("PAHAWH HMONG VOWEL KOV"                                                      . #x16B13)
    ("PAHAWH HMONG VOWEL KIAB"                                                     . #x16B14)
    ("PAHAWH HMONG VOWEL KIAV"                                                     . #x16B15)
    ("PAHAWH HMONG VOWEL KAB"                                                      . #x16B16)
    ("PAHAWH HMONG VOWEL KAV"                                                      . #x16B17)
    ("PAHAWH HMONG VOWEL KWB"                                                      . #x16B18)
    ("PAHAWH HMONG VOWEL KWV"                                                      . #x16B19)
    ("PAHAWH HMONG VOWEL KAAB"                                                     . #x16B1A)
    ("PAHAWH HMONG VOWEL KAAV"                                                     . #x16B1B)
    ("PAHAWH HMONG CONSONANT VAU"                                                  . #x16B1C)
    ("PAHAWH HMONG CONSONANT NTSAU"                                                . #x16B1D)
    ("PAHAWH HMONG CONSONANT LAU"                                                  . #x16B1E)
    ("PAHAWH HMONG CONSONANT HAU"                                                  . #x16B1F)
    ("PAHAWH HMONG CONSONANT NLAU"                                                 . #x16B20)
    ("PAHAWH HMONG CONSONANT RAU"                                                  . #x16B21)
    ("PAHAWH HMONG CONSONANT NKAU"                                                 . #x16B22)
    ("PAHAWH HMONG CONSONANT QHAU"                                                 . #x16B23)
    ("PAHAWH HMONG CONSONANT YAU"                                                  . #x16B24)
    ("PAHAWH HMONG CONSONANT HLAU"                                                 . #x16B25)
    ("PAHAWH HMONG CONSONANT MAU"                                                  . #x16B26)
    ("PAHAWH HMONG CONSONANT CHAU"                                                 . #x16B27)
    ("PAHAWH HMONG CONSONANT NCHAU"                                                . #x16B28)
    ("PAHAWH HMONG CONSONANT HNAU"                                                 . #x16B29)
    ("PAHAWH HMONG CONSONANT PLHAU"                                                . #x16B2A)
    ("PAHAWH HMONG CONSONANT NTHAU"                                                . #x16B2B)
    ("PAHAWH HMONG CONSONANT NAU"                                                  . #x16B2C)
    ("PAHAWH HMONG CONSONANT AU"                                                   . #x16B2D)
    ("PAHAWH HMONG CONSONANT XAU"                                                  . #x16B2E)
    ("PAHAWH HMONG CONSONANT CAU"                                                  . #x16B2F)
    ("PAHAWH HMONG MARK CIM TUB"                                                   . #x16B30)
    ("PAHAWH HMONG MARK CIM SO"                                                    . #x16B31)
    ("PAHAWH HMONG MARK CIM KES"                                                   . #x16B32)
    ("PAHAWH HMONG MARK CIM KHAV"                                                  . #x16B33)
    ("PAHAWH HMONG MARK CIM SUAM"                                                  . #x16B34)
    ("PAHAWH HMONG MARK CIM HOM"                                                   . #x16B35)
    ("PAHAWH HMONG MARK CIM TAUM"                                                  . #x16B36)
    ("PAHAWH HMONG SIGN VOS THOM"                                                  . #x16B37)
    ("PAHAWH HMONG SIGN VOS TSHAB CEEB"                                            . #x16B38)
    ("PAHAWH HMONG SIGN CIM CHEEM"                                                 . #x16B39)
    ("PAHAWH HMONG SIGN VOS THIAB"                                                 . #x16B3A)
    ("PAHAWH HMONG SIGN VOS FEEM"                                                  . #x16B3B)
    ("PAHAWH HMONG SIGN XYEEM NTXIV"                                               . #x16B3C)
    ("PAHAWH HMONG SIGN XYEEM RHO"                                                 . #x16B3D)
    ("PAHAWH HMONG SIGN XYEEM TOV"                                                 . #x16B3E)
    ("PAHAWH HMONG SIGN XYEEM FAIB"                                                . #x16B3F)
    ("PAHAWH HMONG SIGN VOS SEEV"                                                  . #x16B40)
    ("PAHAWH HMONG SIGN MEEJ SUAB"                                                 . #x16B41)
    ("PAHAWH HMONG SIGN VOS NRUA"                                                  . #x16B42)
    ("PAHAWH HMONG SIGN IB YAM"                                                    . #x16B43)
    ("PAHAWH HMONG SIGN XAUS"                                                      . #x16B44)
    ("PAHAWH HMONG SIGN CIM TSOV ROG"                                              . #x16B45)
    ("PAHAWH HMONG DIGIT ZERO"                                                     . #x16B50)
    ("PAHAWH HMONG DIGIT ONE"                                                      . #x16B51)
    ("PAHAWH HMONG DIGIT TWO"                                                      . #x16B52)
    ("PAHAWH HMONG DIGIT THREE"                                                    . #x16B53)
    ("PAHAWH HMONG DIGIT FOUR"                                                     . #x16B54)
    ("PAHAWH HMONG DIGIT FIVE"                                                     . #x16B55)
    ("PAHAWH HMONG DIGIT SIX"                                                      . #x16B56)
    ("PAHAWH HMONG DIGIT SEVEN"                                                    . #x16B57)
    ("PAHAWH HMONG DIGIT EIGHT"                                                    . #x16B58)
    ("PAHAWH HMONG DIGIT NINE"                                                     . #x16B59)
    ("PAHAWH HMONG NUMBER TENS"                                                    . #x16B5B)
    ("PAHAWH HMONG NUMBER HUNDREDS"                                                . #x16B5C)
    ("PAHAWH HMONG NUMBER TEN THOUSANDS"                                           . #x16B5D)
    ("PAHAWH HMONG NUMBER MILLIONS"                                                . #x16B5E)
    ("PAHAWH HMONG NUMBER HUNDRED MILLIONS"                                        . #x16B5F)
    ("PAHAWH HMONG NUMBER TEN BILLIONS"                                            . #x16B60)
    ("PAHAWH HMONG NUMBER TRILLIONS"                                               . #x16B61)
    ("PAHAWH HMONG SIGN VOS LUB"                                                   . #x16B63)
    ("PAHAWH HMONG SIGN XYOO"                                                      . #x16B64)
    ("PAHAWH HMONG SIGN HLI"                                                       . #x16B65)
    ("PAHAWH HMONG SIGN THIRD-STAGE HLI"                                           . #x16B66)
    ("PAHAWH HMONG SIGN ZWJ THAJ"                                                  . #x16B67)
    ("PAHAWH HMONG SIGN HNUB"                                                      . #x16B68)
    ("PAHAWH HMONG SIGN NQIG"                                                      . #x16B69)
    ("PAHAWH HMONG SIGN XIAB"                                                      . #x16B6A)
    ("PAHAWH HMONG SIGN NTUJ"                                                      . #x16B6B)
    ("PAHAWH HMONG SIGN AV"                                                        . #x16B6C)
    ("PAHAWH HMONG SIGN TXHEEJ CEEV"                                               . #x16B6D)
    ("PAHAWH HMONG SIGN MEEJ TSEEB"                                                . #x16B6E)
    ("PAHAWH HMONG SIGN TAU"                                                       . #x16B6F)
    ("PAHAWH HMONG SIGN LOS"                                                       . #x16B70)
    ("PAHAWH HMONG SIGN MUS"                                                       . #x16B71)
    ("PAHAWH HMONG SIGN CIM HAIS LUS NTOG NTOG"                                    . #x16B72)
    ("PAHAWH HMONG SIGN CIM CUAM TSHOOJ"                                           . #x16B73)
    ("PAHAWH HMONG SIGN CIM TXWV"                                                  . #x16B74)
    ("PAHAWH HMONG SIGN CIM TXWV CHWV"                                             . #x16B75)
    ("PAHAWH HMONG SIGN CIM PUB DAWB"                                              . #x16B76)
    ("PAHAWH HMONG SIGN CIM NRES TOS"                                              . #x16B77)
    ("PAHAWH HMONG CLAN SIGN TSHEEJ"                                               . #x16B7D)
    ("PAHAWH HMONG CLAN SIGN YEEG"                                                 . #x16B7E)
    ("PAHAWH HMONG CLAN SIGN LIS"                                                  . #x16B7F)
    ("PAHAWH HMONG CLAN SIGN LAUJ"                                                 . #x16B80)
    ("PAHAWH HMONG CLAN SIGN XYOOJ"                                                . #x16B81)
    ("PAHAWH HMONG CLAN SIGN KOO"                                                  . #x16B82)
    ("PAHAWH HMONG CLAN SIGN HAWJ"                                                 . #x16B83)
    ("PAHAWH HMONG CLAN SIGN MUAS"                                                 . #x16B84)
    ("PAHAWH HMONG CLAN SIGN THOJ"                                                 . #x16B85)
    ("PAHAWH HMONG CLAN SIGN TSAB"                                                 . #x16B86)
    ("PAHAWH HMONG CLAN SIGN PHAB"                                                 . #x16B87)
    ("PAHAWH HMONG CLAN SIGN KHAB"                                                 . #x16B88)
    ("PAHAWH HMONG CLAN SIGN HAM"                                                  . #x16B89)
    ("PAHAWH HMONG CLAN SIGN VAJ"                                                  . #x16B8A)
    ("PAHAWH HMONG CLAN SIGN FAJ"                                                  . #x16B8B)
    ("PAHAWH HMONG CLAN SIGN YAJ"                                                  . #x16B8C)
    ("PAHAWH HMONG CLAN SIGN TSWB"                                                 . #x16B8D)
    ("PAHAWH HMONG CLAN SIGN KWM"                                                  . #x16B8E)
    ("PAHAWH HMONG CLAN SIGN VWJ"                                                  . #x16B8F)
    ("DUPLOYAN LETTER H"                                                           . #x1BC00)
    ("DUPLOYAN LETTER X"                                                           . #x1BC01)
    ("DUPLOYAN LETTER P"                                                           . #x1BC02)
    ("DUPLOYAN LETTER T"                                                           . #x1BC03)
    ("DUPLOYAN LETTER F"                                                           . #x1BC04)
    ("DUPLOYAN LETTER K"                                                           . #x1BC05)
    ("DUPLOYAN LETTER L"                                                           . #x1BC06)
    ("DUPLOYAN LETTER B"                                                           . #x1BC07)
    ("DUPLOYAN LETTER D"                                                           . #x1BC08)
    ("DUPLOYAN LETTER V"                                                           . #x1BC09)
    ("DUPLOYAN LETTER G"                                                           . #x1BC0A)
    ("DUPLOYAN LETTER R"                                                           . #x1BC0B)
    ("DUPLOYAN LETTER P N"                                                         . #x1BC0C)
    ("DUPLOYAN LETTER D S"                                                         . #x1BC0D)
    ("DUPLOYAN LETTER F N"                                                         . #x1BC0E)
    ("DUPLOYAN LETTER K M"                                                         . #x1BC0F)
    ("DUPLOYAN LETTER R S"                                                         . #x1BC10)
    ("DUPLOYAN LETTER TH"                                                          . #x1BC11)
    ("DUPLOYAN LETTER SLOAN DH"                                                    . #x1BC12)
    ("DUPLOYAN LETTER DH"                                                          . #x1BC13)
    ("DUPLOYAN LETTER KK"                                                          . #x1BC14)
    ("DUPLOYAN LETTER SLOAN J"                                                     . #x1BC15)
    ("DUPLOYAN LETTER HL"                                                          . #x1BC16)
    ("DUPLOYAN LETTER LH"                                                          . #x1BC17)
    ("DUPLOYAN LETTER RH"                                                          . #x1BC18)
    ("DUPLOYAN LETTER M"                                                           . #x1BC19)
    ("DUPLOYAN LETTER N"                                                           . #x1BC1A)
    ("DUPLOYAN LETTER J"                                                           . #x1BC1B)
    ("DUPLOYAN LETTER S"                                                           . #x1BC1C)
    ("DUPLOYAN LETTER M N"                                                         . #x1BC1D)
    ("DUPLOYAN LETTER N M"                                                         . #x1BC1E)
    ("DUPLOYAN LETTER J M"                                                         . #x1BC1F)
    ("DUPLOYAN LETTER S J"                                                         . #x1BC20)
    ("DUPLOYAN LETTER M WITH DOT"                                                  . #x1BC21)
    ("DUPLOYAN LETTER N WITH DOT"                                                  . #x1BC22)
    ("DUPLOYAN LETTER J WITH DOT"                                                  . #x1BC23)
    ("DUPLOYAN LETTER J WITH DOTS INSIDE AND ABOVE"                                . #x1BC24)
    ("DUPLOYAN LETTER S WITH DOT"                                                  . #x1BC25)
    ("DUPLOYAN LETTER S WITH DOT BELOW"                                            . #x1BC26)
    ("DUPLOYAN LETTER M S"                                                         . #x1BC27)
    ("DUPLOYAN LETTER N S"                                                         . #x1BC28)
    ("DUPLOYAN LETTER J S"                                                         . #x1BC29)
    ("DUPLOYAN LETTER S S"                                                         . #x1BC2A)
    ("DUPLOYAN LETTER M N S"                                                       . #x1BC2B)
    ("DUPLOYAN LETTER N M S"                                                       . #x1BC2C)
    ("DUPLOYAN LETTER J M S"                                                       . #x1BC2D)
    ("DUPLOYAN LETTER S J S"                                                       . #x1BC2E)
    ("DUPLOYAN LETTER J S WITH DOT"                                                . #x1BC2F)
    ("DUPLOYAN LETTER J N"                                                         . #x1BC30)
    ("DUPLOYAN LETTER J N S"                                                       . #x1BC31)
    ("DUPLOYAN LETTER S T"                                                         . #x1BC32)
    ("DUPLOYAN LETTER S T R"                                                       . #x1BC33)
    ("DUPLOYAN LETTER S P"                                                         . #x1BC34)
    ("DUPLOYAN LETTER S P R"                                                       . #x1BC35)
    ("DUPLOYAN LETTER T S"                                                         . #x1BC36)
    ("DUPLOYAN LETTER T R S"                                                       . #x1BC37)
    ("DUPLOYAN LETTER W"                                                           . #x1BC38)
    ("DUPLOYAN LETTER WH"                                                          . #x1BC39)
    ("DUPLOYAN LETTER W R"                                                         . #x1BC3A)
    ("DUPLOYAN LETTER S N"                                                         . #x1BC3B)
    ("DUPLOYAN LETTER S M"                                                         . #x1BC3C)
    ("DUPLOYAN LETTER K R S"                                                       . #x1BC3D)
    ("DUPLOYAN LETTER G R S"                                                       . #x1BC3E)
    ("DUPLOYAN LETTER S K"                                                         . #x1BC3F)
    ("DUPLOYAN LETTER S K R"                                                       . #x1BC40)
    ("DUPLOYAN LETTER A"                                                           . #x1BC41)
    ("DUPLOYAN LETTER SLOAN OW"                                                    . #x1BC42)
    ("DUPLOYAN LETTER OA"                                                          . #x1BC43)
    ("DUPLOYAN LETTER O"                                                           . #x1BC44)
    ("DUPLOYAN LETTER AOU"                                                         . #x1BC45)
    ("DUPLOYAN LETTER I"                                                           . #x1BC46)
    ("DUPLOYAN LETTER E"                                                           . #x1BC47)
    ("DUPLOYAN LETTER IE"                                                          . #x1BC48)
    ("DUPLOYAN LETTER SHORT I"                                                     . #x1BC49)
    ("DUPLOYAN LETTER UI"                                                          . #x1BC4A)
    ("DUPLOYAN LETTER EE"                                                          . #x1BC4B)
    ("DUPLOYAN LETTER SLOAN EH"                                                    . #x1BC4C)
    ("DUPLOYAN LETTER ROMANIAN I"                                                  . #x1BC4D)
    ("DUPLOYAN LETTER SLOAN EE"                                                    . #x1BC4E)
    ("DUPLOYAN LETTER LONG I"                                                      . #x1BC4F)
    ("DUPLOYAN LETTER YE"                                                          . #x1BC50)
    ("DUPLOYAN LETTER U"                                                           . #x1BC51)
    ("DUPLOYAN LETTER EU"                                                          . #x1BC52)
    ("DUPLOYAN LETTER XW"                                                          . #x1BC53)
    ("DUPLOYAN LETTER U N"                                                         . #x1BC54)
    ("DUPLOYAN LETTER LONG U"                                                      . #x1BC55)
    ("DUPLOYAN LETTER ROMANIAN U"                                                  . #x1BC56)
    ("DUPLOYAN LETTER UH"                                                          . #x1BC57)
    ("DUPLOYAN LETTER SLOAN U"                                                     . #x1BC58)
    ("DUPLOYAN LETTER OOH"                                                         . #x1BC59)
    ("DUPLOYAN LETTER OW"                                                          . #x1BC5A)
    ("DUPLOYAN LETTER OU"                                                          . #x1BC5B)
    ("DUPLOYAN LETTER WA"                                                          . #x1BC5C)
    ("DUPLOYAN LETTER WO"                                                          . #x1BC5D)
    ("DUPLOYAN LETTER WI"                                                          . #x1BC5E)
    ("DUPLOYAN LETTER WEI"                                                         . #x1BC5F)
    ("DUPLOYAN LETTER WOW"                                                         . #x1BC60)
    ("DUPLOYAN LETTER NASAL U"                                                     . #x1BC61)
    ("DUPLOYAN LETTER NASAL O"                                                     . #x1BC62)
    ("DUPLOYAN LETTER NASAL I"                                                     . #x1BC63)
    ("DUPLOYAN LETTER NASAL A"                                                     . #x1BC64)
    ("DUPLOYAN LETTER PERNIN AN"                                                   . #x1BC65)
    ("DUPLOYAN LETTER PERNIN AM"                                                   . #x1BC66)
    ("DUPLOYAN LETTER SLOAN EN"                                                    . #x1BC67)
    ("DUPLOYAN LETTER SLOAN AN"                                                    . #x1BC68)
    ("DUPLOYAN LETTER SLOAN ON"                                                    . #x1BC69)
    ("DUPLOYAN LETTER VOCALIC M"                                                   . #x1BC6A)
    ("DUPLOYAN AFFIX LEFT HORIZONTAL SECANT"                                       . #x1BC70)
    ("DUPLOYAN AFFIX MID HORIZONTAL SECANT"                                        . #x1BC71)
    ("DUPLOYAN AFFIX RIGHT HORIZONTAL SECANT"                                      . #x1BC72)
    ("DUPLOYAN AFFIX LOW VERTICAL SECANT"                                          . #x1BC73)
    ("DUPLOYAN AFFIX MID VERTICAL SECANT"                                          . #x1BC74)
    ("DUPLOYAN AFFIX HIGH VERTICAL SECANT"                                         . #x1BC75)
    ("DUPLOYAN AFFIX ATTACHED SECANT"                                              . #x1BC76)
    ("DUPLOYAN AFFIX ATTACHED LEFT-TO-RIGHT SECANT"                                . #x1BC77)
    ("DUPLOYAN AFFIX ATTACHED TANGENT"                                             . #x1BC78)
    ("DUPLOYAN AFFIX ATTACHED TAIL"                                                . #x1BC79)
    ("DUPLOYAN AFFIX ATTACHED E HOOK"                                              . #x1BC7A)
    ("DUPLOYAN AFFIX ATTACHED I HOOK"                                              . #x1BC7B)
    ("DUPLOYAN AFFIX ATTACHED TANGENT HOOK"                                        . #x1BC7C)
    ("DUPLOYAN AFFIX HIGH ACUTE"                                                   . #x1BC80)
    ("DUPLOYAN AFFIX HIGH TIGHT ACUTE"                                             . #x1BC81)
    ("DUPLOYAN AFFIX HIGH GRAVE"                                                   . #x1BC82)
    ("DUPLOYAN AFFIX HIGH LONG GRAVE"                                              . #x1BC83)
    ("DUPLOYAN AFFIX HIGH DOT"                                                     . #x1BC84)
    ("DUPLOYAN AFFIX HIGH CIRCLE"                                                  . #x1BC85)
    ("DUPLOYAN AFFIX HIGH LINE"                                                    . #x1BC86)
    ("DUPLOYAN AFFIX HIGH WAVE"                                                    . #x1BC87)
    ("DUPLOYAN AFFIX HIGH VERTICAL"                                                . #x1BC88)
    ("DUPLOYAN AFFIX LOW ACUTE"                                                    . #x1BC90)
    ("DUPLOYAN AFFIX LOW TIGHT ACUTE"                                              . #x1BC91)
    ("DUPLOYAN AFFIX LOW GRAVE"                                                    . #x1BC92)
    ("DUPLOYAN AFFIX LOW LONG GRAVE"                                               . #x1BC93)
    ("DUPLOYAN AFFIX LOW DOT"                                                      . #x1BC94)
    ("DUPLOYAN AFFIX LOW CIRCLE"                                                   . #x1BC95)
    ("DUPLOYAN AFFIX LOW LINE"                                                     . #x1BC96)
    ("DUPLOYAN AFFIX LOW WAVE"                                                     . #x1BC97)
    ("DUPLOYAN AFFIX LOW VERTICAL"                                                 . #x1BC98)
    ("DUPLOYAN AFFIX LOW ARROW"                                                    . #x1BC99)
    ("DUPLOYAN SIGN O WITH CROSS"                                                  . #x1BC9C)
    ("DUPLOYAN THICK LETTER SELECTOR"                                              . #x1BC9D)
    ("DUPLOYAN DOUBLE MARK"                                                        . #x1BC9E)
    ("DUPLOYAN PUNCTUATION CHINOOK FULL STOP"                                      . #x1BC9F)
    ("SHORTHAND FORMAT LETTER OVERLAP"                                             . #x1BCA0)
    ("SHORTHAND FORMAT CONTINUING OVERLAP"                                         . #x1BCA1)
    ("SHORTHAND FORMAT DOWN STEP"                                                  . #x1BCA2)
    ("SHORTHAND FORMAT UP STEP"                                                    . #x1BCA3)
    ("MENDE KIKAKUI SYLLABLE M001 KI"                                              . #x1E800)
    ("MENDE KIKAKUI SYLLABLE M002 KA"                                              . #x1E801)
    ("MENDE KIKAKUI SYLLABLE M003 KU"                                              . #x1E802)
    ("MENDE KIKAKUI SYLLABLE M065 KEE"                                             . #x1E803)
    ("MENDE KIKAKUI SYLLABLE M095 KE"                                              . #x1E804)
    ("MENDE KIKAKUI SYLLABLE M076 KOO"                                             . #x1E805)
    ("MENDE KIKAKUI SYLLABLE M048 KO"                                              . #x1E806)
    ("MENDE KIKAKUI SYLLABLE M179 KUA"                                             . #x1E807)
    ("MENDE KIKAKUI SYLLABLE M004 WI"                                              . #x1E808)
    ("MENDE KIKAKUI SYLLABLE M005 WA"                                              . #x1E809)
    ("MENDE KIKAKUI SYLLABLE M006 WU"                                              . #x1E80A)
    ("MENDE KIKAKUI SYLLABLE M126 WEE"                                             . #x1E80B)
    ("MENDE KIKAKUI SYLLABLE M118 WE"                                              . #x1E80C)
    ("MENDE KIKAKUI SYLLABLE M114 WOO"                                             . #x1E80D)
    ("MENDE KIKAKUI SYLLABLE M045 WO"                                              . #x1E80E)
    ("MENDE KIKAKUI SYLLABLE M194 WUI"                                             . #x1E80F)
    ("MENDE KIKAKUI SYLLABLE M143 WEI"                                             . #x1E810)
    ("MENDE KIKAKUI SYLLABLE M061 WVI"                                             . #x1E811)
    ("MENDE KIKAKUI SYLLABLE M049 WVA"                                             . #x1E812)
    ("MENDE KIKAKUI SYLLABLE M139 WVE"                                             . #x1E813)
    ("MENDE KIKAKUI SYLLABLE M007 MIN"                                             . #x1E814)
    ("MENDE KIKAKUI SYLLABLE M008 MAN"                                             . #x1E815)
    ("MENDE KIKAKUI SYLLABLE M009 MUN"                                             . #x1E816)
    ("MENDE KIKAKUI SYLLABLE M059 MEN"                                             . #x1E817)
    ("MENDE KIKAKUI SYLLABLE M094 MON"                                             . #x1E818)
    ("MENDE KIKAKUI SYLLABLE M154 MUAN"                                            . #x1E819)
    ("MENDE KIKAKUI SYLLABLE M189 MUEN"                                            . #x1E81A)
    ("MENDE KIKAKUI SYLLABLE M010 BI"                                              . #x1E81B)
    ("MENDE KIKAKUI SYLLABLE M011 BA"                                              . #x1E81C)
    ("MENDE KIKAKUI SYLLABLE M012 BU"                                              . #x1E81D)
    ("MENDE KIKAKUI SYLLABLE M150 BEE"                                             . #x1E81E)
    ("MENDE KIKAKUI SYLLABLE M097 BE"                                              . #x1E81F)
    ("MENDE KIKAKUI SYLLABLE M103 BOO"                                             . #x1E820)
    ("MENDE KIKAKUI SYLLABLE M138 BO"                                              . #x1E821)
    ("MENDE KIKAKUI SYLLABLE M013 I"                                               . #x1E822)
    ("MENDE KIKAKUI SYLLABLE M014 A"                                               . #x1E823)
    ("MENDE KIKAKUI SYLLABLE M015 U"                                               . #x1E824)
    ("MENDE KIKAKUI SYLLABLE M163 EE"                                              . #x1E825)
    ("MENDE KIKAKUI SYLLABLE M100 E"                                               . #x1E826)
    ("MENDE KIKAKUI SYLLABLE M165 OO"                                              . #x1E827)
    ("MENDE KIKAKUI SYLLABLE M147 O"                                               . #x1E828)
    ("MENDE KIKAKUI SYLLABLE M137 EI"                                              . #x1E829)
    ("MENDE KIKAKUI SYLLABLE M131 IN"                                              . #x1E82A)
    ("MENDE KIKAKUI SYLLABLE M135 IN"                                              . #x1E82B)
    ("MENDE KIKAKUI SYLLABLE M195 AN"                                              . #x1E82C)
    ("MENDE KIKAKUI SYLLABLE M178 EN"                                              . #x1E82D)
    ("MENDE KIKAKUI SYLLABLE M019 SI"                                              . #x1E82E)
    ("MENDE KIKAKUI SYLLABLE M020 SA"                                              . #x1E82F)
    ("MENDE KIKAKUI SYLLABLE M021 SU"                                              . #x1E830)
    ("MENDE KIKAKUI SYLLABLE M162 SEE"                                             . #x1E831)
    ("MENDE KIKAKUI SYLLABLE M116 SE"                                              . #x1E832)
    ("MENDE KIKAKUI SYLLABLE M136 SOO"                                             . #x1E833)
    ("MENDE KIKAKUI SYLLABLE M079 SO"                                              . #x1E834)
    ("MENDE KIKAKUI SYLLABLE M196 SIA"                                             . #x1E835)
    ("MENDE KIKAKUI SYLLABLE M025 LI"                                              . #x1E836)
    ("MENDE KIKAKUI SYLLABLE M026 LA"                                              . #x1E837)
    ("MENDE KIKAKUI SYLLABLE M027 LU"                                              . #x1E838)
    ("MENDE KIKAKUI SYLLABLE M084 LEE"                                             . #x1E839)
    ("MENDE KIKAKUI SYLLABLE M073 LE"                                              . #x1E83A)
    ("MENDE KIKAKUI SYLLABLE M054 LOO"                                             . #x1E83B)
    ("MENDE KIKAKUI SYLLABLE M153 LO"                                              . #x1E83C)
    ("MENDE KIKAKUI SYLLABLE M110 LONG LE"                                         . #x1E83D)
    ("MENDE KIKAKUI SYLLABLE M016 DI"                                              . #x1E83E)
    ("MENDE KIKAKUI SYLLABLE M017 DA"                                              . #x1E83F)
    ("MENDE KIKAKUI SYLLABLE M018 DU"                                              . #x1E840)
    ("MENDE KIKAKUI SYLLABLE M089 DEE"                                             . #x1E841)
    ("MENDE KIKAKUI SYLLABLE M180 DOO"                                             . #x1E842)
    ("MENDE KIKAKUI SYLLABLE M181 DO"                                              . #x1E843)
    ("MENDE KIKAKUI SYLLABLE M022 TI"                                              . #x1E844)
    ("MENDE KIKAKUI SYLLABLE M023 TA"                                              . #x1E845)
    ("MENDE KIKAKUI SYLLABLE M024 TU"                                              . #x1E846)
    ("MENDE KIKAKUI SYLLABLE M091 TEE"                                             . #x1E847)
    ("MENDE KIKAKUI SYLLABLE M055 TE"                                              . #x1E848)
    ("MENDE KIKAKUI SYLLABLE M104 TOO"                                             . #x1E849)
    ("MENDE KIKAKUI SYLLABLE M069 TO"                                              . #x1E84A)
    ("MENDE KIKAKUI SYLLABLE M028 JI"                                              . #x1E84B)
    ("MENDE KIKAKUI SYLLABLE M029 JA"                                              . #x1E84C)
    ("MENDE KIKAKUI SYLLABLE M030 JU"                                              . #x1E84D)
    ("MENDE KIKAKUI SYLLABLE M157 JEE"                                             . #x1E84E)
    ("MENDE KIKAKUI SYLLABLE M113 JE"                                              . #x1E84F)
    ("MENDE KIKAKUI SYLLABLE M160 JOO"                                             . #x1E850)
    ("MENDE KIKAKUI SYLLABLE M063 JO"                                              . #x1E851)
    ("MENDE KIKAKUI SYLLABLE M175 LONG JO"                                         . #x1E852)
    ("MENDE KIKAKUI SYLLABLE M031 YI"                                              . #x1E853)
    ("MENDE KIKAKUI SYLLABLE M032 YA"                                              . #x1E854)
    ("MENDE KIKAKUI SYLLABLE M033 YU"                                              . #x1E855)
    ("MENDE KIKAKUI SYLLABLE M109 YEE"                                             . #x1E856)
    ("MENDE KIKAKUI SYLLABLE M080 YE"                                              . #x1E857)
    ("MENDE KIKAKUI SYLLABLE M141 YOO"                                             . #x1E858)
    ("MENDE KIKAKUI SYLLABLE M121 YO"                                              . #x1E859)
    ("MENDE KIKAKUI SYLLABLE M034 FI"                                              . #x1E85A)
    ("MENDE KIKAKUI SYLLABLE M035 FA"                                              . #x1E85B)
    ("MENDE KIKAKUI SYLLABLE M036 FU"                                              . #x1E85C)
    ("MENDE KIKAKUI SYLLABLE M078 FEE"                                             . #x1E85D)
    ("MENDE KIKAKUI SYLLABLE M075 FE"                                              . #x1E85E)
    ("MENDE KIKAKUI SYLLABLE M133 FOO"                                             . #x1E85F)
    ("MENDE KIKAKUI SYLLABLE M088 FO"                                              . #x1E860)
    ("MENDE KIKAKUI SYLLABLE M197 FUA"                                             . #x1E861)
    ("MENDE KIKAKUI SYLLABLE M101 FAN"                                             . #x1E862)
    ("MENDE KIKAKUI SYLLABLE M037 NIN"                                             . #x1E863)
    ("MENDE KIKAKUI SYLLABLE M038 NAN"                                             . #x1E864)
    ("MENDE KIKAKUI SYLLABLE M039 NUN"                                             . #x1E865)
    ("MENDE KIKAKUI SYLLABLE M117 NEN"                                             . #x1E866)
    ("MENDE KIKAKUI SYLLABLE M169 NON"                                             . #x1E867)
    ("MENDE KIKAKUI SYLLABLE M176 HI"                                              . #x1E868)
    ("MENDE KIKAKUI SYLLABLE M041 HA"                                              . #x1E869)
    ("MENDE KIKAKUI SYLLABLE M186 HU"                                              . #x1E86A)
    ("MENDE KIKAKUI SYLLABLE M040 HEE"                                             . #x1E86B)
    ("MENDE KIKAKUI SYLLABLE M096 HE"                                              . #x1E86C)
    ("MENDE KIKAKUI SYLLABLE M042 HOO"                                             . #x1E86D)
    ("MENDE KIKAKUI SYLLABLE M140 HO"                                              . #x1E86E)
    ("MENDE KIKAKUI SYLLABLE M083 HEEI"                                            . #x1E86F)
    ("MENDE KIKAKUI SYLLABLE M128 HOOU"                                            . #x1E870)
    ("MENDE KIKAKUI SYLLABLE M053 HIN"                                             . #x1E871)
    ("MENDE KIKAKUI SYLLABLE M130 HAN"                                             . #x1E872)
    ("MENDE KIKAKUI SYLLABLE M087 HUN"                                             . #x1E873)
    ("MENDE KIKAKUI SYLLABLE M052 HEN"                                             . #x1E874)
    ("MENDE KIKAKUI SYLLABLE M193 HON"                                             . #x1E875)
    ("MENDE KIKAKUI SYLLABLE M046 HUAN"                                            . #x1E876)
    ("MENDE KIKAKUI SYLLABLE M090 NGGI"                                            . #x1E877)
    ("MENDE KIKAKUI SYLLABLE M043 NGGA"                                            . #x1E878)
    ("MENDE KIKAKUI SYLLABLE M082 NGGU"                                            . #x1E879)
    ("MENDE KIKAKUI SYLLABLE M115 NGGEE"                                           . #x1E87A)
    ("MENDE KIKAKUI SYLLABLE M146 NGGE"                                            . #x1E87B)
    ("MENDE KIKAKUI SYLLABLE M156 NGGOO"                                           . #x1E87C)
    ("MENDE KIKAKUI SYLLABLE M120 NGGO"                                            . #x1E87D)
    ("MENDE KIKAKUI SYLLABLE M159 NGGAA"                                           . #x1E87E)
    ("MENDE KIKAKUI SYLLABLE M127 NGGUA"                                           . #x1E87F)
    ("MENDE KIKAKUI SYLLABLE M086 LONG NGGE"                                       . #x1E880)
    ("MENDE KIKAKUI SYLLABLE M106 LONG NGGOO"                                      . #x1E881)
    ("MENDE KIKAKUI SYLLABLE M183 LONG NGGO"                                       . #x1E882)
    ("MENDE KIKAKUI SYLLABLE M155 GI"                                              . #x1E883)
    ("MENDE KIKAKUI SYLLABLE M111 GA"                                              . #x1E884)
    ("MENDE KIKAKUI SYLLABLE M168 GU"                                              . #x1E885)
    ("MENDE KIKAKUI SYLLABLE M190 GEE"                                             . #x1E886)
    ("MENDE KIKAKUI SYLLABLE M166 GUEI"                                            . #x1E887)
    ("MENDE KIKAKUI SYLLABLE M167 GUAN"                                            . #x1E888)
    ("MENDE KIKAKUI SYLLABLE M184 NGEN"                                            . #x1E889)
    ("MENDE KIKAKUI SYLLABLE M057 NGON"                                            . #x1E88A)
    ("MENDE KIKAKUI SYLLABLE M177 NGUAN"                                           . #x1E88B)
    ("MENDE KIKAKUI SYLLABLE M068 PI"                                              . #x1E88C)
    ("MENDE KIKAKUI SYLLABLE M099 PA"                                              . #x1E88D)
    ("MENDE KIKAKUI SYLLABLE M050 PU"                                              . #x1E88E)
    ("MENDE KIKAKUI SYLLABLE M081 PEE"                                             . #x1E88F)
    ("MENDE KIKAKUI SYLLABLE M051 PE"                                              . #x1E890)
    ("MENDE KIKAKUI SYLLABLE M102 POO"                                             . #x1E891)
    ("MENDE KIKAKUI SYLLABLE M066 PO"                                              . #x1E892)
    ("MENDE KIKAKUI SYLLABLE M145 MBI"                                             . #x1E893)
    ("MENDE KIKAKUI SYLLABLE M062 MBA"                                             . #x1E894)
    ("MENDE KIKAKUI SYLLABLE M122 MBU"                                             . #x1E895)
    ("MENDE KIKAKUI SYLLABLE M047 MBEE"                                            . #x1E896)
    ("MENDE KIKAKUI SYLLABLE M188 MBEE"                                            . #x1E897)
    ("MENDE KIKAKUI SYLLABLE M072 MBE"                                             . #x1E898)
    ("MENDE KIKAKUI SYLLABLE M172 MBOO"                                            . #x1E899)
    ("MENDE KIKAKUI SYLLABLE M174 MBO"                                             . #x1E89A)
    ("MENDE KIKAKUI SYLLABLE M187 MBUU"                                            . #x1E89B)
    ("MENDE KIKAKUI SYLLABLE M161 LONG MBE"                                        . #x1E89C)
    ("MENDE KIKAKUI SYLLABLE M105 LONG MBOO"                                       . #x1E89D)
    ("MENDE KIKAKUI SYLLABLE M142 LONG MBO"                                        . #x1E89E)
    ("MENDE KIKAKUI SYLLABLE M132 KPI"                                             . #x1E89F)
    ("MENDE KIKAKUI SYLLABLE M092 KPA"                                             . #x1E8A0)
    ("MENDE KIKAKUI SYLLABLE M074 KPU"                                             . #x1E8A1)
    ("MENDE KIKAKUI SYLLABLE M044 KPEE"                                            . #x1E8A2)
    ("MENDE KIKAKUI SYLLABLE M108 KPE"                                             . #x1E8A3)
    ("MENDE KIKAKUI SYLLABLE M112 KPOO"                                            . #x1E8A4)
    ("MENDE KIKAKUI SYLLABLE M158 KPO"                                             . #x1E8A5)
    ("MENDE KIKAKUI SYLLABLE M124 GBI"                                             . #x1E8A6)
    ("MENDE KIKAKUI SYLLABLE M056 GBA"                                             . #x1E8A7)
    ("MENDE KIKAKUI SYLLABLE M148 GBU"                                             . #x1E8A8)
    ("MENDE KIKAKUI SYLLABLE M093 GBEE"                                            . #x1E8A9)
    ("MENDE KIKAKUI SYLLABLE M107 GBE"                                             . #x1E8AA)
    ("MENDE KIKAKUI SYLLABLE M071 GBOO"                                            . #x1E8AB)
    ("MENDE KIKAKUI SYLLABLE M070 GBO"                                             . #x1E8AC)
    ("MENDE KIKAKUI SYLLABLE M171 RA"                                              . #x1E8AD)
    ("MENDE KIKAKUI SYLLABLE M123 NDI"                                             . #x1E8AE)
    ("MENDE KIKAKUI SYLLABLE M129 NDA"                                             . #x1E8AF)
    ("MENDE KIKAKUI SYLLABLE M125 NDU"                                             . #x1E8B0)
    ("MENDE KIKAKUI SYLLABLE M191 NDEE"                                            . #x1E8B1)
    ("MENDE KIKAKUI SYLLABLE M119 NDE"                                             . #x1E8B2)
    ("MENDE KIKAKUI SYLLABLE M067 NDOO"                                            . #x1E8B3)
    ("MENDE KIKAKUI SYLLABLE M064 NDO"                                             . #x1E8B4)
    ("MENDE KIKAKUI SYLLABLE M152 NJA"                                             . #x1E8B5)
    ("MENDE KIKAKUI SYLLABLE M192 NJU"                                             . #x1E8B6)
    ("MENDE KIKAKUI SYLLABLE M149 NJEE"                                            . #x1E8B7)
    ("MENDE KIKAKUI SYLLABLE M134 NJOO"                                            . #x1E8B8)
    ("MENDE KIKAKUI SYLLABLE M182 VI"                                              . #x1E8B9)
    ("MENDE KIKAKUI SYLLABLE M185 VA"                                              . #x1E8BA)
    ("MENDE KIKAKUI SYLLABLE M151 VU"                                              . #x1E8BB)
    ("MENDE KIKAKUI SYLLABLE M173 VEE"                                             . #x1E8BC)
    ("MENDE KIKAKUI SYLLABLE M085 VE"                                              . #x1E8BD)
    ("MENDE KIKAKUI SYLLABLE M144 VOO"                                             . #x1E8BE)
    ("MENDE KIKAKUI SYLLABLE M077 VO"                                              . #x1E8BF)
    ("MENDE KIKAKUI SYLLABLE M164 NYIN"                                            . #x1E8C0)
    ("MENDE KIKAKUI SYLLABLE M058 NYAN"                                            . #x1E8C1)
    ("MENDE KIKAKUI SYLLABLE M170 NYUN"                                            . #x1E8C2)
    ("MENDE KIKAKUI SYLLABLE M098 NYEN"                                            . #x1E8C3)
    ("MENDE KIKAKUI SYLLABLE M060 NYON"                                            . #x1E8C4)
    ("MENDE KIKAKUI DIGIT ONE"                                                     . #x1E8C7)
    ("MENDE KIKAKUI DIGIT TWO"                                                     . #x1E8C8)
    ("MENDE KIKAKUI DIGIT THREE"                                                   . #x1E8C9)
    ("MENDE KIKAKUI DIGIT FOUR"                                                    . #x1E8CA)
    ("MENDE KIKAKUI DIGIT FIVE"                                                    . #x1E8CB)
    ("MENDE KIKAKUI DIGIT SIX"                                                     . #x1E8CC)
    ("MENDE KIKAKUI DIGIT SEVEN"                                                   . #x1E8CD)
    ("MENDE KIKAKUI DIGIT EIGHT"                                                   . #x1E8CE)
    ("MENDE KIKAKUI DIGIT NINE"                                                    . #x1E8CF)
    ("MENDE KIKAKUI COMBINING NUMBER TEENS"                                        . #x1E8D0)
    ("MENDE KIKAKUI COMBINING NUMBER TENS"                                         . #x1E8D1)
    ("MENDE KIKAKUI COMBINING NUMBER HUNDREDS"                                     . #x1E8D2)
    ("MENDE KIKAKUI COMBINING NUMBER THOUSANDS"                                    . #x1E8D3)
    ("MENDE KIKAKUI COMBINING NUMBER TEN THOUSANDS"                                . #x1E8D4)
    ("MENDE KIKAKUI COMBINING NUMBER HUNDRED THOUSANDS"                            . #x1E8D5)
    ("MENDE KIKAKUI COMBINING NUMBER MILLIONS"                                     . #x1E8D6)
    ("PLAYING CARD RED JOKER"                                                      . #x1F0BF)
    ("PLAYING CARD FOOL"                                                           . #x1F0E0)
    ("PLAYING CARD TRUMP-1"                                                        . #x1F0E1)
    ("PLAYING CARD TRUMP-2"                                                        . #x1F0E2)
    ("PLAYING CARD TRUMP-3"                                                        . #x1F0E3)
    ("PLAYING CARD TRUMP-4"                                                        . #x1F0E4)
    ("PLAYING CARD TRUMP-5"                                                        . #x1F0E5)
    ("PLAYING CARD TRUMP-6"                                                        . #x1F0E6)
    ("PLAYING CARD TRUMP-7"                                                        . #x1F0E7)
    ("PLAYING CARD TRUMP-8"                                                        . #x1F0E8)
    ("PLAYING CARD TRUMP-9"                                                        . #x1F0E9)
    ("PLAYING CARD TRUMP-10"                                                       . #x1F0EA)
    ("PLAYING CARD TRUMP-11"                                                       . #x1F0EB)
    ("PLAYING CARD TRUMP-12"                                                       . #x1F0EC)
    ("PLAYING CARD TRUMP-13"                                                       . #x1F0ED)
    ("PLAYING CARD TRUMP-14"                                                       . #x1F0EE)
    ("PLAYING CARD TRUMP-15"                                                       . #x1F0EF)
    ("PLAYING CARD TRUMP-16"                                                       . #x1F0F0)
    ("PLAYING CARD TRUMP-17"                                                       . #x1F0F1)
    ("PLAYING CARD TRUMP-18"                                                       . #x1F0F2)
    ("PLAYING CARD TRUMP-19"                                                       . #x1F0F3)
    ("PLAYING CARD TRUMP-20"                                                       . #x1F0F4)
    ("PLAYING CARD TRUMP-21"                                                       . #x1F0F5)
    ("DINGBAT CIRCLED SANS-SERIF DIGIT ZERO"                                       . #x1F10B)
    ("DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT ZERO"                              . #x1F10C)
    ("THERMOMETER"                                                                 . #x1F321)
    ("BLACK DROPLET"                                                               . #x1F322)
    ("WHITE SUN"                                                                   . #x1F323)
    ("WHITE SUN WITH SMALL CLOUD"                                                  . #x1F324)
    ("WHITE SUN BEHIND CLOUD"                                                      . #x1F325)
    ("WHITE SUN BEHIND CLOUD WITH RAIN"                                            . #x1F326)
    ("CLOUD WITH RAIN"                                                             . #x1F327)
    ("CLOUD WITH SNOW"                                                             . #x1F328)
    ("CLOUD WITH LIGHTNING"                                                        . #x1F329)
    ("CLOUD WITH TORNADO"                                                          . #x1F32A)
    ("FOG"                                                                         . #x1F32B)
    ("WIND BLOWING FACE"                                                           . #x1F32C)
    ("HOT PEPPER"                                                                  . #x1F336)
    ("FORK AND KNIFE WITH PLATE"                                                   . #x1F37D)
    ("HEART WITH TIP ON THE LEFT"                                                  . #x1F394)
    ("BOUQUET OF FLOWERS"                                                          . #x1F395)
    ("MILITARY MEDAL"                                                              . #x1F396)
    ("REMINDER RIBBON"                                                             . #x1F397)
    ("MUSICAL KEYBOARD WITH JACKS"                                                 . #x1F398)
    ("STUDIO MICROPHONE"                                                           . #x1F399)
    ("LEVEL SLIDER"                                                                . #x1F39A)
    ("CONTROL KNOBS"                                                               . #x1F39B)
    ("BEAMED ASCENDING MUSICAL NOTES"                                              . #x1F39C)
    ("BEAMED DESCENDING MUSICAL NOTES"                                             . #x1F39D)
    ("FILM FRAMES"                                                                 . #x1F39E)
    ("ADMISSION TICKETS"                                                           . #x1F39F)
    ("SPORTS MEDAL"                                                                . #x1F3C5)
    ("WEIGHT LIFTER"                                                               . #x1F3CB)
    ("GOLFER"                                                                      . #x1F3CC)
    ("RACING MOTORCYCLE"                                                           . #x1F3CD)
    ("RACING CAR"                                                                  . #x1F3CE)
    ("SNOW CAPPED MOUNTAIN"                                                        . #x1F3D4)
    ("CAMPING"                                                                     . #x1F3D5)
    ("BEACH WITH UMBRELLA"                                                         . #x1F3D6)
    ("BUILDING CONSTRUCTION"                                                       . #x1F3D7)
    ("HOUSE BUILDINGS"                                                             . #x1F3D8)
    ("CITYSCAPE"                                                                   . #x1F3D9)
    ("DERELICT HOUSE BUILDING"                                                     . #x1F3DA)
    ("CLASSICAL BUILDING"                                                          . #x1F3DB)
    ("DESERT"                                                                      . #x1F3DC)
    ("DESERT ISLAND"                                                               . #x1F3DD)
    ("NATIONAL PARK"                                                               . #x1F3DE)
    ("STADIUM"                                                                     . #x1F3DF)
    ("WHITE PENNANT"                                                               . #x1F3F1)
    ("BLACK PENNANT"                                                               . #x1F3F2)
    ("WAVING WHITE FLAG"                                                           . #x1F3F3)
    ("WAVING BLACK FLAG"                                                           . #x1F3F4)
    ("ROSETTE"                                                                     . #x1F3F5)
    ("BLACK ROSETTE"                                                               . #x1F3F6)
    ("LABEL"                                                                       . #x1F3F7)
    ("CHIPMUNK"                                                                    . #x1F43F)
    ("EYE"                                                                         . #x1F441)
    ("CAMERA WITH FLASH"                                                           . #x1F4F8)
    ("FILM PROJECTOR"                                                              . #x1F4FD)
    ("PORTABLE STEREO"                                                             . #x1F4FE)
    ("LOWER RIGHT SHADOWED WHITE CIRCLE"                                           . #x1F53E)
    ("UPPER RIGHT SHADOWED WHITE CIRCLE"                                           . #x1F53F)
    ("NOTCHED RIGHT SEMICIRCLE WITH THREE DOTS"                                    . #x1F544)
    ("SYMBOL FOR MARKS CHAPTER"                                                    . #x1F545)
    ("WHITE LATIN CROSS"                                                           . #x1F546)
    ("HEAVY LATIN CROSS"                                                           . #x1F547)
    ("CELTIC CROSS"                                                                . #x1F548)
    ("OM SYMBOL"                                                                   . #x1F549)
    ("DOVE OF PEACE"                                                               . #x1F54A)
    ("RIGHT SPEAKER"                                                               . #x1F568)
    ("RIGHT SPEAKER WITH ONE SOUND WAVE"                                           . #x1F569)
    ("RIGHT SPEAKER WITH THREE SOUND WAVES"                                        . #x1F56A)
    ("BULLHORN"                                                                    . #x1F56B)
    ("BULLHORN WITH SOUND WAVES"                                                   . #x1F56C)
    ("RINGING BELL"                                                                . #x1F56D)
    ("BOOK"                                                                        . #x1F56E)
    ("CANDLE"                                                                      . #x1F56F)
    ("MANTELPIECE CLOCK"                                                           . #x1F570)
    ("BLACK SKULL AND CROSSBONES"                                                  . #x1F571)
    ("NO PIRACY"                                                                   . #x1F572)
    ("HOLE"                                                                        . #x1F573)
    ("MAN IN BUSINESS SUIT LEVITATING"                                             . #x1F574)
    ("SLEUTH OR SPY"                                                               . #x1F575)
    ("DARK SUNGLASSES"                                                             . #x1F576)
    ("SPIDER"                                                                      . #x1F577)
    ("SPIDER WEB"                                                                  . #x1F578)
    ("JOYSTICK"                                                                    . #x1F579)
    ("LEFT HAND TELEPHONE RECEIVER"                                                . #x1F57B)
    ("TELEPHONE RECEIVER WITH PAGE"                                                . #x1F57C)
    ("RIGHT HAND TELEPHONE RECEIVER"                                               . #x1F57D)
    ("WHITE TOUCHTONE TELEPHONE"                                                   . #x1F57E)
    ("BLACK TOUCHTONE TELEPHONE"                                                   . #x1F57F)
    ("TELEPHONE ON TOP OF MODEM"                                                   . #x1F580)
    ("CLAMSHELL MOBILE PHONE"                                                      . #x1F581)
    ("BACK OF ENVELOPE"                                                            . #x1F582)
    ("STAMPED ENVELOPE"                                                            . #x1F583)
    ("ENVELOPE WITH LIGHTNING"                                                     . #x1F584)
    ("FLYING ENVELOPE"                                                             . #x1F585)
    ("PEN OVER STAMPED ENVELOPE"                                                   . #x1F586)
    ("LINKED PAPERCLIPS"                                                           . #x1F587)
    ("BLACK PUSHPIN"                                                               . #x1F588)
    ("LOWER LEFT PENCIL"                                                           . #x1F589)
    ("LOWER LEFT BALLPOINT PEN"                                                    . #x1F58A)
    ("LOWER LEFT FOUNTAIN PEN"                                                     . #x1F58B)
    ("LOWER LEFT PAINTBRUSH"                                                       . #x1F58C)
    ("LOWER LEFT CRAYON"                                                           . #x1F58D)
    ("LEFT WRITING HAND"                                                           . #x1F58E)
    ("TURNED OK HAND SIGN"                                                         . #x1F58F)
    ("RAISED HAND WITH FINGERS SPLAYED"                                            . #x1F590)
    ("REVERSED RAISED HAND WITH FINGERS SPLAYED"                                   . #x1F591)
    ("REVERSED THUMBS UP SIGN"                                                     . #x1F592)
    ("REVERSED THUMBS DOWN SIGN"                                                   . #x1F593)
    ("REVERSED VICTORY HAND"                                                       . #x1F594)
    ("REVERSED HAND WITH MIDDLE FINGER EXTENDED"                                   . #x1F595)
    ("RAISED HAND WITH PART BETWEEN MIDDLE AND RING FINGERS"                       . #x1F596)
    ("WHITE DOWN POINTING LEFT HAND INDEX"                                         . #x1F597)
    ("SIDEWAYS WHITE LEFT POINTING INDEX"                                          . #x1F598)
    ("SIDEWAYS WHITE RIGHT POINTING INDEX"                                         . #x1F599)
    ("SIDEWAYS BLACK LEFT POINTING INDEX"                                          . #x1F59A)
    ("SIDEWAYS BLACK RIGHT POINTING INDEX"                                         . #x1F59B)
    ("BLACK LEFT POINTING BACKHAND INDEX"                                          . #x1F59C)
    ("BLACK RIGHT POINTING BACKHAND INDEX"                                         . #x1F59D)
    ("SIDEWAYS WHITE UP POINTING INDEX"                                            . #x1F59E)
    ("SIDEWAYS WHITE DOWN POINTING INDEX"                                          . #x1F59F)
    ("SIDEWAYS BLACK UP POINTING INDEX"                                            . #x1F5A0)
    ("SIDEWAYS BLACK DOWN POINTING INDEX"                                          . #x1F5A1)
    ("BLACK UP POINTING BACKHAND INDEX"                                            . #x1F5A2)
    ("BLACK DOWN POINTING BACKHAND INDEX"                                          . #x1F5A3)
    ("DESKTOP COMPUTER"                                                            . #x1F5A5)
    ("KEYBOARD AND MOUSE"                                                          . #x1F5A6)
    ("THREE NETWORKED COMPUTERS"                                                   . #x1F5A7)
    ("PRINTER"                                                                     . #x1F5A8)
    ("POCKET CALCULATOR"                                                           . #x1F5A9)
    ("BLACK HARD SHELL FLOPPY DISK"                                                . #x1F5AA)
    ("WHITE HARD SHELL FLOPPY DISK"                                                . #x1F5AB)
    ("SOFT SHELL FLOPPY DISK"                                                      . #x1F5AC)
    ("TAPE CARTRIDGE"                                                              . #x1F5AD)
    ("WIRED KEYBOARD"                                                              . #x1F5AE)
    ("ONE BUTTON MOUSE"                                                            . #x1F5AF)
    ("TWO BUTTON MOUSE"                                                            . #x1F5B0)
    ("THREE BUTTON MOUSE"                                                          . #x1F5B1)
    ("TRACKBALL"                                                                   . #x1F5B2)
    ("OLD PERSONAL COMPUTER"                                                       . #x1F5B3)
    ("HARD DISK"                                                                   . #x1F5B4)
    ("SCREEN"                                                                      . #x1F5B5)
    ("PRINTER ICON"                                                                . #x1F5B6)
    ("FAX ICON"                                                                    . #x1F5B7)
    ("OPTICAL DISC ICON"                                                           . #x1F5B8)
    ("DOCUMENT WITH TEXT"                                                          . #x1F5B9)
    ("DOCUMENT WITH TEXT AND PICTURE"                                              . #x1F5BA)
    ("DOCUMENT WITH PICTURE"                                                       . #x1F5BB)
    ("FRAME WITH PICTURE"                                                          . #x1F5BC)
    ("FRAME WITH TILES"                                                            . #x1F5BD)
    ("FRAME WITH AN X"                                                             . #x1F5BE)
    ("BLACK FOLDER"                                                                . #x1F5BF)
    ("FOLDER"                                                                      . #x1F5C0)
    ("OPEN FOLDER"                                                                 . #x1F5C1)
    ("CARD INDEX DIVIDERS"                                                         . #x1F5C2)
    ("CARD FILE BOX"                                                               . #x1F5C3)
    ("FILE CABINET"                                                                . #x1F5C4)
    ("EMPTY NOTE"                                                                  . #x1F5C5)
    ("EMPTY NOTE PAGE"                                                             . #x1F5C6)
    ("EMPTY NOTE PAD"                                                              . #x1F5C7)
    ("NOTE"                                                                        . #x1F5C8)
    ("NOTE PAGE"                                                                   . #x1F5C9)
    ("NOTE PAD"                                                                    . #x1F5CA)
    ("EMPTY DOCUMENT"                                                              . #x1F5CB)
    ("EMPTY PAGE"                                                                  . #x1F5CC)
    ("EMPTY PAGES"                                                                 . #x1F5CD)
    ("DOCUMENT"                                                                    . #x1F5CE)
    ("PAGE"                                                                        . #x1F5CF)
    ("PAGES"                                                                       . #x1F5D0)
    ("WASTEBASKET"                                                                 . #x1F5D1)
    ("SPIRAL NOTE PAD"                                                             . #x1F5D2)
    ("SPIRAL CALENDAR PAD"                                                         . #x1F5D3)
    ("DESKTOP WINDOW"                                                              . #x1F5D4)
    ("MINIMIZE"                                                                    . #x1F5D5)
    ("MAXIMIZE"                                                                    . #x1F5D6)
    ("OVERLAP"                                                                     . #x1F5D7)
    ("CLOCKWISE RIGHT AND LEFT SEMICIRCLE ARROWS"                                  . #x1F5D8)
    ("CANCELLATION X"                                                              . #x1F5D9)
    ("INCREASE FONT SIZE SYMBOL"                                                   . #x1F5DA)
    ("DECREASE FONT SIZE SYMBOL"                                                   . #x1F5DB)
    ("COMPRESSION"                                                                 . #x1F5DC)
    ("OLD KEY"                                                                     . #x1F5DD)
    ("ROLLED-UP NEWSPAPER"                                                         . #x1F5DE)
    ("PAGE WITH CIRCLED TEXT"                                                      . #x1F5DF)
    ("STOCK CHART"                                                                 . #x1F5E0)
    ("DAGGER KNIFE"                                                                . #x1F5E1)
    ("LIPS"                                                                        . #x1F5E2)
    ("SPEAKING HEAD IN SILHOUETTE"                                                 . #x1F5E3)
    ("THREE RAYS ABOVE"                                                            . #x1F5E4)
    ("THREE RAYS BELOW"                                                            . #x1F5E5)
    ("THREE RAYS LEFT"                                                             . #x1F5E6)
    ("THREE RAYS RIGHT"                                                            . #x1F5E7)
    ("LEFT SPEECH BUBBLE"                                                          . #x1F5E8)
    ("RIGHT SPEECH BUBBLE"                                                         . #x1F5E9)
    ("TWO SPEECH BUBBLES"                                                          . #x1F5EA)
    ("THREE SPEECH BUBBLES"                                                        . #x1F5EB)
    ("LEFT THOUGHT BUBBLE"                                                         . #x1F5EC)
    ("RIGHT THOUGHT BUBBLE"                                                        . #x1F5ED)
    ("LEFT ANGER BUBBLE"                                                           . #x1F5EE)
    ("RIGHT ANGER BUBBLE"                                                          . #x1F5EF)
    ("MOOD BUBBLE"                                                                 . #x1F5F0)
    ("LIGHTNING MOOD BUBBLE"                                                       . #x1F5F1)
    ("LIGHTNING MOOD"                                                              . #x1F5F2)
    ("BALLOT BOX WITH BALLOT"                                                      . #x1F5F3)
    ("BALLOT SCRIPT X"                                                             . #x1F5F4)
    ("BALLOT BOX WITH SCRIPT X"                                                    . #x1F5F5)
    ("BALLOT BOLD SCRIPT X"                                                        . #x1F5F6)
    ("BALLOT BOX WITH BOLD SCRIPT X"                                               . #x1F5F7)
    ("LIGHT CHECK MARK"                                                            . #x1F5F8)
    ("BALLOT BOX WITH BOLD CHECK"                                                  . #x1F5F9)
    ("WORLD MAP"                                                                   . #x1F5FA)
    ("SLIGHTLY FROWNING FACE"                                                      . #x1F641)
    ("SLIGHTLY SMILING FACE"                                                       . #x1F642)
    ("NORTH WEST POINTING LEAF"                                                    . #x1F650)
    ("SOUTH WEST POINTING LEAF"                                                    . #x1F651)
    ("NORTH EAST POINTING LEAF"                                                    . #x1F652)
    ("SOUTH EAST POINTING LEAF"                                                    . #x1F653)
    ("TURNED NORTH WEST POINTING LEAF"                                             . #x1F654)
    ("TURNED SOUTH WEST POINTING LEAF"                                             . #x1F655)
    ("TURNED NORTH EAST POINTING LEAF"                                             . #x1F656)
    ("TURNED SOUTH EAST POINTING LEAF"                                             . #x1F657)
    ("NORTH WEST POINTING VINE LEAF"                                               . #x1F658)
    ("SOUTH WEST POINTING VINE LEAF"                                               . #x1F659)
    ("NORTH EAST POINTING VINE LEAF"                                               . #x1F65A)
    ("SOUTH EAST POINTING VINE LEAF"                                               . #x1F65B)
    ("HEAVY NORTH WEST POINTING VINE LEAF"                                         . #x1F65C)
    ("HEAVY SOUTH WEST POINTING VINE LEAF"                                         . #x1F65D)
    ("HEAVY NORTH EAST POINTING VINE LEAF"                                         . #x1F65E)
    ("HEAVY SOUTH EAST POINTING VINE LEAF"                                         . #x1F65F)
    ("NORTH WEST POINTING BUD"                                                     . #x1F660)
    ("SOUTH WEST POINTING BUD"                                                     . #x1F661)
    ("NORTH EAST POINTING BUD"                                                     . #x1F662)
    ("SOUTH EAST POINTING BUD"                                                     . #x1F663)
    ("HEAVY NORTH WEST POINTING BUD"                                               . #x1F664)
    ("HEAVY SOUTH WEST POINTING BUD"                                               . #x1F665)
    ("HEAVY NORTH EAST POINTING BUD"                                               . #x1F666)
    ("HEAVY SOUTH EAST POINTING BUD"                                               . #x1F667)
    ("HOLLOW QUILT SQUARE ORNAMENT"                                                . #x1F668)
    ("HOLLOW QUILT SQUARE ORNAMENT IN BLACK SQUARE"                                . #x1F669)
    ("SOLID QUILT SQUARE ORNAMENT"                                                 . #x1F66A)
    ("SOLID QUILT SQUARE ORNAMENT IN BLACK SQUARE"                                 . #x1F66B)
    ("LEFTWARDS ROCKET"                                                            . #x1F66C)
    ("UPWARDS ROCKET"                                                              . #x1F66D)
    ("RIGHTWARDS ROCKET"                                                           . #x1F66E)
    ("DOWNWARDS ROCKET"                                                            . #x1F66F)
    ("SCRIPT LIGATURE ET ORNAMENT"                                                 . #x1F670)
    ("HEAVY SCRIPT LIGATURE ET ORNAMENT"                                           . #x1F671)
    ("LIGATURE OPEN ET ORNAMENT"                                                   . #x1F672)
    ("HEAVY LIGATURE OPEN ET ORNAMENT"                                             . #x1F673)
    ("HEAVY AMPERSAND ORNAMENT"                                                    . #x1F674)
    ("SWASH AMPERSAND ORNAMENT"                                                    . #x1F675)
    ("SANS-SERIF HEAVY DOUBLE TURNED COMMA QUOTATION MARK ORNAMENT"                . #x1F676)
    ("SANS-SERIF HEAVY DOUBLE COMMA QUOTATION MARK ORNAMENT"                       . #x1F677)
    ("SANS-SERIF HEAVY LOW DOUBLE COMMA QUOTATION MARK ORNAMENT"                   . #x1F678)
    ("HEAVY INTERROBANG ORNAMENT"                                                  . #x1F679)
    ("SANS-SERIF INTERROBANG ORNAMENT"                                             . #x1F67A)
    ("HEAVY SANS-SERIF INTERROBANG ORNAMENT"                                       . #x1F67B)
    ("VERY HEAVY SOLIDUS"                                                          . #x1F67C)
    ("VERY HEAVY REVERSE SOLIDUS"                                                  . #x1F67D)
    ("CHECKER BOARD"                                                               . #x1F67E)
    ("REVERSE CHECKER BOARD"                                                       . #x1F67F)
    ("TRIANGLE WITH ROUNDED CORNERS"                                               . #x1F6C6)
    ("PROHIBITED SIGN"                                                             . #x1F6C7)
    ("CIRCLED INFORMATION SOURCE"                                                  . #x1F6C8)
    ("BOYS SYMBOL"                                                                 . #x1F6C9)
    ("GIRLS SYMBOL"                                                                . #x1F6CA)
    ("COUCH AND LAMP"                                                              . #x1F6CB)
    ("SLEEPING ACCOMMODATION"                                                      . #x1F6CC)
    ("SHOPPING BAGS"                                                               . #x1F6CD)
    ("BELLHOP BELL"                                                                . #x1F6CE)
    ("BED"                                                                         . #x1F6CF)
    ("HAMMER AND WRENCH"                                                           . #x1F6E0)
    ("SHIELD"                                                                      . #x1F6E1)
    ("OIL DRUM"                                                                    . #x1F6E2)
    ("MOTORWAY"                                                                    . #x1F6E3)
    ("RAILWAY TRACK"                                                               . #x1F6E4)
    ("MOTOR BOAT"                                                                  . #x1F6E5)
    ("UP-POINTING MILITARY AIRPLANE"                                               . #x1F6E6)
    ("UP-POINTING AIRPLANE"                                                        . #x1F6E7)
    ("UP-POINTING SMALL AIRPLANE"                                                  . #x1F6E8)
    ("SMALL AIRPLANE"                                                              . #x1F6E9)
    ("NORTHEAST-POINTING AIRPLANE"                                                 . #x1F6EA)
    ("AIRPLANE DEPARTURE"                                                          . #x1F6EB)
    ("AIRPLANE ARRIVING"                                                           . #x1F6EC)
    ("SATELLITE"                                                                   . #x1F6F0)
    ("ONCOMING FIRE ENGINE"                                                        . #x1F6F1)
    ("DIESEL LOCOMOTIVE"                                                           . #x1F6F2)
    ("PASSENGER SHIP"                                                              . #x1F6F3)
    ("BLACK LEFT-POINTING ISOSCELES RIGHT TRIANGLE"                                . #x1F780)
    ("BLACK UP-POINTING ISOSCELES RIGHT TRIANGLE"                                  . #x1F781)
    ("BLACK RIGHT-POINTING ISOSCELES RIGHT TRIANGLE"                               . #x1F782)
    ("BLACK DOWN-POINTING ISOSCELES RIGHT TRIANGLE"                                . #x1F783)
    ("BLACK SLIGHTLY SMALL CIRCLE"                                                 . #x1F784)
    ("MEDIUM BOLD WHITE CIRCLE"                                                    . #x1F785)
    ("BOLD WHITE CIRCLE"                                                           . #x1F786)
    ("HEAVY WHITE CIRCLE"                                                          . #x1F787)
    ("VERY HEAVY WHITE CIRCLE"                                                     . #x1F788)
    ("EXTREMELY HEAVY WHITE CIRCLE"                                                . #x1F789)
    ("WHITE CIRCLE CONTAINING BLACK SMALL CIRCLE"                                  . #x1F78A)
    ("ROUND TARGET"                                                                . #x1F78B)
    ("BLACK TINY SQUARE"                                                           . #x1F78C)
    ("BLACK SLIGHTLY SMALL SQUARE"                                                 . #x1F78D)
    ("LIGHT WHITE SQUARE"                                                          . #x1F78E)
    ("MEDIUM WHITE SQUARE"                                                         . #x1F78F)
    ("BOLD WHITE SQUARE"                                                           . #x1F790)
    ("HEAVY WHITE SQUARE"                                                          . #x1F791)
    ("VERY HEAVY WHITE SQUARE"                                                     . #x1F792)
    ("EXTREMELY HEAVY WHITE SQUARE"                                                . #x1F793)
    ("WHITE SQUARE CONTAINING BLACK VERY SMALL SQUARE"                             . #x1F794)
    ("WHITE SQUARE CONTAINING BLACK MEDIUM SQUARE"                                 . #x1F795)
    ("SQUARE TARGET"                                                               . #x1F796)
    ("BLACK TINY DIAMOND"                                                          . #x1F797)
    ("BLACK VERY SMALL DIAMOND"                                                    . #x1F798)
    ("BLACK MEDIUM SMALL DIAMOND"                                                  . #x1F799)
    ("WHITE DIAMOND CONTAINING BLACK VERY SMALL DIAMOND"                           . #x1F79A)
    ("WHITE DIAMOND CONTAINING BLACK MEDIUM DIAMOND"                               . #x1F79B)
    ("DIAMOND TARGET"                                                              . #x1F79C)
    ("BLACK TINY LOZENGE"                                                          . #x1F79D)
    ("BLACK VERY SMALL LOZENGE"                                                    . #x1F79E)
    ("BLACK MEDIUM SMALL LOZENGE"                                                  . #x1F79F)
    ("WHITE LOZENGE CONTAINING BLACK SMALL LOZENGE"                                . #x1F7A0)
    ("THIN GREEK CROSS"                                                            . #x1F7A1)
    ("LIGHT GREEK CROSS"                                                           . #x1F7A2)
    ("MEDIUM GREEK CROSS"                                                          . #x1F7A3)
    ("BOLD GREEK CROSS"                                                            . #x1F7A4)
    ("VERY BOLD GREEK CROSS"                                                       . #x1F7A5)
    ("VERY HEAVY GREEK CROSS"                                                      . #x1F7A6)
    ("EXTREMELY HEAVY GREEK CROSS"                                                 . #x1F7A7)
    ("THIN SALTIRE"                                                                . #x1F7A8)
    ("LIGHT SALTIRE"                                                               . #x1F7A9)
    ("MEDIUM SALTIRE"                                                              . #x1F7AA)
    ("BOLD SALTIRE"                                                                . #x1F7AB)
    ("HEAVY SALTIRE"                                                               . #x1F7AC)
    ("VERY HEAVY SALTIRE"                                                          . #x1F7AD)
    ("EXTREMELY HEAVY SALTIRE"                                                     . #x1F7AE)
    ("LIGHT FIVE SPOKED ASTERISK"                                                  . #x1F7AF)
    ("MEDIUM FIVE SPOKED ASTERISK"                                                 . #x1F7B0)
    ("BOLD FIVE SPOKED ASTERISK"                                                   . #x1F7B1)
    ("HEAVY FIVE SPOKED ASTERISK"                                                  . #x1F7B2)
    ("VERY HEAVY FIVE SPOKED ASTERISK"                                             . #x1F7B3)
    ("EXTREMELY HEAVY FIVE SPOKED ASTERISK"                                        . #x1F7B4)
    ("LIGHT SIX SPOKED ASTERISK"                                                   . #x1F7B5)
    ("MEDIUM SIX SPOKED ASTERISK"                                                  . #x1F7B6)
    ("BOLD SIX SPOKED ASTERISK"                                                    . #x1F7B7)
    ("HEAVY SIX SPOKED ASTERISK"                                                   . #x1F7B8)
    ("VERY HEAVY SIX SPOKED ASTERISK"                                              . #x1F7B9)
    ("EXTREMELY HEAVY SIX SPOKED ASTERISK"                                         . #x1F7BA)
    ("LIGHT EIGHT SPOKED ASTERISK"                                                 . #x1F7BB)
    ("MEDIUM EIGHT SPOKED ASTERISK"                                                . #x1F7BC)
    ("BOLD EIGHT SPOKED ASTERISK"                                                  . #x1F7BD)
    ("HEAVY EIGHT SPOKED ASTERISK"                                                 . #x1F7BE)
    ("VERY HEAVY EIGHT SPOKED ASTERISK"                                            . #x1F7BF)
    ("LIGHT THREE POINTED BLACK STAR"                                              . #x1F7C0)
    ("MEDIUM THREE POINTED BLACK STAR"                                             . #x1F7C1)
    ("THREE POINTED BLACK STAR"                                                    . #x1F7C2)
    ("MEDIUM THREE POINTED PINWHEEL STAR"                                          . #x1F7C3)
    ("LIGHT FOUR POINTED BLACK STAR"                                               . #x1F7C4)
    ("MEDIUM FOUR POINTED BLACK STAR"                                              . #x1F7C5)
    ("FOUR POINTED BLACK STAR"                                                     . #x1F7C6)
    ("MEDIUM FOUR POINTED PINWHEEL STAR"                                           . #x1F7C7)
    ("REVERSE LIGHT FOUR POINTED PINWHEEL STAR"                                    . #x1F7C8)
    ("LIGHT FIVE POINTED BLACK STAR"                                               . #x1F7C9)
    ("HEAVY FIVE POINTED BLACK STAR"                                               . #x1F7CA)
    ("MEDIUM SIX POINTED BLACK STAR"                                               . #x1F7CB)
    ("HEAVY SIX POINTED BLACK STAR"                                                . #x1F7CC)
    ("SIX POINTED PINWHEEL STAR"                                                   . #x1F7CD)
    ("MEDIUM EIGHT POINTED BLACK STAR"                                             . #x1F7CE)
    ("HEAVY EIGHT POINTED BLACK STAR"                                              . #x1F7CF)
    ("VERY HEAVY EIGHT POINTED BLACK STAR"                                         . #x1F7D0)
    ("HEAVY EIGHT POINTED PINWHEEL STAR"                                           . #x1F7D1)
    ("LIGHT TWELVE POINTED BLACK STAR"                                             . #x1F7D2)
    ("HEAVY TWELVE POINTED BLACK STAR"                                             . #x1F7D3)
    ("HEAVY TWELVE POINTED PINWHEEL STAR"                                          . #x1F7D4)
    ("LEFTWARDS ARROW WITH SMALL TRIANGLE ARROWHEAD"                               . #x1F800)
    ("UPWARDS ARROW WITH SMALL TRIANGLE ARROWHEAD"                                 . #x1F801)
    ("RIGHTWARDS ARROW WITH SMALL TRIANGLE ARROWHEAD"                              . #x1F802)
    ("DOWNWARDS ARROW WITH SMALL TRIANGLE ARROWHEAD"                               . #x1F803)
    ("LEFTWARDS ARROW WITH MEDIUM TRIANGLE ARROWHEAD"                              . #x1F804)
    ("UPWARDS ARROW WITH MEDIUM TRIANGLE ARROWHEAD"                                . #x1F805)
    ("RIGHTWARDS ARROW WITH MEDIUM TRIANGLE ARROWHEAD"                             . #x1F806)
    ("DOWNWARDS ARROW WITH MEDIUM TRIANGLE ARROWHEAD"                              . #x1F807)
    ("LEFTWARDS ARROW WITH LARGE TRIANGLE ARROWHEAD"                               . #x1F808)
    ("UPWARDS ARROW WITH LARGE TRIANGLE ARROWHEAD"                                 . #x1F809)
    ("RIGHTWARDS ARROW WITH LARGE TRIANGLE ARROWHEAD"                              . #x1F80A)
    ("DOWNWARDS ARROW WITH LARGE TRIANGLE ARROWHEAD"                               . #x1F80B)
    ("LEFTWARDS ARROW WITH SMALL EQUILATERAL ARROWHEAD"                            . #x1F810)
    ("UPWARDS ARROW WITH SMALL EQUILATERAL ARROWHEAD"                              . #x1F811)
    ("RIGHTWARDS ARROW WITH SMALL EQUILATERAL ARROWHEAD"                           . #x1F812)
    ("DOWNWARDS ARROW WITH SMALL EQUILATERAL ARROWHEAD"                            . #x1F813)
    ("LEFTWARDS ARROW WITH EQUILATERAL ARROWHEAD"                                  . #x1F814)
    ("UPWARDS ARROW WITH EQUILATERAL ARROWHEAD"                                    . #x1F815)
    ("RIGHTWARDS ARROW WITH EQUILATERAL ARROWHEAD"                                 . #x1F816)
    ("DOWNWARDS ARROW WITH EQUILATERAL ARROWHEAD"                                  . #x1F817)
    ("HEAVY LEFTWARDS ARROW WITH EQUILATERAL ARROWHEAD"                            . #x1F818)
    ("HEAVY UPWARDS ARROW WITH EQUILATERAL ARROWHEAD"                              . #x1F819)
    ("HEAVY RIGHTWARDS ARROW WITH EQUILATERAL ARROWHEAD"                           . #x1F81A)
    ("HEAVY DOWNWARDS ARROW WITH EQUILATERAL ARROWHEAD"                            . #x1F81B)
    ("HEAVY LEFTWARDS ARROW WITH LARGE EQUILATERAL ARROWHEAD"                      . #x1F81C)
    ("HEAVY UPWARDS ARROW WITH LARGE EQUILATERAL ARROWHEAD"                        . #x1F81D)
    ("HEAVY RIGHTWARDS ARROW WITH LARGE EQUILATERAL ARROWHEAD"                     . #x1F81E)
    ("HEAVY DOWNWARDS ARROW WITH LARGE EQUILATERAL ARROWHEAD"                      . #x1F81F)
    ("LEFTWARDS TRIANGLE-HEADED ARROW WITH NARROW SHAFT"                           . #x1F820)
    ("UPWARDS TRIANGLE-HEADED ARROW WITH NARROW SHAFT"                             . #x1F821)
    ("RIGHTWARDS TRIANGLE-HEADED ARROW WITH NARROW SHAFT"                          . #x1F822)
    ("DOWNWARDS TRIANGLE-HEADED ARROW WITH NARROW SHAFT"                           . #x1F823)
    ("LEFTWARDS TRIANGLE-HEADED ARROW WITH MEDIUM SHAFT"                           . #x1F824)
    ("UPWARDS TRIANGLE-HEADED ARROW WITH MEDIUM SHAFT"                             . #x1F825)
    ("RIGHTWARDS TRIANGLE-HEADED ARROW WITH MEDIUM SHAFT"                          . #x1F826)
    ("DOWNWARDS TRIANGLE-HEADED ARROW WITH MEDIUM SHAFT"                           . #x1F827)
    ("LEFTWARDS TRIANGLE-HEADED ARROW WITH BOLD SHAFT"                             . #x1F828)
    ("UPWARDS TRIANGLE-HEADED ARROW WITH BOLD SHAFT"                               . #x1F829)
    ("RIGHTWARDS TRIANGLE-HEADED ARROW WITH BOLD SHAFT"                            . #x1F82A)
    ("DOWNWARDS TRIANGLE-HEADED ARROW WITH BOLD SHAFT"                             . #x1F82B)
    ("LEFTWARDS TRIANGLE-HEADED ARROW WITH HEAVY SHAFT"                            . #x1F82C)
    ("UPWARDS TRIANGLE-HEADED ARROW WITH HEAVY SHAFT"                              . #x1F82D)
    ("RIGHTWARDS TRIANGLE-HEADED ARROW WITH HEAVY SHAFT"                           . #x1F82E)
    ("DOWNWARDS TRIANGLE-HEADED ARROW WITH HEAVY SHAFT"                            . #x1F82F)
    ("LEFTWARDS TRIANGLE-HEADED ARROW WITH VERY HEAVY SHAFT"                       . #x1F830)
    ("UPWARDS TRIANGLE-HEADED ARROW WITH VERY HEAVY SHAFT"                         . #x1F831)
    ("RIGHTWARDS TRIANGLE-HEADED ARROW WITH VERY HEAVY SHAFT"                      . #x1F832)
    ("DOWNWARDS TRIANGLE-HEADED ARROW WITH VERY HEAVY SHAFT"                       . #x1F833)
    ("LEFTWARDS FINGER-POST ARROW"                                                 . #x1F834)
    ("UPWARDS FINGER-POST ARROW"                                                   . #x1F835)
    ("RIGHTWARDS FINGER-POST ARROW"                                                . #x1F836)
    ("DOWNWARDS FINGER-POST ARROW"                                                 . #x1F837)
    ("LEFTWARDS SQUARED ARROW"                                                     . #x1F838)
    ("UPWARDS SQUARED ARROW"                                                       . #x1F839)
    ("RIGHTWARDS SQUARED ARROW"                                                    . #x1F83A)
    ("DOWNWARDS SQUARED ARROW"                                                     . #x1F83B)
    ("LEFTWARDS COMPRESSED ARROW"                                                  . #x1F83C)
    ("UPWARDS COMPRESSED ARROW"                                                    . #x1F83D)
    ("RIGHTWARDS COMPRESSED ARROW"                                                 . #x1F83E)
    ("DOWNWARDS COMPRESSED ARROW"                                                  . #x1F83F)
    ("LEFTWARDS HEAVY COMPRESSED ARROW"                                            . #x1F840)
    ("UPWARDS HEAVY COMPRESSED ARROW"                                              . #x1F841)
    ("RIGHTWARDS HEAVY COMPRESSED ARROW"                                           . #x1F842)
    ("DOWNWARDS HEAVY COMPRESSED ARROW"                                            . #x1F843)
    ("LEFTWARDS HEAVY ARROW"                                                       . #x1F844)
    ("UPWARDS HEAVY ARROW"                                                         . #x1F845)
    ("RIGHTWARDS HEAVY ARROW"                                                      . #x1F846)
    ("DOWNWARDS HEAVY ARROW"                                                       . #x1F847)
    ("LEFTWARDS SANS-SERIF ARROW"                                                  . #x1F850)
    ("UPWARDS SANS-SERIF ARROW"                                                    . #x1F851)
    ("RIGHTWARDS SANS-SERIF ARROW"                                                 . #x1F852)
    ("DOWNWARDS SANS-SERIF ARROW"                                                  . #x1F853)
    ("NORTH WEST SANS-SERIF ARROW"                                                 . #x1F854)
    ("NORTH EAST SANS-SERIF ARROW"                                                 . #x1F855)
    ("SOUTH EAST SANS-SERIF ARROW"                                                 . #x1F856)
    ("SOUTH WEST SANS-SERIF ARROW"                                                 . #x1F857)
    ("LEFT RIGHT SANS-SERIF ARROW"                                                 . #x1F858)
    ("UP DOWN SANS-SERIF ARROW"                                                    . #x1F859)
    ("WIDE-HEADED LEFTWARDS LIGHT BARB ARROW"                                      . #x1F860)
    ("WIDE-HEADED UPWARDS LIGHT BARB ARROW"                                        . #x1F861)
    ("WIDE-HEADED RIGHTWARDS LIGHT BARB ARROW"                                     . #x1F862)
    ("WIDE-HEADED DOWNWARDS LIGHT BARB ARROW"                                      . #x1F863)
    ("WIDE-HEADED NORTH WEST LIGHT BARB ARROW"                                     . #x1F864)
    ("WIDE-HEADED NORTH EAST LIGHT BARB ARROW"                                     . #x1F865)
    ("WIDE-HEADED SOUTH EAST LIGHT BARB ARROW"                                     . #x1F866)
    ("WIDE-HEADED SOUTH WEST LIGHT BARB ARROW"                                     . #x1F867)
    ("WIDE-HEADED LEFTWARDS BARB ARROW"                                            . #x1F868)
    ("WIDE-HEADED UPWARDS BARB ARROW"                                              . #x1F869)
    ("WIDE-HEADED RIGHTWARDS BARB ARROW"                                           . #x1F86A)
    ("WIDE-HEADED DOWNWARDS BARB ARROW"                                            . #x1F86B)
    ("WIDE-HEADED NORTH WEST BARB ARROW"                                           . #x1F86C)
    ("WIDE-HEADED NORTH EAST BARB ARROW"                                           . #x1F86D)
    ("WIDE-HEADED SOUTH EAST BARB ARROW"                                           . #x1F86E)
    ("WIDE-HEADED SOUTH WEST BARB ARROW"                                           . #x1F86F)
    ("WIDE-HEADED LEFTWARDS MEDIUM BARB ARROW"                                     . #x1F870)
    ("WIDE-HEADED UPWARDS MEDIUM BARB ARROW"                                       . #x1F871)
    ("WIDE-HEADED RIGHTWARDS MEDIUM BARB ARROW"                                    . #x1F872)
    ("WIDE-HEADED DOWNWARDS MEDIUM BARB ARROW"                                     . #x1F873)
    ("WIDE-HEADED NORTH WEST MEDIUM BARB ARROW"                                    . #x1F874)
    ("WIDE-HEADED NORTH EAST MEDIUM BARB ARROW"                                    . #x1F875)
    ("WIDE-HEADED SOUTH EAST MEDIUM BARB ARROW"                                    . #x1F876)
    ("WIDE-HEADED SOUTH WEST MEDIUM BARB ARROW"                                    . #x1F877)
    ("WIDE-HEADED LEFTWARDS HEAVY BARB ARROW"                                      . #x1F878)
    ("WIDE-HEADED UPWARDS HEAVY BARB ARROW"                                        . #x1F879)
    ("WIDE-HEADED RIGHTWARDS HEAVY BARB ARROW"                                     . #x1F87A)
    ("WIDE-HEADED DOWNWARDS HEAVY BARB ARROW"                                      . #x1F87B)
    ("WIDE-HEADED NORTH WEST HEAVY BARB ARROW"                                     . #x1F87C)
    ("WIDE-HEADED NORTH EAST HEAVY BARB ARROW"                                     . #x1F87D)
    ("WIDE-HEADED SOUTH EAST HEAVY BARB ARROW"                                     . #x1F87E)
    ("WIDE-HEADED SOUTH WEST HEAVY BARB ARROW"                                     . #x1F87F)
    ("WIDE-HEADED LEFTWARDS VERY HEAVY BARB ARROW"                                 . #x1F880)
    ("WIDE-HEADED UPWARDS VERY HEAVY BARB ARROW"                                   . #x1F881)
    ("WIDE-HEADED RIGHTWARDS VERY HEAVY BARB ARROW"                                . #x1F882)
    ("WIDE-HEADED DOWNWARDS VERY HEAVY BARB ARROW"                                 . #x1F883)
    ("WIDE-HEADED NORTH WEST VERY HEAVY BARB ARROW"                                . #x1F884)
    ("WIDE-HEADED NORTH EAST VERY HEAVY BARB ARROW"                                . #x1F885)
    ("WIDE-HEADED SOUTH EAST VERY HEAVY BARB ARROW"                                . #x1F886)
    ("WIDE-HEADED SOUTH WEST VERY HEAVY BARB ARROW"                                . #x1F887)
    ("LEFTWARDS TRIANGLE ARROWHEAD"                                                . #x1F890)
    ("UPWARDS TRIANGLE ARROWHEAD"                                                  . #x1F891)
    ("RIGHTWARDS TRIANGLE ARROWHEAD"                                               . #x1F892)
    ("DOWNWARDS TRIANGLE ARROWHEAD"                                                . #x1F893)
    ("LEFTWARDS WHITE ARROW WITHIN TRIANGLE ARROWHEAD"                             . #x1F894)
    ("UPWARDS WHITE ARROW WITHIN TRIANGLE ARROWHEAD"                               . #x1F895)
    ("RIGHTWARDS WHITE ARROW WITHIN TRIANGLE ARROWHEAD"                            . #x1F896)
    ("DOWNWARDS WHITE ARROW WITHIN TRIANGLE ARROWHEAD"                             . #x1F897)
    ("LEFTWARDS ARROW WITH NOTCHED TAIL"                                           . #x1F898)
    ("UPWARDS ARROW WITH NOTCHED TAIL"                                             . #x1F899)
    ("RIGHTWARDS ARROW WITH NOTCHED TAIL"                                          . #x1F89A)
    ("DOWNWARDS ARROW WITH NOTCHED TAIL"                                           . #x1F89B)
    ("HEAVY ARROW SHAFT WIDTH ONE"                                                 . #x1F89C)
    ("HEAVY ARROW SHAFT WIDTH TWO THIRDS"                                          . #x1F89D)
    ("HEAVY ARROW SHAFT WIDTH ONE HALF"                                            . #x1F89E)
    ("HEAVY ARROW SHAFT WIDTH ONE THIRD"                                           . #x1F89F)
    ("LEFTWARDS BOTTOM-SHADED WHITE ARROW"                                         . #x1F8A0)
    ("RIGHTWARDS BOTTOM SHADED WHITE ARROW"                                        . #x1F8A1)
    ("LEFTWARDS TOP SHADED WHITE ARROW"                                            . #x1F8A2)
    ("RIGHTWARDS TOP SHADED WHITE ARROW"                                           . #x1F8A3)
    ("LEFTWARDS LEFT-SHADED WHITE ARROW"                                           . #x1F8A4)
    ("RIGHTWARDS RIGHT-SHADED WHITE ARROW"                                         . #x1F8A5)
    ("LEFTWARDS RIGHT-SHADED WHITE ARROW"                                          . #x1F8A6)
    ("RIGHTWARDS LEFT-SHADED WHITE ARROW"                                          . #x1F8A7)
    ("LEFTWARDS BACK-TILTED SHADOWED WHITE ARROW"                                  . #x1F8A8)
    ("RIGHTWARDS BACK-TILTED SHADOWED WHITE ARROW"                                 . #x1F8A9)
    ("LEFTWARDS FRONT-TILTED SHADOWED WHITE ARROW"                                 . #x1F8AA)
    ("RIGHTWARDS FRONT-TILTED SHADOWED WHITE ARROW"                                . #x1F8AB)
    ("WHITE ARROW SHAFT WIDTH ONE"                                                 . #x1F8AC)
    ("WHITE ARROW SHAFT WIDTH TWO THIRDS"                                          . #x1F8AD)
    ;; Unicode 7.0 to 8.0 delta
    ("ARABIC LETTER AIN WITH THREE DOTS BELOW"                                . #x008B3)
    ("ARABIC LETTER KAF WITH DOT BELOW"                                       . #x008B4)
    ("ARABIC TURNED DAMMA BELOW"                                              . #x008E3)
    ("GUJARATI LETTER ZHA"                                                    . #x00AF9)
    ("TELUGU LETTER RRRA"                                                     . #x00C5A)
    ("MALAYALAM LETTER ARCHAIC II"                                            . #x00D5F)
    ("CHEROKEE LETTER MV"                                                     . #x013F5)
    ("CHEROKEE SMALL LETTER YE"                                               . #x013F8)
    ("CHEROKEE SMALL LETTER YI"                                               . #x013F9)
    ("CHEROKEE SMALL LETTER YO"                                               . #x013FA)
    ("CHEROKEE SMALL LETTER YU"                                               . #x013FB)
    ("CHEROKEE SMALL LETTER YV"                                               . #x013FC)
    ("CHEROKEE SMALL LETTER MV"                                               . #x013FD)
    ("LARI SIGN"                                                              . #x020BE)
    ("TURNED DIGIT TWO"                                                       . #x0218A)
    ("TURNED DIGIT THREE"                                                     . #x0218B)
    ("LEFTWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADS"                    . #x02BEC)
    ("UPWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADS"                      . #x02BED)
    ("RIGHTWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADS"                   . #x02BEE)
    ("DOWNWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADS"                    . #x02BEF)
    ("COMBINING CYRILLIC LETTER EF"                                           . #x0A69E)
    ("LATIN LETTER SINOLOGICAL DOT"                                           . #x0A78F)
    ("LATIN CAPITAL LETTER J WITH CROSSED-TAIL"                               . #x0A7B2)
    ("LATIN CAPITAL LETTER CHI"                                               . #x0A7B3)
    ("LATIN CAPITAL LETTER BETA"                                              . #x0A7B4)
    ("LATIN SMALL LETTER BETA"                                                . #x0A7B5)
    ("LATIN CAPITAL LETTER OMEGA"                                             . #x0A7B6)
    ("LATIN SMALL LETTER OMEGA"                                               . #x0A7B7)
    ("DEVANAGARI SIGN SIDDHAM"                                                . #x0A8FC)
    ("DEVANAGARI JAIN OM"                                                     . #x0A8FD)
    ("LATIN SMALL LETTER SAKHA YAT"                                           . #x0AB60)
    ("LATIN SMALL LETTER IOTIFIED E"                                          . #x0AB61)
    ("LATIN SMALL LETTER OPEN OE"                                             . #x0AB62)
    ("LATIN SMALL LETTER UO"                                                  . #x0AB63)
    ("CHEROKEE SMALL LETTER A"                                                . #x0AB70)
    ("CHEROKEE SMALL LETTER E"                                                . #x0AB71)
    ("CHEROKEE SMALL LETTER I"                                                . #x0AB72)
    ("CHEROKEE SMALL LETTER O"                                                . #x0AB73)
    ("CHEROKEE SMALL LETTER U"                                                . #x0AB74)
    ("CHEROKEE SMALL LETTER V"                                                . #x0AB75)
    ("CHEROKEE SMALL LETTER GA"                                               . #x0AB76)
    ("CHEROKEE SMALL LETTER KA"                                               . #x0AB77)
    ("CHEROKEE SMALL LETTER GE"                                               . #x0AB78)
    ("CHEROKEE SMALL LETTER GI"                                               . #x0AB79)
    ("CHEROKEE SMALL LETTER GO"                                               . #x0AB7A)
    ("CHEROKEE SMALL LETTER GU"                                               . #x0AB7B)
    ("CHEROKEE SMALL LETTER GV"                                               . #x0AB7C)
    ("CHEROKEE SMALL LETTER HA"                                               . #x0AB7D)
    ("CHEROKEE SMALL LETTER HE"                                               . #x0AB7E)
    ("CHEROKEE SMALL LETTER HI"                                               . #x0AB7F)
    ("CHEROKEE SMALL LETTER HO"                                               . #x0AB80)
    ("CHEROKEE SMALL LETTER HU"                                               . #x0AB81)
    ("CHEROKEE SMALL LETTER HV"                                               . #x0AB82)
    ("CHEROKEE SMALL LETTER LA"                                               . #x0AB83)
    ("CHEROKEE SMALL LETTER LE"                                               . #x0AB84)
    ("CHEROKEE SMALL LETTER LI"                                               . #x0AB85)
    ("CHEROKEE SMALL LETTER LO"                                               . #x0AB86)
    ("CHEROKEE SMALL LETTER LU"                                               . #x0AB87)
    ("CHEROKEE SMALL LETTER LV"                                               . #x0AB88)
    ("CHEROKEE SMALL LETTER MA"                                               . #x0AB89)
    ("CHEROKEE SMALL LETTER ME"                                               . #x0AB8A)
    ("CHEROKEE SMALL LETTER MI"                                               . #x0AB8B)
    ("CHEROKEE SMALL LETTER MO"                                               . #x0AB8C)
    ("CHEROKEE SMALL LETTER MU"                                               . #x0AB8D)
    ("CHEROKEE SMALL LETTER NA"                                               . #x0AB8E)
    ("CHEROKEE SMALL LETTER HNA"                                              . #x0AB8F)
    ("CHEROKEE SMALL LETTER NAH"                                              . #x0AB90)
    ("CHEROKEE SMALL LETTER NE"                                               . #x0AB91)
    ("CHEROKEE SMALL LETTER NI"                                               . #x0AB92)
    ("CHEROKEE SMALL LETTER NO"                                               . #x0AB93)
    ("CHEROKEE SMALL LETTER NU"                                               . #x0AB94)
    ("CHEROKEE SMALL LETTER NV"                                               . #x0AB95)
    ("CHEROKEE SMALL LETTER QUA"                                              . #x0AB96)
    ("CHEROKEE SMALL LETTER QUE"                                              . #x0AB97)
    ("CHEROKEE SMALL LETTER QUI"                                              . #x0AB98)
    ("CHEROKEE SMALL LETTER QUO"                                              . #x0AB99)
    ("CHEROKEE SMALL LETTER QUU"                                              . #x0AB9A)
    ("CHEROKEE SMALL LETTER QUV"                                              . #x0AB9B)
    ("CHEROKEE SMALL LETTER SA"                                               . #x0AB9C)
    ("CHEROKEE SMALL LETTER S"                                                . #x0AB9D)
    ("CHEROKEE SMALL LETTER SE"                                               . #x0AB9E)
    ("CHEROKEE SMALL LETTER SI"                                               . #x0AB9F)
    ("CHEROKEE SMALL LETTER SO"                                               . #x0ABA0)
    ("CHEROKEE SMALL LETTER SU"                                               . #x0ABA1)
    ("CHEROKEE SMALL LETTER SV"                                               . #x0ABA2)
    ("CHEROKEE SMALL LETTER DA"                                               . #x0ABA3)
    ("CHEROKEE SMALL LETTER TA"                                               . #x0ABA4)
    ("CHEROKEE SMALL LETTER DE"                                               . #x0ABA5)
    ("CHEROKEE SMALL LETTER TE"                                               . #x0ABA6)
    ("CHEROKEE SMALL LETTER DI"                                               . #x0ABA7)
    ("CHEROKEE SMALL LETTER TI"                                               . #x0ABA8)
    ("CHEROKEE SMALL LETTER DO"                                               . #x0ABA9)
    ("CHEROKEE SMALL LETTER DU"                                               . #x0ABAA)
    ("CHEROKEE SMALL LETTER DV"                                               . #x0ABAB)
    ("CHEROKEE SMALL LETTER DLA"                                              . #x0ABAC)
    ("CHEROKEE SMALL LETTER TLA"                                              . #x0ABAD)
    ("CHEROKEE SMALL LETTER TLE"                                              . #x0ABAE)
    ("CHEROKEE SMALL LETTER TLI"                                              . #x0ABAF)
    ("CHEROKEE SMALL LETTER TLO"                                              . #x0ABB0)
    ("CHEROKEE SMALL LETTER TLU"                                              . #x0ABB1)
    ("CHEROKEE SMALL LETTER TLV"                                              . #x0ABB2)
    ("CHEROKEE SMALL LETTER TSA"                                              . #x0ABB3)
    ("CHEROKEE SMALL LETTER TSE"                                              . #x0ABB4)
    ("CHEROKEE SMALL LETTER TSI"                                              . #x0ABB5)
    ("CHEROKEE SMALL LETTER TSO"                                              . #x0ABB6)
    ("CHEROKEE SMALL LETTER TSU"                                              . #x0ABB7)
    ("CHEROKEE SMALL LETTER TSV"                                              . #x0ABB8)
    ("CHEROKEE SMALL LETTER WA"                                               . #x0ABB9)
    ("CHEROKEE SMALL LETTER WE"                                               . #x0ABBA)
    ("CHEROKEE SMALL LETTER WI"                                               . #x0ABBB)
    ("CHEROKEE SMALL LETTER WO"                                               . #x0ABBC)
    ("CHEROKEE SMALL LETTER WU"                                               . #x0ABBD)
    ("CHEROKEE SMALL LETTER WV"                                               . #x0ABBE)
    ("CHEROKEE SMALL LETTER YA"                                               . #x0ABBF)
    ("COMBINING CYRILLIC TITLO LEFT HALF"                                     . #x0FE2E)
    ("COMBINING CYRILLIC TITLO RIGHT HALF"                                    . #x0FE2F)
    ("HATRAN LETTER ALEPH"                                                    . #x108E0)
    ("HATRAN LETTER BETH"                                                     . #x108E1)
    ("HATRAN LETTER GIMEL"                                                    . #x108E2)
    ("HATRAN LETTER DALETH-RESH"                                              . #x108E3)
    ("HATRAN LETTER HE"                                                       . #x108E4)
    ("HATRAN LETTER WAW"                                                      . #x108E5)
    ("HATRAN LETTER ZAYN"                                                     . #x108E6)
    ("HATRAN LETTER HETH"                                                     . #x108E7)
    ("HATRAN LETTER TETH"                                                     . #x108E8)
    ("HATRAN LETTER YODH"                                                     . #x108E9)
    ("HATRAN LETTER KAPH"                                                     . #x108EA)
    ("HATRAN LETTER LAMEDH"                                                   . #x108EB)
    ("HATRAN LETTER MEM"                                                      . #x108EC)
    ("HATRAN LETTER NUN"                                                      . #x108ED)
    ("HATRAN LETTER SAMEKH"                                                   . #x108EE)
    ("HATRAN LETTER AYN"                                                      . #x108EF)
    ("HATRAN LETTER PE"                                                       . #x108F0)
    ("HATRAN LETTER SADHE"                                                    . #x108F1)
    ("HATRAN LETTER QOPH"                                                     . #x108F2)
    ("HATRAN LETTER SHIN"                                                     . #x108F4)
    ("HATRAN LETTER TAW"                                                      . #x108F5)
    ("HATRAN NUMBER ONE"                                                      . #x108FB)
    ("HATRAN NUMBER FIVE"                                                     . #x108FC)
    ("HATRAN NUMBER TEN"                                                      . #x108FD)
    ("HATRAN NUMBER TWENTY"                                                   . #x108FE)
    ("HATRAN NUMBER ONE HUNDRED"                                              . #x108FF)
    ("MEROITIC CURSIVE FRACTION ELEVEN TWELFTHS"                              . #x109BC)
    ("MEROITIC CURSIVE FRACTION ONE HALF"                                     . #x109BD)
    ("MEROITIC CURSIVE NUMBER ONE"                                            . #x109C0)
    ("MEROITIC CURSIVE NUMBER TWO"                                            . #x109C1)
    ("MEROITIC CURSIVE NUMBER THREE"                                          . #x109C2)
    ("MEROITIC CURSIVE NUMBER FOUR"                                           . #x109C3)
    ("MEROITIC CURSIVE NUMBER FIVE"                                           . #x109C4)
    ("MEROITIC CURSIVE NUMBER SIX"                                            . #x109C5)
    ("MEROITIC CURSIVE NUMBER SEVEN"                                          . #x109C6)
    ("MEROITIC CURSIVE NUMBER EIGHT"                                          . #x109C7)
    ("MEROITIC CURSIVE NUMBER NINE"                                           . #x109C8)
    ("MEROITIC CURSIVE NUMBER TEN"                                            . #x109C9)
    ("MEROITIC CURSIVE NUMBER TWENTY"                                         . #x109CA)
    ("MEROITIC CURSIVE NUMBER THIRTY"                                         . #x109CB)
    ("MEROITIC CURSIVE NUMBER FORTY"                                          . #x109CC)
    ("MEROITIC CURSIVE NUMBER FIFTY"                                          . #x109CD)
    ("MEROITIC CURSIVE NUMBER SIXTY"                                          . #x109CE)
    ("MEROITIC CURSIVE NUMBER SEVENTY"                                        . #x109CF)
    ("MEROITIC CURSIVE NUMBER ONE HUNDRED"                                    . #x109D2)
    ("MEROITIC CURSIVE NUMBER TWO HUNDRED"                                    . #x109D3)
    ("MEROITIC CURSIVE NUMBER THREE HUNDRED"                                  . #x109D4)
    ("MEROITIC CURSIVE NUMBER FOUR HUNDRED"                                   . #x109D5)
    ("MEROITIC CURSIVE NUMBER FIVE HUNDRED"                                   . #x109D6)
    ("MEROITIC CURSIVE NUMBER SIX HUNDRED"                                    . #x109D7)
    ("MEROITIC CURSIVE NUMBER SEVEN HUNDRED"                                  . #x109D8)
    ("MEROITIC CURSIVE NUMBER EIGHT HUNDRED"                                  . #x109D9)
    ("MEROITIC CURSIVE NUMBER NINE HUNDRED"                                   . #x109DA)
    ("MEROITIC CURSIVE NUMBER ONE THOUSAND"                                   . #x109DB)
    ("MEROITIC CURSIVE NUMBER TWO THOUSAND"                                   . #x109DC)
    ("MEROITIC CURSIVE NUMBER THREE THOUSAND"                                 . #x109DD)
    ("MEROITIC CURSIVE NUMBER FOUR THOUSAND"                                  . #x109DE)
    ("MEROITIC CURSIVE NUMBER FIVE THOUSAND"                                  . #x109DF)
    ("MEROITIC CURSIVE NUMBER SIX THOUSAND"                                   . #x109E0)
    ("MEROITIC CURSIVE NUMBER SEVEN THOUSAND"                                 . #x109E1)
    ("MEROITIC CURSIVE NUMBER EIGHT THOUSAND"                                 . #x109E2)
    ("MEROITIC CURSIVE NUMBER NINE THOUSAND"                                  . #x109E3)
    ("MEROITIC CURSIVE NUMBER TEN THOUSAND"                                   . #x109E4)
    ("MEROITIC CURSIVE NUMBER TWENTY THOUSAND"                                . #x109E5)
    ("MEROITIC CURSIVE NUMBER THIRTY THOUSAND"                                . #x109E6)
    ("MEROITIC CURSIVE NUMBER FORTY THOUSAND"                                 . #x109E7)
    ("MEROITIC CURSIVE NUMBER FIFTY THOUSAND"                                 . #x109E8)
    ("MEROITIC CURSIVE NUMBER SIXTY THOUSAND"                                 . #x109E9)
    ("MEROITIC CURSIVE NUMBER SEVENTY THOUSAND"                               . #x109EA)
    ("MEROITIC CURSIVE NUMBER EIGHTY THOUSAND"                                . #x109EB)
    ("MEROITIC CURSIVE NUMBER NINETY THOUSAND"                                . #x109EC)
    ("MEROITIC CURSIVE NUMBER ONE HUNDRED THOUSAND"                           . #x109ED)
    ("MEROITIC CURSIVE NUMBER TWO HUNDRED THOUSAND"                           . #x109EE)
    ("MEROITIC CURSIVE NUMBER THREE HUNDRED THOUSAND"                         . #x109EF)
    ("MEROITIC CURSIVE NUMBER FOUR HUNDRED THOUSAND"                          . #x109F0)
    ("MEROITIC CURSIVE NUMBER FIVE HUNDRED THOUSAND"                          . #x109F1)
    ("MEROITIC CURSIVE NUMBER SIX HUNDRED THOUSAND"                           . #x109F2)
    ("MEROITIC CURSIVE NUMBER SEVEN HUNDRED THOUSAND"                         . #x109F3)
    ("MEROITIC CURSIVE NUMBER EIGHT HUNDRED THOUSAND"                         . #x109F4)
    ("MEROITIC CURSIVE NUMBER NINE HUNDRED THOUSAND"                          . #x109F5)
    ("MEROITIC CURSIVE FRACTION ONE TWELFTH"                                  . #x109F6)
    ("MEROITIC CURSIVE FRACTION TWO TWELFTHS"                                 . #x109F7)
    ("MEROITIC CURSIVE FRACTION THREE TWELFTHS"                               . #x109F8)
    ("MEROITIC CURSIVE FRACTION FOUR TWELFTHS"                                . #x109F9)
    ("MEROITIC CURSIVE FRACTION FIVE TWELFTHS"                                . #x109FA)
    ("MEROITIC CURSIVE FRACTION SIX TWELFTHS"                                 . #x109FB)
    ("MEROITIC CURSIVE FRACTION SEVEN TWELFTHS"                               . #x109FC)
    ("MEROITIC CURSIVE FRACTION EIGHT TWELFTHS"                               . #x109FD)
    ("MEROITIC CURSIVE FRACTION NINE TWELFTHS"                                . #x109FE)
    ("MEROITIC CURSIVE FRACTION TEN TWELFTHS"                                 . #x109FF)
    ("OLD HUNGARIAN CAPITAL LETTER A"                                         . #x10C80)
    ("OLD HUNGARIAN CAPITAL LETTER AA"                                        . #x10C81)
    ("OLD HUNGARIAN CAPITAL LETTER EB"                                        . #x10C82)
    ("OLD HUNGARIAN CAPITAL LETTER AMB"                                       . #x10C83)
    ("OLD HUNGARIAN CAPITAL LETTER EC"                                        . #x10C84)
    ("OLD HUNGARIAN CAPITAL LETTER ENC"                                       . #x10C85)
    ("OLD HUNGARIAN CAPITAL LETTER ECS"                                       . #x10C86)
    ("OLD HUNGARIAN CAPITAL LETTER ED"                                        . #x10C87)
    ("OLD HUNGARIAN CAPITAL LETTER AND"                                       . #x10C88)
    ("OLD HUNGARIAN CAPITAL LETTER E"                                         . #x10C89)
    ("OLD HUNGARIAN CAPITAL LETTER CLOSE E"                                   . #x10C8A)
    ("OLD HUNGARIAN CAPITAL LETTER EE"                                        . #x10C8B)
    ("OLD HUNGARIAN CAPITAL LETTER EF"                                        . #x10C8C)
    ("OLD HUNGARIAN CAPITAL LETTER EG"                                        . #x10C8D)
    ("OLD HUNGARIAN CAPITAL LETTER EGY"                                       . #x10C8E)
    ("OLD HUNGARIAN CAPITAL LETTER EH"                                        . #x10C8F)
    ("OLD HUNGARIAN CAPITAL LETTER I"                                         . #x10C90)
    ("OLD HUNGARIAN CAPITAL LETTER II"                                        . #x10C91)
    ("OLD HUNGARIAN CAPITAL LETTER EJ"                                        . #x10C92)
    ("OLD HUNGARIAN CAPITAL LETTER EK"                                        . #x10C93)
    ("OLD HUNGARIAN CAPITAL LETTER AK"                                        . #x10C94)
    ("OLD HUNGARIAN CAPITAL LETTER UNK"                                       . #x10C95)
    ("OLD HUNGARIAN CAPITAL LETTER EL"                                        . #x10C96)
    ("OLD HUNGARIAN CAPITAL LETTER ELY"                                       . #x10C97)
    ("OLD HUNGARIAN CAPITAL LETTER EM"                                        . #x10C98)
    ("OLD HUNGARIAN CAPITAL LETTER EN"                                        . #x10C99)
    ("OLD HUNGARIAN CAPITAL LETTER ENY"                                       . #x10C9A)
    ("OLD HUNGARIAN CAPITAL LETTER O"                                         . #x10C9B)
    ("OLD HUNGARIAN CAPITAL LETTER OO"                                        . #x10C9C)
    ("OLD HUNGARIAN CAPITAL LETTER NIKOLSBURG OE"                             . #x10C9D)
    ("OLD HUNGARIAN CAPITAL LETTER RUDIMENTA OE"                              . #x10C9E)
    ("OLD HUNGARIAN CAPITAL LETTER OEE"                                       . #x10C9F)
    ("OLD HUNGARIAN CAPITAL LETTER EP"                                        . #x10CA0)
    ("OLD HUNGARIAN CAPITAL LETTER EMP"                                       . #x10CA1)
    ("OLD HUNGARIAN CAPITAL LETTER ER"                                        . #x10CA2)
    ("OLD HUNGARIAN CAPITAL LETTER SHORT ER"                                  . #x10CA3)
    ("OLD HUNGARIAN CAPITAL LETTER ES"                                        . #x10CA4)
    ("OLD HUNGARIAN CAPITAL LETTER ESZ"                                       . #x10CA5)
    ("OLD HUNGARIAN CAPITAL LETTER ET"                                        . #x10CA6)
    ("OLD HUNGARIAN CAPITAL LETTER ENT"                                       . #x10CA7)
    ("OLD HUNGARIAN CAPITAL LETTER ETY"                                       . #x10CA8)
    ("OLD HUNGARIAN CAPITAL LETTER ECH"                                       . #x10CA9)
    ("OLD HUNGARIAN CAPITAL LETTER U"                                         . #x10CAA)
    ("OLD HUNGARIAN CAPITAL LETTER UU"                                        . #x10CAB)
    ("OLD HUNGARIAN CAPITAL LETTER NIKOLSBURG UE"                             . #x10CAC)
    ("OLD HUNGARIAN CAPITAL LETTER RUDIMENTA UE"                              . #x10CAD)
    ("OLD HUNGARIAN CAPITAL LETTER EV"                                        . #x10CAE)
    ("OLD HUNGARIAN CAPITAL LETTER EZ"                                        . #x10CAF)
    ("OLD HUNGARIAN CAPITAL LETTER EZS"                                       . #x10CB0)
    ("OLD HUNGARIAN CAPITAL LETTER ENT-SHAPED SIGN"                           . #x10CB1)
    ("OLD HUNGARIAN CAPITAL LETTER US"                                        . #x10CB2)
    ("OLD HUNGARIAN SMALL LETTER A"                                           . #x10CC0)
    ("OLD HUNGARIAN SMALL LETTER AA"                                          . #x10CC1)
    ("OLD HUNGARIAN SMALL LETTER EB"                                          . #x10CC2)
    ("OLD HUNGARIAN SMALL LETTER AMB"                                         . #x10CC3)
    ("OLD HUNGARIAN SMALL LETTER EC"                                          . #x10CC4)
    ("OLD HUNGARIAN SMALL LETTER ENC"                                         . #x10CC5)
    ("OLD HUNGARIAN SMALL LETTER ECS"                                         . #x10CC6)
    ("OLD HUNGARIAN SMALL LETTER ED"                                          . #x10CC7)
    ("OLD HUNGARIAN SMALL LETTER AND"                                         . #x10CC8)
    ("OLD HUNGARIAN SMALL LETTER E"                                           . #x10CC9)
    ("OLD HUNGARIAN SMALL LETTER CLOSE E"                                     . #x10CCA)
    ("OLD HUNGARIAN SMALL LETTER EE"                                          . #x10CCB)
    ("OLD HUNGARIAN SMALL LETTER EF"                                          . #x10CCC)
    ("OLD HUNGARIAN SMALL LETTER EG"                                          . #x10CCD)
    ("OLD HUNGARIAN SMALL LETTER EGY"                                         . #x10CCE)
    ("OLD HUNGARIAN SMALL LETTER EH"                                          . #x10CCF)
    ("OLD HUNGARIAN SMALL LETTER I"                                           . #x10CD0)
    ("OLD HUNGARIAN SMALL LETTER II"                                          . #x10CD1)
    ("OLD HUNGARIAN SMALL LETTER EJ"                                          . #x10CD2)
    ("OLD HUNGARIAN SMALL LETTER EK"                                          . #x10CD3)
    ("OLD HUNGARIAN SMALL LETTER AK"                                          . #x10CD4)
    ("OLD HUNGARIAN SMALL LETTER UNK"                                         . #x10CD5)
    ("OLD HUNGARIAN SMALL LETTER EL"                                          . #x10CD6)
    ("OLD HUNGARIAN SMALL LETTER ELY"                                         . #x10CD7)
    ("OLD HUNGARIAN SMALL LETTER EM"                                          . #x10CD8)
    ("OLD HUNGARIAN SMALL LETTER EN"                                          . #x10CD9)
    ("OLD HUNGARIAN SMALL LETTER ENY"                                         . #x10CDA)
    ("OLD HUNGARIAN SMALL LETTER O"                                           . #x10CDB)
    ("OLD HUNGARIAN SMALL LETTER OO"                                          . #x10CDC)
    ("OLD HUNGARIAN SMALL LETTER NIKOLSBURG OE"                               . #x10CDD)
    ("OLD HUNGARIAN SMALL LETTER RUDIMENTA OE"                                . #x10CDE)
    ("OLD HUNGARIAN SMALL LETTER OEE"                                         . #x10CDF)
    ("OLD HUNGARIAN SMALL LETTER EP"                                          . #x10CE0)
    ("OLD HUNGARIAN SMALL LETTER EMP"                                         . #x10CE1)
    ("OLD HUNGARIAN SMALL LETTER ER"                                          . #x10CE2)
    ("OLD HUNGARIAN SMALL LETTER SHORT ER"                                    . #x10CE3)
    ("OLD HUNGARIAN SMALL LETTER ES"                                          . #x10CE4)
    ("OLD HUNGARIAN SMALL LETTER ESZ"                                         . #x10CE5)
    ("OLD HUNGARIAN SMALL LETTER ET"                                          . #x10CE6)
    ("OLD HUNGARIAN SMALL LETTER ENT"                                         . #x10CE7)
    ("OLD HUNGARIAN SMALL LETTER ETY"                                         . #x10CE8)
    ("OLD HUNGARIAN SMALL LETTER ECH"                                         . #x10CE9)
    ("OLD HUNGARIAN SMALL LETTER U"                                           . #x10CEA)
    ("OLD HUNGARIAN SMALL LETTER UU"                                          . #x10CEB)
    ("OLD HUNGARIAN SMALL LETTER NIKOLSBURG UE"                               . #x10CEC)
    ("OLD HUNGARIAN SMALL LETTER RUDIMENTA UE"                                . #x10CED)
    ("OLD HUNGARIAN SMALL LETTER EV"                                          . #x10CEE)
    ("OLD HUNGARIAN SMALL LETTER EZ"                                          . #x10CEF)
    ("OLD HUNGARIAN SMALL LETTER EZS"                                         . #x10CF0)
    ("OLD HUNGARIAN SMALL LETTER ENT-SHAPED SIGN"                             . #x10CF1)
    ("OLD HUNGARIAN SMALL LETTER US"                                          . #x10CF2)
    ("OLD HUNGARIAN NUMBER ONE"                                               . #x10CFA)
    ("OLD HUNGARIAN NUMBER FIVE"                                              . #x10CFB)
    ("OLD HUNGARIAN NUMBER TEN"                                               . #x10CFC)
    ("OLD HUNGARIAN NUMBER FIFTY"                                             . #x10CFD)
    ("OLD HUNGARIAN NUMBER ONE HUNDRED"                                       . #x10CFE)
    ("OLD HUNGARIAN NUMBER ONE THOUSAND"                                      . #x10CFF)
    ("SHARADA SANDHI MARK"                                                    . #x111C9)
    ("SHARADA SIGN NUKTA"                                                     . #x111CA)
    ("SHARADA VOWEL MODIFIER MARK"                                            . #x111CB)
    ("SHARADA EXTRA SHORT VOWEL MARK"                                         . #x111CC)
    ("SHARADA SIGN SIDDHAM"                                                   . #x111DB)
    ("SHARADA HEADSTROKE"                                                     . #x111DC)
    ("SHARADA CONTINUATION SIGN"                                              . #x111DD)
    ("SHARADA SECTION MARK-1"                                                 . #x111DE)
    ("SHARADA SECTION MARK-2"                                                 . #x111DF)
    ("MULTANI LETTER A"                                                       . #x11280)
    ("MULTANI LETTER I"                                                       . #x11281)
    ("MULTANI LETTER U"                                                       . #x11282)
    ("MULTANI LETTER E"                                                       . #x11283)
    ("MULTANI LETTER KA"                                                      . #x11284)
    ("MULTANI LETTER KHA"                                                     . #x11285)
    ("MULTANI LETTER GA"                                                      . #x11286)
    ("MULTANI LETTER GHA"                                                     . #x11288)
    ("MULTANI LETTER CA"                                                      . #x1128A)
    ("MULTANI LETTER CHA"                                                     . #x1128B)
    ("MULTANI LETTER JA"                                                      . #x1128C)
    ("MULTANI LETTER JJA"                                                     . #x1128D)
    ("MULTANI LETTER NYA"                                                     . #x1128F)
    ("MULTANI LETTER TTA"                                                     . #x11290)
    ("MULTANI LETTER TTHA"                                                    . #x11291)
    ("MULTANI LETTER DDA"                                                     . #x11292)
    ("MULTANI LETTER DDDA"                                                    . #x11293)
    ("MULTANI LETTER DDHA"                                                    . #x11294)
    ("MULTANI LETTER NNA"                                                     . #x11295)
    ("MULTANI LETTER TA"                                                      . #x11296)
    ("MULTANI LETTER THA"                                                     . #x11297)
    ("MULTANI LETTER DA"                                                      . #x11298)
    ("MULTANI LETTER DHA"                                                     . #x11299)
    ("MULTANI LETTER NA"                                                      . #x1129A)
    ("MULTANI LETTER PA"                                                      . #x1129B)
    ("MULTANI LETTER PHA"                                                     . #x1129C)
    ("MULTANI LETTER BA"                                                      . #x1129D)
    ("MULTANI LETTER BHA"                                                     . #x1129F)
    ("MULTANI LETTER MA"                                                      . #x112A0)
    ("MULTANI LETTER YA"                                                      . #x112A1)
    ("MULTANI LETTER RA"                                                      . #x112A2)
    ("MULTANI LETTER LA"                                                      . #x112A3)
    ("MULTANI LETTER VA"                                                      . #x112A4)
    ("MULTANI LETTER SA"                                                      . #x112A5)
    ("MULTANI LETTER HA"                                                      . #x112A6)
    ("MULTANI LETTER RRA"                                                     . #x112A7)
    ("MULTANI LETTER RHA"                                                     . #x112A8)
    ("MULTANI SECTION MARK"                                                   . #x112A9)
    ("GRANTHA SIGN COMBINING ANUSVARA ABOVE"                                  . #x11300)
    ("GRANTHA OM"                                                             . #x11350)
    ("SIDDHAM SECTION MARK WITH TRIDENT AND U-SHAPED ORNAMENTS"               . #x115CA)
    ("SIDDHAM SECTION MARK WITH TRIDENT AND DOTTED CRESCENTS"                 . #x115CB)
    ("SIDDHAM SECTION MARK WITH RAYS AND DOTTED CRESCENTS"                    . #x115CC)
    ("SIDDHAM SECTION MARK WITH RAYS AND DOTTED DOUBLE CRESCENTS"             . #x115CD)
    ("SIDDHAM SECTION MARK WITH RAYS AND DOTTED TRIPLE CRESCENTS"             . #x115CE)
    ("SIDDHAM SECTION MARK DOUBLE RING"                                       . #x115CF)
    ("SIDDHAM SECTION MARK DOUBLE RING WITH RAYS"                             . #x115D0)
    ("SIDDHAM SECTION MARK WITH DOUBLE CRESCENTS"                             . #x115D1)
    ("SIDDHAM SECTION MARK WITH TRIPLE CRESCENTS"                             . #x115D2)
    ("SIDDHAM SECTION MARK WITH QUADRUPLE CRESCENTS"                          . #x115D3)
    ("SIDDHAM SECTION MARK WITH SEPTUPLE CRESCENTS"                           . #x115D4)
    ("SIDDHAM SECTION MARK WITH CIRCLES AND RAYS"                             . #x115D5)
    ("SIDDHAM SECTION MARK WITH CIRCLES AND TWO ENCLOSURES"                   . #x115D6)
    ("SIDDHAM SECTION MARK WITH CIRCLES AND FOUR ENCLOSURES"                  . #x115D7)
    ("SIDDHAM LETTER THREE-CIRCLE ALTERNATE I"                                . #x115D8)
    ("SIDDHAM LETTER TWO-CIRCLE ALTERNATE I"                                  . #x115D9)
    ("SIDDHAM LETTER TWO-CIRCLE ALTERNATE II"                                 . #x115DA)
    ("SIDDHAM LETTER ALTERNATE U"                                             . #x115DB)
    ("SIDDHAM VOWEL SIGN ALTERNATE U"                                         . #x115DC)
    ("SIDDHAM VOWEL SIGN ALTERNATE UU"                                        . #x115DD)
    ("AHOM LETTER KA"                                                         . #x11700)
    ("AHOM LETTER KHA"                                                        . #x11701)
    ("AHOM LETTER NGA"                                                        . #x11702)
    ("AHOM LETTER NA"                                                         . #x11703)
    ("AHOM LETTER TA"                                                         . #x11704)
    ("AHOM LETTER ALTERNATE TA"                                               . #x11705)
    ("AHOM LETTER PA"                                                         . #x11706)
    ("AHOM LETTER PHA"                                                        . #x11707)
    ("AHOM LETTER BA"                                                         . #x11708)
    ("AHOM LETTER MA"                                                         . #x11709)
    ("AHOM LETTER JA"                                                         . #x1170A)
    ("AHOM LETTER CHA"                                                        . #x1170B)
    ("AHOM LETTER THA"                                                        . #x1170C)
    ("AHOM LETTER RA"                                                         . #x1170D)
    ("AHOM LETTER LA"                                                         . #x1170E)
    ("AHOM LETTER SA"                                                         . #x1170F)
    ("AHOM LETTER NYA"                                                        . #x11710)
    ("AHOM LETTER HA"                                                         . #x11711)
    ("AHOM LETTER A"                                                          . #x11712)
    ("AHOM LETTER DA"                                                         . #x11713)
    ("AHOM LETTER DHA"                                                        . #x11714)
    ("AHOM LETTER GA"                                                         . #x11715)
    ("AHOM LETTER ALTERNATE GA"                                               . #x11716)
    ("AHOM LETTER GHA"                                                        . #x11717)
    ("AHOM LETTER BHA"                                                        . #x11718)
    ("AHOM LETTER JHA"                                                        . #x11719)
    ("AHOM CONSONANT SIGN MEDIAL LA"                                          . #x1171D)
    ("AHOM CONSONANT SIGN MEDIAL RA"                                          . #x1171E)
    ("AHOM CONSONANT SIGN MEDIAL LIGATING RA"                                 . #x1171F)
    ("AHOM VOWEL SIGN A"                                                      . #x11720)
    ("AHOM VOWEL SIGN AA"                                                     . #x11721)
    ("AHOM VOWEL SIGN I"                                                      . #x11722)
    ("AHOM VOWEL SIGN II"                                                     . #x11723)
    ("AHOM VOWEL SIGN U"                                                      . #x11724)
    ("AHOM VOWEL SIGN UU"                                                     . #x11725)
    ("AHOM VOWEL SIGN E"                                                      . #x11726)
    ("AHOM VOWEL SIGN AW"                                                     . #x11727)
    ("AHOM VOWEL SIGN O"                                                      . #x11728)
    ("AHOM VOWEL SIGN AI"                                                     . #x11729)
    ("AHOM VOWEL SIGN AM"                                                     . #x1172A)
    ("AHOM SIGN KILLER"                                                       . #x1172B)
    ("AHOM DIGIT ZERO"                                                        . #x11730)
    ("AHOM DIGIT ONE"                                                         . #x11731)
    ("AHOM DIGIT TWO"                                                         . #x11732)
    ("AHOM DIGIT THREE"                                                       . #x11733)
    ("AHOM DIGIT FOUR"                                                        . #x11734)
    ("AHOM DIGIT FIVE"                                                        . #x11735)
    ("AHOM DIGIT SIX"                                                         . #x11736)
    ("AHOM DIGIT SEVEN"                                                       . #x11737)
    ("AHOM DIGIT EIGHT"                                                       . #x11738)
    ("AHOM DIGIT NINE"                                                        . #x11739)
    ("AHOM NUMBER TEN"                                                        . #x1173A)
    ("AHOM NUMBER TWENTY"                                                     . #x1173B)
    ("AHOM SIGN SMALL SECTION"                                                . #x1173C)
    ("AHOM SIGN SECTION"                                                      . #x1173D)
    ("AHOM SIGN RULAI"                                                        . #x1173E)
    ("AHOM SYMBOL VI"                                                         . #x1173F)
    ("CUNEIFORM SIGN U U"                                                     . #x12399)
    ("CUNEIFORM SIGN AB TIMES NUN TENU"                                       . #x12480)
    ("CUNEIFORM SIGN AB TIMES SHU2"                                           . #x12481)
    ("CUNEIFORM SIGN AD TIMES ESH2"                                           . #x12482)
    ("CUNEIFORM SIGN BAD TIMES DISH TENU"                                     . #x12483)
    ("CUNEIFORM SIGN BAHAR2 TIMES AB2"                                        . #x12484)
    ("CUNEIFORM SIGN BAHAR2 TIMES NI"                                         . #x12485)
    ("CUNEIFORM SIGN BAHAR2 TIMES ZA"                                         . #x12486)
    ("CUNEIFORM SIGN BU OVER BU TIMES NA2"                                    . #x12487)
    ("CUNEIFORM SIGN DA TIMES TAK4"                                           . #x12488)
    ("CUNEIFORM SIGN DAG TIMES KUR"                                           . #x12489)
    ("CUNEIFORM SIGN DIM TIMES IGI"                                           . #x1248A)
    ("CUNEIFORM SIGN DIM TIMES U U U"                                         . #x1248B)
    ("CUNEIFORM SIGN DIM2 TIMES UD"                                           . #x1248C)
    ("CUNEIFORM SIGN DUG TIMES ANSHE"                                         . #x1248D)
    ("CUNEIFORM SIGN DUG TIMES ASH"                                           . #x1248E)
    ("CUNEIFORM SIGN DUG TIMES ASH AT LEFT"                                   . #x1248F)
    ("CUNEIFORM SIGN DUG TIMES DIN"                                           . #x12490)
    ("CUNEIFORM SIGN DUG TIMES DUN"                                           . #x12491)
    ("CUNEIFORM SIGN DUG TIMES ERIN2"                                         . #x12492)
    ("CUNEIFORM SIGN DUG TIMES GA"                                            . #x12493)
    ("CUNEIFORM SIGN DUG TIMES GI"                                            . #x12494)
    ("CUNEIFORM SIGN DUG TIMES GIR2 GUNU"                                     . #x12495)
    ("CUNEIFORM SIGN DUG TIMES GISH"                                          . #x12496)
    ("CUNEIFORM SIGN DUG TIMES HA"                                            . #x12497)
    ("CUNEIFORM SIGN DUG TIMES HI"                                            . #x12498)
    ("CUNEIFORM SIGN DUG TIMES IGI GUNU"                                      . #x12499)
    ("CUNEIFORM SIGN DUG TIMES KASKAL"                                        . #x1249A)
    ("CUNEIFORM SIGN DUG TIMES KUR"                                           . #x1249B)
    ("CUNEIFORM SIGN DUG TIMES KUSHU2"                                        . #x1249C)
    ("CUNEIFORM SIGN DUG TIMES KUSHU2 PLUS KASKAL"                            . #x1249D)
    ("CUNEIFORM SIGN DUG TIMES LAK-020"                                       . #x1249E)
    ("CUNEIFORM SIGN DUG TIMES LAM"                                           . #x1249F)
    ("CUNEIFORM SIGN DUG TIMES LAM TIMES KUR"                                 . #x124A0)
    ("CUNEIFORM SIGN DUG TIMES LUH PLUS GISH"                                 . #x124A1)
    ("CUNEIFORM SIGN DUG TIMES MASH"                                          . #x124A2)
    ("CUNEIFORM SIGN DUG TIMES MES"                                           . #x124A3)
    ("CUNEIFORM SIGN DUG TIMES MI"                                            . #x124A4)
    ("CUNEIFORM SIGN DUG TIMES NI"                                            . #x124A5)
    ("CUNEIFORM SIGN DUG TIMES PI"                                            . #x124A6)
    ("CUNEIFORM SIGN DUG TIMES SHE"                                           . #x124A7)
    ("CUNEIFORM SIGN DUG TIMES SI GUNU"                                       . #x124A8)
    ("CUNEIFORM SIGN E2 TIMES KUR"                                            . #x124A9)
    ("CUNEIFORM SIGN E2 TIMES PAP"                                            . #x124AA)
    ("CUNEIFORM SIGN ERIN2 X"                                                 . #x124AB)
    ("CUNEIFORM SIGN ESH2 CROSSING ESH2"                                      . #x124AC)
    ("CUNEIFORM SIGN EZEN SHESHIG TIMES ASH"                                  . #x124AD)
    ("CUNEIFORM SIGN EZEN SHESHIG TIMES HI"                                   . #x124AE)
    ("CUNEIFORM SIGN EZEN SHESHIG TIMES IGI GUNU"                             . #x124AF)
    ("CUNEIFORM SIGN EZEN SHESHIG TIMES LA"                                   . #x124B0)
    ("CUNEIFORM SIGN EZEN SHESHIG TIMES LAL"                                  . #x124B1)
    ("CUNEIFORM SIGN EZEN SHESHIG TIMES ME"                                   . #x124B2)
    ("CUNEIFORM SIGN EZEN SHESHIG TIMES MES"                                  . #x124B3)
    ("CUNEIFORM SIGN EZEN SHESHIG TIMES SU"                                   . #x124B4)
    ("CUNEIFORM SIGN EZEN TIMES SU"                                           . #x124B5)
    ("CUNEIFORM SIGN GA2 TIMES BAHAR2"                                        . #x124B6)
    ("CUNEIFORM SIGN GA2 TIMES DIM GUNU"                                      . #x124B7)
    ("CUNEIFORM SIGN GA2 TIMES DUG TIMES IGI GUNU"                            . #x124B8)
    ("CUNEIFORM SIGN GA2 TIMES DUG TIMES KASKAL"                              . #x124B9)
    ("CUNEIFORM SIGN GA2 TIMES EREN"                                          . #x124BA)
    ("CUNEIFORM SIGN GA2 TIMES GA"                                            . #x124BB)
    ("CUNEIFORM SIGN GA2 TIMES GAR PLUS DI"                                   . #x124BC)
    ("CUNEIFORM SIGN GA2 TIMES GAR PLUS NE"                                   . #x124BD)
    ("CUNEIFORM SIGN GA2 TIMES HA PLUS A"                                     . #x124BE)
    ("CUNEIFORM SIGN GA2 TIMES KUSHU2 PLUS KASKAL"                            . #x124BF)
    ("CUNEIFORM SIGN GA2 TIMES LAM"                                           . #x124C0)
    ("CUNEIFORM SIGN GA2 TIMES LAM TIMES KUR"                                 . #x124C1)
    ("CUNEIFORM SIGN GA2 TIMES LUH"                                           . #x124C2)
    ("CUNEIFORM SIGN GA2 TIMES MUSH"                                          . #x124C3)
    ("CUNEIFORM SIGN GA2 TIMES NE"                                            . #x124C4)
    ("CUNEIFORM SIGN GA2 TIMES NE PLUS E2"                                    . #x124C5)
    ("CUNEIFORM SIGN GA2 TIMES NE PLUS GI"                                    . #x124C6)
    ("CUNEIFORM SIGN GA2 TIMES SHIM"                                          . #x124C7)
    ("CUNEIFORM SIGN GA2 TIMES ZIZ2"                                          . #x124C8)
    ("CUNEIFORM SIGN GABA ROTATED NINETY DEGREES"                             . #x124C9)
    ("CUNEIFORM SIGN GESHTIN TIMES U"                                         . #x124CA)
    ("CUNEIFORM SIGN GISH TIMES GISH CROSSING GISH"                           . #x124CB)
    ("CUNEIFORM SIGN GU2 TIMES IGI GUNU"                                      . #x124CC)
    ("CUNEIFORM SIGN GUD PLUS GISH TIMES TAK4"                                . #x124CD)
    ("CUNEIFORM SIGN HA TENU GUNU"                                            . #x124CE)
    ("CUNEIFORM SIGN HI TIMES ASH OVER HI TIMES ASH"                          . #x124CF)
    ("CUNEIFORM SIGN KA TIMES BU"                                             . #x124D0)
    ("CUNEIFORM SIGN KA TIMES KA"                                             . #x124D1)
    ("CUNEIFORM SIGN KA TIMES U U U"                                          . #x124D2)
    ("CUNEIFORM SIGN KA TIMES UR"                                             . #x124D3)
    ("CUNEIFORM SIGN LAGAB TIMES ZU OVER ZU"                                  . #x124D4)
    ("CUNEIFORM SIGN LAK-003"                                                 . #x124D5)
    ("CUNEIFORM SIGN LAK-021"                                                 . #x124D6)
    ("CUNEIFORM SIGN LAK-025"                                                 . #x124D7)
    ("CUNEIFORM SIGN LAK-030"                                                 . #x124D8)
    ("CUNEIFORM SIGN LAK-050"                                                 . #x124D9)
    ("CUNEIFORM SIGN LAK-051"                                                 . #x124DA)
    ("CUNEIFORM SIGN LAK-062"                                                 . #x124DB)
    ("CUNEIFORM SIGN LAK-079 OVER LAK-079 GUNU"                               . #x124DC)
    ("CUNEIFORM SIGN LAK-080"                                                 . #x124DD)
    ("CUNEIFORM SIGN LAK-081 OVER LAK-081"                                    . #x124DE)
    ("CUNEIFORM SIGN LAK-092"                                                 . #x124DF)
    ("CUNEIFORM SIGN LAK-130"                                                 . #x124E0)
    ("CUNEIFORM SIGN LAK-142"                                                 . #x124E1)
    ("CUNEIFORM SIGN LAK-210"                                                 . #x124E2)
    ("CUNEIFORM SIGN LAK-219"                                                 . #x124E3)
    ("CUNEIFORM SIGN LAK-220"                                                 . #x124E4)
    ("CUNEIFORM SIGN LAK-225"                                                 . #x124E5)
    ("CUNEIFORM SIGN LAK-228"                                                 . #x124E6)
    ("CUNEIFORM SIGN LAK-238"                                                 . #x124E7)
    ("CUNEIFORM SIGN LAK-265"                                                 . #x124E8)
    ("CUNEIFORM SIGN LAK-266"                                                 . #x124E9)
    ("CUNEIFORM SIGN LAK-343"                                                 . #x124EA)
    ("CUNEIFORM SIGN LAK-347"                                                 . #x124EB)
    ("CUNEIFORM SIGN LAK-348"                                                 . #x124EC)
    ("CUNEIFORM SIGN LAK-383"                                                 . #x124ED)
    ("CUNEIFORM SIGN LAK-384"                                                 . #x124EE)
    ("CUNEIFORM SIGN LAK-390"                                                 . #x124EF)
    ("CUNEIFORM SIGN LAK-441"                                                 . #x124F0)
    ("CUNEIFORM SIGN LAK-449"                                                 . #x124F1)
    ("CUNEIFORM SIGN LAK-449 TIMES GU"                                        . #x124F2)
    ("CUNEIFORM SIGN LAK-449 TIMES IGI"                                       . #x124F3)
    ("CUNEIFORM SIGN LAK-449 TIMES PAP PLUS LU3"                              . #x124F4)
    ("CUNEIFORM SIGN LAK-449 TIMES PAP PLUS PAP PLUS LU3"                     . #x124F5)
    ("CUNEIFORM SIGN LAK-449 TIMES U2 PLUS BA"                                . #x124F6)
    ("CUNEIFORM SIGN LAK-450"                                                 . #x124F7)
    ("CUNEIFORM SIGN LAK-457"                                                 . #x124F8)
    ("CUNEIFORM SIGN LAK-470"                                                 . #x124F9)
    ("CUNEIFORM SIGN LAK-483"                                                 . #x124FA)
    ("CUNEIFORM SIGN LAK-490"                                                 . #x124FB)
    ("CUNEIFORM SIGN LAK-492"                                                 . #x124FC)
    ("CUNEIFORM SIGN LAK-493"                                                 . #x124FD)
    ("CUNEIFORM SIGN LAK-495"                                                 . #x124FE)
    ("CUNEIFORM SIGN LAK-550"                                                 . #x124FF)
    ("CUNEIFORM SIGN LAK-608"                                                 . #x12500)
    ("CUNEIFORM SIGN LAK-617"                                                 . #x12501)
    ("CUNEIFORM SIGN LAK-617 TIMES ASH"                                       . #x12502)
    ("CUNEIFORM SIGN LAK-617 TIMES BAD"                                       . #x12503)
    ("CUNEIFORM SIGN LAK-617 TIMES DUN3 GUNU GUNU"                            . #x12504)
    ("CUNEIFORM SIGN LAK-617 TIMES KU3"                                       . #x12505)
    ("CUNEIFORM SIGN LAK-617 TIMES LA"                                        . #x12506)
    ("CUNEIFORM SIGN LAK-617 TIMES TAR"                                       . #x12507)
    ("CUNEIFORM SIGN LAK-617 TIMES TE"                                        . #x12508)
    ("CUNEIFORM SIGN LAK-617 TIMES U2"                                        . #x12509)
    ("CUNEIFORM SIGN LAK-617 TIMES UD"                                        . #x1250A)
    ("CUNEIFORM SIGN LAK-617 TIMES URUDA"                                     . #x1250B)
    ("CUNEIFORM SIGN LAK-636"                                                 . #x1250C)
    ("CUNEIFORM SIGN LAK-648"                                                 . #x1250D)
    ("CUNEIFORM SIGN LAK-648 TIMES DUB"                                       . #x1250E)
    ("CUNEIFORM SIGN LAK-648 TIMES GA"                                        . #x1250F)
    ("CUNEIFORM SIGN LAK-648 TIMES IGI"                                       . #x12510)
    ("CUNEIFORM SIGN LAK-648 TIMES IGI GUNU"                                  . #x12511)
    ("CUNEIFORM SIGN LAK-648 TIMES NI"                                        . #x12512)
    ("CUNEIFORM SIGN LAK-648 TIMES PAP PLUS PAP PLUS LU3"                     . #x12513)
    ("CUNEIFORM SIGN LAK-648 TIMES SHESH PLUS KI"                             . #x12514)
    ("CUNEIFORM SIGN LAK-648 TIMES UD"                                        . #x12515)
    ("CUNEIFORM SIGN LAK-648 TIMES URUDA"                                     . #x12516)
    ("CUNEIFORM SIGN LAK-724"                                                 . #x12517)
    ("CUNEIFORM SIGN LAK-749"                                                 . #x12518)
    ("CUNEIFORM SIGN LU2 GUNU TIMES ASH"                                      . #x12519)
    ("CUNEIFORM SIGN LU2 TIMES DISH"                                          . #x1251A)
    ("CUNEIFORM SIGN LU2 TIMES HAL"                                           . #x1251B)
    ("CUNEIFORM SIGN LU2 TIMES PAP"                                           . #x1251C)
    ("CUNEIFORM SIGN LU2 TIMES PAP PLUS PAP PLUS LU3"                         . #x1251D)
    ("CUNEIFORM SIGN LU2 TIMES TAK4"                                          . #x1251E)
    ("CUNEIFORM SIGN MI PLUS ZA7"                                             . #x1251F)
    ("CUNEIFORM SIGN MUSH OVER MUSH TIMES GA"                                 . #x12520)
    ("CUNEIFORM SIGN MUSH OVER MUSH TIMES KAK"                                . #x12521)
    ("CUNEIFORM SIGN NINDA2 TIMES DIM GUNU"                                   . #x12522)
    ("CUNEIFORM SIGN NINDA2 TIMES GISH"                                       . #x12523)
    ("CUNEIFORM SIGN NINDA2 TIMES GUL"                                        . #x12524)
    ("CUNEIFORM SIGN NINDA2 TIMES HI"                                         . #x12525)
    ("CUNEIFORM SIGN NINDA2 TIMES KESH2"                                      . #x12526)
    ("CUNEIFORM SIGN NINDA2 TIMES LAK-050"                                    . #x12527)
    ("CUNEIFORM SIGN NINDA2 TIMES MASH"                                       . #x12528)
    ("CUNEIFORM SIGN NINDA2 TIMES PAP PLUS PAP"                               . #x12529)
    ("CUNEIFORM SIGN NINDA2 TIMES U"                                          . #x1252A)
    ("CUNEIFORM SIGN NINDA2 TIMES U PLUS U"                                   . #x1252B)
    ("CUNEIFORM SIGN NINDA2 TIMES URUDA"                                      . #x1252C)
    ("CUNEIFORM SIGN SAG GUNU TIMES HA"                                       . #x1252D)
    ("CUNEIFORM SIGN SAG TIMES EN"                                            . #x1252E)
    ("CUNEIFORM SIGN SAG TIMES SHE AT LEFT"                                   . #x1252F)
    ("CUNEIFORM SIGN SAG TIMES TAK4"                                          . #x12530)
    ("CUNEIFORM SIGN SHA6 TENU"                                               . #x12531)
    ("CUNEIFORM SIGN SHE OVER SHE"                                            . #x12532)
    ("CUNEIFORM SIGN SHE PLUS HUB2"                                           . #x12533)
    ("CUNEIFORM SIGN SHE PLUS NAM2"                                           . #x12534)
    ("CUNEIFORM SIGN SHE PLUS SAR"                                            . #x12535)
    ("CUNEIFORM SIGN SHU2 PLUS DUG TIMES NI"                                  . #x12536)
    ("CUNEIFORM SIGN SHU2 PLUS E2 TIMES AN"                                   . #x12537)
    ("CUNEIFORM SIGN SI TIMES TAK4"                                           . #x12538)
    ("CUNEIFORM SIGN TAK4 PLUS SAG"                                           . #x12539)
    ("CUNEIFORM SIGN TUM TIMES GAN2 TENU"                                     . #x1253A)
    ("CUNEIFORM SIGN TUM TIMES THREE DISH"                                    . #x1253B)
    ("CUNEIFORM SIGN UR2 INVERTED"                                            . #x1253C)
    ("CUNEIFORM SIGN UR2 TIMES UD"                                            . #x1253D)
    ("CUNEIFORM SIGN URU TIMES DARA3"                                         . #x1253E)
    ("CUNEIFORM SIGN URU TIMES LAK-668"                                       . #x1253F)
    ("CUNEIFORM SIGN URU TIMES LU3"                                           . #x12540)
    ("CUNEIFORM SIGN ZA7"                                                     . #x12541)
    ("CUNEIFORM SIGN ZU OVER ZU PLUS SAR"                                     . #x12542)
    ("CUNEIFORM SIGN ZU5 TIMES THREE DISH TENU"                               . #x12543)
    ("ANATOLIAN HIEROGLYPH A001"                                              . #x14400)
    ("ANATOLIAN HIEROGLYPH A002"                                              . #x14401)
    ("ANATOLIAN HIEROGLYPH A003"                                              . #x14402)
    ("ANATOLIAN HIEROGLYPH A004"                                              . #x14403)
    ("ANATOLIAN HIEROGLYPH A005"                                              . #x14404)
    ("ANATOLIAN HIEROGLYPH A006"                                              . #x14405)
    ("ANATOLIAN HIEROGLYPH A007"                                              . #x14406)
    ("ANATOLIAN HIEROGLYPH A008"                                              . #x14407)
    ("ANATOLIAN HIEROGLYPH A009"                                              . #x14408)
    ("ANATOLIAN HIEROGLYPH A010"                                              . #x14409)
    ("ANATOLIAN HIEROGLYPH A010A"                                             . #x1440A)
    ("ANATOLIAN HIEROGLYPH A011"                                              . #x1440B)
    ("ANATOLIAN HIEROGLYPH A012"                                              . #x1440C)
    ("ANATOLIAN HIEROGLYPH A013"                                              . #x1440D)
    ("ANATOLIAN HIEROGLYPH A014"                                              . #x1440E)
    ("ANATOLIAN HIEROGLYPH A015"                                              . #x1440F)
    ("ANATOLIAN HIEROGLYPH A016"                                              . #x14410)
    ("ANATOLIAN HIEROGLYPH A017"                                              . #x14411)
    ("ANATOLIAN HIEROGLYPH A018"                                              . #x14412)
    ("ANATOLIAN HIEROGLYPH A019"                                              . #x14413)
    ("ANATOLIAN HIEROGLYPH A020"                                              . #x14414)
    ("ANATOLIAN HIEROGLYPH A021"                                              . #x14415)
    ("ANATOLIAN HIEROGLYPH A022"                                              . #x14416)
    ("ANATOLIAN HIEROGLYPH A023"                                              . #x14417)
    ("ANATOLIAN HIEROGLYPH A024"                                              . #x14418)
    ("ANATOLIAN HIEROGLYPH A025"                                              . #x14419)
    ("ANATOLIAN HIEROGLYPH A026"                                              . #x1441A)
    ("ANATOLIAN HIEROGLYPH A026A"                                             . #x1441B)
    ("ANATOLIAN HIEROGLYPH A027"                                              . #x1441C)
    ("ANATOLIAN HIEROGLYPH A028"                                              . #x1441D)
    ("ANATOLIAN HIEROGLYPH A029"                                              . #x1441E)
    ("ANATOLIAN HIEROGLYPH A030"                                              . #x1441F)
    ("ANATOLIAN HIEROGLYPH A031"                                              . #x14420)
    ("ANATOLIAN HIEROGLYPH A032"                                              . #x14421)
    ("ANATOLIAN HIEROGLYPH A033"                                              . #x14422)
    ("ANATOLIAN HIEROGLYPH A034"                                              . #x14423)
    ("ANATOLIAN HIEROGLYPH A035"                                              . #x14424)
    ("ANATOLIAN HIEROGLYPH A036"                                              . #x14425)
    ("ANATOLIAN HIEROGLYPH A037"                                              . #x14426)
    ("ANATOLIAN HIEROGLYPH A038"                                              . #x14427)
    ("ANATOLIAN HIEROGLYPH A039"                                              . #x14428)
    ("ANATOLIAN HIEROGLYPH A039A"                                             . #x14429)
    ("ANATOLIAN HIEROGLYPH A040"                                              . #x1442A)
    ("ANATOLIAN HIEROGLYPH A041"                                              . #x1442B)
    ("ANATOLIAN HIEROGLYPH A041A"                                             . #x1442C)
    ("ANATOLIAN HIEROGLYPH A042"                                              . #x1442D)
    ("ANATOLIAN HIEROGLYPH A043"                                              . #x1442E)
    ("ANATOLIAN HIEROGLYPH A044"                                              . #x1442F)
    ("ANATOLIAN HIEROGLYPH A045"                                              . #x14430)
    ("ANATOLIAN HIEROGLYPH A045A"                                             . #x14431)
    ("ANATOLIAN HIEROGLYPH A046"                                              . #x14432)
    ("ANATOLIAN HIEROGLYPH A046A"                                             . #x14433)
    ("ANATOLIAN HIEROGLYPH A046B"                                             . #x14434)
    ("ANATOLIAN HIEROGLYPH A047"                                              . #x14435)
    ("ANATOLIAN HIEROGLYPH A048"                                              . #x14436)
    ("ANATOLIAN HIEROGLYPH A049"                                              . #x14437)
    ("ANATOLIAN HIEROGLYPH A050"                                              . #x14438)
    ("ANATOLIAN HIEROGLYPH A051"                                              . #x14439)
    ("ANATOLIAN HIEROGLYPH A052"                                              . #x1443A)
    ("ANATOLIAN HIEROGLYPH A053"                                              . #x1443B)
    ("ANATOLIAN HIEROGLYPH A054"                                              . #x1443C)
    ("ANATOLIAN HIEROGLYPH A055"                                              . #x1443D)
    ("ANATOLIAN HIEROGLYPH A056"                                              . #x1443E)
    ("ANATOLIAN HIEROGLYPH A057"                                              . #x1443F)
    ("ANATOLIAN HIEROGLYPH A058"                                              . #x14440)
    ("ANATOLIAN HIEROGLYPH A059"                                              . #x14441)
    ("ANATOLIAN HIEROGLYPH A060"                                              . #x14442)
    ("ANATOLIAN HIEROGLYPH A061"                                              . #x14443)
    ("ANATOLIAN HIEROGLYPH A062"                                              . #x14444)
    ("ANATOLIAN HIEROGLYPH A063"                                              . #x14445)
    ("ANATOLIAN HIEROGLYPH A064"                                              . #x14446)
    ("ANATOLIAN HIEROGLYPH A065"                                              . #x14447)
    ("ANATOLIAN HIEROGLYPH A066"                                              . #x14448)
    ("ANATOLIAN HIEROGLYPH A066A"                                             . #x14449)
    ("ANATOLIAN HIEROGLYPH A066B"                                             . #x1444A)
    ("ANATOLIAN HIEROGLYPH A066C"                                             . #x1444B)
    ("ANATOLIAN HIEROGLYPH A067"                                              . #x1444C)
    ("ANATOLIAN HIEROGLYPH A068"                                              . #x1444D)
    ("ANATOLIAN HIEROGLYPH A069"                                              . #x1444E)
    ("ANATOLIAN HIEROGLYPH A070"                                              . #x1444F)
    ("ANATOLIAN HIEROGLYPH A071"                                              . #x14450)
    ("ANATOLIAN HIEROGLYPH A072"                                              . #x14451)
    ("ANATOLIAN HIEROGLYPH A073"                                              . #x14452)
    ("ANATOLIAN HIEROGLYPH A074"                                              . #x14453)
    ("ANATOLIAN HIEROGLYPH A075"                                              . #x14454)
    ("ANATOLIAN HIEROGLYPH A076"                                              . #x14455)
    ("ANATOLIAN HIEROGLYPH A077"                                              . #x14456)
    ("ANATOLIAN HIEROGLYPH A078"                                              . #x14457)
    ("ANATOLIAN HIEROGLYPH A079"                                              . #x14458)
    ("ANATOLIAN HIEROGLYPH A080"                                              . #x14459)
    ("ANATOLIAN HIEROGLYPH A081"                                              . #x1445A)
    ("ANATOLIAN HIEROGLYPH A082"                                              . #x1445B)
    ("ANATOLIAN HIEROGLYPH A083"                                              . #x1445C)
    ("ANATOLIAN HIEROGLYPH A084"                                              . #x1445D)
    ("ANATOLIAN HIEROGLYPH A085"                                              . #x1445E)
    ("ANATOLIAN HIEROGLYPH A086"                                              . #x1445F)
    ("ANATOLIAN HIEROGLYPH A087"                                              . #x14460)
    ("ANATOLIAN HIEROGLYPH A088"                                              . #x14461)
    ("ANATOLIAN HIEROGLYPH A089"                                              . #x14462)
    ("ANATOLIAN HIEROGLYPH A090"                                              . #x14463)
    ("ANATOLIAN HIEROGLYPH A091"                                              . #x14464)
    ("ANATOLIAN HIEROGLYPH A092"                                              . #x14465)
    ("ANATOLIAN HIEROGLYPH A093"                                              . #x14466)
    ("ANATOLIAN HIEROGLYPH A094"                                              . #x14467)
    ("ANATOLIAN HIEROGLYPH A095"                                              . #x14468)
    ("ANATOLIAN HIEROGLYPH A096"                                              . #x14469)
    ("ANATOLIAN HIEROGLYPH A097"                                              . #x1446A)
    ("ANATOLIAN HIEROGLYPH A097A"                                             . #x1446B)
    ("ANATOLIAN HIEROGLYPH A098"                                              . #x1446C)
    ("ANATOLIAN HIEROGLYPH A098A"                                             . #x1446D)
    ("ANATOLIAN HIEROGLYPH A099"                                              . #x1446E)
    ("ANATOLIAN HIEROGLYPH A100"                                              . #x1446F)
    ("ANATOLIAN HIEROGLYPH A100A"                                             . #x14470)
    ("ANATOLIAN HIEROGLYPH A101"                                              . #x14471)
    ("ANATOLIAN HIEROGLYPH A101A"                                             . #x14472)
    ("ANATOLIAN HIEROGLYPH A102"                                              . #x14473)
    ("ANATOLIAN HIEROGLYPH A102A"                                             . #x14474)
    ("ANATOLIAN HIEROGLYPH A103"                                              . #x14475)
    ("ANATOLIAN HIEROGLYPH A104"                                              . #x14476)
    ("ANATOLIAN HIEROGLYPH A104A"                                             . #x14477)
    ("ANATOLIAN HIEROGLYPH A104B"                                             . #x14478)
    ("ANATOLIAN HIEROGLYPH A104C"                                             . #x14479)
    ("ANATOLIAN HIEROGLYPH A105"                                              . #x1447A)
    ("ANATOLIAN HIEROGLYPH A105A"                                             . #x1447B)
    ("ANATOLIAN HIEROGLYPH A105B"                                             . #x1447C)
    ("ANATOLIAN HIEROGLYPH A106"                                              . #x1447D)
    ("ANATOLIAN HIEROGLYPH A107"                                              . #x1447E)
    ("ANATOLIAN HIEROGLYPH A107A"                                             . #x1447F)
    ("ANATOLIAN HIEROGLYPH A107B"                                             . #x14480)
    ("ANATOLIAN HIEROGLYPH A107C"                                             . #x14481)
    ("ANATOLIAN HIEROGLYPH A108"                                              . #x14482)
    ("ANATOLIAN HIEROGLYPH A109"                                              . #x14483)
    ("ANATOLIAN HIEROGLYPH A110"                                              . #x14484)
    ("ANATOLIAN HIEROGLYPH A110A"                                             . #x14485)
    ("ANATOLIAN HIEROGLYPH A110B"                                             . #x14486)
    ("ANATOLIAN HIEROGLYPH A111"                                              . #x14487)
    ("ANATOLIAN HIEROGLYPH A112"                                              . #x14488)
    ("ANATOLIAN HIEROGLYPH A113"                                              . #x14489)
    ("ANATOLIAN HIEROGLYPH A114"                                              . #x1448A)
    ("ANATOLIAN HIEROGLYPH A115"                                              . #x1448B)
    ("ANATOLIAN HIEROGLYPH A115A"                                             . #x1448C)
    ("ANATOLIAN HIEROGLYPH A116"                                              . #x1448D)
    ("ANATOLIAN HIEROGLYPH A117"                                              . #x1448E)
    ("ANATOLIAN HIEROGLYPH A118"                                              . #x1448F)
    ("ANATOLIAN HIEROGLYPH A119"                                              . #x14490)
    ("ANATOLIAN HIEROGLYPH A120"                                              . #x14491)
    ("ANATOLIAN HIEROGLYPH A121"                                              . #x14492)
    ("ANATOLIAN HIEROGLYPH A122"                                              . #x14493)
    ("ANATOLIAN HIEROGLYPH A123"                                              . #x14494)
    ("ANATOLIAN HIEROGLYPH A124"                                              . #x14495)
    ("ANATOLIAN HIEROGLYPH A125"                                              . #x14496)
    ("ANATOLIAN HIEROGLYPH A125A"                                             . #x14497)
    ("ANATOLIAN HIEROGLYPH A126"                                              . #x14498)
    ("ANATOLIAN HIEROGLYPH A127"                                              . #x14499)
    ("ANATOLIAN HIEROGLYPH A128"                                              . #x1449A)
    ("ANATOLIAN HIEROGLYPH A129"                                              . #x1449B)
    ("ANATOLIAN HIEROGLYPH A130"                                              . #x1449C)
    ("ANATOLIAN HIEROGLYPH A131"                                              . #x1449D)
    ("ANATOLIAN HIEROGLYPH A132"                                              . #x1449E)
    ("ANATOLIAN HIEROGLYPH A133"                                              . #x1449F)
    ("ANATOLIAN HIEROGLYPH A134"                                              . #x144A0)
    ("ANATOLIAN HIEROGLYPH A135"                                              . #x144A1)
    ("ANATOLIAN HIEROGLYPH A135A"                                             . #x144A2)
    ("ANATOLIAN HIEROGLYPH A136"                                              . #x144A3)
    ("ANATOLIAN HIEROGLYPH A137"                                              . #x144A4)
    ("ANATOLIAN HIEROGLYPH A138"                                              . #x144A5)
    ("ANATOLIAN HIEROGLYPH A139"                                              . #x144A6)
    ("ANATOLIAN HIEROGLYPH A140"                                              . #x144A7)
    ("ANATOLIAN HIEROGLYPH A141"                                              . #x144A8)
    ("ANATOLIAN HIEROGLYPH A142"                                              . #x144A9)
    ("ANATOLIAN HIEROGLYPH A143"                                              . #x144AA)
    ("ANATOLIAN HIEROGLYPH A144"                                              . #x144AB)
    ("ANATOLIAN HIEROGLYPH A145"                                              . #x144AC)
    ("ANATOLIAN HIEROGLYPH A146"                                              . #x144AD)
    ("ANATOLIAN HIEROGLYPH A147"                                              . #x144AE)
    ("ANATOLIAN HIEROGLYPH A148"                                              . #x144AF)
    ("ANATOLIAN HIEROGLYPH A149"                                              . #x144B0)
    ("ANATOLIAN HIEROGLYPH A150"                                              . #x144B1)
    ("ANATOLIAN HIEROGLYPH A151"                                              . #x144B2)
    ("ANATOLIAN HIEROGLYPH A152"                                              . #x144B3)
    ("ANATOLIAN HIEROGLYPH A153"                                              . #x144B4)
    ("ANATOLIAN HIEROGLYPH A154"                                              . #x144B5)
    ("ANATOLIAN HIEROGLYPH A155"                                              . #x144B6)
    ("ANATOLIAN HIEROGLYPH A156"                                              . #x144B7)
    ("ANATOLIAN HIEROGLYPH A157"                                              . #x144B8)
    ("ANATOLIAN HIEROGLYPH A158"                                              . #x144B9)
    ("ANATOLIAN HIEROGLYPH A159"                                              . #x144BA)
    ("ANATOLIAN HIEROGLYPH A160"                                              . #x144BB)
    ("ANATOLIAN HIEROGLYPH A161"                                              . #x144BC)
    ("ANATOLIAN HIEROGLYPH A162"                                              . #x144BD)
    ("ANATOLIAN HIEROGLYPH A163"                                              . #x144BE)
    ("ANATOLIAN HIEROGLYPH A164"                                              . #x144BF)
    ("ANATOLIAN HIEROGLYPH A165"                                              . #x144C0)
    ("ANATOLIAN HIEROGLYPH A166"                                              . #x144C1)
    ("ANATOLIAN HIEROGLYPH A167"                                              . #x144C2)
    ("ANATOLIAN HIEROGLYPH A168"                                              . #x144C3)
    ("ANATOLIAN HIEROGLYPH A169"                                              . #x144C4)
    ("ANATOLIAN HIEROGLYPH A170"                                              . #x144C5)
    ("ANATOLIAN HIEROGLYPH A171"                                              . #x144C6)
    ("ANATOLIAN HIEROGLYPH A172"                                              . #x144C7)
    ("ANATOLIAN HIEROGLYPH A173"                                              . #x144C8)
    ("ANATOLIAN HIEROGLYPH A174"                                              . #x144C9)
    ("ANATOLIAN HIEROGLYPH A175"                                              . #x144CA)
    ("ANATOLIAN HIEROGLYPH A176"                                              . #x144CB)
    ("ANATOLIAN HIEROGLYPH A177"                                              . #x144CC)
    ("ANATOLIAN HIEROGLYPH A178"                                              . #x144CD)
    ("ANATOLIAN HIEROGLYPH A179"                                              . #x144CE)
    ("ANATOLIAN HIEROGLYPH A180"                                              . #x144CF)
    ("ANATOLIAN HIEROGLYPH A181"                                              . #x144D0)
    ("ANATOLIAN HIEROGLYPH A182"                                              . #x144D1)
    ("ANATOLIAN HIEROGLYPH A183"                                              . #x144D2)
    ("ANATOLIAN HIEROGLYPH A184"                                              . #x144D3)
    ("ANATOLIAN HIEROGLYPH A185"                                              . #x144D4)
    ("ANATOLIAN HIEROGLYPH A186"                                              . #x144D5)
    ("ANATOLIAN HIEROGLYPH A187"                                              . #x144D6)
    ("ANATOLIAN HIEROGLYPH A188"                                              . #x144D7)
    ("ANATOLIAN HIEROGLYPH A189"                                              . #x144D8)
    ("ANATOLIAN HIEROGLYPH A190"                                              . #x144D9)
    ("ANATOLIAN HIEROGLYPH A191"                                              . #x144DA)
    ("ANATOLIAN HIEROGLYPH A192"                                              . #x144DB)
    ("ANATOLIAN HIEROGLYPH A193"                                              . #x144DC)
    ("ANATOLIAN HIEROGLYPH A194"                                              . #x144DD)
    ("ANATOLIAN HIEROGLYPH A195"                                              . #x144DE)
    ("ANATOLIAN HIEROGLYPH A196"                                              . #x144DF)
    ("ANATOLIAN HIEROGLYPH A197"                                              . #x144E0)
    ("ANATOLIAN HIEROGLYPH A198"                                              . #x144E1)
    ("ANATOLIAN HIEROGLYPH A199"                                              . #x144E2)
    ("ANATOLIAN HIEROGLYPH A200"                                              . #x144E3)
    ("ANATOLIAN HIEROGLYPH A201"                                              . #x144E4)
    ("ANATOLIAN HIEROGLYPH A202"                                              . #x144E5)
    ("ANATOLIAN HIEROGLYPH A202A"                                             . #x144E6)
    ("ANATOLIAN HIEROGLYPH A202B"                                             . #x144E7)
    ("ANATOLIAN HIEROGLYPH A203"                                              . #x144E8)
    ("ANATOLIAN HIEROGLYPH A204"                                              . #x144E9)
    ("ANATOLIAN HIEROGLYPH A205"                                              . #x144EA)
    ("ANATOLIAN HIEROGLYPH A206"                                              . #x144EB)
    ("ANATOLIAN HIEROGLYPH A207"                                              . #x144EC)
    ("ANATOLIAN HIEROGLYPH A207A"                                             . #x144ED)
    ("ANATOLIAN HIEROGLYPH A208"                                              . #x144EE)
    ("ANATOLIAN HIEROGLYPH A209"                                              . #x144EF)
    ("ANATOLIAN HIEROGLYPH A209A"                                             . #x144F0)
    ("ANATOLIAN HIEROGLYPH A210"                                              . #x144F1)
    ("ANATOLIAN HIEROGLYPH A211"                                              . #x144F2)
    ("ANATOLIAN HIEROGLYPH A212"                                              . #x144F3)
    ("ANATOLIAN HIEROGLYPH A213"                                              . #x144F4)
    ("ANATOLIAN HIEROGLYPH A214"                                              . #x144F5)
    ("ANATOLIAN HIEROGLYPH A215"                                              . #x144F6)
    ("ANATOLIAN HIEROGLYPH A215A"                                             . #x144F7)
    ("ANATOLIAN HIEROGLYPH A216"                                              . #x144F8)
    ("ANATOLIAN HIEROGLYPH A216A"                                             . #x144F9)
    ("ANATOLIAN HIEROGLYPH A217"                                              . #x144FA)
    ("ANATOLIAN HIEROGLYPH A218"                                              . #x144FB)
    ("ANATOLIAN HIEROGLYPH A219"                                              . #x144FC)
    ("ANATOLIAN HIEROGLYPH A220"                                              . #x144FD)
    ("ANATOLIAN HIEROGLYPH A221"                                              . #x144FE)
    ("ANATOLIAN HIEROGLYPH A222"                                              . #x144FF)
    ("ANATOLIAN HIEROGLYPH A223"                                              . #x14500)
    ("ANATOLIAN HIEROGLYPH A224"                                              . #x14501)
    ("ANATOLIAN HIEROGLYPH A225"                                              . #x14502)
    ("ANATOLIAN HIEROGLYPH A226"                                              . #x14503)
    ("ANATOLIAN HIEROGLYPH A227"                                              . #x14504)
    ("ANATOLIAN HIEROGLYPH A227A"                                             . #x14505)
    ("ANATOLIAN HIEROGLYPH A228"                                              . #x14506)
    ("ANATOLIAN HIEROGLYPH A229"                                              . #x14507)
    ("ANATOLIAN HIEROGLYPH A230"                                              . #x14508)
    ("ANATOLIAN HIEROGLYPH A231"                                              . #x14509)
    ("ANATOLIAN HIEROGLYPH A232"                                              . #x1450A)
    ("ANATOLIAN HIEROGLYPH A233"                                              . #x1450B)
    ("ANATOLIAN HIEROGLYPH A234"                                              . #x1450C)
    ("ANATOLIAN HIEROGLYPH A235"                                              . #x1450D)
    ("ANATOLIAN HIEROGLYPH A236"                                              . #x1450E)
    ("ANATOLIAN HIEROGLYPH A237"                                              . #x1450F)
    ("ANATOLIAN HIEROGLYPH A238"                                              . #x14510)
    ("ANATOLIAN HIEROGLYPH A239"                                              . #x14511)
    ("ANATOLIAN HIEROGLYPH A240"                                              . #x14512)
    ("ANATOLIAN HIEROGLYPH A241"                                              . #x14513)
    ("ANATOLIAN HIEROGLYPH A242"                                              . #x14514)
    ("ANATOLIAN HIEROGLYPH A243"                                              . #x14515)
    ("ANATOLIAN HIEROGLYPH A244"                                              . #x14516)
    ("ANATOLIAN HIEROGLYPH A245"                                              . #x14517)
    ("ANATOLIAN HIEROGLYPH A246"                                              . #x14518)
    ("ANATOLIAN HIEROGLYPH A247"                                              . #x14519)
    ("ANATOLIAN HIEROGLYPH A248"                                              . #x1451A)
    ("ANATOLIAN HIEROGLYPH A249"                                              . #x1451B)
    ("ANATOLIAN HIEROGLYPH A250"                                              . #x1451C)
    ("ANATOLIAN HIEROGLYPH A251"                                              . #x1451D)
    ("ANATOLIAN HIEROGLYPH A252"                                              . #x1451E)
    ("ANATOLIAN HIEROGLYPH A253"                                              . #x1451F)
    ("ANATOLIAN HIEROGLYPH A254"                                              . #x14520)
    ("ANATOLIAN HIEROGLYPH A255"                                              . #x14521)
    ("ANATOLIAN HIEROGLYPH A256"                                              . #x14522)
    ("ANATOLIAN HIEROGLYPH A257"                                              . #x14523)
    ("ANATOLIAN HIEROGLYPH A258"                                              . #x14524)
    ("ANATOLIAN HIEROGLYPH A259"                                              . #x14525)
    ("ANATOLIAN HIEROGLYPH A260"                                              . #x14526)
    ("ANATOLIAN HIEROGLYPH A261"                                              . #x14527)
    ("ANATOLIAN HIEROGLYPH A262"                                              . #x14528)
    ("ANATOLIAN HIEROGLYPH A263"                                              . #x14529)
    ("ANATOLIAN HIEROGLYPH A264"                                              . #x1452A)
    ("ANATOLIAN HIEROGLYPH A265"                                              . #x1452B)
    ("ANATOLIAN HIEROGLYPH A266"                                              . #x1452C)
    ("ANATOLIAN HIEROGLYPH A267"                                              . #x1452D)
    ("ANATOLIAN HIEROGLYPH A267A"                                             . #x1452E)
    ("ANATOLIAN HIEROGLYPH A268"                                              . #x1452F)
    ("ANATOLIAN HIEROGLYPH A269"                                              . #x14530)
    ("ANATOLIAN HIEROGLYPH A270"                                              . #x14531)
    ("ANATOLIAN HIEROGLYPH A271"                                              . #x14532)
    ("ANATOLIAN HIEROGLYPH A272"                                              . #x14533)
    ("ANATOLIAN HIEROGLYPH A273"                                              . #x14534)
    ("ANATOLIAN HIEROGLYPH A274"                                              . #x14535)
    ("ANATOLIAN HIEROGLYPH A275"                                              . #x14536)
    ("ANATOLIAN HIEROGLYPH A276"                                              . #x14537)
    ("ANATOLIAN HIEROGLYPH A277"                                              . #x14538)
    ("ANATOLIAN HIEROGLYPH A278"                                              . #x14539)
    ("ANATOLIAN HIEROGLYPH A279"                                              . #x1453A)
    ("ANATOLIAN HIEROGLYPH A280"                                              . #x1453B)
    ("ANATOLIAN HIEROGLYPH A281"                                              . #x1453C)
    ("ANATOLIAN HIEROGLYPH A282"                                              . #x1453D)
    ("ANATOLIAN HIEROGLYPH A283"                                              . #x1453E)
    ("ANATOLIAN HIEROGLYPH A284"                                              . #x1453F)
    ("ANATOLIAN HIEROGLYPH A285"                                              . #x14540)
    ("ANATOLIAN HIEROGLYPH A286"                                              . #x14541)
    ("ANATOLIAN HIEROGLYPH A287"                                              . #x14542)
    ("ANATOLIAN HIEROGLYPH A288"                                              . #x14543)
    ("ANATOLIAN HIEROGLYPH A289"                                              . #x14544)
    ("ANATOLIAN HIEROGLYPH A289A"                                             . #x14545)
    ("ANATOLIAN HIEROGLYPH A290"                                              . #x14546)
    ("ANATOLIAN HIEROGLYPH A291"                                              . #x14547)
    ("ANATOLIAN HIEROGLYPH A292"                                              . #x14548)
    ("ANATOLIAN HIEROGLYPH A293"                                              . #x14549)
    ("ANATOLIAN HIEROGLYPH A294"                                              . #x1454A)
    ("ANATOLIAN HIEROGLYPH A294A"                                             . #x1454B)
    ("ANATOLIAN HIEROGLYPH A295"                                              . #x1454C)
    ("ANATOLIAN HIEROGLYPH A296"                                              . #x1454D)
    ("ANATOLIAN HIEROGLYPH A297"                                              . #x1454E)
    ("ANATOLIAN HIEROGLYPH A298"                                              . #x1454F)
    ("ANATOLIAN HIEROGLYPH A299"                                              . #x14550)
    ("ANATOLIAN HIEROGLYPH A299A"                                             . #x14551)
    ("ANATOLIAN HIEROGLYPH A300"                                              . #x14552)
    ("ANATOLIAN HIEROGLYPH A301"                                              . #x14553)
    ("ANATOLIAN HIEROGLYPH A302"                                              . #x14554)
    ("ANATOLIAN HIEROGLYPH A303"                                              . #x14555)
    ("ANATOLIAN HIEROGLYPH A304"                                              . #x14556)
    ("ANATOLIAN HIEROGLYPH A305"                                              . #x14557)
    ("ANATOLIAN HIEROGLYPH A306"                                              . #x14558)
    ("ANATOLIAN HIEROGLYPH A307"                                              . #x14559)
    ("ANATOLIAN HIEROGLYPH A308"                                              . #x1455A)
    ("ANATOLIAN HIEROGLYPH A309"                                              . #x1455B)
    ("ANATOLIAN HIEROGLYPH A309A"                                             . #x1455C)
    ("ANATOLIAN HIEROGLYPH A310"                                              . #x1455D)
    ("ANATOLIAN HIEROGLYPH A311"                                              . #x1455E)
    ("ANATOLIAN HIEROGLYPH A312"                                              . #x1455F)
    ("ANATOLIAN HIEROGLYPH A313"                                              . #x14560)
    ("ANATOLIAN HIEROGLYPH A314"                                              . #x14561)
    ("ANATOLIAN HIEROGLYPH A315"                                              . #x14562)
    ("ANATOLIAN HIEROGLYPH A316"                                              . #x14563)
    ("ANATOLIAN HIEROGLYPH A317"                                              . #x14564)
    ("ANATOLIAN HIEROGLYPH A318"                                              . #x14565)
    ("ANATOLIAN HIEROGLYPH A319"                                              . #x14566)
    ("ANATOLIAN HIEROGLYPH A320"                                              . #x14567)
    ("ANATOLIAN HIEROGLYPH A321"                                              . #x14568)
    ("ANATOLIAN HIEROGLYPH A322"                                              . #x14569)
    ("ANATOLIAN HIEROGLYPH A323"                                              . #x1456A)
    ("ANATOLIAN HIEROGLYPH A324"                                              . #x1456B)
    ("ANATOLIAN HIEROGLYPH A325"                                              . #x1456C)
    ("ANATOLIAN HIEROGLYPH A326"                                              . #x1456D)
    ("ANATOLIAN HIEROGLYPH A327"                                              . #x1456E)
    ("ANATOLIAN HIEROGLYPH A328"                                              . #x1456F)
    ("ANATOLIAN HIEROGLYPH A329"                                              . #x14570)
    ("ANATOLIAN HIEROGLYPH A329A"                                             . #x14571)
    ("ANATOLIAN HIEROGLYPH A330"                                              . #x14572)
    ("ANATOLIAN HIEROGLYPH A331"                                              . #x14573)
    ("ANATOLIAN HIEROGLYPH A332A"                                             . #x14574)
    ("ANATOLIAN HIEROGLYPH A332B"                                             . #x14575)
    ("ANATOLIAN HIEROGLYPH A332C"                                             . #x14576)
    ("ANATOLIAN HIEROGLYPH A333"                                              . #x14577)
    ("ANATOLIAN HIEROGLYPH A334"                                              . #x14578)
    ("ANATOLIAN HIEROGLYPH A335"                                              . #x14579)
    ("ANATOLIAN HIEROGLYPH A336"                                              . #x1457A)
    ("ANATOLIAN HIEROGLYPH A336A"                                             . #x1457B)
    ("ANATOLIAN HIEROGLYPH A336B"                                             . #x1457C)
    ("ANATOLIAN HIEROGLYPH A336C"                                             . #x1457D)
    ("ANATOLIAN HIEROGLYPH A337"                                              . #x1457E)
    ("ANATOLIAN HIEROGLYPH A338"                                              . #x1457F)
    ("ANATOLIAN HIEROGLYPH A339"                                              . #x14580)
    ("ANATOLIAN HIEROGLYPH A340"                                              . #x14581)
    ("ANATOLIAN HIEROGLYPH A341"                                              . #x14582)
    ("ANATOLIAN HIEROGLYPH A342"                                              . #x14583)
    ("ANATOLIAN HIEROGLYPH A343"                                              . #x14584)
    ("ANATOLIAN HIEROGLYPH A344"                                              . #x14585)
    ("ANATOLIAN HIEROGLYPH A345"                                              . #x14586)
    ("ANATOLIAN HIEROGLYPH A346"                                              . #x14587)
    ("ANATOLIAN HIEROGLYPH A347"                                              . #x14588)
    ("ANATOLIAN HIEROGLYPH A348"                                              . #x14589)
    ("ANATOLIAN HIEROGLYPH A349"                                              . #x1458A)
    ("ANATOLIAN HIEROGLYPH A350"                                              . #x1458B)
    ("ANATOLIAN HIEROGLYPH A351"                                              . #x1458C)
    ("ANATOLIAN HIEROGLYPH A352"                                              . #x1458D)
    ("ANATOLIAN HIEROGLYPH A353"                                              . #x1458E)
    ("ANATOLIAN HIEROGLYPH A354"                                              . #x1458F)
    ("ANATOLIAN HIEROGLYPH A355"                                              . #x14590)
    ("ANATOLIAN HIEROGLYPH A356"                                              . #x14591)
    ("ANATOLIAN HIEROGLYPH A357"                                              . #x14592)
    ("ANATOLIAN HIEROGLYPH A358"                                              . #x14593)
    ("ANATOLIAN HIEROGLYPH A359"                                              . #x14594)
    ("ANATOLIAN HIEROGLYPH A359A"                                             . #x14595)
    ("ANATOLIAN HIEROGLYPH A360"                                              . #x14596)
    ("ANATOLIAN HIEROGLYPH A361"                                              . #x14597)
    ("ANATOLIAN HIEROGLYPH A362"                                              . #x14598)
    ("ANATOLIAN HIEROGLYPH A363"                                              . #x14599)
    ("ANATOLIAN HIEROGLYPH A364"                                              . #x1459A)
    ("ANATOLIAN HIEROGLYPH A364A"                                             . #x1459B)
    ("ANATOLIAN HIEROGLYPH A365"                                              . #x1459C)
    ("ANATOLIAN HIEROGLYPH A366"                                              . #x1459D)
    ("ANATOLIAN HIEROGLYPH A367"                                              . #x1459E)
    ("ANATOLIAN HIEROGLYPH A368"                                              . #x1459F)
    ("ANATOLIAN HIEROGLYPH A368A"                                             . #x145A0)
    ("ANATOLIAN HIEROGLYPH A369"                                              . #x145A1)
    ("ANATOLIAN HIEROGLYPH A370"                                              . #x145A2)
    ("ANATOLIAN HIEROGLYPH A371"                                              . #x145A3)
    ("ANATOLIAN HIEROGLYPH A371A"                                             . #x145A4)
    ("ANATOLIAN HIEROGLYPH A372"                                              . #x145A5)
    ("ANATOLIAN HIEROGLYPH A373"                                              . #x145A6)
    ("ANATOLIAN HIEROGLYPH A374"                                              . #x145A7)
    ("ANATOLIAN HIEROGLYPH A375"                                              . #x145A8)
    ("ANATOLIAN HIEROGLYPH A376"                                              . #x145A9)
    ("ANATOLIAN HIEROGLYPH A377"                                              . #x145AA)
    ("ANATOLIAN HIEROGLYPH A378"                                              . #x145AB)
    ("ANATOLIAN HIEROGLYPH A379"                                              . #x145AC)
    ("ANATOLIAN HIEROGLYPH A380"                                              . #x145AD)
    ("ANATOLIAN HIEROGLYPH A381"                                              . #x145AE)
    ("ANATOLIAN HIEROGLYPH A381A"                                             . #x145AF)
    ("ANATOLIAN HIEROGLYPH A382"                                              . #x145B0)
    ("ANATOLIAN HIEROGLYPH A383 RA OR RI"                                     . #x145B1)
    ("ANATOLIAN HIEROGLYPH A383A"                                             . #x145B2)
    ("ANATOLIAN HIEROGLYPH A384"                                              . #x145B3)
    ("ANATOLIAN HIEROGLYPH A385"                                              . #x145B4)
    ("ANATOLIAN HIEROGLYPH A386"                                              . #x145B5)
    ("ANATOLIAN HIEROGLYPH A386A"                                             . #x145B6)
    ("ANATOLIAN HIEROGLYPH A387"                                              . #x145B7)
    ("ANATOLIAN HIEROGLYPH A388"                                              . #x145B8)
    ("ANATOLIAN HIEROGLYPH A389"                                              . #x145B9)
    ("ANATOLIAN HIEROGLYPH A390"                                              . #x145BA)
    ("ANATOLIAN HIEROGLYPH A391"                                              . #x145BB)
    ("ANATOLIAN HIEROGLYPH A392"                                              . #x145BC)
    ("ANATOLIAN HIEROGLYPH A393 EIGHT"                                        . #x145BD)
    ("ANATOLIAN HIEROGLYPH A394"                                              . #x145BE)
    ("ANATOLIAN HIEROGLYPH A395"                                              . #x145BF)
    ("ANATOLIAN HIEROGLYPH A396"                                              . #x145C0)
    ("ANATOLIAN HIEROGLYPH A397"                                              . #x145C1)
    ("ANATOLIAN HIEROGLYPH A398"                                              . #x145C2)
    ("ANATOLIAN HIEROGLYPH A399"                                              . #x145C3)
    ("ANATOLIAN HIEROGLYPH A400"                                              . #x145C4)
    ("ANATOLIAN HIEROGLYPH A401"                                              . #x145C5)
    ("ANATOLIAN HIEROGLYPH A402"                                              . #x145C6)
    ("ANATOLIAN HIEROGLYPH A403"                                              . #x145C7)
    ("ANATOLIAN HIEROGLYPH A404"                                              . #x145C8)
    ("ANATOLIAN HIEROGLYPH A405"                                              . #x145C9)
    ("ANATOLIAN HIEROGLYPH A406"                                              . #x145CA)
    ("ANATOLIAN HIEROGLYPH A407"                                              . #x145CB)
    ("ANATOLIAN HIEROGLYPH A408"                                              . #x145CC)
    ("ANATOLIAN HIEROGLYPH A409"                                              . #x145CD)
    ("ANATOLIAN HIEROGLYPH A410 BEGIN LOGOGRAM MARK"                          . #x145CE)
    ("ANATOLIAN HIEROGLYPH A410A END LOGOGRAM MARK"                           . #x145CF)
    ("ANATOLIAN HIEROGLYPH A411"                                              . #x145D0)
    ("ANATOLIAN HIEROGLYPH A412"                                              . #x145D1)
    ("ANATOLIAN HIEROGLYPH A413"                                              . #x145D2)
    ("ANATOLIAN HIEROGLYPH A414"                                              . #x145D3)
    ("ANATOLIAN HIEROGLYPH A415"                                              . #x145D4)
    ("ANATOLIAN HIEROGLYPH A416"                                              . #x145D5)
    ("ANATOLIAN HIEROGLYPH A417"                                              . #x145D6)
    ("ANATOLIAN HIEROGLYPH A418"                                              . #x145D7)
    ("ANATOLIAN HIEROGLYPH A419"                                              . #x145D8)
    ("ANATOLIAN HIEROGLYPH A420"                                              . #x145D9)
    ("ANATOLIAN HIEROGLYPH A421"                                              . #x145DA)
    ("ANATOLIAN HIEROGLYPH A422"                                              . #x145DB)
    ("ANATOLIAN HIEROGLYPH A423"                                              . #x145DC)
    ("ANATOLIAN HIEROGLYPH A424"                                              . #x145DD)
    ("ANATOLIAN HIEROGLYPH A425"                                              . #x145DE)
    ("ANATOLIAN HIEROGLYPH A426"                                              . #x145DF)
    ("ANATOLIAN HIEROGLYPH A427"                                              . #x145E0)
    ("ANATOLIAN HIEROGLYPH A428"                                              . #x145E1)
    ("ANATOLIAN HIEROGLYPH A429"                                              . #x145E2)
    ("ANATOLIAN HIEROGLYPH A430"                                              . #x145E3)
    ("ANATOLIAN HIEROGLYPH A431"                                              . #x145E4)
    ("ANATOLIAN HIEROGLYPH A432"                                              . #x145E5)
    ("ANATOLIAN HIEROGLYPH A433"                                              . #x145E6)
    ("ANATOLIAN HIEROGLYPH A434"                                              . #x145E7)
    ("ANATOLIAN HIEROGLYPH A435"                                              . #x145E8)
    ("ANATOLIAN HIEROGLYPH A436"                                              . #x145E9)
    ("ANATOLIAN HIEROGLYPH A437"                                              . #x145EA)
    ("ANATOLIAN HIEROGLYPH A438"                                              . #x145EB)
    ("ANATOLIAN HIEROGLYPH A439"                                              . #x145EC)
    ("ANATOLIAN HIEROGLYPH A440"                                              . #x145ED)
    ("ANATOLIAN HIEROGLYPH A441"                                              . #x145EE)
    ("ANATOLIAN HIEROGLYPH A442"                                              . #x145EF)
    ("ANATOLIAN HIEROGLYPH A443"                                              . #x145F0)
    ("ANATOLIAN HIEROGLYPH A444"                                              . #x145F1)
    ("ANATOLIAN HIEROGLYPH A445"                                              . #x145F2)
    ("ANATOLIAN HIEROGLYPH A446"                                              . #x145F3)
    ("ANATOLIAN HIEROGLYPH A447"                                              . #x145F4)
    ("ANATOLIAN HIEROGLYPH A448"                                              . #x145F5)
    ("ANATOLIAN HIEROGLYPH A449"                                              . #x145F6)
    ("ANATOLIAN HIEROGLYPH A450"                                              . #x145F7)
    ("ANATOLIAN HIEROGLYPH A450A"                                             . #x145F8)
    ("ANATOLIAN HIEROGLYPH A451"                                              . #x145F9)
    ("ANATOLIAN HIEROGLYPH A452"                                              . #x145FA)
    ("ANATOLIAN HIEROGLYPH A453"                                              . #x145FB)
    ("ANATOLIAN HIEROGLYPH A454"                                              . #x145FC)
    ("ANATOLIAN HIEROGLYPH A455"                                              . #x145FD)
    ("ANATOLIAN HIEROGLYPH A456"                                              . #x145FE)
    ("ANATOLIAN HIEROGLYPH A457"                                              . #x145FF)
    ("ANATOLIAN HIEROGLYPH A457A"                                             . #x14600)
    ("ANATOLIAN HIEROGLYPH A458"                                              . #x14601)
    ("ANATOLIAN HIEROGLYPH A459"                                              . #x14602)
    ("ANATOLIAN HIEROGLYPH A460"                                              . #x14603)
    ("ANATOLIAN HIEROGLYPH A461"                                              . #x14604)
    ("ANATOLIAN HIEROGLYPH A462"                                              . #x14605)
    ("ANATOLIAN HIEROGLYPH A463"                                              . #x14606)
    ("ANATOLIAN HIEROGLYPH A464"                                              . #x14607)
    ("ANATOLIAN HIEROGLYPH A465"                                              . #x14608)
    ("ANATOLIAN HIEROGLYPH A466"                                              . #x14609)
    ("ANATOLIAN HIEROGLYPH A467"                                              . #x1460A)
    ("ANATOLIAN HIEROGLYPH A468"                                              . #x1460B)
    ("ANATOLIAN HIEROGLYPH A469"                                              . #x1460C)
    ("ANATOLIAN HIEROGLYPH A470"                                              . #x1460D)
    ("ANATOLIAN HIEROGLYPH A471"                                              . #x1460E)
    ("ANATOLIAN HIEROGLYPH A472"                                              . #x1460F)
    ("ANATOLIAN HIEROGLYPH A473"                                              . #x14610)
    ("ANATOLIAN HIEROGLYPH A474"                                              . #x14611)
    ("ANATOLIAN HIEROGLYPH A475"                                              . #x14612)
    ("ANATOLIAN HIEROGLYPH A476"                                              . #x14613)
    ("ANATOLIAN HIEROGLYPH A477"                                              . #x14614)
    ("ANATOLIAN HIEROGLYPH A478"                                              . #x14615)
    ("ANATOLIAN HIEROGLYPH A479"                                              . #x14616)
    ("ANATOLIAN HIEROGLYPH A480"                                              . #x14617)
    ("ANATOLIAN HIEROGLYPH A481"                                              . #x14618)
    ("ANATOLIAN HIEROGLYPH A482"                                              . #x14619)
    ("ANATOLIAN HIEROGLYPH A483"                                              . #x1461A)
    ("ANATOLIAN HIEROGLYPH A484"                                              . #x1461B)
    ("ANATOLIAN HIEROGLYPH A485"                                              . #x1461C)
    ("ANATOLIAN HIEROGLYPH A486"                                              . #x1461D)
    ("ANATOLIAN HIEROGLYPH A487"                                              . #x1461E)
    ("ANATOLIAN HIEROGLYPH A488"                                              . #x1461F)
    ("ANATOLIAN HIEROGLYPH A489"                                              . #x14620)
    ("ANATOLIAN HIEROGLYPH A490"                                              . #x14621)
    ("ANATOLIAN HIEROGLYPH A491"                                              . #x14622)
    ("ANATOLIAN HIEROGLYPH A492"                                              . #x14623)
    ("ANATOLIAN HIEROGLYPH A493"                                              . #x14624)
    ("ANATOLIAN HIEROGLYPH A494"                                              . #x14625)
    ("ANATOLIAN HIEROGLYPH A495"                                              . #x14626)
    ("ANATOLIAN HIEROGLYPH A496"                                              . #x14627)
    ("ANATOLIAN HIEROGLYPH A497"                                              . #x14628)
    ("ANATOLIAN HIEROGLYPH A501"                                              . #x14629)
    ("ANATOLIAN HIEROGLYPH A502"                                              . #x1462A)
    ("ANATOLIAN HIEROGLYPH A503"                                              . #x1462B)
    ("ANATOLIAN HIEROGLYPH A504"                                              . #x1462C)
    ("ANATOLIAN HIEROGLYPH A505"                                              . #x1462D)
    ("ANATOLIAN HIEROGLYPH A506"                                              . #x1462E)
    ("ANATOLIAN HIEROGLYPH A507"                                              . #x1462F)
    ("ANATOLIAN HIEROGLYPH A508"                                              . #x14630)
    ("ANATOLIAN HIEROGLYPH A509"                                              . #x14631)
    ("ANATOLIAN HIEROGLYPH A510"                                              . #x14632)
    ("ANATOLIAN HIEROGLYPH A511"                                              . #x14633)
    ("ANATOLIAN HIEROGLYPH A512"                                              . #x14634)
    ("ANATOLIAN HIEROGLYPH A513"                                              . #x14635)
    ("ANATOLIAN HIEROGLYPH A514"                                              . #x14636)
    ("ANATOLIAN HIEROGLYPH A515"                                              . #x14637)
    ("ANATOLIAN HIEROGLYPH A516"                                              . #x14638)
    ("ANATOLIAN HIEROGLYPH A517"                                              . #x14639)
    ("ANATOLIAN HIEROGLYPH A518"                                              . #x1463A)
    ("ANATOLIAN HIEROGLYPH A519"                                              . #x1463B)
    ("ANATOLIAN HIEROGLYPH A520"                                              . #x1463C)
    ("ANATOLIAN HIEROGLYPH A521"                                              . #x1463D)
    ("ANATOLIAN HIEROGLYPH A522"                                              . #x1463E)
    ("ANATOLIAN HIEROGLYPH A523"                                              . #x1463F)
    ("ANATOLIAN HIEROGLYPH A524"                                              . #x14640)
    ("ANATOLIAN HIEROGLYPH A525"                                              . #x14641)
    ("ANATOLIAN HIEROGLYPH A526"                                              . #x14642)
    ("ANATOLIAN HIEROGLYPH A527"                                              . #x14643)
    ("ANATOLIAN HIEROGLYPH A528"                                              . #x14644)
    ("ANATOLIAN HIEROGLYPH A529"                                              . #x14645)
    ("ANATOLIAN HIEROGLYPH A530"                                              . #x14646)
    ("MUSICAL SYMBOL KIEVAN C CLEF"                                           . #x1D1DE)
    ("MUSICAL SYMBOL KIEVAN END OF PIECE"                                     . #x1D1DF)
    ("MUSICAL SYMBOL KIEVAN FINAL NOTE"                                       . #x1D1E0)
    ("MUSICAL SYMBOL KIEVAN RECITATIVE MARK"                                  . #x1D1E1)
    ("MUSICAL SYMBOL KIEVAN WHOLE NOTE"                                       . #x1D1E2)
    ("MUSICAL SYMBOL KIEVAN HALF NOTE"                                        . #x1D1E3)
    ("MUSICAL SYMBOL KIEVAN QUARTER NOTE STEM DOWN"                           . #x1D1E4)
    ("MUSICAL SYMBOL KIEVAN QUARTER NOTE STEM UP"                             . #x1D1E5)
    ("MUSICAL SYMBOL KIEVAN EIGHTH NOTE STEM DOWN"                            . #x1D1E6)
    ("MUSICAL SYMBOL KIEVAN EIGHTH NOTE STEM UP"                              . #x1D1E7)
    ("MUSICAL SYMBOL KIEVAN FLAT SIGN"                                        . #x1D1E8)
    ("SIGNWRITING HAND-FIST INDEX"                                            . #x1D800)
    ("SIGNWRITING HAND-CIRCLE INDEX"                                          . #x1D801)
    ("SIGNWRITING HAND-CUP INDEX"                                             . #x1D802)
    ("SIGNWRITING HAND-OVAL INDEX"                                            . #x1D803)
    ("SIGNWRITING HAND-HINGE INDEX"                                           . #x1D804)
    ("SIGNWRITING HAND-ANGLE INDEX"                                           . #x1D805)
    ("SIGNWRITING HAND-FIST INDEX BENT"                                       . #x1D806)
    ("SIGNWRITING HAND-CIRCLE INDEX BENT"                                     . #x1D807)
    ("SIGNWRITING HAND-FIST THUMB UNDER INDEX BENT"                           . #x1D808)
    ("SIGNWRITING HAND-FIST INDEX RAISED KNUCKLE"                             . #x1D809)
    ("SIGNWRITING HAND-FIST INDEX CUPPED"                                     . #x1D80A)
    ("SIGNWRITING HAND-FIST INDEX HINGED"                                     . #x1D80B)
    ("SIGNWRITING HAND-FIST INDEX HINGED LOW"                                 . #x1D80C)
    ("SIGNWRITING HAND-CIRCLE INDEX HINGE"                                    . #x1D80D)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE"                                     . #x1D80E)
    ("SIGNWRITING HAND-CIRCLE INDEX MIDDLE"                                   . #x1D80F)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE BENT"                                . #x1D810)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE RAISED KNUCKLES"                     . #x1D811)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE HINGED"                              . #x1D812)
    ("SIGNWRITING HAND-FIST INDEX UP MIDDLE HINGED"                           . #x1D813)
    ("SIGNWRITING HAND-FIST INDEX HINGED MIDDLE UP"                           . #x1D814)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED"                           . #x1D815)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED INDEX BENT"                . #x1D816)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED MIDDLE BENT"               . #x1D817)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED CUPPED"                    . #x1D818)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED HINGED"                    . #x1D819)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE CROSSED"                             . #x1D81A)
    ("SIGNWRITING HAND-CIRCLE INDEX MIDDLE CROSSED"                           . #x1D81B)
    ("SIGNWRITING HAND-FIST MIDDLE BENT OVER INDEX"                           . #x1D81C)
    ("SIGNWRITING HAND-FIST INDEX BENT OVER MIDDLE"                           . #x1D81D)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE THUMB"                               . #x1D81E)
    ("SIGNWRITING HAND-CIRCLE INDEX MIDDLE THUMB"                             . #x1D81F)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE STRAIGHT THUMB BENT"                 . #x1D820)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE BENT THUMB STRAIGHT"                 . #x1D821)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE THUMB BENT"                          . #x1D822)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE HINGED SPREAD THUMB SIDE"            . #x1D823)
    ("SIGNWRITING HAND-FIST INDEX UP MIDDLE HINGED THUMB SIDE"                . #x1D824)
    ("SIGNWRITING HAND-FIST INDEX UP MIDDLE HINGED THUMB CONJOINED"           . #x1D825)
    ("SIGNWRITING HAND-FIST INDEX HINGED MIDDLE UP THUMB SIDE"                . #x1D826)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE UP SPREAD THUMB FORWARD"             . #x1D827)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE THUMB CUPPED"                        . #x1D828)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE THUMB CIRCLED"                       . #x1D829)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE THUMB HOOKED"                        . #x1D82A)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE THUMB HINGED"                        . #x1D82B)
    ("SIGNWRITING HAND-FIST THUMB BETWEEN INDEX MIDDLE STRAIGHT"              . #x1D82C)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED THUMB SIDE"                . #x1D82D)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED THUMB SIDE CONJOINED"      . #x1D82E)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED THUMB SIDE BENT"           . #x1D82F)
    ("SIGNWRITING HAND-FIST MIDDLE THUMB HOOKED INDEX UP"                     . #x1D830)
    ("SIGNWRITING HAND-FIST INDEX THUMB HOOKED MIDDLE UP"                     . #x1D831)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED HINGED THUMB SIDE"         . #x1D832)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE CROSSED THUMB SIDE"                  . #x1D833)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED THUMB FORWARD"             . #x1D834)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE CONJOINED CUPPED THUMB FORWARD"      . #x1D835)
    ("SIGNWRITING HAND-FIST MIDDLE THUMB CUPPED INDEX UP"                     . #x1D836)
    ("SIGNWRITING HAND-FIST INDEX THUMB CUPPED MIDDLE UP"                     . #x1D837)
    ("SIGNWRITING HAND-FIST MIDDLE THUMB CIRCLED INDEX UP"                    . #x1D838)
    ("SIGNWRITING HAND-FIST MIDDLE THUMB CIRCLED INDEX HINGED"                . #x1D839)
    ("SIGNWRITING HAND-FIST INDEX THUMB ANGLED OUT MIDDLE UP"                 . #x1D83A)
    ("SIGNWRITING HAND-FIST INDEX THUMB ANGLED IN MIDDLE UP"                  . #x1D83B)
    ("SIGNWRITING HAND-FIST INDEX THUMB CIRCLED MIDDLE UP"                    . #x1D83C)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE THUMB CONJOINED HINGED"              . #x1D83D)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE THUMB ANGLED OUT"                    . #x1D83E)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE THUMB ANGLED"                        . #x1D83F)
    ("SIGNWRITING HAND-FIST MIDDLE THUMB ANGLED OUT INDEX UP"                 . #x1D840)
    ("SIGNWRITING HAND-FIST MIDDLE THUMB ANGLED OUT INDEX CROSSED"            . #x1D841)
    ("SIGNWRITING HAND-FIST MIDDLE THUMB ANGLED INDEX UP"                     . #x1D842)
    ("SIGNWRITING HAND-FIST INDEX THUMB HOOKED MIDDLE HINGED"                 . #x1D843)
    ("SIGNWRITING HAND-FLAT FOUR FINGERS"                                     . #x1D844)
    ("SIGNWRITING HAND-FLAT FOUR FINGERS BENT"                                . #x1D845)
    ("SIGNWRITING HAND-FLAT FOUR FINGERS HINGED"                              . #x1D846)
    ("SIGNWRITING HAND-FLAT FOUR FINGERS CONJOINED"                           . #x1D847)
    ("SIGNWRITING HAND-FLAT FOUR FINGERS CONJOINED SPLIT"                     . #x1D848)
    ("SIGNWRITING HAND-CLAW FOUR FINGERS CONJOINED"                           . #x1D849)
    ("SIGNWRITING HAND-FIST FOUR FINGERS CONJOINED BENT"                      . #x1D84A)
    ("SIGNWRITING HAND-HINGE FOUR FINGERS CONJOINED"                          . #x1D84B)
    ("SIGNWRITING HAND-FLAT FIVE FINGERS SPREAD"                              . #x1D84C)
    ("SIGNWRITING HAND-FLAT HEEL FIVE FINGERS SPREAD"                         . #x1D84D)
    ("SIGNWRITING HAND-FLAT FIVE FINGERS SPREAD FOUR BENT"                    . #x1D84E)
    ("SIGNWRITING HAND-FLAT HEEL FIVE FINGERS SPREAD FOUR BENT"               . #x1D84F)
    ("SIGNWRITING HAND-FLAT FIVE FINGERS SPREAD BENT"                         . #x1D850)
    ("SIGNWRITING HAND-FLAT HEEL FIVE FINGERS SPREAD BENT"                    . #x1D851)
    ("SIGNWRITING HAND-FLAT FIVE FINGERS SPREAD THUMB FORWARD"                . #x1D852)
    ("SIGNWRITING HAND-CUP FIVE FINGERS SPREAD"                               . #x1D853)
    ("SIGNWRITING HAND-CUP FIVE FINGERS SPREAD OPEN"                          . #x1D854)
    ("SIGNWRITING HAND-HINGE FIVE FINGERS SPREAD OPEN"                        . #x1D855)
    ("SIGNWRITING HAND-OVAL FIVE FINGERS SPREAD"                              . #x1D856)
    ("SIGNWRITING HAND-FLAT FIVE FINGERS SPREAD HINGED"                       . #x1D857)
    ("SIGNWRITING HAND-FLAT FIVE FINGERS SPREAD HINGED THUMB SIDE"            . #x1D858)
    ("SIGNWRITING HAND-FLAT FIVE FINGERS SPREAD HINGED NO THUMB"              . #x1D859)
    ("SIGNWRITING HAND-FLAT"                                                  . #x1D85A)
    ("SIGNWRITING HAND-FLAT BETWEEN PALM FACINGS"                             . #x1D85B)
    ("SIGNWRITING HAND-FLAT HEEL"                                             . #x1D85C)
    ("SIGNWRITING HAND-FLAT THUMB SIDE"                                       . #x1D85D)
    ("SIGNWRITING HAND-FLAT HEEL THUMB SIDE"                                  . #x1D85E)
    ("SIGNWRITING HAND-FLAT THUMB BENT"                                       . #x1D85F)
    ("SIGNWRITING HAND-FLAT THUMB FORWARD"                                    . #x1D860)
    ("SIGNWRITING HAND-FLAT SPLIT INDEX THUMB SIDE"                           . #x1D861)
    ("SIGNWRITING HAND-FLAT SPLIT CENTRE"                                     . #x1D862)
    ("SIGNWRITING HAND-FLAT SPLIT CENTRE THUMB SIDE"                          . #x1D863)
    ("SIGNWRITING HAND-FLAT SPLIT CENTRE THUMB SIDE BENT"                     . #x1D864)
    ("SIGNWRITING HAND-FLAT SPLIT LITTLE"                                     . #x1D865)
    ("SIGNWRITING HAND-CLAW"                                                  . #x1D866)
    ("SIGNWRITING HAND-CLAW THUMB SIDE"                                       . #x1D867)
    ("SIGNWRITING HAND-CLAW NO THUMB"                                         . #x1D868)
    ("SIGNWRITING HAND-CLAW THUMB FORWARD"                                    . #x1D869)
    ("SIGNWRITING HAND-HOOK CURLICUE"                                         . #x1D86A)
    ("SIGNWRITING HAND-HOOK"                                                  . #x1D86B)
    ("SIGNWRITING HAND-CUP OPEN"                                              . #x1D86C)
    ("SIGNWRITING HAND-CUP"                                                   . #x1D86D)
    ("SIGNWRITING HAND-CUP OPEN THUMB SIDE"                                   . #x1D86E)
    ("SIGNWRITING HAND-CUP THUMB SIDE"                                        . #x1D86F)
    ("SIGNWRITING HAND-CUP OPEN NO THUMB"                                     . #x1D870)
    ("SIGNWRITING HAND-CUP NO THUMB"                                          . #x1D871)
    ("SIGNWRITING HAND-CUP OPEN THUMB FORWARD"                                . #x1D872)
    ("SIGNWRITING HAND-CUP THUMB FORWARD"                                     . #x1D873)
    ("SIGNWRITING HAND-CURLICUE OPEN"                                         . #x1D874)
    ("SIGNWRITING HAND-CURLICUE"                                              . #x1D875)
    ("SIGNWRITING HAND-CIRCLE"                                                . #x1D876)
    ("SIGNWRITING HAND-OVAL"                                                  . #x1D877)
    ("SIGNWRITING HAND-OVAL THUMB SIDE"                                       . #x1D878)
    ("SIGNWRITING HAND-OVAL NO THUMB"                                         . #x1D879)
    ("SIGNWRITING HAND-OVAL THUMB FORWARD"                                    . #x1D87A)
    ("SIGNWRITING HAND-HINGE OPEN"                                            . #x1D87B)
    ("SIGNWRITING HAND-HINGE OPEN THUMB FORWARD"                              . #x1D87C)
    ("SIGNWRITING HAND-HINGE"                                                 . #x1D87D)
    ("SIGNWRITING HAND-HINGE SMALL"                                           . #x1D87E)
    ("SIGNWRITING HAND-HINGE OPEN THUMB SIDE"                                 . #x1D87F)
    ("SIGNWRITING HAND-HINGE THUMB SIDE"                                      . #x1D880)
    ("SIGNWRITING HAND-HINGE OPEN NO THUMB"                                   . #x1D881)
    ("SIGNWRITING HAND-HINGE NO THUMB"                                        . #x1D882)
    ("SIGNWRITING HAND-HINGE THUMB SIDE TOUCHING INDEX"                       . #x1D883)
    ("SIGNWRITING HAND-HINGE THUMB BETWEEN MIDDLE RING"                       . #x1D884)
    ("SIGNWRITING HAND-ANGLE"                                                 . #x1D885)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE RING"                                . #x1D886)
    ("SIGNWRITING HAND-CIRCLE INDEX MIDDLE RING"                              . #x1D887)
    ("SIGNWRITING HAND-HINGE INDEX MIDDLE RING"                               . #x1D888)
    ("SIGNWRITING HAND-ANGLE INDEX MIDDLE RING"                               . #x1D889)
    ("SIGNWRITING HAND-HINGE LITTLE"                                          . #x1D88A)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE RING BENT"                           . #x1D88B)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE RING CONJOINED"                      . #x1D88C)
    ("SIGNWRITING HAND-HINGE INDEX MIDDLE RING CONJOINED"                     . #x1D88D)
    ("SIGNWRITING HAND-FIST LITTLE DOWN"                                      . #x1D88E)
    ("SIGNWRITING HAND-FIST LITTLE DOWN RIPPLE STRAIGHT"                      . #x1D88F)
    ("SIGNWRITING HAND-FIST LITTLE DOWN RIPPLE CURVED"                        . #x1D890)
    ("SIGNWRITING HAND-FIST LITTLE DOWN OTHERS CIRCLED"                       . #x1D891)
    ("SIGNWRITING HAND-FIST LITTLE UP"                                        . #x1D892)
    ("SIGNWRITING HAND-FIST THUMB UNDER LITTLE UP"                            . #x1D893)
    ("SIGNWRITING HAND-CIRCLE LITTLE UP"                                      . #x1D894)
    ("SIGNWRITING HAND-OVAL LITTLE UP"                                        . #x1D895)
    ("SIGNWRITING HAND-ANGLE LITTLE UP"                                       . #x1D896)
    ("SIGNWRITING HAND-FIST LITTLE RAISED KNUCKLE"                            . #x1D897)
    ("SIGNWRITING HAND-FIST LITTLE BENT"                                      . #x1D898)
    ("SIGNWRITING HAND-FIST LITTLE TOUCHES THUMB"                             . #x1D899)
    ("SIGNWRITING HAND-FIST LITTLE THUMB"                                     . #x1D89A)
    ("SIGNWRITING HAND-HINGE LITTLE THUMB"                                    . #x1D89B)
    ("SIGNWRITING HAND-FIST LITTLE INDEX THUMB"                               . #x1D89C)
    ("SIGNWRITING HAND-HINGE LITTLE INDEX THUMB"                              . #x1D89D)
    ("SIGNWRITING HAND-ANGLE LITTLE INDEX THUMB INDEX THUMB OUT"              . #x1D89E)
    ("SIGNWRITING HAND-ANGLE LITTLE INDEX THUMB INDEX THUMB"                  . #x1D89F)
    ("SIGNWRITING HAND-FIST LITTLE INDEX"                                     . #x1D8A0)
    ("SIGNWRITING HAND-CIRCLE LITTLE INDEX"                                   . #x1D8A1)
    ("SIGNWRITING HAND-HINGE LITTLE INDEX"                                    . #x1D8A2)
    ("SIGNWRITING HAND-ANGLE LITTLE INDEX"                                    . #x1D8A3)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE LITTLE"                              . #x1D8A4)
    ("SIGNWRITING HAND-CIRCLE INDEX MIDDLE LITTLE"                            . #x1D8A5)
    ("SIGNWRITING HAND-HINGE INDEX MIDDLE LITTLE"                             . #x1D8A6)
    ("SIGNWRITING HAND-HINGE RING"                                            . #x1D8A7)
    ("SIGNWRITING HAND-ANGLE INDEX MIDDLE LITTLE"                             . #x1D8A8)
    ("SIGNWRITING HAND-FIST INDEX MIDDLE CROSS LITTLE"                        . #x1D8A9)
    ("SIGNWRITING HAND-CIRCLE INDEX MIDDLE CROSS LITTLE"                      . #x1D8AA)
    ("SIGNWRITING HAND-FIST RING DOWN"                                        . #x1D8AB)
    ("SIGNWRITING HAND-HINGE RING DOWN INDEX THUMB HOOK MIDDLE"               . #x1D8AC)
    ("SIGNWRITING HAND-ANGLE RING DOWN MIDDLE THUMB INDEX CROSS"              . #x1D8AD)
    ("SIGNWRITING HAND-FIST RING UP"                                          . #x1D8AE)
    ("SIGNWRITING HAND-FIST RING RAISED KNUCKLE"                              . #x1D8AF)
    ("SIGNWRITING HAND-FIST RING LITTLE"                                      . #x1D8B0)
    ("SIGNWRITING HAND-CIRCLE RING LITTLE"                                    . #x1D8B1)
    ("SIGNWRITING HAND-OVAL RING LITTLE"                                      . #x1D8B2)
    ("SIGNWRITING HAND-ANGLE RING LITTLE"                                     . #x1D8B3)
    ("SIGNWRITING HAND-FIST RING MIDDLE"                                      . #x1D8B4)
    ("SIGNWRITING HAND-FIST RING MIDDLE CONJOINED"                            . #x1D8B5)
    ("SIGNWRITING HAND-FIST RING MIDDLE RAISED KNUCKLES"                      . #x1D8B6)
    ("SIGNWRITING HAND-FIST RING INDEX"                                       . #x1D8B7)
    ("SIGNWRITING HAND-FIST RING THUMB"                                       . #x1D8B8)
    ("SIGNWRITING HAND-HOOK RING THUMB"                                       . #x1D8B9)
    ("SIGNWRITING HAND-FIST INDEX RING LITTLE"                                . #x1D8BA)
    ("SIGNWRITING HAND-CIRCLE INDEX RING LITTLE"                              . #x1D8BB)
    ("SIGNWRITING HAND-CURLICUE INDEX RING LITTLE ON"                         . #x1D8BC)
    ("SIGNWRITING HAND-HOOK INDEX RING LITTLE OUT"                            . #x1D8BD)
    ("SIGNWRITING HAND-HOOK INDEX RING LITTLE IN"                             . #x1D8BE)
    ("SIGNWRITING HAND-HOOK INDEX RING LITTLE UNDER"                          . #x1D8BF)
    ("SIGNWRITING HAND-CUP INDEX RING LITTLE"                                 . #x1D8C0)
    ("SIGNWRITING HAND-HINGE INDEX RING LITTLE"                               . #x1D8C1)
    ("SIGNWRITING HAND-ANGLE INDEX RING LITTLE OUT"                           . #x1D8C2)
    ("SIGNWRITING HAND-ANGLE INDEX RING LITTLE"                               . #x1D8C3)
    ("SIGNWRITING HAND-FIST MIDDLE DOWN"                                      . #x1D8C4)
    ("SIGNWRITING HAND-HINGE MIDDLE"                                          . #x1D8C5)
    ("SIGNWRITING HAND-FIST MIDDLE UP"                                        . #x1D8C6)
    ("SIGNWRITING HAND-CIRCLE MIDDLE UP"                                      . #x1D8C7)
    ("SIGNWRITING HAND-FIST MIDDLE RAISED KNUCKLE"                            . #x1D8C8)
    ("SIGNWRITING HAND-FIST MIDDLE UP THUMB SIDE"                             . #x1D8C9)
    ("SIGNWRITING HAND-HOOK MIDDLE THUMB"                                     . #x1D8CA)
    ("SIGNWRITING HAND-FIST MIDDLE THUMB LITTLE"                              . #x1D8CB)
    ("SIGNWRITING HAND-FIST MIDDLE LITTLE"                                    . #x1D8CC)
    ("SIGNWRITING HAND-FIST MIDDLE RING LITTLE"                               . #x1D8CD)
    ("SIGNWRITING HAND-CIRCLE MIDDLE RING LITTLE"                             . #x1D8CE)
    ("SIGNWRITING HAND-CURLICUE MIDDLE RING LITTLE ON"                        . #x1D8CF)
    ("SIGNWRITING HAND-CUP MIDDLE RING LITTLE"                                . #x1D8D0)
    ("SIGNWRITING HAND-HINGE MIDDLE RING LITTLE"                              . #x1D8D1)
    ("SIGNWRITING HAND-ANGLE MIDDLE RING LITTLE OUT"                          . #x1D8D2)
    ("SIGNWRITING HAND-ANGLE MIDDLE RING LITTLE IN"                           . #x1D8D3)
    ("SIGNWRITING HAND-ANGLE MIDDLE RING LITTLE"                              . #x1D8D4)
    ("SIGNWRITING HAND-CIRCLE MIDDLE RING LITTLE BENT"                        . #x1D8D5)
    ("SIGNWRITING HAND-CLAW MIDDLE RING LITTLE CONJOINED"                     . #x1D8D6)
    ("SIGNWRITING HAND-CLAW MIDDLE RING LITTLE CONJOINED SIDE"                . #x1D8D7)
    ("SIGNWRITING HAND-HOOK MIDDLE RING LITTLE CONJOINED OUT"                 . #x1D8D8)
    ("SIGNWRITING HAND-HOOK MIDDLE RING LITTLE CONJOINED IN"                  . #x1D8D9)
    ("SIGNWRITING HAND-HOOK MIDDLE RING LITTLE CONJOINED"                     . #x1D8DA)
    ("SIGNWRITING HAND-HINGE INDEX HINGED"                                    . #x1D8DB)
    ("SIGNWRITING HAND-FIST INDEX THUMB SIDE"                                 . #x1D8DC)
    ("SIGNWRITING HAND-HINGE INDEX THUMB SIDE"                                . #x1D8DD)
    ("SIGNWRITING HAND-FIST INDEX THUMB SIDE THUMB DIAGONAL"                  . #x1D8DE)
    ("SIGNWRITING HAND-FIST INDEX THUMB SIDE THUMB CONJOINED"                 . #x1D8DF)
    ("SIGNWRITING HAND-FIST INDEX THUMB SIDE THUMB BENT"                      . #x1D8E0)
    ("SIGNWRITING HAND-FIST INDEX THUMB SIDE INDEX BENT"                      . #x1D8E1)
    ("SIGNWRITING HAND-FIST INDEX THUMB SIDE BOTH BENT"                       . #x1D8E2)
    ("SIGNWRITING HAND-FIST INDEX THUMB SIDE INDEX HINGE"                     . #x1D8E3)
    ("SIGNWRITING HAND-FIST INDEX THUMB FORWARD INDEX STRAIGHT"               . #x1D8E4)
    ("SIGNWRITING HAND-FIST INDEX THUMB FORWARD INDEX BENT"                   . #x1D8E5)
    ("SIGNWRITING HAND-FIST INDEX THUMB HOOK"                                 . #x1D8E6)
    ("SIGNWRITING HAND-FIST INDEX THUMB CURLICUE"                             . #x1D8E7)
    ("SIGNWRITING HAND-FIST INDEX THUMB CURVE THUMB INSIDE"                   . #x1D8E8)
    ("SIGNWRITING HAND-CLAW INDEX THUMB CURVE THUMB INSIDE"                   . #x1D8E9)
    ("SIGNWRITING HAND-FIST INDEX THUMB CURVE THUMB UNDER"                    . #x1D8EA)
    ("SIGNWRITING HAND-FIST INDEX THUMB CIRCLE"                               . #x1D8EB)
    ("SIGNWRITING HAND-CUP INDEX THUMB"                                       . #x1D8EC)
    ("SIGNWRITING HAND-CUP INDEX THUMB OPEN"                                  . #x1D8ED)
    ("SIGNWRITING HAND-HINGE INDEX THUMB OPEN"                                . #x1D8EE)
    ("SIGNWRITING HAND-HINGE INDEX THUMB LARGE"                               . #x1D8EF)
    ("SIGNWRITING HAND-HINGE INDEX THUMB"                                     . #x1D8F0)
    ("SIGNWRITING HAND-HINGE INDEX THUMB SMALL"                               . #x1D8F1)
    ("SIGNWRITING HAND-ANGLE INDEX THUMB OUT"                                 . #x1D8F2)
    ("SIGNWRITING HAND-ANGLE INDEX THUMB IN"                                  . #x1D8F3)
    ("SIGNWRITING HAND-ANGLE INDEX THUMB"                                     . #x1D8F4)
    ("SIGNWRITING HAND-FIST THUMB"                                            . #x1D8F5)
    ("SIGNWRITING HAND-FIST THUMB HEEL"                                       . #x1D8F6)
    ("SIGNWRITING HAND-FIST THUMB SIDE DIAGONAL"                              . #x1D8F7)
    ("SIGNWRITING HAND-FIST THUMB SIDE CONJOINED"                             . #x1D8F8)
    ("SIGNWRITING HAND-FIST THUMB SIDE BENT"                                  . #x1D8F9)
    ("SIGNWRITING HAND-FIST THUMB FORWARD"                                    . #x1D8FA)
    ("SIGNWRITING HAND-FIST THUMB BETWEEN INDEX MIDDLE"                       . #x1D8FB)
    ("SIGNWRITING HAND-FIST THUMB BETWEEN MIDDLE RING"                        . #x1D8FC)
    ("SIGNWRITING HAND-FIST THUMB BETWEEN RING LITTLE"                        . #x1D8FD)
    ("SIGNWRITING HAND-FIST THUMB UNDER TWO FINGERS"                          . #x1D8FE)
    ("SIGNWRITING HAND-FIST THUMB OVER TWO FINGERS"                           . #x1D8FF)
    ("SIGNWRITING HAND-FIST THUMB UNDER THREE FINGERS"                        . #x1D900)
    ("SIGNWRITING HAND-FIST THUMB UNDER FOUR FINGERS"                         . #x1D901)
    ("SIGNWRITING HAND-FIST THUMB OVER FOUR RAISED KNUCKLES"                  . #x1D902)
    ("SIGNWRITING HAND-FIST"                                                  . #x1D903)
    ("SIGNWRITING HAND-FIST HEEL"                                             . #x1D904)
    ("SIGNWRITING TOUCH SINGLE"                                               . #x1D905)
    ("SIGNWRITING TOUCH MULTIPLE"                                             . #x1D906)
    ("SIGNWRITING TOUCH BETWEEN"                                              . #x1D907)
    ("SIGNWRITING GRASP SINGLE"                                               . #x1D908)
    ("SIGNWRITING GRASP MULTIPLE"                                             . #x1D909)
    ("SIGNWRITING GRASP BETWEEN"                                              . #x1D90A)
    ("SIGNWRITING STRIKE SINGLE"                                              . #x1D90B)
    ("SIGNWRITING STRIKE MULTIPLE"                                            . #x1D90C)
    ("SIGNWRITING STRIKE BETWEEN"                                             . #x1D90D)
    ("SIGNWRITING BRUSH SINGLE"                                               . #x1D90E)
    ("SIGNWRITING BRUSH MULTIPLE"                                             . #x1D90F)
    ("SIGNWRITING BRUSH BETWEEN"                                              . #x1D910)
    ("SIGNWRITING RUB SINGLE"                                                 . #x1D911)
    ("SIGNWRITING RUB MULTIPLE"                                               . #x1D912)
    ("SIGNWRITING RUB BETWEEN"                                                . #x1D913)
    ("SIGNWRITING SURFACE SYMBOLS"                                            . #x1D914)
    ("SIGNWRITING SURFACE BETWEEN"                                            . #x1D915)
    ("SIGNWRITING SQUEEZE LARGE SINGLE"                                       . #x1D916)
    ("SIGNWRITING SQUEEZE SMALL SINGLE"                                       . #x1D917)
    ("SIGNWRITING SQUEEZE LARGE MULTIPLE"                                     . #x1D918)
    ("SIGNWRITING SQUEEZE SMALL MULTIPLE"                                     . #x1D919)
    ("SIGNWRITING SQUEEZE SEQUENTIAL"                                         . #x1D91A)
    ("SIGNWRITING FLICK LARGE SINGLE"                                         . #x1D91B)
    ("SIGNWRITING FLICK SMALL SINGLE"                                         . #x1D91C)
    ("SIGNWRITING FLICK LARGE MULTIPLE"                                       . #x1D91D)
    ("SIGNWRITING FLICK SMALL MULTIPLE"                                       . #x1D91E)
    ("SIGNWRITING FLICK SEQUENTIAL"                                           . #x1D91F)
    ("SIGNWRITING SQUEEZE FLICK ALTERNATING"                                  . #x1D920)
    ("SIGNWRITING MOVEMENT-HINGE UP DOWN LARGE"                               . #x1D921)
    ("SIGNWRITING MOVEMENT-HINGE UP DOWN SMALL"                               . #x1D922)
    ("SIGNWRITING MOVEMENT-HINGE UP SEQUENTIAL"                               . #x1D923)
    ("SIGNWRITING MOVEMENT-HINGE DOWN SEQUENTIAL"                             . #x1D924)
    ("SIGNWRITING MOVEMENT-HINGE UP DOWN ALTERNATING LARGE"                   . #x1D925)
    ("SIGNWRITING MOVEMENT-HINGE UP DOWN ALTERNATING SMALL"                   . #x1D926)
    ("SIGNWRITING MOVEMENT-HINGE SIDE TO SIDE SCISSORS"                       . #x1D927)
    ("SIGNWRITING MOVEMENT-WALLPLANE FINGER CONTACT"                          . #x1D928)
    ("SIGNWRITING MOVEMENT-FLOORPLANE FINGER CONTACT"                         . #x1D929)
    ("SIGNWRITING MOVEMENT-WALLPLANE SINGLE STRAIGHT SMALL"                   . #x1D92A)
    ("SIGNWRITING MOVEMENT-WALLPLANE SINGLE STRAIGHT MEDIUM"                  . #x1D92B)
    ("SIGNWRITING MOVEMENT-WALLPLANE SINGLE STRAIGHT LARGE"                   . #x1D92C)
    ("SIGNWRITING MOVEMENT-WALLPLANE SINGLE STRAIGHT LARGEST"                 . #x1D92D)
    ("SIGNWRITING MOVEMENT-WALLPLANE SINGLE WRIST FLEX"                       . #x1D92E)
    ("SIGNWRITING MOVEMENT-WALLPLANE DOUBLE STRAIGHT"                         . #x1D92F)
    ("SIGNWRITING MOVEMENT-WALLPLANE DOUBLE WRIST FLEX"                       . #x1D930)
    ("SIGNWRITING MOVEMENT-WALLPLANE DOUBLE ALTERNATING"                      . #x1D931)
    ("SIGNWRITING MOVEMENT-WALLPLANE DOUBLE ALTERNATING WRIST FLEX"           . #x1D932)
    ("SIGNWRITING MOVEMENT-WALLPLANE CROSS"                                   . #x1D933)
    ("SIGNWRITING MOVEMENT-WALLPLANE TRIPLE STRAIGHT MOVEMENT"                . #x1D934)
    ("SIGNWRITING MOVEMENT-WALLPLANE TRIPLE WRIST FLEX"                       . #x1D935)
    ("SIGNWRITING MOVEMENT-WALLPLANE TRIPLE ALTERNATING"                      . #x1D936)
    ("SIGNWRITING MOVEMENT-WALLPLANE TRIPLE ALTERNATING WRIST FLEX"           . #x1D937)
    ("SIGNWRITING MOVEMENT-WALLPLANE BEND SMALL"                              . #x1D938)
    ("SIGNWRITING MOVEMENT-WALLPLANE BEND MEDIUM"                             . #x1D939)
    ("SIGNWRITING MOVEMENT-WALLPLANE BEND LARGE"                              . #x1D93A)
    ("SIGNWRITING MOVEMENT-WALLPLANE CORNER SMALL"                            . #x1D93B)
    ("SIGNWRITING MOVEMENT-WALLPLANE CORNER MEDIUM"                           . #x1D93C)
    ("SIGNWRITING MOVEMENT-WALLPLANE CORNER LARGE"                            . #x1D93D)
    ("SIGNWRITING MOVEMENT-WALLPLANE CORNER ROTATION"                         . #x1D93E)
    ("SIGNWRITING MOVEMENT-WALLPLANE CHECK SMALL"                             . #x1D93F)
    ("SIGNWRITING MOVEMENT-WALLPLANE CHECK MEDIUM"                            . #x1D940)
    ("SIGNWRITING MOVEMENT-WALLPLANE CHECK LARGE"                             . #x1D941)
    ("SIGNWRITING MOVEMENT-WALLPLANE BOX SMALL"                               . #x1D942)
    ("SIGNWRITING MOVEMENT-WALLPLANE BOX MEDIUM"                              . #x1D943)
    ("SIGNWRITING MOVEMENT-WALLPLANE BOX LARGE"                               . #x1D944)
    ("SIGNWRITING MOVEMENT-WALLPLANE ZIGZAG SMALL"                            . #x1D945)
    ("SIGNWRITING MOVEMENT-WALLPLANE ZIGZAG MEDIUM"                           . #x1D946)
    ("SIGNWRITING MOVEMENT-WALLPLANE ZIGZAG LARGE"                            . #x1D947)
    ("SIGNWRITING MOVEMENT-WALLPLANE PEAKS SMALL"                             . #x1D948)
    ("SIGNWRITING MOVEMENT-WALLPLANE PEAKS MEDIUM"                            . #x1D949)
    ("SIGNWRITING MOVEMENT-WALLPLANE PEAKS LARGE"                             . #x1D94A)
    ("SIGNWRITING TRAVEL-WALLPLANE ROTATION-WALLPLANE SINGLE"                 . #x1D94B)
    ("SIGNWRITING TRAVEL-WALLPLANE ROTATION-WALLPLANE DOUBLE"                 . #x1D94C)
    ("SIGNWRITING TRAVEL-WALLPLANE ROTATION-WALLPLANE ALTERNATING"            . #x1D94D)
    ("SIGNWRITING TRAVEL-WALLPLANE ROTATION-FLOORPLANE SINGLE"                . #x1D94E)
    ("SIGNWRITING TRAVEL-WALLPLANE ROTATION-FLOORPLANE DOUBLE"                . #x1D94F)
    ("SIGNWRITING TRAVEL-WALLPLANE ROTATION-FLOORPLANE ALTERNATING"           . #x1D950)
    ("SIGNWRITING TRAVEL-WALLPLANE SHAKING"                                   . #x1D951)
    ("SIGNWRITING TRAVEL-WALLPLANE ARM SPIRAL SINGLE"                         . #x1D952)
    ("SIGNWRITING TRAVEL-WALLPLANE ARM SPIRAL DOUBLE"                         . #x1D953)
    ("SIGNWRITING TRAVEL-WALLPLANE ARM SPIRAL TRIPLE"                         . #x1D954)
    ("SIGNWRITING MOVEMENT-DIAGONAL AWAY SMALL"                               . #x1D955)
    ("SIGNWRITING MOVEMENT-DIAGONAL AWAY MEDIUM"                              . #x1D956)
    ("SIGNWRITING MOVEMENT-DIAGONAL AWAY LARGE"                               . #x1D957)
    ("SIGNWRITING MOVEMENT-DIAGONAL AWAY LARGEST"                             . #x1D958)
    ("SIGNWRITING MOVEMENT-DIAGONAL TOWARDS SMALL"                            . #x1D959)
    ("SIGNWRITING MOVEMENT-DIAGONAL TOWARDS MEDIUM"                           . #x1D95A)
    ("SIGNWRITING MOVEMENT-DIAGONAL TOWARDS LARGE"                            . #x1D95B)
    ("SIGNWRITING MOVEMENT-DIAGONAL TOWARDS LARGEST"                          . #x1D95C)
    ("SIGNWRITING MOVEMENT-DIAGONAL BETWEEN AWAY SMALL"                       . #x1D95D)
    ("SIGNWRITING MOVEMENT-DIAGONAL BETWEEN AWAY MEDIUM"                      . #x1D95E)
    ("SIGNWRITING MOVEMENT-DIAGONAL BETWEEN AWAY LARGE"                       . #x1D95F)
    ("SIGNWRITING MOVEMENT-DIAGONAL BETWEEN AWAY LARGEST"                     . #x1D960)
    ("SIGNWRITING MOVEMENT-DIAGONAL BETWEEN TOWARDS SMALL"                    . #x1D961)
    ("SIGNWRITING MOVEMENT-DIAGONAL BETWEEN TOWARDS MEDIUM"                   . #x1D962)
    ("SIGNWRITING MOVEMENT-DIAGONAL BETWEEN TOWARDS LARGE"                    . #x1D963)
    ("SIGNWRITING MOVEMENT-DIAGONAL BETWEEN TOWARDS LARGEST"                  . #x1D964)
    ("SIGNWRITING MOVEMENT-FLOORPLANE SINGLE STRAIGHT SMALL"                  . #x1D965)
    ("SIGNWRITING MOVEMENT-FLOORPLANE SINGLE STRAIGHT MEDIUM"                 . #x1D966)
    ("SIGNWRITING MOVEMENT-FLOORPLANE SINGLE STRAIGHT LARGE"                  . #x1D967)
    ("SIGNWRITING MOVEMENT-FLOORPLANE SINGLE STRAIGHT LARGEST"                . #x1D968)
    ("SIGNWRITING MOVEMENT-FLOORPLANE SINGLE WRIST FLEX"                      . #x1D969)
    ("SIGNWRITING MOVEMENT-FLOORPLANE DOUBLE STRAIGHT"                        . #x1D96A)
    ("SIGNWRITING MOVEMENT-FLOORPLANE DOUBLE WRIST FLEX"                      . #x1D96B)
    ("SIGNWRITING MOVEMENT-FLOORPLANE DOUBLE ALTERNATING"                     . #x1D96C)
    ("SIGNWRITING MOVEMENT-FLOORPLANE DOUBLE ALTERNATING WRIST FLEX"          . #x1D96D)
    ("SIGNWRITING MOVEMENT-FLOORPLANE CROSS"                                  . #x1D96E)
    ("SIGNWRITING MOVEMENT-FLOORPLANE TRIPLE STRAIGHT MOVEMENT"               . #x1D96F)
    ("SIGNWRITING MOVEMENT-FLOORPLANE TRIPLE WRIST FLEX"                      . #x1D970)
    ("SIGNWRITING MOVEMENT-FLOORPLANE TRIPLE ALTERNATING MOVEMENT"            . #x1D971)
    ("SIGNWRITING MOVEMENT-FLOORPLANE TRIPLE ALTERNATING WRIST FLEX"          . #x1D972)
    ("SIGNWRITING MOVEMENT-FLOORPLANE BEND"                                   . #x1D973)
    ("SIGNWRITING MOVEMENT-FLOORPLANE CORNER SMALL"                           . #x1D974)
    ("SIGNWRITING MOVEMENT-FLOORPLANE CORNER MEDIUM"                          . #x1D975)
    ("SIGNWRITING MOVEMENT-FLOORPLANE CORNER LARGE"                           . #x1D976)
    ("SIGNWRITING MOVEMENT-FLOORPLANE CHECK"                                  . #x1D977)
    ("SIGNWRITING MOVEMENT-FLOORPLANE BOX SMALL"                              . #x1D978)
    ("SIGNWRITING MOVEMENT-FLOORPLANE BOX MEDIUM"                             . #x1D979)
    ("SIGNWRITING MOVEMENT-FLOORPLANE BOX LARGE"                              . #x1D97A)
    ("SIGNWRITING MOVEMENT-FLOORPLANE ZIGZAG SMALL"                           . #x1D97B)
    ("SIGNWRITING MOVEMENT-FLOORPLANE ZIGZAG MEDIUM"                          . #x1D97C)
    ("SIGNWRITING MOVEMENT-FLOORPLANE ZIGZAG LARGE"                           . #x1D97D)
    ("SIGNWRITING MOVEMENT-FLOORPLANE PEAKS SMALL"                            . #x1D97E)
    ("SIGNWRITING MOVEMENT-FLOORPLANE PEAKS MEDIUM"                           . #x1D97F)
    ("SIGNWRITING MOVEMENT-FLOORPLANE PEAKS LARGE"                            . #x1D980)
    ("SIGNWRITING TRAVEL-FLOORPLANE ROTATION-FLOORPLANE SINGLE"               . #x1D981)
    ("SIGNWRITING TRAVEL-FLOORPLANE ROTATION-FLOORPLANE DOUBLE"               . #x1D982)
    ("SIGNWRITING TRAVEL-FLOORPLANE ROTATION-FLOORPLANE ALTERNATING"          . #x1D983)
    ("SIGNWRITING TRAVEL-FLOORPLANE ROTATION-WALLPLANE SINGLE"                . #x1D984)
    ("SIGNWRITING TRAVEL-FLOORPLANE ROTATION-WALLPLANE DOUBLE"                . #x1D985)
    ("SIGNWRITING TRAVEL-FLOORPLANE ROTATION-WALLPLANE ALTERNATING"           . #x1D986)
    ("SIGNWRITING TRAVEL-FLOORPLANE SHAKING"                                  . #x1D987)
    ("SIGNWRITING MOVEMENT-WALLPLANE CURVE QUARTER SMALL"                     . #x1D988)
    ("SIGNWRITING MOVEMENT-WALLPLANE CURVE QUARTER MEDIUM"                    . #x1D989)
    ("SIGNWRITING MOVEMENT-WALLPLANE CURVE QUARTER LARGE"                     . #x1D98A)
    ("SIGNWRITING MOVEMENT-WALLPLANE CURVE QUARTER LARGEST"                   . #x1D98B)
    ("SIGNWRITING MOVEMENT-WALLPLANE CURVE HALF-CIRCLE SMALL"                 . #x1D98C)
    ("SIGNWRITING MOVEMENT-WALLPLANE CURVE HALF-CIRCLE MEDIUM"                . #x1D98D)
    ("SIGNWRITING MOVEMENT-WALLPLANE CURVE HALF-CIRCLE LARGE"                 . #x1D98E)
    ("SIGNWRITING MOVEMENT-WALLPLANE CURVE HALF-CIRCLE LARGEST"               . #x1D98F)
    ("SIGNWRITING MOVEMENT-WALLPLANE CURVE THREE-QUARTER CIRCLE SMALL"        . #x1D990)
    ("SIGNWRITING MOVEMENT-WALLPLANE CURVE THREE-QUARTER CIRCLE MEDIUM"       . #x1D991)
    ("SIGNWRITING MOVEMENT-WALLPLANE HUMP SMALL"                              . #x1D992)
    ("SIGNWRITING MOVEMENT-WALLPLANE HUMP MEDIUM"                             . #x1D993)
    ("SIGNWRITING MOVEMENT-WALLPLANE HUMP LARGE"                              . #x1D994)
    ("SIGNWRITING MOVEMENT-WALLPLANE LOOP SMALL"                              . #x1D995)
    ("SIGNWRITING MOVEMENT-WALLPLANE LOOP MEDIUM"                             . #x1D996)
    ("SIGNWRITING MOVEMENT-WALLPLANE LOOP LARGE"                              . #x1D997)
    ("SIGNWRITING MOVEMENT-WALLPLANE LOOP SMALL DOUBLE"                       . #x1D998)
    ("SIGNWRITING MOVEMENT-WALLPLANE WAVE CURVE DOUBLE SMALL"                 . #x1D999)
    ("SIGNWRITING MOVEMENT-WALLPLANE WAVE CURVE DOUBLE MEDIUM"                . #x1D99A)
    ("SIGNWRITING MOVEMENT-WALLPLANE WAVE CURVE DOUBLE LARGE"                 . #x1D99B)
    ("SIGNWRITING MOVEMENT-WALLPLANE WAVE CURVE TRIPLE SMALL"                 . #x1D99C)
    ("SIGNWRITING MOVEMENT-WALLPLANE WAVE CURVE TRIPLE MEDIUM"                . #x1D99D)
    ("SIGNWRITING MOVEMENT-WALLPLANE WAVE CURVE TRIPLE LARGE"                 . #x1D99E)
    ("SIGNWRITING MOVEMENT-WALLPLANE CURVE THEN STRAIGHT"                     . #x1D99F)
    ("SIGNWRITING MOVEMENT-WALLPLANE CURVED CROSS SMALL"                      . #x1D9A0)
    ("SIGNWRITING MOVEMENT-WALLPLANE CURVED CROSS MEDIUM"                     . #x1D9A1)
    ("SIGNWRITING ROTATION-WALLPLANE SINGLE"                                  . #x1D9A2)
    ("SIGNWRITING ROTATION-WALLPLANE DOUBLE"                                  . #x1D9A3)
    ("SIGNWRITING ROTATION-WALLPLANE ALTERNATE"                               . #x1D9A4)
    ("SIGNWRITING MOVEMENT-WALLPLANE SHAKING"                                 . #x1D9A5)
    ("SIGNWRITING MOVEMENT-WALLPLANE CURVE HITTING FRONT WALL"                . #x1D9A6)
    ("SIGNWRITING MOVEMENT-WALLPLANE HUMP HITTING FRONT WALL"                 . #x1D9A7)
    ("SIGNWRITING MOVEMENT-WALLPLANE LOOP HITTING FRONT WALL"                 . #x1D9A8)
    ("SIGNWRITING MOVEMENT-WALLPLANE WAVE HITTING FRONT WALL"                 . #x1D9A9)
    ("SIGNWRITING ROTATION-WALLPLANE SINGLE HITTING FRONT WALL"               . #x1D9AA)
    ("SIGNWRITING ROTATION-WALLPLANE DOUBLE HITTING FRONT WALL"               . #x1D9AB)
    ("SIGNWRITING ROTATION-WALLPLANE ALTERNATING HITTING FRONT WALL"          . #x1D9AC)
    ("SIGNWRITING MOVEMENT-WALLPLANE CURVE HITTING CHEST"                     . #x1D9AD)
    ("SIGNWRITING MOVEMENT-WALLPLANE HUMP HITTING CHEST"                      . #x1D9AE)
    ("SIGNWRITING MOVEMENT-WALLPLANE LOOP HITTING CHEST"                      . #x1D9AF)
    ("SIGNWRITING MOVEMENT-WALLPLANE WAVE HITTING CHEST"                      . #x1D9B0)
    ("SIGNWRITING ROTATION-WALLPLANE SINGLE HITTING CHEST"                    . #x1D9B1)
    ("SIGNWRITING ROTATION-WALLPLANE DOUBLE HITTING CHEST"                    . #x1D9B2)
    ("SIGNWRITING ROTATION-WALLPLANE ALTERNATING HITTING CHEST"               . #x1D9B3)
    ("SIGNWRITING MOVEMENT-WALLPLANE WAVE DIAGONAL PATH SMALL"                . #x1D9B4)
    ("SIGNWRITING MOVEMENT-WALLPLANE WAVE DIAGONAL PATH MEDIUM"               . #x1D9B5)
    ("SIGNWRITING MOVEMENT-WALLPLANE WAVE DIAGONAL PATH LARGE"                . #x1D9B6)
    ("SIGNWRITING MOVEMENT-FLOORPLANE CURVE HITTING CEILING SMALL"            . #x1D9B7)
    ("SIGNWRITING MOVEMENT-FLOORPLANE CURVE HITTING CEILING LARGE"            . #x1D9B8)
    ("SIGNWRITING MOVEMENT-FLOORPLANE HUMP HITTING CEILING SMALL DOUBLE"      . #x1D9B9)
    ("SIGNWRITING MOVEMENT-FLOORPLANE HUMP HITTING CEILING LARGE DOUBLE"      . #x1D9BA)
    ("SIGNWRITING MOVEMENT-FLOORPLANE HUMP HITTING CEILING SMALL TRIPLE"      . #x1D9BB)
    ("SIGNWRITING MOVEMENT-FLOORPLANE HUMP HITTING CEILING LARGE TRIPLE"      . #x1D9BC)
    ("SIGNWRITING MOVEMENT-FLOORPLANE LOOP HITTING CEILING SMALL SINGLE"      . #x1D9BD)
    ("SIGNWRITING MOVEMENT-FLOORPLANE LOOP HITTING CEILING LARGE SINGLE"      . #x1D9BE)
    ("SIGNWRITING MOVEMENT-FLOORPLANE LOOP HITTING CEILING SMALL DOUBLE"      . #x1D9BF)
    ("SIGNWRITING MOVEMENT-FLOORPLANE LOOP HITTING CEILING LARGE DOUBLE"      . #x1D9C0)
    ("SIGNWRITING MOVEMENT-FLOORPLANE WAVE HITTING CEILING SMALL"             . #x1D9C1)
    ("SIGNWRITING MOVEMENT-FLOORPLANE WAVE HITTING CEILING LARGE"             . #x1D9C2)
    ("SIGNWRITING ROTATION-FLOORPLANE SINGLE HITTING CEILING"                 . #x1D9C3)
    ("SIGNWRITING ROTATION-FLOORPLANE DOUBLE HITTING CEILING"                 . #x1D9C4)
    ("SIGNWRITING ROTATION-FLOORPLANE ALTERNATING HITTING CEILING"            . #x1D9C5)
    ("SIGNWRITING MOVEMENT-FLOORPLANE CURVE HITTING FLOOR SMALL"              . #x1D9C6)
    ("SIGNWRITING MOVEMENT-FLOORPLANE CURVE HITTING FLOOR LARGE"              . #x1D9C7)
    ("SIGNWRITING MOVEMENT-FLOORPLANE HUMP HITTING FLOOR SMALL DOUBLE"        . #x1D9C8)
    ("SIGNWRITING MOVEMENT-FLOORPLANE HUMP HITTING FLOOR LARGE DOUBLE"        . #x1D9C9)
    ("SIGNWRITING MOVEMENT-FLOORPLANE HUMP HITTING FLOOR TRIPLE SMALL TRIPLE" . #x1D9CA)
    ("SIGNWRITING MOVEMENT-FLOORPLANE HUMP HITTING FLOOR TRIPLE LARGE TRIPLE" . #x1D9CB)
    ("SIGNWRITING MOVEMENT-FLOORPLANE LOOP HITTING FLOOR SMALL SINGLE"        . #x1D9CC)
    ("SIGNWRITING MOVEMENT-FLOORPLANE LOOP HITTING FLOOR LARGE SINGLE"        . #x1D9CD)
    ("SIGNWRITING MOVEMENT-FLOORPLANE LOOP HITTING FLOOR SMALL DOUBLE"        . #x1D9CE)
    ("SIGNWRITING MOVEMENT-FLOORPLANE LOOP HITTING FLOOR LARGE DOUBLE"        . #x1D9CF)
    ("SIGNWRITING MOVEMENT-FLOORPLANE WAVE HITTING FLOOR SMALL"               . #x1D9D0)
    ("SIGNWRITING MOVEMENT-FLOORPLANE WAVE HITTING FLOOR LARGE"               . #x1D9D1)
    ("SIGNWRITING ROTATION-FLOORPLANE SINGLE HITTING FLOOR"                   . #x1D9D2)
    ("SIGNWRITING ROTATION-FLOORPLANE DOUBLE HITTING FLOOR"                   . #x1D9D3)
    ("SIGNWRITING ROTATION-FLOORPLANE ALTERNATING HITTING FLOOR"              . #x1D9D4)
    ("SIGNWRITING MOVEMENT-FLOORPLANE CURVE SMALL"                            . #x1D9D5)
    ("SIGNWRITING MOVEMENT-FLOORPLANE CURVE MEDIUM"                           . #x1D9D6)
    ("SIGNWRITING MOVEMENT-FLOORPLANE CURVE LARGE"                            . #x1D9D7)
    ("SIGNWRITING MOVEMENT-FLOORPLANE CURVE LARGEST"                          . #x1D9D8)
    ("SIGNWRITING MOVEMENT-FLOORPLANE CURVE COMBINED"                         . #x1D9D9)
    ("SIGNWRITING MOVEMENT-FLOORPLANE HUMP SMALL"                             . #x1D9DA)
    ("SIGNWRITING MOVEMENT-FLOORPLANE LOOP SMALL"                             . #x1D9DB)
    ("SIGNWRITING MOVEMENT-FLOORPLANE WAVE SNAKE"                             . #x1D9DC)
    ("SIGNWRITING MOVEMENT-FLOORPLANE WAVE SMALL"                             . #x1D9DD)
    ("SIGNWRITING MOVEMENT-FLOORPLANE WAVE LARGE"                             . #x1D9DE)
    ("SIGNWRITING ROTATION-FLOORPLANE SINGLE"                                 . #x1D9DF)
    ("SIGNWRITING ROTATION-FLOORPLANE DOUBLE"                                 . #x1D9E0)
    ("SIGNWRITING ROTATION-FLOORPLANE ALTERNATING"                            . #x1D9E1)
    ("SIGNWRITING MOVEMENT-FLOORPLANE SHAKING PARALLEL"                       . #x1D9E2)
    ("SIGNWRITING MOVEMENT-WALLPLANE ARM CIRCLE SMALL SINGLE"                 . #x1D9E3)
    ("SIGNWRITING MOVEMENT-WALLPLANE ARM CIRCLE MEDIUM SINGLE"                . #x1D9E4)
    ("SIGNWRITING MOVEMENT-WALLPLANE ARM CIRCLE SMALL DOUBLE"                 . #x1D9E5)
    ("SIGNWRITING MOVEMENT-WALLPLANE ARM CIRCLE MEDIUM DOUBLE"                . #x1D9E6)
    ("SIGNWRITING MOVEMENT-FLOORPLANE ARM CIRCLE HITTING WALL SMALL SINGLE"   . #x1D9E7)
    ("SIGNWRITING MOVEMENT-FLOORPLANE ARM CIRCLE HITTING WALL MEDIUM SINGLE"  . #x1D9E8)
    ("SIGNWRITING MOVEMENT-FLOORPLANE ARM CIRCLE HITTING WALL LARGE SINGLE"   . #x1D9E9)
    ("SIGNWRITING MOVEMENT-FLOORPLANE ARM CIRCLE HITTING WALL SMALL DOUBLE"   . #x1D9EA)
    ("SIGNWRITING MOVEMENT-FLOORPLANE ARM CIRCLE HITTING WALL MEDIUM DOUBLE"  . #x1D9EB)
    ("SIGNWRITING MOVEMENT-FLOORPLANE ARM CIRCLE HITTING WALL LARGE DOUBLE"   . #x1D9EC)
    ("SIGNWRITING MOVEMENT-WALLPLANE WRIST CIRCLE FRONT SINGLE"               . #x1D9ED)
    ("SIGNWRITING MOVEMENT-WALLPLANE WRIST CIRCLE FRONT DOUBLE"               . #x1D9EE)
    ("SIGNWRITING MOVEMENT-FLOORPLANE WRIST CIRCLE HITTING WALL SINGLE"       . #x1D9EF)
    ("SIGNWRITING MOVEMENT-FLOORPLANE WRIST CIRCLE HITTING WALL DOUBLE"       . #x1D9F0)
    ("SIGNWRITING MOVEMENT-WALLPLANE FINGER CIRCLES SINGLE"                   . #x1D9F1)
    ("SIGNWRITING MOVEMENT-WALLPLANE FINGER CIRCLES DOUBLE"                   . #x1D9F2)
    ("SIGNWRITING MOVEMENT-FLOORPLANE FINGER CIRCLES HITTING WALL SINGLE"     . #x1D9F3)
    ("SIGNWRITING MOVEMENT-FLOORPLANE FINGER CIRCLES HITTING WALL DOUBLE"     . #x1D9F4)
    ("SIGNWRITING DYNAMIC ARROWHEAD SMALL"                                    . #x1D9F5)
    ("SIGNWRITING DYNAMIC ARROWHEAD LARGE"                                    . #x1D9F6)
    ("SIGNWRITING DYNAMIC FAST"                                               . #x1D9F7)
    ("SIGNWRITING DYNAMIC SLOW"                                               . #x1D9F8)
    ("SIGNWRITING DYNAMIC TENSE"                                              . #x1D9F9)
    ("SIGNWRITING DYNAMIC RELAXED"                                            . #x1D9FA)
    ("SIGNWRITING DYNAMIC SIMULTANEOUS"                                       . #x1D9FB)
    ("SIGNWRITING DYNAMIC SIMULTANEOUS ALTERNATING"                           . #x1D9FC)
    ("SIGNWRITING DYNAMIC EVERY OTHER TIME"                                   . #x1D9FD)
    ("SIGNWRITING DYNAMIC GRADUAL"                                            . #x1D9FE)
    ("SIGNWRITING HEAD"                                                       . #x1D9FF)
    ("SIGNWRITING HEAD RIM"                                                   . #x1DA00)
    ("SIGNWRITING HEAD MOVEMENT-WALLPLANE STRAIGHT"                           . #x1DA01)
    ("SIGNWRITING HEAD MOVEMENT-WALLPLANE TILT"                               . #x1DA02)
    ("SIGNWRITING HEAD MOVEMENT-FLOORPLANE STRAIGHT"                          . #x1DA03)
    ("SIGNWRITING HEAD MOVEMENT-WALLPLANE CURVE"                              . #x1DA04)
    ("SIGNWRITING HEAD MOVEMENT-FLOORPLANE CURVE"                             . #x1DA05)
    ("SIGNWRITING HEAD MOVEMENT CIRCLE"                                       . #x1DA06)
    ("SIGNWRITING FACE DIRECTION POSITION NOSE FORWARD TILTING"               . #x1DA07)
    ("SIGNWRITING FACE DIRECTION POSITION NOSE UP OR DOWN"                    . #x1DA08)
    ("SIGNWRITING FACE DIRECTION POSITION NOSE UP OR DOWN TILTING"            . #x1DA09)
    ("SIGNWRITING EYEBROWS STRAIGHT UP"                                       . #x1DA0A)
    ("SIGNWRITING EYEBROWS STRAIGHT NEUTRAL"                                  . #x1DA0B)
    ("SIGNWRITING EYEBROWS STRAIGHT DOWN"                                     . #x1DA0C)
    ("SIGNWRITING DREAMY EYEBROWS NEUTRAL DOWN"                               . #x1DA0D)
    ("SIGNWRITING DREAMY EYEBROWS DOWN NEUTRAL"                               . #x1DA0E)
    ("SIGNWRITING DREAMY EYEBROWS UP NEUTRAL"                                 . #x1DA0F)
    ("SIGNWRITING DREAMY EYEBROWS NEUTRAL UP"                                 . #x1DA10)
    ("SIGNWRITING FOREHEAD NEUTRAL"                                           . #x1DA11)
    ("SIGNWRITING FOREHEAD CONTACT"                                           . #x1DA12)
    ("SIGNWRITING FOREHEAD WRINKLED"                                          . #x1DA13)
    ("SIGNWRITING EYES OPEN"                                                  . #x1DA14)
    ("SIGNWRITING EYES SQUEEZED"                                              . #x1DA15)
    ("SIGNWRITING EYES CLOSED"                                                . #x1DA16)
    ("SIGNWRITING EYE BLINK SINGLE"                                           . #x1DA17)
    ("SIGNWRITING EYE BLINK MULTIPLE"                                         . #x1DA18)
    ("SIGNWRITING EYES HALF OPEN"                                             . #x1DA19)
    ("SIGNWRITING EYES WIDE OPEN"                                             . #x1DA1A)
    ("SIGNWRITING EYES HALF CLOSED"                                           . #x1DA1B)
    ("SIGNWRITING EYES WIDENING MOVEMENT"                                     . #x1DA1C)
    ("SIGNWRITING EYE WINK"                                                   . #x1DA1D)
    ("SIGNWRITING EYELASHES UP"                                               . #x1DA1E)
    ("SIGNWRITING EYELASHES DOWN"                                             . #x1DA1F)
    ("SIGNWRITING EYELASHES FLUTTERING"                                       . #x1DA20)
    ("SIGNWRITING EYEGAZE-WALLPLANE STRAIGHT"                                 . #x1DA21)
    ("SIGNWRITING EYEGAZE-WALLPLANE STRAIGHT DOUBLE"                          . #x1DA22)
    ("SIGNWRITING EYEGAZE-WALLPLANE STRAIGHT ALTERNATING"                     . #x1DA23)
    ("SIGNWRITING EYEGAZE-FLOORPLANE STRAIGHT"                                . #x1DA24)
    ("SIGNWRITING EYEGAZE-FLOORPLANE STRAIGHT DOUBLE"                         . #x1DA25)
    ("SIGNWRITING EYEGAZE-FLOORPLANE STRAIGHT ALTERNATING"                    . #x1DA26)
    ("SIGNWRITING EYEGAZE-WALLPLANE CURVED"                                   . #x1DA27)
    ("SIGNWRITING EYEGAZE-FLOORPLANE CURVED"                                  . #x1DA28)
    ("SIGNWRITING EYEGAZE-WALLPLANE CIRCLING"                                 . #x1DA29)
    ("SIGNWRITING CHEEKS PUFFED"                                              . #x1DA2A)
    ("SIGNWRITING CHEEKS NEUTRAL"                                             . #x1DA2B)
    ("SIGNWRITING CHEEKS SUCKED"                                              . #x1DA2C)
    ("SIGNWRITING TENSE CHEEKS HIGH"                                          . #x1DA2D)
    ("SIGNWRITING TENSE CHEEKS MIDDLE"                                        . #x1DA2E)
    ("SIGNWRITING TENSE CHEEKS LOW"                                           . #x1DA2F)
    ("SIGNWRITING EARS"                                                       . #x1DA30)
    ("SIGNWRITING NOSE NEUTRAL"                                               . #x1DA31)
    ("SIGNWRITING NOSE CONTACT"                                               . #x1DA32)
    ("SIGNWRITING NOSE WRINKLES"                                              . #x1DA33)
    ("SIGNWRITING NOSE WIGGLES"                                               . #x1DA34)
    ("SIGNWRITING AIR BLOWING OUT"                                            . #x1DA35)
    ("SIGNWRITING AIR SUCKING IN"                                             . #x1DA36)
    ("SIGNWRITING AIR BLOW SMALL ROTATIONS"                                   . #x1DA37)
    ("SIGNWRITING AIR SUCK SMALL ROTATIONS"                                   . #x1DA38)
    ("SIGNWRITING BREATH INHALE"                                              . #x1DA39)
    ("SIGNWRITING BREATH EXHALE"                                              . #x1DA3A)
    ("SIGNWRITING MOUTH CLOSED NEUTRAL"                                       . #x1DA3B)
    ("SIGNWRITING MOUTH CLOSED FORWARD"                                       . #x1DA3C)
    ("SIGNWRITING MOUTH CLOSED CONTACT"                                       . #x1DA3D)
    ("SIGNWRITING MOUTH SMILE"                                                . #x1DA3E)
    ("SIGNWRITING MOUTH SMILE WRINKLED"                                       . #x1DA3F)
    ("SIGNWRITING MOUTH SMILE OPEN"                                           . #x1DA40)
    ("SIGNWRITING MOUTH FROWN"                                                . #x1DA41)
    ("SIGNWRITING MOUTH FROWN WRINKLED"                                       . #x1DA42)
    ("SIGNWRITING MOUTH FROWN OPEN"                                           . #x1DA43)
    ("SIGNWRITING MOUTH OPEN CIRCLE"                                          . #x1DA44)
    ("SIGNWRITING MOUTH OPEN FORWARD"                                         . #x1DA45)
    ("SIGNWRITING MOUTH OPEN WRINKLED"                                        . #x1DA46)
    ("SIGNWRITING MOUTH OPEN OVAL"                                            . #x1DA47)
    ("SIGNWRITING MOUTH OPEN OVAL WRINKLED"                                   . #x1DA48)
    ("SIGNWRITING MOUTH OPEN OVAL YAWN"                                       . #x1DA49)
    ("SIGNWRITING MOUTH OPEN RECTANGLE"                                       . #x1DA4A)
    ("SIGNWRITING MOUTH OPEN RECTANGLE WRINKLED"                              . #x1DA4B)
    ("SIGNWRITING MOUTH OPEN RECTANGLE YAWN"                                  . #x1DA4C)
    ("SIGNWRITING MOUTH KISS"                                                 . #x1DA4D)
    ("SIGNWRITING MOUTH KISS FORWARD"                                         . #x1DA4E)
    ("SIGNWRITING MOUTH KISS WRINKLED"                                        . #x1DA4F)
    ("SIGNWRITING MOUTH TENSE"                                                . #x1DA50)
    ("SIGNWRITING MOUTH TENSE FORWARD"                                        . #x1DA51)
    ("SIGNWRITING MOUTH TENSE SUCKED"                                         . #x1DA52)
    ("SIGNWRITING LIPS PRESSED TOGETHER"                                      . #x1DA53)
    ("SIGNWRITING LIP LOWER OVER UPPER"                                       . #x1DA54)
    ("SIGNWRITING LIP UPPER OVER LOWER"                                       . #x1DA55)
    ("SIGNWRITING MOUTH CORNERS"                                              . #x1DA56)
    ("SIGNWRITING MOUTH WRINKLES SINGLE"                                      . #x1DA57)
    ("SIGNWRITING MOUTH WRINKLES DOUBLE"                                      . #x1DA58)
    ("SIGNWRITING TONGUE STICKING OUT FAR"                                    . #x1DA59)
    ("SIGNWRITING TONGUE LICKING LIPS"                                        . #x1DA5A)
    ("SIGNWRITING TONGUE TIP BETWEEN LIPS"                                    . #x1DA5B)
    ("SIGNWRITING TONGUE TIP TOUCHING INSIDE MOUTH"                           . #x1DA5C)
    ("SIGNWRITING TONGUE INSIDE MOUTH RELAXED"                                . #x1DA5D)
    ("SIGNWRITING TONGUE MOVES AGAINST CHEEK"                                 . #x1DA5E)
    ("SIGNWRITING TONGUE CENTRE STICKING OUT"                                 . #x1DA5F)
    ("SIGNWRITING TONGUE CENTRE INSIDE MOUTH"                                 . #x1DA60)
    ("SIGNWRITING TEETH"                                                      . #x1DA61)
    ("SIGNWRITING TEETH MOVEMENT"                                             . #x1DA62)
    ("SIGNWRITING TEETH ON TONGUE"                                            . #x1DA63)
    ("SIGNWRITING TEETH ON TONGUE MOVEMENT"                                   . #x1DA64)
    ("SIGNWRITING TEETH ON LIPS"                                              . #x1DA65)
    ("SIGNWRITING TEETH ON LIPS MOVEMENT"                                     . #x1DA66)
    ("SIGNWRITING TEETH BITE LIPS"                                            . #x1DA67)
    ("SIGNWRITING MOVEMENT-WALLPLANE JAW"                                     . #x1DA68)
    ("SIGNWRITING MOVEMENT-FLOORPLANE JAW"                                    . #x1DA69)
    ("SIGNWRITING NECK"                                                       . #x1DA6A)
    ("SIGNWRITING HAIR"                                                       . #x1DA6B)
    ("SIGNWRITING EXCITEMENT"                                                 . #x1DA6C)
    ("SIGNWRITING SHOULDER HIP SPINE"                                         . #x1DA6D)
    ("SIGNWRITING SHOULDER HIP POSITIONS"                                     . #x1DA6E)
    ("SIGNWRITING WALLPLANE SHOULDER HIP MOVE"                                . #x1DA6F)
    ("SIGNWRITING FLOORPLANE SHOULDER HIP MOVE"                               . #x1DA70)
    ("SIGNWRITING SHOULDER TILTING FROM WAIST"                                . #x1DA71)
    ("SIGNWRITING TORSO-WALLPLANE STRAIGHT STRETCH"                           . #x1DA72)
    ("SIGNWRITING TORSO-WALLPLANE CURVED BEND"                                . #x1DA73)
    ("SIGNWRITING TORSO-FLOORPLANE TWISTING"                                  . #x1DA74)
    ("SIGNWRITING UPPER BODY TILTING FROM HIP JOINTS"                         . #x1DA75)
    ("SIGNWRITING LIMB COMBINATION"                                           . #x1DA76)
    ("SIGNWRITING LIMB LENGTH-1"                                              . #x1DA77)
    ("SIGNWRITING LIMB LENGTH-2"                                              . #x1DA78)
    ("SIGNWRITING LIMB LENGTH-3"                                              . #x1DA79)
    ("SIGNWRITING LIMB LENGTH-4"                                              . #x1DA7A)
    ("SIGNWRITING LIMB LENGTH-5"                                              . #x1DA7B)
    ("SIGNWRITING LIMB LENGTH-6"                                              . #x1DA7C)
    ("SIGNWRITING LIMB LENGTH-7"                                              . #x1DA7D)
    ("SIGNWRITING FINGER"                                                     . #x1DA7E)
    ("SIGNWRITING LOCATION-WALLPLANE SPACE"                                   . #x1DA7F)
    ("SIGNWRITING LOCATION-FLOORPLANE SPACE"                                  . #x1DA80)
    ("SIGNWRITING LOCATION HEIGHT"                                            . #x1DA81)
    ("SIGNWRITING LOCATION WIDTH"                                             . #x1DA82)
    ("SIGNWRITING LOCATION DEPTH"                                             . #x1DA83)
    ("SIGNWRITING LOCATION HEAD NECK"                                         . #x1DA84)
    ("SIGNWRITING LOCATION TORSO"                                             . #x1DA85)
    ("SIGNWRITING LOCATION LIMBS DIGITS"                                      . #x1DA86)
    ("SIGNWRITING COMMA"                                                      . #x1DA87)
    ("SIGNWRITING FULL STOP"                                                  . #x1DA88)
    ("SIGNWRITING SEMICOLON"                                                  . #x1DA89)
    ("SIGNWRITING COLON"                                                      . #x1DA8A)
    ("SIGNWRITING PARENTHESIS"                                                . #x1DA8B)
    ("SIGNWRITING FILL MODIFIER-2"                                            . #x1DA9B)
    ("SIGNWRITING FILL MODIFIER-3"                                            . #x1DA9C)
    ("SIGNWRITING FILL MODIFIER-4"                                            . #x1DA9D)
    ("SIGNWRITING FILL MODIFIER-5"                                            . #x1DA9E)
    ("SIGNWRITING FILL MODIFIER-6"                                            . #x1DA9F)
    ("SIGNWRITING ROTATION MODIFIER-2"                                        . #x1DAA1)
    ("SIGNWRITING ROTATION MODIFIER-3"                                        . #x1DAA2)
    ("SIGNWRITING ROTATION MODIFIER-4"                                        . #x1DAA3)
    ("SIGNWRITING ROTATION MODIFIER-5"                                        . #x1DAA4)
    ("SIGNWRITING ROTATION MODIFIER-6"                                        . #x1DAA5)
    ("SIGNWRITING ROTATION MODIFIER-7"                                        . #x1DAA6)
    ("SIGNWRITING ROTATION MODIFIER-8"                                        . #x1DAA7)
    ("SIGNWRITING ROTATION MODIFIER-9"                                        . #x1DAA8)
    ("SIGNWRITING ROTATION MODIFIER-10"                                       . #x1DAA9)
    ("SIGNWRITING ROTATION MODIFIER-11"                                       . #x1DAAA)
    ("SIGNWRITING ROTATION MODIFIER-12"                                       . #x1DAAB)
    ("SIGNWRITING ROTATION MODIFIER-13"                                       . #x1DAAC)
    ("SIGNWRITING ROTATION MODIFIER-14"                                       . #x1DAAD)
    ("SIGNWRITING ROTATION MODIFIER-15"                                       . #x1DAAE)
    ("SIGNWRITING ROTATION MODIFIER-16"                                       . #x1DAAF)
    ("HOT DOG"                                                                . #x1F32D)
    ("TACO"                                                                   . #x1F32E)
    ("BURRITO"                                                                . #x1F32F)
    ("BOTTLE WITH POPPING CORK"                                               . #x1F37E)
    ("POPCORN"                                                                . #x1F37F)
    ("CRICKET BAT AND BALL"                                                   . #x1F3CF)
    ("VOLLEYBALL"                                                             . #x1F3D0)
    ("FIELD HOCKEY STICK AND BALL"                                            . #x1F3D1)
    ("ICE HOCKEY STICK AND PUCK"                                              . #x1F3D2)
    ("TABLE TENNIS PADDLE AND BALL"                                           . #x1F3D3)
    ("BADMINTON RACQUET AND SHUTTLECOCK"                                      . #x1F3F8)
    ("BOW AND ARROW"                                                          . #x1F3F9)
    ("AMPHORA"                                                                . #x1F3FA)
    ("EMOJI MODIFIER FITZPATRICK TYPE-1-2"                                    . #x1F3FB)
    ("EMOJI MODIFIER FITZPATRICK TYPE-3"                                      . #x1F3FC)
    ("EMOJI MODIFIER FITZPATRICK TYPE-4"                                      . #x1F3FD)
    ("EMOJI MODIFIER FITZPATRICK TYPE-5"                                      . #x1F3FE)
    ("EMOJI MODIFIER FITZPATRICK TYPE-6"                                      . #x1F3FF)
    ("PRAYER BEADS"                                                           . #x1F4FF)
    ("KAABA"                                                                  . #x1F54B)
    ("MOSQUE"                                                                 . #x1F54C)
    ("SYNAGOGUE"                                                              . #x1F54D)
    ("MENORAH WITH NINE BRANCHES"                                             . #x1F54E)
    ("BOWL OF HYGIEIA"                                                        . #x1F54F)
    ("UPSIDE-DOWN FACE"                                                       . #x1F643)
    ("FACE WITH ROLLING EYES"                                                 . #x1F644)
    ("PLACE OF WORSHIP"                                                       . #x1F6D0)
    ("ZIPPER-MOUTH FACE"                                                      . #x1F910)
    ("MONEY-MOUTH FACE"                                                       . #x1F911)
    ("FACE WITH THERMOMETER"                                                  . #x1F912)
    ("NERD FACE"                                                              . #x1F913)
    ("THINKING FACE"                                                          . #x1F914)
    ("FACE WITH HEAD-BANDAGE"                                                 . #x1F915)
    ("ROBOT FACE"                                                             . #x1F916)
    ("HUGGING FACE"                                                           . #x1F917)
    ("SIGN OF THE HORNS"                                                      . #x1F918)
    ("CRAB"                                                                   . #x1F980)
    ("LION FACE"                                                              . #x1F981)
    ("SCORPION"                                                               . #x1F982)
    ("TURKEY"                                                                 . #x1F983)
    ("UNICORN FACE"                                                           . #x1F984)
    ("CHEESE WEDGE"                                                           . #x1F9C0))
  "Corrections for ambiguities or omissions in `ucs-names', resolved in favor of Unicode 8.0.")

;; supplement with generated names for CJK Unified Ideographs Extension E
;; (8.0 delta - new block)
(nconc ucs-utils-names-corrections
       (mapcar #'(lambda (x)
                   (cons (format "CJK IDEOGRAPH-%X" x) x))
               (number-sequence #x2B820 #x2CEA1)))

;; attempt to load Unicode 6.0 characters for Emacs 23.x
(when (< emacs-major-version 24)
  (require 'ucs-utils-6.0-delta nil t))

;; note: outside the ucs-utils- namespace
(defvar character-name-history nil "History of character names entered in the minibuffer.")

(defvar ucs-utils-char-mem (make-hash-table :test 'equal)
  "Memoization data for `ucs-utils-char'.")

;;; macros

(defmacro ucs-utils--with-mocked-function (func ret-val &rest body)
  "Execute BODY, mocking FUNC (a symbol) to unconditionally return RET-VAL.

This is portable to versions of Emacs without dynamic `flet`."
  (declare (debug t) (indent 2))
  (let ((o (gensym "--function--")))
    `(let ((,o (ignore-errors (symbol-function ,func))))
       (fset ,func #'(lambda (&rest _ignored) ,ret-val))
       (unwind-protect
           (progn ,@body)
         (when ,o
           (fset ,func ,o))))))

;;; compatibility functions

(unless (fboundp 'string-match-p)
  ;; added in 23.x
  (defun string-match-p (regexp string &optional start)
    "Same as `string-match' except this function does not change the match data."
    (let ((inhibit-changing-match-data t))
      (string-match regexp string start))))

(unless (fboundp 'characterp)
  (defun characterp (char)
    "Return non-nil if CHAR is a character."
    (and (integerp char)
         (> char 0)
         (<= char #x3FFFFF))))

(defun persistent-softest-store (symbol value location &optional expiration)
  "Call `persistent-soft-store' but don't fail when library not present."
  (ignore-errors (persistent-soft-store symbol value location expiration)))
(defun persistent-softest-fetch (symbol location)
  "Call `persistent-soft-fetch' but don't fail when library not present."
  (ignore-errors (persistent-soft-fetch symbol location)))
(defun persistent-softest-exists-p (symbol location)
  "Call `persistent-soft-exists-p' but don't fail when library not present."
  (ignore-errors (persistent-soft-exists-p symbol location)))
(defun persistent-softest-flush (location)
  "Call `persistent-soft-flush' but don't fail when library not present."
  (ignore-errors (persistent-soft-flush location)))
(defun persistent-softest-location-readable (location)
  "Call `persistent-soft-location-readable' but don't fail when library not present."
  (ignore-errors (persistent-soft-location-readable location)))
(defun persistent-softest-location-destroy (location)
  "Call `persistent-soft-location-destroy' but don't fail when library not present."
  (ignore-errors (persistent-soft-location-destroy location)))

;;; utility functions

(defun ucs-utils-names ()
  "Return alist of (CHAR-NAME . CHAR-CODE) pairs cached in `ucs-utils-names'.

Like function `ucs-names' but with more characters."
  (or ucs-utils-names
      (let ((unicode-ranges              ; non-private Unicode 8.0 blocks, merged
             '((#x00000 . #x0085F)
               (#x008A0 . #x01C7F)
               (#x01CC0 . #x02FDF)
               (#x02FF0 . #x0DB7F)
               (#x0DC00 . #x0DFFF)
               (#x0F900 . #x101FF)
               (#x10280 . #x103DF)
               (#x10400 . #x104AF)
               (#x10500 . #x1056F)
               (#x10600 . #x1077F)
               (#x10800 . #x108AF)
               (#x108E0 . #x1093F)
               (#x10980 . #x10A9F)
               (#x10AC0 . #x10BAF)
               (#x10C00 . #x10C4F)
               (#x10C80 . #x10CFF)
               (#x10E60 . #x10E7F)
               (#x11000 . #x1124F)
               (#x11280 . #x1137F)
               (#x11480 . #x114DF)
               (#x11580 . #x1165F)
               (#x11680 . #x116CF)
               (#x11700 . #x1173F)
               (#x118A0 . #x118FF)
               (#x11AC0 . #x11AFF)
               (#x12000 . #x1254F)
               (#x13000 . #x1342F)
               (#x14400 . #x1467F)
               (#x16800 . #x16A6F)
               (#x16AD0 . #x16B8F)
               (#x16F00 . #x16F9F)
               (#x1B000 . #x1B0FF)
               (#x1BC00 . #x1BCAF)
               (#x1D000 . #x1D24F)
               (#x1D300 . #x1D37F)
               (#x1D400 . #x1DAAF)
               (#x1E800 . #x1E8DF)
               (#x1EE00 . #x1EEFF)
               (#x1F000 . #x1F9FF)
               (#x20000 . #x2A6DF)
               (#x2A700 . #x2CEAF)
               (#x2F800 . #x2FA1F)
               (#xE0000 . #xE007F)
               (#xE0100 . #xE01EF)))
            (names (make-tconc))
            (gc-cons-threshold 80000000)
            (gc-cons-percentage .5)
            (name nil))
        (tconc-list names (copy-tree ucs-utils-names-corrections))
        (dolist (range unicode-ranges)
          (when (< (car range) #x010000)
            (tconc-list names (delq nil (mapcar #'(lambda (char)
                                                    (if (setq name (get-char-code-property char 'old-name))
                                                        (cons name char)))
                                                (number-sequence (car range) (cdr range))))))
          (tconc-list names (delq nil (mapcar #'(lambda (char)
                                                  (if (setq name (get-char-code-property char 'name))
                                                      (cons name char)))
                                              (number-sequence (car range) (cdr range))))))
        (setq ucs-utils-names (list-utils-uniq (tconc names)))
        (dolist (cell ucs-utils-names-deletions)
          (setq ucs-utils-names (delete cell ucs-utils-names)))
        ucs-utils-names)))

(defun ucs-utils-names-hash ()
  "All UCS names, cached in hash `ucs-utils-names-hash'.

Content will be taken from persistent storage if available.

Returns nil unless `ucs-utils-trade-memory-for-speed' is set."
  (when ucs-utils-trade-memory-for-speed
    (or
     ucs-utils-names-hash
     (let* ((cache-id (format "e:%s-l:%s"
                              emacs-version
                              (get 'ucs-utils 'custom-version)))
            (store-key (intern (format "names-hash-%s" cache-id)))
            (store-place ucs-utils-use-persistent-storage))
       (unless (and (persistent-softest-exists-p store-key store-place)
                    (hash-table-p (setq ucs-utils-names-hash
                                        (persistent-softest-fetch store-key store-place))))
         (let ((dupes nil)
               (key nil)
               (gc-cons-threshold 80000000)
               (gc-cons-percentage .5))
           (setq ucs-utils-names-hash (make-hash-table :size (length (ucs-utils-names))
                                                       :test 'equal))
           (dolist (cell (ucs-utils-names))
             (setq key (car cell))
             (when (and (gethash key ucs-utils-names-hash)
                        (not (eq (gethash key ucs-utils-names-hash) (cdr cell))))
               (push key dupes))
             (puthash key (cdr cell) ucs-utils-names-hash))
           (delete-dups dupes)
           (dolist (key dupes)
             (remhash key ucs-utils-names-hash))
           (dolist (cell ucs-utils-names-corrections)
             (puthash (car cell) (cdr cell) ucs-utils-names-hash))
           (let ((persistent-soft-inhibit-sanity-checks t))
             (persistent-softest-store store-key
                                       ucs-utils-names-hash
                                       store-place))
           (persistent-softest-flush store-place)))
     ucs-utils-names-hash))))

;; Unfortunately we can't be dash-insensitive b/c UCS names are
;; sensitive to dashes eg TIBETAN LETTER -A vs TIBETAN LETTER A
;; or HANGUL JUNGSEONG O-E vs HANGUL JUNGSEONG OE.
;;
;; Counting deprecated forms, ucs-list is even space-sensitive,
;; though that could be resolved by using only the latest version.
;; Example LATIN SMALL LETTER D Z vs LATIN SMALL LETTER DZ
(defun ucs-utils--lookup (name)
  "Look up the character corresponding to NAME, a UCS name.

NAME is not sensitive to case, allows leading and trailing
double-quotes, and does not distinguish between spaces and
underscores in a UCS name.

Deprecated UCS names are permitted, but conflicts are resolved
in favor of later versions of the Unicode spec.  \"BELL\" is
a famous example of a conflict.

Returns nil if NAME does not exist."
  (save-match-data
    (callf upcase name)
    (setq name (replace-regexp-in-string "\\<\\(BRAILLE DOTS\\|SELECTOR\\) \\([0-9]+\\)\\'" "\\1-\\2"
                 (replace-regexp-in-string "\\<\\(IDEOGRAPH\\) \\([0-9A-F]+\\)\\'" "\\1-\\2"
                   (replace-regexp-in-string "\\`[ \"]+" ""
                     (replace-regexp-in-string "[ \"]+\\'" ""
                       (replace-regexp-in-string " \\{2,\\}" " "
                         (replace-regexp-in-string "[_\t]" " "  name)))))))
    (cond
      ((hash-table-p (ucs-utils-names-hash))
       (gethash name (ucs-utils-names-hash)))
      (t
       (cdr (assoc-string name (ucs-utils-names) t))))))

(defun ucs-utils-vector-flatten (vec)
  "Flatten vector or string VEC which may contain other vectors or strings."
  (cond
    ((stringp vec)
     (string-to-vector vec))
    ((null vec)
     (vector nil))
    ((and (vectorp vec)
          (= 0 (length vec)))
     nil)
    ((vectorp vec)
     (vconcat (ucs-utils-vector-flatten (aref vec 0))
              (ucs-utils-vector-flatten (vconcat (cdr (append vec nil))))))
    (t
     (vector vec))))

(defun ucs-utils-prettify-ucs-string (name)
  "Prettify NAME, a string holding the UCS name for a character."
  (callf capitalize name)
  (save-match-data
    (when (string-match "([^()]+?)\\'" name)
      (setq name (replace-match (upcase (match-string-no-properties 0 name)) 'fixed-case 'literal name)))
    (let ((case-fold-search nil))
      (while (string-match " \\(?:And\\|Or\\|Of\\|On\\|The\\|For\\|In\\|With\\|Over\\|Under\\|At\\) " name)
        (setq name (replace-match (downcase (match-string-no-properties 0 name)) 'fixed-case 'literal name)))
      (while (and (string-match "\\`Cuneiform Sign " name)
                  (string-match " \\(?:Times\\|Plus\\) " name))
        (setq name (replace-match (downcase (match-string-no-properties 0 name)) 'fixed-case 'literal name)))
      (while (string-match "\\<Logical [Aa]nd\\>" name)
        (setq name (replace-match "Logical AND" 'fixed-case 'literal name)))
      (while (string-match "\\<Logical [Oo]r\\>" name)
        (setq name (replace-match "Logical OR" 'fixed-case 'literal name)))
      (while (string-match "\\<N-Ary\\>" name)
        (setq name (replace-match "N-ary" 'fixed-case 'literal name)))
      (while (string-match "\\<Per-Em\\>" name)
        (setq name (replace-match "per-Em" 'fixed-case 'literal name)))
      (while (string-match "\\<Jack-O-Lantern\\>" name)
        (setq name (replace-match "Jack-o-Lantern" 'fixed-case 'literal name)))
      (while (string-match "\\<Fleur-De-Lis\\>" name)
        (setq name (replace-match "Fleur-de-Lis" 'fixed-case 'literal name)))
      (while (string-match "\\<Left-To-Right\\>" name)
        (setq name (replace-match "Left-to-Right" 'fixed-case 'literal name)))
      (while (string-match "\\<Right-To-Left\\>" name)
        (setq name (replace-match "Right-to-Left" 'fixed-case 'literal name)))
      (while (string-match "\\<Minus-Or-Plus\\>" name)
        (setq name (replace-match "Minus-or-Plus" 'fixed-case 'literal name)))
      (while (string-match "\\<Plus-Or-Minus\\>" name)
        (setq name (replace-match "Plus-or-Minus" 'fixed-case 'literal name)))
      (while (string-match "\\<\\(?:Cjk\\|Apl\\|Ocr\\)\\>" name)
        (setq name (replace-match (upcase (match-string-no-properties 0 name)) 'fixed-case 'literal name)))
      (when (string-match "\\<\\(Ideograph\\)-\\([0-9A-Fa-f]+\\)\\'" name)
        (setq name (replace-match (concat (match-string-no-properties 1 name)
                                          "-"
                                          (upcase (match-string-no-properties 2 name))) 'fixed-case 'literal name)))
      (when (string-match "\\`Nko\\>" name)
        (setq name (replace-match "NKo" 'fixed-case 'literal name))))
    name))

;;;###autoload
(defun ucs-utils-pretty-name (char &optional no-hex)
  "Return a prettified UCS name for CHAR.

Based on `get-char-code-property'.  The result has been
title-cased for readability, and will not match into the
`ucs-utils-names' alist until it has been upcased.
`ucs-utils-char' can be used on the title-cased name.

Returns a hexified string if no name is found.  If NO-HEX is
non-nil, then a nil value will be returned when no name is
found."
  (when (characterp char)
    (let ((name (or (car (rassoc char ucs-utils-names-corrections))
                    (get-char-code-property char 'name))))
      (when (equal "<control>" name)
        (setq name (get-char-code-property char 'old-name)))
      (when (eq char ?\s)
        (callf or name "Space"))
      (when (rassoc char ucs-utils-names-deletions)
        (setq name nil))
      (cond
        ((and no-hex
              (= (length name) 0))
         (setq name nil))
        ((= (length name) 0)
         (setq name (concat "#x" (upcase (format "%02x" char)))))
        (t
         (ucs-utils-prettify-ucs-string name))))))

;;;###autoload
(defun ucs-utils-all-prettified-names (&optional progress regenerate)
  "All prettified UCS names, cached in list `ucs-utils-all-prettified-names'.

When optional PROGRESS is given, show progress when generating
cache.

When optional REGENERATE is given, re-generate cache."
  (let* ((cache-id (format "e:%s-l:%s"
                           emacs-version
                           (get 'ucs-utils 'custom-version)))
         (store-key (intern (format "all-prettified-names-%s" cache-id)))
         (store-place ucs-utils-use-persistent-storage))
    (when regenerate
      (setq ucs-utils-all-prettified-names nil)
      (persistent-softest-store store-key nil store-place))
    (cond
      (ucs-utils-all-prettified-names
       t)
      ((and (not regenerate)
            (persistent-softest-exists-p store-key store-place)
            (consp (setq ucs-utils-all-prettified-names
                         (persistent-softest-fetch store-key store-place))))
       t)
      (t
       (let ((reporter (make-progress-reporter "Caching formatted UCS names... " 0 (length (ucs-utils-names))))
             (counter 0)
             (draft-list nil)
             (prev-name nil)
             (gc-cons-threshold 80000000)
             (gc-cons-percentage .5))
         (dolist (cell (ucs-utils-names))
           (when progress
             (progress-reporter-update reporter (incf counter)))
           (push (replace-regexp-in-string " " "_" (or (ucs-utils-pretty-name (cdr cell) 'no-hex) "")) draft-list))
         (dolist (name (delete "" (sort draft-list 'string<)))
           (unless (equal name prev-name)
             (push name ucs-utils-all-prettified-names))
           (setq prev-name name))
         (callf nreverse ucs-utils-all-prettified-names)
         (let ((persistent-soft-inhibit-sanity-checks t))
           (persistent-softest-store store-key
                                     ucs-utils-all-prettified-names
                                     store-place))
         (persistent-softest-flush store-place)
         (progress-reporter-done reporter)))))
  ucs-utils-all-prettified-names)

(defun ucs-utils--subst-char-in-region-1 (start end from-char to-char)
  "Internal driver for `usr-utils-subst-char-in-region'.

Arguments START, END, FROM-CHAR, and TO-CHAR are as documented at
`ucs-utils-subst-char-in-region'."
  ;; done in a specific fashion to maintain markers
  (loop for i from start to (1- end)
        if (eq (char-after i) from-char)
        do (save-excursion
             (goto-char i)
             (insert to-char)
             (delete-char 1))))

;;; external interface (plus `ucs-utils-pretty-name' above)

;;;###autoload
(defun ucs-utils-char (name &optional fallback test)
  "Return the character corresponding to NAME, a UCS name.

NAME is matched leniently by `ucs-utils--lookup'.

Returns FALLBACK if NAME does not exist or is not displayable
according to TEST.  FALLBACK may be either a UCS name or
character, or one of the special symbols described in the next
paragraph.

If FALLBACK is nil or 'drop, returns nil on failure.  If FALLBACK
is 'error, throws an error on failure.

TEST is an optional predicate which characters must pass.  A
useful value is 'char-displayable-p, which is available as
the abbreviation 'cdp, unless you have otherwise defined that
symbol.

When NAME is a character, it passes through unchanged, unless
TEST is set, in which case it must pass TEST."
  (let ((char name)
        (orig-fallback fallback)
        (args (list name fallback test))
        (retval nil))
    (if (and ucs-utils-trade-memory-for-speed
             (gethash args ucs-utils-char-mem))
        (gethash args ucs-utils-char-mem)
      ;; else
      (when (and (eq test 'cdp)
                 (not (fboundp 'cdp)))
        (setq test 'char-displayable-p))
      (when (stringp char)
        (setq char (ucs-utils--lookup char)))
      (when (stringp fallback)
        (setq fallback (ucs-utils--lookup fallback))
        (assert (characterp fallback) nil "Invalid fallback: %s" orig-fallback))
      (setq retval (cond
                     ((and (characterp char)
                           (or (not test) (funcall test char)))
                      char)
                     ((eq fallback 'error)
                      (error "Character invalid or undisplayable: %s" name))
                     ((or (eq fallback 'drop)
                          (null fallback))
                      nil)
                     ((vectorp fallback)
                      fallback)
                     (t
                      (assert (characterp fallback) nil "Invalid fallback: %s" orig-fallback)
                      fallback)))
      (when ucs-utils-trade-memory-for-speed
        (puthash args retval ucs-utils-char-mem))
      retval)))

;;;###autoload
(defun ucs-utils-first-existing-char (sequence &optional test)
  "Return the first existing character from SEQUENCE of character names.

TEST is an optional predicate which characters must pass.  A
useful value is 'char-displayable-p, which is available as
the abbreviation 'cdp, unless you have otherwise defined that
symbol."
 (catch 'char
   (dolist (name sequence)
     (let ((char (ucs-utils-char name nil test)))
     (when char
       (throw 'char char))))))

;;;###autoload
(defun ucs-utils-vector (sequence &optional fallback test no-flatten)
  "Return a vector corresponding to SEQUENCE of UCS names or characters.

If SEQUENCE is a single string or character, it will be coerced
to a list of length 1.  Each name in SEQUENCE is matched
leniently by `ucs-utils--lookup'.

FALLBACK should be a sequence of equal length to SEQUENCE, (or
one of the special symbols described in the next paragraph).  For
any element of SEQUENCE which does not exist or is not
displayable according to TEST, that element degrades to the
corresponding element of FALLBACK.

When FALLBACK is nil, characters which do not exist or are
undisplayable will be given as nils in the return value.  When
FALLBACK is 'drop, such characters will be silently dropped from
the return value.  When FALLBACK is 'error, such characters cause
an error to be thrown.

To allow fallbacks of arbitrary length, give FALLBACK as a vector-
of-vectors, with outer length equal to the length of sequence.  The
inner vectors may contain a sequence of characters, a literal
string, or nil.  Eg

   (ucs-utils-vector '(\"Middle Dot\" \"Ampersand\" \"Horizontal Ellipsis\")
                     '[?.           [?a ?n ?d]  [\"...\"]              ])

or

   (ucs-utils-vector \"Horizontal Ellipsis\" '[[\"...\"]])

TEST is an optional predicate which characters must pass.  A
useful value is 'char-displayable-p, which is available as
the abbreviation 'cdp, unless you have otherwise defined that
symbol.

If NO-FLATTEN is non-nil, then a vector-of-vectors may be returned
if multi-character fallbacks were used as in the example above."
  (let ((strip-char :nonexistent)
        (flattener (if no-flatten 'identity 'ucs-utils-vector-flatten)))
    (cond
      ((vectorp sequence)
       (callf append sequence nil))
      ((or (stringp sequence)
           (characterp sequence))
       (callf list sequence)))
    (cond
      ((eq fallback 'drop)
       (setq strip-char nil)
       (setq fallback (make-list (length sequence) nil)))
      ((null fallback)
       (setq fallback (make-list (length sequence) nil)))
      ((eq fallback 'error)
       (setq fallback (make-list (length sequence) 'error)))
      ((vectorp fallback)
       (callf append fallback nil))
      ((or (stringp fallback)
           (characterp fallback))
       (callf list fallback)))
    (assert (and (listp sequence) (listp fallback)) nil "SEQUENCE and FALLBACK should be lists or vectors.")
    (assert (= (length sequence) (length fallback)) nil "SEQUENCE and FALLBACK should be the same length.")
    (funcall flattener (apply 'vector (delq strip-char (loop for elt in sequence
                                                             for back-elt in fallback
                                                             collect (ucs-utils-char elt back-elt test)))))))

;;;###autoload
(defun ucs-utils-string (sequence &optional fallback test)
  "Return a string corresponding to SEQUENCE of UCS names or characters.

If SEQUENCE is a single string, it will be coerced to a list of
length 1.  Each name in SEQUENCE is matched leniently by
`ucs-utils--lookup'.

FALLBACK should be a sequence of equal length to SEQUENCE, (or
one of the special symbols described in the next paragraph).  For
any element of SEQUENCE which does not exist or is not
displayable according to TEST, that element degrades to the
corresponding element of FALLBACK.

When FALLBACK is nil or 'drop, characters which do not exist or
are undisplayable will be silently dropped from the return value.
When FALLBACK is 'error, such characters cause an error to be
thrown.

TEST is an optional predicate which characters must pass.  A
useful value is 'char-displayable-p, which is available as
the abbreviation 'cdp, unless you have otherwise defined that
symbol."
  (callf or fallback 'drop)
  (concat (delete nil (ucs-utils-vector sequence fallback test))))

;;;###autoload
(defun ucs-utils-intact-string (sequence fallback &optional test)
  "Return a string corresponding to SEQUENCE of UCS names or characters.

This function differs from `ucs-utils-string' in that FALLBACK is
a non-optional single string, to be used unless every member of
SEQUENCE exists and passes TEST.  FALLBACK may not be nil, 'error,
or 'drop as in `ucs-utils-string'.

If SEQUENCE is a single string, it will be coerced to a list of
length 1.  Each name in SEQUENCE is matched leniently by
`ucs-utils--lookup'.

TEST is an optional predicate which characters must pass.  A
useful value is 'char-displayable-p, which is available as
the abbreviation 'cdp, unless you have otherwise defined that
symbol."
  (assert (stringp fallback) nil "FALLBACK must be a string")
  (condition-case nil
      (concat (ucs-utils-vector sequence 'error test))
    (error nil
           fallback)))

;;;###autoload
(defun ucs-utils-subst-char-in-region (start end from-char to-char &optional no-undo)
  "From START to END, replace FROM-CHAR with TO-CHAR each time it occurs.

If optional arg NO-UNDO is non-nil, don't record this change for
undo and don't mark the buffer as really changed.

Characters may be of differing byte-lengths.

The character at the position END is not included, matching the
behavior of `subst-char-in-region'.

This function is slower than `subst-char-in-region'."
  (setq from-char (ucs-utils-char from-char 'error))
  (setq to-char   (ucs-utils-char to-char 'error))
  (if no-undo
      (let ((buffer-undo-list nil)
            (modified (buffer-modified-p)))
        (ucs-utils--subst-char-in-region-1 start end from-char to-char)
        (set-buffer-modified-p modified))
    ;; else
    (ucs-utils--subst-char-in-region-1 start end from-char to-char)))

(fset 'ucs-utils-orig-read-char-by-name (symbol-function 'read-char-by-name))

;;;###autoload
(defun ucs-utils-read-char-by-name (prompt &optional ido)
  "Read a character by its Unicode name or hex number string.

A wrapper for `read-char-by-name', with the option to use
`ido-completing-read'.

PROMPT is displayed, and a string that represents a character by
its name is read.

When IDO is set, several seconds are required on the first
run as all completion candidates are pre-generated."
  (if (not ido)
      (ucs-utils-orig-read-char-by-name prompt)
    ;; else
    (let* ((gc-cons-threshold 80000000)
           (gc-cons-percentage .5)
           (input (ido-completing-read
                  prompt
                  (remove-if #'(lambda (x)
                                 (and ucs-utils-hide-numbered-cjk-ideographs (string-match-p "_Ideograph[_-][0-9a-fA-F]+\\'" x)))
                             (ucs-utils-all-prettified-names 'progress)) nil nil nil 'character-name-history)))
      (when (string-match-p "\\`[0-9a-fA-F]+\\'" input)
        (callf2 concat "#x" input))
      (if (string-match-p "\\`#" input)
          (read input)
        ;; else
        (ucs-utils--lookup input)))))

;;; interactive commands

;;;###autoload
(defun ucs-utils-eval (&optional pos arg)
  "Display a string UCS name for the character at POS.

POS defaults to the current point.

If `transient-mark-mode' is enabled and there is an active
region, return a list of strings UCS names, one for each
character in the region.  If called from Lisp with an
explicit POS, ignores the region.

If called with universal prefix ARG, display the result
in a separate buffer.  If called with two universal
prefix ARGs, replace the current character or region with
its UCS name translation."
  (interactive)
  (let ((result nil)
        (print-level nil)
        (print-length nil))
    (callf or arg current-prefix-arg)
    (cond
      ((and (not pos)
            (use-region-p))
       (dolist (char (string-to-list (buffer-substring-no-properties (region-beginning) (region-end))))
         (push (ucs-utils-pretty-name char) result))
       (setq result (nreverse result)))
      (t
       (save-excursion
         (goto-char (or pos (point)))
         (setq result (ucs-utils-pretty-name (char-after))))))
    (assert result nil "Failed to find name for character at: %s" pos)
    (cond
      ((equal arg '(4))
       (ucs-utils--with-mocked-function 'frame-width 0
         (pp-display-expression result "*Pp Eval Output*")))
      ((consp arg)
       (if (and (not pos)
                (use-region-p))
           (progn
             (let ((begin (region-beginning)))
               (delete-region (region-beginning) (region-end))
               (goto-char begin)
               (pp result (current-buffer))))
         ;; else
         (goto-char (or pos (point)))
         (insert (concat "\"" (prin1 result) "\""))
         (delete-char 1)))
      (t
       (pp-display-expression result "*Pp Eval Output*")))
    result))

;;;###autoload
(defun ucs-utils-ucs-insert (character &optional count inherit)
  "Insert CHARACTER in COUNT copies, where CHARACTER is a Unicode code point.

Works like `ucs-insert', with the following differences

    * Uses `ido-completing-read' at the interactive prompt

    * If `transient-mark-mode' is enabled, and the region contains
      a valid UCS character name, that value is used as the
      character name and the region is replaced.

    * A UCS character name string may be passed for CHARACTER.

INHERIT is as documented at `ucs-insert'."
  (interactive
   (list
    (if (and (use-region-p)
             (ucs-utils-char (buffer-substring-no-properties (region-beginning) (region-end)) nil))
        (prog1
          (ucs-utils-char (buffer-substring-no-properties (region-beginning) (region-end)) nil)
          (delete-region (region-beginning) (region-end)))
      ;; else
      (ucs-utils-read-char-by-name "Unicode (name or hex): " 'ido))
    (prefix-numeric-value current-prefix-arg)
    t))
  (let ((insert-func 'ucs-insert))
    (when (get insert-func 'byte-obsolete-info)
      (setq insert-func 'insert-char))
    (funcall insert-func (ucs-utils-char character 'error) count inherit)))

;;;###autoload
(defun ucs-utils-install-aliases ()
  "Install aliases outside the \"ucs-utils-\" namespace.

The following aliases will be installed:

    `ucs-char'                  for   `ucs-utils-char'
    `ucs-first-existing-char'   for   `ucs-utils-first-existing-char'
    `ucs-string'                for   `ucs-utils-string'
    `ucs-intact-string'         for   `ucs-utils-intact-string'
    `ucs-vector'                for   `ucs-utils-vector'
    `ucs-pretty-name'           for   `ucs-utils-pretty-name'
    `ucs-eval'                  for   `ucs-utils-eval'"
  (interactive)
  (defalias 'ucs-char                 'ucs-utils-char)
  (defalias 'ucs-first-existing-char  'ucs-utils-first-existing-char)
  (defalias 'ucs-string               'ucs-utils-string)
  (defalias 'ucs-intact-string        'ucs-utils-intact-string)
  (defalias 'ucs-vector               'ucs-utils-vector)
  (defalias 'ucs-pretty-name          'ucs-utils-pretty-name)
  (defalias 'ucs-eval                 'ucs-utils-eval))

(provide 'ucs-utils)

;;
;; Emacs
;;
;; Local Variables:
;; indent-tabs-mode: nil
;; mangle-whitespace: t
;; require-final-newline: t
;; coding: utf-8
;; byte-compile-warnings: (not cl-functions redefine)
;; End:
;;
;; LocalWords:  UcsUtils utils eval callf flet YOGH alist ZHAR PHAR
;; LocalWords:  KHAR GHAN JHAN KIYEOK PIEUP CIEUC HALFWIDTH Fleur
;; LocalWords:  JUNGSEONG
;;

;;; ucs-utils.el ends here
