;;; soothe-tva.el --- Autothemer TVA engine for soothe-theme -*- lexical-binding: t; -*-
;;
;; Copyright (C) 2012-2022 Jason Milkins
;;
;; Author: Jason Milkins <jasonm23@gmail.com>
;; Maintainer: Jason Milkins <jasonm23@gmail.com>
;; URL: https://github.com/emacsfodder/emacs-soothe-theme
;; Package-Requires: ((emacs "24.3") (autothemer "0.2"))
;;
;; This file is not part of GNU Emacs.
;;
;;; License:
;;
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 3, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 51 Franklin Street, Fifth
;; Floor, Boston, MA 02110-1301, USA.

;;; Commentary:
;;
;;  TVA/autothemer engine for soothe-theme
;;
;;; Code:

(require 'autothemer)

(unless (>= emacs-major-version 24)
  (error "Requires Emacs 24 or later"))
(defmacro soothe-tva-deftheme (name description palette &rest body)
  "Create a soothe-theme: NAME, DESCRIPTION, PALETTE, BODY."
  `(autothemer-deftheme ,name ,description ,palette
    ;;; Face specs
    ((default                                   (:foreground soothe-foreground                :background soothe-background))
     (ac-candidate-face                         (:foreground soothe-background                :background soothe-foreground-2))
     (ac-candidate-mouse-face                   (:foreground soothe-foreground-1              :background soothe-turquoise-1))
     (ac-completion-face                        (:foreground soothe-foreground-0              :background soothe-purple-3bg))
     (ac-gtags-candidate-face                   (:foreground soothe-background                :background soothe-purple-3))
     (ac-gtags-selection-face                   (:foreground soothe-foreground-2              :background soothe-background-2))
     (ac-selection-face                         (:foreground soothe-foreground-2              :background soothe-background-2))
     (ac-yasnippet-candidate-face               (:foreground soothe-background                :background soothe-green-2))
     (ac-yasnippet-selection-face               (:foreground soothe-foreground-1              :background soothe-background-2))
     (anzu-mode-line                            (:foreground soothe-foreground-1))
     (anzu-mode-line-no-match                   (:foreground soothe-foreground-1))
     (anzu-replace-highlight                    (:foreground soothe-prime-purple               :background soothe-bg-purple))
     (anzu-replace-to                           (:foreground soothe-orange-2                   :background soothe-turquoise-3))
     (anzu-match-1                              (:foreground soothe-prime-turquoise            :background soothe-dark-purple))
     (anzu-match-2                              (:foreground soothe-prime-purple               :background soothe-dark-purple))
     (anzu-match-3                              (:inherit 'diredfl-flag-mark))
     (company-echo                              (:inherit 'company-echo-common))
     (company-echo-common                       (:foreground soothe-prime-blue                 :background soothe-turquoise-1bg))
     (company-preview                           (:inherit 'company-preview-common))
     (company-preview-common                    (:inherit 'font-lock-builtin-face              :underline soothe-foreground-1))
     (company-preview-search                    (:inherit 'company-preview-common))
     (company-scrollbar-bg                      (:background soothe-turquoise-2bg))
     (company-scrollbar-fg                      (:background soothe-turquoise-2))
     (company-template-field                    (:foreground soothe-prime-blue                 :background nil :underline soothe-bg-blue))
     (company-tooltip                           (:foreground soothe-foreground-0               :background soothe-turquoise-1bg))
     (company-tooltip-annotation                (:foreground soothe-foreground-0               :background soothe-turquoise-2bg))
     (company-tooltip-common                    (:foreground soothe-prime-turquoise            :background soothe-turquoise-2bg))
     (company-tooltip-common-selection          (:foreground soothe-prime-orange               :background soothe-turquoise-2bg))
     (company-tooltip-mouse                     (:foreground soothe-prime-purple               :background soothe-turquoise-2bg))
     (company-tooltip-selection                 (:inherit 'font-lock-constant-face             :background soothe-background-dark-0))
     (cua-global-mark                           (:foreground soothe-foreground-1               :background soothe-turquoise-1))
     (cua-rectangle                             (:foreground soothe-foreground-1               :background soothe-purple-4))
     (cua-rectangle-noselect                    (:foreground soothe-foreground-1               :background soothe-prime-orange))
     (cursor                                    (:background soothe-prime-orange))
     (diff-added                                (:foreground soothe-overexposed-green          :background soothe-dark-green))
     (diff-changed                              (:foreground soothe-overexposed-purple         :background soothe-dark-purple))
     (diff-changed-unspcified                   (:foreground soothe-overexposed-purple         :background soothe-dark-purple))
     (diff-context                              (:foreground soothe-foreground-1))
     (diff-file-header                          (:foreground soothe-prime-orange               :background soothe-orange-1bg))
     (diff-function                             (:foreground soothe-foreground-1               :background soothe-dark-orange))
     (diff-header                               (:foreground soothe-prime-turquoise            :background soothe-dark-orange))
     (diff-hl-change                            (:foreground soothe-prime-purple               :background soothe-purple-1bg))
     (diff-hl-delete                            (:foreground soothe-prime-red                  :background soothe-red-1bg))
     (diff-hl-insert                            (:foreground soothe-prime-green                :background soothe-green-2bg))
     (diff-hunk-header                          (:foreground soothe-prime-purple               :background soothe-purple-1bg))
     (diff-refine-added                         (:inherit 'diff-added                          :background soothe-low-green))
     (diff-refine-changed                       (:inherit 'diff-changed                        :background soothe-low-purple))
     (diff-refine-removed                       (:inherit 'diff-removed                        :background soothe-low-red))
     (diff-removed                              (:foreground soothe-overexposed-red            :background soothe-dark-red))
     (dired-directory                           (:foreground soothe-prime-turquoise            :background soothe-bg-turquoise))
     (dired-flagged                             (:foreground soothe-prime-red                  :background soothe-orange-1bg))
     (dired-header                              (:inherit 'header-line))
     (dired-ignored                             (:foreground soothe-turquoise-1                :background soothe-background))
     (dired-mark                                (:foreground soothe-orange-2                   :background soothe-background))
     (dired-marked                              (:foreground soothe-prime-orange               :background soothe-purple-2bg))
     (dired-perm-write                          (:foreground soothe-prime-turquoise            :background soothe-turquoise-1bg))
     (dired-symlink                             (:foreground soothe-prime-blue                 :background soothe-blue-4bg))
     (dired-warning                             (:foreground soothe-prime-red                  :background soothe-red-2bg))
     (diredfl-autofile-name                     (:foreground soothe-prime-blue                 :background soothe-dark-purple))
     (diredfl-compressed-file-name              (:foreground soothe-prime-turquoise))
     (diredfl-compressed-file-suffix            (:foreground soothe-turquoise-0))
     (diredfl-date-time                         (:foreground soothe-prime-orange               :background soothe-dark-purple))
     (diredfl-deletion                          (:foreground soothe-prime-orange               :background soothe-red-3))
     (diredfl-deletion-file-name                (:foreground soothe-red-3                      :background-color soothe-red-2bg))
     (diredfl-dir-heading                       (:inherit 'font-lock-builtin-face))
     (diredfl-dir-name                          (:inherit 'dired-directory))
     (diredfl-dir-priv                          (:foreground soothe-prime-orange               :background soothe-bg-orange))
     (diredfl-exec-priv                         (:foreground soothe-prime-red                  :background soothe-bg-red))
     (diredfl-executable-tag                    (:foreground soothe-red-3))
     (diredfl-file-name                         (:foreground soothe-prime-purple               :background soothe-dark-purple))
     (diredfl-file-suffix                       (:foreground soothe-faded-purple               :background soothe-dark-purple))
     (diredfl-flag-mark                         (:foreground soothe-prime-orange               :background soothe-bg-orange))
     (diredfl-flag-mark-line                    (:background soothe-low-red))
     (diredfl-flagged                           (:inherit 'dired-flagged))
     (diredfl-header                            (:inherit 'dired-header))
     (diredfl-ignored                           (:inherit 'dired-ignored))
     (diredfl-ignored-file-name                 (:foreground soothe-prime-orange))
     (diredfl-link-priv                         (:foreground soothe-prime-turquoise            :background soothe-bg-turquoise))
     (diredfl-mark                              (:inherit 'dired-mark))
     (diredfl-marked                            (:inherit 'dired-marked))
     (diredfl-no-priv                           (:foreground soothe-faded-purple               :background soothe-dark-purple))
     (diredfl-number                            (:foreground soothe-prime-orange               :background soothe-orange-1bg))
     (diredfl-other-priv                        (:background soothe-blue-5bg))
     (diredfl-rare-priv                         (:foreground soothe-prime-green                :background soothe-prime-red))
     (diredfl-read-priv                         (:foreground soothe-prime-purple               :background soothe-bg-purple))
     (diredfl-symlink                           (:inherit 'dired-symlink))
     (diredfl-tagged-autofile-name              (:background soothe-purple-3bg))
     (diredfl-warning                           (:inherit 'dired-warning))
     (diredfl-write-priv                        (:inherit 'dired-perm-write))
     (diredp-autofile-name                      (:inherit 'diredfl-autofile-name))
     (diredp-compressed-file-name               (:inherit 'diredfl-compressed-file-name))
     (diredp-compressed-file-suffix             (:inherit 'diredfl-compressed-file-suffix))
     (diredp-date-time                          (:inherit 'diredfl-date-time))
     (diredp-deletion                           (:inherit 'diredfl-deletion))
     (diredp-deletion-file-name                 (:inherit 'diredfl-deletion-file-name))
     (diredp-dir-heading                        (:inherit 'diredfl-dir-heading))
     (diredp-dir-name                           (:inherit 'diredfl-dir-name))
     (diredp-dir-priv                           (:inherit 'diredfl-dir-priv))
     (diredp-exec-priv                          (:inherit 'diredfl-exec-priv))
     (diredp-executable-tag                     (:inherit 'diredfl-executable-tag))
     (diredp-file-name                          (:foreground soothe-prime-purple               :background soothe-dark-purple))
     (diredp-file-suffix                        (:inherit 'diredfl-file-suffix))
     (diredp-flag-mark                          (:inherit 'diredfl-flag-mark))
     (diredp-flag-mark-line                     (:inherit 'diredfl-flag-mark-line))
     (diredp-ignored-file-name                  (:inherit 'diredfl-ignored-file-name))
     (diredp-link-priv                          (:inherit 'diredfl-link-priv))
     (diredp-no-priv                            (:inherit 'diredfl-no-priv))
     (diredp-number                             (:inherit 'diredfl-number))
     (diredp-other-priv                         (:inherit 'diredfl-other-priv))
     (diredp-rare-priv                          (:inherit 'diredfl-rare-priv))
     (diredp-read-priv                          (:inherit 'diredfl-read-priv))
     (diredp-symlink                            (:inherit 'diredfl-symlink))
     (diredp-tagged-autofile-name               (:inherit 'diredfl-tagged-autofile-name))
     (diredp-write-priv                         (:inherit 'diredfl-write-priv))
     (dropdown-list-face                        (:foreground soothe-background                 :background soothe-foreground-1))
     (dropdown-list-selection-face              (:foreground soothe-foreground-1               :background soothe-purple-1bg))
     (ecb-default-general-face                  (:foreground soothe-foreground))
     (ecb-default-highlight-face                (:foreground soothe-prime-purple               :background soothe-purple-1bg))
     (ecb-method-face                           (:foreground soothe-prime-orange               :background soothe-orange-1bg))
     (ecb-tag-header-face                       (:foreground soothe-prime-turquoise            :background soothe-turquoise-1bg))
     (ediff-current-diff-A                      (:background soothe-bg-red))
     (ediff-current-diff-Ancestor               (:background soothe-dark-blue))
     (ediff-current-diff-B                      (:background soothe-turquoise-1bg))
     (ediff-current-diff-C                      (:background soothe-bg-green))
     (ediff-even-diff-A                         (:background soothe-dark-red))
     (ediff-even-diff-Ancestor                  (:background soothe-dark-blue))
     (ediff-even-diff-B                         (:background soothe-turquoise-3))
     (ediff-even-diff-C                         (:background soothe-bg-green))
     (ediff-fine-diff-A                         (:background soothe-dark-red))
     (ediff-fine-diff-Ancestor                  (:background soothe-bg-blue))
     (ediff-fine-diff-B                         (:background soothe-turquoise-2bg))
     (ediff-fine-diff-C                         (:background soothe-dark-green))
     (ediff-odd-diff-A                          (:background soothe-red-2bg))
     (ediff-odd-diff-Ancestor                   (:background soothe-bg-blue))
     (ediff-odd-diff-B                          (:background soothe-turquoise-3))
     (ediff-odd-diff-C                          (:background soothe-green-1bg))
     (error                                     (:foreground soothe-prime-red))
     (escape-glyph                              (:foreground soothe-prime-red                  :background soothe-purple-1bg))
     (flycheck-error                            (:underline (:style 'wave :color soothe-red-3)))
     (flycheck-warning                          (:underline (:style 'wave :color soothe-orange-2)))
     (flycheck-info                             (:underline (:style 'wave :color soothe-rainbow-delimiters-0)))
     (flycheck-fringe-error                     (:inherit 'error))
     (flycheck-fringe-warning                   (:inherit 'warning))
     (flycheck-fringe-info                      (:inherit 'success))
     (flycheck-error-list-error                 (:inherit 'error))
     (flycheck-error-list-warning               (:inherit 'warning))
     (flycheck-error-list-info                  (:inherit 'success))
     (flycheck-error-list-filename              (:inherit 'mode-line-buffer-id :weight 'normal))
     (flycheck-error-list-id                    (:inherit 'font-lock-type-face))
     (flycheck-error-list-id-with-explainer     (:inherit 'flycheck-error-list-id :box (:style 'released-button)))
     (flycheck-error-list-checker-name          (:inherit 'error))
     (flycheck-error-list-highlight             (:weight 'bold))
     (flycheck-verify-select-checker            (:box (:style 'released-button)))
     (flymake-errline                           (:underline soothe-red-2                       :background nil :inherit nil))
     (flymake-warnline                          (:underline soothe-green-2                     :background nil :inherit nil))
     (flyspell-duplicate                        (:underline soothe-green-2))
     (flyspell-incorrect                        (:underline soothe-red-2))
     (font-lock-builtin-face                    (:foreground soothe-red-2                      :background soothe-red-1bg))
     (font-lock-comment-delimiter-face          (:foreground soothe-turquoise-1                :background soothe-alt-background :italic t))
     (font-lock-comment-face                    (:foreground soothe-prime-purple               :background soothe-alt-background :italic t))
     (font-lock-constant-face                   (:foreground soothe-prime-purple               :background soothe-purple-1bg))
     (font-lock-doc-face                        (:foreground soothe-prime-turquoise            :background soothe-gray-1bg))
     (font-lock-function-name-face              (:foreground soothe-prime-red                  :background soothe-red-1bg))
     (font-lock-keyword-face                    (:foreground soothe-prime-orange               :background soothe-orange-1bg))
     (font-lock-negation-char-face              (:foreground soothe-prime-yellow               :background soothe-yellow-1bg))
     (font-lock-preprocessor-face               (:foreground soothe-prime-orange               :background soothe-orange-1bg))
     (font-lock-string-face                     (:foreground soothe-prime-turquoise            :background soothe-turquoise-2bg))
     (font-lock-type-face                       (:foreground soothe-red-2                      :background soothe-red-1bg :bold nil))
     (font-lock-variable-name-face              (:foreground soothe-prime-blue                 :background soothe-blue-1bg))
     (font-lock-warning-face                    (:foreground soothe-red-2                      :background soothe-red-2bg))
     (fringe                                    (:foreground soothe-turquoise-2                :background soothe-gray-3bg))
     (git-gutter:added                          (:foreground soothe-prime-green                :background soothe-green-2bg))
     (git-gutter:deleted                        (:foreground soothe-prime-red                  :background soothe-red-1bg))
     (git-gutter:modified                       (:foreground soothe-prime-purple               :background soothe-purple-1bg))
     (git-gutter:unchanged                      (:background soothe-yellow-1bg))
     (header-line                               (:foreground soothe-prime-orange               :background soothe-background))
     (helm-action                               (:underline t))
     (helm-candidate-number                     (:foreground soothe-orange-2))
     (helm-candidate-number-suspended           (:foreground soothe-turquoise-0))
     (helm-eob-line                             (:inherit 'default))
     (helm-grep-cmd-line                        (:inherit 'font-lock-type-face))
     (helm-grep-file                            (:foreground soothe-prime-purple :underline t))
     (helm-grep-finish                          (:inherit 'success))
     (helm-grep-lineno                          (:foreground soothe-orange-2))
     (helm-grep-match                           (:foreground soothe-prime-turquoise            :background soothe-dark-purple))
     (helm-header                               (:inherit 'header-line))
     (helm-header-line-left-margin              (:foreground soothe-background-dark-0          :background soothe-orange-2))
     (helm-helper                               (:inherit 'helm-header))
     (helm-mark-prefix                          (:inherit 'default))
     (helm-match                                (:foreground soothe-prime-turquoise            :background soothe-dark-purple))
     (helm-match-item                           (:foreground soothe-orange-2                   :background soothe-turquoise-3))
     (helm-minibuffer-prompt                    (:inherit 'minibuffer-prompt))
     (helm-prefarg                              (:foreground soothe-turquoise-0))
     (helm-selection                            (:inherit 'region))
     (helm-selection-line                       (:inherit 'highlight))
     (helm-separator                            (:inherit 'ivy-separator))
     (helm-source-header                        (:inherit 'variable-pitch :foreground soothe-foreground :background soothe-turquoise-2bg :height 1.1))
     (helm-swoop-line-number-face               (:foreground soothe-prime-purple))
     (helm-swoop-target-line-block-face         (:foreground soothe-foreground-2               :background soothe-turquoise-1))
     (helm-swoop-target-line-face               (:foreground soothe-foreground-3               :background soothe-bg-blue))
     (helm-swoop-target-word-face               (:foreground soothe-overexposed-purple         :background soothe-low-purple))
     (helm-visible-mark                         (:foreground soothe-foreground-3               :background soothe-low-purple))
     (highlight-changes                         (:background soothe-background-dark-0))
     (highlight-changes-delete                  (:background soothe-background-dark-0 :foreground soothe-foreground-3 :underline (:color soothe-prime-red :style 'wave)))
     (highlight-changes-1                       (:background soothe-background-dark-0 :foreground soothe-prime-yellow))
     (highlight-changes-2                       (:background soothe-background-dark-0 :foreground soothe-prime-purple))
     (highlight-changes-3                       (:background soothe-background-dark-0 :foreground soothe-prime-blue))
     (highlight-changes-4                       (:background soothe-background-dark-0 :foreground soothe-prime-red))
     (highlight-changes-5                       (:background soothe-background-dark-0 :foreground soothe-prime-orange))
     (highlight-changes-6                       (:background soothe-background-dark-0 :foreground soothe-prime-green))
     (highlight-changes-7                       (:background soothe-background-dark-0 :foreground soothe-prime-turquoise))
     (highlight                                 (:background soothe-low-blue))
     (highlight-indentation-current-column-face (:background soothe-gray-5))
     (highlight-indentation-face                (:background soothe-background-dark))
     (hl-line                                   (:background soothe-bg-turquoise))
     (hl-sexp-face                              (:background soothe-bg-turquoise))
     (ido-first-match                           (:foreground soothe-prime-orange               :background soothe-orange-1bg))
     (ido-incomplete-regexp                     (:foreground soothe-prime-red                  :background soothe-orange-1bg))
     (ido-indicator                             (:foreground soothe-turquoise-1                :background soothe-turquoise-1bg))
     (ido-only-match                            (:foreground soothe-prime-green                :background soothe-green-1bg))
     (ido-subdir                                (:foreground soothe-prime-purple               :background soothe-purple-1bg))
     (ido-virtual                               (:foreground soothe-green-3                    :background soothe-turquoise-1bg))
     (iedit-occurrence                          (:foreground soothe-prime-turquoise            :background soothe-dark-purple                               :foreground soothe-prime-purple))
     (iedit-read-only-occurrence                (:foreground soothe-prime-turquoise            :background soothe-dark-purple                               :foreground soothe-prime-red))
     (isearch                                   (:foreground soothe-orange-2                   :background soothe-turquoise-3))
     (isearch-fail                              (:foreground soothe-foreground-0               :background soothe-red-3))
     (isearch-group-1                           (:foreground soothe-prime-purple               :background soothe-low-blue))
     (isearch-group-2                           (:foreground soothe-prime-purple               :background soothe-low-purple))
     (ivy-action                                (:inherit 'font-lock-builtin-face))
     (ivy-completions-annotations               (:inherit 'completions-annotations))
     (ivy-confirm-face                          (:inherit 'minibuffer-prompt                   :foreground soothe-rainbow-delimiters-0))
     (ivy-current-match                         (:foreground soothe-prime-turquoise            :background soothe-dark-purple))
     (ivy-cursor                                (:foreground soothe-foreground-1               :background soothe-bg-turquoise))
     (ivy-grep-info                             (:inherit 'compilation-info))
     (ivy-grep-line-number                      (:inherit 'compilation-line-number))
     (ivy-highlight-face                        (:inherit 'highlight))
     (ivy-match-required-face                   (:foreground soothe-red-3))
     (ivy-minibuffer-match-face-1               (:foreground soothe-prime-turquoise            :background soothe-dark-purple))
     (ivy-minibuffer-match-face-2               (:inherit 'highlight))
     (ivy-minibuffer-match-face-3               (:inherit 'secondary-selection))
     (ivy-minibuffer-match-face-4               (:inherit 'region))
     (ivy-minibuffer-match-highlight            (:inherit 'highlight))
     (ivy-modified-buffer                       (:inherit 'default))
     (ivy-modified-outside-buffer               (:inherit 'default))
     (ivy-org                                   (:inherit 'org-level-4))
     (ivy-prompt-match                          (:inherit 'ivy-current-match))
     (ivy-remote                                (:inherit 'font-lock-builtin-face))
     (ivy-separator                             (:inherit 'font-lock-doc-face))
     (ivy-subdir                                (:inherit 'dired-directory))
     (ivy-virtual                               (:inherit 'font-lock-builtin-face))
     (ivy-yanked-word                           (:inherit 'highlight))
     (js3-error-face                            (:background soothe-red-1bg                    :underline (:color soothe-prime-red :style 'wave)))
     (js3-external-variable-face                (:foreground soothe-prime-purple               :background soothe-purple-1bg))
     (js3-function-param-face                   (:foreground soothe-prime-turquoise            :background soothe-blue-3bg))
     (js3-instance-member-face                  (:foreground soothe-foreground-2               :background soothe-purple-1bg))
     (js3-jsdoc-html-tag-delimiter-face         (:foreground soothe-blue-4                     :background soothe-blue-2bg))
     (js3-jsdoc-html-tag-name-face              (:foreground soothe-foreground-1               :background soothe-blue-3bg))
     (js3-jsdoc-tag-face                        (:foreground soothe-green-3                    :background soothe-green-2bg))
     (js3-jsdoc-type-face                       (:foreground soothe-green-2                    :background soothe-green-2bg))
     (js3-jsdoc-value-face                      (:foreground soothe-prime-green                :background soothe-green-1bg))
     (js3-magic-paren-face                      (:foreground soothe-foreground-0               :background soothe-purple-1bg))
     (js3-private-function-call-face            (:foreground soothe-prime-orange               :background soothe-orange-1bg))
     (js3-private-member-face                   (:foreground soothe-orange-2                   :background soothe-orange-1bg))
     (js3-warning-face                          (:background soothe-yellow-1bg                 :underline (:color soothe-prime-yellow :style 'wave)))
     (lazy-highlight                            (:foreground soothe-prime-purple               :background soothe-bg-purple))
     (link                                      (:foreground soothe-prime-blue                 :background soothe-blue-1bg))
     (link-visited                              (:foreground soothe-prime-turquoise            :background soothe-blue-4bg))
     (linum                                     (:foreground soothe-gray-6                     :background soothe-alt-background))
     (magit-branch                              (:foreground soothe-prime-orange               :background soothe-orange-2bg))
     (magit-branch-current                      (:foreground soothe-prime-purple               :background soothe-purple-1bg :box (:line-width -1  :color soothe-low-purple)))
     (magit-branch-local                        (:foreground soothe-prime-purple               :background soothe-purple-1bg))
     (magit-branch-remote                       (:foreground soothe-prime-orange               :background soothe-orange-2bg))
     (magit-diff-added                          (:foreground soothe-foreground-1               :background soothe-bg-green))
     (magit-diff-added-highlight                (:foreground soothe-foreground-1               :background soothe-dark-green))
     (magit-diff-base                           (:foreground soothe-foreground                 :background soothe-rainbow-delimiters-0))
     (magit-diff-base-highlight                 (:foreground soothe-foreground-0               :background soothe-rainbow-delimiters-0))
     (magit-diff-conflict-heading               (:inherit 'magit-diff-hunk-heading))
     (magit-diff-context                        (:background soothe-background))
     (magit-diff-context-highlight              (:background soothe-alt-background))
     (magit-diff-file-heading                   (:weight 'bold))
     (magit-diff-file-heading-highlight         (:inherit 'magit-section-highlight))
     (magit-diff-file-heading-selection         (:inherit 'magit-diff-file-heading-highlight   :foreground soothe-prime-yellow))
     (magit-diff-hunk-heading                   (:foreground soothe-foreground                 :background soothe-low-purple))
     (magit-diff-hunk-heading-highlight         (:foreground soothe-foreground                 :background soothe-low-purple))
     (magit-diff-hunk-heading-selection         (:foreground soothe-prime-yellow               :inherit 'magit-diff-hunk-heading-highlight))
     (magit-diff-hunk-region                    (:inherit 'bold))
     (magit-diff-lines-boundary                 (:inherit 'magit-diff-lines-heading))
     (magit-diff-lines-heading                  (:inherit 'magit-diff-hunk-heading-highlight   :foreground soothe-foreground-3 :background soothe-red-2))
     (magit-diff-merge-current                  (:foreground soothe-prime-blue))
     (magit-diff-merge-proposed                 (:foreground soothe-foreground-1))
     (magit-diff-merge-separator                (:foreground soothe-blue-2))
     (magit-diff-our                            (:inherit 'magit-diff-removed))
     (magit-diff-our-highlight                  (:inherit 'magit-diff-removed-highlight))
     (magit-diff-removed                        (:foreground soothe-foreground-1               :background soothe-bg-red))
     (magit-diff-removed-highlight              (:foreground soothe-foreground-1               :background soothe-dark-red))
     (magit-diff-revision-summary               (:inherit 'magit-diff-hunk-heading))
     (magit-diff-revision-summary-highlight     (:inherit 'magit-diff-hunk-heading-highlight))
     (magit-diff-their                          (:inherit 'magit-diff-added))
     (magit-diff-their-highlight                (:inherit 'magit-diff-added-highlight))
     (magit-diff-whitespace-warning             (:inherit 'trailing-whitespace))
     (magit-diffstat-added                      (:foreground soothe-overexposed-green))
     (magit-diffstat-removed                    (:foreground soothe-overexposed-red))
     (magit-hash                                (:foreground soothe-prime-purple               :background soothe-purple-1bg))
     (magit-header                              (:foreground soothe-prime-orange               :background soothe-orange-1bg))
     (magit-item-highlight                      (:background soothe-purple-3bg))
     (magit-item-mark                           (:foreground soothe-prime-green))
     (magit-log-author                          (:foreground soothe-prime-purple               :background soothe-purple-1bg))
     (magit-log-author                          (:foreground soothe-prime-purple               :background soothe-purple-1bg))
     (magit-log-date                            (:foreground soothe-blue-4                     :background soothe-background))
     (magit-log-graph                           (:foreground soothe-blue-2                     :background soothe-blue-2bg))
     (magit-log-head-label-bisect-bad           (:foreground soothe-prime-red                  :background soothe-red-1bg))
     (magit-log-head-label-bisect-good          (:foreground soothe-turquoise-1                :background soothe-turquoise-1bg))
     (magit-log-head-label-default              (:foreground soothe-foreground-1               :background soothe-turquoise-1bg))
     (magit-log-head-label-local                (:foreground soothe-foreground-1               :background soothe-turquoise-1bg))
     (magit-log-head-label-patches              (:foreground soothe-blue-2                     :background soothe-blue-1bg))
     (magit-log-head-label-remote               (:foreground soothe-foreground-1               :background soothe-purple-2bg))
     (magit-log-head-label-tags                 (:foreground soothe-prime-orange               :background soothe-orange-1bg))
     (magit-log-message                         (:foreground soothe-foreground-2               :background soothe-background))
     (magit-log-sha1                            (:foreground soothe-turquoise-1                :background soothe-turquoise-1bg))
     (magit-reflog-rebase                       (:foreground soothe-prime-turquoise            :background soothe-turquoise-1bg))
     (magit-reflog-other                        (:foreground soothe-prime-turquoise            :background soothe-turquoise-1bg))
     (magit-reflog-remote                       (:foreground soothe-prime-turquoise            :background soothe-turquoise-1bg))
     (magit-section-heading                     (:foreground soothe-prime-orange               :background soothe-purple-1bg))
     (magit-section-highlight                   (:foreground soothe-prime-purple               :background soothe-purple-1bg))
     (magit-section-highlight-selection         (:foreground soothe-prime-purple               :background soothe-purple-1bg))
     (magit-section-title                       (:foreground soothe-prime-purple               :background soothe-purple-1bg))
     (magit-tag                                 (:foreground soothe-prime-orange               :background soothe-orange-1bg))
     (magit-whitespace-warning-face             (:foreground soothe-red-3                      :background soothe-red-1bg))
     (markdown-header-face-1                    (:inherit 'variable-pitch                      :foreground soothe-overexposed-purple :height 1.9))
     (markdown-header-face-2                    (:inherit 'variable-pitch                      :foreground soothe-overexposed-blue :height 1.8))
     (markdown-header-face-3                    (:inherit 'variable-pitch                      :foreground soothe-overexposed-green :height 1.6))
     (markdown-header-face-4                    (:inherit 'variable-pitch                      :foreground soothe-overexposed-orange :height 1.5))
     (markdown-header-face-5                    (:inherit 'variable-pitch                      :foreground soothe-overexposed-purple :height 1.4))
     (markdown-header-face-6                    (:inherit 'variable-pitch                      :foreground soothe-overexposed-blue :height 1.3))
     (match                                     (:foreground soothe-prime-turquoise            :background soothe-dark-purple))
     (minibuffer-prompt                         (:foreground soothe-prime-orange               :background soothe-orange-1bg))
     (mode-line                                 (:foreground soothe-prime-turquoise            :background soothe-bg-turquoise :height 1.0))
     (mode-line-inactive                        (:foreground soothe-faded-purple               :background soothe-dark-purple :inherit 'mode-line :height 1.0))
     (mode-line-highlight                       (:foreground soothe-prime-red))
     (mode-line-buffer-id                       (:foreground soothe-prime-orange))
     (mode-line-emphasis                        (:bold))
     (lambda-line-active                        (:inherit 'mode-line))
     (lambda-line-active-name                   (:inherit 'mode-line))
     (lambda-line-active-primary                (:inherit 'mode-line))
     (lambda-line-active-secondary              (:inherit 'mode-line))
     (lambda-line-active-status-MD              (:inherit 'mode-line                           :background soothe-dark-purple :foreground soothe-overexposed-purple))
     (lambda-line-active-status-RO              (:inherit 'mode-line                           :foreground soothe-orange-2))
     (lambda-line-active-status-RW              (:inherit 'mode-line                           :foreground soothe-prime-turquoise))
     (lambda-line-active-tertiary               (:inherit 'mode-line))
     (lambda-line-hspace-active                 (:inherit 'mode-line))
     (lambda-line-hspace-inactive               (:inherit 'mode-line-inactive))
     (lambda-line-inactive                      (:inherit 'mode-line-inactive))
     (lambda-line-inactive-name                 (:inherit 'mode-line-inactive))
     (lambda-line-inactive-primary              (:inherit 'mode-line-inactive))
     (lambda-line-inactive-secondary            (:inherit 'mode-line-inactive))
     (lambda-line-inactive-status-MD            (:inherit 'mode-line-inactive                  :background soothe-low-purple  :foreground soothe-faded-purple))
     (lambda-line-inactive-status-RO            (:inherit 'mode-line-inactive                  :foreground soothe-faded-orange))
     (lambda-line-inactive-status-RW            (:inherit 'mode-line-inactive                  :foreground soothe-faded-purple))
     (lambda-line-inactive-tertiary             (:inherit 'mode-line-inactive))
     (lambda-line-visual-bell                   (:background soothe-turquoise-2))
     (orderless-match-face-0                    (:foreground soothe-prime-turquoise            :background soothe-dark-purple                               :foreground soothe-prime-turquoise))
     (orderless-match-face-1                    (:inherit 'highlight                           :foreground soothe-prime-purple))
     (orderless-match-face-2                    (:foreground soothe-prime-purple               :background soothe-bg-purple                      :foreground soothe-prime-orange))
     (orderless-match-face-3                    (:inherit 'secondary-selection                 :foreground soothe-prime-blue))
     (org-date                                  (:foreground soothe-prime-purple               :background soothe-purple-1bg))
     (org-done                                  (:foreground soothe-prime-green                :background soothe-green-1bg))
     (org-hide                                  (:foreground soothe-gray-3                     :background soothe-gray-1bg))
     (org-link                                  (:foreground soothe-prime-blue                 :background soothe-blue-1bg))
     (org-todo                                  (:foreground soothe-prime-red                  :background soothe-red-1bg))
     (popup-tip-face                            (:foreground soothe-foreground-2               :background soothe-background-2))
     (popup-isearch-match                       (:background soothe-bg-turquoise               :inherit 'default))
     (rainbow-delimiters-depth-1-face           (:foreground soothe-rainbow-delimiters-0))
     (rainbow-delimiters-depth-2-face           (:foreground soothe-rainbow-delimiters-1))
     (rainbow-delimiters-depth-3-face           (:foreground soothe-rainbow-delimiters-2))
     (rainbow-delimiters-depth-4-face           (:foreground soothe-rainbow-delimiters-3))
     (rainbow-delimiters-depth-5-face           (:foreground soothe-rainbow-delimiters-4))
     (rainbow-delimiters-depth-6-face           (:foreground soothe-rainbow-delimiters-5))
     (rainbow-delimiters-depth-7-face           (:foreground soothe-rainbow-delimiters-6))
     (rainbow-delimiters-depth-8-face           (:foreground soothe-rainbow-delimiters-7))
     (rainbow-delimiters-depth-9-face           (:foreground soothe-rainbow-delimiters-8))
     (rainbow-delimiters-unmatched-face         (:foreground soothe-rainbow-delimiters-9))
     (reb-regexp-grouping-backslash             (:foreground soothe-prime-turquoise))
     (reb-regexp-grouping-construct             (:foreground soothe-prime-orange))
     (reb-match-0                               (:foreground soothe-prime-turquoise            :background soothe-turquoise-2bg))
     (reb-match-1                               (:foreground soothe-faded-purple               :background soothe-turquoise-2bg))
     (reb-match-2                               (:foreground soothe-prime-turquoise            :background soothe-bg-purple))
     (reb-match-3                               (:foreground soothe-prime-orange               :background soothe-bg-purple))
     (region                                    (:background soothe-blue-5bg))
     (secondary-selection                       (:background soothe-bg-turquoise))
     (show-paren-match                          (:foreground soothe-prime-turquoise            :background soothe-bg-purple))
     (show-paren-mismatch                       (:foreground soothe-prime-orange               :background soothe-red-2bg))
     (smerge-base                               (:inherit 'diff-context))
     (smerge-lower                              (:background soothe-turquoise-1bg))
     (smerge-markers                            (:foreground soothe-prime-purple               :background soothe-purple-1bg))
     (smerge-refined-added                      (:inherit 'diff-added))
     (smerge-refined-changed                    (:inherit 'diff-changed))
     (smerge-refined-removed                    (:inherit 'diff-removed))
     (smerge-upper                              (:background soothe-blue-1bg))
     (success                                   (:foreground soothe-prime-turquoise))
     (swiper-background-match-face-1            (:inherit 'swiper-match-face-1))
     (swiper-background-match-face-2            (:inherit 'swiper-match-face-2))
     (swiper-background-match-face-3            (:inherit 'swiper-match-face-3))
     (swiper-background-match-face-4            (:inherit 'swiper-match-face-4))
     (swiper-line-face                          (:foreground soothe-prime-turquoise            :background soothe-dark-purple))
     (swiper-match-face-1                       (:foreground soothe-prime-purple               :background soothe-bg-purple))
     (swiper-match-face-2                       (:foreground soothe-orange-2                   :background soothe-turquoise-3))
     (swiper-match-face-3                       (:foreground soothe-prime-turquoise            :background soothe-dark-purple))
     (swiper-match-face-4                       (:inherit 'isearch-fail))
     (tooltip                                   (:foreground soothe-background-2               :background soothe-foreground-2))
     (vertical-border                           (:foreground soothe-gray-4                     :background soothe-background))
     (vertico-posframe                          (:background soothe-background-dark))
     (vertico-posframe-border                   (:background soothe-background-dark-0))
     (vertico-posframe-border-2                 (:background soothe-alt-background))
     (vertico-posframe-border-3                 (:background soothe-background-2))
     (vertico-posframe-border-4                 (:background soothe-background-3))
     (vline                                     (:background soothe-bg-turquoise))
     (vline-visual                              (:background soothe-bg-turquoise))
     (warning                                   (:foreground soothe-orange-2 :weight 'bold))
     (web-mode-doctype-face                     (:foreground soothe-foreground-2 :weight 'bold))
     (web-mode-html-attr-equal-face             (:inherit 'default))
     (web-mode-html-attr-name-face              (:inherit 'font-lock-keyword-face))
     (web-mode-html-tag-bracket-face            (:inherit 'default))
     (web-mode-html-tag-face                    (:inherit 'font-lock-builtin-face))
     (which-func                                (:foreground soothe-prime-blue))
     (which-key-command-description-face        (:inherit 'font-lock-function-name-face))
     (which-key-group-description-face          (:inherit 'font-lock-keyword-face))
     (which-key-highlighted-command-face        (:inherit 'which-key-command-description-face  :underline t))
     (which-key-key-face                        (:inherit 'font-lock-constant-face))
     (which-key-local-map-description-face      (:inherit 'which-key-command-description-face))
     (which-key-note-face                       (:inherit 'which-key-separator-face))
     (which-key-separator-face                  (:inherit 'font-lock-comment-face))
     (which-key-special-key-face                (:inherit 'which-key-key-face                  :weight 'bold :inverse-video t))
     (whitespace-empty                          (:foreground soothe-prime-yellow               :background soothe-turquoise-2bg))
     (whitespace-hspace                         (:foreground soothe-turquoise-2                :background soothe-turquoise-2bg))
     (whitespace-indentation                    (:foreground soothe-turquoise-2                :background soothe-turquoise-2bg))
     (whitespace-line                           (:foreground soothe-prime-orange               :background soothe-turquoise-2bg))
     (whitespace-newline                        (:foreground soothe-turquoise-2                :background soothe-turquoise-2bg))
     (whitespace-space                          (:foreground soothe-turquoise-2                :background soothe-turquoise-2bg))
     (whitespace-space-after-tab                (:foreground soothe-turquoise-2                :background soothe-turquoise-2bg))
     (whitespace-tab                            (:foreground soothe-turquoise-2                :background soothe-turquoise-2bg))
     (whitespace-trailing                       (:foreground soothe-prime-red                  :background soothe-turquoise-2bg))
     (window-divider-first-pixel                (:foreground soothe-background-dark-0))
     (window-divider-last-pixel                 (:foreground soothe-background-dark-0))
     (window-divider                            (:foreground soothe-background-dark-0)))

    ,@body))

  ;;;###autoload
(and load-file-name
     (boundp 'custom-theme-load-path)
     (add-to-list 'custom-theme-load-path
                  (file-name-as-directory
                   (file-name-directory load-file-name))))

;; For dev.
(defun soothe-tva--dev-reload (prefix)
  "Reload dev soothe-theme. non-nil PREFIX keep complilation window open."
  (interactive "P")
  (let* ((theme-file-name (or (let* ((f "soothe-theme.el")
                                     (i (- 0 (length f))))
                                  (--find
                                      (string= f (substring it i))
                                    (directory-files default-directory t)))
                              (read-file-name "Locate sooth-theme.el: " nil nil t nil)))
         (tva-file-name (format "%ssoothe-tva.el" (file-name-directory theme-file-name))))

    (dolist (it (list 'soothe 'soothe-theme 'soothe-tva))
      (when (featurep it) (unload-feature it t)))

    (dolist (it (list tva-file-name theme-file-name))
      (byte-compile-file it)
      (load-file it)
      (load-file
       (replace-regexp-in-string
        "[.]el" ".elc" it)))
   (unless prefix
     (let ((compilation-buffer (get-buffer "*Compile-Log*")))
      (when compilation-buffer (kill-buffer compilation-buffer))))
   (disable-theme 'soothe)
   (enable-theme 'soothe)))


;;; Formatting:
;;
;; The first two attribute pairs in the spec list should be column aligned for
;; visibility. Subsequent specs should be single spaced to keep them over to the left side. e.g.
;;
;; (:foreground soothe-prime-red    :background soothe-background-3)
;; (:foreground soothe-turquoise-2  :background soothe-background-0 :box ( ... ) :underline t)
;; (:foreground soothe-foreground-2 :background soothe-background-3 :underline soothe-prime-red)
;; (:inhert 'default                :foreground soothe-foreground-1 :background soothe-background-3)
;;
;; Inherit should always appear first.

(provide 'soothe-tva)

;;; soothe-tva.el ends here
