;;; run-command.el --- Run an external command from a context-dependent list -*- lexical-binding: t -*-

;; Copyright (C) 2020-2023 Massimiliano Mirra

;; Author: Massimiliano Mirra <hyperstruct@gmail.com>
;; URL: https://github.com/bard/emacs-run-command
;; Package-Version: 1.0.0
;; Package-Revision: e44bc5fb9712
;; Package-Requires: ((emacs "27.1"))
;; Keywords: processes

;; This file is not part of GNU Emacs

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 3, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; For a full copy of the GNU General Public License
;; see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; Leave Emacs less.  Launch both one-off utilities and long-lived processes
;; from context-sensitive lists with autocompletion.
;;
;; Supports Helm, Ivy, and Emacs's `completing-read' for selection, and
;; `term-mode', `compilation-mode', `vterm-mode', and `eat-mode' for command
;; output and interaction.

;;; Code:

;;;; Dependencies

(require 'run-command-core)
(require 'run-command-selector-helm)
(require 'run-command-selector-ivy)
(require 'run-command-selector-completing-read)
(require 'run-command-runner-term)
(require 'run-command-runner-compile)
(require 'run-command-runner-vterm)
(require 'run-command-runner-eat)

;;;; Customization

(defgroup run-command nil
  "Run an external command from a context-sensitive list."
  :group 'convenience
  :group 'processes
  :prefix "run-command-"
  :link '(url-link "https://github.com/bard/emacs-run-command"))

(defcustom run-command-default-runner 'run-command-runner-term
  "Runner function to use when when a command recipe does not specify one.

If nil, falls back to `run-command-runner-term', except on
Windows where it falls back to `run-command-runner-compile'.

The following runners are shipped with `run-command':

- `run-command-runner-term': display command output in a
  `term-mode' buffer

- `run-command-runner-compile': display command output in a
  `compilation-mode' buffer

- `run-command-runner-vterm': display command output in a `vterm'
  buffer (requires `vterm' package to be available)

- `run-command-runner-eat': display command output in a `eat'
  buffer (requires `eat' package to be available)"
  :type 'function
  :group 'run-command)

(defcustom run-command-selector nil
  "Interface to use to select a command.

If nil, choose one based on active global modes (such as
`helm-mode', `ivy-mode') or fall back to `completing-read'.

Selectors shipped with `run-command':

- `run-command-selector-completing-read': use built-in `completing-read'

- `run-command-selector-helm': use Helm

- `run-command-selector-ivy': use Ivy"
  :type 'function)

(defcustom run-command-recipes nil
  "List of functions that will produce command lists.

Some recipes ship with `run-command', see
`run-command-load-cookbook'.

Each function is called without arguments and must return a list
of command specifications.

A command specification is a property list with the following
properties:

- `:command-name' (string, required): A name for the command,
  used to generate the output buffer name, as well as a fallback
  in case `:display' isn't provided.

- `:display' (string, optional): A descriptive name that will be
  displayed in the selector interface.

- `:command-line' (string or function, required): The command
  line that will be executed.  Can be a string or a function
  returning a string.  A function can be used to e.g. ask the user
  for additional information to use in assembling a command line.

- `:working-dir' (string, optional): Directory to run the command
  in.  If not provided, defaults to `default-directory'.

- `:runner' (function, optional): a runner function specifically
  for this command.  If not provided, `run-command-default-runner'
  will be used.

- `:hook' (function, optional): a function to run just after the
  command has been launched.  Can be used to e.g. activate
  compilation minor mode."

  :type '(repeat function)
  :group 'run-command)


;;;; Entry points

;;;###autoload
(defun run-command ()
  "Display a context-sensitive list of external commands and run one.

The command list is generated by running the functions configured in
`run-command-recipes'."
  (interactive)
  (run-command--check-experiments)

  (when (not run-command-recipes)
    (error
     "[run-command] Please customize `run-command-recipes' in order to use `run-command'"))

  (funcall (run-command--get-selector) run-command-recipes))

;;;; Internals

(defun run-command--get-selector ()
  "Determine what selector to use."
  (or (pcase run-command-selector
        ((pred null)
         (cond
          ((and (boundp 'helm-mode) helm-mode)
           'run-command-selector-helm)
          ((and (boundp 'ivy-mode) ivy-mode)
           'run-command-selector-ivy)
          (t
           'run-command-selector-completing-read)))
        ((pred functionp) run-command-selector)
        (_ (error "[run-command] Could not determine selector")))
      ;; Backward compatibility
      (and (fboundp 'run-command-completion-method)
           (pcase run-command-completion-method
             ('auto
              (cond
               ((and (boundp 'helm-mode) helm-mode)
                'run-command-selector-helm)
               ((and (boundp 'ivy-mode) ivy-mode)
                'run-command-selector-ivy)
               (t
                'run-command-selector-completing-read)))
             ('helm 'run-command-selector-helm)
             ('ivy 'run-command-selector-ivy)
             ('completing-read 'run-command-selector-completing-read)))
      'run-command-selector-completing-read))

;;;###autoload
(defun run-command-load-cookbook ()
  "Load all recipes from the cookbook.

Use this in your init file to make all recipes shipped with
`run-command' available.  (You will still need to customize
`run-command-recipes' to activate specific ones.)"
  (interactive)
  (let* ((examples-dir
          (concat
           (file-name-directory (locate-library "run-command")) "cookbook/")))
    (thread-last
     (directory-files examples-dir t)
     (seq-filter #'file-regular-p)
     (seq-do #'load-file))))

;;;; Backward compatibility

(defcustom run-command-completion-method 'auto
  "Completion framework to use to select a command.
If nil, choose one based on active global modes (such as
`helm-mode', `ivy-mode') or fall back to `completing-read'.

- `autodetect' (default): pick helm, ivy, or none, based on active
  global completion mode

- `helm': force use helm

- `ivy': force use ivy

- `completing-read': force use `completing-read'"
  :type
  '(choice
    (const :tag "autodetect" auto)
    (const :tag "helm" helm)
    (const :tag "ivy" ivy)))
(make-obsolete-variable
 'run-command-completion-method 'run-command-selector "0.2.0")

(defcustom run-command-run-method 'compile
  "Run strategy.

- `compile' (default): display command output in a `compilation-mode' buffer

- `term': display command output in a `term-mode' buffer"
  :type
  '(choice
    (const :tag "Terminal Mode" term)
    (const :tag "Compilation Mode" compile)))
(make-obsolete-variable
 'run-command-run-method 'run-command-default-runner "0.2.0")

;;;; Meta

(provide 'run-command)

;;; run-command.el ends here
