;;; prefab.el --- Integration for project generation tools like cookiecutter -*- lexical-binding: t -*-

;; Author: Laurence Warne
;; Maintainer: Laurence Warne
;; Package-Version: 0.1.1
;; Package-Revision: v0.1.1-0-ge4775959e541
;; URL: https://github.com/laurencewarne/prefab.el
;; Package-Requires: ((emacs "27.1") (f "0.2.0") (transient "0.3.7"))

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.

;;; Commentary:

;; prefab.el is a tool aiming to provide integration for project generation
;; tools like cookiecutter.  It's main entry point is `prefab', after
;; invocation the steps are:
;; 1) Prompt you for pre-used templates (or you can paste in a new one)
;; 2) Edit the template variables through the transient interface
;; 3) Generate the project

;;; Code:

(require 'python)
(require 'json)
(require 'f)
(require 'transient)
(require 'subr-x)

;;; Custom variables

(defgroup prefab nil
  "Project generation for Emacs."
  :group 'applications)

(defcustom prefab-cookiecutter-template-sources
  (list (format "%s/.cookiecutters" (getenv "HOME")))
  "List of directories to search for cookiecutter templates."
  :group 'prefab
  :type 'directory)

(defcustom prefab-cookiecutter-config-file
  nil
  "The cookiecutter config file location.

Only change this if your cookiecutter config file is not located in the
standard location: '~/.cookiecutterrc'.

Creating a config file is not necessary but can save some boilerplate.  For
example specifying 'author' and 'email' may be a good idea as these are common
template attributes.

See
URL 'https://cookiecutter.readthedocs.io/en/latest/advanced/user_config.html'
for more information."
  :group 'prefab
  :type 'file)

(defcustom prefab-cookiecutter-output-dir
  (format "%s/projects" (getenv "HOME"))
  "Where to output projects generated by cookiecutter."
  :group 'prefab
  :type 'directory)

(defcustom prefab-cookiecutter-replay-dir
  (format "%s/.cookiecutter_replay/" (getenv "HOME"))
  "Where to look for cookiecutter replays."
  :group 'prefab
  :type 'directory)

(defcustom prefab-cookiecutter-python-executable
  python-shell-interpreter
  "The path of the python executable to invoke for cookiecutter code."
  :group 'prefab
  :type 'string)

(defcustom prefab-cookiecutter-get-context-from-replay
  nil
  "If non-nil pre-populate the prefab transient with context from the last run.

Else pre-populate it using the template defaults, note these can be toggled
in the prefab transient prefix with the '-' key."
  :group 'prefab
  :type 'boolean)

(defcustom prefab-default-templates
  '((cookiecutter . ("https://github.com/LaurenceWarne/cookiecutter-eldev")))
  "Templates to prompt the user for if completing read would otherwise be empty."
  :group 'prefab
  :type '(alist :key-type symbol :value-type (repeat string)))

;;; Constants

(defconst prefab-version "0.1.0")

(defconst prefab--cookiecutter-not-found-err-msg
  "'cookiecutter' was not found on the PATH.

Visit https://cookiecutter.readthedocs.io/en/latest/installation.html for
installation instructions.")

;;; Internal variables

(defvar prefab-debug nil)
(defvar prefab-all-keys
  (mapcar #'char-to-string
          (string-to-list
           "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ")))

;;; Internal functions

(defun prefab--keys (keywords blacklist)
  "Return an alist of keywords in KEYWORDS to keys not in BLACKLIST."
  (if-let ((kw (car keywords)))
      (let ((key (cl-find-if (lambda (c) (or (not (member c blacklist))
                                             (not (member (upcase c) blacklist))))
                             (mapcar #'char-to-string (string-to-list kw)))))
        (cond ((member key blacklist)
               (cons `(,kw . ,(upcase key))
                     (prefab--keys (cdr keywords) `(,(upcase key) . ,blacklist))))
              (key (cons `(,kw . ,key)
                         (prefab--keys (cdr keywords) `(,key . ,blacklist))))
              (t (if-let ((fallback
                           (cl-find-if (lambda (c) (or (not (member c blacklist))))
                                       prefab-all-keys)))
                     (cons `(,kw . ,fallback)
                           (prefab--keys (cdr keywords) `(,fallback . ,blacklist)))
                   (error "Could not find a complete set of keys")))))))

(defun prefab--alist-to-python-dict (alist)
  "Convert ALIST to a python dictionary (as a string)."
  (format "{%s}"
          (mapconcat #'identity
                     (cl-loop for (key . value) in alist
                              collect
                              (format "'%s': '%s'"
                                      (prefab--escape-quotes key)
                                      (prefab--escape-quotes value)))
                     ", ")))

(defun prefab--json-from-python (python-src)
  "Return JSON returned by executing PYTHON-SRC.

The returned JSON will be of the form returned by `json-read'"
  (let ((shell-output
         (shell-command-to-string
          (format "%s -c \"%s\""
                  prefab-cookiecutter-python-executable
                  python-src)))
        (error-fmt-string "Got error: '%s' parsing the result of:
%s
shell output:
%s"))
    (condition-case err
        (json-read-from-string shell-output)
      (error
       (error error-fmt-string
              (error-message-string err) python-src shell-output)))))

(defun prefab--python-check-for-err (output error-msg-prefix)
  "Check the string OUTPUT for python errors.

Check the string OUTPUT for errors.  If it is an error string signal
an error starting with ERROR-MSG-PREFIX, else return OUTPUT."
  (if (string-match-p "^Error:" output)
      (error (concat error-msg-prefix output))
    output))

(defun prefab--cookiecutter-conf ()
  "Return the cookiecutter config file as an appropriate python object."
  (if prefab-cookiecutter-config-file
      (format "'%s'" prefab-cookiecutter-config-file)
    "None"))

(defun prefab--cookiecutter-created-dir (template-dir ctx)
  "Return the created directory implied by TEMPLATE-DIR and CTX (an alist)."
  (let* ((ctx-str (format "{'cookiecutter': %s}"
                          (prefab--alist-to-python-dict ctx)))
         (src (format "from cookiecutter.find import find_template
from cookiecutter.environment import StrictEnvironment
import os.path

ctx = %s
template_dir = find_template('%s')
dirname = os.path.split(template_dir)[1]
envvars = ctx.get('cookiecutter', {}).get('_jinja2_env_vars', {})
env = StrictEnvironment(context=ctx, keep_trailing_newline=True, **envvars)
output_dir = '%s'

name_tmpl = env.from_string(dirname)
rendered_dirname = name_tmpl.render(**ctx)
prev = None

# https://stackoverflow.com/questions/8862731/jinja-nested-rendering-on-variable-content
while rendered_dirname != prev:
    rendered_dirname, prev = env.from_string(rendered_dirname).render(**ctx), rendered_dirname

dir_to_create = os.path.normpath(os.path.join(output_dir, rendered_dirname))

print(dir_to_create, end='')"
                      ctx-str template-dir prefab-cookiecutter-output-dir))
         (output
          (shell-command-to-string
           (format "%s -c \"%s\"" prefab-cookiecutter-python-executable src))))
    (prefab--python-check-for-err output "Error getting output directory: ")))

(defun prefab--cookiecutter-download-template (template)
  "Download TEMPLATE and return the template directory."
  (let* ((src (format "from cookiecutter.config import get_user_config
from cookiecutter.repository import determine_repo_dir

config_dict = get_user_config(config_file=%s)
repo_dir, cleanup = determine_repo_dir(
    template='%s',
    abbreviations=config_dict['abbreviations'],
    clone_to_dir=config_dict['cookiecutters_dir'],
    checkout=None,
    no_input=False,
    #password=password,
    #directory=directory,
)
print(repo_dir, end='')" (prefab--cookiecutter-conf) template))
         (output (shell-command-to-string
                  (format "%s -c \"%s\""
                          prefab-cookiecutter-python-executable src))))
    (prefab--python-check-for-err output "Error downloading template: ")))

(defun prefab--escape-quotes (s)
  "Return S with quotes escaped."
  (replace-regexp-in-string "'" "\\'" s nil t nil 0))

(defun prefab--escape-quotes-bash (s)
  "Return S with quotes escaped, suitable for bash.

See URL 'https://stackoverflow.com/questions/1250079/how-to-escape-single-quotes-within-single-quoted-strings'."
  (replace-regexp-in-string "'" "'\"'\"'" s nil t nil 0))

(defun prefab--transient-set-value (project-src template replay)
  "Set the infixes and suffixes of the prefab transient.

PROJECT-SRC should be an instance of `prefab-source' and TEMPLATE should be
a member of the value returned by `prefab-templates' of the same object.
If REPLAY is non-nil add a use replay suffix to the transient, else add a use
original suffix."
  (pcase-let*
      ((`(,ctx . ,resolved-template)
        (prefab-default-context project-src template replay))
       (keywords (mapcar (lambda (cell) (symbol-name (car cell))) ctx))
       (key-lookup (prefab--keys keywords '("t" "c")))
       (options (cl-loop for (key-sym . value) in ctx
                         for key = (symbol-name key-sym)
                         collect
                         (list (alist-get key key-lookup)
                               (replace-regexp-in-string "[_-]+" " " key)
                               (concat key "="))))
       (v-options (vconcat ["Context"] options))
       (template-str (prefab-template-display-string project-src template))
       (template-options
        (vconcat (vector (format "Template: %s" template-str))
                 (when (prefab-replay-exists-p project-src template)
                   (list (list "-"
                               (if replay "Template defaults" "Replay Last")
                               (lambda () (interactive) (prefab--transient-set-value project-src template (not replay))) ':transient t))))))
    (transient-replace-suffix 'prefab--uri (list 0) template-options)
    (transient-replace-suffix 'prefab--uri (list 1) v-options)
    (oset (get 'prefab--uri 'transient--prefix)
          :value (cl-loop for (key . value)
                          in (cons (cons 'template template-str) ctx)
                          collect (format "%s=%s" (symbol-name key) value)))
    (put 'prefab--uri 'prefab-project-src project-src)
    (put 'prefab--uri 'prefab-template resolved-template)
    (prefab--uri)))

(defun prefab--run (args)
  "Run cookiecutter using ARGS."
  (interactive (list (transient-args transient-current-command)))
  (let* ((context (mapcar (lambda (s) (let ((split (split-string s "=")))
                                        (cons (car split) (cadr split))))
                          args))
         (template (get 'prefab--uri 'prefab-template))
         (project-src (get 'prefab--uri 'prefab-project-src))
         (out-dir (prefab-run project-src template context)))
    (dired out-dir)))

(transient-define-prefix prefab--uri ()
  :value '("???=nonempty")
  ["Template: ???"]
  ["Context"
   ("?" "???" "" read-string)]
  ["Actions"
   ("c" "Create"    prefab--run)])

(defclass prefab-source ()
  nil
  "A class representing a project generation method.")

(cl-defgeneric prefab-template-display-string ((source prefab-source) template)
  "Convert TEMPLATE to a string using SOURCE.

TEMPLATE should be of the form returned by `prefab-templates'.")

(cl-defgeneric prefab-templates ((source prefab-source))
  "Return a list of templates from SOURCE.")

(cl-defgeneric prefab-replay-exists-p ((source prefab-source) template)
  "Return t if a replay exists for this TEMPLATE for this SOURCE else nil.

TEMPLATE should be of the form returned by `prefab-templates'")

(cl-defgeneric prefab-default-context ((source prefab-source) template replay)
  "Return the default template resource context for TEMPLATE using SOURCE.

This method should return a cons cell whose `car' is the default context and
whose `cdr' is a template resource.  The format of the input template and
output template resource should is specific to the implementation though they
should both be strings.  A non-nil value for REPLAY indicates that context
from a replay is preferred.")

(cl-defgeneric prefab-run ((source prefab-source) template context)
  "Create a project from TEMPLATE and CONTEXT using SOURCE.

CONTEXT is an alist with string keys (template attributes) and values
 (attribute values).  TEMPLATE should be of the form returned by
`prefab-templates'")

;; Implementations

(defclass prefab-cookiecutter-source (prefab-source) nil)

(cl-defmethod prefab-templates ((_ prefab-cookiecutter-source))
  "Return a list of templates from SOURCE."
  (mapcan #'f-directories
          (cl-remove-if-not #'f-exists-p prefab-cookiecutter-template-sources)))

(cl-defmethod prefab-template-display-string
  ((_ prefab-cookiecutter-source) template)
  "Convert TEMPLATE to a string suitable to display to a user.

TEMPLATE should be of the form returned by `prefab-templates'."
  (if (f-exists-p template) (f-filename template) template))

(cl-defmethod prefab-replay-exists-p ((_ prefab-cookiecutter-source) template)
  "Return t if a replay exists for this TEMPLATE for this SOURCE else nil.

TEMPLATE should be of the form returned by `prefab-templates'"
  (f-exists-p
   (f-swap-ext
    (f-join prefab-cookiecutter-replay-dir (f-filename template)) "json")))

(cl-defmethod prefab-default-context ((source prefab-cookiecutter-source)
                                      template replay)
  "Return the default context for TEMPLATE using SOURCE.

This method should return a cons cell whose `car' is the default context and
whose `cdr' is a template resource.  TEMPLATE should be of the form returned
by `prefab-templates' or a remote URI.  A non-nil value for REPLAY indicates
that context from a replay is preferred."
  (let* ((template-path
          (if (f-exists-p template) template
            (progn (message "Downloading template %s" template)
                   (prefab--cookiecutter-download-template template))))
         (ctx-file (format "%s/cookiecutter.json" template-path))
         (truth (if replay "False" "True"))
         (template-name (prefab-template-display-string source template))
         (src (format "from cookiecutter.config import get_user_config
from cookiecutter.generate import generate_context
from cookiecutter.replay import load
import json

def default_context():
    return generate_context(
        context_file='%s',
        default_context=config_dict['default_context'],
    )

config_dict = get_user_config(config_file=%s)
if %s:
    ctx = default_context()
else:
    try:
        ctx = load(config_dict['replay_dir'], '%s')
    except:
        ctx = default_context()
print(json.dumps(dict(ctx['cookiecutter'])))" ctx-file (prefab--cookiecutter-conf) truth template-name)))
    (cons
     (cl-remove-if
      (lambda (alist-entry) (string-match-p "^_.*" (symbol-name (car alist-entry))))
      (prefab--json-from-python src))
     template-path)))

(cl-defmethod prefab-run ((_ prefab-source) template context)
  "Create a project from TEMPLATE and CONTEXT using SOURCE.

CONTEXT is an alist with string keys (template attributes) and values
 (attribute values).  TEMPLATE should be of the form returned by
`prefab-templates'"
  (let* ((extra-args
          (mapconcat
           (lambda (s)
             (format "%s='%s'" (car s) (prefab--escape-quotes-bash (cdr s))))
           context " "))
         (cmd (format "cookiecutter %s %s --no-input --output-dir %s %s"
                      template
                      (if prefab-cookiecutter-config-file
                          (format "--config-file %s"
                                  prefab-cookiecutter-config-file)
                        "")
                      prefab-cookiecutter-output-dir
                      extra-args))
         (response (shell-command-to-string cmd)))
    (prefab--python-check-for-err response "Error running cookiecutter: ")
    (prefab--cookiecutter-created-dir template context)))

;;; Commands

(defun prefab ()
  "Generate a project from a template."
  (interactive)
  (if (executable-find "cookiecutter")
      (let* ((project-src (prefab-cookiecutter-source))
             (templates (prefab-templates project-src))
             (alist (mapcar (lambda (p)
                              (cons (prefab-template-display-string project-src p) p))
                            templates))
             (template-str
              (completing-read "Template: "
                               (or (mapcar #'f-filename templates)
                                   (mapcan #'cdr prefab-default-templates)))))
        (prefab--transient-set-value
         project-src
         (or (alist-get template-str alist nil nil #'string=) template-str)
         prefab-cookiecutter-get-context-from-replay))
    (error prefab--cookiecutter-not-found-err-msg)))

(provide 'prefab)

;;; prefab.el ends here
