;;; mugur.el --- Configurator for QMK compatible keyboards -*- lexical-binding: t -*-

;; Copyright (C) 2020-2021 Mihai Olteanu

;; Author: Mihai Olteanu <mihai_olteanu@fastmail.fm>
;; Package-Version: 2.0
;; Package-Revision: b8ebfd18a579
;; Package-Requires: ((emacs "26.1") (s "1.12.0") (anaphora "1.0.4") (dash "2.18.1") (cl-lib "1.0"))
;; Keywords: multimedia
;; URL: https://github.com/mihaiolteanu/mugur

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; Generate all the keymaps, enums, arrays, make files and everything needed for
;; building a hex file and flashing a qmk powered keyboard.  See the github
;; README for details and usage.

;;; Definitions used in this package:
;; * mugur-key - The most basic entry in the mugur-keymap.  It represents the
;; list of all valid and mugur specific symbols, strings, characters and lists
;; that can be used by the user in building its mugur-keymap.  It is interpreted
;; by the mugur package and used in the end to generate the qmk-keymap.  There
;; need not be a one-to-one correspondence with a mugur-key and a qmk-keycode,
;; as it happens in the case of macros, for example (a single mugur-key is used
;; to fill multiple places in the keymap.c, for example)

;; * mugur-modifier - One of the symbols C, M, G or S which stand for Ctrl, Alt,
;; Win and Shift, respectively and which correspond to the qmk-modifier KC_LCTL,
;; KC_LALT, KC_LGUI and KC_LSHIFT respectively.  A mugur-modifier is a
;; restricted list of mugur-keys.

;; * qmk-modifier - One of the KC_LCTL, KC_LALT, KC_LGUI and KC_LSHIFT
;; qmk-keycodes, respectively, which stand for Ctrl, Alt, Win and Shift,
;; respectively.

;; * qmk-raw-modifier - Similar to qmk-modifier, but without the KC_ part, that
;; is LCTL, LALT, LGUI and LSHIFT

;; * qmk-keycode - The qmk equivalent code for the mugur-key given by the user.
;; Thus, a symbol like 'a, given in the mugur-keymap is transformed into the
;; equivalent qmk-keycode "KC_A".  These are the codes that are written to the
;; qmk-matrix

;; * qmk-layer - The totality of all qmk-keycode's, toghether with a layer
;; code/name that is part of a qmk-matrix.

;; * qmk-matrix - The matrix found in the qmk_firmware's file keymap.c.  It
;; contains all the qmk-keycode's and all the qmk-layer's, as specified by the
;; qmk documentation.  This is what is actually used to decide which keys does
;; what on your keyboard

;; * mugur-layer - The totality of all the mugur-key's, together with a string
;; representing the layer name, that is used to generate the qmk-layer.

;; * mugur-keymap - The totally of all the mugur-layer's.  This is the mugur
;; equivalent for the qmk-matrix.

;;; Code:

(require 's)
(require 'anaphora)
(require 'dash)
(require 'cl-lib)

(defgroup mugur ()
  "ErgoDox EZ keyboard configurator."
  :group 'tools
  :prefix "mugur-")

;; qmk paths, keyboards and keymaps
(defcustom mugur-qmk-path nil
  "Path to the qmk firmware source code."
  :type '(string :tag "path")
  :group 'mugur)

(defcustom mugur-keyboard-name nil
  "The name of your qmk keyboard."
  :type '(string :tag "name")
  :group 'mugur)

(defcustom mugur-layout-name nil
  "The keymap name used in the keymaps matrix.
Check the 'uint16_t keymaps' matrix in the default keymap.c of
your keyboard.  Some have just \"LAYOUT\", others
\"LAYOUT_ergodox\", etc. Adapt accordingly."
  :type '(string :tag "name")
  :group 'mugur)

(defcustom mugur-keymap-name "mugur"
  "The name of qmk keymap generated by mugur."
  :type '(string :tag "name")
  :group 'mugur)

;; Rules
(defcustom mugur-leader-enable "no"
  "Enable the leader key functionality."
  :type '(string :tag "yes/no")
  :group 'mugur)

(defcustom mugur-rgblight-enable "no"
  "Enable the rgblight functionality."
  :type '(string :tag "yes/no")
  :group 'mugur)

;; Configs
(defcustom mugur-tapping-term 180
  "Tapping term, in ms."
  :type '(integer :tag "ms")
  :group 'mugur)

;; Others
(defcustom mugur-user-defined-keys nil
  "User defined keys for long or often used combinations.
A list of lists, where the car of each antry is a symbol (a name
for the new key) and the cadr is any valid mugur-key."
  :type  '(alist :tag "keys")
  :group 'mugur)

(defcustom mugur-ignore-key '-x-
  "The symbol used in the mugur-keymap for an ignored key.
This is transformed into the qmk-keycode KC_NO."
  :type  '(symbol :tag "symbol")
  :group 'mugur)

(defcustom mugur-transparent-key '---
  "The symbol used in the mugur keymap for a transparent key.
This is transformed into the qmk-keycode KC_TRANSPARENT."
  :type  '(symbol :tag "symbol")
  :group 'mugur)


;;;; ---------------------------------------------------------------------------
;;;; This section handles the transformation of individual mugur-keys into their
;;;; qmk-keycode equivalents.
;;;; ---------------------------------------------------------------------------
(defun mugur--keycode (mugur-key)
  "Return the qmk-keycode for the MUGUR-KEY.
MUGUR-KEY is any of the valid symbols, digits, strings or lists
that can appear in a mugur-keymap.  The returned string is what
would one use in the keymaps matrix in the keymap.c qmk file.  If
no transformation is possible, return nil."
  (or (mugur--letter       mugur-key)
      (mugur--digit        mugur-key)
      (mugur--f            mugur-key)
      (mugur--symbol       mugur-key)
      ;; Strings can always be handled as macros, but we can try and interpret
      ;; strings of a single character as just that character.  So instead of
      ;; SEND_STRING(\":\") we can have KC_COLON as a return value.  For special
      ;; characters that do not have a qmk-keycode, just continue and use a
      ;; macro instead.
      (and (stringp mugur-key)
           (= (length mugur-key) 1)
           (or (mugur--keycode (string-to-char mugur-key))
               (mugur--keycode (intern mugur-key))))

      ;; Strings of type "C-x" can be handled as modifiers instead of macros.
      (and (stringp mugur-key)
           (mugur--keycode (intern mugur-key)))
      
      ;; Some mugur-keys, like the symbol >, do not have a qmk-keycode, but they
      ;; can be transformed into their character equivalent (that is, ?\>),
      ;; which does have a qmk-keycode.  Useful for combinations like M->.
      (aand (symbolp mugur-key)
            (symbol-name mugur-key)
            (and (= (length it) 1)
                 (mugur--keycode (string-to-char it))))
      (mugur--oneshot      mugur-key)
      (mugur--layer-toggle mugur-key)
      (mugur--modifier     mugur-key)
      (mugur--modtap       mugur-key)
      (mugur--macro        mugur-key)
      (mugur--user-defined mugur-key)
      (mugur--emacs-fn     mugur-key)
      (and (listp mugur-key)            ;Destructure '(k) and search for 'k
           (= (length mugur-key) 1)
           (mugur--keycode (car mugur-key)))))

(defun mugur--letter (mugur-key)
  "If MUGUR-KEY is a letter, return its qmk-keycode, nil otherwise.
MUGUR-KEY can be a single down-case letter, between 'a' and 'z',
and can be specified either as a symbol or as a string (i.e. 'a
or \"a\" are both transformed to the qmk-keycode KC_A, as a string)."
  (aand (pcase mugur-key
          ((pred symbolp) (symbol-name mugur-key))
          ((pred stringp) mugur-key))
        (and (let ((case-fold-search nil))
               (string-match "^[a-z]$" it))
             it)
        (format "KC_%s" (upcase it))))

(defun mugur--digit (mugur-key)
  "If MUGUR-KEY is a digit, return its qmk-keycode, nil otherwise.
MUGUR-KEY can be a digit between '0' and '9' and can also be
specified as a string (i.e. 5 or \"5\" are both transformed to
the qmk-keycode KC_5, as a string)."
  (aand (pcase mugur-key
          ((pred integerp) (number-to-string mugur-key))
          ((pred stringp ) mugur-key))
        (and (string-match "^[0-9]$" it)
             it)
        (format "KC_%s" it)))

(defun mugur--f (mugur-key)
  "If MUGUR-KEY is an F key, return its qmk-keycode, nil otherwise.
MUGUR-KEY can be a function key between 'f1' and 'f24', uppercase
or lowercase, and can also be specified as a string (i.e. 'f6,
'F6, \"f6\" or \"F6\" are all transformed to the qmk-keycode
KC_F6, as a string)."
  (aand (pcase mugur-key
          ((pred symbolp) (symbol-name mugur-key))
          ((pred stringp) mugur-key))
        (s-match (rx bol
                     (or (seq "f1" digit)
                         (seq "f2" (in "0-4"))
                         (seq "f"  digit))
                     eol)
                 it)
        (format "KC_%s" (upcase (car it)))))

(defun mugur--symbol (mugur-key)
  "If MUGUR-KEY is a valid qmk symbol, return its qmk-keycode, nil otherwise.
MUGUR-KEY can be any of the basic qmk keycodes, like punctuation,
modifiers, media keys, mouse keys, symbols like '?' and '.' and
other special keys that all have a direct qmk-keycode
equivalent."
  (pcase mugur-key
    ;; Punctuation
    ((or 'RET               'ent       ) "KC_ENTER"               ) ;Return (Enter)
    ((or 'escape            'esc       ) "KC_ESCAPE"              ) ;Escape
    ((or 'backspace         'bspace    ) "KC_BSPACE"              ) ;Delete (Backspace)
    ((or 'tab               'tab       ) "KC_TAB"                 ) ;Tab
    ((or 'space             'spc       ) "KC_SPACE"               ) ;Spacebar
    ((or 'minus             ?\-)         "KC_MINUS"               ) ;- and _
    ((or 'equal             ?\=)         "KC_EQUAL"               ) ;= and +
    ((or 'lbracket          ?\[        ) "KC_LBRACKET"            ) ;[ and {
    ((or 'rbracket          ?\]        ) "KC_RBRACKET"            ) ;] and }
    ((or 'bslash            ?\\        ) "KC_BSLASH"              ) ;\ and |
    (    'nonus-hash                     "KC_NONUS_HASH"          ) ;Non-US # and ~
    ((or 'scolon            ?\;        ) "KC_SCOLON"              ) ;; and :
    ((or 'quote             ?\'        ) "KC_QUOTE"               ) ;' and
    ((or 'grave             ?\`        ) "KC_GRAVE"               ) ;` and ~, JIS Zenkaku/Hankaku
    ((or 'comma             ?\,        ) "KC_COMMA"               ) ;, and <
    ((or 'dot               ?\.        ) "KC_DOT"                 ) ;. and >
    ((or 'slash             ?\/        ) "KC_SLASH"               ) ;/ and ?

    ;; Lock keys
    ((or 'capslock          'caps      ) "KC_CAPSLOCK"            ) ;Caps Lock
    ((or 'scrollock         'slck      ) "KC_SCROLLOCK"           ) ;Scroll Lock, Brightness Down (macOS)
    ((or 'numlock           'nlck      ) "KC_NUMLOCK"             ) ;Keypad Num Lock and Clear
    ((or 'locking_caps      'lcap      ) "KC_LOCKING_CAPS"        ) ;Locking Caps Lock
    ((or 'locking_num       'lnum      ) "KC_LOCKING_NUM"         ) ;Locking Num Lock
    ((or 'locking_scroll    'lscr      ) "KC_LOCKING_SCROLL"      ) ;Locking Scroll Lock

    ;; Modifiers
    ((or 'lctl              'C         ) "KC_LCTL"                ) ;Left Control
    ((or 'lalt              'M         ) "KC_LALT"                ) ;Left Alt
    ((or 'lshift            'S         ) "KC_LSFT"                ) ;Left Shift
    ((or 'lgui              'G         ) "KC_LGUI"                ) ;Left GUI (Windows/Command/Meta key)
    ((or 'rctl              'rctrl     ) "KC_RCTRL"               ) ;Right Control
    ((or 'ralt              'ropt      ) "KC_RALT"                ) ;Right Alt (Option/AltGr)
    ((or 'rshift            'rsft      ) "KC_RSHIFT"              ) ;Right Shift
    ((or 'rgui              'rcmd      ) "KC_RGUI"                ) ;Right GUI (Windows/Command/Meta key)

    ;; International
    ((or 'ro                'int1      ) "INT1"                   ) ;JIS \ and _
    ((or 'kana              'int2      ) "INT2"                   ) ;JIS Katakana/Hiragana
    ((or 'jyen              'int3      ) "INT3"                   ) ;JIS ¥ and |
    ((or 'henk              'int4      ) "INT4"                   ) ;JIS Henkan
    ((or 'mhen              'int5      ) "INT5"                   ) ;JIS Muhenkan
    (    'int6                           "INT6"                   ) ;JIS Numpad ,
    (    'int7                           "INT7"                   ) ;International 7
    (    'int8                           "INT8"                   ) ;International 8
    (    'int9                           "INT9"                   ) ;International 9
    ((or 'lang1             'haen      ) "LANG1"                  ) ;Hangul/English
    ((or 'lang2             'hanj      ) "LANG2"                  ) ;Hanja
    (    'lang3                          "LANG3"                  ) ;JIS Katakana
    (    'lang4                          "LANG4"                  ) ;JIS Hiragana
    (    'lang5                          "LANG5"                  ) ;JIS Zenkaku/Hankaku
    (    'lang6                          "LANG6"                  ) ;Language 6
    (    'lang7                          "LANG7"                  ) ;Language 7
    (    'lang8                          "LANG8"                  ) ;Language 8
    (    'lang9                          "LANG9"                  ) ;Language 9

    ;; Commands
    ((or 'pscreen           'pscr      ) "KC_PSCREEN"             ) ;Print Screen
    ((or 'pause             'brk       ) "KC_PAUSE"               ) ;Pause, Brightness Up (macOS)
    ((or 'insert            'ins       ) "KC_INSERT"              ) ;Insert
    (    'home                           "KC_HOME"                ) ;Home
    (    'pgup                           "KC_PGUP"                ) ;Page Up
    ((or 'delete            'del       ) "KC_DELETE"              ) ;Forward Delete
    (    'end                            "KC_END"                 ) ;End
    ((or 'pgdown            'pgdn      ) "KC_PGDOWN"              ) ;Page Down
    (    'right                          "KC_RIGHT"               ) ;Right Arrow
    (    'left                           "KC_LEFT"                ) ;Left Arrow
    (    'down                           "KC_DOWN"                ) ;Down Arrow
    (    'up                             "KC_UP"                  ) ;Up Arrow
    ((or 'application       'app       ) "KC_APPLICATION"         ) ;Application (Windows Context Menu Key)
    (    'power                          "KC_POWER"               ) ;System Power
    ((or 'execute           'exec      ) "KC_EXECUTE"             ) ;Execute
    (    'help                           "KC_HELP"                ) ;Help
    (    'menu                           "KC_MENU"                ) ;Menu
    ((or 'select            'slct      ) "KC_SELECT"              ) ;Select
    (    'stop                           "KC_STOP"                ) ;Stop
    ((or 'again             'agin      ) "KC_AGAIN"               ) ;Again
    (    'undo                           "KC_UNDO"                ) ;Undo
    (    'cut                            "KC_CUT"                 ) ;Cut
    (    'copy                           "KC_COPY"                ) ;Copy
    ((or 'paste             'pste      ) "KC_PASTE"               ) ;Paste
    (    'find                           "KC_FIND"                ) ;Find
    (    '_mute                          "KC__MUTE"               ) ;Mute
    (    '_volup                         "KC__VOLUP"              ) ;Volume Up
    (    '_voldown                       "KC__VOLDOWN"            ) ;Volume Down
    ((or 'alt_erase         'eras      ) "KC_ALT_erase"           ) ;Aternate Erase
    (    'sysreq                         "KC_SYSREQ"              ) ;SysReq/Attention
    (    'cancel                         "KC_CANCEL"              ) ;Cancel
    ((or 'clear             'clr       ) "KC_CLEAR"               ) ;Clear
    (    'prior                          "KC_PRIOR"               ) ;Prior
    (    'return                         "KC_RETURN"              ) ;Return
    (    'separator                      "KC_SEPARATOR"           ) ;Separator
    (    'out                            "KC_OUT"                 ) ;Out
    (    'oper                           "KC_OPER"                ) ;Open
    (    'clear_again                    "KC_CLEAR_again"         ) ;Clear/Again
    (    'crsel                          "KC_CRSEL"               ) ;CrSel/Props
    (    'exsel                          "KC_EXSEL"               ) ;ExSel

    ;; Media Keys
    ((or 'system-power      'pwr       ) "KC_SYSTEM_POWER"        ) ;System Power Down
    ((or 'system-sleep      'slep      ) "KC_SYSTEM_SLEEP"        ) ;System Sleep
    ((or 'system-wake       'wake      ) "KC_SYSTEM_WAKE"         ) ;System Wake
    ((or 'audio-mute        'mute      ) "KC_AUDIO_MUTE"          ) ;Mute
    ((or 'vol-up            'volu      ) "KC_AUDIO_VOL_UP"        ) ;Volume Up
    ((or 'vol-down          'vold      ) "KC_AUDIO_VOL_DOWN"      ) ;Volume Down
    ((or 'next-track        'mnxt      ) "KC_MEDIA_NEXT_TRACK"    ) ;Next Track
    ((or 'prev-track        'mprv      ) "KC_MEDIA_PREV_TRACK"    ) ;Previous Track
    ((or 'media-stop        'mstp      ) "KC_MEDIA_STOP"          ) ;Stop Track
    ((or 'media-play-pause  'mply      ) "KC_MEDIA_PLAY_PAUSE"    ) ;Play/Pause Track
    ((or 'media-select      'msel      ) "KC_MEDIA_SELECT"        ) ;Launch Media Player
    ((or 'media-eject       'ejct      ) "KC_MEDIA_EJECT"         ) ;Eject
    (    'mail                           "KC_MAIL"                ) ;Launch Mail
    ((or 'calculator        'calc      ) "KC_CALCULATOR"          ) ;Launch Calculator
    ((or 'my-computer       'mycm      ) "KC_MY_COMPUTER"         ) ;Launch My Computer
    ((or 'www-search        'wsch      ) "KC_WWW_SEARCH"          ) ;Browser Search
    ((or 'www-home          'whom      ) "KC_WWW_HOME"            ) ;Browser Home
    ((or 'www-back          'wbak      ) "KC_WWW_BACK"            ) ;Browser Back
    ((or 'www-forward       'wfwd      ) "KC_WWW_FORWARD"         ) ;Browser Forward
    ((or 'www-stop          'wstp      ) "KC_WWW_STOP"            ) ;Browser Stop
    ((or 'www-refresh       'wref      ) "KC_WWW_REFRESH"         ) ;Browser Refresh
    ((or 'www-favorites     'wfav      ) "KC_WWW_FAVORITES"       ) ;Browser Favorites
    ((or 'fast-forward      'mffd      ) "KC_MEDIA_FAST_FORWARD"  ) ;Next Track
    ((or 'rewind            'mrwd      ) "KC_MEDIA_REWIND"        ) ;Previous Track
    ((or 'brigthness-up     'briu      ) "KC_BRIGTHNESS_UP"       ) ;Brightness Up
    ((or 'brigthness-down   'brid      ) "KC_BRIGTHNESS_DOWN"     ) ;Brightness Down

    ;; Number Pad
    ((or 'kp_slash          'psls      ) "KP_SLASH"               ) ;Keypad /
    ((or 'kp_asterisk       'past      ) "KP_ASTERISK"            ) ;Keypad *
    ((or 'kp_minus          'pmns      ) "KP_MINUS"               ) ;Keypad -
    ((or 'kp_plus           'ppls      ) "KP_PLUS"                ) ;Keypad +
    ((or 'kp_enter          'pent      ) "KP_ENTER"               ) ;Enter
    ((or 'kp_1              'p1        ) "KP_1"                   ) ;Keypad 1 and End
    ((or 'kp_2              'p2        ) "KP_2"                   ) ;Keypad 2 and Down Arrow
    ((or 'kp_3              'p3        ) "KP_3"                   ) ;Keypad 3 and Page Down
    ((or 'kp_4              'p4        ) "KP_4"                   ) ;Keypad 4 and Left Arrow
    ((or 'kp_5              'p5        ) "KP_5"                   ) ;Keypad 5
    ((or 'kp_6              'p6        ) "KP_6"                   ) ;Keypad 6 and Right Arrow
    ((or 'kp_7              'p7        ) "KP_7"                   ) ;Keypad 7 and Home
    ((or 'kp_8              'p8        ) "KP_8"                   ) ;Keypad 8 and Up Arrow
    ((or 'kp_9              'p9        ) "KP_9"                   ) ;Keypad 9 and Page Up
    ((or 'kp_0              'p0        ) "KP_0"                   ) ;Keypad 0 and Insert
    ((or 'kp_dot            'pdot      ) "KP_DOT"                 ) ;Keypad . and Delete
    ((or 'kp_equal          'peql      ) "KP_EQUAL"               ) ;Keypad =
    ((or 'kp_comma          'pcmm      ) "KP_COMMA"               ) ;Keypad ,
    (    'kp_equal_as400                 "KP_EQUAL_AS400"         ) ;Keypad = on AS/400 keyboards

    ;; Special Keys
    ((pred (eq mugur-ignore-key       )) "KC_NO"                 ) ;Ignore this key (NOOP)
    ((pred (eq mugur-transparent-key  )) "KC_TRNS"               ) ;Use the next lowest non-transparent key
    
    ;; Quantum Keycodes
    (    'reset                          "RESET"                  ) ;Put the keyboard into bootloader mode for flashing
    (    'debug                          "DEBUG"                  ) ;Toggle debug mode
    ((or 'eeprom-reset      'eep_rst   ) "EEPROM_RESET"           ) ;Reinitializes the keyboard’s EEPROM (persistent memory)
        
    ;; Dynamic Macros
    ((or 'dyn_rec_start1    'dm_rec1   ) "KC_DYN_REC_START1"      ) ;Start recording Macro 1
    ((or 'dyn_rec_start2    'dm_rec2   ) "KC_DYN_REC_START1"      ) ;Start recording Macro 2
    ((or 'dyn_macro_play1   'dm_ply1   ) "KC_DYN_MACRO_PLAY1"     ) ;Replay Macro 1
    ((or 'dyn_macro_play2   'dm_ply2   ) "KC_DYN_MACRO_PLAY1"     ) ;Replay Macro 2
    ((or 'dyn_rec_stop      'dm_rstp   ) "KC_DYN_REC_STOP"        ) ;Finish the macro that is currently being recorded.

    ;; Grave Escape
    ((or 'gesc              'grave_esc ) "KC_GESC"                ) ;Escape when pressed, ` when Shift or GUI are held

    ;; Leader Key
    (    'lead                           "KC_LEADER"              ) ;The Leader Key

    ;; Mouse Keys
    ((or 'ms_up             'ms_u      ) "KC_MS_UP"               ) ;Move cursor up
    ((or 'ms_down           'ms_d      ) "KC_MS_DOWN"             ) ;Move cursor down
    ((or 'ms_left           'ms_l      ) "KC_MS_LEFT"             ) ;Move cursor left
    ((or 'ms_right          'ms_r      ) "KC_MS_RIGHT"            ) ;Move cursor right
    ((or 'ms_btn1           'btn1      ) "KC_MS_BTN1"             ) ;Press button 1
    ((or 'ms_btn2           'btn2      ) "KC_MS_BTN2"             ) ;Press button 2
    ((or 'ms_btn3           'btn3      ) "KC_MS_BTN3"             ) ;Press button 3
    ((or 'ms_btn4           'btn4      ) "KC_MS_BTN4"             ) ;Press button 4
    ((or 'ms_btn5           'btn5      ) "KC_MS_BTN5"             ) ;Press button 5
    ((or 'ms_btn6           'btn6      ) "KC_MS_BTN6"             ) ;Press button 6
    ((or 'ms_btn7           'btn7      ) "KC_MS_BTN7"             ) ;Press button 7
    ((or 'ms_btn8           'btn8      ) "KC_MS_BTN8"             ) ;Press button 8
    ((or 'ms_wh_up          'wh_u      ) "KC_MS_WH_UP"            ) ;Move wheel up
    ((or 'ms_wh_down        'wh_d      ) "KC_MS_WH_DOWN"          ) ;Move wheel down
    ((or 'ms_wh_left        'wh_l      ) "KC_MS_WH_LEFT"          ) ;Move wheel left
    ((or 'ms_wh_right       'wh_r      ) "KC_MS_WH_RIGHT"         ) ;Move wheel right
    ((or 'ms_accel0         'acl0      ) "KC_MS_ACCEL0"           ) ;Set speed to 0
    ((or 'ms_accel1         'acl1      ) "KC_MS_ACCEL1"           ) ;Set speed to 1
    ((or 'ms_accel2         'acl2      ) "KC_MS_ACCEL2"           ) ;Set speed to 2
        
    ;; Space Cadet
    (    'lspo                           "KC_LSPO"                ) ;Left Shift when held, ( when tapped
    (    'rspc                           "KC_RSPC"                ) ;Right Shift when held, ) when tapped
    (    'lcpo                           "KC_LCPO"                ) ;Left Control when held, ( when tapped
    (    'rcpc                           "KC_RCPC"                ) ;Right Control when held, ) when tapped
    (    'lapo                           "KC_LAPO"                ) ;Left Alt when held, ( when tapped
    (    'rapc                           "KC_RAPC"                ) ;Right Alt when held, ) when tapped
    (    'sftent                         "KC_SFTENT"              ) ;Right Shift when held, Enter when tapped

    ;; US ANSI Shifted Symbols
    ((or 'tilde              ?\~       ) "KC_TILDE"               ) ;~
    ((or 'exclaim            ?\!       ) "KC_EXCLAIM"             ) ;!
    ((or 'at                 ?\@       ) "KC_AT"                  ) ;@
    ((or 'hash               ?\#       ) "KC_HASH"                ) ;#
    ((or 'dollar             ?\$       ) "KC_DOLLAR"              ) ;$
    ((or 'percent            ?\%       ) "KC_PERCENT"             ) ;%
    ((or 'circumflex         ?\^       ) "KC_CIRCUMFLEX"          ) ;^
    ((or 'ampersand          ?\&       ) "KC_AMPERSAND"           ) ;&
    ((or 'asterisk           ?\*       ) "KC_ASTERISK"            ) ;*
    ((or 'lparen             ?\(       ) "KC_LEFT_PAREN"          ) ;(
    ((or 'rparen             ?\)       ) "KC_RIGHT_PAREN"         ) ;)
    ((or 'under              ?\_       ) "KC_UNDERSCORE"          ) ;_
    ((or 'plus               ?\+       ) "KC_PLUS"                ) ;+
    ((or 'left_curly         ?\{       ) "KC_LEFT_CURLY_BRACE"    ) ;{
    ((or 'right_curly        ?\}       ) "KC_RIGHT_CURLY_BRACE"   ) ;}
    ((or 'pipe               ?\|       ) "KC_PIPE"                ) ;|
    ((or 'colon              ?\:       ) "KC_COLON"               ) ;:
    ((or 'double_quote       ?\"       ) "KC_DOUBLE_QUOTE"        ) ;"
    ((or 'left_angle         ?\<       ) "KC_LEFT_ANGLE_BRACKET"  ) ;<
    ((or 'right_angle        ?\>       ) "KC_RIGHT_ANGLE_BRACKET" ) ;>
    ((or 'question           ?\?       ) "KC_QUESTION"            ) ;?

    ;; RGB Ligthing
    (    'rgb_tog                        "RGB_TOG"                ) ;Toggle RGB lighting on or off
    ((or 'rgb_mode_forward  'rgb_mod   ) "RGB_MOD"                ) ;Cycle through modes, reverse direction when Shift is held
    ((or 'rgb_mode_reverse  'rgb_mod   ) "RGB_RMOD"               ) ;Cycle through modes in reverse, forward direction when Shift is held
    (    'rgb_hui                        "RGB_HUI"                ) ;Increase hue, decrease hue when Shift is held
    (    'rgb_hud                        "RGB_HUD"                ) ;Decrease hue, increase hue when Shift is held
    (    'rgb_sai                        "RGB_SAI"                ) ;Increase saturation, decrease saturation when Shift is held
    (    'rgb_sad                        "RGB_SAD"                ) ;Decrease saturation, increase saturation when Shift is held
    (    'rgb_vai                        "RGB_VAI"                ) ;Increase value (brightness), decrease value when Shift is held
    (    'rgb_vad                        "RGB_VAD"                ) ;Decrease value (brightness), increase value when Shift is held
    ((or 'rgb_mode_plain    'rgb_m_p   ) "RGB_MOIDE_PLAIN"        ) ;Static (no animation) mode
    ((or 'rgb_mode_breathe  'rgb_m_b   ) "RGB_MODE_BREATHE"       ) ;Breathing animation mode
    ((or 'rgb_mode_rainbow  'rgb_m_r   ) "RGB_MODE_RAINBOW"       ) ;Rainbow animation mode
    ((or 'rgb_mode_swirl    'rgb_m_sw  ) "RGB_MODE_SWIRL"         ) ;Swirl animation mode
    ((or 'rgb_mode_snake    'rgb_m_sn  ) "RGB_MODE_SNAKE"         ) ;Snake animation mode
    ((or 'rgb_mode_knight   'rgb_m_k   ) "RGB_MODE_KNIGHT"        ) ;"Knight Rider" animation mode
    ((or 'rgb_mode_xmas     'rgb_m_x   ) "RGB_MODE_XMAS"          ) ;Christmas animation mode
    ((or 'rgb_mode_gradient 'rgb_m_g   ) "RGB_MODE_GRADIENT"      ) ;Static gradient animation mode
    ((or 'rgb_mode_rgbtest  'rgb_m_t   ) "RGB_MODE_RGBTEST"       ) ;Red, Green, Blue test animation mode

    ;; Key Lock
    (    'lock                           "KC_LOCK"              ))) ;Hold down the next key pressed, until the key is pressed again

(defun mugur--modifier (mugur-key)
  "Handle a modifier MUGUR-KEY.
MUGUR-KEY is a symbol containing mugur-modifiers separated by
dashes and a mugur-key after the last dash.  If all the elements
are valid, return the qmk-keycode for the modifier key
functionality, otherwise return nil.

For example, 'C-M-a means hold down Ctrl, Meta and send 'a', and
is transformed into the qmk-key \"LCTL(LALT(KC_A))\"."
  (aand (mugur--qmk-dashed-modifier mugur-key)
        (--reduce-r (format "%s(%s)" it acc) it)))

(defun mugur--modtap (mugur-key)
  "If MUGUR-KEY is a mugur-modtap, return its qmk-keycode, nil otherwise.
MUGUR-KEY is a list containing mugur-modifiers and a mugur-key as
its last element.  If all the elements are valid, return the
qmk-code for the mod-tap functionality, otherwise return nil.
For example, '(C M a) becomes \"MT(MOD_LCTL | MOD_LALT, KC_A)\""
  (aand (listp mugur-key)
        (mugur--qmk-raw-modifiers (butlast mugur-key))
        (--map (format "MOD_%s" it) it)
        (format "MT(%s, %s)"
                (--reduce-r (format "%s | %s" it acc) it)
                (mugur--keycode (car (last mugur-key))))
        ;; Invalid modtap if there is no valid mugur-key on the last position.
        (and (not (s-match "nil" it))
             it)))

(defun mugur--macro (mugur-key)
  "If MUGUR-KEY is a mugur-macro, return its qmk-keycode, nil otherwise.
MUGUR-KEY is either a string or a list of more than one element,
where each element is a string or a mugur-key.  If all the
elements are valid, return the SEND_STRING qmk-keycode for the
macro functionality, nil otherwise.  For example, '(C-u a)
becomes \"SEND_STRING(SS_LCTL(SS_TAP(X_U)) SS_TAP(X_A))\""
  (or
   ;;If this is either a list of more than one element
   (aand (and (listp mugur-key)
              (> (length mugur-key) 1))
         ;;..where each element can be transformed into a valid qmk SEND_STRING
         ;;equivalent,
         (mapcar (lambda (k)
              ;; that is,
              (or               
               ;; - A mugur-dashed-modifier
               ;; For example, C-M-a becomes SS_LCTL(SS_LALT(SS_TAP(X_A)))
               (aand (mugur--qmk-dashed-modifier k)
                     `(;; Add SS to all the modifiers,
                       ,@(--map (format "SS_%s" it) (butlast it))
                       ;; Add SS_TAP around the qmk-keycode and replace KC_ with
                       ;; X_, as required by the qmk macro definition
                       ,(format "SS_TAP(%s)"
                                (s-replace "KC_" "X_" (car (last it)))))
                     ;; Add parantheses to all the elements, such that the last
                     ;; element (the qmk-keycode) is the innermost element
                     (--reduce-r (format "%s(%s)" it acc) it))

               ;; - A valid mugur-keycode that is not a macro (a simple string).
               ;; "enter", which becomes KC_ENTER is valid, but "my enter",
               ;; which becomes SEND_STRING("my enter") is not.  This is because
               ;; keycodes need to be wrapped around SS_TAPs, but strings as
               ;; such are wrapped around quotes.
               (aand (mugur--keycode k)
                     (and (not (s-match "SEND_STRING" it))
                          it)
                     (format "SS_TAP(%s)"
                             (s-replace "KC_" "X_" it)))
               
               ;; - Or a simple string which is wrapped in quotes and sent as
               ;; such (the "my enter" from above, for example).
               (and (stringp k)
                    (format "\"%s\"" k))))
            mugur-key)
         ;; Then, if all the elements have had a SEND_STRING equivalent, return
         ;; the SEND_STRING macro.
         (and (not (member nil it))
              (format "SEND_STRING(%s)"
                      (mapconcat #'identity it " "))))
   
   ;;...or a simple string
   (and (stringp mugur-key)
        (format "SEND_STRING(\"%s\")" mugur-key))))

(defun mugur--user-defined (mugur-key)
  "Handle the user-defined MUGUR-KEYs.
If MUGUR-KEY is any of the mugur-keys defined in
`mugur-user-defined-keys', return its qmk-keycode, otherwise
return nil."
  (aand (alist-get mugur-key mugur-user-defined-keys)
        (mugur--keycode it)))

(defun mugur--emacs-fn (mugur-key)
  "Handle MUGUR-KEYs which are keybound EMACS functions.
If MUGUR-KEY is an EMACS function which has a keybinding, handle
that keybinding string as a normal mugur-key and transform it
into a qmk-keycode, otherwise return nil."
  (aand
   ;; Get the function keybinding, if any.
   (where-is-internal mugur-key nil t)
   (key-description it)
   (or
    ;; Some keybindings are given with angle brackets, <C-tab> for example.
    (aand (s-match (rx bol "<" (1+ anything) ">" eol)
                   it)
          (car it)
          ;; Remove the angle brackets and treat the resulting string like a
          ;; mugur-key.
          (s-replace-regexp "^<" "" it)           
          (s-replace-regexp ">$" "" it)
          (mugur--keycode it))
    ;; Other keybindings have more than one key, "C-x * RET" for example.
    (aand (s-split " " it)
          ;; s-split always return the string, even if no matches were found.
          ;; If there were spaces, and a split was done, the resulting list of
          ;; strings can be treated as a mugur-key.  If there were no spaces,
          ;; and no split was done, that can also be treated as a mugur-key.
          (mugur--keycode it)))))

(defun mugur--oneshot (mugur-key)
  "Handle the oneshot MUGUR-KEYs.
MUGUR-KEY is a list that starts either with OSM or with OSL and
is followed by a mugur-modifier or by a mugur-key, respectively."
  (pcase mugur-key
    ;; One Shot Modifier
    ((and `(,'OSM ,m)
          (guard (mugur--qmk-raw-modifiers (list m))))
     (format "OSM(MOD_%s)" (car (mugur--qmk-raw-modifiers (list m)))))
    ;; One Shot Layer
    (`(,'OSL ,x) (format "OSL(%s)" x))))

(defun mugur--layer-toggle (mugur-key)
  "Handle the layer toggle MUGUR-KEYs.
MUGUR-KEY can be a list that starts with any of the DF, MO, OSL,
TG, TT, LL or LT, followed by a mugur-modifier, mugur-layer or
mugur-key, depending on the case (see the official qmk
documentation for details).  If all the items are valid, return
the qmk-keycode, nil otherwise.  For example '(lt \"mylayer\" x)
becomes \"LT(MYLAYER, KC_X)\"."
  (aand (pcase mugur-key
          (`(,(or 'DF 'MO 'OSL 'TG 'TT) ,layer)
           (format "%s(%s)" (car mugur-key) layer))
          (`(LM ,layer ,mod)
           (aand (mugur--qmk-raw-modifiers (list mod))
                 (format "LM(%s, %s)"
                         layer
                         (format "MOD_%s" (car it)))))
          (`(LT ,layer ,kc)
           (aand (mugur--keycode kc)
                 (format "LT(%s, %s)" layer it))))
        (upcase it)))

(defun mugur--qmk-raw-modifiers (mugur-modifiers)
  "Transform the list of MUGUR-MODIFIERS into qmk-raw-modifiers.
MUGUR-MODIFIERS can be a list of mugur-modifiers, given either as
symbols or as strings.  If *all* the items in the list are valid
mugur-modifiers, return a list of qmk-raw-modifiers, as strings,
otherwise return nil."
  (aand (listp mugur-modifiers)
        (mapcar #'mugur--to-string mugur-modifiers)
        (and (not (-difference it '("C" "M" "S" "G")))
             it)
        (mapcar #'mugur--keycode it)
        (and (not (-some #'null it))
             it)
        (--map (s-replace "KC_" "" it)
               it)))

(defun mugur--qmk-dashed-modifier (mugur-dashed-modifier)
  "Transform the list of MUGUR-DASHED-MODIFIER into a qmk-raw-modifier.
Similar to `mugur--qmk-raw-modifiers', but the modifiers are
given in a single overall symbol, separated by dashes, where the
item after the last dash is a valid `mugur-keycode', for example
'C-M-a.  If not all items in the MUGUR-DASHED-MODIFIER can be
successfully transformed, return nil"
  (aand (symbolp mugur-dashed-modifier)
        (symbol-name mugur-dashed-modifier)
        (and (s-match "-" it)
             it)
        (s-split "-" it)
        ;; Splice the elements into the resulting list.
        `(;; All the items before the last dash should be `mugur-modifier's
          ,@(mugur--qmk-raw-modifiers (butlast it))
          ;; The last item should be a `mugur-key'. To avoid transforming
          ;; multi-character keys, like "space", into a qmk-macro instead of
          ;; KC_SPACE, intern it.
          ,(mugur--keycode (intern (car (last it)))))
        (and (not (-some #'null it))
             it)))

(defun mugur--to-string (mugur-key)
  "Force a transformation of MUGUR-KEY into string.
MUGUR-KEY is either a number, string, symbol, character or a list
of any of the previous.  In that case, return its string
equivalent, otherwise return nil."
  (pcase mugur-key
    ((pred numberp)    (number-to-string mugur-key))
    ((pred stringp)    mugur-key)
    ((pred symbolp)    (symbol-name mugur-key))
    ((pred characterp) (char-to-string mugur-key))    
    (`(,v) (mugur--to-string v))))


;;;; ---------------------------------------------------------------------------
;;;; This section handles the transformation of the mugur-keymap into its
;;;; qmk-keymap equivalent and the generation of the qmk C code (keymap.c,
;;;; config.h and rules.mk files).
;;;; ---------------------------------------------------------------------------
(defun mugur-mugur (mugur-keymap)
  "Use the MUGUR-KEMAP to generate the entire equivalent qmk-keymap.
MUGUR-KEYMAP is a list of mugur-layers."
  (mugur--write-rules-mk)
  (mugur--write-config-h)
  (mugur--write-keymap-c mugur-keymap))

(defun mugur--write-rules-mk ()
  "Generate the qmk rules.mk file.
Use the user supplied custom variables to set up all the rules.
The output of the file is in the generated qmk-keymaps folder, as
required by the qmk rules."
  (mugur--write-file "rules.mk"
   (format
    "FORCE_NKRO      = yes
     LEADER_ENABLE   = %s
     RGBLIGHT_ENABLE = %s"
    mugur-leader-enable
    mugur-rgblight-enable)))

(defun mugur--write-config-h ()
  "Generate the qmk config.h file.
Use the user supplied custom variables to set up all the rules.
The output of the file is in the generated qmk-keymaps folder, as
required by the qmk rules."
  (mugur--write-file "config.h"
   (format
    "#undef TAPPING_TERM
     #define TAPPING_TERM %s
     #define FORCE_NKRO
     #undef RGBLIGHT_ANIMATIONS"
    mugur-tapping-term)))

(defun mugur--write-keymap-c (mugur-keymap)
  "Generate the qmk keymap.c file from the MUGUR-KEYMAP.
This is the main qmk file that contains the qmk-matrix and all
the qmk-keycodes.  The output of the file is in the generated
qmk-keymaps folder, as required by the qmk rules."
  (let* ((qmk-keymap (mugur--transform-keymap mugur-keymap))
         (qmk-layers (mugur--qmk-keymap-layers qmk-keymap))
         (qmk-macros (mugur--qmk-keymap-macros qmk-keymap)))

    (mugur--write-file "keymap.c"
     (format
      "#include QMK_KEYBOARD_H
      #include \"version.h\"
      
      /* Layer names */
      enum layer_codes {
          %s
      };
      
      /* Macro names */
      enum custom_keycodes {
          EPRM = SAFE_RANGE,
          %s
      };
      
      /* All the qmk-layers, with layer names and keys */
      const uint16_t PROGMEM keymaps[][MATRIX_ROWS][MATRIX_COLS] = {
          %s
      };
      
      /* Macros processing */
      bool process_record_user(uint16_t keycode, keyrecord_t *record) {
          if (record->event.pressed) {
              switch (keycode) {
                 case EPRM:
                 eeconfig_init();
                 return false;
              %s
      	      }
          }
         return true;
      }"
      ;; Layer names
      (--reduce-r
       (format "%s, \n       %s" acc it)
       (reverse (--map (upcase (car it)) qmk-layers)))

      ;; Macro names
      (--reduce-r
        ;; macro1, macro2, etc..
       (format "%s, \n       %s" acc it)
       (mapcar #'car qmk-macros))

      ;; All the qmk-layers
      (--reduce-r
       ;; layer1, layer2, etc..
       (format "%s, \n\n       %s" it acc)
       (mapcar (lambda (k)
            ;; Layer name, layout name, layer keys
            (format "[%s] = %s(%s)"
                    (upcase (car k))
                    (or mugur-layout-name
                        (error "The mugur-layout-name variable is not set"))
                    ;; Add comma between the keys and transform
                    ;; the keys into a big string.
                    (--reduce-r (format "%s, %s" it acc) (cdr k))))
          qmk-layers))

      ;; Macros
      (--reduce-r
       (format "%s\n         %s" it acc)
       (--map
        ;; case macro_name: SEND_STRING(...); return false;
        (format "case %s:\n %s;\n return false;"
                (car it) (cadr it))
        qmk-macros))))))

(defun mugur--write-file (file contents)
  "Write CONTENTS to FILE in the correct qmk_firmware location.
Use the `mugur-qmk-path' and `mugur-keymap-name' to figure out
where to write the FILE.  Create all the paths and the FILE if
they don't already exist."
  (with-temp-file
      (aand (concat (file-name-as-directory mugur-qmk-path)
                    (file-name-as-directory "keyboards")
                    (file-name-as-directory mugur-keyboard-name)
                    (file-name-as-directory "keymaps")
                    (file-name-as-directory mugur-keymap-name))
            (and (or (file-directory-p it)
                     (make-directory it))
                 it)
            (concat it file))
    (insert contents)))

(defun mugur--transform-keymap (mugur-keymap)
  "Transform the MUGUR-KEYMAP into its qmk-keymap equivalent.
Every macro string is replaced with a macro name, created in
place (MACRO_1, MACRO_2, etc.), and all the macros, toghether
with their names are added to one list.

If all the mugur-keys have qmk-keycodes and all the mugur-layers
are valid, return a list of two elements, where the first element
is a list of all the macros discovered together with their newly
created names, and the second element is a list of all the
layers, with all the mugur-keys transformed into their
qmk-keycodes equivalents.  Both lists are tagged with their
respective contents, macros and layers, respectively.  Return nil
otherwise."
  (let* ((macros-list)
         (macro-count 0)
         (qmk-keymap
          ;;For each layer in the mugur-keymap
          (mapcar (lambda (mugur-layer)
               (cons
                ;; Transform the mugur-layer's name, if its valid
                (or (mugur--to-string (car mugur-layer))
                    (error (format "Invalid mugur layer name, %s" (car mugur-layer))))
                ;; ...and all layer's mugur-keys
                (mapcar (lambda (mugur-key)
                     (let ((qmk-keycode (mugur--keycode mugur-key)))
                       (pcase qmk-keycode
                         ;;No such qmk key
                         ((pred null)
                          (error (format "Invalid mugur key, %s" mugur-key)))
                         
                         ;; If it's a macro, add the macro name together with
                         ;; its value to the list of macros. The resulting
                         ;; qmk-keycode for the mugur-macro is the newly created
                         ;; macro name and not the SEND_STRING string.
                         ((rx "SEND_STRING" anything)
                          (let ((macro-name (format "MACRO_%s" (cl-incf macro-count))))
                            (push (list macro-name qmk-keycode)
                                  macros-list)
                            macro-name))
                         
                         ;; Any other qmk-key we leave as it was returned from
                         ;; `mugur--keycode'.
                         ((pred stringp) qmk-keycode))))
                   ;; The first item of the mugur-layer is the layer's name,
                   ;; handled above.
                   (cdr mugur-layer))))
             mugur-keymap)))
    ;; Build the returning list of macros and layers, tagging them.
    `((macros ,(reverse macros-list))
      (layers ,qmk-keymap))))

(defun mugur--qmk-keymap-layers (qmk-keymap)
  "Extract the qmk-layers from a QMK-KEYMAP."
  (aand (--find (eq (car it) 'layers)
                qmk-keymap)
        (cadr it)))

(defun mugur--qmk-keymap-macros (qmk-keymap)
  "Extract the qmk-macros from a QMK-KEYMAP."
  (aand (--find (eq (car it) 'macros)
                qmk-keymap)
        (cadr it)))


;;;; ---------------------------------------------------------------------------
;;;; This section handles all the mugur defined tests using the ert framework.
;;;; ---------------------------------------------------------------------------
(defmacro mugur--test (name testp tests)
  "Define NAME as an ert test for all the mugur-keycodes in TESTS.
TESTP is the predicate used for the should clauses.  TESTS is a
list of tests, where for each tests, its first element is the
mugur-key and the second element is the expectd qmk-keycode for
that mugur-key."
  `(ert-deftest ,name ()
     ,@(mapcar (lambda (test)
            `(should (,testp (mugur--keycode ',(car test))
                              ,(cadr test))))
          tests)))

(mugur--test mugur-valid-keycodes string=
 ((a               "KC_A")              ;Send A
  ("a"             "KC_A")              ;Send A
  ((a)             "KC_A")              ;Send A
  (("a")           "KC_A")              ;Send A
  (c               "KC_C")
  ("c"             "KC_C")
  ("C"             "KC_LCTL")
  (C               "KC_LCTL")           ;Ctrl Modifier
  (M               "KC_LALT")           ;Meta Modifier 
  (G               "KC_LGUI")           ;Gui/Win/Super Modifier
  (S               "KC_LSFT")           ;Shift Modifier
  (5               "KC_5")
  (?\>             "KC_RIGHT_ANGLE_BRACKET")
  (-x-             "KC_NO")
  (---             "KC_TRNS")
  (C-a             "LCTL(KC_A)")        ;Hold Ctrl and send A
  ((C-a)           "LCTL(KC_A)")        ;Equivalent
  ("C-a"           "LCTL(KC_A)")        ;Equivalent
  (("C-a")         "LCTL(KC_A)")        ;Equivalent
  (C-M-a           "LCTL(LALT(KC_A))")  ;Hold Ctrl and Meta and sent A
  ((C M a)         "MT(MOD_LCTL | MOD_LALT, KC_A)")
  ((C-u "this" a)  "SEND_STRING(SS_LCTL(SS_TAP(X_U)) \"this\" SS_TAP(X_A))")
  ("a flip-flop"   "SEND_STRING(\"a flip-flop\")")
  ;; Strings of valid keycodes should result in those keycodes.
  (("C-x" "8" "RET") "SEND_STRING(SS_TAP(LCTL(X_X)) SS_TAP(X_8) SS_TAP(X_ENTER))")
  (("C-x"  8   RET)  "SEND_STRING(SS_TAP(LCTL(X_X)) SS_TAP(X_8) SS_TAP(X_ENTER))")

  ;;; Keybound emacs functions.
  ;; Multikeys keybindings (C-x 8 RET)
  (insert-char     "SEND_STRING(SS_TAP(LCTL(X_X)) SS_TAP(X_8) SS_TAP(X_ENTER))")  
  (query-replace   "LALT(KC_PERCENT)") ;Normal/simple keybindings (M-%)  
  (kill-region     "LSFT(KC_DELETE)")  ;Keybindings with angle brackets (<S-delete>)
  ((kill-region)   "LSFT(KC_DELETE)")  ;Same, but given as a list
  
  ;;; Layers switching and toggle.
  ((OSL mylayer)   "OSL(mylayer)")
  ((OSM C)         "OSM(MOD_LCTL)")

  ((DF base)       "DF(BASE)")
  ((DF "base")     "DF(BASE)")
  ((LM "base" C)   "LM(BASE, MOD_LCTL)")
  ((LT "base" x)   "LT(BASE, KC_X)")))

(mugur--test mugur-invalid-keycodes eq
 ((aa              nil)
  (C-aa            nil)
  ((C M aa)        nil)
  ((C Mm aa)       nil)
  ((C-u "this" aa) nil)
  ((LM "base" c)   nil)
  ((OSM c)         nil)
  ;; One Shot Keys can have only one modifier.
  ((OSM C M)       nil)))

(ert-deftest mugur-valid-keymaps ()
  "Test valid mugur-keymaps."
  (should
   (equal (mugur--transform-keymap
           '(("base" a b c)))
          '((macros nil)
            (layers (("base" "KC_A" "KC_B" "KC_C"))))))
  (should
   (equal (mugur--transform-keymap
           '(("base" a b c)
             ("numbers" 1 2 "three")))
          '((macros (("MACRO_1" "SEND_STRING(\"three\")")))
            (layers (("base" "KC_A" "KC_B" "KC_C")
                     ("numbers" "KC_1" "KC_2" "MACRO_1")))))))

(ert-deftest mugur-invalid-keymaps ()
  "Test invalid mugur-keymaps."
  (should-error
   (mugur--transform-keymap
    ;; Invalid mugur-key, xx.
    '(("base" xx)))))

(defun mugur-test ()
  "Helper function to run all the mugur-tests."
  (interactive)
  (ert t))

(provide 'mugur)

;;; mugur.el ends here
