;;; ibrowse-bookmark-chromium.el --- Interact with your browser -*- lexical-binding: t -*-

;; Copyright © 2022 Nicolas Graves <ngraves@ngraves.fr>
;; Copyright © 2021 BlueBoxWare

;; Author: Nicolas Graves <ngraves@ngraves.fr>
;; Package-Requires: ((emacs "27.1"))
;; Keywords: comm, data, files, tools
;; URL: https://git.sr.ht/~ngraves/ibrowse.el

;;; License:

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.


;;; Commentary:
;; Interact with your browser from Emacs

;;; Code:

(require 'json)
(require 'ibrowse-core)
(require 'radix-tree)

(defconst ibrowse-bookmark-chromium--separator ".")
(defvar ibrowse-bookmark-file)

(defvar ibrowse-bookmark-chromium-default-bar
  ".Bookmarks bar."
  "Default folder to save bookmarks inserted through Emacs in Chromium.
You can use a subfolder if you add an additional
`ibrowse-bookmark--separator'.")

(defun ibrowse-bookmark-chromium--generate-item (title-url)
  "Generate a bookmark item entry from TITLE-URL."
  `((date_added . "")
    (date_last_used . "")
    (guid . "")
    (id . "")
    (meta_info
     (power_bookmark_meta . ""))
    (name . ,(car title-url))
    (type . "url")
    (url . ,(cdr title-url))))

(defun ibrowse-bookmark-chromium--generate-dir (content name)
  "Generate a bookmark directory entry from CONTENT named NAME."
  `((children . ,(vconcat
                  (ibrowse-bookmark-chromium--generate-children content)))
    (date_added . "")
    (date_last_used . "")
    (date_modified . "")
    (guid . "")
    (id . "")
    (name . ,name)
    (type . "folder")))

(defun ibrowse-bookmark-chromium--generate-children (content)
  "Generate a bookmark children array from directory entry from CONTENT."
  (mapcar
   (lambda (cand)
     (if (not (nested-alist-p cand))
         (ibrowse-bookmark-chromium--generate-item cand)
       ;; Case when a bookmark begins with the same name as a directory.
       (if (not (string-match-p "\\." (car cand)))
           (ibrowse-bookmark-chromium--generate-children
            (mapcar
             (lambda (y) (cons (concat (car cand) (car y)) (cdr y)))
             (cdr cand)))
         ;; Case when better split the directory.
         (if (not (string-suffix-p "." (car cand)))
             (let ((prefix (car (split-string (car cand) "\\."))))
               (ibrowse-bookmark-chromium--generate-dir
                (radix-tree-subtree (list cand) (concat prefix ".")) prefix))
           (ibrowse-bookmark-chromium--generate-dir
            (radix-tree-subtree (list cand) (car cand))
            (string-remove-suffix "." (car cand)))))))
   content))

(defun ibrowse-bookmark-chromium--generate-json (bookmark-list)
  "Generate the content of a bookmark file from BOOKMARK-LIST."
  (let* ((radix        (ibrowse-bookmark-chromium--radix-tree (list bookmark-list)))
         (bookmark-bar_ (car radix))
         (other_        (cadr radix))
         (synced_       (caddr radix))
         (bookmark-bar  (or (radix-tree-subtree bookmark-bar_ (caar bookmark-bar_)) (list)))
         (other         (radix-tree-subtree other_ (caar other_)))
         (synced        (or (radix-tree-subtree synced_ (caar synced_)) (list))))
    `((checksum . "")
      (roots
       (bookmark_bar . ,(ibrowse-bookmark-chromium--generate-dir
                         bookmark-bar (substring (caar bookmark-bar_) 1 -1)))
       (other .        ,(ibrowse-bookmark-chromium--generate-dir
                         other (substring (caar other_) 1 -1)))
       (synced .       ,(ibrowse-bookmark-chromium--generate-dir
                         synced (substring (caar synced_) 1 -1))))
      (version . 1))))

(defun ibrowse-bookmark-chromium--write-file (bookmark-list filename)
  "Write the file generated from BOOKMARK-LIST to FILENAME.

The file is generated by applying `ibrowse-bookmark--generate-json' to
BOOKMARK-LIST."
  (with-temp-file filename
    (insert (json-encode (ibrowse-bookmark-chromium--generate-json bookmark-list)))))

(defun ibrowse-bookmark-chromium--extract-fields (item recursion-id)
  "Prepare a search result ITEM for display and store directory to \
RECURSION-ID."
  (let-alist item
    (let ((id (concat recursion-id ibrowse-bookmark-chromium--separator .name)))
      (cond
       ((and .children (string= .type "folder"))
        (seq-mapcat
         (lambda (x) (ibrowse-bookmark-chromium--extract-fields x id))
         .children))
       ((string= .type "url")
        (list (list .name .url id)))))))

(defun ibrowse-bookmark-chromium--radix-tree (bookmark-list)
  "Generate a radix-tree from BOOKMARK-LIST."
  (mapcar
   (lambda (sublist)
     (seq-reduce (lambda (acc x) (radix-tree-insert acc (caddr x) (cadr x)))
                 sublist
                 radix-tree-empty))
   bookmark-list))

(defun ibrowse-bookmark-chromium--get-candidates ()
  "Get an alist with candidates."
  (seq-mapcat
   #'identity
   (delq nil
         (mapcar (lambda (x)
                   (ibrowse-bookmark-chromium--extract-fields (cdr x) ""))
                 (alist-get 'roots (json-read-file
                                    ibrowse-bookmark-file))))))

(defun ibrowse-bookmark-chromium--delete-item (item)
  "Delete ITEM from bookmarks.  ITEM is a list of title, url and id."
  (ibrowse-bookmark-chromium--write-file
   (delete item (ibrowse-bookmark-chromium--get-candidates))
   ibrowse-bookmark-file))

(defun ibrowse-bookmark-chromium-add-item-1 (title url)
  "Same as `ibrowse-add-item' on TITLE and URL, but never prompt."
  (ibrowse-bookmark-chromium--write-file
   (append `((,title ,url ,(concat ibrowse-bookmark-chromium-default-bar title)))
           (ibrowse-bookmark-chromium--get-candidates))
   ibrowse-bookmark-file))

(provide 'ibrowse-bookmark-chromium)
;;; ibrowse-bookmark-chromium.el ends here
