;;; apropospriate.el --- A light & dark theme set for Emacs.

;; Copyright (C) 2015 Justin Talbott

;; Author: Justin Talbott <justin@waymondo.com>
;; URL: http://github.com/waymondo/apropospriate-theme

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.

;;; Commentary:

;; A light & dark theme set for Emacs.

;;; Code:

(defgroup apropospriate nil
  "Apropospriate theme options."
  :group 'faces)

(defcustom apropospriate-mode-line-height 0.95
  "The proportional font size of the mode-line in the apropospriate theme.
Set to `1.0' or nil to prevent font size manipulation."
  :type 'number
  :group 'apropospriate)

(defcustom apropospriate-org-level-resizing t
  "Set to non-nil for `org-level-*' faces to be different larger
  than the default font height."
  :type 'boolean
  :group 'apropospriate)

(defmacro apropospriate-with-color-variables (variant &rest body)
  (declare (indent 0))
  `(let* ((class '((class color) (min-colors 89)))
          (variant ,variant)
          (base00   (if (eq variant 'light) "#F5F5F5" "#424242"))
          (base01   (if (eq variant 'light) "#90A4AE" "#757575"))
          (base02   (if (eq variant 'light) "#78909C" "#9E9E9E"))
          (base03   (if (eq variant 'light) "#546E7A" "#E0E0E0"))
          (yellow   (if (eq variant 'light) "#F57F17" "#FFEE58"))
          (yellow-1 (if (eq variant 'light) "#F9A725" "#FFF59D"))
          (brown    (if (eq variant 'light) "#4E342E" "#BCAAA4"))
          (brown-1  (if (eq variant 'light) "#6D4C41" "#D7CCC8"))
          (orange   (if (eq variant 'light) "#D84315" "#FFCC80"))
          (orange-1 (if (eq variant 'light) "#FF5722" "#FFA726"))
          (red      (if (eq variant 'light) "#D50000" "#E57373"))
          (red-1    (if (eq variant 'light) "#FF1744" "#EF9A9A"))
          (pink     (if (eq variant 'light) "#F8BBD0" "#F8BBD0"))
          (pink-1   (if (eq variant 'light) "#EC407A" "#FF80AB"))
          (purple   (if (eq variant 'light) "#7E57C2" "#E1BEE7"))
          (purple-1 (if (eq variant 'light) "#B388FF" "#9575CD"))
          (blue     (if (eq variant 'light) "#42A5F5" "#64B5F6"))
          (blue-1   (if (eq variant 'light) "#1E88E5" "#42A5F5"))
          (indigo   (if (eq variant 'light) "#5C6BC0" "#C5CAE9"))
          (indigo-1 (if (eq variant 'light) "#9FA8DA" "#7986CB"))
          (cyan     (if (eq variant 'light) "#0097A7" "#80DEEA"))
          (cyan-1   (if (eq variant 'light) "#00B8D4" "#26C6DA"))
          (teal     (if (eq variant 'light) "#26A69A" "#80CBC4"))
          (teal-1   (if (eq variant 'light) "#00897B" "#4DB6AC"))
          (green    (if (eq variant 'light) "#66BB6A" "#C5E1A5"))
          (green-1  (if (eq variant 'light) "#558B2F" "#F4FF81"))
          (base00-1 (if (eq variant 'light) "#FBFBFB" "#3A3A3A"))
          (base00-2 (if (eq variant 'light) "#FDFDFD" "#323232"))
          (base00-3 (if (eq variant 'light) "#FFFFFF" "#2A2A2A"))
          (base00+1 (if (eq variant 'light) "#F0F0F0" "#494949"))
          (base00+2 (if (eq variant 'light) "#EBEBEB" "#515151"))
          (base00+3 (if (eq variant 'light) "#E6E6E6" "#595959"))
          (light-emphasis (if (eq variant 'light) base00+3 base00-3))
          (light-emphasis-1 (if (eq variant 'light) base00+2 base00-2))
          (light-emphasis-2 (if (eq variant 'light) base00+1 base00-1))
          (flashing-color (if (eq variant 'light) "#FCE2EB" "#EE758C"))
          (highlight-line-color (if (eq variant 'light) "#EEEEEE" "#444444")))
     ,@body))

(defun create-apropospriate-theme (variant theme-name)
  "Create a VARIANT of the theme named THEME-NAME."
  (apropospriate-with-color-variables variant
    (custom-theme-set-faces
     theme-name
     `(default ((,class (:background ,base00 :foreground ,base03))))
     `(bold ((,class (:weight bold))))
     `(shadow ((,class (:foreground ,base02))))
     `(border ((,class (:background ,base02))))
     `(cursor ((,class (:background ,pink-1 :inverse-video t))))
     `(highlight ((,class (:background ,base00+1))))
     `(hl-line ((,class (:background ,(if (eq variant 'light) base00-1 base00+1)))))
     `(link ((,class (:foreground ,blue :underline t))))
     `(link-visited ((,class (:inherit link :foreground ,purple))))
     `(minibuffer-prompt ((,class (:foreground ,blue :weight bold))))
     `(region ((,class (:background ,base00+2))))
     `(vhl/default-face ((,class (:background ,base00+1))))
     `(trailing-whitespace ((,class (:background ,base00+1 :foreground ,yellow))))
     `(next-error ((,class (:background ,base01))))
     `(secondary-selection ((,class (:background ,base00-1))))
     `(header-line ((,class (:foreground ,purple :background nil))))
     `(auto-dim-other-buffers-face ((,class (:background ,base00+1))))
     `(fringe ((,class (:background ,base00 :foreground ,base02))))
     `(linum ((,class (:inherit fringe :foreground ,base01))))
     `(linum-relative-current-face ((,class (:inherit fringe :foreground ,base02))))
     `(line-number ((,class (:inherit fringe :foreground ,base01))))
     `(line-number-current-line ((,class (:inherit fringe :foreground ,base02))))
     `(vertical-border ((,class (:foreground ,base00+2))))
     `(window-divider ((,class (:foreground ,base00+2))))
     `(window-divider-first-pixel ((,class (:foreground ,base00+2))))
     `(window-divider-last-pixel ((,class (:foreground ,base00+2))))
     `(widget-button ((,class (:underline t))))
     `(widget-field ((,class (:background ,base02 :box (:line-width 1 :color ,base03)))))
     `(error ((,class (:foreground ,red :weight bold))))
     `(warning ((,class (:foreground ,orange :weight bold))))
     `(success ((,class (:foreground ,green :weight bold))))
     `(ace-jump-face-background ((,class (:foreground ,base00+3))))
     `(ace-jump-face-foreground ((,class (:foreground ,pink-1 :background nil :weight bold))))
     `(ajb-face ((,class (:background ,base00+1))))
     `(avy-lead-face ((,class (:foreground ,pink-1 :background nil :weight bold))))
     `(avy-lead-face-0 ((,class (:inherit avy-lead-face))))
     `(avy-lead-face-1 ((,class (:inherit avy-lead-face))))
     `(avy-lead-face-2 ((,class (:inherit avy-lead-face))))
     `(vertico-quick1 ((,class (:inherit avy-lead-face))))
     `(vertico-quick2 ((,class (:inherit avy-lead-face))))
     `(avy-background-face ((,class (:foreground ,base00+3))))
     `(aw-leading-char-face ((,class (:inherit avy-lead-face :height 2.0))))
     `(aw-background-face ((,class (:inherit avy-background-face))))
     `(highlight-indentation-face ((,class (:background ,base00))))
     `(highlight-indentation-current-column-face ((,class (:background ,base00+1))))
     `(highlight-indent-guides-odd-face ((,class (:background ,base00+1))))
     `(highlight-indent-guides-even-face ((,class (:background ,base00))))
     `(highlight-indent-guides-character-face ((,class (:foreground ,base00+2))))
     `(parenthesis ((,class (:foreground ,base00+3))))
     `(font-lock-comment-face ((,class (:foreground ,base01))))
     `(font-lock-comment-delimiter-face ((,class (:foreground ,base01))))
     `(font-lock-builtin-face ((,class (:foreground ,cyan))))
     `(font-lock-doc-face ((,class (:foreground ,base02))))
     `(font-lock-constant-face ((,class (:foreground ,indigo))))
     `(font-lock-function-name-face ((,class (:foreground ,blue))))
     `(font-lock-keyword-face ((,class (:foreground ,purple))))
     `(font-lock-negation-char-face ((,class (:foreground ,red-1))))
     `(font-lock-preprocessor-face ((,class (:foreground ,blue-1))))
     `(font-lock-regexp-grouping-backslash ((,class (:foreground ,purple-1))))
     `(font-lock-regexp-grouping-construct ((,class (:foreground ,purple-1))))
     `(font-lock-string-face ((,class (:foreground ,green))))
     `(font-lock-type-face ((,class (:foreground ,orange))))
     `(font-lock-variable-name-face ((,class (:foreground ,teal))))
     `(font-lock-warning-face ((,class (:foreground ,red))))
     `(mode-line ((,class (:box (:line-width 4 :color ,light-emphasis :style nil)
                                :background ,base00-2 :foreground ,base03
                                :height ,(or apropospriate-mode-line-height 1.0)))))
     `(mode-line-inactive ((,class (:box (:line-width 4 :color ,base00+1 :style nil)
                                         :background ,base00+1 :foreground ,base02
                                         :height ,(or apropospriate-mode-line-height 1.0)))))
     `(mode-line-emphasis ((,class (:foreground ,base02 :slant italic))))
     `(mode-line-highlight ((,class (:foreground ,base02 :box nil))))
     `(powerline-active1 ((,class (:background ,base00 :height ,(or apropospriate-mode-line-height 1.0)))))
     `(powerline-active2 ((,class (:background ,base00+1 :height ,(or apropospriate-mode-line-height 1.0)))))
     `(powerline-inactive1 ((,class (:background ,base00+1 :height ,(or apropospriate-mode-line-height 1.0)))))
     `(powerline-inactive2 ((,class (:background ,base00+1 :height ,(or apropospriate-mode-line-height 1.0)))))
     `(alert-trivial-face ((,class (:inherit nil :foreground nil :background nil))))
     `(anzu-mode-line ((,class (:foreground ,yellow))))
     `(persp-selected-face ((,class (:foreground ,base03))))
     `(mm-command-output ((,class (:foreground unspecified :background unspecified))))
     `(spaceline-highlight-face ((,class (:background ,light-emphasis :foreground ,base03))))
     `(spaceline-evil-normal ((,class (:background ,yellow :foreground ,base00))))
     `(spaceline-evil-insert ((,class (:background ,red :foreground ,base00))))
     `(spaceline-evil-emacs ((,class (:background ,cyan-1 :foreground ,base00))))
     `(spaceline-evil-replace ((,class (:background ,brown :foreground ,base00))))
     `(spaceline-evil-visual ((,class (:background ,green :foreground ,base00))))
     `(spaceline-evil-motion ((,class (:background ,purple :foreground ,base00))))
     `(spaceline-unmodified ((,class (:background ,orange-1 :foreground ,base00))))
     `(spaceline-modified ((,class (:background ,cyan-1 :foreground ,base00))))
     `(spaceline-read-only ((,class (:background ,purple :foreground ,base00))))
     `(match ((,class (:foreground ,yellow :background ,base00+1 :weight bold))))
     `(isearch ((,class (:inherit match))))
     `(query-replace ((,class (:inherit isearch))))
     `(anzu-replace-to ((,class (:foreground ,yellow :background ,base00+1))))
     `(lazy-highlight ((,class (:foreground ,yellow))))
     `(isearch-lazy-highlight-face ((,class (:foreground ,yellow))))
     `(isearch-fail ((,class (:inherit font-lock-warning-face))))
     `(regex-tool-matched-face ((,class (:foreground nil :background nil :inherit match))))
     `(ag-match-face ((,class (:inherit isearch-lazy-highlight-face))))
     `(custom-state ((,class (:foreground ,green))))
     `(visible-mark-active ((,class (:foreground unspecified :background ,light-emphasis))))
     `(visible-mark-face1 ((,class (:background ,light-emphasis-1))))
     `(visible-mark-face2 ((,class (:background ,light-emphasis-2))))
     `(ido-subdir ((,class (:foreground ,brown))))
     `(ido-first-match ((,class (:inherit match))))
     `(ido-only-match ((,class (:inherit ido-first-match))))
     `(ido-indicator ((,class (:foreground ,red :background ,base00+1))))
     `(ido-virtual ((,class (:foreground ,cyan))))
     `(ido-vertical-match-face ((,class (:foreground ,yellow))))
     `(vertico-current ((,class (:background ,highlight-line-color))))
     `(grizzl-selection-face ((,class (:inherit match))))
     `(orderless-match-face-0 ((,class (:foreground ,blue :background ,base00-1 :weight bold))))
     `(orderless-match-face-1 ((,class (:foreground ,purple :background ,base00-1 :weight bold))))
     `(orderless-match-face-2 ((,class (:foreground ,green :background ,base00-1 :weight bold))))
     `(orderless-match-face-3 ((,class (:foreground ,yellow :background ,base00-1 :weight bold))))
     `(lsp-ui-doc-background ((,class (:background ,base00-2))))
     `(lsp-ui-doc-header ((,class (:background ,base00-2 :foreground ,purple))))
     `(lsp-ui-sideline-symbol ((,class (:background ,base00-1 :foreground ,base03 :box nil))))
     `(lsp-ui-sideline-current-symbol ((,class (:background ,base00-1 :foreground ,purple :box nil))))
     `(lsp-ui-sideline-code-action ((,class (:background ,base00-1 :foreground ,base02))))
     `(lsp-ui-sideline-symbol-info ((,class (:background ,base00-1 :foreground ,base02))))
     `(lsp-ui-peek-peek ((,class (:background ,base00-1))))
     `(lsp-ui-peek-list ((,class (:background ,base00-1))))
     `(lsp-ui-peek-filename ((,class (:background ,base03))))
     `(lsp-ui-peek-line-number ((,class (:background ,base02))))
     `(lsp-ui-peek-highlight ((,class (:background ,base00+1 :foreground ,light-emphasis))))
     `(lsp-ui-peek-header ((,class (:background ,base00-1 :foreground ,light-emphasis))))
     `(lsp-ui-peek-selection ((,class (:background ,base00-1 :inherit secondary-selection))))
     `(ivy-current-match ((,class (:foreground unspecified :background ,light-emphasis))))
     `(ivy-confirm-face ((,class (:foreground ,green))))
     `(ivy-match-required-face ((,class (:foreground ,red))))
     `(ivy-remote ((,class (:foreground ,cyan))))
     `(ivy-virtual ((,class (:foreground ,cyan))))
     `(ivy-minibuffer-match-face-1 ((,class (:foreground nil :background nil :underline (:style line :color ,base02)))))

     `(ivy-minibuffer-match-face-2 ((,class (:foreground ,yellow :background nil :underline (:style line :color ,base02)))))
     `(ivy-minibuffer-match-face-3 ((,class (:foreground ,yellow :background nil :underline (:style line :color ,base02)))))
     `(ivy-minibuffer-match-face-4 ((,class (:foreground ,yellow :background nil :underline (:style line :color ,base02)))))
     `(swiper-match-face-1 ((,class (:inherit ivy-minibuffer-match-face-1))))
     `(swiper-match-face-2 ((,class (:inherit ivy-minibuffer-match-face-2))))
     `(swiper-match-face-3 ((,class (:inherit ivy-minibuffer-match-face-3))))
     `(swiper-match-face-4 ((,class (:inherit ivy-minibuffer-match-face-4))))
     `(swiper-line-face ((,class (:background ,highlight-line-color))))
     `(wgrep-face ((,class (:background ,base00+2))))
     `(wgrep-file-face ((,class (:background ,base00+1))))
     `(wgrep-delete-face ((,class (:foreground ,red))))
     `(wgrep-reject-face ((,class (:foreground ,red))))
     `(wgrep-done-face ((,class (:foreground ,green))))
     `(guide-key/highlight-command-face ((,class (:foreground ,yellow))))
     `(guide-key/key-face ((,class (:foreground ,pink-1))))
     `(guide-key/prefix-command-face ((,class (:foreground ,cyan))))
     `(which-key-key-face ((,class (:foreground ,pink-1))))
     `(which-key-command-description-face ((,class (:foreground ,base03))))
     `(which-key-special-key-face ((,class (:inherit which-key-key-face :inverse-video nil))))
     `(tabbar-default ((,class (:inherit default :foreground ,base02 :background "black" :box (:line-width 4 :color ,base00 :style nil)))))
     `(tabbar-button ((,class (:height 0.8 :box (:line-width 4 :color ,base00-2 :style nil)))))
     `(tabbar-unselected ((,class (:inherit tabbar-button :background ,base00-2))))
     `(tabbar-modified ((,class (:inherit tabbar-unselected))))
     `(tabbar-selected ((,class (:inherit tabbar-button :box (:line-width 4 :color ,base00 :style nil)))))
     `(tabbar-separator ((,class (:inherit tabbar-selected))))
     `(lsp-face-highlight-read ((,class (:inherit highlight-symbol-face))))
     `(lsp-face-highlight-textual ((,class (:inherit highlight-symbol-face))))
     `(lsp-face-highlight-write ((,class (:inherit highlight-symbol-face :background ,base00+2))))
     `(transient-separator ((,class (:background ,base00-2))))
     `(transient-red ((,class (:foreground ,red))))
     `(transient-blue ((,class (:foreground ,blue))))
     `(transient-amaranth ((,class (:foreground ,purple))))
     `(transient-pink ((,class (:foreground ,pink))))
     `(transient-teal ((,class (:foreground ,teal))))
     `(transient-enabled-suffix ((,class (:foreground ,green :background ,base00-2))))
     `(transient-disabled-suffix ((,class (:foreground ,red :background ,base00-2))))
     `(transient-posframe ((,class (:background ,base00-2))))
     `(transient-posframe-border ((,class (:background ,base00-2))))
     `(vertico-posframe ((,class (:background ,base00-2))))
     `(vertico-posframe-border ((,class (:background ,base00-2))))
     `(vertico-posframe-border-2 ((,class (:background ,base00-1))))
     `(vertico-posframe-border-3 ((,class (:background ,base00-1))))
     `(vertico-posframe-border-4 ((,class (:background ,base00-1))))
     `(vertico-posframe-border-fallback ((,class (:background ,base00-1))))
     `(ivy-posframe ((,class :background ,base00-2 :foreground ,base03)))
     `(which-key-posframe ((,class :background ,base00-2 :foreground ,base03)))
     `(which-key-posframe-border ((,class :background ,base00-2)))
     `(company-posframe-quickhelp ((,class :background ,base00-2 :foreground ,base03)))
     `(frog-menu-prompt-face ((,class :foreground ,base03)))
     `(frog-menu-actions-face ((,class :foreground ,base03)))
     `(frog-menu-posframe-background-face ((,class :background ,base00-2)))
     `(frog-menu-action-keybinding-face ((,class :inherit avy-lead-face)))
     `(ivy-posframe-border ((,class :background ,base00-2)))
     `(eldoc-box-body ((,class :background ,base00-2)))
     `(eldoc-box-border ((,class :background ,base00-2)))
     `(solaire-default-face ((,class :background ,base00-1)))
     `(solaire-line-number-face ((,class :background ,base00-2)))
     `(company-tooltip ((,class (:background ,base00-2 :foreground ,base02))))
     `(company-tooltip-selection ((,class (:background ,base00-1 :foreground ,base03))))
     `(company-tooltip-common ((,class (:inherit company-tooltip))))
     `(company-tooltip-common-selection ((,class (:inherit company-tooltip-selection))))
     `(company-tooltip-search ((,class (:foreground ,yellow))))
     `(company-echo-common ((,class (:inherit company-tooltip :foreground ,yellow))))
     `(company-scrollbar-bg ((,class (:background ,base00+1))))
     `(company-scrollbar-fg ((,class (:background ,base00+3))))
     `(company-tooltip-annotation ((,class (:inherit company-tooltip :foreground ,base01))))
     `(company-preview ((,class (:background ,base00 :foreground ,base02))))
     `(company-preview-common ((,class (:foreground ,base02))))
     `(company-preview-search ((,class (:background ,base00))))
     `(corfu-default ((,class (:background ,base00-2 :foreground ,base02))))
     `(corfu-current ((,class (:background ,highlight-line-color :foreground ,base03))))
     `(corfu-bar ((,class (:background ,base00+1 :foreground ,base00+3))))
     `(corfu-border ((,class (:background ,base00-2 :foreground ,base03))))
     `(tempel-field ((,class (:background ,base00-2 :foreground ,base03))))
     `(tempel-form ((,class (:background ,base00-2 :foreground ,base03))))
     `(tempel-default ((,class (:background ,base00-2 :foreground ,base03))))
     `(ac-completion-face ((,class :inherit company-preview)))
     `(popper-echo-area ((,class :foreground ,base03)))
     `(tooltip ((,class (:background ,base00-1 :foreground ,base03))))
     `(popup-tip-face ((,class (:inherit tooltip))))
     `(popup-face ((,class (:inherit company-tooltip))))
     `(popup-menu-mouse-face ((,class (:inherit popup-face))))
     `(popup-menu-selection-face ((,class (:inherit company-tooltip-selection))))
     `(popup-isearch-match ((,class (:foreground ,yellow :background nil))))
     `(popup-scroll-bar-foreground-face ((,class (:inherit company-scrollbar-fg))))
     `(popup-scroll-bar-background-face ((,class (:inherit company-scrollbar-bg))))
     `(tab-bar ((,class (:background ,base00-2))))
     `(tab-line ((,class (:background ,base00-2))))
     `(tab-line-tab ((,class (:foreground ,base02 :background ,base00-2 :box (:line-width 4 :color ,base00-2 :style nil)))))
     `(tab-line-tab-inactive ((,class (:foreground ,base02 :background ,base00-2 :box (:line-width 4 :color ,base00-2 :style nil) :inverse-video nil))))
     `(tab-line-tab-inactive-alternate ((,class (:background ,base00-2))))
     `(tab-line-tab-current ((,class (:foreground ,base03 :background ,base00 :box (:line-width 4 :color ,base00 :style nil)))))
     `(tab-line-highlight ((,class (:background ,highlight-line-color :box (:line-width 4 :color ,highlight-line-color :style nil)))))
     `(tab-line-close-highlight ((,class (:color ,red))))
     `(tab-bar-tab ((,class (:background ,base00 :box (:line-width 4 :color ,base00 :style nil)))))
     `(tab-bar-tab-inactive ((,class (:foreground ,base02 :background ,base00-2))))
     `(flymake-warnline ((,class (:underline ,orange :background ,base00+1))))
     `(flymake-errline ((,class (:underline ,red :background ,base00+1))))
     `(flycheck-error ((,class (:underline (:style wave :color ,red) :inherit unspecified))))
     `(flycheck-info ((,class (:underline (:style wave :color ,green) :inherit unspecified))))
     `(flycheck-warning ((,class (:underline (:style wave :color ,orange) :inherit unspecified))))
     `(flycheck-inline-error ((,class (:foreground ,base03 :background ,base00-1 :inherit unspecified))))
     `(flycheck-inline-warning ((,class (:foreground ,base03 :background ,base00-1 :inherit unspecified))))
     `(flycheck-inline-info ((,class (:foreground ,base03 :background ,base00-1 :inherit unspecified))))
     `(flymake-error ((,class (:underline (:style wave :color ,red) :inherit unspecified))))
     `(flymake-warning ((,class (:underline (:style wave :color ,orange) :inherit unspecified))))
     `(flymake-note ((,class (:underline (:style wave :color ,green) :inherit unspecified))))
     `(flymake-diagnostic-at-point-posframe-background-face ((,class (:background ,base00-2))))
     `(swoop-face-target-words ((,class (:foreground ,yellow))))
     `(swoop-face-target-line ((,class (:background ,base01))))
     `(swoop-face-line-buffer-name ((,class (:inherit header-line :foreground ,base02))))
     `(swoop-face-header-format-line ((,class (:inherit swoop-face-line-buffer-name :foreground ,purple))))
     `(swoop-face-line-number ((,class (:foreground ,base02))))
     `(helm-swoop-target-word-face ((,class (:foreground ,yellow))))
     `(helm-swoop-target-line-face ((,class (:background ,base00+2))))
     `(helm-swoop-target-line-block-face ((,class (:background ,base00+1))))
     `(helm-swoop-line-number-face ((,class (:foreground ,base01))))
     `(helm-match ((,class (:foreground ,yellow-1))))
     `(helm-source-header ((,class (:family inherit :inherit header-line :foreground ,purple))))
     `(helm-visible-mark ((,class (:background ,green))))
     `(helm-header ((,class (:inherit header-line))))
     `(helm-candidate-number ((,class (:background unspecified))))
     `(helm-selection ((,class (:background ,base00+2))))
     `(helm-selection-line ((,class (:inherit helm-selection))))
     `(helm-separator ((,class (:foreground ,base01))))
     `(helm-action ((,class (:underline nil))))
     `(helm-prefarg ((,class (:foreground ,green))))
     `(helm-buffer-saved-out ((,class (:foreground ,red))))
     `(helm-buffer-not-saved ((,class (:foreground ,red))))
     `(helm-buffer-size ((,class (:foreground ,base02))))
     `(helm-buffer-process ((,class (:foreground ,base02))))
     `(helm-buffer-directory ((,class (:foreground ,brown))))
     `(helm-M-x-key ((,class (:foreground ,base02))))
     `(helm-ff-prefix ((,class (:foreground ,yellow))))
     `(helm-ff-executable ((,class (:foreground ,green))))
     `(helm-ff-directory ((,class (:inherit helm-buffer-directory))))
     `(helm-ff-symlink ((,class (:foreground ,brown))))
     `(helm-ff-invalid-symlink ((,class (:inherit helm-ff-symlink :underline (:style wave :color ,red)))))
     `(helm-grep-match ((,class (:inherit match))))
     `(helm-grep-file ((,class (:foreground ,base01))))
     `(helm-grep-lineno ((,class (:foreground ,base02))))
     `(helm-grep-running ((,class (:inherit compilation-mode-line-run))))
     `(helm-grep-finish ((,class (:inherit success))))
     `(helm-visible-mark ((,class (:inverse-video t))))
     `(helm-ls-git-added-modified-face ((,class (:foreground ,cyan))))
     `(helm-ls-git-conflict-face ((,class (:foreground ,purple-1))))
     `(helm-ls-git-deleted-and-staged-face ((,class (:foreground ,base02))))
     `(helm-ls-git-deleted-not-staged-face ((,class (:foreground ,orange))))
     `(helm-ls-git-modified-and-staged-face ((,class (:foreground ,orange-1))))
     `(helm-ls-git-modified-not-staged-face ((,class (:foreground ,orange-1))))
     `(helm-ls-git-renamed-modified-face ((,class (:foreground ,orange-1))))
     `(helm-ls-git-untracked-face ((,class (:foreground ,blue))))
     `(helm-css-scss-target-line-face ((,class (:background ,base00 :foreground ,yellow))))
     `(helm-css-scss-selector-depth-face-1 ((,class (:foreground ,yellow-1))))
     `(helm-css-scss-selector-depth-face-2 ((,class (:foreground ,orange))))
     `(helm-css-scss-selector-depth-face-3 ((,class (:foreground ,yellow-1))))
     `(helm-css-scss-selector-depth-face-4 ((,class (:foreground ,orange))))
     `(helm-css-scss-selector-depth-face-5 ((,class (:foreground ,yellow-1))))
     `(helm-css-scss-selector-depth-face-6 ((,class (:foreground ,orange))))
     `(counsel-css-selector-depth-face-1 ((,class (:foreground ,yellow-1))))
     `(counsel-css-selector-depth-face-2 ((,class (:foreground ,orange))))
     `(counsel-css-selector-depth-face-3 ((,class (:foreground ,yellow-1))))
     `(counsel-css-selector-depth-face-4 ((,class (:foreground ,orange))))
     `(counsel-css-selector-depth-face-5 ((,class (:foreground ,yellow-1))))
     `(counsel-css-selector-depth-face-6 ((,class (:foreground ,orange))))
     `(show-paren-match ((,class (:background unspecified :underline (:style line :color ,pink)))))
     `(show-paren-mismatch ((,class (:background unspecified :foreground ,red :inverse-video t :underline (:style line :color ,red)))))
     `(highlight-symbol-face ((,class (:foreground unspecified :background unspecified :underline (:style line :color ,base02)))))
     `(symbol-overlay-temp-face ((,class (:foreground unspecified :background unspecified :underline (:style line :color ,base02)))))
     `(symbol-overlay-default-face ((,class (:foreground unspecified :background unspecified :underline (:style line :color ,base02)))))
     `(sp-show-pair-match-face ((,class (:inherit show-paren-match))))
     `(sp-show-pair-mismatch-face ((,class (:inherit show-paren-mismatch))))
     `(rainbow-delimiters-depth-1-face ((,class (:foreground ,purple-1))))
     `(rainbow-delimiters-depth-2-face ((,class (:foreground ,indigo-1))))
     `(rainbow-delimiters-depth-3-face ((,class (:foreground ,cyan-1))))
     `(rainbow-delimiters-depth-4-face ((,class (:foreground ,blue-1))))
     `(rainbow-delimiters-depth-5-face ((,class (:foreground ,teal-1))))
     `(rainbow-delimiters-depth-6-face ((,class (:foreground ,pink-1))))
     `(rainbow-delimiters-depth-7-face ((,class (:foreground ,purple))))
     `(rainbow-delimiters-depth-8-face ((,class (:foreground ,indigo))))
     `(rainbow-delimiters-depth-9-face ((,class (:foreground ,cyan))))
     `(rainbow-delimiters-depth-10-face ((,class (:foreground ,blue))))
     `(rainbow-delimiters-depth-11-face ((,class (:foreground ,teal))))
     `(rainbow-delimiters-depth-12-face ((,class (:foreground ,pink))))
     `(rainbow-delimiters-mismatched-face ((,class (:foreground ,red))))
     `(rainbow-delimiters-unmatched-face ((,class (:foreground ,red))))
     `(highlight-blocks-depth-1-face ((,class (:background ,base00-1))))
     `(highlight-blocks-depth-2-face ((,class (:background ,base00-2))))
     `(highlight-blocks-depth-3-face ((,class (:background ,base00-3))))
     `(highlight-blocks-depth-4-face ((,class (:background ,base00-3))))
     `(highlight-blocks-depth-5-face ((,class (:background ,base00-3))))
     `(highlight-blocks-depth-6-face ((,class (:background ,base00-3))))
     `(highlight-blocks-depth-7-face ((,class (:background ,base00-3))))
     `(highlight-blocks-depth-8-face ((,class (:background ,base00-3))))
     `(highlight-blocks-depth-9-face ((,class (:background ,base00-3))))
     `(flx-highlight-face ((,class (:foreground ,yellow))))
     `(diff-added ((,class (:foreground ,green))))
     `(diff-changed ((,class (:foreground ,blue))))
     `(diff-removed ((,class (:foreground ,red))))
     `(diff-header ((,class (:background ,base00+1))))
     `(diff-file-header ((,class (:foreground ,base01))))
     `(diff-hunk-header ((,class (:foreground ,base01))))
     `(git-gutter:added ((,class (:foreground ,base01))))
     `(git-gutter:deleted ((,class (:foreground ,base01))))
     `(git-gutter:modified ((,class (:foreground ,base01))))
     `(diff-hl-insert ((,class (:foreground ,green :background ,base00+1))))
     `(diff-hl-unknown ((,class (:foreground ,orange-1 :background ,base00+1))))
     `(diff-hl-delete ((,class (:foreground ,red-1 :background ,base00+1))))
     `(diff-hl-change ((,class (:foreground ,blue-1 :background ,base00+1))))
     `(eldoc-highlight-function-argument ((,class (:foreground ,green))))
     `(undo-tree-visualizer-default-face ((,class (:foreground ,base03))))
     `(undo-tree-visualizer-current-face ((,class (:foreground ,green :weight bold))))
     `(undo-tree-visualizer-active-branch-face ((,class (:foreground ,red))))
     `(undo-tree-visualizer-register-face ((,class (:foreground ,yellow))))
     `(erm-syn-errline ((,class (:box nil :underline (:style wave :color ,red)))))
     `(erm-syn-warnline ((,class (:box nil :underline (:style wave :color ,orange)))))
     `(dirvish-hl-line ((,class (:background ,highlight-line-color))))
     `(dired-directory ((,class (:foreground ,brown))))
     `(dired-header ((,class (:inherit header-line))))
     `(dired-subtree-depth-1-face ((,class (:background ,base00-1))))
     `(dired-subtree-depth-2-face ((,class (:background ,base00-2))))
     `(dired-subtree-depth-3-face ((,class (:background ,base00-2))))
     `(dired-subtree-depth-4-face ((,class (:background ,base00-3))))
     `(dired-subtree-depth-5-face ((,class (:background ,base00-3))))
     `(dired-subtree-depth-6-face ((,class (:background ,base00-3))))
     `(diredp-compressed-file-suffix ((,class (:foreground ,blue))))
     `(diredp-dir-heading ((,class (:inherit dired-header))))
     `(diredp-dir-priv ((,class (:foreground ,cyan :background nil))))
     `(diredp-exec-priv ((,class (:foreground ,blue :background nil))))
     `(diredp-executable-tag ((,class (:foreground ,red :background nil))))
     `(diredp-file-name ((,class (:foreground ,orange))))
     `(diredp-file-suffix ((,class (:foreground ,green))))
     `(diredp-flag-mark-line ((,class (:background nil :inherit highlight))))
     `(diredp-ignored-file-name ((,class (:foreground ,base03))))
     `(diredp-link-priv ((,class (:background nil :foreground ,purple))))
     `(diredp-mode-line-flagged ((,class (:foreground ,red))))
     `(diredp-mode-line-marked ((,class (:foreground ,green))))
     `(diredp-no-priv ((,class (:background nil))))
     `(diredp-number ((,class (:foreground ,orange-1))))
     `(diredp-other-priv ((,class (:background nil :foreground ,purple))))
     `(diredp-rare-priv ((,class (:foreground ,red :background nil))))
     `(diredp-read-priv ((,class (:foreground ,green :background nil))))
     `(diredp-symlink ((,class (:foreground ,purple))))
     `(diredp-write-priv ((,class (:foreground ,yellow :background nil))))
     `(magit-process-ok ((,class (:foreground ,green))))
     `(magit-process-ng ((,class (:foreground ,red))))
     `(magit-tag ((,class (:foreground ,cyan-1))))
     `(magit-log-author ((,class (:foreground ,base02))))
     `(magit-log-date ((,class (:foreground ,base02))))
     `(magit-log-graph ((,class (:foreground ,base02))))
     `(magit-log-head-label-head ((,class (:box nil :background ,base00+1 :foreground ,green-1))))
     `(magit-log-head-label-remote ((,class (:box nil :background ,base00+1 :foreground ,green-1))))
     `(magit-log-head-label-local ((,class (:box nil :background ,base00+1 :foreground ,green-1))))
     `(magit-log-head-label-tags ((,class (:box nil :background ,base00+1 :foreground ,green-1))))
     `(magit-log-head-label-patches ((,class (:box nil :background ,base00+1 :foreground ,green-1))))
     `(magit-hash ((,class (:foreground ,cyan))))
     `(magit-branch ((,class (:background ,base00+1 :foreground ,green-1))))
     `(magit-branch-local ((,class (:foreground ,green-1))))
     `(magit-branch-remote ((,class (:foreground ,green-1))))
     `(magit-branch-label ((,class (:foreground ,green-1))))
     `(magit-branch-current ((,class (:foreground ,green-1))))
     `(magit-head ((,class (:foreground ,indigo))))
     `(magit-section-highlight ((,class (:inherit highlight))))
     `(magit-section-heading ((,class (:foreground ,purple))))
     `(magit-dimmed ((,class (:background ,base00-2))))
     `(magit-refname ((,class (:foreground ,orange))))
     `(magit-diff-added ((,class (:foreground ,green :background unspecified))))
     `(magit-diff-removed ((,class (:foreground ,red :background unspecified))))
     `(magit-diff-context ((,class (:foreground unspecified :background unspecified))))
     `(diff-refine-added ((,class (:foreground ,green :weight bold :background ,base00-1))))
     `(diff-refine-changed ((,class (:foreground ,green :weight bold :background ,base00-1))))
     `(diff-refine-removed ((,class (:foreground ,red :weight bold :background ,base00-1))))
     `(magit-diff-added-highlight ((,class (:foreground ,green :background ,base00-2))))
     `(magit-diff-removed-highlight ((,class (:foreground ,red :background ,base00-2))))
     `(magit-diff-context-highlight ((,class (:foreground unspecified :background ,base00-2))))
     `(magit-diffstat-added ((,class (:foreground ,green :background unspecified))))
     `(magit-diffstat-removed ((,class (:foreground ,red :background unspecified))))
     `(magit-cherry-unmatched ((,class (:foreground ,cyan))))
     `(magit-cherry-equivalent ((,class (:foreground ,purple))))
     `(magit-blame-heading ((,class (:foreground unspecified :background unspecified))))
     `(magit-diff-hunk-heading ((,class (:foreground ,base01))))
     `(magit-diff-hunk-heading-highlight ((,class (:inherit magit-diff-hunk-heading :background ,base00-2))))
     `(magit-diff-hunk-heading-selection ((,class (:foreground unspecified :background ,base00-2))))
     `(magit-diff-lines-heading ((,class (:background unspecified))))
     `(magit-popup-argument ((,class (:foreground ,orange))))
     `(magit-popup-disabled-argument ((,class (:foreground ,base01))))
     `(magit-item-highlight ((,class (:background ,highlight-line-color))))
     `(git-commit-summary ((,class (:foreground ,base03))))
     `(git-timemachine-minibuffer-detail-face ((,class (:foreground ,orange))))
     `(smerge-markers ((,class (:inherit highlight))))
     `(smerge-upper ((,class (:background ,base00-1))))
     `(smerge-lower ((,class (:background ,base00-1))))
     `(smerge-refined-removed ((,class (:foreground ,red :background ,base00-1))))
     `(smerge-refined-added ((,class (:foreground ,green :background ,base00-1))))
     `(ediff-current-diff-A ((,class (:background ,base00-2 :inverse-video nil :extend t))))
     `(ediff-current-diff-Ancestor ((,class (:inherit ediff-current-diff-A))))
     `(ediff-current-diff-B ((,class (:inherit ediff-current-diff-A))))
     `(ediff-current-diff-C ((,class (:inherit ediff-current-diff-A))))
     `(ediff-fine-diff-A ((,class (:background ,base00 :stipple nil :extend t))))
     `(ediff-fine-diff-Ancestor ((,class (:inherit ediff-fine-diff-A))))
     `(ediff-fine-diff-B ((,class (:inherit ediff-fine-diff-A))))
     `(ediff-fine-diff-C ((,class (:inherit ediff-fine-diff-A ))))
     `(ediff-even-diff-A ((,class (:background ,base00+1 :stipple nil :extend t))))
     `(ediff-even-diff-Ancestor ((,class (:inherit ediff-even-diff-A))))
     `(ediff-even-diff-B ((,class (:inherit ediff-even-diff-A))))
     `(ediff-even-diff-C ((,class (:inherit ediff-even-diff-A))))
     `(ediff-odd-diff-A ((,class (:inherit ediff-even-diff-A))))
     `(ediff-odd-diff-Ancestor ((,class (:inherit ediff-odd-diff-A))))
     `(ediff-odd-diff-B ((,class (:inherit ediff-odd-diff-A))))
     `(ediff-odd-diff-C ((,class (:inherit ediff-odd-diff-A))))
     `(compilation-info ((,class (:foreground ,base02 :underline t))))
     `(compilation-column-number ((,class (:foreground ,base02))))
     `(compilation-line-number ((,class (:foreground ,base02))))
     `(compilation-message-face ((,class (:foreground ,blue))))
     `(compilation-mode-line-exit ((,class (:inherit nil :foreground unspecified :background unspecified))))
     `(compilation-mode-line-fail ((,class (:inherit nil :foreground unspecified :background unspecified))))
     `(compilation-mode-line-run ((,class (:inherit nil :foreground unspecified :background unspecified))))
     `(outline-1 ((,class (:foreground ,base03))))
     `(org-date ((,class (:foreground ,purple))))
     `(org-done ((,class (:foreground ,green))))
     `(org-checkbox ((,class (:foreground ,purple))))
     `(org-agenda-structure ((,class (:foreground ,purple))))
     `(org-agenda-date ((,class (:foreground ,blue :underline nil))))
     `(org-agenda-done ((,class (:foreground ,green))))
     `(org-agenda-dimmed-todo-face ((,class (:foreground ,base03))))
     `(org-code ((,class (:foreground ,orange-1))))
     `(org-column ((,class (:background ,base00+1))))
     `(org-column-title ((,class (:background ,base00 :weight bold :underline t))))
     `(org-document-info ((,class (:foreground ,cyan))))
     `(org-document-info-keyword ((,class (:foreground ,green))))
     `(org-document-title ((,class (:weight bold :foreground ,orange :height 1.44))))
     `(org-ellipsis ((,class (:foreground ,base03))))
     `(org-footnote ((,class (:foreground ,cyan))))
     `(org-formula ((,class (:foreground ,red))))
     `(org-hide ((,class (:foreground ,base02))))
     `(org-link ((,class (:inherit link))))
     `(org-scheduled ((,class (:foreground ,green))))
     `(org-scheduled-previously ((,class (:foreground ,orange))))
     `(org-scheduled-today ((,class (:foreground ,green))))
     `(org-special-keyword ((,class (:foreground ,orange))))
     `(org-table ((,class (:foreground ,purple))))
     `(org-todo ((,class (:foreground ,red))))
     `(org-upcoming-deadline ((,class (:foreground ,orange))))
     `(org-warning ((,class (:foreground ,red))))
     `(org-block ((,class (:foreground ,orange :background ,base00+1))))
     `(org-meta-line ((,class (:inherit font-lock-comment-face))))
     `(org-block-end-line ((,class (:inherit font-lock-comment-face))))
     `(org-block-end-line ((,class (:inherit font-lock-comment-face))))
     `(org-block-background ((,class (:background ,base00+1))))
     `(markdown-url-face ((,class (:inherit link))))
     `(markdown-link-face ((,class (:foreground ,blue :underline t))))
     `(js2-warning ((,class (:inherit flycheck-warning))))
     `(js2-error ((,class (:foreground nil :inherit flycheck-error))))
     `(js2-external-variable ((,class (:foreground ,orange))))
     `(js2-function-param ((,class (:foreground ,indigo))))
     `(js2-instance-member ((,class (:foreground ,blue))))
     `(js2-private-function-call ((,class (:foreground ,pink))))
     `(js3-warning-face ((,class (:inherit js2-warning))))
     `(js3-error-face ((,class (:inherit js2-error))))
     `(js3-external-variable-face ((,class (:inherit js2-external-variable))))
     `(js3-function-param-face ((,class (:inherit js2-function-param))))
     `(js3-instance-member-face ((,class (:inherit js2-instance-member))))
     `(js3-private-function-call-face ((,class (:inherit js2-private-function-call))))
     `(js3-jsdoc-tag-face ((,class (:foreground ,orange))))
     `(js3-jsdoc-type-face ((,class (:foreground ,cyan))))
     `(js3-jsdoc-value-face ((,class (:foreground ,orange-1))))
     `(js3-jsdoc-html-tag-name-face ((,class (:foreground ,blue))))
     `(js3-jsdoc-html-tag-delimiter-face ((,class (:foreground ,green))))
     `(erb-face ((,class (:background ,base00+1 :foreground ,base03))))
     `(erb-delim-face ((,class (:background ,base00+1 :foreground ,red))))
     `(erb-exec-face ((,class (:background ,base00+1 :foreground ,base02))))
     `(erb-exec-delim-face ((,class (:background ,base00+1 :foreground ,red))))
     `(erb-out-face ((,class (:background ,base00+1 :foreground ,base03))))
     `(erb-out-delim-face ((,class (:background ,base00+1 :foreground ,red))))
     `(erb-comment-face ((,class (:background ,base00+1 :foreground ,base02))))
     `(erb-comment-delim-face ((,class (:background ,base00+1 :foreground ,base01))))
     `(web-mode-error-face ((,class (:inherit error))))
     `(web-mode-symbol-face ((,class (:inherit font-lock-constant-face))))
     `(web-mode-doctype-face ((,class (:inherit font-lock-comment-face))))
     `(web-mode-html-tag-face ((,class (:inherit font-lock-function-name-face))))
     `(web-mode-html-tag-bracket-face ((,class (:foreground ,purple))))
     `(web-mode-html-attr-name-face ((,class (:inherit font-lock-variable-name-face))))
     `(web-mode-current-element-highlight-face ((,class (:inherit highlight-symbol-face))))
     `(web-mode-current-column-highlight-face ((,class (:inherit hl-line))))
     `(pulse-highlight-start-face ((,class (:background ,flashing-color))))
     `(goggles-changed ((,class (:background ,flashing-color))))
     `(goggles-removed ((,class (:background ,flashing-color))))
     `(goggles-added ((,class (:background ,flashing-color))))
     `(eshell-prompt ((,class (:foreground ,green :weight bold))))
     `(eshell-ls-archive ((,class (:foreground ,teal))))
     `(eshell-ls-backup ((,class (:inherit font-lock-comment-face))))
     `(eshell-ls-clutter ((,class (:inherit font-lock-comment-face))))
     `(eshell-ls-directory ((,class (:foreground ,brown))))
     `(eshell-ls-executable ((,class (:foreground ,red))))
     `(eshell-ls-unreadable ((,class (:inherit font-lock-comment-face))))
     `(eshell-ls-missing ((,class (:inherit font-lock-warning-face))))
     `(eshell-ls-product ((,class (:inherit font-lock-doc-face))))
     `(eshell-ls-special ((,class (:foreground ,orange))))
     `(eshell-ls-symlink ((,class (:foreground ,purple))))
     `(ansi-color-black ((,class (:foreground ,base03))))
     `(ansi-color-bright-black ((,class (:foreground ,base03))))
     `(ansi-color-red ((,class (:foreground ,red))))
     `(ansi-color-bright-red ((,class (:foreground ,red-1))))
     `(ansi-color-green ((,class (:foreground ,green))))
     `(ansi-color-bright-green ((,class (:foreground ,green-1))))
     `(ansi-color-yellow ((,class (:foreground ,yellow))))
     `(ansi-color-bright-yellow ((,class (:foreground ,yellow-1))))
     `(ansi-color-blue ((,class (:foreground ,blue))))
     `(ansi-color-bright-blue ((,class (:foreground ,blue-1))))
     `(ansi-color-magenta ((,class (:foreground ,purple))))
     `(ansi-color-bright-magenta ((,class (:foreground ,purple-1))))
     `(ansi-color-cyan ((,class (:foreground ,cyan))))
     `(ansi-color-bright-cyan ((,class (:foreground ,cyan-1))))
     `(ansi-color-white ((,class (:foreground ,base00))))
     `(ansi-color-bright-white ((,class (:foreground ,base00-1))))
     `(term ((,class (:foreground ,base03))))
     `(term-color-black ((,class (:foreground ,base03))))
     `(term-color-red ((,class (:foreground ,red))))
     `(term-color-green ((,class (:foreground ,green))))
     `(term-color-yellow ((,class (:foreground ,yellow))))
     `(term-color-blue ((,class (:foreground ,blue))))
     `(term-color-magenta ((,class (:foreground ,purple))))
     `(term-color-cyan ((,class (:foreground ,cyan))))
     `(term-color-white ((,class (:foreground ,base00))))
     `(erc-action-face ((,class (:inherit erc-default-face))))
     `(erc-bold-face ((,class (:weight bold))))
     `(erc-current-nick-face ((,class (:foreground ,blue :weight bold))))
     `(erc-dangerous-host-face ((,class (:inherit font-lock-warning-face))))
     `(erc-default-face ((,class (:foreground ,base03))))
     `(erc-direct-msg-face ((,class (:inherit erc-default))))
     `(erc-error-face ((,class (:inherit font-lock-warning-face))))
     `(erc-fool-face ((,class (:inherit erc-default))))
     `(erc-highlight-face ((,class (:inherit hover-highlight))))
     `(erc-input-face ((,class (:foreground ,yellow))))
     `(erc-keyword-face ((,class (:foreground ,blue))))
     `(erc-nick-default-face ((,class (:foreground ,yellow :weight bold))))
     `(erc-my-nick-face ((,class (:foreground ,red :weight bold))))
     `(erc-nick-msg-face ((,class (:inherit erc-default))))
     `(erc-notice-face ((,class (:foreground ,green))))
     `(erc-pal-face ((,class (:foreground ,orange))))
     `(erc-prompt-face ((,class (:foreground ,orange :background ,base00 :weight bold))))
     `(erc-timestamp-face ((,class (:foreground ,green))))
     `(erc-underline-face ((,class (:underline t))))
     `(bm-face ((,class (:inherit secondary-selection))))
     `(bm-persistent-face ((,class (:inherit secondary-selection))))
     `(bm-fringe-face ((,class (:inherit fringe))))
     `(bm-fringe-persistent-face ((,class (:inherit fringe))))
     `(neo-dir-link-face ((,class (:foreground ,brown))))
     `(neo-expand-btn-face ((,class (:foreground ,base02))))
     `(neo-header-face ((,class (:inherit header-line))))
     `(neo-file-link-face ((,class (:foreground ,base03))))
     `(neo-root-dir-face ((,class (:foreground ,base02))))
     `(org-ellipsis ((,class (:inherit font-lock-keyword-face))))
     `(flyspell-incorrect ((,class (:underline (:style wave :color ,red)))))
     `(flyspell-duplicate ((,class (:underline (:style wave :color ,orange)))))
     `(org-level-1 ((,class (:inherit header-line :height ,(if apropospriate-org-level-resizing 1.3 1.0)))))
     `(org-level-2 ((,class (:inherit header-line :height ,(if apropospriate-org-level-resizing 1.2 1.0)))))
     `(org-level-3 ((,class (:inherit header-line :height ,(if apropospriate-org-level-resizing 1.1 1.0)))))
     `(org-level-4 ((,class (:inherit header-line))))
     `(org-level-5 ((,class (:inherit header-line))))
     `(org-level-6 ((,class (:inherit header-line))))
     `(org-level-7 ((,class (:inherit header-line))))
     `(org-level-8 ((,class (:inherit header-line))))
     `(hydra-face-red ((,class (:foreground ,red :bold t))))
     `(hydra-face-blue ((,class (:foreground ,blue :bold t))))
     `(hydra-face-amaranth ((,class (:foreground ,red-1 :bold t))))
     `(hydra-face-pink ((,class (:foreground ,pink-1 :bold t))))
     `(hydra-face-teal ((,class (:foreground ,cyan :bold t))))
     `(hi-yellow ((,class (:inherit highlight :foreground ,yellow :inverse-video t))))
     `(hi-pink ((,class (:inherit highlight :foreground ,pink :inverse-video t))))
     `(hi-green ((,class (:inherit highlight :foreground ,green :inverse-video t))))
     `(hi-blue ((,class (:inherit highlight :foreground ,blue :inverse-video t))))
     `(hi-black-b ((,class (:inherit highlight :foreground ,light-emphasis :inverse-video t :bold t))))
     `(hi-blue-b ((,class (:inherit hi-blue :bold t))))
     `(hi-green-b ((,class (:inherit hi-green :bold t))))
     `(hi-red-b ((,class (:inherit highlight :foreground ,red :inverse-video t :bold t))))
     `(hi-black-hb ((,class (:inherit hi-black-b :height 1.2))))
     `(vr/match-0 ((,class (:background ,light-emphasis-1))))
     `(vr/match-1 ((,class (:background ,light-emphasis-2))))
     `(vr/group-0 ((,class (:foreground ,red :bold t))))
     `(vr/group-1 ((,class (:foreground ,orange :bold t))))
     `(vr/group-2 ((,class (:foreground ,green :bold t))))
     `(vr/match-separator-face ((,class (:foreground ,red :bold t))))
     `(yafolding-ellipsis-face ((,class (:inherit font-lock-comment-face)))))

    (custom-theme-set-variables
     theme-name
     `(diff-hl-show-hunk-posframe-internal-border-color ,base00-2)
     `(evil-emacs-state-cursor '(,red hbar))
     `(evil-insert-state-cursor '(,red bar))
     `(evil-normal-state-cursor '(,yellow box))
     `(evil-visual-state-cursor '(,green box))
     `(pos-tip-foreground-color ,base03)
     `(pos-tip-background-color ,base00-1)
     `(mlscroll-in-color ,base00+2)
     `(mlscroll-out-color ,base00)
     `(highlight-indent-guides-auto-enabled nil)
     `(highlight-symbol-foreground-color ,base03)
     `(highlight-symbol-colors '(,yellow
                                 ,green
                                 ,cyan
                                 ,blue
                                 ,purple
                                 ,orange))
     `(vc-annotate-color-map '((20 . ,red)
                               (40 . ,orange-1)
                               (60 . ,orange)
                               (80 . ,yellow-1)
                               (100 . ,yellow)
                               (120 . ,green)
                               (140 . ,green)
                               (160 . ,green)
                               (180 . ,green)
                               (200 . ,green-1)
                               (220 . ,green-1)
                               (240 . ,cyan)
                               (260 . ,cyan)
                               (280 . ,cyan-1)
                               (300 . ,cyan-1)
                               (320 . ,blue)
                               (340 . ,blue)
                               (360 . ,blue-1)))
     `(vc-annotate-very-old-color ,base02)
     `(beacon-color ,flashing-color)
     `(highlight-tail-colors
       '((,flashing-color . 0) (,base00 . 100)))
     `(tabbar-background-color ,base00-2)
     `(ansi-color-names-vector
       [,base00 ,red-1 ,green ,yellow ,blue ,purple ,cyan ,base03]))))

;;;###autoload
(and load-file-name
     (boundp 'custom-theme-load-path)
     (add-to-list 'custom-theme-load-path
                  (file-name-as-directory
                   (file-name-directory load-file-name))))

(provide 'apropospriate)

;; Local Variables:
;; no-byte-compile: t
;; eval: (when (require 'rainbow-mode nil t) (rainbow-mode 1))
;; fill-column: 105
;; End:
;;; apropospriate.el ends here
