"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerQueryAssistRoutes = registerQueryAssistRoutes;
var _configSchema = require("@osd/config-schema");
var _errors = require("../../../../../src/core/server/opensearch/client/errors");
var _query_assist = require("../../../common/constants/query_assist");
var _generate_field_context = require("../../common/helpers/query_assist/generate_field_context");
var _agents = require("./utils/agents");
var _constants = require("./utils/constants");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerQueryAssistRoutes(router) {
  /**
   * Returns whether the PPL agent is configured.
   */
  router.get({
    path: _query_assist.QUERY_ASSIST_API.CONFIGURED,
    validate: {}
  }, async (context, request, response) => {
    const client = context.core.opensearch.client.asCurrentUser;
    try {
      // if the call does not throw any error, then the agent is properly configured
      await (0, _agents.getAgentIdByConfig)(client, _constants.AGENT_CONFIGS.PPL_AGENT);
      return response.ok({
        body: {
          configured: true
        }
      });
    } catch (error) {
      return response.ok({
        body: {
          configured: false,
          error: error.message
        }
      });
    }
  });
  router.post({
    path: _query_assist.QUERY_ASSIST_API.GENERATE_PPL,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        question: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const client = context.core.opensearch.client.asCurrentUser;
    try {
      const pplRequest = await (0, _agents.getAgentIdAndRequest)({
        client,
        configName: _constants.AGENT_CONFIGS.PPL_AGENT,
        body: {
          parameters: {
            index: request.body.index,
            question: request.body.question
          }
        }
      });
      if (!pplRequest.body.inference_results[0].output[0].result) throw new Error('Generated PPL query not found.');
      const result = JSON.parse(pplRequest.body.inference_results[0].output[0].result);
      const ppl = result.ppl.replace(/[\r\n]/g, ' ').trim().replace(/ISNOTNULL/g, 'isnotnull') // https://github.com/opensearch-project/sql/issues/2431
      .replace(/\bSPAN\(/g, 'span('); // https://github.com/opensearch-project/dashboards-observability/issues/759
      return response.ok({
        body: ppl
      });
    } catch (error) {
      if ((0, _errors.isResponseError)(error) && error.statusCode === 400 &&
      // on opensearch >= 2.17, error.body is an object https://github.com/opensearch-project/ml-commons/pull/2858
      JSON.stringify(error.body).includes(_query_assist.ERROR_DETAILS.GUARDRAILS_TRIGGERED)) {
        return response.badRequest({
          body: _query_assist.ERROR_DETAILS.GUARDRAILS_TRIGGERED
        });
      }
      return response.custom({
        statusCode: error.statusCode || 500,
        body: typeof error.body === 'string' ? error.body : JSON.stringify(error.body)
      });
    }
  });
  router.post({
    path: _query_assist.QUERY_ASSIST_API.SUMMARIZE,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        question: _configSchema.schema.string(),
        query: _configSchema.schema.maybe(_configSchema.schema.string()),
        response: _configSchema.schema.string(),
        isError: _configSchema.schema.boolean()
      })
    }
  }, async (context, request, response) => {
    const client = context.core.opensearch.client.asCurrentUser;
    const {
      index,
      question,
      query,
      response: _response,
      isError
    } = request.body;
    const queryResponse = JSON.stringify(_response);
    let summaryRequest;
    try {
      var _summaryRequest$body$;
      if (!isError) {
        summaryRequest = await (0, _agents.getAgentIdAndRequest)({
          client,
          configName: _constants.AGENT_CONFIGS.RESPONSE_SUMMARY_AGENT,
          body: {
            parameters: {
              index,
              question,
              query,
              response: queryResponse
            }
          }
        });
      } else {
        const [mappings, sampleDoc] = await Promise.all([client.indices.getMapping({
          index
        }), client.search({
          index,
          size: 1
        })]);
        const fields = (0, _generate_field_context.generateFieldContext)(mappings, sampleDoc);
        summaryRequest = await (0, _agents.getAgentIdAndRequest)({
          client,
          configName: _constants.AGENT_CONFIGS.ERROR_SUMMARY_AGENT,
          body: {
            parameters: {
              index,
              question,
              query,
              response: queryResponse,
              fields
            }
          }
        });
      }
      const summary = summaryRequest.body.inference_results[0].output[0].result;
      if (!summary) throw new Error('Generated summary not found.');
      const suggestedQuestions = Array.from((((_summaryRequest$body$ = summaryRequest.body.inference_results[0].output[1]) === null || _summaryRequest$body$ === void 0 ? void 0 : _summaryRequest$body$.result) || '').matchAll(/<question>((.|[\r\n])+?)<\/question>/g)).map(m => m[1]);
      return response.ok({
        body: {
          summary,
          suggestedQuestions
        }
      });
    } catch (error) {
      if ((0, _errors.isResponseError)(error) && error.statusCode === 400 &&
      // on opensearch >= 2.17, error.body is an object https://github.com/opensearch-project/ml-commons/pull/2858
      JSON.stringify(error.body).includes(_query_assist.ERROR_DETAILS.GUARDRAILS_TRIGGERED)) {
        return response.badRequest({
          body: _query_assist.ERROR_DETAILS.GUARDRAILS_TRIGGERED
        });
      }
      return response.custom({
        statusCode: error.statusCode || 500,
        body: typeof error.body === 'string' ? error.body : JSON.stringify(error.body)
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfY29uZmlnU2NoZW1hIiwicmVxdWlyZSIsIl9lcnJvcnMiLCJfcXVlcnlfYXNzaXN0IiwiX2dlbmVyYXRlX2ZpZWxkX2NvbnRleHQiLCJfYWdlbnRzIiwiX2NvbnN0YW50cyIsInJlZ2lzdGVyUXVlcnlBc3Npc3RSb3V0ZXMiLCJyb3V0ZXIiLCJnZXQiLCJwYXRoIiwiUVVFUllfQVNTSVNUX0FQSSIsIkNPTkZJR1VSRUQiLCJ2YWxpZGF0ZSIsImNvbnRleHQiLCJyZXF1ZXN0IiwicmVzcG9uc2UiLCJjbGllbnQiLCJjb3JlIiwib3BlbnNlYXJjaCIsImFzQ3VycmVudFVzZXIiLCJnZXRBZ2VudElkQnlDb25maWciLCJBR0VOVF9DT05GSUdTIiwiUFBMX0FHRU5UIiwib2siLCJib2R5IiwiY29uZmlndXJlZCIsImVycm9yIiwibWVzc2FnZSIsInBvc3QiLCJHRU5FUkFURV9QUEwiLCJzY2hlbWEiLCJvYmplY3QiLCJpbmRleCIsInN0cmluZyIsInF1ZXN0aW9uIiwicHBsUmVxdWVzdCIsImdldEFnZW50SWRBbmRSZXF1ZXN0IiwiY29uZmlnTmFtZSIsInBhcmFtZXRlcnMiLCJpbmZlcmVuY2VfcmVzdWx0cyIsIm91dHB1dCIsInJlc3VsdCIsIkVycm9yIiwiSlNPTiIsInBhcnNlIiwicHBsIiwicmVwbGFjZSIsInRyaW0iLCJpc1Jlc3BvbnNlRXJyb3IiLCJzdGF0dXNDb2RlIiwic3RyaW5naWZ5IiwiaW5jbHVkZXMiLCJFUlJPUl9ERVRBSUxTIiwiR1VBUkRSQUlMU19UUklHR0VSRUQiLCJiYWRSZXF1ZXN0IiwiY3VzdG9tIiwiU1VNTUFSSVpFIiwicXVlcnkiLCJtYXliZSIsImlzRXJyb3IiLCJib29sZWFuIiwiX3Jlc3BvbnNlIiwicXVlcnlSZXNwb25zZSIsInN1bW1hcnlSZXF1ZXN0IiwiX3N1bW1hcnlSZXF1ZXN0JGJvZHkkIiwiUkVTUE9OU0VfU1VNTUFSWV9BR0VOVCIsIm1hcHBpbmdzIiwic2FtcGxlRG9jIiwiUHJvbWlzZSIsImFsbCIsImluZGljZXMiLCJnZXRNYXBwaW5nIiwic2VhcmNoIiwic2l6ZSIsImZpZWxkcyIsImdlbmVyYXRlRmllbGRDb250ZXh0IiwiRVJST1JfU1VNTUFSWV9BR0VOVCIsInN1bW1hcnkiLCJzdWdnZXN0ZWRRdWVzdGlvbnMiLCJBcnJheSIsImZyb20iLCJtYXRjaEFsbCIsIm1hcCIsIm0iXSwic291cmNlcyI6WyJyb3V0ZXMudHMiXSwic291cmNlc0NvbnRlbnQiOlsiLypcbiAqIENvcHlyaWdodCBPcGVuU2VhcmNoIENvbnRyaWJ1dG9yc1xuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqL1xuXG5pbXBvcnQgeyBzY2hlbWEgfSBmcm9tICdAb3NkL2NvbmZpZy1zY2hlbWEnO1xuaW1wb3J0IHtcbiAgSU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2UsXG4gIElSb3V0ZXIsXG4gIFJlc3BvbnNlRXJyb3IsXG59IGZyb20gJy4uLy4uLy4uLy4uLy4uL3NyYy9jb3JlL3NlcnZlcic7XG5pbXBvcnQgeyBpc1Jlc3BvbnNlRXJyb3IgfSBmcm9tICcuLi8uLi8uLi8uLi8uLi9zcmMvY29yZS9zZXJ2ZXIvb3BlbnNlYXJjaC9jbGllbnQvZXJyb3JzJztcbmltcG9ydCB7IEVSUk9SX0RFVEFJTFMsIFFVRVJZX0FTU0lTVF9BUEkgfSBmcm9tICcuLi8uLi8uLi9jb21tb24vY29uc3RhbnRzL3F1ZXJ5X2Fzc2lzdCc7XG5pbXBvcnQgeyBnZW5lcmF0ZUZpZWxkQ29udGV4dCB9IGZyb20gJy4uLy4uL2NvbW1vbi9oZWxwZXJzL3F1ZXJ5X2Fzc2lzdC9nZW5lcmF0ZV9maWVsZF9jb250ZXh0JztcbmltcG9ydCB7IGdldEFnZW50SWRCeUNvbmZpZywgZ2V0QWdlbnRJZEFuZFJlcXVlc3QgfSBmcm9tICcuL3V0aWxzL2FnZW50cyc7XG5pbXBvcnQgeyBBR0VOVF9DT05GSUdTIH0gZnJvbSAnLi91dGlscy9jb25zdGFudHMnO1xuXG5leHBvcnQgZnVuY3Rpb24gcmVnaXN0ZXJRdWVyeUFzc2lzdFJvdXRlcyhyb3V0ZXI6IElSb3V0ZXIpIHtcbiAgLyoqXG4gICAqIFJldHVybnMgd2hldGhlciB0aGUgUFBMIGFnZW50IGlzIGNvbmZpZ3VyZWQuXG4gICAqL1xuICByb3V0ZXIuZ2V0KFxuICAgIHtcbiAgICAgIHBhdGg6IFFVRVJZX0FTU0lTVF9BUEkuQ09ORklHVVJFRCxcbiAgICAgIHZhbGlkYXRlOiB7fSxcbiAgICB9LFxuICAgIGFzeW5jIChcbiAgICAgIGNvbnRleHQsXG4gICAgICByZXF1ZXN0LFxuICAgICAgcmVzcG9uc2VcbiAgICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueSB8IFJlc3BvbnNlRXJyb3I+PiA9PiB7XG4gICAgICBjb25zdCBjbGllbnQgPSBjb250ZXh0LmNvcmUub3BlbnNlYXJjaC5jbGllbnQuYXNDdXJyZW50VXNlcjtcbiAgICAgIHRyeSB7XG4gICAgICAgIC8vIGlmIHRoZSBjYWxsIGRvZXMgbm90IHRocm93IGFueSBlcnJvciwgdGhlbiB0aGUgYWdlbnQgaXMgcHJvcGVybHkgY29uZmlndXJlZFxuICAgICAgICBhd2FpdCBnZXRBZ2VudElkQnlDb25maWcoY2xpZW50LCBBR0VOVF9DT05GSUdTLlBQTF9BR0VOVCk7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7IGJvZHk6IHsgY29uZmlndXJlZDogdHJ1ZSB9IH0pO1xuICAgICAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHsgYm9keTogeyBjb25maWd1cmVkOiBmYWxzZSwgZXJyb3I6IGVycm9yLm1lc3NhZ2UgfSB9KTtcbiAgICAgIH1cbiAgICB9XG4gICk7XG5cbiAgcm91dGVyLnBvc3QoXG4gICAge1xuICAgICAgcGF0aDogUVVFUllfQVNTSVNUX0FQSS5HRU5FUkFURV9QUEwsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBpbmRleDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICAgIHF1ZXN0aW9uOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGFzeW5jIChcbiAgICAgIGNvbnRleHQsXG4gICAgICByZXF1ZXN0LFxuICAgICAgcmVzcG9uc2VcbiAgICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueSB8IFJlc3BvbnNlRXJyb3I+PiA9PiB7XG4gICAgICBjb25zdCBjbGllbnQgPSBjb250ZXh0LmNvcmUub3BlbnNlYXJjaC5jbGllbnQuYXNDdXJyZW50VXNlcjtcbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IHBwbFJlcXVlc3QgPSBhd2FpdCBnZXRBZ2VudElkQW5kUmVxdWVzdCh7XG4gICAgICAgICAgY2xpZW50LFxuICAgICAgICAgIGNvbmZpZ05hbWU6IEFHRU5UX0NPTkZJR1MuUFBMX0FHRU5ULFxuICAgICAgICAgIGJvZHk6IHtcbiAgICAgICAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAgICAgaW5kZXg6IHJlcXVlc3QuYm9keS5pbmRleCxcbiAgICAgICAgICAgICAgcXVlc3Rpb246IHJlcXVlc3QuYm9keS5xdWVzdGlvbixcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICAgIGlmICghcHBsUmVxdWVzdC5ib2R5LmluZmVyZW5jZV9yZXN1bHRzWzBdLm91dHB1dFswXS5yZXN1bHQpXG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdHZW5lcmF0ZWQgUFBMIHF1ZXJ5IG5vdCBmb3VuZC4nKTtcbiAgICAgICAgY29uc3QgcmVzdWx0ID0gSlNPTi5wYXJzZShwcGxSZXF1ZXN0LmJvZHkuaW5mZXJlbmNlX3Jlc3VsdHNbMF0ub3V0cHV0WzBdLnJlc3VsdCkgYXMge1xuICAgICAgICAgIHBwbDogc3RyaW5nO1xuICAgICAgICAgIGV4ZWN1dGlvblJlc3VsdDogc3RyaW5nO1xuICAgICAgICB9O1xuICAgICAgICBjb25zdCBwcGwgPSByZXN1bHQucHBsXG4gICAgICAgICAgLnJlcGxhY2UoL1tcXHJcXG5dL2csICcgJylcbiAgICAgICAgICAudHJpbSgpXG4gICAgICAgICAgLnJlcGxhY2UoL0lTTk9UTlVMTC9nLCAnaXNub3RudWxsJykgLy8gaHR0cHM6Ly9naXRodWIuY29tL29wZW5zZWFyY2gtcHJvamVjdC9zcWwvaXNzdWVzLzI0MzFcbiAgICAgICAgICAucmVwbGFjZSgvXFxiU1BBTlxcKC9nLCAnc3BhbignKTsgLy8gaHR0cHM6Ly9naXRodWIuY29tL29wZW5zZWFyY2gtcHJvamVjdC9kYXNoYm9hcmRzLW9ic2VydmFiaWxpdHkvaXNzdWVzLzc1OVxuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soeyBib2R5OiBwcGwgfSk7XG4gICAgICB9IGNhdGNoIChlcnJvcikge1xuICAgICAgICBpZiAoXG4gICAgICAgICAgaXNSZXNwb25zZUVycm9yKGVycm9yKSAmJlxuICAgICAgICAgIGVycm9yLnN0YXR1c0NvZGUgPT09IDQwMCAmJlxuICAgICAgICAgIC8vIG9uIG9wZW5zZWFyY2ggPj0gMi4xNywgZXJyb3IuYm9keSBpcyBhbiBvYmplY3QgaHR0cHM6Ly9naXRodWIuY29tL29wZW5zZWFyY2gtcHJvamVjdC9tbC1jb21tb25zL3B1bGwvMjg1OFxuICAgICAgICAgIEpTT04uc3RyaW5naWZ5KGVycm9yLmJvZHkpLmluY2x1ZGVzKEVSUk9SX0RFVEFJTFMuR1VBUkRSQUlMU19UUklHR0VSRUQpXG4gICAgICAgICkge1xuICAgICAgICAgIHJldHVybiByZXNwb25zZS5iYWRSZXF1ZXN0KHsgYm9keTogRVJST1JfREVUQUlMUy5HVUFSRFJBSUxTX1RSSUdHRVJFRCB9KTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBlcnJvci5zdGF0dXNDb2RlIHx8IDUwMCxcbiAgICAgICAgICBib2R5OiB0eXBlb2YgZXJyb3IuYm9keSA9PT0gJ3N0cmluZycgPyBlcnJvci5ib2R5IDogSlNPTi5zdHJpbmdpZnkoZXJyb3IuYm9keSksXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICByb3V0ZXIucG9zdChcbiAgICB7XG4gICAgICBwYXRoOiBRVUVSWV9BU1NJU1RfQVBJLlNVTU1BUklaRSxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIGJvZHk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIGluZGV4OiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgICAgcXVlc3Rpb246IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICBxdWVyeTogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoKSksXG4gICAgICAgICAgcmVzcG9uc2U6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICBpc0Vycm9yOiBzY2hlbWEuYm9vbGVhbigpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBhc3luYyAoXG4gICAgICBjb250ZXh0LFxuICAgICAgcmVxdWVzdCxcbiAgICAgIHJlc3BvbnNlXG4gICAgKTogUHJvbWlzZTxJT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZTxhbnkgfCBSZXNwb25zZUVycm9yPj4gPT4ge1xuICAgICAgY29uc3QgY2xpZW50ID0gY29udGV4dC5jb3JlLm9wZW5zZWFyY2guY2xpZW50LmFzQ3VycmVudFVzZXI7XG4gICAgICBjb25zdCB7IGluZGV4LCBxdWVzdGlvbiwgcXVlcnksIHJlc3BvbnNlOiBfcmVzcG9uc2UsIGlzRXJyb3IgfSA9IHJlcXVlc3QuYm9keTtcbiAgICAgIGNvbnN0IHF1ZXJ5UmVzcG9uc2UgPSBKU09OLnN0cmluZ2lmeShfcmVzcG9uc2UpO1xuICAgICAgbGV0IHN1bW1hcnlSZXF1ZXN0O1xuXG4gICAgICB0cnkge1xuICAgICAgICBpZiAoIWlzRXJyb3IpIHtcbiAgICAgICAgICBzdW1tYXJ5UmVxdWVzdCA9IGF3YWl0IGdldEFnZW50SWRBbmRSZXF1ZXN0KHtcbiAgICAgICAgICAgIGNsaWVudCxcbiAgICAgICAgICAgIGNvbmZpZ05hbWU6IEFHRU5UX0NPTkZJR1MuUkVTUE9OU0VfU1VNTUFSWV9BR0VOVCxcbiAgICAgICAgICAgIGJvZHk6IHtcbiAgICAgICAgICAgICAgcGFyYW1ldGVyczogeyBpbmRleCwgcXVlc3Rpb24sIHF1ZXJ5LCByZXNwb25zZTogcXVlcnlSZXNwb25zZSB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9KTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBjb25zdCBbbWFwcGluZ3MsIHNhbXBsZURvY10gPSBhd2FpdCBQcm9taXNlLmFsbChbXG4gICAgICAgICAgICBjbGllbnQuaW5kaWNlcy5nZXRNYXBwaW5nKHsgaW5kZXggfSksXG4gICAgICAgICAgICBjbGllbnQuc2VhcmNoKHsgaW5kZXgsIHNpemU6IDEgfSksXG4gICAgICAgICAgXSk7XG4gICAgICAgICAgY29uc3QgZmllbGRzID0gZ2VuZXJhdGVGaWVsZENvbnRleHQobWFwcGluZ3MsIHNhbXBsZURvYyk7XG4gICAgICAgICAgc3VtbWFyeVJlcXVlc3QgPSBhd2FpdCBnZXRBZ2VudElkQW5kUmVxdWVzdCh7XG4gICAgICAgICAgICBjbGllbnQsXG4gICAgICAgICAgICBjb25maWdOYW1lOiBBR0VOVF9DT05GSUdTLkVSUk9SX1NVTU1BUllfQUdFTlQsXG4gICAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICAgIHBhcmFtZXRlcnM6IHsgaW5kZXgsIHF1ZXN0aW9uLCBxdWVyeSwgcmVzcG9uc2U6IHF1ZXJ5UmVzcG9uc2UsIGZpZWxkcyB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBzdW1tYXJ5ID0gc3VtbWFyeVJlcXVlc3QuYm9keS5pbmZlcmVuY2VfcmVzdWx0c1swXS5vdXRwdXRbMF0ucmVzdWx0O1xuICAgICAgICBpZiAoIXN1bW1hcnkpIHRocm93IG5ldyBFcnJvcignR2VuZXJhdGVkIHN1bW1hcnkgbm90IGZvdW5kLicpO1xuICAgICAgICBjb25zdCBzdWdnZXN0ZWRRdWVzdGlvbnMgPSBBcnJheS5mcm9tKFxuICAgICAgICAgIChzdW1tYXJ5UmVxdWVzdC5ib2R5LmluZmVyZW5jZV9yZXN1bHRzWzBdLm91dHB1dFsxXT8ucmVzdWx0IHx8ICcnKS5tYXRjaEFsbChcbiAgICAgICAgICAgIC88cXVlc3Rpb24+KCgufFtcXHJcXG5dKSs/KTxcXC9xdWVzdGlvbj4vZ1xuICAgICAgICAgIClcbiAgICAgICAgKS5tYXAoKG0pID0+IChtIGFzIHVua25vd25bXSlbMV0pO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soe1xuICAgICAgICAgIGJvZHk6IHtcbiAgICAgICAgICAgIHN1bW1hcnksXG4gICAgICAgICAgICBzdWdnZXN0ZWRRdWVzdGlvbnMsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9IGNhdGNoIChlcnJvcikge1xuICAgICAgICBpZiAoXG4gICAgICAgICAgaXNSZXNwb25zZUVycm9yKGVycm9yKSAmJlxuICAgICAgICAgIGVycm9yLnN0YXR1c0NvZGUgPT09IDQwMCAmJlxuICAgICAgICAgIC8vIG9uIG9wZW5zZWFyY2ggPj0gMi4xNywgZXJyb3IuYm9keSBpcyBhbiBvYmplY3QgaHR0cHM6Ly9naXRodWIuY29tL29wZW5zZWFyY2gtcHJvamVjdC9tbC1jb21tb25zL3B1bGwvMjg1OFxuICAgICAgICAgIEpTT04uc3RyaW5naWZ5KGVycm9yLmJvZHkpLmluY2x1ZGVzKEVSUk9SX0RFVEFJTFMuR1VBUkRSQUlMU19UUklHR0VSRUQpXG4gICAgICAgICkge1xuICAgICAgICAgIHJldHVybiByZXNwb25zZS5iYWRSZXF1ZXN0KHsgYm9keTogRVJST1JfREVUQUlMUy5HVUFSRFJBSUxTX1RSSUdHRVJFRCB9KTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBlcnJvci5zdGF0dXNDb2RlIHx8IDUwMCxcbiAgICAgICAgICBib2R5OiB0eXBlb2YgZXJyb3IuYm9keSA9PT0gJ3N0cmluZycgPyBlcnJvci5ib2R5IDogSlNPTi5zdHJpbmdpZnkoZXJyb3IuYm9keSksXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcbn1cbiJdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBS0EsSUFBQUEsYUFBQSxHQUFBQyxPQUFBO0FBTUEsSUFBQUMsT0FBQSxHQUFBRCxPQUFBO0FBQ0EsSUFBQUUsYUFBQSxHQUFBRixPQUFBO0FBQ0EsSUFBQUcsdUJBQUEsR0FBQUgsT0FBQTtBQUNBLElBQUFJLE9BQUEsR0FBQUosT0FBQTtBQUNBLElBQUFLLFVBQUEsR0FBQUwsT0FBQTtBQWZBO0FBQ0E7QUFDQTtBQUNBOztBQWNPLFNBQVNNLHlCQUF5QkEsQ0FBQ0MsTUFBZSxFQUFFO0VBQ3pEO0FBQ0Y7QUFDQTtFQUNFQSxNQUFNLENBQUNDLEdBQUcsQ0FDUjtJQUNFQyxJQUFJLEVBQUVDLDhCQUFnQixDQUFDQyxVQUFVO0lBQ2pDQyxRQUFRLEVBQUUsQ0FBQztFQUNiLENBQUMsRUFDRCxPQUNFQyxPQUFPLEVBQ1BDLE9BQU8sRUFDUEMsUUFBUSxLQUN3RDtJQUNoRSxNQUFNQyxNQUFNLEdBQUdILE9BQU8sQ0FBQ0ksSUFBSSxDQUFDQyxVQUFVLENBQUNGLE1BQU0sQ0FBQ0csYUFBYTtJQUMzRCxJQUFJO01BQ0Y7TUFDQSxNQUFNLElBQUFDLDBCQUFrQixFQUFDSixNQUFNLEVBQUVLLHdCQUFhLENBQUNDLFNBQVMsQ0FBQztNQUN6RCxPQUFPUCxRQUFRLENBQUNRLEVBQUUsQ0FBQztRQUFFQyxJQUFJLEVBQUU7VUFBRUMsVUFBVSxFQUFFO1FBQUs7TUFBRSxDQUFDLENBQUM7SUFDcEQsQ0FBQyxDQUFDLE9BQU9DLEtBQUssRUFBRTtNQUNkLE9BQU9YLFFBQVEsQ0FBQ1EsRUFBRSxDQUFDO1FBQUVDLElBQUksRUFBRTtVQUFFQyxVQUFVLEVBQUUsS0FBSztVQUFFQyxLQUFLLEVBQUVBLEtBQUssQ0FBQ0M7UUFBUTtNQUFFLENBQUMsQ0FBQztJQUMzRTtFQUNGLENBQ0YsQ0FBQztFQUVEcEIsTUFBTSxDQUFDcUIsSUFBSSxDQUNUO0lBQ0VuQixJQUFJLEVBQUVDLDhCQUFnQixDQUFDbUIsWUFBWTtJQUNuQ2pCLFFBQVEsRUFBRTtNQUNSWSxJQUFJLEVBQUVNLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNsQkMsS0FBSyxFQUFFRixvQkFBTSxDQUFDRyxNQUFNLENBQUMsQ0FBQztRQUN0QkMsUUFBUSxFQUFFSixvQkFBTSxDQUFDRyxNQUFNLENBQUM7TUFDMUIsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQ0VwQixPQUFPLEVBQ1BDLE9BQU8sRUFDUEMsUUFBUSxLQUN3RDtJQUNoRSxNQUFNQyxNQUFNLEdBQUdILE9BQU8sQ0FBQ0ksSUFBSSxDQUFDQyxVQUFVLENBQUNGLE1BQU0sQ0FBQ0csYUFBYTtJQUMzRCxJQUFJO01BQ0YsTUFBTWdCLFVBQVUsR0FBRyxNQUFNLElBQUFDLDRCQUFvQixFQUFDO1FBQzVDcEIsTUFBTTtRQUNOcUIsVUFBVSxFQUFFaEIsd0JBQWEsQ0FBQ0MsU0FBUztRQUNuQ0UsSUFBSSxFQUFFO1VBQ0pjLFVBQVUsRUFBRTtZQUNWTixLQUFLLEVBQUVsQixPQUFPLENBQUNVLElBQUksQ0FBQ1EsS0FBSztZQUN6QkUsUUFBUSxFQUFFcEIsT0FBTyxDQUFDVSxJQUFJLENBQUNVO1VBQ3pCO1FBQ0Y7TUFDRixDQUFDLENBQUM7TUFDRixJQUFJLENBQUNDLFVBQVUsQ0FBQ1gsSUFBSSxDQUFDZSxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsQ0FBQ0MsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDQyxNQUFNLEVBQ3hELE1BQU0sSUFBSUMsS0FBSyxDQUFDLGdDQUFnQyxDQUFDO01BQ25ELE1BQU1ELE1BQU0sR0FBR0UsSUFBSSxDQUFDQyxLQUFLLENBQUNULFVBQVUsQ0FBQ1gsSUFBSSxDQUFDZSxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsQ0FBQ0MsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDQyxNQUFNLENBRzlFO01BQ0QsTUFBTUksR0FBRyxHQUFHSixNQUFNLENBQUNJLEdBQUcsQ0FDbkJDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsR0FBRyxDQUFDLENBQ3ZCQyxJQUFJLENBQUMsQ0FBQyxDQUNORCxPQUFPLENBQUMsWUFBWSxFQUFFLFdBQVcsQ0FBQyxDQUFDO01BQUEsQ0FDbkNBLE9BQU8sQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQztNQUNsQyxPQUFPL0IsUUFBUSxDQUFDUSxFQUFFLENBQUM7UUFBRUMsSUFBSSxFQUFFcUI7TUFBSSxDQUFDLENBQUM7SUFDbkMsQ0FBQyxDQUFDLE9BQU9uQixLQUFLLEVBQUU7TUFDZCxJQUNFLElBQUFzQix1QkFBZSxFQUFDdEIsS0FBSyxDQUFDLElBQ3RCQSxLQUFLLENBQUN1QixVQUFVLEtBQUssR0FBRztNQUN4QjtNQUNBTixJQUFJLENBQUNPLFNBQVMsQ0FBQ3hCLEtBQUssQ0FBQ0YsSUFBSSxDQUFDLENBQUMyQixRQUFRLENBQUNDLDJCQUFhLENBQUNDLG9CQUFvQixDQUFDLEVBQ3ZFO1FBQ0EsT0FBT3RDLFFBQVEsQ0FBQ3VDLFVBQVUsQ0FBQztVQUFFOUIsSUFBSSxFQUFFNEIsMkJBQWEsQ0FBQ0M7UUFBcUIsQ0FBQyxDQUFDO01BQzFFO01BQ0EsT0FBT3RDLFFBQVEsQ0FBQ3dDLE1BQU0sQ0FBQztRQUNyQk4sVUFBVSxFQUFFdkIsS0FBSyxDQUFDdUIsVUFBVSxJQUFJLEdBQUc7UUFDbkN6QixJQUFJLEVBQUUsT0FBT0UsS0FBSyxDQUFDRixJQUFJLEtBQUssUUFBUSxHQUFHRSxLQUFLLENBQUNGLElBQUksR0FBR21CLElBQUksQ0FBQ08sU0FBUyxDQUFDeEIsS0FBSyxDQUFDRixJQUFJO01BQy9FLENBQUMsQ0FBQztJQUNKO0VBQ0YsQ0FDRixDQUFDO0VBRURqQixNQUFNLENBQUNxQixJQUFJLENBQ1Q7SUFDRW5CLElBQUksRUFBRUMsOEJBQWdCLENBQUM4QyxTQUFTO0lBQ2hDNUMsUUFBUSxFQUFFO01BQ1JZLElBQUksRUFBRU0sb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ2xCQyxLQUFLLEVBQUVGLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQ3RCQyxRQUFRLEVBQUVKLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQ3pCd0IsS0FBSyxFQUFFM0Isb0JBQU0sQ0FBQzRCLEtBQUssQ0FBQzVCLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDcENsQixRQUFRLEVBQUVlLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQ3pCMEIsT0FBTyxFQUFFN0Isb0JBQU0sQ0FBQzhCLE9BQU8sQ0FBQztNQUMxQixDQUFDO0lBQ0g7RUFDRixDQUFDLEVBQ0QsT0FDRS9DLE9BQU8sRUFDUEMsT0FBTyxFQUNQQyxRQUFRLEtBQ3dEO0lBQ2hFLE1BQU1DLE1BQU0sR0FBR0gsT0FBTyxDQUFDSSxJQUFJLENBQUNDLFVBQVUsQ0FBQ0YsTUFBTSxDQUFDRyxhQUFhO0lBQzNELE1BQU07TUFBRWEsS0FBSztNQUFFRSxRQUFRO01BQUV1QixLQUFLO01BQUUxQyxRQUFRLEVBQUU4QyxTQUFTO01BQUVGO0lBQVEsQ0FBQyxHQUFHN0MsT0FBTyxDQUFDVSxJQUFJO0lBQzdFLE1BQU1zQyxhQUFhLEdBQUduQixJQUFJLENBQUNPLFNBQVMsQ0FBQ1csU0FBUyxDQUFDO0lBQy9DLElBQUlFLGNBQWM7SUFFbEIsSUFBSTtNQUFBLElBQUFDLHFCQUFBO01BQ0YsSUFBSSxDQUFDTCxPQUFPLEVBQUU7UUFDWkksY0FBYyxHQUFHLE1BQU0sSUFBQTNCLDRCQUFvQixFQUFDO1VBQzFDcEIsTUFBTTtVQUNOcUIsVUFBVSxFQUFFaEIsd0JBQWEsQ0FBQzRDLHNCQUFzQjtVQUNoRHpDLElBQUksRUFBRTtZQUNKYyxVQUFVLEVBQUU7Y0FBRU4sS0FBSztjQUFFRSxRQUFRO2NBQUV1QixLQUFLO2NBQUUxQyxRQUFRLEVBQUUrQztZQUFjO1VBQ2hFO1FBQ0YsQ0FBQyxDQUFDO01BQ0osQ0FBQyxNQUFNO1FBQ0wsTUFBTSxDQUFDSSxRQUFRLEVBQUVDLFNBQVMsQ0FBQyxHQUFHLE1BQU1DLE9BQU8sQ0FBQ0MsR0FBRyxDQUFDLENBQzlDckQsTUFBTSxDQUFDc0QsT0FBTyxDQUFDQyxVQUFVLENBQUM7VUFBRXZDO1FBQU0sQ0FBQyxDQUFDLEVBQ3BDaEIsTUFBTSxDQUFDd0QsTUFBTSxDQUFDO1VBQUV4QyxLQUFLO1VBQUV5QyxJQUFJLEVBQUU7UUFBRSxDQUFDLENBQUMsQ0FDbEMsQ0FBQztRQUNGLE1BQU1DLE1BQU0sR0FBRyxJQUFBQyw0Q0FBb0IsRUFBQ1QsUUFBUSxFQUFFQyxTQUFTLENBQUM7UUFDeERKLGNBQWMsR0FBRyxNQUFNLElBQUEzQiw0QkFBb0IsRUFBQztVQUMxQ3BCLE1BQU07VUFDTnFCLFVBQVUsRUFBRWhCLHdCQUFhLENBQUN1RCxtQkFBbUI7VUFDN0NwRCxJQUFJLEVBQUU7WUFDSmMsVUFBVSxFQUFFO2NBQUVOLEtBQUs7Y0FBRUUsUUFBUTtjQUFFdUIsS0FBSztjQUFFMUMsUUFBUSxFQUFFK0MsYUFBYTtjQUFFWTtZQUFPO1VBQ3hFO1FBQ0YsQ0FBQyxDQUFDO01BQ0o7TUFDQSxNQUFNRyxPQUFPLEdBQUdkLGNBQWMsQ0FBQ3ZDLElBQUksQ0FBQ2UsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLENBQUNDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQ0MsTUFBTTtNQUN6RSxJQUFJLENBQUNvQyxPQUFPLEVBQUUsTUFBTSxJQUFJbkMsS0FBSyxDQUFDLDhCQUE4QixDQUFDO01BQzdELE1BQU1vQyxrQkFBa0IsR0FBR0MsS0FBSyxDQUFDQyxJQUFJLENBQ25DLENBQUMsRUFBQWhCLHFCQUFBLEdBQUFELGNBQWMsQ0FBQ3ZDLElBQUksQ0FBQ2UsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLENBQUNDLE1BQU0sQ0FBQyxDQUFDLENBQUMsY0FBQXdCLHFCQUFBLHVCQUFsREEscUJBQUEsQ0FBb0R2QixNQUFNLEtBQUksRUFBRSxFQUFFd0MsUUFBUSxDQUN6RSx1Q0FDRixDQUNGLENBQUMsQ0FBQ0MsR0FBRyxDQUFFQyxDQUFDLElBQU1BLENBQUMsQ0FBZSxDQUFDLENBQUMsQ0FBQztNQUNqQyxPQUFPcEUsUUFBUSxDQUFDUSxFQUFFLENBQUM7UUFDakJDLElBQUksRUFBRTtVQUNKcUQsT0FBTztVQUNQQztRQUNGO01BQ0YsQ0FBQyxDQUFDO0lBQ0osQ0FBQyxDQUFDLE9BQU9wRCxLQUFLLEVBQUU7TUFDZCxJQUNFLElBQUFzQix1QkFBZSxFQUFDdEIsS0FBSyxDQUFDLElBQ3RCQSxLQUFLLENBQUN1QixVQUFVLEtBQUssR0FBRztNQUN4QjtNQUNBTixJQUFJLENBQUNPLFNBQVMsQ0FBQ3hCLEtBQUssQ0FBQ0YsSUFBSSxDQUFDLENBQUMyQixRQUFRLENBQUNDLDJCQUFhLENBQUNDLG9CQUFvQixDQUFDLEVBQ3ZFO1FBQ0EsT0FBT3RDLFFBQVEsQ0FBQ3VDLFVBQVUsQ0FBQztVQUFFOUIsSUFBSSxFQUFFNEIsMkJBQWEsQ0FBQ0M7UUFBcUIsQ0FBQyxDQUFDO01BQzFFO01BQ0EsT0FBT3RDLFFBQVEsQ0FBQ3dDLE1BQU0sQ0FBQztRQUNyQk4sVUFBVSxFQUFFdkIsS0FBSyxDQUFDdUIsVUFBVSxJQUFJLEdBQUc7UUFDbkN6QixJQUFJLEVBQUUsT0FBT0UsS0FBSyxDQUFDRixJQUFJLEtBQUssUUFBUSxHQUFHRSxLQUFLLENBQUNGLElBQUksR0FBR21CLElBQUksQ0FBQ08sU0FBUyxDQUFDeEIsS0FBSyxDQUFDRixJQUFJO01BQy9FLENBQUMsQ0FBQztJQUNKO0VBQ0YsQ0FDRixDQUFDO0FBQ0gifQ==