"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processTaskError = exports.isIndexNotFoundError = exports.getTaskState = exports.getTaskInitProgress = exports.getLatestTaskForForecasterQuery = exports.getLatestForecasterTasksQuery = exports.getErrorMessage = exports.convertTaskAndJobFieldsToCamelCase = exports.convertStaticFieldsToCamelCase = exports.convertForecastKeysToSnakeCase = exports.convertForecastKeysToCamelCase = exports.combineTaskState = exports.buildEntityListQuery = void 0;
var _lodash = require("lodash");
var _constants = require("../../utils/constants");
var _helpers = require("../../utils/helpers");
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

const convertForecastKeysToSnakeCase = payload => {
  return {
    ...(0, _helpers.mapKeysDeep)({
      ...(0, _lodash.omit)(payload, ['filterQuery', 'featureAttributes']) // Exclude the filterQuery,
    }, _helpers.toSnake),
    filter_query: (0, _lodash.get)(payload, 'filterQuery', {}),
    ui_metadata: (0, _lodash.get)(payload, 'uiMetadata', {}),
    feature_attributes: (0, _lodash.get)(payload, 'featureAttributes', []).map(feature => ({
      ...(0, _helpers.mapKeysDeep)({
        ...(0, _lodash.omit)(feature, ['aggregationQuery'])
      }, _helpers.toSnake),
      aggregation_query: feature.aggregationQuery
    }))
  };
};
exports.convertForecastKeysToSnakeCase = convertForecastKeysToSnakeCase;
const processTaskError = error => {
  const errorWithPrefixRemoved = error.replace(_constants.OPENSEARCH_EXCEPTION_PREFIX, '');
  return (0, _lodash.isEmpty)(errorWithPrefixRemoved) || errorWithPrefixRemoved.endsWith('.') ? errorWithPrefixRemoved : errorWithPrefixRemoved + '.';
};

// Following checks/transformations need to be made here:
// - set to INACTIVE_NOT_STARTED if there is no existing task for this forecaster
// - set to UNEXPECTED_FAILURE if the task is in a FAILED state & the error message is unreadable / is a stack trace
// - set to INITIALIZING_FORECAST if the task is in a CREATED state
// - set to INACTIVE_STOPPED if the task is in a STOPPED state
// - override INACTIVE states (INACTIVE_STOPPED or INACTIVE_NOT_STARTED) with test-related states
//   (INITIALIZING_TEST, TEST_COMPLETE, or INIT_TEST_FAILURE) if runOnceTask exists with these states
exports.processTaskError = processTaskError;
const combineTaskState = (realTimeTask, runOnceTask) => {
  console.log('realTimeTask', realTimeTask);
  console.log('runOnceTask', runOnceTask);
  const realTimeState = (0, _lodash.get)(realTimeTask, 'state', 'INACTIVE_NOT_STARTED');
  const updatedStateString = realTimeState === 'CREATED' ? 'INITIALIZING_FORECAST' : realTimeState === 'STOPPED' ? 'INACTIVE_STOPPED' : realTimeState;
  //@ts-ignore
  let updatedState = _constants.FORECASTER_STATE[updatedStateString];
  console.log('updatedState', updatedState);

  // at the beginning, runOnceTask is inactive before going into initializing test state
  // if runOnceTask is not empty and the error is empty, set the state to INIT_TEST
  const runOnceState = (0, _lodash.get)(runOnceTask, 'state', 'INACTIVE_NOT_STARTED');
  const runOnceStateError = (0, _lodash.get)(runOnceTask, 'error', '');
  const updatedRunOnceStateString = runOnceState === 'INACTIVE' && runOnceStateError === '' ? 'INIT_TEST' : runOnceState;
  console.log('updatedRunOnceStateString', updatedRunOnceStateString);
  const realTimeLastUpdateTime = (0, _lodash.get)(realTimeTask, 'last_update_time', 0);
  const runOnceLastUpdateTime = (0, _lodash.get)(runOnceTask, 'last_update_time', 0);

  // Check if current state is INACTIVE and runOnceTask has a priority state
  // when realTimeTask is not updated for a while
  if ((updatedState === _constants.FORECASTER_STATE.INACTIVE_STOPPED || updatedState === _constants.FORECASTER_STATE.INACTIVE_NOT_STARTED) && runOnceTask && runOnceLastUpdateTime > realTimeLastUpdateTime) {
    // Convert runOnceState to enum value before comparison
    const runOnceStateEnum = _constants.FORECASTER_STATE[updatedRunOnceStateString];
    if (runOnceStateEnum === _constants.FORECASTER_STATE.INIT_TEST || runOnceStateEnum === _constants.FORECASTER_STATE.TEST_COMPLETE || runOnceStateEnum === _constants.FORECASTER_STATE.INIT_TEST_FAILED) {
      updatedState = runOnceStateEnum;
    }
  }
  return updatedState;
};
exports.combineTaskState = combineTaskState;
const getTaskState = realTimeTask => {
  const state = (0, _lodash.get)(realTimeTask, 'state', 'INACTIVE_NOT_STARTED');
  const updatedStateString = state === 'CREATED' ? 'INITIALIZING_FORECAST' : state === 'STOPPED' ? 'INACTIVE_STOPPED' : state;
  //@ts-ignore
  return _constants.FORECASTER_STATE[updatedStateString];
};
exports.getTaskState = getTaskState;
const convertForecastKeysToCamelCase = response => {
  let camelCaseResponse = {
    ...(0, _helpers.mapKeysDeep)((0, _lodash.omit)(response, ['filter_query', 'ui_metadata', 'feature_query', 'feature_attributes', 'forecasterJob', 'runOnceTask']), _helpers.toCamel),
    filterQuery: (0, _lodash.get)(response, 'filter_query', {}),
    featureAttributes: (0, _lodash.get)(response, 'feature_attributes', []).map(feature => ({
      ...(0, _helpers.mapKeysDeep)({
        ...(0, _lodash.omit)(feature, ['aggregation_query'])
      }, _helpers.toCamel),
      aggregationQuery: feature.aggregation_query
    })),
    uiMetadata: (0, _lodash.get)(response, 'ui_metadata', {}),
    enabled: (0, _lodash.get)(response, 'adJob.enabled', false),
    enabledTime: (0, _lodash.get)(response, 'adJob.enabled_time'),
    disabledTime: (0, _lodash.get)(response, 'adJob.disabled_time'),
    categoryField: (0, _lodash.get)(response, 'category_field')
  };
  if (!(0, _lodash.isEmpty)((0, _lodash.get)(response, 'runOnceTask', {}))) {
    camelCaseResponse = {
      ...camelCaseResponse,
      //@ts-ignore
      taskId: (0, _lodash.get)(response, 'runOnceTask.task_id'),
      taskState: getTaskState((0, _lodash.get)(response, 'runOnceTask', {})),
      taskProgress: (0, _lodash.get)(response, 'runOnceTask.task_progress'),
      taskError: processTaskError((0, _lodash.get)(response, 'runOnceTask.error', '')),
      detectionDateRange: {
        startTime: (0, _lodash.get)(response, 'historicalTask.detection_date_range.start_time'),
        endTime: (0, _lodash.get)(response, 'historicalTask.detection_date_range.end_time')
      }
    };
  }
  return camelCaseResponse;
};
exports.convertForecastKeysToCamelCase = convertForecastKeysToCamelCase;
const isIndexNotFoundError = err => {
  return err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception';
};
exports.isIndexNotFoundError = isIndexNotFoundError;
const getErrorMessage = err => {
  return !(0, _lodash.isEmpty)((0, _lodash.get)(err, 'body.error.reason')) ? (0, _lodash.get)(err, 'body.error.reason') : (0, _lodash.get)(err, 'message');
};

// Filtering by 'is_latest=true' is not enough here. It is possible that multiple realtime
// tasks with 'is_latest=true' are created. We sort by latest execution_start_time
// (which is equivalent to it's creation timestamp), and only return the latest one.
exports.getErrorMessage = getErrorMessage;
const getLatestForecasterTasksQuery = realtime => {
  const taskTypes = realtime ? _constants.FORECAST_REALTIME_TASK_TYPES : _constants.FORECAST_RUN_ONCE_TASK_TYPES;
  return {
    size: 0,
    query: {
      bool: {
        filter: [{
          term: {
            is_latest: 'true'
          }
        }, {
          terms: {
            task_type: taskTypes
          }
        }]
      }
    },
    aggs: {
      forecasters: {
        terms: {
          field: 'forecaster_id',
          size: _constants.MAX_FORECASTER
        },
        aggs: {
          latest_tasks: {
            top_hits: {
              size: 1,
              sort: {
                execution_start_time: _constants.SORT_DIRECTION.DESC
              }
            }
          }
        }
      }
    }
  };
};

// Converts the static forecaster fields into camelcase. Ignores any job or task-related fields
exports.getLatestForecasterTasksQuery = getLatestForecasterTasksQuery;
const convertStaticFieldsToCamelCase = response => {
  return {
    ...(0, _helpers.mapKeysDeep)((0, _lodash.omit)(response, ['filter_query', 'feature_query', 'feature_attributes', 'ui_metadata', 'forecaster_job', 'realtime_task', 'run_once_task']), _helpers.toCamel),
    filterQuery: (0, _lodash.get)(response, 'filter_query', {}),
    featureAttributes: (0, _lodash.get)(response, 'feature_attributes', []).map(feature => ({
      ...(0, _helpers.mapKeysDeep)({
        ...(0, _lodash.omit)(feature, ['aggregation_query'])
      }, _helpers.toCamel),
      aggregationQuery: feature.aggregation_query
    })),
    uiMetadata: (0, _lodash.get)(response, 'ui_metadata', {}),
    lastUiBreakingChangeTime: (0, _lodash.get)(response, 'last_ui_breaking_change_time'),
    lastUpdateTime: (0, _lodash.get)(response, 'last_update_time')
  };
};

// Filtering by 'is_latest=true' is not enough here. It is possible that multiple realtime
// tasks with 'is_latest=true' are created. We sort by latest execution_start_time
// (which is equivalent to it's creation timestamp), and only return the latest one.
exports.convertStaticFieldsToCamelCase = convertStaticFieldsToCamelCase;
const getLatestTaskForForecasterQuery = (forecasterId, realtime) => {
  const taskTypes = realtime ? _constants.FORECAST_REALTIME_TASK_TYPES : _constants.FORECAST_RUN_ONCE_TASK_TYPES;
  return {
    size: 1,
    sort: {
      execution_start_time: _constants.SORT_DIRECTION.DESC
    },
    query: {
      bool: {
        filter: [{
          term: {
            forecaster_id: forecasterId
          }
        }, {
          term: {
            is_latest: 'true'
          }
        }, {
          terms: {
            task_type: taskTypes
          }
        }]
      }
    }
  };
};
exports.getLatestTaskForForecasterQuery = getLatestTaskForForecasterQuery;
const getTaskInitProgress = task => {
  if ((task === null || task === void 0 ? void 0 : task.init_progress) !== undefined) {
    return {
      percentageStr: `${((0, _lodash.get)(task, 'init_progress', 0) * 100).toFixed(0)}%`,
      estimatedMinutesLeft: task.estimated_minutes_left
    };
  }
  return undefined;
};

// Converts the task-related detector fields into camelcase
exports.getTaskInitProgress = getTaskInitProgress;
const convertTaskAndJobFieldsToCamelCase = (realtimeTask, runOnceTask, forecasterJob) => {
  // Populate Forecaster job fields
  return {
    enabled: (0, _lodash.get)(forecasterJob, 'enabled', false),
    enabledTime: (0, _lodash.get)(forecasterJob, 'enabled_time'),
    disabledTime: (0, _lodash.get)(forecasterJob, 'disabled_time'),
    curState: combineTaskState(realtimeTask, runOnceTask),
    stateError: (0, _lodash.get)(realtimeTask, 'error') || (0, _lodash.get)(runOnceTask, 'error'),
    initProgress: getTaskInitProgress(realtimeTask),
    realTimeLastUpdateTime: (0, _lodash.get)(realtimeTask, 'last_update_time'),
    runOnceLastUpdateTime: (0, _lodash.get)(runOnceTask, 'last_update_time'),
    taskId: (0, _lodash.get)(runOnceTask, 'task_id'),
    taskState: getTaskState(runOnceTask),
    taskProgress: (0, _lodash.get)(runOnceTask, 'task_progress'),
    taskError: processTaskError((0, _lodash.get)(runOnceTask, 'error', ''))
  };
};

/**
 * Builds an OpenSearch query for matching multiple ForecastEntity objects
 * Each entity's key-value pairs are grouped with "must" clauses
 * Different entities are combined with "should" clauses
 */
exports.convertTaskAndJobFieldsToCamelCase = convertTaskAndJobFieldsToCamelCase;
const buildEntityListQuery = entityList => {
  if (!entityList || entityList.length === 0) {
    return undefined;
  }

  // Create an array of bool queries - one for each entity
  const shouldClauses = entityList.map(entity => {
    // For each entity, create nested queries for each key-value pair
    const mustClauses = Object.entries(entity).map(([name, value]) => {
      return {
        nested: {
          path: "entity",
          query: {
            bool: {
              must: [{
                term: {
                  "entity.name": name
                }
              }, {
                term: {
                  "entity.value": value
                }
              }]
            }
          },
          ignore_unmapped: false,
          score_mode: "avg"
        }
      };
    });

    // All key-value pairs for this entity must match
    return {
      bool: {
        must: mustClauses
      }
    };
  });

  // At least one entity should match
  return {
    bool: {
      should: shouldClauses,
      minimum_should_match: 1
    }
  };
};
exports.buildEntityListQuery = buildEntityListQuery;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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