/*******************************************************************************
* Copyright 2013-2022 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*   Content : Intel(R) oneAPI Math Kernel Library (oneMKL) IE Sparse BLAS C
*             example for CSR format
*
********************************************************************************
*
* Example program for using Intel oneMKL Inspector-Executor Sparse BLAS routines
* for matrices represented in the compressed sparse row (CSR) sparse storage format.
*
* The following Inspector Executor Sparse Blas routines are used in the example:
*
*   Initialization/Destruction stage:
*          mkl_sparse_d_create_csr
*          mkl_sparse_destroy
*
*   Inspector stage:
*          mkl_sparse_set_mv_hint  mkl_sparse_set_sv_hint
*          mkl_sparse_set_mm_hint  mkl_sparse_set_sm_hint
*          mkl_sparse_optimize
*
*   Executor stage:
*          mkl_sparse_d_mv         mkl_sparse_d_trsv
*          mkl_sparse_d_mm         mkl_sparse_d_trsm
*
* Consider the matrix A (see 'Sparse Storage Formats for Sparse BLAS Level 2
* and Level 3 in the  Intel oneMKL Reference Manual')
*
*                 |   1       -1      0   -3     0   |
*                 |  -2        5      0    0     0   |
*   A    =        |   0        0      4    6     4   |,
*                 |  -4        0      2    7     0   |
*                 |   0        8      0    0    -5   |
*
* decomposed as
*
*                      A = L + D + U,
*
*  where L is the strict  lower triangle of A, U is the strictly  upper triangle
*  of A, D is the main diagonal. Namely
*
*        |   0    0   0    0     0   |       |  0   -1    0   -3   0   |
*        |  -2    0   0    0     0   |       |  0    0    0    0   0   |
*   L  = |   0    0   0    0     0   |,  U=  |  0    0    0    6   4   |
*        |  -4    0   2    0     0   |       |  0    0    0    0   0   |
*        |   0    8   0    0     0   |       |  0    0    0    0   0   |
*
*
*           |   1  0  0   0   0   |
*           |   0  5  0   0   0   |
*   D    =  |   0  0  4   0   0   |.
*           |   0  0  0   7   0   |
*           |   0  0  0   0  -5   |
*
*
*  The matrix A is represented in a zero-based compressed sparse row (CSR) storage
*  scheme with three arrays (see 'Sparse Matrix Storage Schemes' in the
*   Intel oneMKL Reference Manual) as follows:
*
*         rowPtr  = ( 0        3     5        8       11    13 )
*         columns = ( 0  1  3  0  1  2  3  4  0  2  3  1  4 )
*         values  = ( 1 -1 -3 -2  5  4  6  4 -4  2  7  8 -5 )
*
*  In what follows the symbol ' means transposition of object preceding the symbol.
*
*  Using notation A = (L + D + U) for provided matrix elements and I for identity
*  matrix, the test performs the following operations :
*
*       1. The example computes (L+D)'*x_v_original = y_v using mkl_sparse_d_mv where x_v_original is 
*          a known vector of size 5 and then the example solves the system 
*          (L+D)'*x_v_calculated = y_v with the help of mkl_sparse_d_trsv. It's evident 
*          that x_v_calculated should be equal to x_v_original.
*
*       2. The example computes (U+I)'*x_v_original = y_v using mkl_sparse_d_mv where x_v_original is 
*          a known vector of size 5 and then the example solves the system 
*          (U+I)'*x_v_calculated = y_v with the help of mkl_sparse_d_trsv. It's evident 
*          that x_v_calculated should be equal to x_v_original.
*
*       3. The example computes A'*x_m_original = y_m using mkl_sparse_d_mm where x_m_original is a 
*          known 5 by 2 matrix with column major ordering.
*
*       4. The example computes A*x_m_original = y_m using mkl_sparse_d_mm where x_m_original is a 
*          known 5 by 2 matrix with row major ordering.
*
*       5. The example computes (L+D)'*x_m_original = y_m using mkl_sparse_d_mm where x_m_original 
*          is a known 5 by 2 matrix and then the example solves the system 
*          (L+D)'*x_m_calculated = y_m with the help of mkl_sparse_d_trsm. It's evident
*          that x_m_calculated should be equal to x_m_original.
*
* The code given below uses only one sparse representation for the operations.
********************************************************************************
*/
#include <stdio.h>
#include <assert.h>
#include <math.h>
#include "mkl_spblas.h"

#define ALMOST_EQUAL(a, b) (fabs((a)-(b)) < 1e-10 ? 1 : 0)

int main() {
    //*******************************************************************************
    //     Declaration and initialization of parameters for sparse representation of
    //     the matrix A in the CSR format:
    //*******************************************************************************
#define M 5
#define NNZ 13
#define NRHS 2

    MKL_INT m = M, nrhs = NRHS;

    //*******************************************************************************
    //    Sparse representation of the matrix A
    //*******************************************************************************

    MKL_INT rowPtr[M+1] = { 0, 3, 5, 8, 11, 13 };

    MKL_INT columns[NNZ]   = { 0,      1,        3,
                               0,      1,
                                            2,   3,   4,
                               0,           2,   3,
                                       1,             4 };

    double values[NNZ]     = { 1.0, -1.0,     -3.0,
                              -2.0,  5.0,
                                          4.0, 6.0, 4.0,
                              -4.0,       2.0, 7.0,
                                     8.0,          -5.0 };


    // Descriptor of main sparse matrix properties
    struct matrix_descr descrA;
    // Structure with sparse matrix stored in CSR format
    sparse_matrix_t       csrA;
    //*******************************************************************************
    //    Declaration of local variables:
    //*******************************************************************************
    double x_m_original[M*NRHS]   = { 1.0, 5.0, 3.0, 4.0, 2.0, 2.0, 10.0, 6.0, 8.0, 4.0};
    double y_m[M*NRHS]            = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    double x_m_calculated[M*NRHS] = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    double x_v_original[M]        = { 3.0, 2.0, 5.0, 4.0, 1.0};
    double y_v[M]                 = { 0.0, 0.0, 0.0, 0.0, 0.0};
    double x_v_calculated[M]      = { 0.0, 0.0, 0.0, 0.0, 0.0};
    double alpha = 1.0, beta = 0.0;
    MKL_INT    i, j;

    sparse_status_t status;
    int exit_status = 0;

    printf( "\n EXAMPLE PROGRAM FOR CSR format routines from IE Sparse BLAS\n" );
    printf( "-------------------------------------------------------\n" );

//*******************************************************************************
//   Create CSR sparse matrix handle and analyze step
//*******************************************************************************

    // Create handle with matrix stored in CSR format
    status = mkl_sparse_d_create_csr ( &csrA,
                                       SPARSE_INDEX_BASE_ZERO,
                                       m,  // number of rows
                                       m,  // number of cols
                                       rowPtr,
                                       rowPtr+1,
                                       columns,
                                       values );


    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_d_create_csr: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //*******************************************************************************
    // First we set hints for the different operations before calling the
    // mkl_sparse_optimize() api which actually does the analyze step.  Not all
    // configurations have optimized steps, so the hint apis may return status
    // MKL_SPARSE_STATUS_NOT_SUPPORTED (=6) if no analysis stage is actually available
    // for that configuration.
    //*******************************************************************************

    //*******************************************************************************
    // Set hints for Task 1: Lower triangular transpose MV and SV solve with
    // non-unit diagonal
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_LOWER;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_set_mv_hint(csrA, SPARSE_OPERATION_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 1: mkl_sparse_set_mv_hint: %d \n", status);
    }

    status = mkl_sparse_set_sv_hint(csrA, SPARSE_OPERATION_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 1: mkl_sparse_set_mv_hint: %d \n", status);
    }

    //*******************************************************************************
    // Set hints for Task 2: Upper triangular non-transpose MV and SV solve with
    // unit diagonal
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_UPPER;
    descrA.diag = SPARSE_DIAG_UNIT;

    status = mkl_sparse_set_mv_hint(csrA, SPARSE_OPERATION_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 2: mkl_sparse_set_mv_hint: %d \n", status);
    }

    //*******************************************************************************
    // Set hints for Task 3: General matrix (transpose sparse) * dense MM  with
    // non-unit diagonal and column-major format
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;

    status = mkl_sparse_set_mm_hint(csrA, SPARSE_OPERATION_TRANSPOSE, descrA,
                                    SPARSE_LAYOUT_COLUMN_MAJOR, nrhs, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 3: mkl_sparse_set_mm_hint: %d \n", status);
    }

    //*******************************************************************************
    // Set hints for Task 4: General matrix sparse * dense MM  with
    // non-unit diagonal and row-major format
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;

    status = mkl_sparse_set_mm_hint(csrA, SPARSE_OPERATION_NON_TRANSPOSE, descrA,
                                    SPARSE_LAYOUT_ROW_MAJOR, nrhs, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 4: mkl_sparse_set_mm_hint: %d \n", status);
    }

    //*******************************************************************************
    // Set hints for Task 5: Lower triangular transpose MM and SM solve with
    // non-unit diagonal and row-major format
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_LOWER;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_set_mm_hint(csrA, SPARSE_OPERATION_TRANSPOSE, descrA,
                                    SPARSE_LAYOUT_ROW_MAJOR, nrhs, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 5: mkl_sparse_set_mm_hint: %d \n", status);
    }

    status = mkl_sparse_set_sm_hint(csrA, SPARSE_OPERATION_TRANSPOSE, descrA,
                                    SPARSE_LAYOUT_ROW_MAJOR, nrhs, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 5: mkl_sparse_set_sm_hint: %d \n", status);
    }

    //*******************************************************************************
    // Analyze sparse matrix; choose proper kernels and workload balancing strategy
    //*******************************************************************************
    status = mkl_sparse_optimize ( csrA );
    if (status != SPARSE_STATUS_SUCCESS) {
      printf(" Error in mkl_sparse_optimize: %d \n", status);
      exit_status = 1;
      goto exit;
    }

//*******************************************************************************
//  Task 1: Obtain matrix-matrix multiply (L+D)' *x_v_original --> y_v
//          and solve triangular system   (L+D)' *x_v_calculated = y_v
//          Array x_v_calculated must be equal to the array x_v_original
//*******************************************************************************
    printf( "-------------------------------------------------------\n" );
    printf( "                                  \n" );
    printf( "   Task 1:                        \n" );
    printf( "   INPUT DATA FOR mkl_sparse_d_mv \n" );
    printf( "   WITH TRIANGULAR SPARSE MATRIX  \n" );
    printf( "   ALPHA = %4.1f  BETA = %4.1f    \n", alpha, beta );
    printf( "   SPARSE_OPERATION_TRANSPOSE     \n" );
    printf( "   Input vector                   \n" );
    for ( i = 0; i < m; i++ )
    {
        printf( "%7.1f\n", x_v_original[i] );
    }

    // Create matrix descriptor
    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_LOWER;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_d_mv ( SPARSE_OPERATION_TRANSPOSE, alpha, csrA,
                               descrA, x_v_original, beta, y_v );
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 1 mkl_sparse_d_mv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf( "   OUTPUT DATA FOR mkl_sparse_d_mv \n" );
    for ( i = 0; i < m; i++ )
    {
        printf( "%7.1f\n", y_v[i] );
    }

    printf("   Solve triangular system   \n");
    printf("   with obtained             \n");
    printf("   right hand side           \n");

    status = mkl_sparse_d_trsv ( SPARSE_OPERATION_TRANSPOSE, alpha, csrA,
                                 descrA, y_v, x_v_calculated );
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 1 mkl_sparse_d_trsv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf( "   OUTPUT DATA FOR mkl_sparse_d_trsv \n" );
    for ( i = 0; i < m; i++ )
    {
        printf( "%7.1f\n", x_v_calculated[i] );
    }

    printf( "   Validating output data\n" );
    printf( "   against input data\n" );
    for (i = 0; i < m; i++) {
        if (!ALMOST_EQUAL(x_v_original[i], x_v_calculated[i])) {
            printf( " Error in Task 1: output data is not equal to" );
            printf( " input data \n" );
            exit_status = 1;
            goto exit;
        }
    };
    printf( "   Done \n" );
    printf( "-------------------------------------------------------\n" );

//*******************************************************************************
//  Task 2: Obtain matrix-matrix multiply (U+I)*x_v_original --> y_v
//          and solve triangular system   (U+I)*x_v_calculated = y_v
//          Array x_v_calculated must be equal to the array x_v_original
//*******************************************************************************
    printf( "                                  \n" );
    printf( "   Task 2:                        \n" );
    printf( "   INPUT DATA FOR mkl_sparse_d_mv \n" );
    printf( "   WITH TRIANGULAR SPARSE MATRIX  \n" );
    printf( "   ALPHA = %4.1f  BETA = %4.1f    \n", alpha, beta );
    printf( "   SPARSE_OPERATION_NON_TRANSPOSE \n" );
    printf( "   Input vector                   \n" );
    // Release matrix handle and deallocate matrix
    for ( i = 0; i < m; i++ )
    {
        printf( "%7.1f\n", x_v_original[i] );
    }

    // Create matrix descriptor
    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_UPPER;
    descrA.diag = SPARSE_DIAG_UNIT;

    status = mkl_sparse_d_mv ( SPARSE_OPERATION_NON_TRANSPOSE, alpha, csrA,
                               descrA, x_v_original, beta, y_v );
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 2 mkl_sparse_d_mv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf( "   OUTPUT DATA FOR mkl_sparse_d_mv \n" );
    for ( i = 0; i < m; i++ )
    {
        printf( "%7.1f\n", y_v[i] );
    }

    printf("   Solve triangular system   \n");
    printf("   with obtained             \n");
    printf("   right hand side           \n");

    status = mkl_sparse_d_trsv ( SPARSE_OPERATION_NON_TRANSPOSE, alpha, csrA,
                                 descrA, y_v, x_v_calculated );
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 2 mkl_sparse_d_trsv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf( "   OUTPUT DATA FOR mkl_sparse_d_trsv \n" );
    for ( i = 0; i < m; i++ )
    {
        printf( "%7.1f\n", x_v_calculated[i] );
    }

    printf( "   Validating output data\n" );
    printf( "   against input data\n" );
    for (i = 0; i < m; i++) {
        if (!ALMOST_EQUAL(x_v_original[i], x_v_calculated[i])) {
            printf( " Error in Task 2: output data is not equal to" );
            printf( " input data \n" );
            exit_status = 1;
            goto exit;
        }
    };
    printf( "   Done \n" );
    printf( "-------------------------------------------------------\n" );

//*******************************************************************************
//  Task 3: Obtain matrix-matrix multiply A' *x_m_original --> y_m
//          A - zero-based indexing,
//          x_m_original - column major ordering
//*******************************************************************************
    printf( "                                  \n" );
    printf( "   Task 3:                        \n" );
    printf( "   INPUT DATA FOR mkl_sparse_d_mm \n" );
    printf( "   WITH GENERAL SPARSE MATRIX     \n" );
    printf( "   COLUMN MAJOR ORDERING for RHS  \n" );
    printf( "   ALPHA = %4.1f  BETA = %4.1f    \n", alpha, beta );
    printf( "   SPARSE_OPERATION_TRANSPOSE     \n" );
    printf( "   Input vectors                  \n" );
    for (i = 0; i < m; i++) {
        for (j = 0; j < nrhs; j++) {
            printf("%7.1f", x_m_original[i+m*j]);
        };
        printf("\n");
    };

    // Create matrix descriptor
    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    // note that column-major format implies  ldx = m, ldy = m
    status = mkl_sparse_d_mm ( SPARSE_OPERATION_TRANSPOSE, alpha, csrA,  descrA,
                               SPARSE_LAYOUT_COLUMN_MAJOR, x_m_original, nrhs, m,
                               beta, y_m, m );
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 3 mkl_sparse_d_mm: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf( "   OUTPUT DATA FOR mkl_sparse_d_mm \n" );
    for (i = 0; i < m; i++) {
        for (j = 0; j < nrhs; j++) {
            printf("%7.1f", y_m[i+m*j]);
        };
        printf("\n");
    };
    printf( "-------------------------------------------------------\n" );

//*******************************************************************************
//  Task 4: Obtain matrix-matrix multiply A*x_m_original --> y_m
//          A - zero-based indexing,
//          x_m_original - row major ordering
//*******************************************************************************
    printf( "                                  \n" );
    printf( "   Task 4:                        \n" );
    printf( "   INPUT DATA FOR mkl_sparse_d_mm \n" );
    printf( "   WITH GENERAL SPARSE MATRIX     \n" );
    printf( "   ROW MAJOR ORDERING for RHS     \n" );
    printf( "   ALPHA = %4.1f  BETA = %4.1f    \n", alpha, beta );
    printf( "   SPARSE_OPERATION_TRANSPOSE     \n" );
    printf( "   Input vectors                  \n" );
    for (i = 0; i < m; i++) {
        for (j = 0; j < nrhs; j++) {
            printf("%7.1f", x_m_original[i*nrhs+j]);
        };
        printf("\n");
    };

    // Create matrix descriptor
    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    // note that row-major format implies  ldx = nrhs, ldy = nrhs
    status = mkl_sparse_d_mm ( SPARSE_OPERATION_NON_TRANSPOSE, alpha, csrA, descrA,
                               SPARSE_LAYOUT_ROW_MAJOR, x_m_original, nrhs, nrhs,
                               beta, y_m, nrhs );
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 4 mkl_sparse_d_mm: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf( "   OUTPUT DATA FOR mkl_sparse_d_mm \n" );
    for (i = 0; i < m; i++) {
        for (j = 0; j < nrhs; j++) {
            printf("%7.1f", y_m[i*nrhs+j]);
        };
        printf("\n");
    };
    printf( "-------------------------------------------------------\n" );

//*******************************************************************************
//    Task 5.  Obtain Triangular matrix-matrix multiply (L+D)' *x_m_original --> y_m
//    and solve triangular system (L+D)' *x_m_calculated = y_m with multiple right-hand
//    sides. Array x_m_calculated must be equal to the array y_m.
//*******************************************************************************
    printf( "                                  \n" );
    printf( "   Task 5:                        \n" );
    printf( "   INPUT DATA FOR mkl_sparse_d_mm \n" );
    printf( "   WITH LOWER TRIANGULAR SPARSE MATRIX   \n" );
    printf( "   WITH NON-UNIT DIAGONAL   \n" );
    printf( "   ALPHA = %4.1f  BETA = %4.1f    \n", alpha, beta );
    printf( "   SPARSE_OPERATION_TRANSPOSE     \n" );
    printf( "   Input matrix                   \n");
    for (i = 0; i < m; i++) {
        for (j = 0; j < nrhs; j++) {
            printf("%7.1f", x_m_original[i*nrhs+j]);
        };
        printf("\n");
    };

    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_LOWER;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_d_mm( SPARSE_OPERATION_TRANSPOSE, alpha, csrA, descrA,
                              SPARSE_LAYOUT_ROW_MAJOR, x_m_original, nrhs, nrhs, beta,
                              y_m, nrhs);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 5 mkl_sparse_d_mm: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf( "   OUTPUT DATA FOR mkl_sparse_d_mm \n" );
    printf( "   WITH TRIANGULAR MATRIX          \n" );
    for (i = 0; i < m; i++) {
        for (j = 0; j < nrhs; j++) {
            printf("%7.1f", y_m[i*nrhs+j]);
        };
        printf("\n");
    };

    printf( "   Solve triangular system   \n" );
    printf( "   with obtained             \n" );
    printf( "   right hand side           \n" );

    status =  mkl_sparse_d_trsm( SPARSE_OPERATION_TRANSPOSE, alpha, csrA, descrA,
                                 SPARSE_LAYOUT_ROW_MAJOR, y_m, nrhs, nrhs, x_m_calculated, 
                                 nrhs);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 5 mkl_sparse_d_trsm: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf( "   OUTPUT DATA FOR mkl_sparse_d_trsm \n" );
    printf( "   WITH TRIANGULAR MATRIX            \n" );
    for (i = 0; i < m; i++) {
        for (j = 0; j < nrhs; j++) {
            printf("%7.1f", x_m_calculated[i*nrhs+j]);
        };
        printf("\n");
    };

    printf( "   Validating output data\n" );
    printf( "   against input data\n" );
    for (i = 0; i < m*nrhs; i++) {
        if (!ALMOST_EQUAL(x_m_original[i], x_m_calculated[i])) {
            printf( " Error in Task 5: output data is not equal to" );
            printf( " input data \n" );
            exit_status = 1;
            goto exit;
        };
    };
    printf( "   Done \n" );
    printf( "-------------------------------------------------------\n" );

exit:
    // Release matrix handle and deallocate matrix
    mkl_sparse_destroy ( csrA );

    return exit_status;
}
