#===============================================================================
# Copyright 2020-2022 Intel Corporation.
#
# This software and the related documents are Intel copyrighted  materials,  and
# your use of  them is  governed by the  express license  under which  they were
# provided to you (License).  Unless the License provides otherwise, you may not
# use, modify, copy, publish, distribute,  disclose or transmit this software or
# the related documents without Intel's prior written permission.
#
# This software and the related documents  are provided as  is,  with no express
# or implied  warranties,  other  than those  that are  expressly stated  in the
# License.
#===============================================================================

include_guard()

# Set MKL_ROOT directory
if(NOT DEFINED MKL_ROOT)
  if(DEFINED ENV{MKLROOT})
    set(MKL_ROOT $ENV{MKLROOT})
  else()
    get_filename_component(MKL_CMAKE_PATH "${CMAKE_CURRENT_LIST_DIR}" REALPATH)
    get_filename_component(MKL_ROOT "${MKL_CMAKE_PATH}/../../" ABSOLUTE)
    message(STATUS "MKL_ROOT ${MKL_ROOT}")
  endif()
endif()

if(NOT DEFINED MKL_DIR)
  set(MKL_DIR ${MKL_ROOT})
endif()

if(ENABLE_BLACS OR ENABLE_CDFT OR ENABLE_SCALAPACK OR ENABLE_CPARDISO)
  set(ENABLE_MPI ON)
endif()

function(define_compiler TARGET_COMPILER DEFAULT_COMPILER SUPPORTED_LIST)
  if(NOT DEFINED ${TARGET_COMPILER})
    set(${TARGET_COMPILER} "${${DEFAULT_COMPILER}}" PARENT_SCOPE)
    message(STATUS "${TARGET_COMPILER}: None, set to `${${DEFAULT_COMPILER}}` by default")
  else()
    set(ITEM_FOUND 0)
    foreach(opt ${${SUPPORTED_LIST}})
      if(${TARGET_COMPILER} MATCHES "^.*/${opt}(\.exe)*$|^${opt}(\.exe)*$")
        message(STATUS "${TARGET_COMPILER}: `${${TARGET_COMPILER}}`")
        set(ITEM_FOUND 1)
      endif()
    endforeach()
    if(ITEM_FOUND EQUAL 0)
      if(ENABLE_MPI AND ${TARGET_COMPILER} MATCHES "mpi")
        message(STATUS "${TARGET_COMPILER}: `${${TARGET_COMPILER}}`")
      else()
        foreach(opt ${${SUPPORTED_LIST}})
          if(opt STREQUAL "mpi")
          set(STR_LIST "${STR_LIST} mpi*")
          else()
            set(STR_LIST "${STR_LIST} ${opt}")
          endif()
        endforeach()
        message(FATAL_ERROR "Unsupported ${TARGET_COMPILER} `${${TARGET_COMPILER}}`, supported: ${STR_LIST}")
      endif()
    endif()
  endif()
endfunction()

function(define_param TARGET_PARAM DEFAULT_PARAM SUPPORTED_LIST)
  if(NOT DEFINED ${TARGET_PARAM} AND NOT DEFINED ${DEFAULT_PARAM})
    message(STATUS "${TARGET_PARAM}: Undefined")
  elseif(NOT DEFINED ${TARGET_PARAM} AND DEFINED ${DEFAULT_PARAM})
    set(${TARGET_PARAM} "${${DEFAULT_PARAM}}" CACHE STRING "Choose ${TARGET_PARAM} options are: ${${SUPPORTED_LIST}}")
    foreach(opt ${${DEFAULT_PARAM}})
      set(STR_LIST "${STR_LIST} ${opt}")
    endforeach()
    message(STATUS "${TARGET_PARAM}: None, set to `${STR_LIST}` by default")
  elseif(${SUPPORTED_LIST})
    set(ITEM_FOUND 1)
    foreach(opt ${${TARGET_PARAM}})
      if(NOT ${opt} IN_LIST ${SUPPORTED_LIST})
        set(ITEM_FOUND 0)
      endif()
    endforeach()
    if(ITEM_FOUND EQUAL 0)
      foreach(opt ${${SUPPORTED_LIST}})
        set(STR_LIST "${STR_LIST} ${opt}")
      endforeach()
      message(FATAL_ERROR "Invalid ${TARGET_PARAM} `${${TARGET_PARAM}}`, options are: ${STR_LIST}")
    else()
      message(STATUS "${TARGET_PARAM}: ${${TARGET_PARAM}}")
    endif()
  else()
    message(STATUS "${TARGET_PARAM}: ${${TARGET_PARAM}}")
  endif()
endfunction()

# Define C compiler
if("C" IN_LIST TEST_LANG)
  if(ENABLE_OMP_OFFLOAD)
    set(CMAKE_C_COMPILER "icx") # OpenMP Offload is supported only by icx
    message(STATUS "CMAKE_C_COMPILER: `${CMAKE_C_COMPILER}`")
  else()
    if(UNIX)
      set(DEFAULT_C_COMPILER "icc")
      set(C_COMPILER_LIST icc gcc clang pgcc icx)
      if(APPLE)
        set(C_COMPILER_LIST icc clang)
      endif()
    else()
      set(DEFAULT_C_COMPILER "icl")
      set(C_COMPILER_LIST icl cl pgcc icx)
    endif()
    if(ENABLE_MPI)
      list(APPEND C_COMPILER_LIST mpi)
    endif()
    define_compiler(CMAKE_C_COMPILER DEFAULT_C_COMPILER C_COMPILER_LIST)
  endif() # ENABLE_OMP_OFFLOAD
endif()

# Define Fortran compiler
if("Fortran" IN_LIST TEST_LANG)

  if(ENABLE_OMP_OFFLOAD)
    set(CMAKE_Fortran_COMPILER "ifx") # OpenMP Offload is supported only by ifx
    message(STATUS "CMAKE_Fortran_COMPILER: `${CMAKE_Fortran_COMPILER}`")
  else()
    set(DEFAULT_F_COMPILER "ifort")
    set(F_COMPILER_LIST ifort pgf95 ifx)
    if(UNIX)
      list(APPEND F_COMPILER_LIST "gfortran")
    endif()
    if(ENABLE_MPI)
      list(APPEND F_COMPILER_LIST mpi)
    endif()
    define_compiler(CMAKE_Fortran_COMPILER DEFAULT_F_COMPILER F_COMPILER_LIST)
  endif() # ENABLE_OMP_OFFLOAD

  # Some Fortran examples need C wrappers to build, define C compiler for them
  if(C_WRAPPERS)
    if(CMAKE_Fortran_COMPILER STREQUAL "ifort")
      if(WIN32)
        set(CMAKE_C_COMPILER "icl")
      else()
        set(CMAKE_C_COMPILER "icc")
      endif()
    elseif(CMAKE_Fortran_COMPILER STREQUAL "pgf95")
      set(CMAKE_C_COMPILER "pgcc")
    elseif(CMAKE_Fortran_COMPILER STREQUAL "ifx")
      set(CMAKE_C_COMPILER "icx")
    elseif(CMAKE_Fortran_COMPILER STREQUAL "gfortran")
      set(CMAKE_C_COMPILER "gcc")
    elseif(CMAKE_Fortran_COMPILER MATCHES "mpi")
      if(WIN32)
        set(CMAKE_C_COMPILER "mpicc.bat")
      else()
        set(CMAKE_C_COMPILER "mpicc")
      endif()
    endif()
  endif()

endif()

if(CMAKE_${TEST_LANG}_COMPILER MATCHES "mpi")
  set(USE_MPI_COMPILER ON)
endif()

# Define C++ compiler
if("CXX" IN_LIST TEST_LANG)
  # Compiler for DPC++ examples
  if(UNIX)
    set(DEFAULT_CXX_COMPILER "icpx")
    set(CXX_COMPILER_LIST icpx)
  else()
    set(DEFAULT_CXX_COMPILER "icx")
    set(CXX_COMPILER_LIST icx)
  endif()
  define_compiler(CMAKE_CXX_COMPILER DEFAULT_CXX_COMPILER CXX_COMPILER_LIST)
endif()

# Check/define Offload precision
if(ENABLE_OMP_OFFLOAD)
  set(DEFAULT_TARGET_OFFLOAD_PRECISION sp dp)
  set(TARGET_OFFLOAD_PRECISION_LIST sp dp)
  define_param(TARGET_OFFLOAD_PRECISION DEFAULT_TARGET_OFFLOAD_PRECISION TARGET_OFFLOAD_PRECISION_LIST)
endif()

# Check/define TARGET_DEVICES
if(CMAKE_CXX_COMPILER MATCHES "^icp?x$|^.*/(icpx|icx\.exe)$")
  if(TARGET_DEVICES)
    # Make sure the input was converted to list
    string(REPLACE " " ";" TARGET_DEVICES "${TARGET_DEVICES}")
  endif()
  set(DEFAULT_TARGET_DEVICES cpu gpu)
  set(TARGET_DEVICES_LIST cpu gpu)
  define_param(TARGET_DEVICES DEFAULT_TARGET_DEVICES TARGET_DEVICES_LIST)
endif()

# Check/define TARGET_DOMAINS
# TARGET_DOMAINS_LIST can be any, all by default
define_param(TARGET_DOMAINS DEFAULT_TARGET_DOMAINS TARGET_DOMAINS_LIST)
# Make sure the input was converted to list
if(TARGET_DOMAINS)
    string(REPLACE " " ";" TARGET_DOMAINS ${TARGET_DOMAINS})
endif()

# Define TARGET_FUNCTIONS
# TARGET_FUNCTIONS_LIST can be any, all by default
define_param(TARGET_FUNCTIONS DEFAULT_TARGET_FUNCTIONS TARGET_FUNCTIONS_LIST)
# Make sure the input was converted to list
if(TARGET_FUNCTIONS)
    string(REPLACE " " ";" TARGET_FUNCTIONS ${TARGET_FUNCTIONS})
endif()


# define additional options to enable/disable warnings
set(COPT_WARNING "")
set(FOPT_WARNING "")

if(NOT USE_MPI_COMPILER AND NOT CMAKE_C_COMPILER STREQUAL "pgcc")
  if(WIN32)
    list(APPEND COPT_WARNING /Wall)
    if(CMAKE_${TEST_LANG}_COMPILER STREQUAL "ifort")
      list(APPEND FOPT_WARNING /warn:all)

      # Suppress warnings about truncated source line
      list(APPEND FOPT_WARNING "/Qdiag-disable:5194")
    endif()
  else()
    list(APPEND COPT_WARNING -Wall)
    if(CMAKE_${TEST_LANG}_COMPILER STREQUAL "ifort")
      list(APPEND FOPT_WARNING -warn all)
      
      # Suppress warnings about truncated source line
      list(APPEND FOPT_WARNING "-diag-disable:5194")
    elseif(CMAKE_${TEST_LANG}_COMPILER STREQUAL "gfortran")
      list(APPEND FOPT_WARNING -Wall)
    endif()
  endif()
endif()

if (CMAKE_${TEST_LANG}_COMPILER STREQUAL "cl")
# Taken from https://docs.microsoft.com/en-us/cpp/preprocessor/compiler-warnings-that-are-off-by-default?view=msvc-160
# These warnings are off by default, because most developers don't find them useful.
# In some cases, they warn about a stylistic choice, or about common idioms in older code.
# warning 4710: the given function was selected for inline expansion, but the compiler did not perform the inlining.
# warning 4820: The type and order of elements caused the compiler to add padding to the end of a struct
# warning 4711: the given function (not specifically marked as inline) was selected for automatic inline expansion
  list(APPEND COPT_WARNING "/wd4710")
  list(APPEND COPT_WARNING "/wd4820")
  list(APPEND COPT_WARNING "/wd4711")
endif()

# Suppress CRT warning on Windows
if(WIN32)
  list(APPEND COPT_WARNING "/D_CRT_SECURE_NO_WARNINGS")
endif()

set(DEFAULT_BUILD_TYPE Release)
set(BUILD_TYPE_LIST Release Debug)
define_param(CMAKE_BUILD_TYPE DEFAULT_BUILD_TYPE BUILD_TYPE_LIST)
