/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.index.store.remote.utils.cache;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Objects;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.BiFunction;
import java.util.function.Predicate;
import org.opensearch.common.cache.RemovalListener;
import org.opensearch.common.cache.RemovalNotification;
import org.opensearch.common.cache.RemovalReason;
import org.opensearch.common.cache.Weigher;
import org.opensearch.index.store.remote.utils.cache.CacheUsage;
import org.opensearch.index.store.remote.utils.cache.RefCountedCache;
import org.opensearch.index.store.remote.utils.cache.stats.CacheStats;
import org.opensearch.index.store.remote.utils.cache.stats.DefaultStatsCounter;
import org.opensearch.index.store.remote.utils.cache.stats.StatsCounter;

class LRUCache<K, V>
implements RefCountedCache<K, V> {
    private final long capacity;
    private final HashMap<K, Node<K, V>> data;
    private final LinkedHashMap<K, Node<K, V>> lru;
    private final RemovalListener<K, V> listener;
    private final Weigher<V> weigher;
    private final StatsCounter<K> statsCounter;
    private final ReentrantLock lock;
    private long usage;
    private long activeUsage;

    public LRUCache(long capacity, RemovalListener<K, V> listener, Weigher<V> weigher) {
        this.capacity = capacity;
        this.listener = listener;
        this.weigher = weigher;
        this.data = new HashMap();
        this.lru = new LinkedHashMap();
        this.lock = new ReentrantLock();
        this.statsCounter = new DefaultStatsCounter();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public V get(K key) {
        Objects.requireNonNull(key);
        this.lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node == null) {
                this.statsCounter.recordMisses(key, 1);
                V v = null;
                return v;
            }
            this.incRef(key);
            this.statsCounter.recordHits(key, 1);
            Object v = node.value;
            return v;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public V put(K key, V value) {
        Objects.requireNonNull(key);
        Objects.requireNonNull(value);
        this.lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node != null) {
                Object oldValue = node.value;
                this.replaceNode(node, value);
                Object v = oldValue;
                return v;
            }
            this.addNode(key, value);
            V v = null;
            return v;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public V compute(K key, BiFunction<? super K, ? super V, ? extends V> remappingFunction) {
        Objects.requireNonNull(key);
        Objects.requireNonNull(remappingFunction);
        this.lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node == null) {
                V newValue = remappingFunction.apply(key, null);
                if (newValue == null) {
                    V v = null;
                    return v;
                }
                this.addNode(key, newValue);
                this.statsCounter.recordMisses(key, 1);
                V v = newValue;
                return v;
            }
            V newValue = remappingFunction.apply(key, node.value);
            if (newValue == null) {
                this.removeNode(key);
                V v = null;
                return v;
            }
            this.statsCounter.recordHits(key, 1);
            this.replaceNode(node, newValue);
            V v = newValue;
            return v;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void remove(K key) {
        Objects.requireNonNull(key);
        this.lock.lock();
        try {
            this.removeNode(key);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void clear() {
        this.lock.lock();
        try {
            this.usage = 0L;
            this.activeUsage = 0L;
            this.lru.clear();
            for (Node<K, V> node : this.data.values()) {
                this.data.remove(node.key);
                this.statsCounter.recordRemoval(node.weight);
                this.listener.onRemoval(new RemovalNotification(node.key, node.value, RemovalReason.EXPLICIT));
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public long size() {
        return this.data.size();
    }

    @Override
    public void incRef(K key) {
        Objects.requireNonNull(key);
        this.lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node != null) {
                if (node.refCount == 0) {
                    this.activeUsage += node.weight;
                }
                if (node.evictable()) {
                    this.lru.remove(node.key);
                }
                ++node.refCount;
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void decRef(K key) {
        Objects.requireNonNull(key);
        this.lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node != null && node.refCount > 0) {
                --node.refCount;
                if (node.evictable()) {
                    this.lru.put(node.key, node);
                }
                if (node.refCount == 0) {
                    this.activeUsage -= node.weight;
                }
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long prune(Predicate<K> keyPredicate) {
        long sum = 0L;
        this.lock.lock();
        try {
            Iterator<Node<K, V>> iterator = this.lru.values().iterator();
            while (iterator.hasNext()) {
                Node<K, V> node = iterator.next();
                if (keyPredicate != null && !keyPredicate.test(node.key)) continue;
                iterator.remove();
                this.data.remove(node.key, node);
                sum += node.weight;
                this.statsCounter.recordRemoval(node.weight);
                this.listener.onRemoval(new RemovalNotification(node.key, node.value, RemovalReason.EXPLICIT));
            }
            this.usage -= sum;
        }
        finally {
            this.lock.unlock();
        }
        return sum;
    }

    @Override
    public CacheUsage usage() {
        this.lock.lock();
        try {
            CacheUsage cacheUsage = new CacheUsage(this.usage, this.activeUsage);
            return cacheUsage;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public CacheStats stats() {
        this.lock.lock();
        try {
            CacheStats cacheStats = this.statsCounter.snapshot();
            return cacheStats;
        }
        finally {
            this.lock.unlock();
        }
    }

    private void addNode(K key, V value) {
        long weight = this.weigher.weightOf(value);
        Node<K, V> newNode = new Node<K, V>(key, value, weight);
        this.data.put(key, newNode);
        this.usage += weight;
        this.incRef(key);
        this.evict();
    }

    private void replaceNode(Node<K, V> node, V newValue) {
        if (node.value != newValue) {
            Object oldValue = node.value;
            long oldWeight = node.weight;
            long newWeight = this.weigher.weightOf(newValue);
            node.value = newValue;
            node.weight = newWeight;
            long weightDiff = newWeight - oldWeight;
            if (node.refCount > 0) {
                this.activeUsage += weightDiff;
            }
            this.usage += weightDiff;
            this.statsCounter.recordReplacement();
            this.listener.onRemoval(new RemovalNotification(node.key, oldValue, RemovalReason.REPLACED));
        }
        this.incRef(node.key);
        this.evict();
    }

    private void removeNode(K key) {
        Node<K, V> node = this.data.remove(key);
        if (node != null) {
            if (node.refCount > 0) {
                this.activeUsage -= node.weight;
            }
            this.usage -= node.weight;
            if (node.evictable()) {
                this.lru.remove(node.key);
            }
            this.statsCounter.recordRemoval(node.weight);
            this.listener.onRemoval(new RemovalNotification(node.key, node.value, RemovalReason.EXPLICIT));
        }
    }

    private boolean hasOverflowed() {
        return this.usage >= this.capacity;
    }

    private void evict() {
        Iterator<Node<K, V>> iterator = this.lru.values().iterator();
        while (this.hasOverflowed() && iterator.hasNext()) {
            Node<K, V> node = iterator.next();
            iterator.remove();
            this.data.remove(node.key, node);
            this.usage -= node.weight;
            this.statsCounter.recordEviction(node.weight);
            this.listener.onRemoval(new RemovalNotification(node.key, node.value, RemovalReason.CAPACITY));
        }
    }

    static class Node<K, V> {
        final K key;
        V value;
        long weight;
        int refCount;

        Node(K key, V value, long weight) {
            this.key = key;
            this.value = value;
            this.weight = weight;
            this.refCount = 0;
        }

        public boolean evictable() {
            return this.refCount == 0;
        }
    }
}

