# $Id: netflow.py 23 2006-11-08 15:45:33Z dugsong $
# -*- coding: utf-8 -*-
"""Cisco Netflow."""
from __future__ import print_function
from __future__ import absolute_import

import struct

from . import dpkt
from .compat import compat_izip


class NetflowBase(dpkt.Packet):
    """Base class for Cisco Netflow packets.

    TODO: Longer class information....

    Attributes:
        __hdr__: Header fields of NetflowBase.
        TODO.
    """

    __hdr__ = (
        ('version', 'H', 1),
        ('count', 'H', 0),
        ('sys_uptime', 'I', 0),
        ('unix_sec', 'I', 0),
        ('unix_nsec', 'I', 0)
    )

    def __len__(self):
        return self.__hdr_len__ + (len(self.data[0]) * self.count)

    def __bytes__(self):
        # for now, don't try to enforce any size limits
        self.count = len(self.data)
        return self.pack_hdr() + b''.join(map(bytes, self.data))

    def unpack(self, buf):
        dpkt.Packet.unpack(self, buf)
        buf = self.data
        l_ = []
        while buf:
            flow = self.NetflowRecord(buf)
            l_.append(flow)
            buf = buf[len(flow):]
        self.data = l_

    class NetflowRecordBase(dpkt.Packet):
        """Base class for netflow v1-v7 netflow records.

        TODO: Longer class information....

        Attributes:
            __hdr__: Header fields of NetflowRecordBase.
            TODO.
        """

        # performance optimizations
        def __len__(self):
            # don't bother with data
            return self.__hdr_len__

        def __bytes__(self):
            # don't bother with data
            return self.pack_hdr()

        def unpack(self, buf):
            # don't bother with data
            for k, v in compat_izip(self.__hdr_fields__,
                                    struct.unpack(self.__hdr_fmt__, buf[:self.__hdr_len__])):
                setattr(self, k, v)
            self.data = b""


class Netflow1(NetflowBase):
    """Netflow Version 1.

    TODO: Longer class information....

    Attributes:
        __hdr__: Header fields of Netflow Version 1.
        TODO.
    """

    class NetflowRecord(NetflowBase.NetflowRecordBase):
        """Netflow v1 flow record.

        TODO: Longer class information....

        Attributes:
            __hdr__: Header fields of Netflow Version 1 flow record.
            TODO.
        """

        __hdr__ = (
            ('src_addr', 'I', 0),
            ('dst_addr', 'I', 0),
            ('next_hop', 'I', 0),
            ('input_iface', 'H', 0),
            ('output_iface', 'H', 0),
            ('pkts_sent', 'I', 0),
            ('bytes_sent', 'I', 0),
            ('start_time', 'I', 0),
            ('end_time', 'I', 0),
            ('src_port', 'H', 0),
            ('dst_port', 'H', 0),
            ('pad1', 'H', 0),
            ('ip_proto', 'B', 0),
            ('tos', 'B', 0),
            ('tcp_flags', 'B', 0),
            ('pad2', 'B', 0),
            ('pad3', 'H', 0),
            ('reserved', 'I', 0)
        )


# FYI, versions 2-4 don't appear to have ever seen the light of day.

class Netflow5(NetflowBase):
    """Netflow Version 5.

    TODO: Longer class information....

    Attributes:
        __hdr__: Header fields of Netflow Version 5.
        TODO.
    """

    __hdr__ = NetflowBase.__hdr__ + (
        ('flow_sequence', 'I', 0),
        ('engine_type', 'B', 0),
        ('engine_id', 'B', 0),
        ('reserved', 'H', 0),
    )

    class NetflowRecord(NetflowBase.NetflowRecordBase):
        """Netflow v5 flow record.

        TODO: Longer class information....

        Attributes:
            __hdr__: Header fields of Netflow Version 5 flow record.
            TODO.
        """

        __hdr__ = (
            ('src_addr', 'I', 0),
            ('dst_addr', 'I', 0),
            ('next_hop', 'I', 0),
            ('input_iface', 'H', 0),
            ('output_iface', 'H', 0),
            ('pkts_sent', 'I', 0),
            ('bytes_sent', 'I', 0),
            ('start_time', 'I', 0),
            ('end_time', 'I', 0),
            ('src_port', 'H', 0),
            ('dst_port', 'H', 0),
            ('pad1', 'B', 0),
            ('tcp_flags', 'B', 0),
            ('ip_proto', 'B', 0),
            ('tos', 'B', 0),
            ('src_as', 'H', 0),
            ('dst_as', 'H', 0),
            ('src_mask', 'B', 0),
            ('dst_mask', 'B', 0),
            ('pad2', 'H', 0),
        )


class Netflow6(NetflowBase):
    """Netflow Version 6.

    XXX - unsupported by Cisco, but may be found in the field.
    TODO: Longer class information....

    Attributes:
        __hdr__: Header fields of Netflow Version 6.
        TODO.
    """

    __hdr__ = Netflow5.__hdr__

    class NetflowRecord(NetflowBase.NetflowRecordBase):
        """Netflow v6 flow record.

        TODO: Longer class information....

        Attributes:
            __hdr__: Header fields of Netflow Version 6 flow record.
            TODO.
        """

        __hdr__ = (
            ('src_addr', 'I', 0),
            ('dst_addr', 'I', 0),
            ('next_hop', 'I', 0),
            ('input_iface', 'H', 0),
            ('output_iface', 'H', 0),
            ('pkts_sent', 'I', 0),
            ('bytes_sent', 'I', 0),
            ('start_time', 'I', 0),
            ('end_time', 'I', 0),
            ('src_port', 'H', 0),
            ('dst_port', 'H', 0),
            ('pad1', 'B', 0),
            ('tcp_flags', 'B', 0),
            ('ip_proto', 'B', 0),
            ('tos', 'B', 0),
            ('src_as', 'H', 0),
            ('dst_as', 'H', 0),
            ('src_mask', 'B', 0),
            ('dst_mask', 'B', 0),
            ('in_encaps', 'B', 0),
            ('out_encaps', 'B', 0),
            ('peer_nexthop', 'I', 0),
        )


class Netflow7(NetflowBase):
    """Netflow Version 7.

    TODO: Longer class information....

    Attributes:
        __hdr__: Header fields of Netflow Version 7.
        TODO.
    """

    __hdr__ = NetflowBase.__hdr__ + (
        ('flow_sequence', 'I', 0),
        ('reserved', 'I', 0),
    )

    class NetflowRecord(NetflowBase.NetflowRecordBase):
        """Netflow v6 flow record.

        TODO: Longer class information....

        Attributes:
            __hdr__: Header fields of Netflow Version 6 flow record.
            TODO.
        """

        __hdr__ = (
            ('src_addr', 'I', 0),
            ('dst_addr', 'I', 0),
            ('next_hop', 'I', 0),
            ('input_iface', 'H', 0),
            ('output_iface', 'H', 0),
            ('pkts_sent', 'I', 0),
            ('bytes_sent', 'I', 0),
            ('start_time', 'I', 0),
            ('end_time', 'I', 0),
            ('src_port', 'H', 0),
            ('dst_port', 'H', 0),
            ('flags', 'B', 0),
            ('tcp_flags', 'B', 0),
            ('ip_proto', 'B', 0),
            ('tos', 'B', 0),
            ('src_as', 'H', 0),
            ('dst_as', 'H', 0),
            ('src_mask', 'B', 0),
            ('dst_mask', 'B', 0),
            ('pad2', 'H', 0),
            ('router_sc', 'I', 0),
        )


# No support for v8 or v9 yet.

__sample_v1 = (
    b'\x00\x01\x00\x18\x67\x7a\x61\x3c\x42\x00\xfc\x1c\x24\x93\x08\x70\xac\x01\x20\x57\xc0\xa8\x63\xf7\x0a\x00\x02\x01\x00\x03'
    b'\x00\x0a\x00\x00\x00\x01\x00\x00\x02\x28\x67\x7a\x37\x2c\x67\x7a\x37\x2c\x5c\x1b\x00\x50\xac\x01\x11\x2c\x10\x00\x00\x00'
    b'\x00\x04\x00\x1b\xac\x01\x18\x53\xac\x18\xd9\xaa\xc0\xa8\x32\x02\x00\x03\x00\x19\x00\x00\x00\x01\x00\x00\x05\xdc\x67\x7a'
    b'\x37\x7c\x67\x7a\x37\x7c\xd8\xe3\x00\x50\xac\x01\x06\x2c\x10\x00\x00\x00\x00\x04\x00\x1b\xac\x01\x14\x18\xac\x18\x8d\xcd'
    b'\xc0\xa8\x32\x66\x00\x03\x00\x07\x00\x00\x00\x01\x00\x00\x05\xdc\x67\x7a\x37\x90\x67\x7a\x37\x90\x8a\x81\x17\x6f\xac\x01'
    b'\x06\x36\x10\x00\x00\x00\x00\x04\x00\x03\xac\x0f\x27\x24\xac\x01\xe5\x1d\xc0\xa8\x32\x06\x00\x04\x00\x1b\x00\x00\x00\x01'
    b'\x00\x00\x02\x28\x67\x7a\x3a\x38\x67\x7a\x3a\x38\xa3\x51\x12\x36\xac\x29\x06\xfd\x18\x00\x00\x00\x00\x04\x00\x1b\xac\x01'
    b'\x16\x45\xac\x23\x17\x8e\xc0\xa8\x32\x06\x00\x03\x00\x1b\x00\x00\x00\x01\x00\x00\x02\x28\x67\x7a\x3a\x4c\x67\x7a\x3a\x4c'
    b'\xc9\xff\x00\x50\xac\x1f\x06\x86\x02\x00\x00\x00\x00\x03\x00\x1b\xac\x0d\x09\xff\xac\x01\x99\x95\xc0\xa8\x32\x06\x00\x04'
    b'\x00\x1b\x00\x00\x00\x01\x00\x00\x05\xdc\x67\x7a\x3a\x58\x67\x7a\x3a\x58\xee\x39\x00\x17\xac\x01\x06\xde\x10\x00\x00\x00'
    b'\x00\x04\x00\x03\xac\x0e\x4a\xd8\xac\x01\xae\x2f\xc0\xa8\x32\x06\x00\x04\x00\x1b\x00\x00\x00\x01\x00\x00\x05\xdc\x67\x7a'
    b'\x3a\x68\x67\x7a\x3a\x68\xb3\x6e\x00\x15\xac\x01\x06\x81\x10\x00\x00\x00\x00\x04\x00\x1b\xac\x01\x23\x38\xac\x01\xd9\x2a'
    b'\xc0\xa8\x32\x06\x00\x03\x00\x1b\x00\x00\x00\x01\x00\x00\x05\xdc\x67\x7a\x3a\x74\x67\x7a\x3a\x74\x00\x00\x83\x50\xac\x21'
    b'\x01\xab\x10\x00\x00\x00\x00\x03\x00\x1b\xac\x0a\x60\x37\xac\x2a\x93\x4a\xc0\xa8\x32\x06\x00\x04\x00\x1b\x00\x00\x00\x01'
    b'\x00\x00\x05\xdc\x67\x7a\x3a\x74\x67\x7a\x3a\x74\x00\x00\x00\x00\xac\x01\x32\xa9\x10\x00\x00\x00\x00\x04\x00\x07\xac\x0a'
    b'\x47\x1f\xac\x01\xfd\x4a\xc0\xa8\x32\x06\x00\x04\x00\x1b\x00\x00\x00\x01\x00\x00\x00\x28\x67\x7a\x3a\x88\x67\x7a\x3a\x88'
    b'\x21\x99\x69\x87\xac\x1e\x06\x7e\x02\x00\x00\x00\x00\x03\x00\x1b\xac\x01\x28\xc9\xac\x01\x42\xc4\xc0\xa8\x32\x02\x00\x03'
    b'\x00\x19\x00\x00\x00\x01\x00\x00\x00\x28\x67\x7a\x3a\x88\x67\x7a\x3a\x88\x7d\x36\x00\x50\xac\x01\x06\xfe\x10\x00\x00\x00'
    b'\x00\x04\x00\x1b\xac\x0b\x08\xe8\xac\x01\x46\xe2\xc0\xa8\x32\x02\x00\x04\x00\x19\x00\x00\x00\x01\x00\x00\x05\xdc\x67\x7a'
    b'\x3a\x9c\x67\x7a\x3a\x9c\x60\x69\x69\x87\xac\x01\x06\x3b\x10\x00\x00\x00\x00\x04\x00\x1b\xac\x01\x1d\x24\xac\x3c\xf0\xc3'
    b'\xc0\xa8\x32\x06\x00\x03\x00\x1b\x00\x00\x00\x01\x00\x00\x05\xdc\x67\x7a\x3a\x9c\x67\x7a\x3a\x9c\x46\x32\x00\x14\xac\x01'
    b'\x06\x73\x18\x00\x00\x00\x00\x04\x00\x03\xac\x0b\x11\x51\xac\x01\xde\x06\xc0\xa8\x32\x06\x00\x04\x00\x1b\x00\x00\x00\x01'
    b'\x00\x00\x05\xdc\x67\x7a\x3a\xb0\x67\x7a\x3a\xb0\xef\x23\x1a\x2b\xac\x29\x06\xe9\x10\x00\x00\x00\x00\x04\x00\x1b\xac\x0c'
    b'\x52\xd9\xac\x01\x6f\xe8\xc0\xa8\x32\x02\x00\x04\x00\x19\x00\x00\x00\x01\x00\x00\x05\xdc\x67\x7a\x3a\xc4\x67\x7a\x3a\xc4'
    b'\x13\x6e\x00\x6e\xac\x19\x06\xa8\x10\x00\x00\x00\x00\x03\x00\x19\xac\x01\x3d\xdd\xac\x01\x7d\xee\xc0\xa8\x32\x66\x00\x03'
    b'\x00\x07\x00\x00\x00\x01\x00\x00\x00\x28\x67\x7a\x3a\xc4\x67\x7a\x3a\xc4\x00\x00\xdc\xbb\xac\x01\x01\xd3\x10\x00\x00\x00'
    b'\x00\x04\x00\x1b\xac\x0f\x28\xd1\xac\x01\xcc\xa5\xc0\xa8\x32\x06\x00\x04\x00\x1b\x00\x00\x00\x01\x00\x00\x05\xdc\x67\x7a'
    b'\x3a\xd8\x67\x7a\x3a\xd8\xc5\x73\x17\x6f\xac\x19\x06\x23\x18\x00\x00\x00\x00\x03\x00\x07\xac\x0a\x85\x5b\xc0\xa8\x63\x6e'
    b'\x0a\x00\x02\x01\x00\x04\x00\x0a\x00\x00\x00\x01\x00\x00\x05\xdc\x67\x7a\x3a\xe4\x67\x7a\x3a\xe4\xbf\x6c\x00\x50\xac\x01'
    b'\x06\xcf\x10\x00\x00\x00\x00\x04\x00\x07\xac\x01\x30\x1f\xac\x18\x21\x45\xc0\xa8\x32\x66\x00\x03\x00\x07\x00\x00\x00\x01'
    b'\x00\x00\x05\xdc\x67\x7a\x3b\x00\x67\x7a\x3b\x00\x11\x95\x04\xbe\xc0\xa8\x06\xea\x10\x00\x00\x00\x00\x03\x00\x0a\xac\x01'
    b'\x30\xb6\xac\x1e\xf4\xaa\xc0\xa8\x32\x06\x00\x03\x00\x1b\x00\x00\x00\x01\x00\x00\x05\xdc\x67\x7a\x3b\x34\x67\x7a\x3b\x34'
    b'\x88\x64\x00\x17\xac\x01\x06\x1f\x10\x00\x00\x00\x00\x04\x00\x1b\xac\x01\x23\x5f\xac\x1e\xb0\x09\xc0\xa8\x32\x06\x00\x03'
    b'\x00\x1b\x00\x00\x00\x01\x00\x00\x05\xdc\x67\x7a\x3b\x48\x67\x7a\x3b\x48\x81\x53\x00\x50\xac\x20\x06\x4e\x10\x00\x00\x00'
    b'\x00\x03\x00\x1b\xac\x01\x04\xd9\xac\x01\x94\x63\xc0\xa8\x32\x06\x00\x03\x00\x1b\x00\x00\x00\x01\x00\x00\x02\x28\x67\x7a'
    b'\x3b\x5c\x67\x7a\x3b\x5c\x55\x10\x00\x50\xac\x01\x06\x50\x18\x00\x00\x00\x00\x04\x00\x1b\xac\x01\x3c\xae\xac\x2a\xac\x21'
    b'\xc0\xa8\x32\x06\x00\x03\x00\x1b\x00\x00\x00\x01\x00\x00\x00\xfa\x67\x7a\x3b\x84\x67\x7a\x3b\x84\x0c\xe7\x00\x50\xac\x01'
    b'\x11\xfd\x10\x00\x00\x00\x00\x04\x00\x1b\xac\x01\x1f\x1f\xac\x17\xed\x69\xc0\xa8\x32\x02\x00\x03\x00\x19\x00\x00\x00\x01'
    b'\x00\x00\x05\xdc\x67\x7a\x3b\x98\x67\x7a\x3b\x98\xba\x17\x00\x16\xac\x01\x06\x7c\x10\x00\x00\x00\x00\x03\x00\x07'
)
__sample_v5 = (
    b'\x00\x05\x00\x1d\xb5\xfa\xc9\xd0\x3a\x0b\x41\x42\x26\x56\x77\xde\x9b\x73\x76\x31\x00\x01\x00\x00\xac\x0a\x86\xa6\xac\x01'
    b'\xaa\xf7\xc0\xa8\x32\x32\x02\x71\x00\x69\x00\x00\x00\x01\x00\x00\x02\x28\xb5\xfa\x81\x14\xb5\xfa\x81\x14\x35\x32\x00\x50'
    b'\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x01\x91\x44\xac\x14\x43\xe4\xc0\xa8\x32\x16\x00\x69\x02\x71\x00\x00'
    b'\x00\x01\x00\x00\x00\x28\xb5\xfa\x9b\xbd\xb5\xfa\x9b\xbd\x00\x50\x85\xd7\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00'
    b'\xac\x17\xe2\xd7\xac\x01\x8c\x56\xc0\xa8\x32\x32\x02\x71\x00\x69\x00\x00\x00\x01\x00\x00\x05\xdc\xb5\xfa\x6f\xb8\xb5\xfa'
    b'\x6f\xb8\x76\xe8\x17\x6f\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x0e\xf2\xe5\xac\x01\x91\xb2\xc0\xa8\x32\x32'
    b'\x02\x71\x00\x69\x00\x00\x00\x01\x00\x00\x00\xfa\xb5\xfa\x81\xee\xb5\xfa\x81\xee\xd0\xeb\x00\x15\x00\x00\x06\x00\x00\x00'
    b'\x00\x00\x00\x00\x00\x00\xac\x0a\x43\x6a\xac\x29\xa7\x09\x0a\x00\x02\x01\x02\x71\x00\xdb\x00\x00\x00\x01\x00\x00\x02\x28'
    b'\xb5\xfa\x85\x92\xb5\xfa\x85\x92\x8c\xb0\x00\x35\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x01\x96\x3d\xac\x15'
    b'\x1a\xa8\xc0\xa8\x32\x16\x00\x69\x02\x71\x00\x00\x00\x01\x00\x00\x05\xdc\xb5\xfa\x86\xe0\xb5\xfa\x86\xe0\xb4\xe7\x00\xc2'
    b'\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x01\x56\xd1\xac\x01\x86\x15\xc0\xa8\x32\x32\x02\x71\x00\x69\x00\x00'
    b'\x00\x01\x00\x00\x05\xdc\xb5\xfa\x7d\x3a\xb5\xfa\x7d\x3a\x5b\x51\x00\x50\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00'
    b'\xac\x32\xf1\xb1\xac\x29\x19\xca\x0a\x00\x02\x01\x02\x71\x00\xdb\x00\x00\x00\x01\x00\x00\x05\xdc\xb5\xfa\x83\xc3\xb5\xfa'
    b'\x83\xc3\x16\x2c\x00\x15\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x0c\x41\x34\xac\x01\x9a\x7a\xc0\xa8\x32\x32'
    b'\x02\x71\x00\x69\x00\x00\x00\x01\x00\x00\x05\xdc\xb5\xfa\x8d\xa7\xb5\xfa\x8d\xa7\x17\x33\x00\x15\x00\x00\x06\x00\x00\x00'
    b'\x00\x00\x00\x00\x00\x00\xac\x1e\xd2\x84\xac\x29\xd8\xd2\x0a\x00\x02\x01\x02\x71\x00\xdb\x00\x00\x00\x01\x00\x00\x05\xdc'
    b'\xb5\xfa\x8e\x97\xb5\xfa\x8e\x97\x37\x2a\x17\x6f\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x01\x85\x4a\xac\x20'
    b'\x11\xfc\xc0\xa8\x32\x16\x00\x69\x02\x71\x00\x00\x00\x01\x00\x00\x02\x28\xb5\xfa\x88\x34\xb5\xfa\x88\x34\xf5\xdd\x00\x8f'
    b'\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x01\x04\x80\xac\x3c\x5b\x6e\x0a\x00\x02\x01\x02\x71\x00\xdb\x00\x00'
    b'\x00\x01\x00\x00\x05\xdc\xb5\xfa\x9d\x72\xb5\xfa\x9d\x72\x73\x24\x00\x16\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00'
    b'\xac\x01\xb9\x4a\xac\x22\xc9\xd7\xc0\xa8\x32\x16\x00\x69\x02\x71\x00\x00\x00\x01\x00\x00\x00\x28\xb5\xfa\x90\x72\xb5\xfa'
    b'\x90\x72\x0f\x8d\x00\xc2\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x2a\xa3\x10\xac\x01\xb4\x19\xc0\xa8\x32\x32'
    b'\x02\x71\x00\x69\x00\x00\x00\x01\x00\x00\x00\x28\xb5\xfa\x92\x03\xb5\xfa\x92\x03\x70\x66\x00\x15\x00\x00\x06\x00\x00\x00'
    b'\x00\x00\x00\x00\x00\x00\xac\x01\xab\x6f\xac\x1e\x7f\x69\xc0\xa8\x32\x16\x00\x69\x02\x71\x00\x00\x00\x01\x00\x00\x05\xdc'
    b'\xb5\xfa\x93\x7f\xb5\xfa\x93\x7f\x00\x50\x0b\x98\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x0c\x0a\xea\xac\x01'
    b'\xa1\x15\xc0\xa8\x32\x32\x02\x71\x00\x69\x00\x00\x00\x01\x00\x00\x05\xdc\xb5\xfa\x79\xcf\xb5\xfa\x79\xcf\x5b\x33\x17\xe0'
    b'\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x01\xbb\xb3\xac\x29\x75\x8c\x0a\x00\x02\x01\x00\x69\x00\xdb\x00\x00'
    b'\x00\x01\x00\x00\x00\xfa\xb5\xfa\x94\x33\xb5\xfa\x94\x33\x00\x50\x1e\xca\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00'
    b'\xac\x0f\x4a\x60\xac\x01\xab\x94\xc0\xa8\x32\x32\x02\x71\x00\x69\x00\x00\x00\x01\x00\x00\x02\x28\xb5\xfa\x87\x5b\xb5\xfa'
    b'\x87\x5b\x9a\xd6\x2f\xab\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x2a\x0f\x93\xac\x01\xb8\xa3\xc0\xa8\x32\x32'
    b'\x02\x71\x00\x69\x00\x00\x00\x01\x00\x00\x00\x28\xb5\xfa\x89\xbb\xb5\xfa\x89\xbb\x6e\xe1\x00\x50\x00\x00\x06\x00\x00\x00'
    b'\x00\x00\x00\x00\x00\x00\xac\x01\x93\xa1\xac\x16\x80\x0c\xc0\xa8\x32\x16\x00\x69\x02\x71\x00\x00\x00\x01\x00\x00\x00\x28'
    b'\xb5\xfa\x87\x26\xb5\xfa\x87\x26\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x01\x83\x5a\xac\x1f'
    b'\x52\xcd\xc0\xa8\x32\x16\x00\x69\x02\x71\x00\x00\x00\x01\x00\x00\x05\xdc\xb5\xfa\x90\x0d\xb5\xfa\x90\x0d\xf7\x2a\x00\x8a'
    b'\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x0c\xe0\xad\xac\x01\xa8\x56\xc0\xa8\x32\x32\x02\x71\x00\x69\x00\x00'
    b'\x00\x01\x00\x00\x05\xdc\xb5\xfa\x9c\xf6\xb5\xfa\x9c\xf6\xe5\x7c\x1a\x2b\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00'
    b'\xac\x1e\xcc\x54\xac\x3c\x78\x26\x0a\x00\x02\x01\x02\x71\x00\xdb\x00\x00\x00\x01\x00\x00\x05\xdc\xb5\xfa\x80\xea\xb5\xfa'
    b'\x80\xea\x00\x00\x00\x00\x00\x00\x2f\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x01\xbb\x18\xac\x01\x7c\x7a\xc0\xa8\x32\x16'
    b'\x00\x69\x02\x71\x00\x00\x00\x01\x00\x00\x00\xfa\xb5\xfa\x88\x70\xb5\xfa\x88\x70\x00\x50\x0b\x7d\x00\x00\x06\x00\x00\x00'
    b'\x00\x00\x00\x00\x00\x00\xac\x17\x0e\x72\xac\x01\x8f\xdd\xc0\xa8\x32\x32\x02\x71\x00\x69\x00\x00\x00\x01\x00\x00\x02\x28'
    b'\xb5\xfa\x89\xf7\xb5\xfa\x89\xf7\x0d\xf7\x00\x8a\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x0a\xbb\x04\xac\x3c'
    b'\xb0\x15\x0a\x00\x02\x01\x02\x71\x00\xdb\x00\x00\x00\x01\x00\x00\x05\xdc\xb5\xfa\x90\xa9\xb5\xfa\x90\xa9\x9c\xd0\x00\x8f'
    b'\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x0a\x7a\x3f\xac\x29\x03\xc8\x0a\x00\x02\x01\x02\x71\x00\xdb\x00\x00'
    b'\x00\x01\x00\x00\x05\xdc\xb5\xfa\x75\x65\xb5\xfa\x75\x65\xee\xa6\x00\x50\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00'
    b'\xac\x01\xb5\x05\xc0\xa8\x63\x9f\x0a\x00\x02\x01\x00\x69\x00\xdb\x00\x00\x00\x01\x00\x00\x05\xdc\xb5\xfa\x7b\xc7\xb5\xfa'
    b'\x7b\xc7\x00\x50\x86\xa9\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\xac\x32\xa5\x1b\xac\x29\x30\xbf\x0a\x00\x02\x01'
    b'\x02\x71\x00\xdb\x00\x00\x00\x01\x00\x00\x00\xfa\xb5\xfa\x9b\x5a\xb5\xfa\x9b\x5a\x43\xf9\x17\xe0\x00\x00\x06\x00\x00\x00'
    b'\x00\x00\x00\x00\x00\x00'
)


def test_net_flow_v1_pack():
    pass


def test_net_flow_v1_unpack():
    nf = Netflow1(__sample_v1)
    assert len(nf.data) == 24
    # print repr(nfv1)


def test_net_flow_v5_pack():
    pass


def test_net_flow_v5_unpack():
    nf = Netflow5(__sample_v5)
    assert len(nf.data) == 29
    # print repr(nfv5)


if __name__ == '__main__':
    test_net_flow_v1_pack()
    test_net_flow_v1_unpack()
    test_net_flow_v5_pack()
    test_net_flow_v5_unpack()
    print('Tests Successful...')
