/***
 *** Copyright 1995  Phil Mercurio (mercurio@acm.org)
 ***
 *** Freely distributable for non-commercial purposes.  
 ***/

/**
 ** rcsFilt
 **
 ** PJM 950331	Begun
 ** PJM 950412	Reworked data structures
 **/

#ifndef rcsFilt_H
#define rcsFilt_H

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fstream.h>
#include <string.h>

/**
 ** Configuration constants
 **/
#define MaxBins		(16)		// maximum number of color bins
#define MaxLines	(50000)		// default maximum number of line slots
#define NameSize	(256)		// size of a file name
#define LineSize	(256)		// input line buffer size

#define TempPrefix	"%"		// temp file name prefix

/*
 * Information about one line in the source file.  These structures
 * form a doubly-linked list.
 *
 * A Line created after the input file has been read is an imaginary
 * line, and is indicated by a number of -1.
 *
 * A Line is marked as being deleted by zeroing its prev and next pointers.
 */
class Line {
public:
	class	Line*	next;		// forward link
	class	Line*	prev;		// backward link
	int	age;			// age of line, in revisions
	int	bin;			// age bin assigned to line [0..nBins)
	int	number;			// original line number, or -1

	/*
	 * Constructors and destructors
	 */
	Line() : age(0), bin(0), number(-1), next(0), prev(0) {}

	/*
	 * Methods
	 */
	bool deleted() {return(prev == 0 && next == 0);}
};
	
/*
 * Main application class
 */
class RcsFilt {
public:
	int	firstColor;		// first color number to use [1..16]
	int	nBins;			// number of bins
	int	maxLines;		// maximum number of lines
	bool	trunc;			// should we truncate or bin?
	bool	major;			// major versions only?
	bool	newZero;		// force age 0 into a bin by itself
	bool	tempci;			// temporarily check in file
	bool	testAge;		// for testing purposes

	char	path[NameSize];		// source file pathname
	char	tmpPath[NameSize];	// temporary input file
	char	rcsPath[NameSize];	// RCS companion to file
	char	tmpRcsPath[NameSize];	// temporary RCS companion to file
	char	dir[NameSize];		// directory
	char	file[NameSize];		// file name portion of path
	char	buffer[LineSize];	// input line buffer

	Line*	head;			// head of linked list
	Line**	slot;			// array of slots for Lines
	Line**	stash;			// slots for original Lines

	int	maxAge;			// highest age value
	int	inLines;		// number of lines in input
	int	curLines;		// current number of lines

	ostream*	err;			// error output file

	/*
	 * Constructors and destructors
	 */
	RcsFilt(int, char**);		// construct from command-line args

	/*
	 * Methods
	 */
	void Run();
	void findRCS();
	void scanSource();
	void scanDiffs();
	void fakeAges();
	void truncBins();
	void computeBins();
	void outputSource();
	bool scanForText(ifstream& in, char* vers);
	int scanFirstText(ifstream& in);
	bool isVersion(char* buf);
	void fillSlots();
	Line* findSlot(int);
	Line* getSlot(int);
	void setSlot(int,Line*);
	void processDiffs(ifstream& in);
	char parseDiffLine(char* p, int& l, int& n);
	bool versionsSame(char* a, char* b);
	void ageLines();
	void deleteLines(int l, int n);
	void appendLines(int l, int n);
	void dumpLines();
	void tempCheckin();
	void tempCleanup();
};

#endif /* rcsFilt_H */
