/***
 *** Copyright 1995  Phil Mercurio (mercurio@acm.org)
 ***
 *** Freely distributable for non-commercial purposes.  
 ***/

/**
 ** rcsFilt
 **
 ** xvile-compatible filter for RCS files.  Produces output which
 ** contains color attributes mapped onto the age of lines in the
 ** source file.
 **
 ** We assume the user has set up a range of colors for our use,
 ** the user gets to specify the first color ([1..16]) and the number of
 ** colors (bins).  If there are more ages than bins, we bin the
 ** ages so that the newest lines are in the first bin and the
 ** oldest are in the last bin, unless the -trunc flag is given,
 ** in which case we just truncate (anything older than nBins is put
 ** in the last bin).  If the -major flag is given, we only increment
 ** the age if the major version number changes.
 **
 ** See the manual page for more details.
 **
 ** PJM 950331	Begun
 **/

#include "rcsFilt.h"

RcsFilt*	rcsFilt;		// main application object

/*
 * Handy macro
 */
#define same(a,b)	(strcmp(a,b) == 0)

/*
 * Main routine
 */
main(int ac, char** av)
{
	rcsFilt = new RcsFilt(ac,av);

	if(rcsFilt) rcsFilt->Run();
}

/**
 ** RcsFilt methods
 **/

/*
 * Create a new RcsFilt given the command line arguments
 */
RcsFilt::RcsFilt(int ac, char** av)
{
	/* stderr-type output goes to "#rcsFilt.out", since
	 * running this from a macro loses the stderr
	 */
	err = (ostream*) new ofstream("#rcsFilt.out");
	if(!err) err = &cerr;

	/* Set defaults */
	firstColor = 1;
	nBins = 16;
	maxLines = MaxLines;
	trunc = false;
	major = false;
	path[0] = 0;
	head = 0;
	slot = 0;
	tempci = false;
	newZero = false;
	testAge = false;

	/* Parse args */
	for(ac--,av++; ac > 0; ac--,av++) {
		if(same(av[0],"-first")) {
			ac--;
			av++;
			firstColor = atoi(av[0]);
		} 
		else if(same(av[0],"-bins")) {
			ac--;
			av++;
			nBins = atoi(av[0]);
		} 
		else if(same(av[0],"-maxlines")) {
			ac--;
			av++;
			maxLines = atoi(av[0]);
		} 
		else if(same(av[0],"-testage")) {
			testAge = true;
		} 
		else if(same(av[0],"-trunc")) {
			trunc = true;
		} 
		else if(same(av[0],"-major")) {
			major = true;
		} 
		else if(same(av[0],"-tempci")) {
			tempci = true;
		} 
		else if(same(av[0],"-newzero")) {
			newZero = true;
		} 
		else if(av[0][0] == '-') {	// bogus flag
			goto usage;
		}
		else {				// file name
			if(path[0]) goto usage;

			strcpy(path,av[0]);
		}
	}

	/*
	 * Sanity checks
	 */
	if(path[0] == 0) goto usage;

	if(firstColor < 1 || firstColor > MaxBins) {
		*err << "First color not in range [1.." << MaxBins 
			<< "]." << endl;
		goto usage;
	}

	if(firstColor + nBins - 1 > MaxBins) {
		*err << "Last color (bin) greater than " << MaxBins 
			<< endl;
		goto usage;
	}

	/*
	 * Quietly shut off tempci if we're testing.
	 */
	if(testAge)
		tempci = false;

	/*
	 * Allocate the slot array
	 */
	slot = new Line*[maxLines];
	if(!slot) {
		*err << "Unable to allocate slot array for " << maxLines
			<< " lines" << endl;
		exit(-1);
	}

#ifdef DEBUG
	*err << "Args: -first " << firstColor << " -bins " << nBins 
		<< " -maxlines " << maxLines 
		<< " -trunc " << trunc << " -major " << major 
		<< " -tempci " << tempci << " -newzero " << newZero 
		<< " " << path << endl;
#endif
	return;

usage:
	*err << "Usage: rcsFilt [-first <color>] [-bins <nBins>] [-major] [-testage] [-trunc] [-tempci] [-newzero] file"
		<< endl;
	exit(0);
}

/*
 * Main processing routine for RcsFilt 
 */
void
RcsFilt::Run()
{
	/*
	 * Find RCS file name
	 */
	findRCS();

	/*
	 * Perform temporary checkin
	 */
	if(tempci) 
		tempCheckin();

	/*
	 * Read source file, initialize line info
	 */
	scanSource();

	/*
	 * Process diff info in RCS file, setting ages
	 * (unless we're in testAge mode)
	 */
	if(testAge)
		fakeAges();
	else
		scanDiffs();

	/*
	 * Compute bins.  Note that if the maxAge is less than or equal
	 * to the number of bins, then truncation is fine.
	 */
	if(trunc || maxAge <= nBins)
		truncBins();
	else
		computeBins();

	/*
	 * Output attributed text
	 */
	outputSource();

	/*
	 * Cleanup after temporary checkin
	 */
	if(tempci) 
		tempCleanup();

}

/*
 * Given the name of the file, find the corresponding RCS file, and the
 * temp paths, just in case we need them later.
 *
 * We also construct the dir and file strings here.  Note that
 * we make sure that dir, if not null, includes the terminating slash.
 */
void
RcsFilt::findRCS()
{
	char*	slash;		// right-most slash in path name

	slash = strrchr(path,'/');

	if(!slash) {
		dir[0] = 0;
		strcpy(file,path);
	}
	else {
		*slash = 0;
		sprintf(dir,"%s/",path);
		strcpy(file,slash+1);
		*slash = '/';
	}

	sprintf(tmpPath,"%s%s%s",dir,TempPrefix,file);
	sprintf(rcsPath,"%sRCS/%s,v",dir,file);
	sprintf(tmpRcsPath,"%sRCS/%s%s,v",dir,TempPrefix,file);
}

/*
 * Temporarily check in the file, by copying the file and its RCS file
 * to temp files and performing the checkin on the copy.
 * 
 * We end up by copying tmpRcsPath to rcsPath, everything else
 * proceeds normally.
 */
void
RcsFilt::tempCheckin()
{
	sprintf(buffer,"cp %s %s",path,tmpPath);
#ifdef DEBUG
	*err << buffer << endl;
#endif
	system(buffer);

	sprintf(buffer,"cp %s %s",rcsPath,tmpRcsPath);
#ifdef DEBUG
	*err << buffer << endl;
#endif
	system(buffer);

	sprintf(buffer,"rcs -q -l  %s",tmpPath);
#ifdef DEBUG
	*err << buffer << endl;
#endif
	system(buffer);

	sprintf(buffer,"ci -q -f -mtemporary %s",tmpPath);
#ifdef DEBUG
	*err << buffer << endl;
#endif
	system(buffer);

	strcpy(rcsPath,tmpRcsPath);
}

/*
 * Clean up after tempCheckin().  The temporary version of the file
 * should have already been removed by ci, but we remove it here
 * just to be certain.
 */
void 
RcsFilt::tempCleanup()
{
	unlink(tmpRcsPath);
	unlink(tmpPath);
}

/*
 * Read the source file once, building the initial linked list of
 * Lines and initializing their ages to 0.
 *
 * We also build the stash array, with a pointer to each of the original
 * Lines.
 */
void
RcsFilt::scanSource()
{
	ifstream in(path);

	if(!in.good()) {
		*err << "Unable to open input file \"" << path << "\"." 
			<< endl;
		exit(-1);
	}

	Line* 	lp;
	Line* 	prev = 0;

	inLines = 0;

	while((in.getline(buffer,LineSize)).good()) {
		inLines++;
		lp = new Line;
		if(!lp) {
			*err << "Unable to allocate memory for line " 
				<< inLines << endl;
			exit(-1);
		}

		lp->age = 0;
		lp->number = inLines;

		if(!prev) {
			head = lp;
		}
		else {
			prev->next = lp;
			lp->prev = prev;
		}

		prev = lp;
	}

	/* 
	 * Build stash
	 */
	int	i;
	stash = new Line*[inLines];
	if(!stash) {
		*err << "Unable to allocate " << inLines
			<< " pointers for stash" << endl;
		exit(-1);
	}

	for(i=0,lp=head; lp != 0; lp = lp->next,i++)
		stash[i] = lp;
}

/*
 * Generate fake ages by setting each line's age to its line number,
 * for debugging.
 */
void
RcsFilt::fakeAges()
{
	for(int j=0; j < inLines; j++) 
		stash[j]->age = j;

	maxAge = inLines - 1;
}

/*
 * Read the RCS file, parsing its contents and augmenting line ages.
 * maxAge gets set here.
 */
void
RcsFilt::scanDiffs()
{
	ifstream in(rcsPath);
	char	version[NameSize];
	char	prevVersion[NameSize];

	if(!in.good()) {
		*err << "Unable to open RCS file \"" << rcsPath << "\"." 
			<< endl;
		exit(-1);
	}

	/*
	 * Get first text block
	 */
	if(!scanForText(in,version)) {
		*err << "Error in RCS file \"" << rcsPath << "\"."
			<< endl;
		exit(-1);
	}

	if(scanFirstText(in) != inLines) {
		*err << "Input does not match RCS file, has it been checked in?"
			<< endl;
		exit(-1);
	}

#ifdef DEBUG
	*err << "After scanFirstText():" << endl;
	dumpLines();
#endif

	/*
	 * Now process each diff in turn, first marking those lines that
	 * are deleted, then aging those which remain.
	 */
	strcpy(prevVersion,version);
	while(scanForText(in,version)) {
		fillSlots();
		processDiffs(in);

		if(!major || !versionsSame(prevVersion,version)) 
			ageLines();

		strcpy(prevVersion,version);

#ifdef DEBUG
		*err << "After version " << version << endl;
		dumpLines();
#endif
	}

	/*
	 * Compute the maximum age
	 */
	maxAge = 0;
	for(int i=0; i < inLines; i++)
		if(stash[i]->age > maxAge) 
			maxAge = stash[i]->age;
}

/*
 * Scan forward thru the RCS file for the next text marker, capturing
 * the last seen version number in the process.
 *
 * Return true if successful, false if out of input.
 */
bool
RcsFilt::scanForText(ifstream& in, char* vers)
{
	while((in.getline(buffer,LineSize)).good()) {
		if(isVersion(buffer))
			strcpy(vers,buffer);
		else if(same("text",buffer))
			return(true);
	}

	/* If we get here, we ran out of input */
	return(false);
}

/*
 * Scan the first text block, returning the number of lines read.
 * We're done when we get to the "@" line.
 */
int 
RcsFilt::scanFirstText(ifstream& in)
{
	int	i = 0;

	while((in.getline(buffer,LineSize)).good()) {
		if(same("@",buffer))
			return(i);

		i++;
	}

	return(i);
}

/*
 * Parse an RCS file input line to see if it's a version number.
 */
bool
RcsFilt::isVersion(char* buf)
{
	char*	p;

	for(p=buf; *p; p++)
		if(!isdigit(*p) && *p != '.') return(false);

	if(p == buf) return(false);

	return(true);
}

/*
 * Return true if two version strings are the same in the major portion
 * of the version numbers.
 */
bool
RcsFilt::versionsSame(char* a, char* b)
{
	while(*a != '.' && *a != 0 && *b != '.' && *b != 0) {
		if(*a != *b) return(false);

		a++;
		b++;
	}


	return(*a == *b);
}

/*
 * Create an array, slot, of pointers to Lines in the correct 
 * order.  This demangles any additions/subtractions from the
 * linked list.  We may also need to change the head pointer
 * here, if any lines were added before line 1.
 *
 * The result is an array we can use to index lines numerically,
 * it should speed processing the diffs up enough to be worth it.
 */
void
RcsFilt::fillSlots()
{
	/*
	 * Fix head, make sure it's the first line in the linked list
	 */
	while(head->prev) 
		head = head->prev;

	/*
	 * Fill slots
	 */
	Line*	lp;
	int	i = 0;

	for(lp=head; lp != 0; lp = lp->next) {
		slot[i++] = lp;

		if(i >= maxLines) {
			*err << "Slot table overflow, use a larger -maxlines"
				<< " value (currently " << maxLines << ")"
				<< endl;
			exit(-1);
		}
	}

	curLines = i;
}

/*
 * Given a line number ([1..nLines)), find the Line by using the slot
 * table.  NULL is returned if it's out of range.
 */
Line*
RcsFilt::getSlot(int l)
{
	if(l >= 1 && l <= curLines)
		return(slot[l-1]);
	else
		return(NULL);
}

/*
 * Given a line number ([1..nLines)), find the Line by using the slot
 * table.  Search backwards until we find the first undeleted slot.
 * NULL is returned if it's out of range.
 */
Line*
RcsFilt::findSlot(int l)
{
	if(l < 1 || l > curLines)
		return(NULL);

	int	i;
	Line*	lp;

	for(i=l-1; i >= 0; i--) {
		lp = slot[i];
		if(lp && !lp->deleted()) 
			return(lp);
	}

	return(NULL);
}

/*
 * Given a line number ([1..nLines)), set the corresponding slot.
 */
void
RcsFilt::setSlot(int l,Line* lp)
{
	if(l >= 1 && l <= curLines)
		slot[l-1] = lp;
}

/*
 * Process a set of diffs.  We should be at the first line, which
 * will begin with an @.  Since time runs backwards in RCS files,
 * we only need to attend to the "d" and "a" lines, appended text 
 * is ignored (since it's really text that was deleted from one 
 * version to the next).
 *
 * We're done when we get to a line containing only "@" or "@@"
 * ("@@" is a difference record with no changes at all).
 */
void
RcsFilt::processDiffs(ifstream& in)
{
	char*	p;
	int	l, n;

	while((in.getline(buffer,LineSize)).good()) {
		if(same("@",buffer)) return;
		if(same("@@",buffer)) return;

		p = buffer;
		if(*p == '@') p++;

		switch(parseDiffLine(p,l,n)) {
			case 'd':		// Delete command
				deleteLines(l,n);
				break;
				
			case 'a':		// Append command
				appendLines(l,n);
				while(n--)
					in.getline(buffer,LineSize);

				break;

			default:
				*err << "Error in RCS input: " << buffer 
					<< endl;
				exit(-1);
				break;
		}
	}
}
				
/*
 * Parse an RCS diff line, returning the command character
 * and filling in the line number and the count.
 */
char
RcsFilt::parseDiffLine(char* p, int& l, int& n)
{
	char	command;

	command = *p++;
	l = atoi(p);

	while(*p != ' ') p++;
	p++;

	n = atoi(p);

	return(command);
}

/*
 * Delete n lines at line l ([1..nLines))
 */
void 
RcsFilt::deleteLines(int l, int n)
{
	Line*	lp;
	int	i;

	for(i=l; i < l+n; i++) {
		lp = getSlot(i);
		if(!lp) {
			*err << "Can't delete line " << i << ": out of range"
				<< endl;
			exit(-1);
		}
		else {
			if(lp == head) 
				head = lp->next;

			if(lp->prev)
				lp->prev->next = lp->next;

			if(lp->next)
				lp->next->prev = lp->prev;
		}

		/*
		 * Mark the line as deleted by zeroing both prev and next
		 * and emptying the slot.
		 */
		lp->prev = 0;
		lp->next = 0;
		setSlot(i,(Line*)0);
	}
}

/*
 * Append n imaginary lines after line l ([0..nLines)).  All the lines
 * we add will have a line number of -1, indicating that they're
 * imaginary lines (and should not be aged).
 */
void 
RcsFilt::appendLines(int l, int n)
{
	Line*	h = 0;	// head
	Line* 	t = 0;	// tail
	Line*	lp;

	/*
	 * Build a chain of n imaginary lines
	 */
	while(n--) {
		lp = new Line;	// imaginary by default
		if(!lp) {
			*err << "Unable to allocate memory for imaginary line" 
				<< endl;
			exit(-1);
		}

		if(!t) {
			h = lp;
		}
		else {
			t->next = lp;
			lp->prev = t;
		}

		t = lp;
	}

	/* 
	 * Use imaginary 0th line to handle insertion before current text
	 */
	if(l == 0) {
		lp = findSlot(1);
		if(lp->deleted()) {	// replace slot with new lines
			setSlot(1,h);
			t->next = findSlot(1);
			if(t->next) t->next->prev = t;
		}
		else {			// prepend new lines
			lp->prev = t;
			t->next = lp;
		}

		head = h;
		return;
	}

	/*
	 * Insert after line l 
	 */
	lp = findSlot(l);
	if(!lp) {
		*err << "Error in processing add" << endl;
		exit(-1);
	}

	t->next = lp->next;
	if(t->next) t->next->prev = t;

	h->prev = lp;
	lp->next = h;
}

/*
 * Increment the age of each line in the stash that hasn't been deleted.
 */
void
RcsFilt::ageLines()
{
	int	i;

	for(i=0; i < inLines; i++)
		if(!stash[i]->deleted()) 
			stash[i]->age++;
}

/*
 * Compute bin values by truncation--anything greater than the number
 * of bins ends up in the last bin.  Note that no special handling is
 * needed for newZero.
 */
void
RcsFilt::truncBins()
{
	int	i;

	for(i=0; i < inLines; i++) {
		if(stash[i]->age >= nBins) 
			stash[i]->bin = nBins - 1;
		else
			stash[i]->bin = stash[i]->age;
	}
}

/*
 * Compute bin values by binning--given the maxAge and the number of bins,
 * partition ages into bins evenly.
 *
 * We don't get here unless the maxAge is greater than the number of bins.
 *
 * If newZero is set, force age 0 into bin 0, by itself.
 */
void
RcsFilt::computeBins()
{

	int i;

	if(!newZero) {		// normal processing
		for(i=0; i < inLines; i++) 
			stash[i]->bin = (stash[i]->age * nBins) / (maxAge+1);
	}
	else {			// special handling for age 0
		for(i=0; i < inLines; i++) 
			if(stash[i]->age == 0)
				stash[i]->bin = 0;
			else
				stash[i]->bin = 
					(stash[i]->age * (nBins-1)) / (maxAge+1)
					+ 1;
	}
}

/*
 * Output the source, with the attributes added.  The bin values are
 * turned into color numbers here.
 *
 * The xvile attribute format is a ^A, followed by the number of
 * characters the attribute applies to, followed by the attribute 
 * and ended with a :
 */
void
RcsFilt::outputSource()
{
	ifstream in(file);

	if(!in.good()) {
		*err << "Unable to open input file \"" << file << "\"." 
			<< endl;
		exit(-1);
	}

	int i = 0;

	while((in.getline(buffer,LineSize)).good()) {
		cout.put('\001');
		cout << strlen(buffer) << 'C' 
			<< hex << (stash[i]->bin + firstColor) % 16 << dec
			<< ':' << buffer << endl;

		i++;
	}
}

/*
 * Used in debugging
 */
void
RcsFilt::dumpLines()
{
	int	i;
	int	n;
	Line*	lp;

	*err << "head: " << "0x" << hex << (unsigned long)head << dec
		<< " inLines: " << inLines << "  curLines: " << curLines << endl;

	*err << "Slots: ";
	for(i=0; i < curLines; i++) {
		if(slot[i] == NULL)
			*err << "@ ";
		else {
			*err << slot[i]->number << " ";
		}
	}
	*err << endl;
		
	for(i=0; i < inLines; i++) {
		lp = stash[i];

		*err << "0x" << hex << (unsigned long)lp << dec
			<< ": age " << lp->age << " bin " << lp->bin
			<< " number " << lp->number 
			<< " prev 0x" << hex << (unsigned long)lp->prev 
			<< " next 0x" << (unsigned long)lp->next << dec << endl;
	}
}
