#include "uart.h"
#include <malloc.h>
#include <stdlib.h>
#include <dos.h>

#ifdef MODEL_SMALL
#define CODE_NEAR		1
#define DATA_NEAR		1
#endif
#ifdef MODEL_COMPACT
#define CODE_NEAR		1
#define DATA_NEAR		0
#endif
#ifdef MODEL_MEDIUM
#define CODE_NEAR		0
#define DATA_NEAR		1
#endif
#ifdef MODEL_LARGE
#define CODE_NEAR		0
#define DATA_NEAR		0
#endif

#ifndef CODE_NEAR
Error: No memory model specified. Compile using the makefiles...
#endif

#define TIMEOUT_TIME	36		// 2 secs

static unsigned declaration_counter=0;

// Prototypes of assembler functions called by this module
int cdecl _uart_asm_define_port_1(unsigned baseaddr, unsigned intlevel, unsigned inbuf_size,
									unsigned outbuf_size, unsigned handshake);
void cdecl _uart_asm_define_port_2(unsigned ofs, unsigned seg);
void cdecl _uart_asm_define_port_3(unsigned ofs, unsigned seg);
void cdecl _uart_asm_define_port_4(unsigned intlevel);
void cdecl _uart_asm_cleanup(void);

/*	Just an example...

void _default_handler(int handle, unsigned baseaddr, unsigned char condition)
{
	// this is the default handler for MSI and LSI conditions
	// it does just nothing
	// please note that you can't call any functions that use DOS inside
	// an interrupt handler!
}

*/

int cdecl uart_int_define_port(unsigned baseaddr, unsigned intlevel, unsigned inbuf_size,
										unsigned outbuf_size, unsigned handshake)
{
	// create a new entry in the uart table, that is, begin servicing a certain uart
	// returns handle (0 - MAX_UARTS-1) if OK
	// returns negative value if not OK

	int h,i;
	struct foo {
		unsigned ofs;
		unsigned seg;
		};
	union {
		void far *ptr;
		struct foo adr;
		} x;

	// check given values for plausibility
	if (inbuf_size<MIN_INBUF_SIZE||inbuf_size>=MAX_INBUF_SIZE) return UE_BAD_INBUF_SIZE;
	if (outbuf_size<MIN_OUTBUF_SIZE||outbuf_size>=MAX_OUTBUF_SIZE) return UE_BAD_OUTBUF_SIZE;
	if (intlevel<3||intlevel==8||intlevel==13||intlevel>15) return UE_BAD_INTLEVEL;
	if (handshake>2) return UE_BAD_HANDSHAKING;

	// disable interrupts from this UART
	outp(baseaddr+4,inp(baseaddr+4)&0xf7);
	outp(baseaddr+1,0);
	if (inp(baseaddr+1)!=0) return UE_ERROR;

	h=_uart_asm_define_port_1(baseaddr,intlevel,inbuf_size,outbuf_size,handshake);
	if (h<0) return h;

	// try to reserve memory for the buffers
	x.ptr=malloc(inbuf_size);
	if (x.ptr==NULL) return -5;
	_uart_asm_define_port_2(x.adr.ofs,x.adr.seg);
	
	x.ptr=malloc(outbuf_size);
	if (x.ptr==NULL) return -5;
	_uart_asm_define_port_3(x.adr.ofs,x.adr.seg);

	// set default handlers for line status interrupt and modem status interrupt
	// these can be redefined using the appropriate functions
	// see if there is already a port in the table with the same intlevel;
	// if not, load vectors and unmask int
	_uart_asm_define_port_4(intlevel);

	if (declaration_counter==0) {
		declaration_counter=1;
		// it's the first interrupt created; make sure all interrupts are properly
		// shut down and all buffers are freed at exit
		atexit(_uart_asm_cleanup);
		}

	// now enable all interrupts for this UART
	outp(baseaddr+4,inp(baseaddr+4)|0x08);
	outp(baseaddr+1,0xf);

	// everything's gone well
	return h;
}

int cdecl uart_int_send_string(int handle, char *buf)
{
	// send a null-terminated string in interrupt driven mode
	// returns -2 if illegal handle
	//          non-negative  # of bytes sent
	
	long far * volatile timer=(long far * volatile)0x0040006cUL;
	long time;
	int x;
	unsigned c=0;
 
	time=*timer;
	while (*buf!=0) {
		while (1) {
			x=uart_int_send_byte(handle,*buf);
			if (x==UE_ERROR) continue;
			if (x==UE_BAD_HANDLE) return UE_BAD_HANDLE;
			if (labs(time-*timer)>=TIMEOUT_TIME) return c;
			break;
			}
		c++;
		buf++;
		time=*timer;
		}
	return c;
}

int cdecl uart_int_send_block(int handle, void *pointer, unsigned count)
{
	// sends a block of data in interrupt driven mode
	// returns -2 if illegal handle
	//          non-negative  # of bytes sent
	
	long far * volatile timer=(long far * volatile)0x0040006cUL;
	long time;
	int x;
	unsigned i;
	char *buf;

 	buf=(char *)pointer;
	time=*timer;
	for (i=0; i<count; i++) {
		while (1) {
			x=uart_int_send_byte(handle,*buf);
			if (x==UE_ERROR) continue;
			if (x==UE_BAD_HANDLE) return UE_BAD_HANDLE;
			if (labs(time-*timer)>=TIMEOUT_TIME) return i;
			break;
			}
		buf++;
		time=*timer;
		}
	return i;
}


