.po +.5i
.TL
The Curses Reference Manual
.AU
Pavel Curtis
.NH
Introduction
.LP
Terminfo is a database describing many capabilities of over 150
different terminals.  Curses is a subroutine package which
presents a high level screen model to the programmer, while
dealing with issues such as terminal differences and optimization of
output to change one screenfull of text into another.
.LP
Terminfo is based on Berkeley's termcap database, but contains a
number of improvements and extensions.  Parameterized strings are
introduced, making it possible to describe such capabilities as
video attributes, and to handle far more unusual terminals than
possible with termcap.
.LP
Curses is also based on Berkeley's curses package, with many
improvements.  The package makes use of the insert and delete
line and character features of terminals so equipped, and
determines how to optimally use these features with no help from the
programmer.  It allows arbitrary combinations of video attributes
to be displayed, even on terminals that leave ``magic cookies''
on the screen to mark changes in attributes.
.NH
An Overview of the Package
.NH 2
Terminology
.PP
In this document,
the following terminology is kept to with reasonable consistency:
.IP \fIwindow\fP 10
An internal representation
containing an image of what a section of the terminal screen may look like
at some point in time.
This subsection can either encompass the entire terminal screen,
or any smaller portion down to a single character within that screen.
.IP \fIterminal\fP 10
Sometimes called \fIterminal screen\fP.
The package's idea of what the terminal's screen currently looks like,
i.e., what the user sees now.
This is a special \fIscreen\fP:
.IP \fIscreen\fP 10
This is a subset of windows which are as large as the terminal screen,
i.e., they start at the upper left hand corner
and encompass the lower right hand corner.
One of these, \fIstdscr\fP,
is automatically provided for the programmer.
.NH 2
Compiling Programs using the Package
.PP
In order to use the library,
it is necessary to have certain types and variables defined.
Therefore, the programmer must have a line:
.DS
.B "#include <ncurses.h>"
.DE
at the top of the program source.
The header file
.B <ncurses.h>
needs to include
.B <sgtty.h> ,
so the one should not do so oneself.\u*\d
.FS
\u*\dThe screen package also uses the Standard I/O library,
so \fB<ncurses.h>\fP includes \fB<stdio.h>\fP.
It is redundant (but harmless) for the programmer to do it, too.
.FE
Also,
compilations should have the following form:
.DS
\fBcc\fR [ \fIflags\fR ] file ... \fB\-lbcurses\fR
.DE
.NH 2
Updating the Screen
.PP
In order to update the screen optimally,
it is necessary for the routines to know what the screen currently looks like
and what the programmer wants it to look like next.
For this purpose, a data type (structure) named \fIWINDOW\fP
is defined which describes a window image to the routines,
including its starting position on the screen
(the (y, x) coordinates of the upper left hand corner)
and its size.
One of these (called \fIcurscr\fP, for \fIcurrent screen\fP)
is a screen image of what the terminal currently looks like.
Another screen (called \fIstdscr\fP, for \fIstandard screen\fP)
is provided by default to make changes on.
.PP
A window is a purely internal representation.
It is used to build and store
a potential image of a portion of the terminal.
It doesn't bear any necessary relation
to what is really on the terminal screen.
It is more like an array of characters on which to make changes.
.PP
When one has a window which describes
what some part of the terminal screen should look like,
the routine \fIrefresh()\fP (or \fIwrefresh()\fP if the window is not
\fIstdscr\fP) is called.
\fIRefresh()\fP in the area covered by the window,
look like that window.
Note, therefore, that changing something on a window
\fIdoes not change the terminal\fP.
Actual updates to the terminal screen are made only by calling
\fIrefresh()\fP or \fIwrefresh()\fP.
This allows the programmer to maintain several different ideas
of what a portion of the terminal screen should look like.
Also, changes can be made to windows in any order,
without regard to motion efficiency.
Then, at will, the programmer can effectively say
``make it look like this,''
and let the package worry about the best way to do this.
.NH 2
Naming Conventions
.PP
As hinted above, the routines can use several windows,
but two are automatically given:
\fIcurscr\fP, which knows what the terminal looks like,
and \fIstdscr\fP,
which is what the programmer wants the terminal to look like next.
The user should never really access \fIcurscr\fP directly.
Changes should be made to the appropriate screen,
and then the routine \fIrefresh()\fP (or \fIwrefresh()\fP)
should be called.
.PP
Many functions are set up to deal with \fIstdscr\fP as a default screen.
For example, to add a character to \fIstdscr\fP,
one calls \fIaddch()\fP with the desired character.
If a different window is to be used, the routine \fIwaddch()\fP
(for `w'indow-specific \fIaddch()\fP) is provided.\u*\d
.FS
\u*\dActually, \fIaddch()\fP is really a ``#define'' macro with arguments,
as are most of the ``functions'' which deal with \fIstdscr\fP
as a default.
.FE
This convention of prepending function names with a ``w''
when they are to be applied to specific windows is consistent.
The only routines which do \fInot\fP do this are those
to which a window must always be specified.
.PP
In order to move the current (y, x) coordinates from one point to another,
the routines \fImove()\fP and \fIwmove()\fP are provided.
However, it is often desirable to first move and then perform some I/O
operation.
In order to avoid clumsyness, most I/O routines can be preceded by the prefix
``mv'' and the desired (y, x) coordinates then can be added to the arguments
to the function.
For example, the calls
.DS
move(y, x);
addch(ch);
.DE
can be replaced by
.DS
mvaddch(y, x, ch);
.DE
and
.DS
wmove(win, y, x);
waddch(win, ch);
.DE
can be replaced by
.DS
mvwaddch(win, y, x, ch);
.DE
Note that the window description pointer (\fIwin\fP)
comes before the added (y, x) coordinates.
If such pointers are need, they are always the first parameters passed.
.NH 1
Variables
.PP
Many variables which are used to describe the terminal environment
are available to the programmer. They are:
.TS
expand;
lw(6m) lw(8n) lw(50n).
type	name	description
_
WINDOW *	curscr	T{
.fi
current version of the screen (terminal screen).
T}
WINDOW *	stdscr	T{
standard screen.  Most updates are usually done here.
T}
int	LINES	T{
number of lines on the terminal
T}
int	COLS	T{
number of columns on the terminal
T}
int	ERR	T{
error flag returned by routines on a fail.
T}
int	OK	T{
error flag returned by routines when things go right.
T}
.TE
.LP
There are also several ``#define'' constants and types
which are of general usefulness:
.ta 11n
.DS L
bool		boolean type, actually a ``char'' (e.g., \fIbool doneit;\fR\|)
TRUE		boolean ``true'' flag (1).
FALSE		boolean ``false'' flag (0).
.DE
.NH 1
Usage
.PP
This is a description of how to actually use the screen package.
In it, we assume all updating, reading, etc. is applied to
\fIstdscr\fP.
All instructions will work on any window,
with changing the function name and parameters as mentioned above.
.NH 2
Starting up
.PP
In order to use the screen package,
the routines must know about terminal characteristics, and the space for
\fIcurscr\fP and \fIstdscr\fP must be allocated.
These functions are performed by \fIinitscr()\fP.
Since it must allocate space for the windows,
it can overflow core when attempting to do so.
On this rather rare occasion, \fIinitscr()\fP
returns ERR.
\fIinitscr()\fP must \fIalways\fP
be called before any of the routines which affect windows are used.
If it is not, the program will core dump as soon as either
\fIcurscr\fP or \fIstdscr\fP are referenced.
However, it is usually best to wait to call it
until after you are sure you will need it,
like after checking for startup errors.
Terminal status changing routines like \fInl()\fP and \fIcbreak()\fP
should be called after \fIinitscr()\fP.
.PP
Now that the screen windows have been allocated,
you can set them up for the run.
If you want to, say, allow the window to scroll, use \fIscrollok()\fP.
If you want the cursor to be left after the last change, use
\fIleaveok()\fP.
If this isn't done, \fIrefresh()\fP
will move the cursor to the window's current (y, x) coordinates
after updating it.
New windows of your own can be created, too, by using the functions
\fInewwin()\fP and \fIsubwin()\fP.
\fIdelwin()\fP will allow you to get rid of old windows.
.NH 2
Output
.PP
Now that we have set things up, we will want to actually update the terminal.
The basic functions used to change what will go on a window are \fIaddch()\fP
and \fImove()\fP.
\fIaddch()\fP adds a character at the current (y, x) coordinates,
returning ERR if it would cause the window to illegally scroll,
i.e., printing a character in the lower right-hand corner
of a terminal which automatically scrolls if scrolling is not allowed.
\fImove()\fP changes the current (y, x) coordinates to whatever you want
them to be.
It returns ERR if you try to move off the window.
As mentioned above, you can combine the two into \fImvaddch()\fP
to do both things at once.
.PP
The other output functions, such as \fIaddstr()\fP
and \fIprintw()\fP, all call \fIaddch()\fP to add characters to the window.
.PP
After you have put on the window what you want there,
when you want the portion of the terminal covered by the window
to be made to look like it, you must call \fIrefresh()\fP.
In order to optimize finding changes, \fIrefresh()\fP
assumes that any part of the window not changed since the last
\fIrefresh()\fP of that window has not been changed on the terminal, i.e.,
that you have not refreshed a portion of the terminal
with an overlapping window.
If this is not the case, the routine \fItouchwin()\fP
is provided to make it look like the entire window has been changed,
thus making \fIrefresh()\fP check the whole subsection of the terminal for
changes.
.PP
If you call \fIwrefresh()\fP with \fIcurscr()\fP,
it will make the screen look like \fIcurscr\fP thinks it looks like.
This is useful for implementing a command
which would redraw the screen in case it get messed up.
.NH 2
Input
.PP
Input is essentially a mirror image of output.
The complementary function to \fIaddch()\fP is \fIgetch()\fP which,
if echo is set, will call \fIaddch()\fP to echo the character.
Since the screen package needs to know what is on the terminal at all times,
if characters are to be echoed, the tty must be in raw or cbreak mode.
If it is not, \fIgetch()\fP sets it to be cbreak, reads in the character,
and then sets it back the way it was.
.NH 2
Miscellaneous
.PP
A plethora of other functions exist for maintaining and changing information
about the windows.
For the most part, the descriptions in section 5 should suffice.
.NH 2
Finishing Up
.PP
In order to do certain optimizations, and, on some terminals,
to work at all, some things must be done before the screen routines start up.
In order to clean up after the routines, the routine \fIendwin()\fP
is provided.
It restores tty modes to what they were when \fIinitscr()\fP
was first called, moves the cursor down to the lower-left corner, etc.
Thus, anytime after the call to initscr, \fIendwin()\fP
should be called before exiting.
.NH
Descriptions of the Functions
.de Lp
.sp
.LP
..
.LP
This section describes all the functions available to the
programmer in the curses package.  For an alphabetical list, see the
manual page \fIncurses\fP(3).
.NH 2
Initialization
.LP
These functions are called when initializing a program.
.Lp
\fBinitscr\fP()
.br
The first function called should always be \fBinitscr\fP.  This will
determine the terminal type and initialize curses data
structures.  \fBinitscr\fP also arranges that the first
call to \fBrefresh\fP will clear the screen.
.Lp
\fBendwin\fP()
.br
A program should always call \fBendwin\fP before exiting.  This
function will restore tty modes, move the cursor to the lower left
corner of the screen, reset the terminal into the proper
nonvisual mode.
.Lp
\fBnewterm\fP(type, fp)
.br
A program which outputs to more than one terminal should use
\fBnewterm\fP instead of \fBinitscr\fP.  \fBnewterm\fP should be called once for
each terminal.  It returns a variable of type \fBstruct\fP \fBscreen\fP \fB*\fP
which should be saved as a reference to that terminal.  The
arguments are the type of the terminal (a string) and a stdio FILE
pointer for output to the terminal.  The FILE pointer
should be open for both reading and writing, if input from the
terminal is desired.
The program should also call \fBendwin\fP for each terminal being used.
.Lp
\fBset_term\fP(new)
.br
This function is used to switch to a different terminal.  The
screen reference for the new terminal is passed as the parameter.
The previous terminal is returned by the function.  All other
calls affect only the current terminal.
.Lp
\fBlongname\fP()
.br
This function returns a pointer to a static area containing a
verbose description of the current terminal.  It is defined only
after a call to \fBinitscr\fP or \fBnewterm\fP.
.NH 2
Option Setting
.LP
These functions set options within curses.  In each case, \fIwin\fP is
the window affected, and \fIbf\fP is a boolean flag with value \fBTRUE\fP or
\fBFALSE\fP indicating whether to enable or disable the option.  All
options are initially \fBFALSE.\fP  It is not necessary to turn these
options off before calling \fBendwin\fP.
.Lp
\fBclearok\fP(win,bf)
.br
If set, the next call to \fBwrefresh\fP with this window will clear the
screen and redraw the entire screen.  If \fIwin\fP is \fBcurscr\fP, the next
call to \fIwrefresh\fP with any window will cause the screen to be
cleared.  This is useful when the contents of the screen are
uncertain, or in some cases for a more pleasing visual effect.
.Lp
\fBidlok\fP(win,bf)
.br
If enabled, curses will consider using the hardware insert/delete
line feature of terminals so equipped.  If disabled, curses will
never use this feature.  The insert/delete character feature is
always considered.  Enable this option only if your application
needs insert/delete line, for example, for a screen editor.  It
is disabled by default because insert/delete line is visually
annoying when used in applications where it isn't really needed.
.Lp
\fBkeypad\fP(win,bf)
.br
This option enables the keypad of the users terminal.  If
enabled, the user can press a function key (such as an arrow key)
and \fBgetch\fP will return a single value representing the function
key.  If disabled, curses will not treat function keys specially.
If the keypad in the terminal can be turned on (made to transmit)
and off (made to work locally), turning on this option will turn
on the terminal keypad.
.Lp
\fBleaveok\fP(win,bf)
.br
Normally, the hardware cursor is left at the location of the
window cursor being refreshed.  This option allows the cursor to be
left wherever the update happens to leave it.  It is useful for
applications where the cursor is not used, since it saves cursor
motions.  If possible, the cursor is made invisible when this
option is enabled.
.Lp
\fBmeta\fP(win,bf)
.br
If enabled, characters returned by \fBgetch\fP are transmitted with all
8 bits, instead of stripping the highest bit.  It is useful for
extending the non-text command set in applications where the
terminal has a meta shift key, such as EMACS.  \fINOTE\fP: This function
is currently unsupported, due to lack of support in current
teletype drivers for 8 bit input in non-raw mode.
.Lp
\fBnodelay\fP(win,bf)
.br
This option causes \fBgetch\fP to be a non-blocking call.  If no input
is ready, \fBgetch\fP will return -1.
If disabled, \fBgetch\fP will hang until a key is pressed.
.Lp
\fBscrollok\fP(win,bf)
.br
This option controls what happens when the cursor of a window is
moved off the edge of the window, either from a newline on the
bottom line, or typing the last character of the last line.  If
disabled, the cursor is left on the bottom line.  If enabled,
\fBwrefresh\fP is called on the window, and then the physical terminal
and window are scrolled up one line.
.Lp
\fBsetscrreg\fP(t,b)
.br
\fBwsetscrreg\fP(win,t,b)
.br
These functions allow the user to set a software scrolling region
in a window \fIwin\fP or \fBstdscr\fP.
\fIt\fP and \fIb\fP are the line numbers of the
top and bottom margin of the scrolling region.  (Line 0 is the
top line of the screen.) If this option and \fBscrollok\fP are enabled,
an attempt to move off the bottom margin line will cause all
lines in the scrolling region to scroll up one line.  Note that
this has nothing to do with use of a physical scrolling region
capability in the terminal, like that in the VT100.  Only the
text of the window is scrolled.
.LP
The scrolling region really acts as a sort of barrier, limiting the
area of a window over which changes take place.  For this reason, an
attempt to create a scrolling region in an area of the screen which
does not contain the current (y, x) coordinates for that window is an
error.
Similarly, attempts to move the (y, x) coordinates out of the region will
also fail with an ERR return.
.LP
When a scrolling region is in place, all changes are limited to the region.
For example, \fIerase()\fP will only erase the area inside the region;
\fIinsertln()\fP will only shift lines down to the bottom of the region,
etc.  It is anticipated that this method of controlling the area of change
will prove quite handy in a number of applications.
.LP
To disable the scrolling region, once defined, simply redefine it to be
the whole window.  For example, to disable the scrolling region on
\fIstdscr\fP, the following call would be used:
.DS
\fBsetscrreg(0, LINES - 1)\fP
.DE
For other windows, the height of the window should be used instead of
(LINES - 1).
.NH 2
Terminal Mode Setting
.LP
These functions are used to set modes in the tty driver.  The
initial mode usually depends on the setting when the program was
called: the initial modes documented here represenet the normal
situation.
.Lp
\fBcbreak\fP()
.br
\fBnocbreak\fP()
.br
\fBcrmode\fP()
.br
\fBnocrmode\fP()
.br
These functions put the terminal into and out of \fBCBREAK\fP mode.
In this mode, characters typed by the user are immediately
available to the program.  When out of this mode, the teletype driver
will buffer characters typed until newline is typed.  Interrupt
and flow control characters are unaffected by this mode. 
Initially the terminal is not in \fBCBREAK\fP mode.  Most interactive
programs using curses will set this mode.
.LP
The functions \fBcrmode\fP() and \fBnocrmode\fP() are the result of an
accident in the first version of curses and are retained solely for
upward compatibility.  \fBcrmode\fP() is the same as \fBcbreak\fP() and
\fBnocrmode\fP() is the same as \fBnocbreak\fP().
.Lp
\fBraw\fP()
.br
\fBnoraw\fP()
.br
These functions put the terminal into and out of \fBRAW\fP mode.
\fBRAW\fP mode is just like \fBCBREAK\fP mode except that \fIno\fP
special character processing is done (e.g. the interrupt character will
be passed through to the program, uninterpreted, as will the kill
character, etc.) and all 8 bits of the input character are retained;
in \fBCBREAK\fP mode, the eighth bit is stripped off before it is
given to the program.  Because of the lack of interpretation of
special characters, it is not recommended that programs use this
mode.
.Lp
\fBecho\fP()
.br
\fBnoecho\fP()
.br
These functions control whether characters typed by the user are
echoed as typed.  Initially, characters typed are echoed by the
teletype driver.  Authors of most interactive programs prefer to
do their own echoing in a controlled area of the screen, or not
to echo at all, so they disable echoing.
.Lp
\fBnl\fP()
.br
\fBnonl\fP()
.br
These functions control whether newline is translated into
carriage return and linefeed on output, and whether return is
translated into newline on input.  Initially, the translations do
occur.  By disabling these translations, curses is able to make
better use of the linefeed capability, resulting in faster cursor
motion.
.Lp
\fBresetty\fP()
.br
\fBsavetty\fP()
.br
These functions save and restore the state of the tty modes.
\fBsavetty\fP saves the current state in a buffer, \fBresetty\fP restores the
state to what it was at the last call to \fBsavetty\fP.
.NH 2
Window Manipulation
.LP
\fBnewwin\fP(num_lines, num_cols, begy, begx)
.br
Create a new window with the given number of lines and columns.
The upper left corner of the window is at line \fBbegy\fP column \fBbegx\fP.
If either \fInum_lines\fP or \fInum_cols\fP is zero,
they will be defaulted
to \fBLINES\fP-\fIbegy\fP and \fBCOLS\fP-\fIbegx\fP.
A new full-screen window is
created by calling \fBnewwin\fP(0,0,0,0).
.Lp
\fBsubwin\fP(orig, num_lines, num_cols, begy, begx)
.br
Create a new window with the given number of lines and columns.
The window is at position (\fIbegy\fP, \fIbegx\fP) on the screen.  (It is
relative to the screen, not \fIorig\fP.) The window is made in the
middle of the window \fIorig\fP, so that changes made to one window
will affect both windows.  When using this function, often it
will be necessary to call \fItouchwin\fP before calling \fIwrefresh\fP.
.Lp
\fBdelwin\fP(win)
.br
Deletes the named window, freeing up all memory associated with
it.  In the case of sub-windows, they should be deleted before the main
window.
.Lp
\fBmvwin\fP(win, by, bx)
.br
Move the window so that the upper left corner will be at position
(\fIby\fP, \fIbx\fP).  If the move would cause the window to be off the
screen, it is an error and the window is not moved.
.Lp
\fBtouchwin\fP(win)
.br
Throw away all optimization information about which parts of the
window have been touched, by pretending the entire window has
been drawn on.  This is sometimes necessary when using
overlapping windows, since a change to one window will affect the other
window, but the optimization records of the other window will not
reflect the change.
.Lp
\fBoverlay\fP(win1, win2)
.br
\fBoverwrite\fP(win1, win2)
.br
These functions overlay \fIwin1\fP on top of \fIwin2\fP, that is, all text in
\fIwin1\fP is copied into \fIwin2\fP, after lining up the two windows'
origins.
The difference between the functions is that \fBoverlay\fP is
nondestructive (blanks are not copied) while \fBoverwrite\fP is
destructive.
.NH 2
Causing Output to the Terminal
.LP
\fBrefresh\fP()
.br
\fBwrefresh\fP(win)
.br
These functions must be called to actually get any output on the
terminal, as other routines merely manipulate data structures.
\fBwrefresh\fP copies the named window to the physical terminal screen,
taking into account what is already there in order to do
optimizations.
\fBrefresh\fP is the same, using \fBstdscr\fP as a default screen.
Unless \fBleaveok\fP has been enabled, the physical cursor of the
terminal is left at the location of the window's cursor.
.Lp
\fBdoupdate\fP()
.br
\fBwnoutrefresh\fP(win)
.br
These two functions allow multiple updates with more efficiency
than \fBwrefresh.\fP  To use them, it is important to understand how
curses works.  In addition to all the window structures, curses
keeps two data structures representing the terminal screen: a
\fIphysical\fP screen, describing what is actually on the screen, and a
\fIvirtual\fP screen, describing what the programmer \fIwants\fP to have on
the screen.  \fBwrefresh\fP works by first copying the named window to
the virtual screen (\fBwnoutrefresh\fP), and then calling the routine
to update the screen (\fBdoupdate\fP).  If the programmer wishes to
output several windows at once, a series of calls to \fBwrefresh\fP
will result in alternating calls to \fBwnoutrefresh\fP and \fBdoupdate\fP,
causing several bursts of output to the screen.  By calling
\fBwnoutrefresh\fP for each window, it is then possible to call
\fBdoupdate\fP once, resulting in only one burst of output, with probably
fewer total characters transmitted.
.NH 2
Writing on Window Structures
.LP
These routines are used to ``draw'' text on windows.  In all
cases, a missing \fIwin\fP is taken to be \fBstdscr\fP.
\fIy\fP and \fIx\fP are the row
and column, respectively.  The upper left corner is always (0, 0) not (1, 1).
The \fBmv\fP functions imply a call to \fBmove\fP before the call to the
other function.
.NH 3
Moving the Cursor
.LP
\fBmove\fP(y, x)
.br
\fBwmove\fP(win, y, x)
.br
The cursor associated with the window is moved to the given
location.  This does not move the physical cursor of the terminal
until \fBrefresh\fP is called.
.NH 3
Writing One Character
.LP
\fBaddch\fP(ch)
.br
\fBwaddch\fP(win, ch)
.br
\fBmvaddch\fP(y, x, ch)
.br
\fBmvwaddch\fP(win, y, x, ch)
.br
The character \fIch\fP is put in the window at the current cursor
position of the window.  If \fIch\fP is a tab, newline, or backspace, the
cursor will be moved appropriately in the window.  If \fIch\fP is a
different control character, it will be drawn in the ^X notation.
The position of the window cursor is advanced.  At the right
margin, an automatic newline is performed.  At the bottom of the
scrolling region, if \fBscrollok\fP is enabled, the scrolling region
will be scrolled up one line.
.NH 3
Writing a String
.LP
\fBaddstr\fP(str)
.br
\fBwaddstr\fP(win,str)
.br
\fBmvaddstr\fP(y,x,str)
.br
\fBmvwaddstr\fP(win,y,x,str)
.br
These functions write all the characters of the null terminated
character string \fIstr\fP on the given window.  They are identical to
a series of calls to \fBaddch\fP.
.NH 3
Clearing Areas of the Screen
.LP
\fBerase\fP()
.br
\fBwerase\fP(win)
.br
These functions copy blanks to every position in the window.
.Lp
\fBclear\fP()
.br
\fBwclear\fP(win)
.br
These functions are like \fBerase\fP and \fBwerase\fP but they also call
\fBclearok\fP, arranging that the screen will be cleared on the next
\fBrefresh\fP.
.Lp
\fBclrtobot\fP()
.br
\fBwclrtobot\fP(win)
.br
All lines below the cursor in this window are erased.  Also, the
current line to the right of the cursor is erased.
.Lp
\fBclrtoeol\fP()
.br
\fBwclrtoeol\fP(win)
.br
The current line to the right of the cursor is erased.
.NH 3
Inserting and Deleting Text
.LP
\fBdelch\fP()
.br
\fBwdelch\fP(win)
.br
\fBmvdelch\fP(y,x)
.br
\fBmvwdelch\fP(win,y,x)
.br
The character under the cursor in the window is deleted.  All
characters to the right on the same line are moved to the left
one position.  This does not imply use of the hardware delete
character feature.
.Lp
\fBdeleteln\fP()
.br
\fBwdeleteln\fP(win)
.br
The line under the cursor in the window is deleted.  All lines
below the current line are moved up one line.  The bottom line of
the window is cleared.  This does not imply use of the hardware
delete line feature.
.Lp
\fBinsch\fP(c)
.br
\fBwinsch\fP(win, c)
.br
\fBmvinsch\fP(y,x,c)
.br
\fBmvwinsch\fP(win,y,x,c)
.br
The character \fIc\fP is inserted before the character under the
cursor.  All characters to the right are moved one space to the
right, possibly losing the rightmost character on the line.  This
does not imply use of the hardware insert character feature.
.Lp
\fBinsertln\fP()
.br
\fBwinsertln\fP(win)
.br
A blank line is inserted above the current line.  The bottom line
is lost.  This does not imply use of the hardware insert line
feature.
.NH 3
Formatted Output
.LP
\fBprintw\fP(fmt, args)
.br
\fBwprintw\fP(win, fmt, args)
.br
\fBmvprintw\fP(y, x, fmt, args)
.br
\fBmvwprintw\fP(win, y, x, fmt, args)
.br
These functions correspond to \fIprintf\fP.  The characters which would
be output by \fIprintf\fP are instead output using \fIwaddch\fP on the given
window.
.NH 3
Miscelaneous
.LP
\fBbox\fP(win, vert, hor)
.br
A box is drawn around the edge of the window.  \fIvert\fP and \fIhor\fP are
the characters the box is to be drawn with.
.Lp
\fBscroll\fP(win)
.br
The window is scrolled up one line.  This involves moving the
lines in the window data structure.
.NH 2
Querying the Contents of a Window
.LP
\fBgetyx\fP(win,y,x)
.br
The cursor position of the window is placed in the two integer
variables \fIy\fP and \fIx\fP.  Since this is a macro, no & is necessary.
.Lp
\fBinch\fP()
.br
\fBwinch\fP(win)
.br
\fBmvinch\fP(y,x)
.br
\fBmvwinch\fP(win,y,x)
.br
The character at the current position in the named window is
returned.
.NH 2
Input from the Terminal
.LP
\fBgetch\fP()
.br
\fBwgetch\fP(win)
.br
\fBmvgetch\fP(y,x)
.br
\fBmvwgetch\fP(win,y,x)
.br
A character is read from the terminal associated with the window.
In nodelay mode, if there is no input waiting, the value -1 is
returned.  In delay mode, the program will hang until a character is typed.
.Lp
If \fIkeypad\fP mode is enabled, and a function key is pressed, the
code for that function key will be returned instead of the raw
characters.  Possible function keys are defined with integers
beginning with 0401, whose names begin with KEY_, defined in
<ncurses.h>.  If a character is received that could be the
beginning of a function key (such as escape), curses will set a one
second timer.  If the remainder of the sequence does not come in
within one second, the character will be passed through,
otherwise the function key value will be returned.  For this reason,
on many terminals, there will be a one second delay after a user
presses the escape key.  (Use by a programmer of the escape key
for a single character function is discouraged.)
.Lp
\fBgetstr\fP(str)
.br
\fBwgetstr\fP(win,str)
.br
\fBmvgetstr\fP(y,x,str)
.br
\fBmvwgetstr\fP(win,y,x,str)
.br
A series of calls to \fIgetch\fP is made, until a newline is received.
The resulting value is placed in the area pointed at by the
character pointer \fIstr\fP.  The users erase and kill characters are
interpreted, and the string is echoed.
.Lp
\fBscanw\fP(fmt, args)
.br
\fBwscanw\fP(win, fmt, args)
.br
\fBmvscanw\fP(y, x, fmt, args)
.br
\fBmvwscanw\fP(win, y, x, fmt, args)
.br
This function corresponds to \fIscanf\fP.  \fIwgetstr\fP is called on the
window, and the resulting line is used as input for the scan.
.NH 2
Video Attributes
.LP
\fBattroff\fP(at)
.br
\fBwattroff\fP(win, attrs)
.br
\fBattron\fP(at)
.br
\fBwattron\fP(win, attrs)
.br
\fBattrset\fP(at)
.br
\fBwattrset\fP(win, attrs)
.br
\fBstandout\fP()
.br
\fBstandend\fP()
.br
\fBwstandout\fP(win)
.br
\fBwstandend\fP(win)
.br
These functions set the \fIcurrent\fP \fIattributes\fP of the named window.
These attributes can be any combination of \fBA_STANDOUT\fP, \fBA_REVERSE\fP,
\fBA_BOLD\fP, \fBA_DIM\fP, \fBA_BLINK\fP, \fBA_BLANK\fP, \fBA_UNDERLINE\fP,
\fBA_PROTECT\fP, and \fBA_ALTCHARSET\fP.
These constants are defined in <ncurses.h> and can
be combined with the C | (or) operator.
.LP
The current attributes of a window are applied to all characters
that are written into the window.  Attributes are a
property of the character, and move with the character through
any scrolling and insert/delete line/character operations.  To
the extent possible on the particular terminal, they will be
displayed as the graphic rendition of characters put on the
screen.
.LP
\fIattrset\fP(at) sets the current attributes of the given window to
\fIat\fP.  \fIattroff\fP(at) turns off the named attributes without affecting
any other attributes.  \fIattron\fP(at) turns on the named attributes
without affecting any others.  \fIstandout\fP is the same as
\fIattrset\fP(A_STANDOUT) \fIstandend\fP is the same as \fIattrset\fP(0),
that is, it turns off all attributes.
.NH 2
Bells and Flashing Lights
.LP
\fBbeep\fP()
.br
\fBflash\fP()
.br
These functions are used to signal the programmer.  \fIbeep\fP will
sound the audible alarm on the terminal, if possible, and if not,
will flash the screen (visible bell), if that is possible.  \fIflash\fP
will flash the screen, and if that is not possible, will sound
the audible signal.  If neither signal is possible, nothing will
happen.  Nearly all terminals have an audible signal (bell or
beep) but only some can flash the screen.
.NH 2
Portability Functions
.LP
These functions do not have anything to do with terminal
dependent character output, but tend to be needed by programs that use
curses.  Unfortunately, their implemention varies from one
version of UNIX\u*\d to another.  They have been included here to
enhance the portability of programs using curses.
.FS
\u*\d UNIX is a trademark of Bell Laboratories.
.FE
.Lp
\fBbaudrate\fP()
.br
\fIbaudrate\fP returns the output speed of the terminal.  The number
returned is the integer baud rate, for example, 9600, rather than
a table index such as \fBB9600\fP.
.Lp
\fBerasechar\fP()
.br
The erase character chosen by the user is returned.  This is the
character typed by the user to erase the character just typed.
.Lp
\fBkillchar\fP()
.br
The line kill character chosen by the user is returned.  This is
the character typed by the user to forget the entire line being
typed.
.Lp
\fBflushinp\fP()
.br
\fIflushinp\fP throws away any typeahead that has been typed by the
user and has not yet been read by the program.
.NH 2
Debugging
.LP
These functions are useful when debugging a program with curses.
.Lp
\fBunctrl\fP(ch)
.br
This macro expands to a character string which is a printable
representation of the character \fIch\fP.  The program must include the
file <unctrl.h>.  Control characters are displayed in the ^x
notation.  Printing characters are displayed as is.
.Lp
\fBtraceoff\fP()
.br
\fBtraceon\fP()
.br
It is possible to compile a debugging version of curses with
tracing turned on, and with the -g option for sdb.  This library
may be available on your system as -ldcurses.  When using this
version, the file ``trace'' will be created each time the program
is run, containing verbose information showing each step done by
curses.  This output is useful for finding bugs in curses, and
may be useful for finding bugs in user programs.  Since the
output is so verbose, with any bug that cannot be easily and quickly
reproduced, it may be necessary to turn the debugging output off
in some parts of the program.  These functions can be used to
turn tracing off and back on.  When \fIinitscr\fP is first called,
tracing is automatically turned on.
.NH 2
Lower Level Functions
.LP
These functions are provided for programs not needing the screen
optimization capabilities of curses.  Programs are discouraged
from working at this level, since they must handle various
glitches in certain terminals.  However, a program can be smaller
if it only brings in the low level routines.
.NH 3
Cursor Motion
.Lp
\fBgettmode\fP()
.br
\fBsetterm\fP(type)
.br
These two initialization routines are provided for upward
compatibility with the old curses.  \fIgettmode\fP does nothing.
\fIsetterm\fP
results in a call to \fIsetupterm\fP with appropriate arguments.
.Lp
\fBmvcur\fP(oldrow, oldcol, newrow, newcol)
.br
This routine optimally moves the cursor from (oldrow, oldcol) to
(newrow, newcol).  The user program is expected to keep track of
the current cursor position.  Note that unless a full screen
image is kept, curses will have to make pessimistic assumptions,
sometimes resulting in less than optimal cursor motion.  For
example, moving the cursor a few spaces to the right can be done
by transmitting the characters being moved over, but if curses
does not have access to the screen image, it doesn't know what
these characters are.
.LP
If either of oldcol or oldrow are negative, \fImvcur()\fP will refrain
from using any relative motions.  This is handy for occasions when
a program is unsure as to the current cursor location.
.NH 3
Terminfo Level
.LP
These routines are called by low level
programs that need access to specific capabilities of terminfo.
A program working at this level should include both <ncurses.h>
and <term.h>.  After a call to \fIsetupterm\fP, the
capabilities will be available with macro names defined in
<term.h>.  See \fBterminfo\fP(5) for a detailed description of the
capabilies.  If the program only needs to handle one terminal,
the definition \fB-DSINGLE\fP can be passed to the C compiler,
resulting in static references to capabilities instead of dynamic
references.  This can result in smaller code, but prevents use of
more than one terminal at a time.  Very few programs use more
than one terminal, so almost all programs can use this flag.
.Lp
\fBsetupterm\fP(term, filenum, errret)
.br
This routine is called to initialize a terminal.  \fIterm\fP is the
character string representing the name of the terminal being
used.  \fIfilenum\fP is the UNIX file descriptor of the terminal being
used for output.  \fIerrret\fP is a pointer to an integer, in which a
success or failure indication is returned.  The values returned
can be 1 (all is well), 0 (no such terminal), or -1 (some problem
locating the terminfo database).
.LP
The value of \fIterm\fP can be given as 0, which will cause the value
of TERM in the environment to be used.  The \fIerrret\fP pointer can
also be given as 0, meaning no error code is wanted.  If \fIerrret\fP
is defaulted, and something goes wrong, \fIsetupterm\fP will print an
appropriate error message and exit, rather than returning.  Thus,
a simple program can call \fIsetupterm\fP(0, 1, 0) and not worry about
initialization errors.
.LP
\fIsetupterm\fP will check the tty driver mode bits, and change any
that might prevent the correct operation of other low level
routines.  Currently, the mode that expands tabs into spaces is
disabled, because the tab character is sometimes used for different
functions by different terminals.  (Some terminals use it to move
right one space.  Others use it to address the cursor to row or
column 9.) If the system is expanding tabs, \fIsetupterm\fP will remove
the definition of the \fBtab\fP and \fBbacktab\fP functions, assuming that
since the user is not using hardware tabs, they may not be
properly set in the terminal.
.LP
After the call to \fIsetupterm\fP, the global variable \fIcur_term\fP is set
to point to the current structure of terminal capabilities.  By
calling \fIsetupterm\fP for each terminal, and saving and restoring
\fIcur_term\fP, it is possible for a program to use two or more
terminals at once.  \fISetupterm\fP also stores the names section of the
terminal description in the global character array \fIttytype[]\fP.
Subsequent calls to \fIsetupterm\fP will overwrite this array, so you'll
have to save it yourself if need be.
.LP
The mode that turns newlines into CRLF on output is not disabled.
Programs that use \fBcud1\fP or \fBind\fP should avoid these capabilities if
their value is linefeed unless they disable this mode.  \fIsetupterm\fP
calls \fIfixterm\fP after any changes it makes.
.Lp
\fBfixterm\fP()
.br
\fBresetterm\fP()
.br
\fBsaveterm\fP()
.br
These routines can be used to change the tty modes between the
two states: \fInormal\fP (the mode they were in before the program was
started) and \fIprogram\fP (the mode needed by the program).  \fIfixterm\fP
puts the terminal into program mode, and \fIresetterm\fP puts the
terminal into normal mode.  These functions are useful for shell
escapes and control-Z suspensions.  In addition, all programs
must call \fIresetterm\fP before they exit.
.LP
The routine \fBsaveterm\fP saves the current state of the tty modes
so that the next time \fBfixterm\fP is called, the same modes will be
used.  This is useful for programs which use some of the functions
described in section 2.3 to tailor the modes.
.LP
Normal mode is stored in \fIcur_term\fP->\fIOttyb\fP, and program mode is in
\fIcur_term\fP->\fINttyb\fP.  These structures are both of type SGTTYB (which
varies depending on the system).  Currently the only possible type
is \fBstruct\fP \fBsgttyb\fP.
.Lp
\fBvidattr\fP(newmode)
.br
\fBvidputs\fP(newmode, outc)
.br
\fInewmode\fP is any combination of attributes, defined in <ncurses.h>.
The proper string to put the terminal in the given video mode is
output. The routine \fBvidattr\fP() sends the output characters to
\fBputchar\fP; \fBvidputs\fP sends them to the given routine \fIoutc\fP,
one character at a time.  That routine should therefore expect one
\fBchar\fP parameter.
The previous mode is remembered by this routine.
.Lp
\fBtparm\fP(instring, p1, p2, p3, p4, p5, p6, p7, p8, p9)
.br
\fItparm\fP is used to instantiate a parameterized string.  The
character string returned is suitable for \fItputs\fP.  Up to 9
parameters can be passed, in addition to the parameterized string.
.Lp
\fBtputs\fP(cp, affcnt, outc)
.br
A string capability, possibly containing padding information, is
processed.  Enough padding characters to delay for the specified
time replace the padding specification, and the resulting string
is passed, one character at a time, to the routine \fIoutc\fP, which
should expect one character parameter.  (This routine often just
calls \fIputchar\fP.) \fIcp\fP is the capability string.  \fIaffcnt\fP is the
number of units affected by the capability, which varies with the
particular capability.  (For example, the \fIaffcnt\fP for \fIinsert_line\fP
is the number of lines below the inserted line on the screen,
that is, the number of lines that will have to be moved by the
terminal.) \fIaffcnt\fP is used by the padding information of some
terminals as a multiplication factor.  If the capability does not
have a factor, the value 1 should be passed.
.Lp
\fBputp\fP(str)
.br
This is a convenient function to output a capability with no
\fIaffcnt\fP.
The string is output to \fIputchar\fP with an \fIaffcnt\fP of 1.
It can be used in simple applications that do not need to process
the output of \fItputs\fP.
