// NVMe datastructures and constants
//
// Copyright 2017 Amazon.com, Inc. or its affiliates.
//
// This file may be distributed under the terms of the GNU LGPLv3 license.

#ifndef __NVME_INT_H
#define __NVME_INT_H

#include <grub/types.h>

/* Data structures */

/* The register file of a NVMe host controller. This struct follows the naming
   scheme in the NVMe specification. */
struct nvme_reg {
    grub_uint64_t cap;                    /* controller capabilities */
    grub_uint32_t vs;                     /* version */
    grub_uint32_t intms;                  /* interrupt mask set */
    grub_uint32_t intmc;                  /* interrupt mask clear */
    grub_uint32_t cc;                     /* controller configuration */
    grub_uint32_t _res0;
    grub_uint32_t csts;                   /* controller status */
    grub_uint32_t _res1;
    grub_uint32_t aqa;                    /* admin queue attributes */
    grub_uint64_t asq;                    /* admin submission queue base address */
    grub_uint64_t acq;                    /* admin completion queue base address */
};

/* Submission queue entry */
struct nvme_sqe {
    union {
        grub_uint32_t dword[16];
        struct {
            grub_uint32_t cdw0;           /* Command DWORD 0 */
            grub_uint32_t nsid;           /* Namespace ID */
            grub_uint64_t _res0;
            grub_uint64_t mptr;           /* metadata ptr */

            grub_uint64_t dptr_prp1;
            grub_uint64_t dptr_prp2;
        };
    };
};

/* Completion queue entry */
struct nvme_cqe {
    union {
        grub_uint32_t dword[4];
        struct {
            grub_uint32_t cdw0;
            grub_uint32_t _res0;
            grub_uint16_t sq_head;
            grub_uint16_t sq_id;
            grub_uint16_t cid;
            grub_uint16_t status;
        };
    };
};

/* The common part of every submission or completion queue. */
struct nvme_queue {
    grub_uint32_t *dbl;                   /* doorbell */
    grub_uint16_t mask;                   /* length - 1 */
};

struct nvme_cq {
    struct nvme_queue common;
    struct nvme_cqe *cqe;

    /* We have read upto (but not including) this entry in the queue. */
    grub_uint16_t head;

    /* The current phase bit the controller uses to indicate that it has written
       a new entry. This is inverted after each wrap. */
    unsigned phase : 1;
};

struct nvme_sq {
    struct nvme_queue common;
    struct nvme_sqe *sqe;

    /* Corresponding completion queue. We only support a single SQ per CQ. */
    struct nvme_cq *cq;

    /* The last entry the controller has fetched. */
    grub_uint16_t head;

    /* The last value we have written to the tail doorbell. */
    grub_uint16_t tail;
};

struct nvme_ctrl {
    grub_pci_device_t pci;
    struct nvme_reg volatile *reg;

    grub_uint32_t ctrlnum;

    grub_uint32_t doorbell_stride;        /* in bytes */

    struct nvme_sq admin_sq;
    struct nvme_cq admin_cq;

    grub_uint32_t ns_count;

    struct nvme_sq io_sq;
    struct nvme_cq io_cq;
};

struct nvme_namespace {
    struct nvme_namespace *next;
    struct nvme_namespace **prev;

    char *devname;

    grub_uint32_t nsnum;

    struct nvme_ctrl *ctrl;

    grub_uint32_t ns_id;

    grub_uint64_t lba_count;              /* The total amount of sectors. */

    grub_uint32_t block_size;
    grub_uint32_t metadata_size;
    grub_uint32_t max_req_size;
};

/* Data structures for NVMe admin identify commands */

struct nvme_identify_ctrl {
    grub_uint16_t vid;
    grub_uint16_t ssvid;
    char sn[20];
    char mn[40];
    char fr[8];

    grub_uint8_t rab;
    grub_uint8_t ieee[3];
    grub_uint8_t cmic;
    grub_uint8_t mdts;

    char _boring[516 - 78];

    grub_uint32_t nn;                     /* number of namespaces */
};

struct nvme_identify_ns_list {
    grub_uint32_t ns_id[1024];
};

struct nvme_lba_format {
    grub_uint16_t ms;
    grub_uint8_t  lbads;
    grub_uint8_t  rp;
};

struct nvme_identify_ns {
    grub_uint64_t nsze;
    grub_uint64_t ncap;
    grub_uint64_t nuse;
    grub_uint8_t  nsfeat;
    grub_uint8_t  nlbaf;
    grub_uint8_t  flbas;

    char _boring[128 - 27];

    struct nvme_lba_format lbaf[16];
};

union nvme_identify {
    struct nvme_identify_ns      ns;
    struct nvme_identify_ctrl    ctrl;
    struct nvme_identify_ns_list ns_list;
};

/* NVMe constants */

#define NVME_CAP_CSS_NVME (1ULL << 37)

#define NVME_CSTS_FATAL   (1U <<  1)
#define NVME_CSTS_RDY     (1U <<  0)

#define NVME_CC_EN        (1U <<  0)

#define NVME_SQE_OPC_ADMIN_CREATE_IO_SQ 1U
#define NVME_SQE_OPC_ADMIN_CREATE_IO_CQ 5U
#define NVME_SQE_OPC_ADMIN_IDENTIFY     6U

#define NVME_SQE_OPC_IO_WRITE 1U
#define NVME_SQE_OPC_IO_READ  2U

#define NVME_ADMIN_IDENTIFY_CNS_ID_NS       0U
#define NVME_ADMIN_IDENTIFY_CNS_ID_CTRL     1U
#define NVME_ADMIN_IDENTIFY_CNS_GET_NS_LIST 2U

#define NVME_CQE_DW3_P (1U << 16)

#define NVME_PAGE_SIZE 4096
#define NVME_PAGE_MASK ~(NVME_PAGE_SIZE - 1)

/* Length for the queue entries. */
#define NVME_SQE_SIZE_LOG 6
#define NVME_CQE_SIZE_LOG 4

#endif

/* EOF */
