/* read functions.

Copyright (C) 1999 Politecnico di Torino

This file is part of the NDIS Packet capture driver.

The GNU C Library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

The GNU C Library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with the GNU C Library; see the file COPYING.LIB.  If
not, write to the Free Software Foundation, Inc., 675 Mass Ave,
Cambridge, MA 02139, USA.  */

#include <basedef.h>
#include <vmm.h>
#include <vwin32.h>
#include <winerror.h>
#include <ndis.h>
#include "debug.h"
#include "packet.h"
#include "..\Inc\ntddpack.h"
#pragma VxD_LOCKED_CODE_SEG
#pragma VxD_LOCKED_DATA_SEG
DWORD _stdcall MyPageLock(DWORD, DWORD);
void  _stdcall MyPageUnlock(DWORD, DWORD);

/************************************************************
Allocates the space for a packet, extracting it from the 
buffer reserved for the driver
************************************************************/
VOID
PacketAllocatePacketBuffer(	PNDIS_STATUS	pStatus,
							POPEN_INSTANCE	pOpen,
							PNDIS_PACKET	*lplpPacket,
							PDIOCPARAMETERS	pDiocParms,
							DWORD			FunctionCode )
{
	PNDIS_BUFFER		pNdisBuffer;
	PPACKET_RESERVED	pReserved;
	TRACE_ENTER( "PacketAllocatePacket" );
	NdisAllocatePacket( pStatus, lplpPacket, pOpen->PacketPool );
	if ( *pStatus != NDIS_STATUS_SUCCESS ) 
	{
		IF_VERY_LOUD( "Read- No free packets" );
		*(DWORD *)(pDiocParms->lpcbBytesReturned) = 0;
		return;
	}
	InitializeListHead( &(RESERVED(*lplpPacket)->ListElement) );
	pReserved = RESERVED(*lplpPacket);
	switch ( FunctionCode )
	{
	case IOCTL_PROTOCOL_READ:
		pReserved->lpBuffer = (PVOID)PacketPageLock( (PVOID)pDiocParms->lpvOutBuffer, 
									 				 pDiocParms->cbOutBuffer );
		pReserved->cbBuffer = pDiocParms->cbOutBuffer;
		break;
	case IOCTL_PROTOCOL_WRITE:
		pReserved->lpBuffer = (PVOID)PacketPageLock( pDiocParms->lpvInBuffer, 
									 				 pDiocParms->cbInBuffer );
		pReserved->cbBuffer = pDiocParms->cbInBuffer;
		break;
	default:
		/*function not valid, free the resource*/
		IF_TRACE_MSG( "Allocate- Invalid FunctionCode %x", FunctionCode );
		NdisReinitializePacket( *lplpPacket );
		NdisFreePacket( *lplpPacket );
		*(DWORD *)(pDiocParms->lpcbBytesReturned) = 0;
		*pStatus = NDIS_STATUS_NOT_ACCEPTED;
		return;
	}
	pReserved->lpcbBytesReturned	= 
			(PVOID)PacketPageLock( (PVOID)pDiocParms->lpcbBytesReturned, sizeof(DWORD) );
	pReserved->lpoOverlapped		= 
	
			(PVOID)PacketPageLock( (PVOID)pDiocParms->lpoOverlapped, sizeof(OVERLAPPED) );
	pReserved->hDevice 				= pDiocParms->hDevice;
	pReserved->tagProcess			= pDiocParms->tagProcess;
	
	NdisAllocateBuffer(	pStatus, 
						&pNdisBuffer, 
						pOpen->BufferPool, 
						(PVOID)pReserved->lpBuffer,
						pDiocParms->cbOutBuffer );
	if ( *pStatus != NDIS_STATUS_SUCCESS )
	{
		IF_TRACE( "Read- No free buffers" );
		NdisReinitializePacket(*lplpPacket);
		NdisFreePacket(*lplpPacket);
		*(DWORD *)(pDiocParms->lpcbBytesReturned) = 0;
		return;
	}
	NdisChainBufferAtFront( *lplpPacket, pNdisBuffer );
	IF_PACKETDEBUG( PACKET_DEBUG_VERY_LOUD ) 
	{
		IF_TRACE_MSG( " lplpPacket : %lx", lplpPacket );		
		IF_TRACE_MSG( "   lpPacket : %lx", *lplpPacket );		
		IF_TRACE_MSG3( "pNdisBuffer : %lx  %lx  %lx", pNdisBuffer, (*lplpPacket)->Private.Head, (*lplpPacket)->Private.Tail );
		IF_TRACE_MSG( "   Reserved : %lx", pReserved );		
		IF_TRACE_MSG4( "   lpBuffer : %lx  %lx  %lx  %lx", pReserved->lpBuffer, pNdisBuffer->VirtualAddress, pDiocParms->lpvOutBuffer, pDiocParms->lpvInBuffer );
		IF_TRACE_MSG3( "   cbBuffer : %lx  %lx  %lx", pReserved->cbBuffer, pDiocParms->cbOutBuffer, pDiocParms->cbInBuffer );
		IF_TRACE_MSG2( " lpcbBytes  : %lx  %lx", pReserved->lpcbBytesReturned, pDiocParms->lpcbBytesReturned );
		IF_TRACE_MSG2( " lpoOverlap : %lx  %lx", pReserved->lpoOverlapped, pDiocParms->lpoOverlapped );
		IF_TRACE_MSG2( "    hDevice : %lx  %lx", pReserved->hDevice, pDiocParms->hDevice );
		IF_TRACE_MSG2( " tagProcess : %lx  %lx", pReserved->tagProcess, pDiocParms->tagProcess );
	}
	PACKETASSERT( pReserved->lpBuffer );
	PACKETASSERT( pReserved->cbBuffer );
	PACKETASSERT( pReserved->lpcbBytesReturned );
	PACKETASSERT( pReserved->lpoOverlapped );
	PACKETASSERT( pReserved->hDevice == pDiocParms->hDevice );
	PACKETASSERT( pReserved->tagProcess == pDiocParms->tagProcess );
	PACKETASSERT( pNdisBuffer == (*lplpPacket)->Private.Head );
	PACKETASSERT( pNdisBuffer->VirtualAddress == pReserved->lpBuffer );
	TRACE_LEAVE( "PacketAllocatePacket" );
	return;
}

/************************************************************
Move a portion of the circular buffer 
updating the head every 1024 bytes
************************************************************/

void PacketMoveMem(PVOID Destination,
				   PVOID Source, 
				   ULONG Length,
				   UINT	 *Bhead)
{
ULONG WordLength;
UINT n,i,NBlocks;

	WordLength=Length>>2;
	NBlocks=WordLength>>8;
	
	for(n=0;n<NBlocks;n++){
		for(i=0;i<256;i++){
			*((PULONG)Destination)++=*((PULONG)Source)++;
		}
	*Bhead+=1024;
	}

	n=WordLength-(NBlocks<<8);
	for(i=0;i<n;i++){
		*((PULONG)Destination)++=*((PULONG)Source)++;
	}
	*Bhead+=n<<2;
	
	n=Length-(WordLength<<2);
	for(i=0;i<n;i++){
		*((PUCHAR)Destination)++=*((PUCHAR)Source)++;
	}
	*Bhead+=n;
}

/************************************************************
Start a read
************************************************************/
DWORD
PacketRead( POPEN_INSTANCE	Open,
			DWORD  			dwDDB,
            DWORD  			hDevice,
		  	PDIOCPARAMETERS pDiocParms
	)
{
	NDIS_STATUS		Status;
	PNDIS_PACKET	pPacket;
    PUCHAR				packp;//buffer that maps the application memory
	UINT				i;
	ULONG				Input_Buffer_Length;
	UINT				Thead;
	UINT				Ttail;
	UINT				TLastByte;
	PUCHAR				CurrBuff;
	UINT				cplen;
	UINT				CpStart;


	TRACE_ENTER( "PacketRead" );
	if ( pDiocParms->cbOutBuffer < ETHERNET_HEADER_LENGTH ) 
	{
		/*the application's buffer if too small to contain the packet*/
		*(DWORD *)(pDiocParms->lpcbBytesReturned) = 0;
		IF_VERY_LOUD( "Read- Buffer too small" );
		TRACE_LEAVE( "ReadPacket" );
		return NDIS_STATUS_FAILURE;
	}

	/*See if there are packets in the buffer*/

	Thead=Open->Bhead;
	Ttail=Open->Btail;
	TLastByte=Open->BLastByte;

	if (Thead == Ttail)
	{
	/*there aren't buffered packet: the application must wait*/


	PacketAllocatePacketBuffer( &Status, Open, &pPacket, pDiocParms, IOCTL_PROTOCOL_READ );
	if ( Status == NDIS_STATUS_SUCCESS )
	{
		PACKETASSERT( Open != NULL );
		PACKETASSERT( pPacket != NULL );
		NdisAcquireSpinLock( &Open->RcvQSpinLock );
		InsertTailList( &Open->RcvList, &RESERVED(pPacket)->ListElement );
		NdisReleaseSpinLock( &Open->RcvQSpinLock );
		IF_TRACE_MSG2( "RcvList Link : %lx  %lx", Open->RcvList.Blink, &RESERVED(pPacket)->ListElement );
		PACKETASSERT( Open->RcvList.Blink == &RESERVED(pPacket)->ListElement );
		PACKETASSERT( &(Open->RcvList) == RESERVED(pPacket)->ListElement.Flink );
	}
	TRACE_LEAVE( "PacketRead" );
	return(-1);	/*Communicate that the operation is asynchronous*/	
	}
	

	
	/*there is at least a buffered packet
	the read call can be completed*/

	Input_Buffer_Length=pDiocParms->cbOutBuffer;
	packp=(PUCHAR)pDiocParms->lpvOutBuffer;
	i=0;
	/*get the address of the buffer*/
	CurrBuff=Open->Buffer;
	
	/*fill the application buffer*/

	/*first of all see if it we can copy all the buffer in one time*/
	if(Ttail>Thead){
		if((Ttail-Thead)<Input_Buffer_Length){
			PacketMoveMem(packp,CurrBuff+Thead,Ttail-Thead,&(Open->Bhead));
			*(DWORD *)(pDiocParms->lpcbBytesReturned)=Ttail-Thead;
			return(NDIS_STATUS_SUCCESS);
		}
	}
	else if((TLastByte-Thead)<Input_Buffer_Length){
			PacketMoveMem(packp,CurrBuff+Thead,TLastByte-Thead,&(Open->Bhead));
			Open->BLastByte=Ttail;
			Open->Bhead=0;
			*(DWORD *)(pDiocParms->lpcbBytesReturned)=TLastByte-Thead;
			return(NDIS_STATUS_SUCCESS);
	}

	/*scan the buffer*/
	CpStart=Thead;
	while(TRUE){
		if(Thead==Ttail)break;
		if(Thead==TLastByte){
			PacketMoveMem(packp,CurrBuff+CpStart,Thead-CpStart,&(Open->Bhead));
//			NdisMoveMemory(packp,CurrBuff+CpStart,Thead-CpStart);
			packp+=(Thead-CpStart);
			Open->Bhead=0;
			Thead=0;
			CpStart=0;
		}
		cplen=((struct bpf_hdr*)(CurrBuff+Thead))->bh_caplen+sizeof(struct bpf_hdr);
		if((i+cplen > Input_Buffer_Length))break; //no more space in the application buffer
		cplen=Packet_WORDALIGN(cplen);
		i+=cplen;
		Thead+=cplen;
	}

//	NdisMoveMemory(packp,CurrBuff+CpStart,Thead-CpStart);
	PacketMoveMem(packp,CurrBuff+CpStart,Thead-CpStart,&(Open->Bhead));
//	Open->Bhead=Thead;

	*(DWORD *)(pDiocParms->lpcbBytesReturned)=i;
    return(NDIS_STATUS_SUCCESS);
	
}

/************************************************************
Callback routine called by NDIS when the adapter receives a
packet
************************************************************/
NDIS_STATUS NDIS_API
Packet_tap (	IN NDIS_HANDLE ProtocolBindingContext,
				IN NDIS_HANDLE MacReceiveContext,
				IN PVOID       pvHeaderBuffer,
				IN UINT        uiHeaderBufferSize,
				IN PVOID       pvLookAheadBuffer,
				IN UINT        uiLookaheadBufferSize,
				IN UINT        uiPacketSize
	)
#define pOpen	((POPEN_INSTANCE)ProtocolBindingContext)
{
	PLIST_ENTRY			PacketListEntry;
	PNDIS_PACKET   		pPacket;
	ULONG          		ulSizeToTransfer;
	NDIS_STATUS    		Status;
	UINT           		uiBytesTransferred;
	PPACKET_RESERVED	pReserved;
	PNDIS_BUFFER		pNdisBuffer;
	PVOID				pvActualVirtualAddress;
	UINT				uiActualLength;
	PUCHAR				CurrBuff;
	UINT				Thead;
	UINT				Ttail;
	UINT				TLastByte;
    PNDIS_PACKET        pPacketb;
	struct bpf_hdr		*header;
	LARGE_INTEGER		CapTime;
	__int64				lCapTime;
	UINT				fres; //filter result
	UINT				maxbufspace;

	PACKETASSERT( (pOpen != NULL) );

	pOpen->Received++;		/*number of packets received by filter ++*/

	/*
	Check if the lookahead buffer follows the mac header.
	If the data follow the header (i.e. there is only a buffer) a normal bpf_filter() is
	executed on the packet.
	Otherwise if there are 2 separate buffers (this could be the case of LAN emulation or
	things like this) bpf_filter_with_2_buffers() is executed.
	*/

	if((int)pvLookAheadBuffer-(int)pvHeaderBuffer != uiHeaderBufferSize)
		fres=bpf_filter_with_2_buffers((struct bpf_insn*)(pOpen->bpfprogram),
									   pvHeaderBuffer,
									   pvLookAheadBuffer,
									   uiHeaderBufferSize,
									   uiPacketSize+uiHeaderBufferSize,
									   uiLookaheadBufferSize+uiHeaderBufferSize);
	else
		fres=bpf_filter((struct bpf_insn*)(pOpen->bpfprogram),
		                pvHeaderBuffer,
						uiPacketSize+uiHeaderBufferSize,
						uiLookaheadBufferSize+uiHeaderBufferSize);

	if(fres==0)return NDIS_STATUS_NOT_ACCEPTED;
	if( fres==-1)fres=uiPacketSize+uiHeaderBufferSize;	//if the filter returns -1 the whole packet must be accepted

	/*see if there are pending requests*/
	NdisAcquireSpinLock( &pOpen->RcvQSpinLock );
	PacketListEntry = PacketRemoveHeadList( &pOpen->RcvList );
	NdisReleaseSpinLock( &pOpen->RcvQSpinLock );

	if ( PacketListEntry == NULL )
	{
	/*no requests. This means that the application is not performing a read in this 
	  moment: the packet must go in the buffer*/

	TRACE_ENTER( "IndicateReceive" );

	if(pOpen->BufSize==0)return NDIS_STATUS_NOT_ACCEPTED;

	Thead=pOpen->Bhead;
	Ttail=pOpen->Btail;
	TLastByte=pOpen->BLastByte;


	maxbufspace=fres+sizeof(struct bpf_hdr);
	if(Ttail+maxbufspace>=pOpen->BufSize){
		if(Thead<=maxbufspace)
		{
			/*the buffer is full: the packet is lost*/
			pOpen->Dropped++;
			return NDIS_STATUS_NOT_ACCEPTED;
		}
		else{
			Ttail=0;
		}
	}

	if((Ttail<Thead)&&(Ttail+maxbufspace>=Thead))
	{
		/*the buffer is full: the packet is lost*/
		pOpen->Dropped++;
		return NDIS_STATUS_NOT_ACCEPTED;
	}
	CurrBuff=pOpen->Buffer+Ttail;


	/*allocate the ndis buffer*/
	NdisAllocateBuffer(	&Status, 
						&pNdisBuffer, 
						pOpen->BufferPool, 
						CurrBuff+sizeof(struct bpf_hdr),
						fres);

	if (Status != NDIS_STATUS_SUCCESS)
		{
			pOpen->Dropped++;
			return NDIS_STATUS_NOT_ACCEPTED;
		}

	/*allocate the packet from NDIS*/
	NdisAllocatePacket(&Status,&pPacketb,pOpen->PacketPool);
	    if (Status != NDIS_STATUS_SUCCESS)
		{
			NdisFreeBuffer( pNdisBuffer );
			pOpen->Dropped++;
			return NDIS_STATUS_NOT_ACCEPTED;
		}
	/*link the buffer to the packet*/
	NdisChainBufferAtFront(pPacketb,pNdisBuffer);

	if ( uiHeaderBufferSize > 0 ) 
	{
		if ( uiHeaderBufferSize > pNdisBuffer->Length )
			uiHeaderBufferSize = pNdisBuffer->Length;
		/*copy the header*/
		NdisMoveMemory(pNdisBuffer->VirtualAddress, pvHeaderBuffer, uiHeaderBufferSize );		
		uiBytesTransferred = uiHeaderBufferSize;
		(BYTE *)(pNdisBuffer->VirtualAddress) += uiHeaderBufferSize;
		pNdisBuffer->Length -= uiHeaderBufferSize;
	}
	
//provare a togliere
	if ( uiPacketSize >= 1500 )
	{
		_asm { int 3 }
	}

	ulSizeToTransfer = uiPacketSize/*uiPacketSize - uiLookaheadBufferSize*/;
	if ( ulSizeToTransfer > pNdisBuffer->Length )
		ulSizeToTransfer = pNdisBuffer->Length;

	/*Copy the remaining part of the packet*/
	NdisTransferData(	&Status,					
						pOpen->AdapterHandle,		
						MacReceiveContext,			
						0,		
						ulSizeToTransfer,			
						pPacketb,					
						&uiBytesTransferred );		

	uiBytesTransferred+=uiHeaderBufferSize;


	/*get the capture time*/
	lCapTime=QuerySystemTime();

 	/*fill the bpf header for this packet*/
	if( fres>uiBytesTransferred )fres=uiBytesTransferred;
	lCapTime+=pOpen->StartTime;
	header=(struct bpf_hdr*)CurrBuff;
	header->bh_tstamp.tv_usec=(long)((lCapTime%1250000)*1000000/1250000);
	header->bh_tstamp.tv_sec=(long)((lCapTime)/1250000);
	header->bh_caplen=(UINT)fres;
	header->bh_datalen=(UINT)uiPacketSize+uiHeaderBufferSize;
	header->bh_hdrlen=sizeof(struct bpf_hdr);
				
	/*update the buffer*/	
	Ttail+=Packet_WORDALIGN(fres+sizeof(struct bpf_hdr));
	if(Ttail>Thead)TLastByte=Ttail;
	pOpen->Btail=Ttail;
	pOpen->BLastByte=TLastByte;

	/*free the allocated buffer*/
	NdisFreeBuffer( pNdisBuffer );
	/*recylcle the packet*/
	NdisReinitializePacket(pPacketb);
	/*Put the packet on the free queue*/
	NdisFreePacket(pPacketb);

	TRACE_LEAVE( "IndicateReceive" );
    return NDIS_STATUS_SUCCESS;


}

	TRACE_ENTER( "IndicateReceive" );

	pReserved = CONTAINING_RECORD( PacketListEntry, PACKET_RESERVED, ListElement );
	pPacket   = CONTAINING_RECORD( pReserved, NDIS_PACKET, ProtocolReserved );
	IF_PACKETDEBUG( PACKET_DEBUG_VERY_LOUD ) 
	{
		IF_TRACE_MSG( "   Reserved : %lx", pReserved );
		IF_TRACE_MSG( "    pPacket : %lx", pPacket );
		IF_TRACE_MSG2( "     Header : %lx  %lx", pvHeaderBuffer, uiHeaderBufferSize );
		IF_TRACE_MSG2( "  LookAhead : %lx  %lx", pvLookAheadBuffer, uiLookaheadBufferSize );
		IF_TRACE_MSG( " PacketSize : %lx", uiPacketSize );
	}
	PACKETASSERT( (pReserved != NULL) );
	PACKETASSERT( (pPacket != NULL) );

	pNdisBuffer = pPacket->Private.Head;
	/*virtual address of the buffer that will contain the packet*/
	pvActualVirtualAddress	= pNdisBuffer->VirtualAddress;
	uiActualLength			= pNdisBuffer->Length;
	
	CurrBuff=pNdisBuffer->VirtualAddress;
	(BYTE *)(pNdisBuffer->VirtualAddress) += sizeof(struct bpf_hdr);


	if ( uiHeaderBufferSize > 0 ) 
	{
		if ( uiHeaderBufferSize > pNdisBuffer->Length )
			uiHeaderBufferSize = pNdisBuffer->Length;
		/*copy the header*/
		NdisMoveMemory(pNdisBuffer->VirtualAddress, pvHeaderBuffer, uiHeaderBufferSize );		
		uiBytesTransferred = uiHeaderBufferSize;
		(BYTE *)(pNdisBuffer->VirtualAddress) += uiHeaderBufferSize;
		pNdisBuffer->Length -= uiHeaderBufferSize;
	}
	
//provare a togliere
	if ( uiPacketSize >= 1500 )
	{
		_asm { int 3 }
	}

	ulSizeToTransfer = uiPacketSize/*uiPacketSize - uiLookaheadBufferSize*/;
	if ( ulSizeToTransfer > pNdisBuffer->Length )
		ulSizeToTransfer = pNdisBuffer->Length;

	/*Copy the remaining part of the packet*/
	NdisTransferData(	&Status,					
						pOpen->AdapterHandle,		
						MacReceiveContext,			
						0,		
						ulSizeToTransfer,			
						pPacket,					
						&uiBytesTransferred );		

	uiBytesTransferred+=uiHeaderBufferSize;

	pNdisBuffer->VirtualAddress = pvActualVirtualAddress;
	pNdisBuffer->Length			= uiActualLength;
	if ( Status != NDIS_STATUS_PENDING ) 
	{

		/*store the capture time*/
		lCapTime=QuerySystemTime();

		/*fill the bpf header for this packet*/
		if( fres>uiBytesTransferred )fres=uiBytesTransferred;
			lCapTime+=pOpen->StartTime;
			header=(struct bpf_hdr*)CurrBuff;
			header->bh_tstamp.tv_usec=(long)((lCapTime%1250000)*1000000/1250000);
			header->bh_tstamp.tv_sec=(long)((lCapTime)/1250000);
			header->bh_caplen=(UINT)fres;
			header->bh_datalen=(UINT)uiPacketSize+uiHeaderBufferSize;
			header->bh_hdrlen=sizeof(struct bpf_hdr);
			/*call the completion routine*/
			PacketTransferDataComplete(	pOpen,				
										pPacket,			
										Status,				
										fres);

	}
	else
	{
		Status = NDIS_STATUS_SUCCESS;
		pNdisBuffer->VirtualAddress = pvActualVirtualAddress;
		pNdisBuffer->Length			= uiActualLength;

		/*store the capture time*/
		lCapTime=QuerySystemTime();

		/*call the filter to see if the packet is valid*/
			/*fill the bpf header for this packet*/
			if( fres>uiBytesTransferred )fres=uiBytesTransferred;
			lCapTime+=pOpen->StartTime;
			header=(struct bpf_hdr*)CurrBuff;
			header->bh_tstamp.tv_usec=(long)((lCapTime%1250000)*1000000/1250000);
			header->bh_tstamp.tv_sec=(long)((lCapTime)/1250000);
			header->bh_caplen=(UINT)fres;
			header->bh_datalen=(UINT)uiPacketSize+uiHeaderBufferSize;
			header->bh_hdrlen=sizeof(struct bpf_hdr);
			/*call the completion routine*/
			PacketTransferDataComplete(	pOpen,			
										pPacket,		
										Status,			
										0 );			
		
	}
	TRACE_LEAVE( "IndicateReceive" );
	return NDIS_STATUS_SUCCESS;
}

/*Ends the transfer started with the Packet_tap function*/
VOID NDIS_API
PacketTransferDataComplete (	IN NDIS_HANDLE   ProtocolBindingContext,
								IN PNDIS_PACKET  pPacket,
								IN NDIS_STATUS   Status,
								IN UINT          uiBytesTransferred
   )
{
	PPACKET_RESERVED	pReserved;
	OVERLAPPED*			pOverlap;
	PNDIS_BUFFER		pNdisBuffer;
	TRACE_ENTER( "TransferDataComplete" );
	pReserved = (PPACKET_RESERVED) pPacket->ProtocolReserved;
	pOverlap  = (OVERLAPPED *) pReserved->lpoOverlapped;
	PACKETASSERT( (pOpen != NULL) );
	PACKETASSERT( (pReserved != NULL) );
	PACKETASSERT( (pOverlap != NULL) );
	
	IF_PACKETDEBUG( PACKET_DEBUG_VERY_LOUD ) 
	{
		IF_TRACE_MSG( "     Status : %lx", Status );
		IF_TRACE_MSG( "BytesXfered : %lx", uiBytesTransferred );
		IF_TRACE_MSG( "Byte Offset : %lx", *(pReserved->lpcbBytesReturned) );
	}
	
	NdisUnchainBufferAtFront( pPacket, &pNdisBuffer );
	PACKETASSERT( (pNdisBuffer != NULL) );
	if ( pNdisBuffer )
		NdisFreeBuffer( pNdisBuffer );
	
	*(pReserved->lpcbBytesReturned) = uiBytesTransferred+sizeof(struct bpf_hdr);
	pOverlap->O_InternalHigh         = *(pReserved->lpcbBytesReturned);
	/*wakes the application process*/
	VWIN32_DIOCCompletionRoutine( pOverlap->O_Internal );
	
	PacketPageUnlock( pReserved->lpBuffer, pReserved->cbBuffer );
	PacketPageUnlock( pReserved->lpcbBytesReturned, sizeof(DWORD) );
	PacketPageUnlock( pReserved->lpoOverlapped, sizeof(OVERLAPPED) );
	
	NdisReinitializePacket( pPacket );
	
	NdisFreePacket( pPacket );
	TRACE_LEAVE( "TransferDataComplete" );
	return;
}

VOID NDIS_API
PacketReceiveComplete( IN NDIS_HANDLE  ProtocolBindingContext )
{
	IF_PACKETDEBUG( PACKET_DEBUG_VERY_LOUD ) 
	{
		TRACE_ENTER( "ReceiveComplete" );
		TRACE_LEAVE( "ReceiveComplete" );
	}
	return;
}
