// NOTE: This file has been generated automatically by “update-keysyms-case-mappings.py”.
//       Do not edit manually!

/*
 * Copyright © 2024 Pierre Le Marre
 * SPDX-License-Identifier: MIT
 */

/* Case mappings for Unicode 17.0
 *
 * There are two kinds of keysyms to consider:
 * • Legacy keysyms: their case mappings is located at `data/keysyms.yaml`.
 * • Unicode keysyms: their case mappings come from the ICU library.
 *
 * These mappings would create huge lookup tables if done naively. Fortunately,
 * we can observe that if we compute only the *difference* between a keysym and
 * its corresponding case mapping, there are a lot of repetitions that can be
 * efficiently compressed.
 *
 * The idea for the compression is, for each kind of keysyms:
 * 1. Compute the deltas between the keysyms and their case mappings.
 * 2. Split the delta array in chunks of a given size.
 * 3. Rearrange the order of the chunks in order to optimize consecutive
 *    chunks overlap.
 * 4. Create a data table with the reordered chunks and an index table that
 *    maps the original chunk index to its offset in the data table.
 *
 * Trivial example (chunk size: 4, from step 2):
 *
 * [1, 2, 3, 4, 2, 3, 4, 5, 0, 1, 2, 3]          # source data
 * -> [[1, 2, 3, 4], [2, 3, 4, 5], [0, 1, 2, 3]] # make chunks
 * -> [[0, 1, 2, 3], [1, 2, 3, 4], [2, 3, 4, 5]] # rearrange to have best overlaps
 * -> {data: [0, 1, 2, 3, 4, 5], offsets: [1, 2, 0]} # overlap chunks & compute
 *                                                   # their offsets
 *
 * Then we can retrieve the data from the original array at index i with the
 * following formula:
 *
 *     mask = (1 << chunk_size) - 1;
 *     original[i] = data[offsets[i >> chunk_size] + (i & mask)];
 *
 * Since the index array is itself quite repetitive with the real data, we apply
 * the compression a second time to the offsets table.
 *
 * The complete algorithm optimizes the chunk sizes for both arrays in order to
 * get the lowest total data size.
 *
 * There are 6 resulting arrays, 3 for each kind of keysyms:
 * 1. The data array. Each item is either:
 *    • 0, if the keysym is not cased.
 *    • A delta to lower case.
 *    • A delta to upper case.
 *    • For some special cases, there are both a lower *and* an upper case
 *      mapping. The delta is identical in both cases.
 * 2. The 1st offsets array, that provides offsets into the data array.
 * 3. The 2nd offsets array, that provides offsets into the 1st index array.
 *
 * Finally, given the chunks sizes `cs_data` and `cs_offsets`:
 * 1. We compute the corresponding masks:
 *    • `mask_data = (1 << cs_data) - 1` and
 *    • `mask_offsets = (1 << cs_offsets) - 1`.
 * 2. We can retrieve the case mapping of a keysyms `ks` with the following
 *    formula:
 *
 *     data[
 *         offsets1[
 *             offsets2[ks >> (cs_data + cs_offsets)] +
 *             ((ks >> cs_data) & mask_offsets)
 *         ] +
 *         (ks & mask_data)
 *     ];
 */

#include "config.h"

#include <stdint.h>
#include <stdbool.h>
#include <string.h>
#include "xkbcommon/xkbcommon.h"
#include "utils.h"
#include "keysym.h"

struct CaseMappings{
    bool lower:1;
    bool upper:1;
    int32_t offset:30;
};

static const struct CaseMappings legacy_keysym_data[47] = {
    {0, 1, 0x0000000}, {1, 0, 0x0000010}, {0, 0, 0x0000000}, {0, 0, 0x0000000},
    {1, 0, 0x0000020}, {0, 1,-0x1001dbf}, {1, 0, 0x0000020}, {1, 0, 0x0000020},
    {0, 0, 0x0000000}, {0, 1,-0x0000717}, {0, 1, 0x0000020}, {0, 0, 0x0000000},
    {0, 1, 0x0000020}, {0, 1, 0x0000020}, {0, 1,-0x00012bf}, {0, 1, 0x0000000},
    {0, 0, 0x0000000}, {1, 0, 0x0000010}, {1, 0, 0x0000010}, {0, 1,-0x0fff89b},
    {0, 0, 0x0000000}, {0, 1, 0x0000010}, {0, 0, 0x0000000}, {0, 1,-0x0000010},
    {0, 1,-0x0000010}, {0, 1, 0x0000000}, {0, 1, 0x0000010}, {0, 1, 0x0000010},
    {1, 0, 0x0000002}, {0, 0, 0x0000000}, {1, 0,-0x0000010}, {1, 0,-0x0000010},
    {1, 0,-0x00012bf}, {1, 0,-0x0000020}, {1, 0,-0x0000020}, {0, 1,-0x0000020},
    {0, 1,-0x0000020}, {0, 0, 0x0000000}, {1, 0,-0x0000240}, {0, 0, 0x0000000},
    {0, 1, 0x0000002}, {0, 0, 0x0000000}, {0, 1, 0x0000270}, {0, 1, 0x0000020},
    {0, 1, 0x0000021}, {1, 0, 0x0000001}, {0, 1, 0x0000001}
};

static const uint8_t legacy_keysym_offsets1[540] = {
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0003, 0x0006, 0x0006, 0x0006, 0x0006, 0x0006, 0x0006, 0x0006,
    0x0006, 0x0006, 0x0006, 0x0006, 0x0006, 0x0007, 0x0002, 0x0002, 0x000b, 0x000c,
    0x000c, 0x000c, 0x000c, 0x000c, 0x000c, 0x000c, 0x000c, 0x000c, 0x000c, 0x000c,
    0x000c, 0x000a, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0010, 0x0011,
    0x0011, 0x0010, 0x0011, 0x0010, 0x0002, 0x0002, 0x0014, 0x001a, 0x001a, 0x0019,
    0x001a, 0x0019, 0x0002, 0x0002, 0x0003, 0x0006, 0x0006, 0x0006, 0x0006, 0x0006,
    0x0006, 0x0006, 0x0006, 0x0007, 0x0006, 0x0006, 0x0006, 0x0002, 0x0002, 0x0002,
    0x000b, 0x000c, 0x000c, 0x000c, 0x000c, 0x000c, 0x000c, 0x000c, 0x000c, 0x002b,
    0x000c, 0x000c, 0x000c, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0013, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x002d, 0x0020,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0000, 0x0010, 0x0001,
    0x0002, 0x0011, 0x0001, 0x0002, 0x0002, 0x0014, 0x0014, 0x0015, 0x0002, 0x001a,
    0x001b, 0x0027, 0x0007, 0x0002, 0x0002, 0x0003, 0x0002, 0x0002, 0x0007, 0x0003,
    0x0003, 0x0006, 0x0002, 0x0002, 0x0003, 0x0002, 0x0003, 0x0007, 0x000a, 0x0002,
    0x0002, 0x000b, 0x0002, 0x0002, 0x000a, 0x000b, 0x000b, 0x000c, 0x0002, 0x0002,
    0x000b, 0x0002, 0x000b, 0x000a, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x000f, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0008, 0x0002, 0x0002, 0x000f, 0x0002, 0x0002, 0x0006, 0x0006, 0x0006, 0x0006,
    0x0006, 0x0006, 0x0006, 0x0006, 0x0006, 0x0006, 0x0006, 0x0007, 0x0006, 0x0006,
    0x0006, 0x0004, 0x000c, 0x000c, 0x000c, 0x000c, 0x000c, 0x000c, 0x000c, 0x000c,
    0x000c, 0x000c, 0x000c, 0x000a, 0x000c, 0x000c, 0x000c, 0x000d, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0010, 0x0002, 0x0002, 0x0001, 0x0025, 0x0010,
    0x0001, 0x0002, 0x0014, 0x0002, 0x0002, 0x0015, 0x0029, 0x0014, 0x0015, 0x0002,
    0x0002, 0x0002, 0x0003, 0x0007, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0003, 0x0002, 0x0007, 0x0002, 0x0003, 0x0007, 0x0002, 0x0002, 0x000b, 0x000a,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x000b, 0x0002, 0x000a, 0x0002,
    0x000b, 0x000a, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0010, 0x0010,
    0x0010, 0x0001, 0x0010, 0x0011, 0x0001, 0x0011, 0x0014, 0x0014, 0x0014, 0x0015,
    0x0014, 0x001a, 0x0015, 0x001a, 0x0007, 0x0003, 0x0003, 0x0007, 0x0007, 0x0007,
    0x0007, 0x0003, 0x0006, 0x0007, 0x0003, 0x0002, 0x0006, 0x0003, 0x0002, 0x0007,
    0x000a, 0x000b, 0x000b, 0x000a, 0x000a, 0x000a, 0x000a, 0x000b, 0x000c, 0x000a,
    0x000b, 0x0002, 0x000c, 0x000b, 0x0002, 0x000a, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0016, 0x0017, 0x0017, 0x0017, 0x0017, 0x0017, 0x0017, 0x0017,
    0x001d, 0x001e, 0x001e, 0x001e, 0x001e, 0x001e, 0x001e, 0x001e, 0x0023, 0x0023,
    0x0023, 0x0023, 0x0023, 0x0023, 0x0023, 0x0023, 0x0023, 0x0023, 0x0023, 0x0023,
    0x0023, 0x0023, 0x0023, 0x0023, 0x0021, 0x0021, 0x0021, 0x0021, 0x0021, 0x0021,
    0x0021, 0x0021, 0x0021, 0x0021, 0x0021, 0x0021, 0x0021, 0x0021, 0x0021, 0x0021
};

static const uint16_t legacy_keysym_offsets2[40] = {
    0x0000, 0x011c, 0x007b, 0x019c, 0x007b, 0x015c, 0x007b, 0x00dc, 0x007b, 0x007b,
    0x007b, 0x007b, 0x007b, 0x01dc, 0x007b, 0x003e, 0x007b, 0x0080, 0x007b, 0x007b,
    0x007b, 0x007b, 0x007b, 0x007b, 0x007b, 0x007b, 0x007b, 0x007b, 0x007b, 0x007b,
    0x007b, 0x007b, 0x007b, 0x007b, 0x007b, 0x007b, 0x007b, 0x007b, 0x007b, 0x00bc
};

static inline const struct CaseMappings *
get_legacy_keysym_entry(xkb_keysym_t ks)
{
    return &legacy_keysym_data[legacy_keysym_offsets1[legacy_keysym_offsets2[ks >> 7] + ((ks >> 1) & 0x3f)] + (ks & 0x01)];
}

static const struct CaseMappings unicode_data[1019] = {
    {0, 1,-0x0080}, {0, 1,-0x0080}, {0, 1,-0x0070}, {0, 1,-0x0070},
    {0, 1,-0x007e}, {0, 1,-0x007e}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {0, 1, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1,-0xa515}, {0, 1,-0xa512},
    {0, 1, 0x0000}, {1, 0, 0x0002}, {1, 1, 0x0001}, {0, 1, 0x0002},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0,-0x0061}, {1, 0,-0x0038},
    {1, 0, 0x0020}, {1, 0, 0x0020}, {0, 0, 0x0000}, {1, 0, 0x0020},
    {1, 0, 0x0020}, {1, 0, 0x0020}, {1, 0, 0x0020}, {1, 0, 0x0020},
    {1, 0, 0x0020}, {1, 0, 0x0020}, {1, 0, 0x0020}, {0, 1,-0x1dbf},
    {1, 0,-0x0008}, {1, 0,-0x0008}, {1, 0,-0x0064}, {1, 0,-0x0064},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {1, 0,-0x0008}, {0, 0, 0x0000}, {1, 0,-0x0008}, {0, 0, 0x0000},
    {1, 0,-0x0008}, {0, 0, 0x0000}, {1, 0,-0x0008}, {1, 0,-0x0008},
    {1, 0,-0x004a}, {1, 0,-0x004a}, {1, 0,-0x0009}, {0, 0, 0x0000},
    {0, 1, 0x1c25}, {0, 0, 0x0000}, {0, 1, 0x0020}, {0, 1, 0x0020},
    {0, 1, 0x0020}, {0, 1, 0x0020}, {0, 1, 0x0020}, {0, 1, 0x0020},
    {0, 1, 0x0020}, {0, 1, 0x0020}, {0, 1,-0x0079}, {0, 1,-0x2a1f},
    {0, 1,-0x2a1c}, {0, 1,-0x2a1e}, {0, 1, 0x00d2}, {0, 1, 0x00ce},
    {0, 1, 0x0000}, {0, 1, 0x00cd}, {0, 1, 0x00cd}, {0, 1,-0xa54b},
    {0, 1, 0x0000}, {0, 1, 0x00cf}, {0, 1,-0xa567}, {0, 1,-0xa528},
    {0, 1,-0xa544}, {0, 1, 0x0000}, {1, 0, 0x0020}, {1, 0, 0x0020},
    {1, 0, 0x0020}, {1, 0, 0x0020}, {1, 0, 0x0020}, {1, 0, 0x0020},
    {1, 0, 0x0020}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 1,-0x02e7}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 1, 0x0000}, {0, 1,-0x0009}, {0, 1, 0x0000},
    {0, 0, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000}, {0, 0, 0x0000},
    {1, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000}, {0, 1, 0x0000},
    {0, 0, 0x0000}, {1, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {0, 1,-0x0008}, {0, 1,-0x0008}, {0, 1, 0x0000}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {0, 1,-0x0007}, {0, 1, 0x0000}, {0, 1, 0x0000},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0, 0x0001}, {0, 1, 0x0001},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0, 0x0001}, {0, 1, 0x0001},
    {0, 1, 0x000f}, {1, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000},
    {1, 0, 0x0000}, {1, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x0000},
    {0, 0, 0x0000}, {1, 0,-0x20bf}, {1, 0,-0x2046}, {1, 0, 0x0000},
    {1, 0, 0x0000}, {0, 0, 0x0000}, {0, 1, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {1, 0, 0x0000},
    {1, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000},
    {0, 0, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x001a},
    {1, 0, 0x001a}, {1, 0, 0x001a}, {1, 0, 0x001a}, {1, 0, 0x001a},
    {1, 0, 0x001a}, {1, 0, 0x001a}, {1, 0, 0x001a}, {0, 1,-0x89c2},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {1, 0, 0x0026}, {0, 0, 0x0000}, {1, 0,-0x0080}, {1, 0,-0x0080},
    {1, 0,-0x007e}, {1, 0,-0x007e}, {1, 0,-0x0009}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x0030}, {1, 0, 0x0030},
    {1, 0, 0x0030}, {1, 0, 0x0030}, {1, 0, 0x0030}, {1, 0, 0x0030},
    {1, 0, 0x0030}, {1, 0, 0x0030}, {0, 0, 0x0000}, {0, 1, 0x003e},
    {0, 1, 0x0039}, {1, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000},
    {0, 1, 0x002f}, {0, 1, 0x0036}, {0, 1, 0x0008}, {0, 1, 0x0008},
    {0, 1, 0x0008}, {0, 1, 0x0008}, {0, 1, 0x0008}, {0, 1, 0x0008},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {1, 0, 0x0001}, {0, 1, 0x0001}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x0002},
    {1, 1, 0x0001}, {0, 1, 0x0002}, {1, 0, 0x0002}, {1, 1, 0x0001},
    {0, 1, 0x0002}, {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0, 0x0001},
    {0, 1, 0x0001}, {1, 0,-0x0030}, {1, 0,-0xa543}, {1, 0,-0x8a38},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {0, 1,-0x00a3}, {0, 1,-0xa641},
    {1, 0, 0x00d3}, {1, 0, 0x00d5}, {0, 1,-0x0082}, {1, 0, 0x00d6},
    {1, 0,-0x00c7}, {0, 1, 0x00e8}, {1, 0, 0x0001}, {0, 1, 0x0001},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0, 0x0001}, {0, 1, 0x0001},
    {1, 0, 0x00da}, {1, 0, 0x0001}, {1, 0, 0x000f}, {1, 0, 0x0001},
    {0, 1, 0x0001}, {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0, 0x0001},
    {0, 1, 0x0001}, {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0, 0x0001},
    {0, 1, 0x0001}, {1, 0,-0x2a1c}, {1, 0,-0x29fd}, {1, 0,-0x2a1f},
    {0, 1, 0x0001}, {0, 1, 0x0000}, {1, 0, 0x0001}, {0, 1, 0x0001},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0, 0x0001}, {0, 1, 0x0001},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000},
    {1, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000},
    {1, 0, 0x001c}, {1, 0, 0x0000}, {0, 1, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 1, 0x0001}, {1, 0,-0x0007},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {0, 1, 0x0000}, {1, 0,-0x0082},
    {1, 0,-0x0082}, {1, 0,-0x0082}, {0, 1, 0x0000}, {1, 0, 0x0001},
    {0, 1, 0x0001}, {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0, 0x0001},
    {0, 1, 0x0001}, {0, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x0001},
    {0, 1, 0x0001}, {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0, 0x0001},
    {0, 1, 0x0001}, {0, 1, 0x004f}, {1, 0, 0x0001}, {0, 1, 0x0001},
    {1, 0,-0xa544}, {1, 0,-0xa54f}, {1, 0,-0xa54b}, {1, 0,-0xa541},
    {1, 0,-0xa544}, {0, 1, 0x0000}, {0, 1, 0x00ca}, {0, 1, 0x0000},
    {0, 1, 0x00cb}, {0, 1,-0xa54f}, {0, 1, 0x0000}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {0, 1, 0x03a0}, {0, 1, 0x0000}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 0, 0x0000},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {1, 0,-0x2a3f}, {1, 0,-0x2a3f},
    {1, 0, 0x1c60}, {1, 0, 0x1c60}, {1, 0, 0x1c60}, {1, 0, 0x1c60},
    {1, 0, 0x1c60}, {1, 0, 0x1c60}, {1, 0, 0x1c60}, {1, 0, 0x1c60},
    {0, 0, 0x0000}, {1, 0, 0x1c60}, {0, 1,-0x0008}, {0, 1,-0x0008},
    {0, 1,-0x0008}, {0, 1,-0x0008}, {0, 1,-0x0008}, {0, 1,-0x0008},
    {0, 1,-0x0008}, {0, 1,-0x0008}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {1, 0,-0x0008}, {1, 0,-0x0008}, {1, 0,-0x0008}, {1, 0,-0x0008},
    {1, 0,-0x0008}, {1, 0,-0x0008}, {1, 0,-0x0008}, {1, 0,-0x0008},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 1, 0x1c60}, {0, 1, 0x1c60},
    {0, 1, 0x1c60}, {0, 1, 0x1c60}, {0, 1, 0x1c60}, {0, 1, 0x1c60},
    {0, 1, 0x1c60}, {0, 1, 0x1c60}, {0, 0, 0x0000}, {0, 1, 0x1c60},
    {1, 0, 0x0022}, {1, 0, 0x0022}, {1, 0, 0x0022}, {1, 0, 0x0022},
    {1, 0, 0x0022}, {1, 0, 0x0022}, {1, 0, 0x0022}, {1, 0, 0x0022},
    {0, 1, 0x0022}, {0, 1, 0x0022}, {0, 1, 0x0022}, {0, 1, 0x0022},
    {0, 1, 0x0022}, {0, 1, 0x0022}, {0, 1, 0x0022}, {0, 1, 0x0022},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0, 0x0001}, {0, 1, 0x0001},
    {0, 0, 0x0000}, {0, 1,-0x0008}, {0, 1,-0x0008}, {0, 1, 0x0000},
    {0, 1,-0x0009}, {0, 1, 0x0000}, {0, 0, 0x0000}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {0, 1, 0x00db}, {0, 1, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000},
    {0, 0, 0x0000}, {0, 1, 0x0000}, {0, 0, 0x0000}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {1, 0, 0x0001},
    {0, 1, 0x0001}, {0, 0, 0x0000}, {0, 1, 0x001a}, {0, 1, 0x001a},
    {0, 1, 0x001a}, {0, 1, 0x001a}, {0, 1, 0x001a}, {0, 1, 0x001a},
    {0, 1, 0x001a}, {0, 1, 0x001a}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {0, 1,-0x0054}, {0, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x0040},
    {1, 0, 0x0040}, {1, 0, 0x0040}, {1, 0, 0x0040}, {1, 0, 0x0040},
    {1, 0, 0x0040}, {1, 0, 0x0040}, {1, 0, 0x0040}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {1, 0, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {1, 0, 0x0000},
    {0, 1, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 1, 0x0040}, {0, 1, 0x0040}, {0, 1, 0x0040},
    {0, 1, 0x0040}, {0, 1, 0x0040}, {0, 1, 0x0040}, {0, 1, 0x0040},
    {0, 1, 0x0040}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 1, 0x1c60}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {1, 0, 0x001b}, {1, 0, 0x001b}, {1, 0, 0x001b},
    {1, 0, 0x001b}, {1, 0, 0x001b}, {1, 0, 0x001b}, {1, 0, 0x001b},
    {1, 0, 0x001b}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 1, 0x001b},
    {0, 1, 0x001b}, {0, 1, 0x001b}, {0, 1, 0x001b}, {0, 1, 0x001b},
    {0, 1, 0x001b}, {0, 1, 0x001b}, {0, 1, 0x001b}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x0001},
    {0, 1, 0x0001}, {1, 0,-0xa528}, {0, 1, 0x0000}, {0, 0, 0x0000},
    {0, 1, 0x0020}, {0, 1, 0x0020}, {0, 1, 0x001f}, {0, 1, 0x0020},
    {0, 1, 0x0020}, {0, 1, 0x0020}, {0, 1, 0x0020}, {0, 1, 0x0020},
    {0, 1, 0x0040}, {0, 1, 0x003f}, {0, 1, 0x003f}, {1, 0, 0x0008},
    {1, 0, 0x0008}, {1, 0, 0x0008}, {1, 0, 0x0008}, {1, 0, 0x0008},
    {1, 0, 0x0008}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 1, 0x0000},
    {1, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000},
    {1, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000},
    {1, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000},
    {1, 0, 0x0000}, {1, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x0000},
    {1, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000},
    {1, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000}, {1, 0, 0x0000},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x003b},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {1, 0,-0x1dbf}, {0, 1, 0x0000},
    {0, 1, 0x0030}, {0, 1, 0x0030}, {0, 1, 0x0030}, {0, 1, 0x0030},
    {0, 1, 0x0030}, {0, 1, 0x0030}, {0, 1, 0x0030}, {0, 1, 0x0030},
    {0, 1, 0x0000}, {0, 1,-0x0008}, {0, 1, 0x0000}, {0, 1,-0x0008},
    {0, 1, 0x0000}, {0, 1,-0x0008}, {0, 1, 0x0000}, {0, 1,-0x0008},
    {0, 1,-0x0008}, {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1,-0x8a04},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1,-0x0ee6},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0020}, {0, 1, 0x0020},
    {0, 1, 0x0020}, {0, 1, 0x0020}, {0, 1, 0x0020}, {0, 1, 0x0020},
    {0, 1, 0x0020}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x1c60}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 1, 0x0000}, {0, 1,-0x0082}, {0, 1,-0x0082},
    {0, 1,-0x0082}, {0, 0, 0x0000}, {1, 0, 0x0074}, {0, 1, 0x0001},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0,-0xa567}, {1, 0, 0x0001},
    {0, 1, 0x0001}, {1, 0, 0x0001}, {0, 1, 0x0001}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 1, 0x001c}, {0, 0, 0x0000}, {1, 0,-0xa512},
    {1, 0,-0xa52a}, {1, 0,-0xa515}, {1, 0, 0x03a0}, {1, 0, 0x0001},
    {0, 1, 0x0001}, {1, 0, 0x0001}, {0, 1, 0x0001}, {0, 1, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x0028},
    {1, 0, 0x0028}, {1, 0, 0x0028}, {1, 0, 0x0028}, {1, 0, 0x0028},
    {1, 0, 0x0028}, {1, 0, 0x0028}, {1, 0, 0x0028}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 1, 0x0028},
    {0, 1, 0x0028}, {0, 1, 0x0028}, {0, 1, 0x0028}, {0, 1, 0x0028},
    {0, 1, 0x0028}, {0, 1, 0x0028}, {0, 1, 0x0028}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x0027},
    {1, 0, 0x0027}, {1, 0, 0x0027}, {0, 0, 0x0000}, {1, 0, 0x0027},
    {1, 0, 0x0027}, {1, 0, 0x0027}, {1, 0, 0x0027}, {1, 0, 0x0027},
    {1, 0, 0x0027}, {1, 0, 0x0027}, {1, 0, 0x0027}, {0, 0, 0x0000},
    {1, 0, 0x0027}, {1, 0, 0x0027}, {0, 0, 0x0000}, {0, 1, 0x0027},
    {0, 1, 0x0027}, {0, 0, 0x0000}, {0, 1, 0x0027}, {0, 1, 0x0027},
    {0, 1, 0x0027}, {0, 1, 0x0027}, {0, 1, 0x0027}, {0, 1, 0x0027},
    {0, 1, 0x0027}, {0, 1, 0x0027}, {0, 0, 0x0000}, {0, 1, 0x0027},
    {0, 1, 0x0027}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000},
    {1, 0,-0x2a1e}, {0, 1, 0x0000}, {1, 0, 0x0001}, {0, 1, 0x0001},
    {0, 1, 0x0000}, {1, 0, 0x0001}, {0, 1, 0x0001}, {0, 1, 0x0000},
    {0, 0, 0x0000}, {1, 0, 0x0001}, {0, 1, 0x0001}, {0, 1, 0x0000},
    {0, 1,-0x0038}, {0, 1, 0x00da}, {0, 1, 0x0045}, {0, 1, 0x00d9},
    {0, 1, 0x00d9}, {0, 1, 0x0047}, {0, 1, 0x0000}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1,-0x29e7},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1,-0x8a38}, {0, 1, 0x0000},
    {0, 1,-0x29fd}, {0, 1, 0x00d5}, {0, 1, 0x0000}, {0, 1, 0x0000},
    {0, 1, 0x00d6}, {0, 1, 0x0000}, {0, 1, 0x0000}, {1, 0, 0x2a2b},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0,-0x00a3}, {1, 0, 0x2a28},
    {0, 1,-0x2a3f}, {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0,-0x00c3},
    {1, 0, 0x0045}, {1, 0, 0x0047}, {1, 0, 0x0001}, {0, 1, 0x0001},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0,-0xa641}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x0000}, {0, 0, 0x0000},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {1, 0, 0x0000},
    {0, 0, 0x0000}, {1, 0,-0x1d5d}, {0, 0, 0x0000}, {0, 1,-0x0bc0},
    {0, 1,-0x0bc0}, {0, 1,-0x0bc0}, {0, 1,-0x0bc0}, {0, 1,-0x0bc0},
    {0, 1,-0x0bc0}, {0, 1,-0x0bc0}, {0, 1,-0x0bc0}, {0, 0, 0x0000},
    {0, 1, 0x0000}, {0, 1,-0x0bc0}, {0, 1,-0x0bc0}, {0, 1,-0x0bc0},
    {1, 0,-0x0bc0}, {1, 0,-0x0bc0}, {1, 0,-0x0bc0}, {1, 0,-0x0bc0},
    {1, 0,-0x0bc0}, {1, 0,-0x0bc0}, {1, 0,-0x0bc0}, {1, 0,-0x0bc0},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {1, 0,-0x0bc0}, {1, 0,-0x0bc0},
    {1, 0,-0x0bc0}, {0, 1, 0x0001}, {1, 0, 0x00da}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0, 0x00da},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0, 0x0001}, {0, 1, 0x0001},
    {0, 1,-0x0030}, {0, 1, 0x0000}, {1, 0, 0x0001}, {0, 1, 0x0001},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0,-0x8a04}, {1, 0, 0x0001},
    {0, 1, 0x0001}, {1, 0,-0x29f7}, {1, 0,-0x0ee6}, {1, 0,-0x29e7},
    {0, 1, 0x2a2b}, {0, 1, 0x2a28}, {1, 0, 0x0001}, {1, 0,-0x0079},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0, 0x0001}, {0, 1, 0x0001},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {0, 1, 0x012c}, {1, 0,-0x0056},
    {1, 0,-0x0056}, {1, 0,-0x0056}, {1, 0,-0x0056}, {1, 0,-0x0009},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {1, 0,-0x0008},
    {1, 0,-0x0008}, {1, 0,-0x0070}, {1, 0,-0x0070}, {1, 0,-0x0007},
    {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 0, 0x0000}, {0, 1,-0x00c3},
    {1, 0, 0x00d2}, {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0, 0x0001},
    {0, 1, 0x0001}, {1, 0, 0x00ce}, {1, 0, 0x0001}, {0, 1,-0x004a},
    {0, 1,-0x004a}, {0, 1,-0x0056}, {0, 1,-0x0056}, {0, 1,-0x0056},
    {0, 1,-0x0056}, {0, 1,-0x0064}, {0, 1,-0x0064}, {0, 1, 0x0001},
    {1, 0, 0x00cd}, {1, 0, 0x00cd}, {1, 0, 0x0001}, {0, 1, 0x0001},
    {0, 1, 0x0000}, {1, 0, 0x004f}, {1, 0, 0x00ca}, {0, 1, 0x0001},
    {1, 0, 0x00d9}, {1, 0, 0x00d9}, {1, 0, 0x0001}, {0, 1, 0x0001},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0, 0x00db}, {0, 1, 0x0010},
    {0, 1, 0x0010}, {0, 1, 0x0010}, {0, 1, 0x0010}, {0, 1, 0x0010},
    {0, 1, 0x0010}, {0, 1, 0x0010}, {0, 1, 0x0010}, {0, 1, 0x0050},
    {0, 1, 0x0050}, {0, 1, 0x0050}, {0, 1, 0x0050}, {0, 1, 0x0050},
    {0, 1, 0x0050}, {0, 1, 0x0050}, {0, 1, 0x0050}, {0, 1, 0x0056},
    {0, 1, 0x0050}, {0, 1,-0x0007}, {0, 1, 0x0074}, {1, 0,-0x003c},
    {0, 1, 0x0060}, {0, 0, 0x0000}, {1, 0, 0x0001}, {0, 1, 0x00d1},
    {0, 1, 0x00d3}, {0, 1,-0xa544}, {0, 1,-0x29f7}, {0, 1,-0xa541},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1, 0x00d3}, {0, 1, 0x00da},
    {0, 1, 0x0000}, {0, 1,-0xa543}, {0, 1, 0x00da}, {0, 1, 0x0000},
    {0, 1, 0x0000}, {0, 1, 0x0000}, {0, 1,-0xa52a}, {0, 1, 0x186e},
    {0, 1, 0x186d}, {0, 1, 0x1864}, {0, 1, 0x1862}, {0, 1, 0x1862},
    {0, 1, 0x1863}, {0, 1, 0x185c}, {0, 1, 0x1825}, {0, 1, 0x97d0},
    {0, 1, 0x97d0}, {0, 1, 0x97d0}, {0, 1, 0x97d0}, {0, 1, 0x97d0},
    {0, 1, 0x97d0}, {0, 1, 0x97d0}, {0, 1, 0x97d0}, {1, 0, 0x0010},
    {1, 0, 0x0010}, {1, 0, 0x0010}, {1, 0, 0x0010}, {1, 0, 0x0010},
    {1, 0, 0x0010}, {1, 0, 0x0010}, {1, 0, 0x0010}, {1, 0, 0x0020},
    {1, 0, 0x0020}, {1, 0, 0x0020}, {1, 0, 0x0020}, {0, 1, 0x0026},
    {0, 1, 0x0025}, {0, 1, 0x0025}, {0, 1, 0x0025}, {1, 0, 0x0025},
    {1, 0, 0x0025}, {1, 0, 0x0025}, {0, 0, 0x0000}, {1, 0, 0x0040},
    {0, 0, 0x0000}, {1, 0, 0x003f}, {1, 0, 0x003f}, {1, 0, 0x0050},
    {1, 0, 0x0050}, {1, 0, 0x0050}, {1, 0, 0x0050}, {1, 0, 0x0050},
    {1, 0, 0x0050}, {1, 0, 0x0050}, {1, 0, 0x0050}, {1, 0, 0x00cb},
    {1, 0, 0x0001}, {0, 1, 0x0001}, {1, 0, 0x00cd}, {1, 0, 0x00cf},
    {0, 1,-0x0061}, {1, 0, 0x00d3}, {1, 0, 0x00d1}, {1, 0, 0x97d0},
    {1, 0, 0x97d0}, {1, 0, 0x97d0}, {1, 0, 0x97d0}, {1, 0, 0x97d0},
    {1, 0, 0x97d0}, {1, 0, 0x97d0}, {1, 0, 0x97d0}
};

static const uint16_t unicode_offsets1[908] = {
    0x00a6, 0x0009, 0x000e, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x00ad, 0x00b3, 0x00b3, 0x00b3, 0x01ca,
    0x01ca, 0x01ca, 0x01d0, 0x0009, 0x0009, 0x018e, 0x018e, 0x018e, 0x018e, 0x0190,
    0x01f9, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0170, 0x0170, 0x0170,
    0x0170, 0x0172, 0x0281, 0x0327, 0x0327, 0x0327, 0x0327, 0x0327, 0x032c, 0x00a5,
    0x0165, 0x00a5, 0x00a6, 0x000c, 0x00aa, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0023, 0x0023, 0x0023, 0x0023, 0x0042, 0x0042, 0x0042, 0x0042, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0201, 0x0201, 0x0201, 0x0208, 0x020b, 0x020b, 0x020f,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0149, 0x0084, 0x0082, 0x0084, 0x0084,
    0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x00a5, 0x034d, 0x0084, 0x0214, 0x0348,
    0x0084, 0x0084, 0x0152, 0x029f, 0x0084, 0x00fd, 0x028f, 0x0084, 0x0316, 0x0009,
    0x0009, 0x01c2, 0x00aa, 0x0239, 0x0239, 0x0239, 0x0231, 0x00a5, 0x00a5, 0x0235,
    0x0239, 0x0239, 0x024e, 0x0162, 0x00a5, 0x00a5, 0x0239, 0x0239, 0x0239, 0x0231,
    0x00a5, 0x00a5, 0x012c, 0x00e4, 0x0241, 0x024e, 0x01be, 0x0163, 0x00a5, 0x0239,
    0x0239, 0x0239, 0x0231, 0x00a5, 0x00a5, 0x0162, 0x0235, 0x0239, 0x0239, 0x009f,
    0x00a5, 0x00a5, 0x00a6, 0x0233, 0x0239, 0x0239, 0x023a, 0x00a5, 0x00a5, 0x00a5,
    0x0166, 0x0239, 0x0239, 0x0239, 0x00a3, 0x00a5, 0x00a5, 0x0164, 0x01e9, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x00a5, 0x00a5, 0x00a5, 0x00a5, 0x015e,
    0x0164, 0x00a5, 0x00ab, 0x03c3, 0x03c3, 0x03c3, 0x03c3, 0x03c3, 0x03c3, 0x03c3,
    0x03c3, 0x03c3, 0x03c3, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x00a5, 0x00a5, 0x00a5, 0x00a5, 0x00a5, 0x00a5, 0x00a5, 0x00a7, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x03bb, 0x00bb, 0x0334,
    0x0334, 0x0334, 0x0334, 0x0334, 0x0339, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x01d3, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0145,
    0x0287, 0x00be, 0x03db, 0x0059, 0x0023, 0x0020, 0x03d3, 0x0279, 0x0042, 0x021c,
    0x0220, 0x00d7, 0x0084, 0x0084, 0x0084, 0x03a3, 0x013a, 0x0198, 0x0198, 0x0198,
    0x0198, 0x019e, 0x01a0, 0x01a0, 0x01a0, 0x01a4, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x01ba,
    0x00a5, 0x00a5, 0x00a5, 0x00a5, 0x00a5, 0x0166, 0x00aa, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0022, 0x0023, 0x0023, 0x005e, 0x0041,
    0x0042, 0x0042, 0x027e, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0006, 0x0061,
    0x0006, 0x0023, 0x0023, 0x005a, 0x0024, 0x0042, 0x0042, 0x027a, 0x0043, 0x031c,
    0x022d, 0x0074, 0x0246, 0x031f, 0x0093, 0x0132, 0x0099, 0x01e3, 0x0297, 0x0009,
    0x0009, 0x03cb, 0x03cb, 0x0393, 0x0393, 0x00ee, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x01db, 0x01db, 0x01db, 0x01db, 0x01db, 0x01db, 0x01e0, 0x0009,
    0x01f1, 0x01f1, 0x01f1, 0x01f1, 0x01f1, 0x01f1, 0x01f6, 0x0009, 0x03e3, 0x03e3,
    0x0023, 0x0023, 0x0023, 0x0023, 0x0042, 0x0042, 0x0042, 0x0042, 0x039b, 0x039b,
    0x0084, 0x0084, 0x0084, 0x0084, 0x00bc, 0x0149, 0x0084, 0x0084, 0x0084, 0x0084,
    0x0084, 0x0084, 0x0116, 0x0085, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084,
    0x00cd, 0x00ce, 0x00ce, 0x00ce, 0x00cf, 0x0009, 0x025b, 0x025c, 0x025c, 0x025c,
    0x025d, 0x0008, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0007, 0x000a, 0x0009, 0x0009,
    0x00a5, 0x00a8, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0239, 0x0239, 0x0239, 0x00e9, 0x0239, 0x0239,
    0x0239, 0x00e9, 0x0239, 0x0239, 0x0239, 0x0071, 0x00a5, 0x00a5, 0x00a5, 0x00a5,
    0x00a5, 0x00a5, 0x00a5, 0x00a5, 0x00a5, 0x00a5, 0x00a5, 0x00a5, 0x00a5, 0x00a5,
    0x00a5, 0x0272, 0x00a5, 0x02fc, 0x00a5, 0x00a5, 0x00a5, 0x00a5, 0x00a5, 0x00a5,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0084, 0x0084,
    0x0084, 0x0084, 0x0084, 0x0143, 0x0009, 0x0009, 0x0084, 0x0084, 0x0084, 0x0293,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0023, 0x0023, 0x005b, 0x0009, 0x0042, 0x0042, 0x027b, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x02c7, 0x02c3, 0x02c7, 0x02c3, 0x02cc, 0x02d6,
    0x02d3, 0x02d6, 0x02d3, 0x02dc, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0022, 0x0023, 0x0023, 0x005e, 0x0041, 0x0042, 0x0042, 0x027e,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009,
    0x03f3, 0x03f3, 0x03f3, 0x03f3, 0x03f3, 0x03f3, 0x03f3, 0x03f3, 0x03f3, 0x03f3,
    0x0227, 0x00de, 0x006d, 0x0249, 0x023c, 0x009f, 0x00a5, 0x00a5, 0x00a5, 0x023f,
    0x008d, 0x024c, 0x00a3, 0x00a5, 0x00a5, 0x0235, 0x0239, 0x0239, 0x024e, 0x00a5,
    0x00a5, 0x00a5, 0x0239, 0x0239, 0x0239, 0x0231, 0x00a5, 0x00a5, 0x0235, 0x0239,
    0x0239, 0x024e, 0x00a5, 0x00a5, 0x00a5, 0x0239, 0x0239, 0x0239, 0x0231, 0x00a5,
    0x00a5, 0x00a7, 0x0239, 0x0239, 0x0239, 0x00a3, 0x00a5, 0x00a5, 0x0164, 0x0237,
    0x0239, 0x0239, 0x00a1, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084,
    0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084,
    0x0084, 0x0126, 0x0254, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084,
    0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x010c, 0x0083, 0x0118, 0x0124, 0x0084,
    0x0084, 0x0084, 0x0084, 0x0084, 0x035b, 0x0373, 0x0383, 0x03eb, 0x0104, 0x010e,
    0x0341, 0x038b, 0x02e9, 0x00f3, 0x00f8, 0x0118, 0x014c, 0x0084, 0x0084, 0x0018,
    0x0084, 0x0084, 0x0084, 0x0084, 0x0141, 0x0084, 0x0128, 0x0309, 0x0310, 0x0084,
    0x004b, 0x0159, 0x0052, 0x03ab, 0x0303, 0x02f6, 0x03b3, 0x02f1, 0x01b7, 0x0011,
    0x00a5, 0x00a5, 0x00a5, 0x0008, 0x00ab, 0x0009, 0x0009, 0x0009, 0x00a8, 0x0009,
    0x0009, 0x0009, 0x00ce, 0x00ce, 0x00ce, 0x00ce, 0x00ce, 0x00ce, 0x025c, 0x025c,
    0x025c, 0x025c, 0x025c, 0x025c, 0x0353, 0x011c, 0x02e4, 0x0168, 0x0084, 0x0084,
    0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084, 0x0084,
    0x02a3, 0x01a9, 0x00ef, 0x0009, 0x017a, 0x0184, 0x017c, 0x0186, 0x017a, 0x0184,
    0x017a, 0x0184, 0x017c, 0x0186, 0x0264, 0x0033, 0x017a, 0x0184, 0x037b, 0x0000,
    0x017a, 0x0184, 0x017a, 0x0184, 0x017a, 0x0184, 0x01b1, 0x003a, 0x0067, 0x0363,
    0x026b, 0x002c, 0x007c, 0x036b, 0x0067, 0x00c6, 0x02ab, 0x02ab, 0x02ab, 0x02ab,
    0x02ab, 0x02b7, 0x02b7, 0x02b7, 0x02b7, 0x02b7, 0x0009, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x02ab, 0x02ab,
    0x02ab, 0x02ab, 0x02af, 0x02b7, 0x02b7, 0x02b7, 0x02b7, 0x02bb
};

static const uint16_t unicode_offsets2[498] = {
    0x016f, 0x02ed, 0x030c, 0x0127, 0x01c0, 0x01da, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0039, 0x0003, 0x0003, 0x0280,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x010f, 0x0218,
    0x02d3, 0x034c, 0x01ec, 0x018f, 0x0003, 0x0003, 0x000d, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x032c, 0x002d, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0230, 0x008d, 0x0003, 0x0003,
    0x0003, 0x00e5, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0000, 0x0003, 0x0003, 0x0003, 0x0274, 0x0003, 0x0003, 0x0003, 0x0003,
    0x036c, 0x0258, 0x0003, 0x0157, 0x0003, 0x0003, 0x0003, 0x0003, 0x01a0, 0x0246,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0065, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0071, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x00ad, 0x02a0,
    0x02b3, 0x00cb, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0059,
    0x00ff, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0147,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0206
};

static inline const struct CaseMappings *
get_unicode_entry(xkb_keysym_t ks)
{
    return &unicode_data[unicode_offsets1[unicode_offsets2[ks >> 8] + ((ks >> 3) & 0x1f)] + (ks & 0x07)];
}

xkb_keysym_t
xkb_keysym_to_lower(xkb_keysym_t ks)
{
    if (ks <= 0x13be) {
        const struct CaseMappings *m = get_legacy_keysym_entry(ks);
        return (m->lower) ? ks + m->offset : ks;
    } else if (XKB_KEYSYM_UNICODE_MIN <= ks && ks <= 0x0101f189) {
        const struct CaseMappings *m = get_unicode_entry(ks - XKB_KEYSYM_UNICODE_OFFSET);
        if (m->lower) {
            ks = ks + m->offset;
            return (ks < XKB_KEYSYM_UNICODE_MIN) ? ks - XKB_KEYSYM_UNICODE_OFFSET : ks;
        } else {
            return ks;
        }
    } else {
        return ks;
    }
}

xkb_keysym_t
xkb_keysym_to_upper(xkb_keysym_t ks)
{
    if (ks <= 0x13be) {
        const struct CaseMappings *m = get_legacy_keysym_entry(ks);
        return (m->upper) ? ks - m->offset : ks;
    } else if (XKB_KEYSYM_UNICODE_MIN <= ks && ks <= 0x0101f189) {
        const struct CaseMappings *m = get_unicode_entry(ks - XKB_KEYSYM_UNICODE_OFFSET);
        if (m->upper) {
            ks = ks - m->offset;
            return (ks < XKB_KEYSYM_UNICODE_MIN) ? ks - XKB_KEYSYM_UNICODE_OFFSET : ks;
        } else {
            return ks;
        }
    } else {
        return ks;
    }
}

bool
xkb_keysym_is_lower(xkb_keysym_t ks)
{
    /* This predicate matches keysyms with their corresponding Unicode code point
     * having the Unicode property “Lowercase”.
     *
     * Here: a keysym is lower case if it has an upper case and no lower case.
     * Note: title case letters may have both. Example for U+01F2:
     * • U+01F1 Ǳ: upper case
     * • U+01F2 ǲ: title case
     * • U+01F3 ǳ: lower case
     */
    if (ks <= 0x13be) {
        const struct CaseMappings *m = get_legacy_keysym_entry(ks);
        return m->upper && !m->lower;
    } else if (XKB_KEYSYM_UNICODE_MIN <= ks && ks <= 0x0101f189) {
        const struct CaseMappings *m = get_unicode_entry(ks - XKB_KEYSYM_UNICODE_OFFSET);
        return m->upper && !m->lower;
    } else {
        return false;
    }
}

bool
xkb_keysym_is_upper_or_title(xkb_keysym_t ks)
{
    /* This predicate matches keysyms with their corresponding Unicode code point
     * having the Unicode properties “Uppercase” or General Category “Lt”.
     *
     * Here: a keysym is upper case or title case if it has a lower case. */
    if (ks <= 0x13be) {
        return get_legacy_keysym_entry(ks)->lower;
    } else if (XKB_KEYSYM_UNICODE_MIN <= ks && ks <= 0x0101f189) {
        return get_unicode_entry(ks - XKB_KEYSYM_UNICODE_OFFSET)->lower;
    } else {
        return false;
    }
}
