/* vi:set ts=8 sts=0 sw=8:
 * $Id: toolbar.c,v 1.15 1999/01/21 21:37:27 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn <kahn@zk3.dec.com>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <glib.h>
#include "main.h"
#include "doc.h"
#include "toolbar.h"
#include "msgbar.h"
#include "file.h"
#include "prefs.h"


/*** local definitions ***/
#define PIC_TEXT		"Toolbar with icons and text"
#define PIC_ONLY		"Toolbar with icons only"
#define TEXT_ONLY		"Toolbar with text only"
#define TOOLBAR_ON		"Toolbar ON"
#define TOOLBAR_OFF		"Toolbar OFF"
#define TOOLTIPS_ON		"Tooltips ON"
#define TOOLTIPS_OFF		"Tooltips OFF"
#define RAISED_ON		"RAISED toolbar buttons"
#define RAISED_OFF		"FLAT toolbar buttons"
#define HTML_TOOLBAR_ON		"HTML Toolbar ON"
#define HTML_TOOLBAR_OFF	"HTML Toolbar OFF"


/*** local function prototypes ***/
static GtkWidget *new_pixmap(char *fname, GtkWidget *gdkw, GtkWidget *w);
static GtkWidget *toolbar_create_common(toolbar_data_t *tbdata, win_t *w);
static void tb_off(GtkWidget *wgt);
static void tb_on(GtkWidget *wgt);
static void tb_ttip_off(GtkWidget *wgt);
static void tb_ttip_on(GtkWidget *wgt);
static void tb_pic_only_private(GtkWidget *tb);
static void tb_pic_text_private(GtkWidget *tb);
static void tb_text_only_private(GtkWidget *tb);
#ifdef GTK_HAVE_FEATURES_1_1_0
static void tb_raised_off(GtkWidget *tb);
static void tb_raised_on(GtkWidget *tb);
#endif


/*** global function definitions ***/
/*
 * redraws all toolbars for a specified window
 */
void
tb_redraw(void *data)
{
	win_t *w = (win_t *)data;

	if (IS_SHOW_TOOLBAR()) {
		tb_on(w->main_tb_h);
		tb_on(w->dlw_tb_h);
	} else {
		tb_off(w->main_tb_h);
		tb_off(w->dlw_tb_h);
	}

#ifdef APP_GNP
	if (IS_SHOW_HTML_TOOLBAR())
		tb_on(w->html_tb_h);
	else
		tb_off(w->html_tb_h);
#endif

#ifdef GTK_HAVE_FEATURES_1_1_0
	if (IS_TOOLBAR_RAISED()) {
		tb_raised_on(w->main_tb);
#ifdef APP_GNP
		tb_raised_on(w->html_tb);
#endif
		tb_raised_on(w->dlw_tb);
	} else {
		tb_raised_off(w->main_tb);
#ifdef APP_GNP
		tb_raised_off(w->html_tb);
#endif
		tb_raised_off(w->dlw_tb);
	}
#endif

	if (IS_TEXT_PIC_TOOLBAR()) {
		tb_pic_text_private(w->main_tb);
#ifdef APP_GNP
		tb_pic_text_private(w->html_tb);
#endif
		tb_pic_text_private(w->dlw_tb);
	} else if (IS_PIC_TOOLBAR()) {
		tb_pic_only_private(w->main_tb);
#ifdef APP_GNP
		tb_pic_only_private(w->html_tb);
#endif
		tb_pic_only_private(w->dlw_tb);
	} else {
		tb_text_only_private(w->main_tb);
#ifdef APP_GNP
		tb_text_only_private(w->html_tb);
#endif
		tb_text_only_private(w->dlw_tb);
	}
} /* tb_redraw */


/*
 * PUBLIC: tb_pic_only
 *
 * updates toolbar to show buttons with icons only
 */
void
tb_pic_only(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	SET_PIC_ONLY_TOOLBAR();
	CLEAR_PIC_TEXT_TOOLBAR();
	CLEAR_TEXT_ONLY_TOOLBAR();
	win_foreach(tb_redraw);
	msgbar_printf(w, PIC_ONLY);
} /* tb_pic_only */


/*
 * PUBLIC: tb_pic_text
 *
 * updates toolbar to show buttons with icons and text
 */
void
tb_pic_text(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	SET_PIC_TEXT_TOOLBAR();
	CLEAR_TEXT_ONLY_TOOLBAR();
	CLEAR_PIC_ONLY_TOOLBAR();
	win_foreach(tb_redraw);
	msgbar_printf(w, PIC_TEXT);
} /* tb_pic_text */


/*
 * PUBLIC: tb_text_only
 *
 * updates toolbar to show buttons with text only
 */
void
tb_text_only(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	SET_TEXT_ONLY_TOOLBAR();
	CLEAR_PIC_TEXT_TOOLBAR();
	CLEAR_PIC_ONLY_TOOLBAR();
	win_foreach(tb_redraw);
	msgbar_printf(w, TEXT_ONLY);
} /* tb_text_only */


/*
 * PUBLIC: tb_toggle
 *
 * toggles main toolbar and doclist toolbar
 */
void
tb_toggle(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	if (IS_SHOW_TOOLBAR()) {
		tb_off(w->main_tb_h);
		tb_off(w->dlw_tb_h);
		CLEAR_SHOW_TOOLBAR();
		msgbar_printf(w, TOOLBAR_OFF);
	} else {
		tb_on(w->main_tb_h);
		tb_on(w->dlw_tb_h);
		SET_SHOW_TOOLBAR();
		msgbar_printf(w, TOOLBAR_ON);
	}
	win_foreach(tb_redraw);
} /* tb_toggle */


#ifdef APP_GNP
/*
 * PUBLIC: tb_html_toggle
 *
 * toggles html toolbar
 */
void
tb_html_toggle(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	if (IS_SHOW_HTML_TOOLBAR()) {
		tb_off(w->html_tb_h);
		CLEAR_SHOW_HTML_TOOLBAR();
		msgbar_printf(w, HTML_TOOLBAR_OFF);
	} else {
		tb_on(w->html_tb_h);
		SET_SHOW_HTML_TOOLBAR();
		msgbar_printf(w, HTML_TOOLBAR_ON);
	}
	win_foreach(tb_redraw);
} /* tb_html_toggle */
#endif


#ifdef GTK_HAVE_FEATURES_1_1_0
/*
 * PUBLIC: tb_raised_toggle
 *
 * toggles toolbar button relief
 */
void
tb_raised_toggle(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	if (IS_TOOLBAR_RAISED()) {
		tb_raised_off(w->main_tb);
#ifdef APP_GNP
		tb_raised_off(w->html_tb);
#endif
		tb_raised_off(w->dlw_tb);
		CLEAR_TOOLBAR_RAISED();
		msgbar_printf(w, RAISED_OFF);
	} else {
		tb_raised_on(w->main_tb);
#ifdef APP_GNP
		tb_raised_on(w->html_tb);
#endif
		tb_raised_on(w->dlw_tb);
		SET_TOOLBAR_RAISED();
		msgbar_printf(w, RAISED_ON);
	}
	win_foreach(tb_redraw);
} /* tb_raised_toggle */
#endif


/*
 * PUBLIC: tb_ttip_toggle
 *
 * toggles toolbar tooltips on/off
 */
void
tb_ttip_toggle(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	if (IS_SHOW_TOOLTIPS()) {
		tb_ttip_off(w->main_tb);
#ifdef APP_GNP
		tb_ttip_off(w->html_tb);
#endif
		tb_ttip_off(w->dlw_tb);
		CLEAR_SHOW_TOOLTIPS();
		msgbar_printf(w, TOOLTIPS_OFF);
	} else {
		tb_ttip_on(w->main_tb);
#ifdef APP_GNP
		tb_ttip_on(w->html_tb);
#endif
		tb_ttip_on(w->dlw_tb);
		SET_SHOW_TOOLTIPS();
		msgbar_printf(w, TOOLTIPS_ON);
	}
} /* tb_ttip_toggle */


/*
 * PUBLIC: toolbar_create
 *
 * in:
 *	w: current window
 *	parent: parent widget (a box type) to hold toolbar (or handle)
 *	tbdata: toolbar data table
 *
 * out:
 *	tb: the toolbar
 *	tb_h: toolbar handle
 */
void
toolbar_create(win_t *w, GtkWidget *parent, toolbar_data_t tbdata[],
	       GtkWidget **tb, GtkWidget **tb_h)
{
	*tb = toolbar_create_common(tbdata, w);
	GTK_WIDGET_UNSET_FLAGS(*tb, GTK_CAN_FOCUS);
#ifdef GTK_HAVE_FEATURES_1_1_0
	gtk_toolbar_set_button_relief(GTK_TOOLBAR(*tb),
		(IS_TOOLBAR_RAISED()) ? GTK_RELIEF_NORMAL : GTK_RELIEF_NONE);
	gtk_container_border_width(GTK_CONTAINER(*tb), 2);
#endif
	*tb_h = gtk_handle_box_new();
	gtk_container_add(GTK_CONTAINER(*tb_h), *tb);
	if (parent)
		gtk_box_pack_start(GTK_BOX(parent), *tb_h, FALSE, TRUE, 0);
	if (IS_SHOW_TOOLBAR())
		gtk_widget_show(*tb_h);
} /* toolbar_create */


/*** local function definitions */
/*
 * PRIVATE: new_pixmap
 *
 * taken from testgtk.c
 */
static GtkWidget*
new_pixmap(char *fname, GtkWidget *gtkw, GtkWidget *wgt)
{
	extern char *pkgdatadir;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	char *buf;
	int len;

	len = strlen(pkgdatadir) + strlen(fname) + 6;
	buf = (char *)g_malloc(len * sizeof(char));
	g_snprintf(buf, len, "%s/xpm/%s", pkgdatadir, fname);
	if (!file_exist(buf))
		g_snprintf(buf, len, "xpm/%s", fname);
	GNPDBG_TOOLBAR(("new_pixmap: buf = '%s'\n", buf));
	pixmap = gdk_pixmap_create_from_xpm(
			gtkw->window,
			&mask,
			&gtkw->style->bg[GTK_STATE_NORMAL],
			buf);
	g_free(buf);
	return gtk_pixmap_new(pixmap, mask);
} /* new pixmap */


/*
 * PRIVATE: tb_off
 *
 * hides the toolbar
 */
static void
tb_off(GtkWidget *tbh)
{
	if (tbh && GTK_WIDGET_VISIBLE(tbh))
		gtk_widget_hide(tbh);
} /* tb_on */


/*
 * PRIVATE: tb_off
 *
 * unhides the toolbar
 */
static void
tb_on(GtkWidget *tbh)
{
	if (tbh && !GTK_WIDGET_VISIBLE(tbh))
		gtk_widget_show(tbh);
} /* tb_on */


#ifdef GTK_HAVE_FEATURES_1_1_0
/*
 * PRIVATE: tb_raised_off
 *
 * sets toolbar buttons to GTK_RELIEF_NONE
 */
static void
tb_raised_off(GtkWidget *tb)
{
	if (tb)
		gtk_toolbar_set_button_relief(GTK_TOOLBAR(tb), GTK_RELIEF_NONE);
} /* tb_raised_off */


/*
 * PRIVATE: tb_raised_on
 *
 * sets toolbar buttons to GTK_RELIEF_NORMAL
 */
static void
tb_raised_on(GtkWidget *tb)
{
	if (tb)
		gtk_toolbar_set_button_relief(GTK_TOOLBAR(tb),
					      GTK_RELIEF_NORMAL);
} /* tb_raised_on */
#endif


/*
 * PRIVATE: tb_ttip_off
 *
 * turns OFF tooltips
 */
static void
tb_ttip_off(GtkWidget *tb)
{
	if (tb)
		gtk_toolbar_set_tooltips(GTK_TOOLBAR(tb), FALSE);
} /* tb_ttip_off */


/*
 * PRIVATE: tb_ttip_on
 *
 * turns ON tooltips
 */
static void
tb_ttip_on(GtkWidget *tb)
{
	if (tb)
		gtk_toolbar_set_tooltips(GTK_TOOLBAR(tb), TRUE);
} /* tb_ttip_on */


/*
 * PRIVATE: toolbar_create_common
 *
 * common routine to create a toolbar.
 *
 * in: toolbar_data_t and pointer to callback data of gE_data type
 * out: GtkWidget *toolbar
 */
static GtkWidget *
toolbar_create_common(toolbar_data_t *tbdata, win_t *w)
{
	GtkWidget *tb;
	toolbar_data_t *tbdp = tbdata;

	g_assert(tbdp != NULL);

	if (IS_TEXT_TOOLBAR()) {
		tb = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL,
				     GTK_TOOLBAR_TEXT);
	} else if (IS_PIC_TOOLBAR()) {
		tb = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL,
				     GTK_TOOLBAR_ICONS);
	} else {
		SET_PIC_TEXT_TOOLBAR();
		tb = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL,
				     GTK_TOOLBAR_BOTH);
	}

	while (tbdp->text != NULL) {
		gtk_toolbar_append_item(
			GTK_TOOLBAR(tb),
			tbdp->text,
			tbdp->tooltip_text,
			tbdp->tooltip_private_text,
			new_pixmap(tbdp->icon, w->toplev, tb),
			(GtkSignalFunc)tbdp->callback,
			(tbdp->cbdata) ? tbdp->cbdata : (gpointer)w);

		tbdp++;

		if (tbdp->text != NULL && strcmp(tbdp->text, " SPACE ") == 0) {
			gtk_toolbar_append_space(GTK_TOOLBAR(tb));
			tbdp++;
		}
	}

	/* set according to preferences */
	gtk_toolbar_set_tooltips(GTK_TOOLBAR(tb), IS_SHOW_TOOLTIPS());

	/*
	 * this is OK to show even if prefs says don't show, because it's
	 * the toolbar handle that gets shown
	 */
	gtk_widget_show(tb);
	return tb;
} /* toolbar_create_common */


static void
tb_pic_only_private(GtkWidget *tb)
{
	if (tb) {
		gtk_toolbar_set_style(GTK_TOOLBAR(tb), GTK_TOOLBAR_ICONS);
		if (GTK_WIDGET_VISIBLE(tb)) {
			gtk_widget_hide(tb);
			gtk_widget_show(tb);
		}

		SET_PIC_ONLY_TOOLBAR();
		CLEAR_PIC_TEXT_TOOLBAR();
		CLEAR_TEXT_ONLY_TOOLBAR();
	}
} /* tb_pic_only_private */


static void
tb_pic_text_private(GtkWidget *tb)
{
	if (tb) {
		gtk_toolbar_set_style(GTK_TOOLBAR(tb), GTK_TOOLBAR_BOTH);
		if (GTK_WIDGET_VISIBLE(tb)) {
			gtk_widget_hide(tb);
			gtk_widget_show(tb);
		}

		SET_PIC_TEXT_TOOLBAR();
		CLEAR_TEXT_ONLY_TOOLBAR();
		CLEAR_PIC_ONLY_TOOLBAR();
	}
} /* tb_pic_text_private */


static void
tb_text_only_private(GtkWidget *tb)
{
	if (tb) {
		gtk_toolbar_set_style(GTK_TOOLBAR(tb), GTK_TOOLBAR_TEXT);
		if (GTK_WIDGET_VISIBLE(tb) && IS_SHOW_TOOLBAR()) {
			gtk_widget_hide(tb);
			gtk_widget_show(tb);
		}

		SET_TEXT_ONLY_TOOLBAR();
		CLEAR_PIC_TEXT_TOOLBAR();
		CLEAR_PIC_ONLY_TOOLBAR();
	}
} /* tb_text_only_private */


/* the end */
