/* vi:set ts=8 sts=0 sw=8:
 * $Id: prefs.c,v 1.36 1999/01/21 21:11:41 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn <kahn@zk3.dec.com>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <errno.h>
#include <values.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <glib.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#ifndef GTK_HAVE_FEATURES_1_1_0
#include "gtkfontsel.h"
#endif
#include "main.h"
#include "menu.h"
#include "file.h"
#include "dialog.h"
#include "win.h"
#include "msgbar.h"
#include "msgbox.h"
#include "toolbar.h"
#ifdef WANT_PROJECT
#include "prjbar.h"
#endif
#include "misc.h"
#include "prefs.h"


/*** global variables ***/
app_prefs_t prefs;

/*** local definitions ***/
#define COL_VAL_GDK	65535.0		/* GdkColor is 16-bit color */
#define PREFS_NAME_LEN	32		/* max len for pref name in rc file */
#ifdef APP_GNP
#define PREFS_DIR	".gnp/"		/* should be in $HOME */
#endif
#define APPRC		"apprc"		/* stores app-specific prefs */
#define APPGTKRC	"appgtkrc"	/* stores app-specific GTK prefs */
#define GTKRC		"gtkrc"		/* stores gtk-related prefs */
#define MAX_FONT_LEN	MAXPATH
#define MAX_RGB_STR	32
#define DEFAULT_FONT	"-adobe-courier-medium-r-normal-*-*-120-*-*-m-*-iso8859-1"

typedef enum {
	PrefGeneric,	/* generic */
	PrefBool,	/* its boolean, but the variable must be a "long" */
	PrefString,	/* (char *) */
	PrefInt,	/* int or uint */
	PrefFloat,	/* float */
	PrefByte,	/* byte */
	PrefShort	/* short */
} ptype_t;

/*
 * f: callback used when updating prefs (typically, win_foreach()).
 * c: callback data function passed to f().
 */
typedef struct {
	char *	name;	/* pref name */
	ptype_t	type;	/* pref type */
	void * 	addr;	/* address of data value */
	long	min;	/* min size/len/value */
	long	max;	/* max size/length/value */
	gulong	val;	/* multi-purpose value.  currently used for */
			/* max string len and boolean bit flags */
	void	(*f)(void (*g)(void *));
	void 	(*c)(void *);
} prefs_t;

/* used by color selection */
typedef enum { Foreground, Background, HighlightFg, HighlightBg } which_text_t;
typedef struct {
	GtkWidget *	cs;
	which_text_t	which;
} prefs_color_sel_t;


/*** local variables ***/
static bool_t		pre102 = FALSE;	/* using gnp+ before v1.0.2 */
static GtkPositionType	prefs_tabpos;	/* prefs tab position */

static char *	appgtkrc = NULL;	/* app specific gtk rc */
static void	appgtk_rc_update(gpointer data);	/* need fwd ref */

/*
 * to add a new configurable preference option:
 *
 * 1. declare a global variable up in the "global variables" section.  make
 * sure that it's one of the following types: int, bool, float, byte, short,
 * or "char *".
 *
 * 2. add a corresponding entry below into the variable 'app_prefs'.  your
 * best bet is to just cut and paste an existing entry and see how it works.
 * keep the entries in alphabetical order.
 *
 * 3. create and add the necessary widgets into the Preferences window popup
 * dialog.  the most commonly used ones are a button and a text entry.  be
 * sure to add these widgets into the wgtopt_list.  see one of the existing
 * routines as an example (e.g., prefs_frame_toolbar_create()).
 *
 *
 * i know this setup may not be easiest way for another developer to use.  for
 * example, ideally one would like to do something such as the following:
 *
 *	prefs_set_data("doc_tab_position", "top");
 *	prefs_get_data("print_command", buf);
 *
 * however, the big downside to this is that when getting data, you need
 * routines to translate the values *ANYWAY*.  in other words, every time you
 * successfully get data, you still have to determing *what* that data is and
 * *how* it affects preferences.  for instance, if data is supposed to
 * represent an integer, you'd have to then convert it to an integer and
 * determine what to do.
 *
 * so although the prefs_set_data() and prefs_get_data() looks nice on the
 * outside, you still have to write a whole bunch of code to figure out what
 * do with it once you've gotten it.  the tabular method that i'm using
 * effectively puts the "what", "how", and "where" into the table itself.
 * so all you have to do is specify it ONCE, and the routines that manipulate
 * the table will take care of it so you don't have to.
 *
 * the prefs_set_data() and prefs_get_data() technique may work well if it
 * were provided as a library (e.g., Gnome does this), but since this part of
 * the code is not intended to be a library, (i think) the tabular method is
 * better.
 */
static prefs_t		app_prefs[] = {
#ifdef APP_GNP
#ifdef USE_AUTO_INDENT
	{ "use_auto_indent",
		PrefBool,
		&prefs.options,
		0, 1, AUTO_INDENT,
		NULL_CALLBACK, NULL,
	},
#endif
	{ "autosave_timeout",
		PrefInt,
		&prefs.autosave,
		0, 1440, 0,
		NULL_CALLBACK, NULL,
	},
#endif
	{ "doc_tab_position",
		PrefInt,
		&prefs.tabpos,
		0, 3, 0,
		win_foreach,
		win_redraw_doc_tab,
	},
	{ "max_num_recent",
		PrefByte,
		&prefs.maxrecent,
		0, 255, 0,
		NULL_CALLBACK, NULL,
	},
	{ "msgbox_max_msg",
		PrefInt,
		&prefs.msgbox_max_msg,
		0, 32768, 0,
		NULL_CALLBACK, NULL,
	},
	{ "msgbox_per_del",
		PrefInt,
		&prefs.msgbox_per_del,
		0, 100, 0,
		NULL_CALLBACK, NULL,
	},
	{ "prefs_tab_position",
		PrefInt,
		&prefs_tabpos,
		0, 3, 0,
		NULL_CALLBACK, NULL,
	},
#ifdef APP_GNP
	{ "print_command",
		PrefString,
		&prefs.printcmd,
		0, 0, MAXPATH,
		NULL_CALLBACK, NULL,
	},
#endif
	{ "save_win_pos",
		PrefBool,
		&prefs.options1,
		0, 1, SAVE_WIN_POS,
		NULL_CALLBACK, NULL,
	},
	{ "save_win_height",
		PrefBool,
		&prefs.options1,
		0, 1, SAVE_WIN_HEIGHT,
		NULL_CALLBACK, NULL,
	},
	{ "save_win_width",
		PrefBool,
		&prefs.options1,
		0, 1, SAVE_WIN_WIDTH,
		NULL_CALLBACK, NULL,
	},
	{ "show_doc_tabs",
		PrefBool,
		&prefs.options1,
		0, 1, SHOW_TABS,
		NULL_CALLBACK, NULL,
	},
#ifdef APP_GNP
	{ "show_html_toolbar",
		PrefBool,
		&prefs.options1,
		0, 1, SHOW_HTML_TOOLBAR,
		win_foreach, tb_redraw,
	},
#ifdef WANT_PROJECT
	{ "show_project_bar",
		PrefBool,
		&prefs.options1,
		0, 1, SHOW_PRJBAR,
		win_foreach, prjbar_redraw,
	},
#endif	/* WANT_PROJECT */
#endif
	{ "show_message_bar",
		PrefBool,
		&prefs.options1,
		0, 1, SHOW_MSGBAR,
		win_foreach, msgbar_redraw,
	},
	{ "show_toolbar",
		PrefBool,
		&prefs.options1,
		0, 1, SHOW_TOOLBAR,
		win_foreach, tb_redraw,
	},
	{ "show_tooltips",
		PrefBool,
		&prefs.options1,
		0, 1, SHOW_TOOLTIPS,
		NULL_CALLBACK, NULL,
	},
#ifdef APP_GNP
	{ "text_bg_color",
		PrefString,
		&prefs.text_bg_str,
		0, 1, MAX_RGB_STR,
		NULL_CALLBACK, NULL,
	},
	{ "text_fg_color",
		PrefString,
		&prefs.text_fg_str,
		0, 0, MAX_RGB_STR,
		NULL_CALLBACK, NULL,
	},
	{ "text_font",
		PrefString,
		&prefs.text_font_str,
		0, 0, MAX_FONT_LEN,
		win_doc_foreach, doc_redraw,
	},
	{ "text_hlbg_color",
		PrefString,
		&prefs.text_hlbg_str,
		0, 1, MAX_RGB_STR,
		NULL_CALLBACK, NULL,
	},
	{ "text_hlfg_color",
		PrefString,
		&prefs.text_hlfg_str,
		0, 0, MAX_RGB_STR,
		NULL_CALLBACK, NULL,
	},
#endif
	{ "tmp_directory",
		PrefString,
		&prefs.tmpdir,
		0, 0, MAXPATH,
		NULL_CALLBACK, NULL,
	},
	{ "toolbar_piconly",
		PrefBool,
		&prefs.options1,
		0, 1, PIC_ONLY_TOOLBAR,
		win_foreach, tb_redraw,
	},
	{ "toolbar_pictext",
		PrefBool,
		&prefs.options1,
		0, 1, PIC_TEXT_TOOLBAR,
		win_foreach, tb_redraw,
	},
#ifdef GTK_HAVE_FEATURES_1_1_0
	{ "toolbar_raised",
		PrefBool,
		&prefs.options1,
		0, 1, TOOLBAR_RAISED,
		win_foreach, tb_redraw,
	},
#endif
	{ "toolbar_textonly",
		PrefBool,
		&prefs.options1,
		0, 1, TEXT_ONLY_TOOLBAR,
		win_foreach, tb_redraw,
	},
	{ "use_msgbox",
		PrefBool,
		&prefs.options1,
		0, 1, USE_MSGBOX,
		NULL_CALLBACK, NULL,
	},
#ifdef APP_GNP
	{ "use_wordwrap",
		PrefBool,
		&prefs.options1,
		0, 1, USE_WORDWRAP,
		NULL_CALLBACK, NULL,
	},
#endif
	{ "win_height",
		PrefInt,
		&prefs.win_height,
		-1, 8191, 0,
		NULL_CALLBACK, NULL,
	},
	{ "win_width",
		PrefInt,
		&prefs.win_width,
		-1, 8191, 0,
		NULL_CALLBACK, NULL,
	},
	{ "win_xpos",
		PrefInt,
		&prefs.win_xpos,
		-1, 8191, 0,
		NULL_CALLBACK, NULL,
	},
	{ "win_ypos",
		PrefInt,
		&prefs.win_ypos,
		-1, 8191, 0,
		NULL_CALLBACK, NULL,
	},
	{ NULL, PrefBool, NULL, 0, 0, 0, NULL_CALLBACK, NULL }
};

/*
 * in order for the preferences to take effect and be saved *after* the user
 * clicked "Ok" (as opposed to the preferences taking effect immediately each
 * time the user selects an option in the prefs window), we build a list of
 * items.  each item contains the following:
 *
 *	1. a pointer into the correct entry into the app_prefs table.  since
 *	once the user selects "Ok", we're effectively saving the new options
 *	anyway, so let's just reuse the code used for saving the prefs.
 *
 *	2. the widget representing the options (e.g., a check box).  this
 *	is used so we know what state it's in.  or rather, what the user
 *	has selected.  the widget is implied by what the preference is.
 *	if it's PrefBool, it must be a button widget.  all others are most
 *	likely a text entry type.
 *
 * once the user hits the "Save Preferences" button, we then scan this list,
 * and depending on the state of the widget/button/etc, we can look in the
 * app_prefs table to see what to set and to what value.
 */
typedef enum {
	GenericType,
	EntryType,
	ButtonType,
	ComboType
} wgttype_t;

typedef struct {
	prefs_t *pap;		/* pointer into app_prefs table */
	GtkWidget *wgt;		/* the widget */
	wgttype_t type;		/* type of widget */
	void *data;		/* data to use */
} wgtopt_t;
static GSList *wgtopt_list = NULL;	/* list of wgtopt_t's */

/* other global variables within this file */
static char *		apprc     = NULL;	/* full path to apprc */
static char *		gtkrc     = NULL;	/* full path to gtkrc */
static GtkWidget *	prefs_win = NULL;	/* preferences popup window */
static GtkWidget *	prefs_nb  = NULL;	/* notebook for prefs window */
#ifdef APP_GNP
#ifdef GTK_HAVE_FEATURES_1_1_0
static GtkWidget *	prefs_txt = NULL;	/* preferences text window */
#endif
#endif
static GList *		tabpos_list = NULL;	/* text for combo widget */


/*** local function prototypes ***/
static char *		prefs_tf(long opts, long mask);
static void		prefs_read(FILE *fp);
static void		prefs_bool_set(prefs_t *pap, char *data);
static void		prefs_byte_set(prefs_t *pap, char *data);
static void		prefs_float_set(prefs_t *pap, char *data);
static void		prefs_int_set(prefs_t *pap, char *data);
static void		prefs_string_set(prefs_t *pap, char *data);
static void		prefs_dialog_create(win_t *w);
static void		prefs_win_cancel(GtkWidget *wgt, gpointer cbdata);
static void		prefs_win_destroy(GtkWidget *wgt, gpointer cbdata);
static void		prefs_save_cb(GtkWidget *wgt, gpointer cbdata);
static void		prefs_wgtopt_list_add(
				GtkWidget *, wgttype_t, void *, char *);
static void		prefs_wgtopt_list_free(void);
static void		prefs_page_document_create(GtkWidget *nb);
static void		prefs_page_misc_create(GtkWidget *nb);
static void		prefs_frame_toolbar_create(GtkWidget *parent_vbox);
static void		prefs_frame_doctab_create(GtkWidget *parent_vbox);
static void		prefs_page_appearance_create(GtkWidget *nb);
static void		prefs_frame_msgbar_create(GtkWidget *parent);
static void		prefs_page_window_create(GtkWidget *nb);
static void		prefs_frame_msgbox_create(GtkWidget *parent);
static void		prefs_update(void);
static void		prefs_tab_pos_change(void);
#ifdef APP_GNP
static void		prefs_page_fonts_colors(GtkWidget *nb);
static void		prefs_text_fg_cb(GtkWidget *wgt, gpointer cbdata);
static void		prefs_text_bg_cb(GtkWidget *wgt, gpointer cbdata);
static void		prefs_text_hlfg_cb(GtkWidget *wgt, gpointer cbdata);
static void		prefs_text_hlbg_cb(GtkWidget *wgt, gpointer cbdata);
#endif
static GtkWidget *	prefs_check_button_with_label_create(
				GtkWidget *, char *, long , char *,
				GtkWidget *table,
				int lft1, int rht1, int top1, int bot1);
static GtkWidget *	prefs_entry_with_label_create(
				GtkWidget *, char *, char *, char *,
				GtkWidget *table,
				int lft1, int rht1, int top1, int bot1,
				int lft2, int rht2, int top2, int bot2);
static GtkWidget *	prefs_spin_button_with_label_create(
				GtkWidget *, char *, float , float , float ,
				float , float , float , char *,
				GtkWidget *table,
				int lft1, int rht1, int top1, int bot1,
				int lft2, int rht2, int top2, int bot2);
static GtkWidget *	prefs_radio_button_with_label_create(
				GtkWidget *, GtkWidget *, char *, long, char *);
static GtkWidget *	prefs_pulldown_with_label_create(
				GtkWidget *, char *, GList *, char *);
static void		prefs_box_in_frame(char *frametext, int border_width,
				GtkWidget **frame, GtkWidget **vbox, bool_t);
#ifdef APP_GNP
static void		prefs_color_sel_changed(GtkWidget *, gpointer cbdata);
static void		prefs_color_sel_ok(GtkWidget *wgt, gpointer cbdata);
static void		prefs_color_sel_create(which_text_t which);
static void		prefs_color_sel_close(GtkWidget *, gpointer cbdata);
static void		prefs_color_sel_destroy(GtkWidget *, gpointer cbdata);
static void		prefs_font_sel_cb(GtkWidget *wgt, gpointer cbdata);
static void		prefs_font_sel_ok(GtkWidget *wgt, gpointer cbdata);
static void		prefs_font_sel_cancel(GtkWidget *wgt, gpointer cbdata);
#endif


/*** global function definitions ***/
/*
 * PUBLIC: prefs_cb
 *
 * callback invoked from menu to bring up preferences window.
 */
void
prefs_cb(GtkWidget *wtg, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	if (prefs_win != NULL) {
		gdk_window_raise(prefs_win->window);
		return;
	}

	prefs_dialog_create(w);
} /* prefs_cb */


/*
 * PUBLIC: prefs_init
 *
 * initializes application wide preferences.  called from main().
 * TODO: specify/use system-wide app-defaults file
 */
void
prefs_init(void)
{
	char *homedir, *buf;
	FILE *fp;
	int num;

	/* initialize to default values */
	prefs.appdir         = NULL;
#ifdef APP_GNP
	prefs.printcmd       = g_strdup("lpr %s");
#endif
	prefs.tmpdir         = g_strdup("/var/tmp");
	prefs.tabpos         = GTK_POS_TOP;
#ifdef APP_GNP
	prefs.autosave       = 5;
#endif
	prefs.options1        = DEFAULT_OPTIONS;
	prefs.maxrecent      = 4;
	prefs.msgbox_max_msg = 200;
	prefs.msgbox_per_del = 50;
	prefs.win_height     = 470;
	prefs.win_width      = -1;
	prefs.win_xpos       = -1;
	prefs.win_ypos       = -1;
#ifdef APP_GNP
	prefs.text_fg_str    = g_strdup("0 0 0");	/* always black */
	prefs.text_bg_str    = g_strdup("65535 65535 65535");	/* 16-bit */
	prefs.text_hlfg_str  = g_strdup("65535 65535 65535");	/* 16-bit */
	prefs.text_hlbg_str  = g_strdup("0 65535 0");	/* blue */
	prefs.text_font_str  = g_strdup(DEFAULT_FONT);
	prefs.text_font      = NULL;
#endif
	prefs_tabpos   = GTK_POS_TOP;

	if ((homedir = getenv("HOME")) == NULL)
		return;

	prefs.appdir = (char *)g_malloc(strlen(homedir) + strlen(PREFS_DIR)+ 3);
	sprintf(prefs.appdir, "%s/%s/", homedir, PREFS_DIR);
	num = mkdir(prefs.appdir, 0777);
	if (num == -1 && errno != ESUCCESS && errno != EEXIST) {
		buf = (char *)g_malloc(strlen(prefs.appdir) + 32);
		sprintf(buf, "prefs_init: could not create '%s'", prefs.appdir);
		perror(buf);
		g_free(buf);
		g_free(prefs.appdir);
		return;
	} 
#if NOT_QUITE
	else if (errno == ESUCCESS) {
		/* hmm, this doesn't work the way i intended to at this
		 * this point because there's no top-level dialog window
		 * associated with it.  consequently, this dialog box just
		 * pops up where-ever it wants.
		 */
		(void)do_dialog_ok(
			"gnotepad+ Initialized", " Created $HOME/.app/ ");
	}
#endif

	/* parse gtkrc specific attributes */
	if (gtkrc)
		g_free(gtkrc);
	num = strlen(prefs.appdir) + strlen(GTKRC) + 1;
	gtkrc = (char *)g_malloc(num);
	g_snprintf(gtkrc, num, "%s%s", prefs.appdir, GTKRC);
	gtk_rc_parse(gtkrc);

	/* now read apprc to get application specific stuff */
	num = strlen(prefs.appdir) + strlen(APPRC) + 1;
	apprc = (char *)g_malloc(num);
	g_snprintf(apprc, num, "%s%s", prefs.appdir, APPRC);
	
	if ((fp = fopen(apprc, "r")) == NULL) {
		if (errno == ENOENT)	/* apprc doesn't exist, so create it */
			prefs_save();	/* using default values */
		else {
			buf = (char *)g_malloc(strlen(apprc) + 32);
			sprintf(buf, "prefs_init: could not open '%s'", apprc);
			perror(buf);
			g_free(buf);
		}
	}

	prefs_read(fp);
#ifdef APP_GNP
	prefs_text_color_update();
	prefs.text_font = (prefs.text_font_str) ?
				gdk_font_load(prefs.text_font_str) : NULL;
#endif

	/*
	 * now that we have the application specific stuff, create a gtk
	 * formattted rc file which contains app settings that may override
	 * anything in gtkrc, and finally, use the settings specified in this
	 * new file.
	 */
	num = strlen(prefs.appdir) + strlen(APPGTKRC) + 1;
	if (appgtkrc)
		g_free(appgtkrc);
	appgtkrc = (char *)g_malloc(num);
	g_snprintf(appgtkrc, num, "%s%s", prefs.appdir, APPGTKRC);
	appgtk_rc_update(appgtkrc);
} /* prefs_init */


/*
 * PUBLIC: prefs_save
 *
 * save application-wide preferences.  basically, scan the app_prefs table and
 * write out the values corresponding to each entry.
 */
void
prefs_save(void)
{
	FILE *fp;
	char *buf;
	prefs_t *pap;

	if ((fp = fopen(apprc, "w")) == NULL) {
		buf = (char *)g_malloc(strlen(apprc) + 32);
		sprintf(buf, "prefs_save: could not open '%s'", apprc);
		perror(buf);
		g_free(buf);
		return;
	}

	(void)file_lock(apprc, fp, TRUE, FALSE, TRUE);
	fprintf(fp,
		"# %s %s initialization file.  "
		"This file is automatically generated.\n"
		"# Data in this file is in no particular order.\n"
		"# Although you could edit this by hand, "
		"it is NOT recommended!\n",
		APP_NAME, APP_VERSION);

	buf = (char *)g_malloc(PREFS_NAME_LEN + MAXPATH);
	buf[0] = '\0';
	pap = app_prefs;
	while (pap->name) {
		if (pap->addr == NULL) {
			printf("prefs_save: '%s' has NULL addr\n", pap->name);
			pap++;
			continue;
		}

		switch (pap->type) {
		case PrefBool:
			g_snprintf(buf, PREFS_NAME_LEN + MAXPATH,
				"%s = %s",
				pap->name,
				prefs_tf(*(long *)(pap->addr), pap->val));
			break;
		case PrefString:
			g_snprintf(buf, PREFS_NAME_LEN + MAXPATH,
				"%s = %s",
				pap->name, *(char **)(pap->addr));
			break;
		case PrefInt:
			g_snprintf(buf, PREFS_NAME_LEN + MAXPATH,
				"%s = %d",
				pap->name, *(int *)(pap->addr));
			break;
		case PrefByte:
			g_snprintf(buf, PREFS_NAME_LEN + MAXPATH,
				"%s = %d",
				pap->name, *(byte *)(pap->addr));
			break;
		case PrefFloat:
			g_snprintf(buf, PREFS_NAME_LEN + MAXPATH,
				"%s = %f",
				pap->name, *(float *)(pap->addr));
			break;
		default:
			printf("prefs_save: ignoring '%s' (unknown type=%d)\n",
				pap->name, pap->type);
			break;
		} /* switch pap->type */

		if (buf[0] != '\0') {
			fprintf(fp, "%s\n", buf);
			GNPDBG_PREFS(("prefs_save: wrote = '%s'\n", buf));
			buf[0] = '\0';
		}

		pap++;
	} /* while pap->name */

	(void)file_unlock(apprc, fp);
	fclose(fp);
	g_free(buf);
} /* prefs_save */


/*
 * PRIVATE: prefs_read
 *
 * actually read the prefs file.  each line's first token is looked up in the
 * app_prefs table.  if it's not in there, then that line is ignored.
 */
static void
prefs_read(FILE *fp)
{
	char *buf, *data, *tok;
	prefs_t *pap;
	bool_t done;

	if (fp == NULL) {
		if ((fp = fopen(apprc, "r")) == NULL) {
			buf = (char *)g_malloc(strlen(apprc) + 32);
			sprintf(buf, "prefs_save: could not open '%s'", apprc);
			perror(buf);
			g_free(buf);
			return;
		}
	}

	(void)file_lock(apprc, fp, FALSE, FALSE, TRUE);
	buf = (char *)g_malloc(PREFS_NAME_LEN + MAXPATH);
	if (fgets(buf, PREFS_NAME_LEN + MAXPATH, fp)) {
		char *p;
		/* XXX - horrible hack for versioning */
		if ((p = strstr(buf, "gnotepad+ 1.0"))) {
			p += strlen("gnotepad+ 1.0") + 1;
			if (!isdigit((int)(*(p+1))))
				*(p+1) = '\0';
			else
				*(p+2) = '\0';
			if (atoi(p) < 2)
				pre102 = TRUE;
		}
	}

	while (fgets(buf, PREFS_NAME_LEN + MAXPATH, fp)) {
		buf[strlen(buf) - 1] = '\0';	/* remove \n */
		if ((tok = strtok(buf, "=")) == NULL)
			continue;

		/* trim any trailing spaces */
		while (isspace((int)tok[strlen(tok) - 1]))
			tok[strlen(tok) - 1] = '\0';

		data = tok + strlen(tok) + 2;
		while (isspace((int)(*data)))	/* skip to actual value */
			data++;

		GNPDBG_PREFS(("prefs_read: tok  = '%s'\n", tok));
		GNPDBG_PREFS(("prefs_read: data = '%s'\n", data));

		/*
		 * TODO - eventually, if app_prefs gets big, this should be a
		 * binary search instead of a linear traversal.
		 */
		done = FALSE;
		pap = app_prefs;
		while (pap->name) {
			if (pap->addr == NULL) {
				printf("prefs_read: '%s' has NULL addr\n",
					pap->name);
				pap++;
				continue;
			}

			if (strncmp(tok, pap->name, strlen(pap->name)) == 0) {
				switch (pap->type) {
				case PrefBool:
					prefs_bool_set(pap, data);
					break;
				case PrefString:
					prefs_string_set(pap, data);
					break;
				case PrefInt:
					prefs_int_set(pap, data);
					break;
				case PrefByte:
					prefs_byte_set(pap, data);
					break;
				case PrefFloat:
					prefs_float_set(pap, data);
					break;
				default:
					printf("prefs_read: ignoring '%s' "
						"(unknown type=%d)\n",
						pap->name, pap->type);
					break;
				}
				done = TRUE;
			} /* if attr match */

			pap++;

		} /* while pap->name */
	} /* while fgets */

	(void)file_unlock(apprc, fp);
	fclose(fp);
	g_free(buf);
} /* prefs_read */


/*
 * PRIVATE: appgtk_rc_update
 *
 * creates an application specific rc file in gtk format.
 */
static void
appgtk_rc_update(gpointer data)
{
#ifdef APP_GNP
	char *appgtkrc = (char *)data;
	FILE *fp;
	char *buf;

	if ((fp = fopen(appgtkrc, "w")) == NULL) {
		buf = (char *)g_malloc(strlen(appgtkrc) + 64);
		sprintf(buf, "appgtk_rc_update: could not open '%s'", appgtkrc);
		perror(buf);
		g_free(buf);
		return;
	}

	(void)file_lock(appgtkrc, fp, TRUE, FALSE, TRUE);

	fprintf(fp, "style \"text\"\n{\n\tfont = \"%s\"\n",
		prefs.text_font_str);
	fprintf(fp, "\tbase[NORMAL]   = { %0.1f, %0.1f, %0.1f }\n", 
		prefs.text_bg_color.red   / COL_VAL_GDK,
		prefs.text_bg_color.green / COL_VAL_GDK,
		prefs.text_bg_color.blue  / COL_VAL_GDK);
	fprintf(fp, "\tbg[NORMAL]     = { %0.1f, %0.1f, %0.1f }\n", 
		prefs.text_bg_color.red   / COL_VAL_GDK,
		prefs.text_bg_color.green / COL_VAL_GDK,
		prefs.text_bg_color.blue  / COL_VAL_GDK);
	fprintf(fp, "\ttext[NORMAL]   = { %0.1f, %0.1f, %0.1f }\n", 
		prefs.text_fg_color.red   / COL_VAL_GDK,
		prefs.text_fg_color.green / COL_VAL_GDK,
		prefs.text_fg_color.blue  / COL_VAL_GDK);
	fprintf(fp, "\tfg[NORMAL]     = { %0.1f, %0.1f, %0.1f }\n", 
		prefs.text_fg_color.red   / COL_VAL_GDK,
		prefs.text_fg_color.green / COL_VAL_GDK,
		prefs.text_fg_color.blue  / COL_VAL_GDK);
	fprintf(fp, "\tbase[SELECTED] = { %0.1f, %0.1f, %0.1f }\n", 
		prefs.text_hlbg_color.red   / COL_VAL_GDK,
		prefs.text_hlbg_color.green / COL_VAL_GDK,
		prefs.text_hlbg_color.blue  / COL_VAL_GDK);
	fprintf(fp, "\tbg[SELECTED]   = { %0.1f, %0.1f, %0.1f }\n", 
		prefs.text_hlbg_color.red   / COL_VAL_GDK,
		prefs.text_hlbg_color.green / COL_VAL_GDK,
		prefs.text_hlbg_color.blue  / COL_VAL_GDK);
	fprintf(fp, "\ttext[SELECTED] = { %0.1f, %0.1f, %0.1f }\n", 
		prefs.text_hlfg_color.red   / COL_VAL_GDK,
		prefs.text_hlfg_color.green / COL_VAL_GDK,
		prefs.text_hlfg_color.blue  / COL_VAL_GDK);
	fprintf(fp, "\tfg[SELECTED]   = { %0.1f, %0.1f, %0.1f }\n", 
		prefs.text_hlfg_color.red   / COL_VAL_GDK,
		prefs.text_hlfg_color.green / COL_VAL_GDK,
		prefs.text_hlfg_color.blue  / COL_VAL_GDK);
	fprintf(fp, "}\nwidget_class \"*GtkText\" style \"text\"\n");

	(void)file_unlock(appgtkrc, fp);
	fclose(fp);
	gtk_rc_parse(appgtkrc);
#endif
} /* appgtk_rc_update */


/*
 * PRIVATE: prefs_string_set
 *
 * set a preference whose setting/value is a "string"
 */
static void
prefs_string_set(prefs_t *pap, char *data)
{
	int len;
	char **string = (char **)(pap->addr);

	if (*string)
		g_free(*string);
	*string = NULL;

	if (data == NULL)
		return;

	if (data[0] == '\0')
		return;

	len = MIN(strlen(data), pap->val);
	*string = (char *)g_malloc(len + 1);
	strncpy(*string, data, len);
	(*string)[len] = '\0';
} /* prefs_string_set */


/*
 * PRIVATE: prefs_byte_set
 *
 * set a preference whose setting/value is an integer
 */
static void
prefs_byte_set(prefs_t *pap, char *data)
{
	byte *num = (byte *)(pap->addr);

	*num = (byte)atoi(data);
	if (*num > pap->max)
		*num = (byte)(pap->max);
	if (*num < pap->min)
		*num = (byte)(pap->min);
} /* prefs_byte_set */


/*
 * PRIVATE: prefs_float_set
 *
 * set a preference whose setting/value is a float
 */
static void
prefs_float_set(prefs_t *pap, char *data)
{
	float *num = (float *)(pap->addr);

	sscanf(data, "%f", num);
	if (*num > pap->max)
		*num = (float)(pap->max);
	if (*num < pap->min)
		*num = (float)(pap->min);
} /* prefs_float_set */


/*
 * PRIVATE: prefs_int_set
 *
 * set a preference whose setting/value is an integer
 */
static void
prefs_int_set(prefs_t *pap, char *data)
{
	int *num = (int *)(pap->addr);

	*num = atoi(data);
	if (*num > pap->max)
		*num = (int)(pap->max);
	if (*num < pap->min)
		*num = (int)(pap->min);
} /* prefs_int_set */


/*
 * PRIVATE: prefs_bool_set
 *
 * set a preference whose setting/value is a boolean.
 * NOTE!!  the convention here is that booleans use bitmasks to toggle their
 * settings.  the variable itself must be stored in a "long".
 */
static void
prefs_bool_set(prefs_t *pap, char *data)
{
	long *opts;

	opts = (long *)(pap->addr);

	if (tolower((int)data[0]) == 't' || data[0] == '1') {
		*opts |= pap->val;
	} else if (tolower((int)data[0]) == 'f' || data[0] == '0') {
		*opts &= ~pap->val;
	}
} /* prefs_bool_set */


/*
 * PRIVATE: prefs_tf
 *
 * returns the string 'true' or 'false'
 */
static char *
prefs_tf(long opts, long mask)
{
	return (opts & mask) ? "true" : "false";
} /* prefs_tf */


/*
 * PRIVATE: prefs_dialog_create
 *
 * creates the preferences window.
 *
 * 1. Appearance
 * 2. Document
 * 3. Window
 * 4. Misc
 */
static void
prefs_dialog_create(win_t *w)
{
	GtkWidget *vbox, *hbox, *tmp;

	/* create top level */
	prefs_win = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(prefs_win), "gnotepad+ Preferences");
	gtk_signal_connect(GTK_OBJECT(prefs_win), "destroy",
		GTK_SIGNAL_FUNC(prefs_win_destroy), NULL);
	gtk_container_border_width(GTK_CONTAINER(prefs_win), 5);

	/* create vbox to hold everything */
	vbox = GTK_DIALOG(prefs_win)->vbox;
	gtk_container_border_width(GTK_CONTAINER(vbox), 5);

	/* create notebook */
	prefs_nb = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(prefs_nb), prefs_tabpos);
	gtk_box_pack_start(GTK_BOX(vbox), prefs_nb, TRUE, TRUE, 0);

	/* create various pages of the notebook */
	prefs_page_appearance_create(prefs_nb);
	prefs_page_document_create(prefs_nb);
	prefs_page_window_create(prefs_nb);
#ifdef APP_GNP
	prefs_page_fonts_colors(prefs_nb);
#endif
	prefs_page_misc_create(prefs_nb);

	/* lastly, the buttons */
#if 0
	/* if we don't use gtk_dialog_new() and just create a regular top level
	   window, we can use the following code snippit to have more control
	   over how the buttons are laid out */
	hbox = gtk_hbutton_box_new();
	gtk_container_border_width(GTK_CONTAINER(hbox), 5);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 5);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
#endif
	hbox = GTK_DIALOG(prefs_win)->action_area;
	(void)misc_button_new_w_label("Save",
				      GTK_SIGNAL_FUNC(prefs_save_cb),
				      w, hbox,
#ifdef GTK_HAVE_FEATURES_1_1_0
				      GTK_RELIEF_NORMAL,
#endif
				      NULL, 0, 0, 0, 0);
	(void)misc_button_new_w_label("Tab Position",
				      GTK_SIGNAL_FUNC(prefs_tab_pos_change),
				      NULL, hbox,
#ifdef GTK_HAVE_FEATURES_1_1_0
				      GTK_RELIEF_NORMAL,
#endif
				      NULL, 0, 0, 0, 0);
	tmp = misc_button_new_w_label("Cancel",
				      GTK_SIGNAL_FUNC(prefs_win_cancel),
				      NULL, hbox,
#ifdef GTK_HAVE_FEATURES_1_1_0
				      GTK_RELIEF_NORMAL,
#endif
				      NULL, 0, 0, 0, 0);
	GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(tmp);

	gtk_widget_show_all(prefs_win);

#ifdef APP_GNP
#ifdef GTK_HAVE_FEATURES_1_1_0
	/* update the prefs window's text widget and insert sample text */
	prefs_update_text_widget_style(prefs_txt);
	gtk_text_set_editable(GTK_TEXT(prefs_txt), FALSE);
	gtk_text_set_word_wrap(GTK_TEXT(prefs_txt), TRUE);
	gtk_widget_realize(prefs_txt);
	gtk_text_insert(GTK_TEXT(prefs_txt),
		NULL, NULL, NULL,
		"This is some sample text.  Note that color changes only take "
		"immediate effect if you are using gtk-1.1.x, not gtk-1.0.x.",
		-1);
#endif
#endif
} /* prefs_dialog_create */


#ifdef APP_GNP
/*
 * PUBLIC: prefs_update_text_widget_style
 *
 * this needs to be called *AFTER* the main prefs window has been drawn and
 * show.  this is because at the time when the text widget was created, nothing
 * was realized, so we cannot muck with the text widget.
 */
void
prefs_update_text_widget_style(GtkWidget *wgt)
{
	GtkStyle *style;

	style = gtk_style_copy(wgt->style);

	sscanf(prefs.text_fg_str, "%hd %hd %hd",
		&prefs.text_fg_color.red, &prefs.text_fg_color.blue, &prefs.text_fg_color.green);
	style->fg[GTK_STATE_NORMAL].red     = prefs.text_fg_color.red;
	style->fg[GTK_STATE_NORMAL].blue    = prefs.text_fg_color.blue;
	style->fg[GTK_STATE_NORMAL].green   = prefs.text_fg_color.green;
	style->text[GTK_STATE_NORMAL].red   = prefs.text_fg_color.red;
	style->text[GTK_STATE_NORMAL].blue  = prefs.text_fg_color.blue;
	style->text[GTK_STATE_NORMAL].green = prefs.text_fg_color.green;

	sscanf(prefs.text_bg_str, "%hd %hd %hd",
		&prefs.text_bg_color.red, &prefs.text_bg_color.blue, &prefs.text_bg_color.green);
	style->bg[GTK_STATE_NORMAL].red     = prefs.text_bg_color.red;
	style->bg[GTK_STATE_NORMAL].blue    = prefs.text_bg_color.blue;
	style->bg[GTK_STATE_NORMAL].green   = prefs.text_bg_color.green;
	style->base[GTK_STATE_NORMAL].red   = prefs.text_bg_color.red;
	style->base[GTK_STATE_NORMAL].blue  = prefs.text_bg_color.blue;
	style->base[GTK_STATE_NORMAL].green = prefs.text_bg_color.green;

	sscanf(prefs.text_hlfg_str, "%hd %hd %hd", &prefs.text_hlfg_color.red,
		&prefs.text_hlfg_color.blue, &prefs.text_hlfg_color.green);
	style->fg[GTK_STATE_SELECTED].red     = prefs.text_hlfg_color.red;
	style->fg[GTK_STATE_SELECTED].blue    = prefs.text_hlfg_color.blue;
	style->fg[GTK_STATE_SELECTED].green   = prefs.text_hlfg_color.green;
	style->text[GTK_STATE_SELECTED].red   = prefs.text_hlfg_color.red;
	style->text[GTK_STATE_SELECTED].blue  = prefs.text_hlfg_color.blue;
	style->text[GTK_STATE_SELECTED].green = prefs.text_hlfg_color.green;

	sscanf(prefs.text_hlbg_str, "%hd %hd %hd", &prefs.text_hlbg_color.red,
		&prefs.text_hlbg_color.blue, &prefs.text_hlbg_color.green);
	style->bg[GTK_STATE_SELECTED].red     = prefs.text_hlbg_color.red;
	style->bg[GTK_STATE_SELECTED].blue    = prefs.text_hlbg_color.blue;
	style->bg[GTK_STATE_SELECTED].green   = prefs.text_hlbg_color.green;
	style->base[GTK_STATE_SELECTED].red   = prefs.text_hlbg_color.red;
	style->base[GTK_STATE_SELECTED].blue  = prefs.text_hlbg_color.blue;
	style->base[GTK_STATE_SELECTED].green = prefs.text_hlbg_color.green;

	gdk_font_unref(style->font);
	gtk_widget_push_style(style);
	if ((style->font = gdk_font_load(prefs.text_font_str)) == NULL)
		style->font = gdk_font_load("7x13");
	gdk_font_unref(prefs.text_font);
	prefs.text_font = style->font;
#ifdef GTK_HAVE_FEATURES_1_1_0
	/* XXX - there seems to be a problem in gtk-1.0.x */
	gtk_widget_set_style(wgt, style);
#endif
	gtk_widget_pop_style();
} /* prefs_update_text_widget_style */
#endif


/*
 * PRIVATE: prefs_page_window_create
 *
 * creates the page titled "Window"
 *
 * 3. Window
 *	save window height on exit	(check button)
 *	default window height		(spin button)
 *	save window width on exit	(check button)
 *	default window width		(spin button)
 *
 *	save window position on exit	(check button)
 *	default window x-position	(spin button)
 *	default window y-position	(spin button)
 */
static void
prefs_page_window_create(GtkWidget *nb)
{
	GtkWidget *frame, *vbox, *tmp, *table;

	prefs_box_in_frame("Window Settings", 10, &frame, &vbox, TRUE);

	table = gtk_table_new(4, 2, FALSE);
	gtk_table_set_col_spacing(GTK_TABLE(table), 0, 10);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

	/* save window height on exit */
	(void)prefs_check_button_with_label_create(
			vbox,
			" Save Window height on exit ",
			IS_SAVE_WIN_HEIGHT(),
			"save_win_height",
			table, 0, 2, 0, 1);

	/* default window height */
	(void)prefs_spin_button_with_label_create(
			vbox,
			" Default height (-1 for auto) ",
			(float)prefs.win_height, -1.0, 2048.0, 1.0, 1.0, 0,
			"win_height",
			table, 0, 1, 1, 2, 1, 2, 1, 2);

	/* save window width on exit */
	(void)prefs_check_button_with_label_create(
			vbox,
			" Save Window width on exit ",
			IS_SAVE_WIN_WIDTH(),
			"save_win_width",
			table, 0, 2, 2, 3);

	/* default window width */
	(void)prefs_spin_button_with_label_create(
			vbox,
			" Default width (-1 for auto) ",
			(float)prefs.win_width, -1.0, 2048.0, 1.0, 1.0, 0,
			"win_width",
			table, 0, 1, 3, 4, 1, 2, 3, 4);

	tmp = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 0);

	/* save window position on exit */
	(void)prefs_check_button_with_label_create(
			vbox,
			" Save Window Position on Exit ",
			IS_SAVE_WIN_POS(),
			"save_win_pos",
			NULL, -1, -1, -1, -1);

	table = gtk_table_new(2, 2, FALSE);
	gtk_table_set_col_spacing(GTK_TABLE(table), 0, 10);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

	/* default window xpos */
	(void)prefs_spin_button_with_label_create(
			vbox,
			" Default x-pos (-1 for auto) ",
			(float)prefs.win_xpos, -1.0, 2048.0, 1.0, 1.0, 0,
			"win_xpos",
			table, 0, 1, 0, 1, 1, 2, 0, 1);

	/* default window ypos */
	(void)prefs_spin_button_with_label_create(
			vbox,
			" Default y-pos (-1 for auto) ",
			(float)prefs.win_ypos, -1.0, 2048.0, 1.0, 1.0, 0,
			"win_ypos",
			table, 0, 1, 1, 2, 1, 2, 1, 2);

	/* stuff into notebook */
	tmp = gtk_label_new("Window");
	gtk_notebook_append_page(GTK_NOTEBOOK(nb), frame, tmp);
} /* prefs_page_window_create */


/*
 * PRIVATE: prefs_page_document_create
 *
 * creates the notebook page titled "Document"
 *
 * 2. Document
 *	autosave		(spin button)
 *	max num of recent docs	(spin button)
 *	wordwrap on/off		(check button)
 *	fonts			(popup must return string)
 *	colors			(popup must return string)
 */
static void
prefs_page_document_create(GtkWidget *nb)
{
	GtkWidget *frame, *vbox, *tmp, *table;

	prefs_box_in_frame("Document Settings", 10, &frame, &vbox, TRUE);

	table = gtk_table_new(2, 2, FALSE);
	gtk_table_set_col_spacing(GTK_TABLE(table), 0, 10);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

#ifdef APP_GNP
	/* autosave spin button */
	(void)prefs_spin_button_with_label_create(
			vbox,
			" Autosave minutes (0 disables) ",
			(float)prefs.autosave, 0.0, 1440.0, 1.0, 1.0, 0.0,
			"autosave_timeout",
			table, 0, 1, 0, 1, 1, 2, 0, 1);
#endif

	/* max num of recent doc */
	(void)prefs_spin_button_with_label_create(
			vbox,
			" Maximum # of recent documents ",
			(float)prefs.maxrecent, 0.0, 255.0, 1.0, 1.0, 0.0,
			"max_num_recent",
			table, 0, 1, 1, 2, 1, 2, 1, 2);

	tmp = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 0);

#ifdef APP_GNP
	/* word wrap */
	(void)prefs_check_button_with_label_create(
			vbox,
			"Use wordwrap",
			IS_USE_WORDWRAP(),
			"use_wordwrap",
			NULL, -1, -1, -1, -1);
#endif

	tmp = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 0);

	/* stuff into notebook */
	tmp = gtk_label_new("Document");
	gtk_notebook_append_page(GTK_NOTEBOOK(nb), frame, tmp);
} /* prefs_page_document_create */


#ifdef APP_GNP
/*
 * PRIVATE: prefs_page_fonts_colors
 *
 * creates the notebook page titled "Fonts and Colors"
 *
 * Fonts and Colors
 *	Font Selection		(push button w/label)
 *	Text Foreground		(push button w/label)
 *	Text Background		(push button w/label)
 *	Highlight Foreground	(push button w/label)
 *	Highlight Background	(push button w/label)
 */
static void
prefs_page_fonts_colors(GtkWidget *nb)
{
	GtkWidget *frame, *vbox, *tmp, *hbox;

	prefs_box_in_frame("Fonts and Colors", 10, &frame, &hbox, FALSE);
	/* stuff into notebook */
	tmp = gtk_label_new("Fonts and Colors");
	gtk_notebook_append_page(GTK_NOTEBOOK(nb), frame, tmp);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 10);

	tmp = misc_button_new_w_label("Font Selection",
				      GTK_SIGNAL_FUNC(prefs_font_sel_cb),
				      NULL, vbox,
#ifdef GTK_HAVE_FEATURES_1_1_0
				      GTK_RELIEF_NONE,
#endif
				      NULL, 0, 1, 0, 1);
	prefs_wgtopt_list_add(tmp, GenericType, NULL, "text_font");

	tmp = misc_button_new_w_label("Text Foreground",
				      GTK_SIGNAL_FUNC(prefs_text_fg_cb),
				      NULL, vbox,
#ifdef GTK_HAVE_FEATURES_1_1_0
				      GTK_RELIEF_NONE,
#endif
				      NULL, 0, 1, 1, 2);
	prefs_wgtopt_list_add(tmp, GenericType, NULL, "text_fg_color");

	tmp = misc_button_new_w_label("Text Background",
				      GTK_SIGNAL_FUNC(prefs_text_bg_cb),
				      NULL, vbox,
#ifdef GTK_HAVE_FEATURES_1_1_0
				      GTK_RELIEF_NONE,
#endif
				      NULL, 0, 1, 2, 3);
	prefs_wgtopt_list_add(tmp, GenericType, NULL, "text_bg_color");

	tmp = misc_button_new_w_label("Highlighted Foreground",
				      GTK_SIGNAL_FUNC(prefs_text_hlfg_cb),
				      NULL, vbox,
#ifdef GTK_HAVE_FEATURES_1_1_0
				      GTK_RELIEF_NONE,
#endif
				      NULL, 0, 1, 3, 4);
	prefs_wgtopt_list_add(tmp, GenericType, NULL, "text_hlfg_color");

	tmp = misc_button_new_w_label("Highlighted Background",
				      GTK_SIGNAL_FUNC(prefs_text_hlbg_cb),
				      NULL, vbox,
#ifdef GTK_HAVE_FEATURES_1_1_0
				      GTK_RELIEF_NONE,
#endif
				      NULL, 0, 1, 4, 5);
	prefs_wgtopt_list_add(tmp, GenericType, NULL, "text_hlbg_color");

#ifdef GTK_HAVE_FEATURES_1_1_0
	frame = gtk_frame_new("Sample Output");
	gtk_container_border_width(GTK_CONTAINER(frame), 10);

	prefs_txt = gtk_text_new(NULL, NULL);
	gtk_container_add(GTK_CONTAINER(frame), prefs_txt);
	gtk_box_pack_start(GTK_BOX(hbox), frame, TRUE, TRUE, 10);
#else
	tmp = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 10);
	tmp = gtk_label_new(
		" NOTE: for gtk-1.0.x, font and color changes \n"
		" take effect the next time you run gnotepad+ ");
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 10);
#endif
} /* prefs_page_fonts_colors */


/*
 * PRIVATE: prefs_text_fg_cb
 *
 * callback when button pushed is for changing the text foreground color
 */
static void
prefs_text_fg_cb(GtkWidget *wgt, gpointer cbdata)
{
	prefs_color_sel_create(Foreground);
} /* prefs_text_fg_cb */


/*
 * PRIVATE: prefs_text_bg_cb
 *
 * callback when button pushed is for changing the text background color
 */
static void
prefs_text_bg_cb(GtkWidget *wgt, gpointer cbdata)
{
	prefs_color_sel_create(Background);
} /* prefs_text_bg_cb */


/*
 * PRIVATE: prefs_text_hlfg_cb
 *
 * callback when button pushed is for changing the text foreground color
 */
static void
prefs_text_hlfg_cb(GtkWidget *wgt, gpointer cbdata)
{
	prefs_color_sel_create(HighlightFg);
} /* prefs_text_hlfg_cb */


/*
 * PRIVATE: prefs_text_hlbg_cb
 *
 * callback when button pushed is for changing the text background color
 */
static void
prefs_text_hlbg_cb(GtkWidget *wgt, gpointer cbdata)
{
	prefs_color_sel_create(HighlightBg);
} /* prefs_text_bg_hlcb */


/*
 * PRIVATE: prefs_color_sel_create
 *
 * creates the color selection dialog and sets up the callbacks.
 * some of this code was taken from testgtk.c
 *
 * rgb_str: points to either text_fg_str or text_bg_str.
 */
static void
prefs_color_sel_create(which_text_t which)
{
	gdouble color[4];
	GtkWidget *colsel;
	prefs_color_sel_t *pcsp;
	char *which_txt[] = {
		"Text Foreground Color Selection",
		"Text Background Color Selection",
		"Text Hightlight Foreground Color Selection",
		"Text Hightlight Background Color Selection"
	};


	colsel = gtk_color_selection_dialog_new(which_txt[which]);

	pcsp = (prefs_color_sel_t *)g_malloc(sizeof(prefs_color_sel_t));
	pcsp->cs = colsel;
	pcsp->which = which;

	switch (which) {
	case Foreground:
		sscanf(prefs.text_fg_str, "%lf %lf %lf",
			&(color[0]), &(color[2]), &(color[1]));
		break;
	case Background:
		sscanf(prefs.text_bg_str, "%lf %lf %lf",
			&(color[0]), &(color[2]), &(color[1]));
		break;
	case HighlightFg:
		sscanf(prefs.text_hlfg_str, "%lf %lf %lf",
			&(color[0]), &(color[2]), &(color[1]));
		break;
	case HighlightBg:
		sscanf(prefs.text_hlbg_str, "%lf %lf %lf",
			&(color[0]), &(color[2]), &(color[1]));
		break;
	}
	color[0] /= COL_VAL_GDK;
	color[1] /= COL_VAL_GDK;
	color[2] /= COL_VAL_GDK;
	gtk_color_selection_set_color(GTK_COLOR_SELECTION(
		GTK_COLOR_SELECTION_DIALOG(colsel)->colorsel), color);

	gtk_color_selection_set_opacity(GTK_COLOR_SELECTION(
		GTK_COLOR_SELECTION_DIALOG(colsel)->colorsel), FALSE);

	gtk_color_selection_set_update_policy(GTK_COLOR_SELECTION(
		GTK_COLOR_SELECTION_DIALOG(colsel)->colorsel),
		GTK_UPDATE_CONTINUOUS);

	gtk_signal_connect(GTK_OBJECT(colsel), "destroy",
		GTK_SIGNAL_FUNC(prefs_color_sel_destroy), pcsp);

	gtk_signal_connect(GTK_OBJECT(
		GTK_COLOR_SELECTION_DIALOG(colsel)->colorsel), "color_changed",
		GTK_SIGNAL_FUNC(prefs_color_sel_changed), colsel);

	gtk_signal_connect(GTK_OBJECT(
		GTK_COLOR_SELECTION_DIALOG(colsel)->ok_button), "clicked",
		GTK_SIGNAL_FUNC(prefs_color_sel_ok), pcsp);

	gtk_signal_connect(GTK_OBJECT(
		GTK_COLOR_SELECTION_DIALOG(colsel)->cancel_button), "clicked",
		GTK_SIGNAL_FUNC(prefs_color_sel_close), pcsp);

	gtk_widget_show(colsel);
} /* prefs_color_sel_create */


/*
 * PRIVATE: prefs_text_color_update
 *
 * reads the RGB string values and converts them into what GDK expects.
 */
void
prefs_text_color_update(void)
{
	GdkColormap *gdkcmap;

	gdkcmap = gdk_colormap_get_system();

	sscanf(prefs.text_fg_str, "%hu %hu %hu", &prefs.text_fg_color.red,
						 &prefs.text_fg_color.blue,
						 &prefs.text_fg_color.green);
	GNPDBG_PREFS(("prefs_text_color_update: FG pixel %lu, "
		      "red %u, blue %u, green %u\n",
		      prefs.text_fg_color.pixel, prefs.text_fg_color.red,
		      prefs.text_fg_color.blue, prefs.text_fg_color.green));

	if (pre102) {
		g_free(prefs.text_bg_str);
		prefs.text_bg_str = g_strdup("65535 65535 65535");
	}
	sscanf(prefs.text_bg_str, "%hu %hu %hu", &prefs.text_bg_color.red,
						 &prefs.text_bg_color.blue,
						 &prefs.text_bg_color.green);
	GNPDBG_PREFS(("prefs_text_color_update: BG pixel %lu, "
		      "red %u, blue %u, green %u\n",
		      prefs.text_bg_color.pixel, prefs.text_bg_color.red,
		      prefs.text_bg_color.blue, prefs.text_bg_color.green));

	sscanf(prefs.text_hlfg_str, "%hu %hu %hu",&prefs.text_hlfg_color.red,
						  &prefs.text_hlfg_color.blue,
						  &prefs.text_hlfg_color.green);
	sscanf(prefs.text_hlbg_str, "%hu %hu %hu",&prefs.text_hlbg_color.red,
						  &prefs.text_hlbg_color.blue,
						  &prefs.text_hlbg_color.green);

	if (!gdk_color_alloc(gdkcmap, &prefs.text_fg_color))
		g_error("prefs_text_color_update: couldn't alloc fg color");
	if (!gdk_color_alloc(gdkcmap, &prefs.text_bg_color))
		g_error("prefs_text_color_update: couldn't alloc bg color");
	if (!gdk_color_alloc(gdkcmap, &prefs.text_hlfg_color))
		g_error("prefs_text_color_update: couldn't alloc hlfg color");
	if (!gdk_color_alloc(gdkcmap, &prefs.text_hlbg_color))
		g_error("prefs_text_color_update: couldn't alloc hlbg color");
} /* prefs_text_color_update */


/*
 * PRIVATE: prefs_color_sel_destroy
 *
 * callback for the "destroy" event for the color selection dialog
 */
static void
prefs_color_sel_destroy(GtkWidget *wgt, gpointer cbdata)
{
	prefs_color_sel_t *pcsp = (prefs_color_sel_t *)cbdata;

	if (pcsp) {
		g_free(pcsp);
		pcsp = NULL;
	}
} /* prefs_color_sel_destroy */


/*
 * PRIVATE: prefs_color_sel_close
 *
 * callback for the "Close" button from the color selection dialog
 */
static void
prefs_color_sel_close(GtkWidget *wgt, gpointer cbdata)
{
	prefs_color_sel_t *pcsp = (prefs_color_sel_t *)cbdata;

	g_assert(pcsp != NULL);
	gtk_widget_destroy(pcsp->cs);
} /* prefs_color_sel_close */


/*
 * PRIVATE: prefs_color_sel_ok
 *
 * callback for the "Ok" button in the color selection dialog.
 * gets the color from the color selection dialog, then converts it into a
 * character string containing RGB values.
 */
static void
prefs_color_sel_ok(GtkWidget *wgt, gpointer cbdata)
{
	prefs_color_sel_t *pcsp = (prefs_color_sel_t *)cbdata;
	GtkColorSelection *colorsel;
	gdouble color[4];
	char buf[MAX_RGB_STR];

	g_assert(pcsp != NULL);
	colorsel = GTK_COLOR_SELECTION(
			GTK_COLOR_SELECTION_DIALOG(pcsp->cs)->colorsel);
	gtk_color_selection_get_color(colorsel, color);

	/*
	 * NOTE!  gtk_color_selection_get_color() returns the color in the
	 * order red-blue-green, instead of red-green-blue.
	 */
	GNPDBG_PREFS(("prefs_color_sel_ok: colors = (red %f) (green %f) "
		      "(blue %f)\n", color[0], color[2], color[1]));
	g_snprintf(buf, MAX_RGB_STR, "%u %u %u",
		   (unsigned)(color[0] * COL_VAL_GDK),
		   (unsigned)(color[2] * COL_VAL_GDK),
		   (unsigned)(color[1] * COL_VAL_GDK));
	GNPDBG_PREFS(("prefs_color_sel_ok: buf = '%s'\n", buf));

	switch (pcsp->which) {
	case Foreground:
		if (prefs.text_fg_str)
			g_free(prefs.text_fg_str);
		prefs.text_fg_str = g_strdup(buf);
		GNPDBG_PREFS(("prefs_color_sel_ok: text_fg_color = '%s'\n",
			      prefs.text_fg_str));
		break;
	case Background:
		if (prefs.text_bg_str)
			g_free(prefs.text_bg_str);
		prefs.text_bg_str = g_strdup(buf);
		GNPDBG_PREFS(("prefs_color_sel_ok: text_bg_color = '%s'\n",
			      prefs.text_bg_str));
		break;
	case HighlightFg:
		if (prefs.text_hlfg_str)
			g_free(prefs.text_hlfg_str);
		prefs.text_hlfg_str = g_strdup(buf);
		GNPDBG_PREFS(("prefs_color_sel_ok: text_hlfg_color = '%s'\n",
			      prefs.text_hlfg_str));
		break;
	case HighlightBg:
		if (prefs.text_hlbg_str)
			g_free(prefs.text_hlbg_str);
		prefs.text_hlbg_str = g_strdup(buf);
		GNPDBG_PREFS(("prefs_color_sel_ok: text_hlbg_color = '%s'\n",
			      prefs.text_hlbg_str));
		break;
	} /* switch pcsp->which */

	gtk_widget_destroy(pcsp->cs);
#ifdef GTK_HAVE_FEATURES_1_1_0
	prefs_update_text_widget_style(prefs_txt);
#endif
} /* prefs_color_sel_ok */


/*
 * PRIVATE: prefs_color_sel_changed
 *
 * callback if color has changed in the color selection dialog
 */
static void
prefs_color_sel_changed(GtkWidget *wgt, gpointer cbdata)
{
	GtkColorSelectionDialog *cs = (GtkColorSelectionDialog *)cbdata;
	GtkColorSelection *colorsel;
	gdouble color[4];

	colorsel = GTK_COLOR_SELECTION(cs->colorsel);
	gtk_color_selection_get_color(colorsel, color);
} /* prefs_color_sel_changed */


/*
 * PRIVATE: prefs_font_sel_cb
 *
 * creates font selection dialog box and sets up callbacks.
 */
static void
prefs_font_sel_cb(GtkWidget *wgt, gpointer cbdata)
{
	GtkWidget *tmp;

	tmp = gtk_font_selection_dialog_new("Font Selection");
	gtk_signal_connect(
		GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(tmp)->ok_button),
		"clicked",
		GTK_SIGNAL_FUNC(prefs_font_sel_ok),
		tmp);

	gtk_signal_connect(
		GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(tmp)->cancel_button),
		"clicked",
		GTK_SIGNAL_FUNC(prefs_font_sel_cancel),
		tmp);

	gtk_signal_connect(
		GTK_OBJECT(tmp),
		"destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroy),
		tmp);

	(void)gtk_font_selection_dialog_set_font_name(
		GTK_FONT_SELECTION_DIALOG(tmp), prefs.text_font_str);

	gtk_widget_show(tmp);
} /* prefs_font_sel_cb */


/*
 * PRIVATE: prefs_font_sel_ok
 *
 * callback for "Ok" button on font selection dialog.
 * gets the font string from the dialog, and calls gdk_font_load() to load it
 * from the system.
 */
static void
prefs_font_sel_ok(GtkWidget *wgt, gpointer cbdata)
{
	GtkFontSelectionDialog *fsd = (GtkFontSelectionDialog *)cbdata;
	char *newfont;

	newfont = g_strdup(gtk_font_selection_dialog_get_font_name(fsd));
	if (newfont == NULL) {
		GNPDBG_PREFS(("prefs_font_sel_ok: no font chosen\n"));
		return;
	}

	if (prefs.text_font_str)
		g_free(prefs.text_font_str);
	prefs.text_font_str = g_strdup(newfont);
	GNPDBG_PREFS(("prefs_font_sel_ok: text_font_str '%s'\n",
		       prefs.text_font_str));
	if ((prefs.text_font = gdk_font_load(prefs.text_font_str)) == NULL) {
		if ((prefs.text_font = gdk_font_load(DEFAULT_FONT)) == NULL)
			prefs.text_font = gdk_font_load("7x13");
	}

	gtk_widget_destroy(GTK_WIDGET(fsd));
#ifdef GTK_HAVE_FEATURES_1_1_0
	prefs_update_text_widget_style(prefs_txt);
#endif
} /* prefs_font_sel_ok */


/*
 * PRIVATE: prefs_font_sel_cancel
 *
 * callback for "Cancel" button on font selection dialog
 */
static void
prefs_font_sel_cancel(GtkWidget *wgt, gpointer cbdata)
{
	GtkFontSelectionDialog *fsd = (GtkFontSelectionDialog *)cbdata;

	gtk_widget_destroy(GTK_WIDGET(fsd));
} /* prefs_font_sel_cancel */
#endif	/* APP_GNP */


/*
 * PRIVATE: prefs_frame_msgbar_create
 *
 * creates the frame to contain options for the message bar
 */
static void
prefs_frame_msgbar_create(GtkWidget *parent)
{
	GtkWidget *frame, *tmp, *vbox;

	prefs_box_in_frame("Message Bar", 0, &frame, &vbox, TRUE);

	/* 'show msg bar' check box */
	tmp = prefs_check_button_with_label_create(
			vbox,
			"Show msg bar",
			IS_SHOW_MSGBAR(),
			"show_message_bar",
			NULL, -1, -1, -1, -1);

	/* stuff into parent */
	gtk_box_pack_start(GTK_BOX(parent), frame, TRUE, TRUE, 0);
} /* prefs_frame_msgbar_create */


/*
 * PRIVATE: prefs_frame_doctab_create
 *
 * creates the frame to contain options for the document tabs
 */
static void
prefs_frame_doctab_create(GtkWidget *parent)
{
	GtkWidget *frame, *vbox;

	g_list_free(tabpos_list);
	tabpos_list = NULL;
	tabpos_list = g_list_prepend(tabpos_list, "Bottom");
	tabpos_list = g_list_prepend(tabpos_list, "Top");
	tabpos_list = g_list_prepend(tabpos_list, "Right");
	tabpos_list = g_list_prepend(tabpos_list, "Left");

	prefs_box_in_frame("Document Tabs", 0, &frame, &vbox, TRUE);

	/* 'show doc tabs' check box */
	(void)prefs_check_button_with_label_create(
			vbox,
			"Show doc tabs",
			IS_SHOW_TABS(),
			"show_doc_tabs",
			NULL, -1, -1, -1, -1);

	/* tab position */
	(void)prefs_pulldown_with_label_create(
			vbox,
			" Tab position ",
			tabpos_list,
			"doc_tab_position"
			);

	/* stuff into parent */
	gtk_box_pack_start(GTK_BOX(parent), frame, TRUE, TRUE, 0);
} /* prefs_frame_doctab_create */


/*
 * PRIVATE: prefs_page_appearance_create
 *
 * creates the notebook page titled "Appearance"
 *
 * 1. Appearance
 *	toolbar on/off		(check button)
 *	toolbar style		(radio buttons)
 *	tooltips on/off		(check button)
 *	doc tabs on/off		(check button)
 *	doc tabs position	(pulldown combo entry text)
 *	msgbar on/off		(check button)
 */
static void
prefs_page_appearance_create(GtkWidget *nb)
{
	GtkWidget *frame, *vbox, *tmp, *hbox;

	prefs_box_in_frame("Appearance Settings", 10, &frame, &vbox, TRUE);

	prefs_frame_toolbar_create(vbox);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);

	prefs_frame_doctab_create(hbox);
	prefs_frame_msgbar_create(hbox);

	/* stuff into notebook */
	tmp = gtk_label_new("Appearance");
	gtk_notebook_append_page(GTK_NOTEBOOK(nb), frame, tmp);
} /* prefs_page_appearance_create */


/*
 * PRIVATE: prefs_frame_toolbar_create
 *
 * creates the frame to contain options for the toolbar
 */
static void
prefs_frame_toolbar_create(GtkWidget *parent)
{
	GtkWidget *frame, *hbox, *vbox, *radio, *tbvbox;

	prefs_box_in_frame("Toolbar Settings", 0, &frame, &tbvbox, TRUE);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(tbvbox), hbox, FALSE, FALSE, 0);

	/* put first two checkboxes into one vbox */
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);

	/* 'show toolbar' check box */
	(void)prefs_check_button_with_label_create(
			vbox,
			"Show Toolbar",
			IS_SHOW_TOOLBAR(),
			"show_toolbar",
			NULL, -1, -1, -1, -1);

	/* 'show tooltips' check box */
	(void)prefs_check_button_with_label_create(
			vbox,
			"Show Tooltips",
			IS_SHOW_TOOLTIPS(),
			"show_tooltips",
			NULL, -1, -1, -1, -1);

#ifdef GTK_HAVE_FEATURES_1_1_0
	/* 'raised buttons' check box */
	(void)prefs_check_button_with_label_create(
			vbox,
			"Raised Buttons",
			IS_TOOLBAR_RAISED(),
			"toolbar_raised",
			NULL, -1, -1, -1, -1);

#endif

	/* now another vbox into the hbox */
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);

	/* radio buttons for toolbar style */
	radio = prefs_radio_button_with_label_create(
			vbox,
			NULL,
			"Pictures and Text",
			IS_TEXT_PIC_TOOLBAR(),
			"toolbar_pictext");
	(void)prefs_radio_button_with_label_create(
			vbox,
			radio,
			"Pictures Only",
			IS_PIC_TOOLBAR(),
			"toolbar_piconly");
	(void)prefs_radio_button_with_label_create(
			vbox,
			radio,
			"Text Only",
			IS_TEXT_TOOLBAR(),
			"toolbar_textonly");
#ifdef APP_GNP
	gtk_box_pack_start(GTK_BOX(tbvbox), gtk_hseparator_new(),
			   FALSE, FALSE, 0);

	/* 'show html toolbar' check box */
	(void)prefs_check_button_with_label_create(
			tbvbox,
			"Show Html Toolbar",
			IS_SHOW_HTML_TOOLBAR(),
			"show_html_toolbar",
			NULL, -1, -1, -1, -1);
#endif
	/* stuff into parent */
	gtk_box_pack_start(GTK_BOX(parent), frame, FALSE, FALSE, 0);
} /* prefs_frame_toolbar_create */


/*
 * PRIVATE: prefs_page_misc_create
 *
 * creates the notebook page titled "Miscellaneous"
 *
 * 4. Misc
 *	tmpdir				(text entry string)
 *	printcmd			(text entry string)
 *	msgbox on/off			(check button)
 *	max # msgs			(spin button w/ text entry)
 *	msgbox truncate percentage	(spin button w/ text entry)
 */
static void
prefs_page_misc_create(GtkWidget *nb)
{
	GtkWidget *frame, *vbox, *tmp, *table;

	prefs_box_in_frame("Miscellaneous", 10, &frame, &vbox, TRUE);

	table = gtk_table_new(1, 2, FALSE);
	gtk_table_set_col_spacing(GTK_TABLE(table), 0, 10);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

#ifdef APP_GNP
	/* print cmd */
	(void)prefs_entry_with_label_create(
			vbox, " Custom print command ", prefs.printcmd,
			"print_cmd", table, 0, 1, 0, 1, 1, 2, 0, 1);
#endif

	/* tmp dir */
	(void)prefs_entry_with_label_create(
			vbox, " Temp directory ", prefs.tmpdir,
			"tmp_directory", table, 0, 1, 1, 2, 1, 2, 1, 2);

	tmp = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 0);

	prefs_frame_msgbox_create(vbox);

	/* stuff into notebook */
	tmp = gtk_label_new("Miscellaneous");
	gtk_notebook_append_page(GTK_NOTEBOOK(nb), frame, tmp);
} /* prefs_page_misc_create */


/*
 * PRIVATE: prefs_frame_msgbox_create
 *
 * creates the frame containing the options for the msgbox
 *
 *	msgbox on/off			(check button)
 *	max # msgs			(spin button w/ text entry)
 *	msgbox truncate percentage	(spin button w/ text entry)
 */
static void
prefs_frame_msgbox_create(GtkWidget *parent)
{
	GtkWidget *frame, *vbox, *table;

	prefs_box_in_frame("Message Box", 0, &frame, &vbox, TRUE);

	table = gtk_table_new(1, 2, FALSE);
	gtk_table_set_col_spacing(GTK_TABLE(table), 0, 10);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

	/* use msg box */
	(void)prefs_check_button_with_label_create(
			vbox,
			"Use message box",
			IS_USE_MSGBOX(),
			"use_msgbox",
			NULL, -1, -1, -1, -1);

	/* msgbox max msgs */
	(void)prefs_spin_button_with_label_create(
			vbox,
			" Max # of messages ",
			(float)prefs.msgbox_max_msg, 0.0, 1024.0, 1.0, 1.0, 0,
			"msgbox_max_msg",
			table, 0, 1, 0, 1, 1, 2, 0, 1);

	/* msgbox del msgs */
	(void)prefs_spin_button_with_label_create(
			vbox,
			" % msgs to delete ",
			(float)prefs.msgbox_per_del, 0.0, 100.0, 1.0, 1.0, 0,
			"msgbox_per_del",
			table, 0, 1, 1, 2, 1, 2, 1, 2);

	/* stuff into parent */
	gtk_box_pack_start(GTK_BOX(parent), frame, FALSE, FALSE, 0);
} /* prefs_frame_msgbox_create */


/*
 * PRIVATE: prefs_pulldown_with_label_create
 *
 * convenience routine for creating combo widget (pulldown widget with text)
 */
static GtkWidget *
prefs_pulldown_with_label_create(
	GtkWidget *parent,
	char *labeltext,
	GList *itemlist,
	char *prefname)
{
	GtkWidget *hbox, *tmp;

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), hbox, FALSE, FALSE, 0);
	tmp = gtk_label_new(labeltext);
	gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 0);

	tmp = gtk_combo_new();
	gtk_combo_set_popdown_strings(GTK_COMBO(tmp), itemlist);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(tmp)->entry),
		(char *)g_list_nth_data(itemlist, prefs.tabpos));
	gtk_widget_set_usize(GTK_COMBO(tmp)->entry, 65, 0);
	gtk_editable_select_region(GTK_EDITABLE(GTK_COMBO(tmp)->entry), 0, -1);
	gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 0);
	prefs_wgtopt_list_add(GTK_WIDGET(GTK_EDITABLE(GTK_COMBO(tmp)->entry)),
		ComboType, itemlist, prefname);

	return tmp;
} /* prefs_pulldown_with_label_create */


/*
 * PRIVATE: prefs_check_button_with_label_create
 *
 * convenience routine for creating a check button and label, and then packing
 * them into a parent (box).
 */
static GtkWidget *
prefs_check_button_with_label_create(
	GtkWidget *parent,
	char *labeltext,
	long expr,
	char *prefname,
	GtkWidget *table,
	int lft1, int rht1, int top1, int bot1)
{
	GtkWidget *tmp;

	tmp = gtk_check_button_new_with_label(labeltext);
	if (table) {
		gtk_table_attach_defaults(GTK_TABLE(table),
					  tmp, lft1, rht1, top1, bot1);
	} else {
		gtk_box_pack_start(GTK_BOX(parent), tmp, FALSE, FALSE, 0);
	}
	if (expr)
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(tmp), TRUE);
	prefs_wgtopt_list_add(tmp, ButtonType, NULL, prefname);

	return tmp;
} /* prefs_check_button_with_label_create */


/*
 * PRIVATE: prefs_entry_with_label_create
 *
 * convenience routine for creating a text entry widget and label
 */
static GtkWidget *
prefs_entry_with_label_create(
	GtkWidget *parent,
	char *labeltext,
	char *entrytext,
	char *prefname,
	GtkWidget *table,
	int lft1, int rht1, int top1, int bot1,
	int lft2, int rht2, int top2, int bot2)
{
	GtkWidget *tmp, *hbox = NULL;

	tmp = gtk_label_new(labeltext);
	if (table) {
		gtk_misc_set_alignment(GTK_MISC(tmp), 0, 0.5);
		gtk_table_attach_defaults(GTK_TABLE(table),
					  tmp, lft1, rht1, top1, bot1);
	} else {
		hbox = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(parent), hbox, FALSE, FALSE, 0);
		gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 0);
	}

	tmp = gtk_entry_new();
	if (table) {
		gtk_table_attach_defaults(GTK_TABLE(table),
					  tmp, lft2, rht2, top2, bot2);
	} else {
		gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 0);
	}
	gtk_entry_set_text(GTK_ENTRY(tmp), entrytext);
	prefs_wgtopt_list_add(tmp, EntryType, NULL, prefname);

	return tmp;
} /* prefs_entry_with_label_create */


/*
 * PRIVATE: prefs_spin_button_with_label_create
 *
 * convenience routine for creating a spin button and a label
 *
 * if 'table' is non-null, the label and spin button are placed into the table.
 */
static GtkWidget *
prefs_spin_button_with_label_create(
	GtkWidget *parent,
	char *labeltext,
	float adj_value,
	float adj_lower,
	float adj_upper,
	float adj_step_incr,
	float adj_page_incr,
	float adj_page_size,
	char *prefname,
	GtkWidget *table,
	int lft1, int rht1, int top1, int bot1,
	int lft2, int rht2, int top2, int bot2)
{
	GtkWidget *tmp, *entry, *hbox = NULL;
	GtkObject *adj;

	tmp = gtk_label_new(labeltext);
	adj = gtk_adjustment_new(adj_value, adj_lower, adj_upper,
				 adj_step_incr, adj_page_incr, adj_page_size);
	if (table) {
		gtk_misc_set_alignment(GTK_MISC(tmp), 0, 0.5);
		gtk_table_attach_defaults(GTK_TABLE(table),
					  tmp, lft1, rht1, top1, bot1);
	} else {
		hbox = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(parent), hbox, FALSE, FALSE, 0);
		gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 0);
	}

	tmp = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
	gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(tmp), TRUE);
	gtk_widget_set_usize(tmp, 65, 0);
	if (table) {
		gtk_table_attach_defaults(GTK_TABLE(table),
					  tmp, lft2, rht2, top2, bot2);
	} else {
		gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, TRUE, 0);
	}
	entry = GTK_WIDGET(&(GTK_SPIN_BUTTON(tmp)->entry));
	prefs_wgtopt_list_add(GTK_WIDGET(entry), EntryType, NULL, prefname);

	return tmp;
} /* prefs_spin_button_with_label_create */


/*
 * PRIVATE: prefs_radio_button_with_label_create
 *
 * convenience routine for creating a radio button and a label.
 */
static GtkWidget *
prefs_radio_button_with_label_create(
	GtkWidget *parent,
	GtkWidget *group,
	char *labeltext,
	long expr,
	char *prefname)
{
	GtkWidget *tmp;

	if (group != NULL)
		tmp = gtk_radio_button_new_with_label(gtk_radio_button_group(
				GTK_RADIO_BUTTON(group)), labeltext);
	else
		tmp = gtk_radio_button_new_with_label(NULL, labeltext);
	gtk_box_pack_start(GTK_BOX(parent), tmp, FALSE, FALSE, 0);
	if (expr)
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(tmp), TRUE);
	prefs_wgtopt_list_add(tmp, ButtonType, NULL, prefname);

	return tmp;

} /* prefs_radio_button_with_label_create */


/*
 * PRIVATE: prefs_box_in_frame
 *
 * convenience routine for creating a box (vbox or hbox) packed inside a frame.
 * NOTE!!  this routine is slightly different from the other convenience
 * routines; this one returns two widgets instead of one: the frame widget,
 * as well as the vbox widget.
 */
static void
prefs_box_in_frame(
	char *frametext,
	int border_width,
	GtkWidget **frame,
	GtkWidget **box,
	bool_t is_vbox
	)
{
	*frame = gtk_frame_new(frametext);
	if (border_width > 0)
		gtk_container_border_width(GTK_CONTAINER(*frame), border_width);
	if (is_vbox)
		*box = gtk_vbox_new(FALSE, 0);
	else
		*box = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(*frame), *box);
} /* prefs_box_in_frame */


/*
 * PRIVATE: prefs_wgtopt_list_add
 *
 * adds a widget and its corresponding information into the wgtopt_list.
 */
static void
prefs_wgtopt_list_add(GtkWidget *wgt, wgttype_t type, void *data,
	char *prefname)
{
	wgtopt_t *wdata;
	prefs_t *pap;

	wdata = (wgtopt_t *)g_malloc0(sizeof(wgtopt_t));
	wdata->wgt = wgt;

	pap = app_prefs;
	while (pap->name) {
		if (strncmp(prefname, pap->name, strlen(pap->name)) == 0) {
			wdata->pap = pap;
			wdata->type = type;
			wdata->data = data;
			wgtopt_list = g_slist_prepend(wgtopt_list, wdata);
			break;
		}
		pap++;
	}
} /* prefs_wgtopt_list_add */


/*
 * PRIVATE: prefs_update
 *
 * reads from wgtopt_list and updates in-memory preferences.  invoked just
 * before calling prefs_save() to save the new changes.
 */
static void
prefs_update(void)
{
	GSList *wolp, *dp;
	wgtopt_t *wdata;
	char buf[8], *data;
	int num;

	wolp = wgtopt_list;
	while (wolp) {
		wdata = (wgtopt_t *)(wolp->data);
		GNPDBG_PREFS(("prefs_update: updating '%s''\n",
			       wdata->pap->name));
		switch (wdata->pap->type) {
		case PrefBool:
			if (GTK_TOGGLE_BUTTON(GTK_BUTTON(wdata->wgt))->active)
				prefs_bool_set(wdata->pap, "true");
			else
				prefs_bool_set(wdata->pap, "false");
			break;
		case PrefString:
			if (wdata->type != GenericType)
				prefs_string_set(wdata->pap,
						 gtk_entry_get_text(
						 	GTK_ENTRY(wdata->wgt)));
			break;
		case PrefInt:
			/*
			 * ugh this is awful.  if the preference variable
			 * stores an integer, we have to allow at least a
			 * couple of ways to represent it in the prefs window.
			 * the obvious way is the text entry method, where we
			 * can simply use atoi() on the text entry.  however,
			 * in the case of document tab positioning, the value
			 * is an integer, but we want to present it as text to
			 * the user (e.g., "Left", "Right", "Top", "Bottom").
			 * to do this, we need some special case code to
			 * handle it.  i will probably rethink this in the
			 * future to see if there's a better way to do this.
			 */
			data = gtk_entry_get_text(GTK_ENTRY(wdata->wgt));

			if (wdata->type == EntryType) {
				prefs_int_set(wdata->pap, data);
			} else if (wdata->type == ComboType) {
				dp = (GSList *)(wdata->data);
				num = 0;
				while (dp) {
					if (strcmp((char *)(dp->data),
						data) == 0) {
						g_snprintf(buf, 8, "%d", num);
						prefs_int_set(wdata->pap, buf);
						break;
					}
					num++;
					dp = dp->next;
				}
			}

			break;
		case PrefByte:
			prefs_byte_set(wdata->pap,
				gtk_entry_get_text(GTK_ENTRY(wdata->wgt)));
			break;
		case PrefFloat:
			prefs_float_set(wdata->pap,
				gtk_entry_get_text(GTK_ENTRY(wdata->wgt)));
			break;
		default:
			printf("prefs_update: ignoring '%s' (unknown type=%d)"
			       "\n", wdata->pap->name, wdata->pap->type);
			break;
		} /* switch */

		if (wdata->pap->f) {
			GNPDBG_PREFS(("prefs_update: invoking callback for "
				      "'%s''\n", wdata->pap->name));
			(wdata->pap->f)(wdata->pap->c);
		}
		wolp = wolp->next;
	} /* while wolp */
} /* prefs_update */


/*
 * PRIVATE: prefs_wgtopt_list_free
 *
 * cleans up and frees the wgtopt_list
 */
static void
prefs_wgtopt_list_free(void)
{
	GSList *wolp;
	wgtopt_t *wdata;

	while (wgtopt_list) {
		wolp = wgtopt_list;
		wgtopt_list = g_slist_remove_link(wgtopt_list, wolp);
		wdata = (wgtopt_t *)(wolp->data);
		g_free(wdata);
	}
} /* prefs_wgtopt_list_free */


/*
 * PRIVATE: prefs_save_cb
 *
 * callback invoked when user clicks on "Ok" button
 */
static void
prefs_save_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_widget_hide(prefs_win);
	prefs_update();
	prefs_save();
	prefs_wgtopt_list_free();
	gtk_widget_destroy(prefs_win);
	appgtk_rc_update(appgtkrc);
	msgbox_printf("Preferences saved...");
	msgbar_printf(w, "Preferences saved...");
} /* prefs_save_cb */


/*
 * PRIVATE: prefs_win_cancel
 *
 * callback for 'Cancel' button
 */
static void
prefs_win_cancel(GtkWidget *wgt, gpointer cbdata)
{
	gtk_widget_destroy(prefs_win);
} /* prefs_win_cancel */


/*
 * PRIVATE: prefs_win_destroy
 *
 * destroys the prefs window and cleans up the wgtopt_list
 */
static void
prefs_win_destroy(GtkWidget *wgt, gpointer cbdata)
{
	prefs_wgtopt_list_free();
	prefs_win = NULL;
} /* prefs_win_destroy */


/*
 * PRIVATE: prefs_tab_pos_change
 *
 * for all windows, redraw the document tabs.  called from prefs_save, when
 * the preferences may have changed.
 */
static void
prefs_tab_pos_change(void)
{
	switch (prefs_tabpos) {
	case GTK_POS_LEFT:
		prefs_tabpos = GTK_POS_TOP;
		break;
	case GTK_POS_RIGHT:
		prefs_tabpos = GTK_POS_BOTTOM;
		break;
	case GTK_POS_TOP:
		prefs_tabpos = GTK_POS_RIGHT;
		break;
	case GTK_POS_BOTTOM:
		prefs_tabpos = GTK_POS_LEFT;
		break;
	}
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(prefs_nb), prefs_tabpos);
} /* prefs_tab_pos_change */


/* the end */
