/*
 *  Copyright 1995  Mike Jagdis (jaggy@purplet.demon.co.uk)
 *
 *  $Id: libnsl_s_stub.c,v 1.3 1996/02/01 15:41:10 mike Exp $
 */

#include <sys/types.h>
#include <sys/stream.h>
#include <sys/stropts.h>
#include <sys/tihdr.h>
#include <sys/timod.h>
#include <errno.h>
#include <fcntl.h>
#include <tiuser.h>
#include <stdlib.h>
#include <unistd.h>

#include "global.h"
#include "import.h"


#ifndef DEBUG
#define dbg_write(d, s, n)	if (0) ; else
#define dbg_write_int(d, n, b)	if (0) ; else
#else
static int
dbg_write_int(int d, int v, int b)
{
	char buf[16];
	unsigned int n;
	int i, f;

	f = 0;
	if (b == 10 && v < 0) {
		f = 1;
		n = -v;
	} else
		n = v;

	i = 16;
	do {
		buf[--i] = "0123456789abcdef"[n % b];
		n /= b;
	} while (n > 0 && i > (f ? 1 : 0));
	if (f)
		buf[--i] = '-';
	dbg_write(d, buf+i, sizeof(buf)-i);
}
#endif


int
t_accept(int fd, int resfd, struct t_call *call)
{
	/* FIXME */
	t_errno = TNOTSUPPORT;
	return -1;
}


char *
t_alloc(int fd, int struct_type, int fields)
{
	struct t_info info;
	char *it;
	struct netbuf *p_addr = NULL;
	struct netbuf *p_opt = NULL;
	struct netbuf *p_udata = NULL;

	dbg_write(2, "t_alloc type=", 13);
	dbg_write_int(2, struct_type, 10);
	dbg_write(2, ", fields=", 9);
	dbg_write_int(2, fields, 10);
	dbg_write(2, "\n", 1);

	if (t_getinfo(fd, &info)) {
		dbg_write(2, "not a transport end point\n", 26);
		t_errno = TBADF;
		return NULL;
	}

	switch (struct_type) {
		case T_BIND:
			dbg_write(2, "alloc struct t_bind\n", 20);
			if (!(it = (char *)calloc(1, sizeof(struct t_bind)))) {
				t_errno = TSYSERR; errno = ENOMEM;
				return NULL;
			}
			p_addr = (struct netbuf *)&((struct t_bind *)it)->addr;
			p_addr->maxlen = 0;
			p_addr->buf = NULL;
			break;

		case T_OPTMGMT:
			dbg_write(2, "alloc struct t_optmgmt\n", 23);
			if (!(it = (char *)calloc(1, sizeof(struct t_optmgmt)))) {
				t_errno = TSYSERR; errno = ENOMEM;
				return NULL;
			}
			p_opt = (struct netbuf *)&((struct t_optmgmt *)it)->opt;
			p_opt->maxlen = 0;
			p_opt->buf = NULL;
			break;

		case T_CALL:
			dbg_write(2, "alloc struct t_call\n", 20);
			if (!(it = (char *)calloc(1, sizeof(struct t_call)))) {
				t_errno = TSYSERR; errno = ENOMEM;
				return NULL;
			}
			p_addr = (struct netbuf *)&((struct t_call *)it)->addr;
			p_addr->maxlen = 0;
			p_addr->buf = NULL;
			p_opt = (struct netbuf *)&((struct t_call *)it)->opt;
			p_opt->maxlen = 0;
			p_opt->buf = NULL;
			p_udata = (struct netbuf *)&((struct t_call *)it)->udata;
			p_udata->maxlen = 0;
			p_udata->buf = NULL;
			break;

		case T_DIS:
			dbg_write(2, "alloc struct t_discon\n", 22);
			if (!(it = (char *)calloc(1, sizeof(struct t_discon)))) {
				t_errno = TSYSERR; errno = ENOMEM;
				return NULL;
			}
			p_udata = (struct netbuf *)&((struct t_discon *)it)->udata;
			p_udata->maxlen = 0;
			p_udata->buf = NULL;
			break;

		case T_UNITDATA:
			dbg_write(2, "alloc struct t_unitdata\n", 24);
			if (!(it = (char *)calloc(1, sizeof(struct t_unitdata)))) {
				t_errno = TSYSERR; errno = ENOMEM;
				return NULL;
			}
			p_addr = (struct netbuf *)&((struct t_unitdata *)it)->addr;
			p_addr->maxlen = 0;
			p_addr->buf = NULL;
			p_opt = (struct netbuf *)&((struct t_unitdata *)it)->opt;
			p_opt->maxlen = 0;
			p_opt->buf = NULL;
			p_udata = (struct netbuf *)&((struct t_unitdata *)it)->udata;
			p_udata->maxlen = 0;
			p_udata->buf = NULL;
			break;

		case T_UDERROR:
			dbg_write(2, "alloc struct t_uderr\n", 21);
			if (!(it = (char *)calloc(1, sizeof(struct t_uderr)))) {
				t_errno = TSYSERR; errno = ENOMEM;
				return NULL;
			}
			p_addr = (struct netbuf *)&((struct t_uderr *)it)->addr;
			p_addr->maxlen = 0;
			p_addr->buf = NULL;
			p_opt = (struct netbuf *)&((struct t_uderr *)it)->opt;
			p_opt->maxlen = 0;
			p_opt->buf = NULL;
			break;

		case T_INFO:
			dbg_write(2, "alloc struct t_info\n", 20);
			if (!(it = (char *)calloc(1, sizeof(struct t_info)))) {
				t_errno = TSYSERR; errno = ENOMEM;
				return NULL;
			}
			break;


		default:
			dbg_write(2, "alloc unknown struct type\n", 26);
#ifdef XTI
			t_errno = TNOSTRUCTYPE;
#else
			t_errno = TSYSERR;
			errno = EINVAL;
#endif
			return NULL;
	}

	if ((fields & T_ADDR) && p_addr) {
		dbg_write(2, "alloc addr buffer\n", 18);
		if (info.addr == -1) info.addr = 1024;
		if (info.addr < 0) {
			dbg_write(2, "no addr size from getinfo\n", 26);
			free(it);
			t_errno = TSYSERR; errno = EINVAL;
			return NULL;
		}
		if (info.addr > 0
		&& !(p_addr->buf = (char *)calloc(1, info.addr))) {
			dbg_write(2, "no memory for addr buffer\n", 26);
			free(it);
			t_errno = TSYSERR; errno = ENOMEM;
			return NULL;
		}
		p_addr->maxlen = info.addr;
	}

	if ((fields & T_OPT) && p_opt) {
		dbg_write(2, "alloc options buffer\n", 21);
		if (info.options == -1) info.options = 1024;
		if (info.options < 0) {
			dbg_write(2, "no options size from getinfo\n", 29);
			if (p_addr) free(p_addr->buf);
			free(it);
			t_errno = TSYSERR; errno = EINVAL;
			return NULL;
		}
		if (info.options
		&& !(p_opt->buf = (char *)calloc(1, info.options))) {
			dbg_write(2, "no memory for options buffer\n", 29);
			if (p_addr) free(p_addr->buf);
			free(it);
			t_errno = TSYSERR; errno = ENOMEM;
			return NULL;
		}
		p_opt->maxlen = info.options;
	}

	if ((fields & T_UDATA) && p_udata) {
		dbg_write(2, "alloc udata buffer\n", 19);
		if (info.tsdu == -1) info.tsdu = 1024;
		if (info.tsdu < 0) {
			dbg_write(2, "no udata size from getinfo\n", 27);
			if (p_addr) free(p_addr->buf);
			if (p_opt) free(p_opt->buf);
			free(it);
			t_errno = TSYSERR; errno = EINVAL;
			return NULL;
		}
		if (info.tsdu > 0
		&& !(p_udata->buf = (char *)calloc(1, info.tsdu))) {
			dbg_write(2, "no memory for udata\n", 20);
			if (p_addr) free(p_addr->buf);
			if (p_opt) free(p_opt->buf);
			free(it);
			t_errno = TSYSERR; errno = ENOMEM;
			return NULL;
		}
		p_udata->maxlen = info.tsdu;
	}

	return it;
}


int
t_bind(int fd, struct t_bind *req, struct t_bind *ret)
{
	struct strioctl si;
	int bufsize;
	char *buf;
	union u_bind {
		struct T_bind_req req;
		struct T_bind_ack ack;
	} *bind;
	int i;

	dbg_write(2, "t_bind\n", 7);

	bufsize = req ? req->addr.len : 0;
	if (ret && ret->addr.maxlen > bufsize)
		bufsize = ret->addr.maxlen;

dbg_write(2, "bufsize=", 8);
dbg_write_int(2, bufsize, 10);
dbg_write(2, "\n", 1);
bufsize += 16; /* It should be at least the address size of this TEP
		* even if we haven't been given a request and aren't
		* to return one. (This is *required* for the SCO stack.
		* The iBCS emulation can handle it)
		*/
	if (!(buf = (char *)calloc(1, sizeof(union u_bind)+bufsize))) {
		dbg_write(2, "no memory for buffer\n", 21);
		t_errno = TSYSERR;
		errno = ENOMEM;
		return -1;
	}

	bind = (union u_bind *)buf;
	bind->req.PRIM_type = T_BIND_REQ;
	bind->req.CONIND_number = req ? req->qlen : 1;
	if (req && req->addr.len) {
		bind->req.ADDR_length = req->addr.len;
		bind->req.ADDR_offset = sizeof(struct T_bind_req);
		memcpy(buf+sizeof(struct T_bind_req), req->addr.buf,
			req->addr.len);
	}

	si.ic_cmd = TI_BIND;
	si.ic_timout = 5;
	si.ic_len = sizeof(union u_bind) + bufsize;
	si.ic_dp = buf;
	if ((i = ioctl(fd, I_STR, &si))) {
		free(buf);
		t_errno = i & 0xff;
		errno = (i >> 8) & 0xff;
		return -1;
	}

	if (ret) {
		if (ret->addr.buf && ret->addr.maxlen > 0) {
			if (ret->addr.maxlen < bind->ack.ADDR_length) {
				free(buf);
				t_errno = TBUFOVFLW;
				return -1;
			}
			ret->addr.len = bind->ack.ADDR_length;
			memcpy(ret->addr.buf, buf+bind->ack.ADDR_offset,
				bind->ack.ADDR_length);
		} else
			ret->addr.len = 0;
	}
	if (ret)
		ret->qlen = bind->ack.CONIND_number;

	free(buf);
	return 0;
}


int
t_close(int fd)
{
	dbg_write(2, "t_close\n", 8);
	close(fd);
	return 0;
}


int
t_connect(int fd, struct t_call *sndcall, struct t_call *rcvcall)
{
	struct strbuf ctl, dat;
	int bufsize;
	char *buf;
	int i, flags;
	union T_primitives *msg;

	dbg_write(2, "t_connect\n", 10);

	if (!sndcall) {
		dbg_write(2, "no destination address\n", 23);
		t_errno = TBADADDR;
		return -1;
	}

	bufsize = sizeof(struct T_conn_req)
		+ (sndcall->addr.len > 0 ? sndcall->addr.len : 0)
		+ (sndcall->opt.len > 0 ? sndcall->opt.len : 0);
	i = sizeof(struct T_conn_con)
		+ ((rcvcall && rcvcall->addr.maxlen) > 0
			? rcvcall->addr.maxlen : 0)
		+ ((rcvcall && rcvcall->opt.maxlen) > 0
			? rcvcall->opt.maxlen : 0);
	if (i > bufsize)
		bufsize = i;
	if (bufsize < sizeof(*msg))
		bufsize = sizeof(*msg);

	if (!(buf = (char *)calloc(1, bufsize))) {
		dbg_write(2, "no memory for buffer\n", 21);
		t_errno = TSYSERR;
		errno = ENOMEM;
		return -1;
	}

	msg = (union T_primitives *)buf;
	msg->conn_req.PRIM_type = T_CONN_REQ;
	msg->conn_req.OPT_offset = msg->conn_req.DEST_offset
		= sizeof(struct T_conn_req);
	msg->conn_req.DEST_length = sndcall->addr.len;
	if (sndcall->addr.len > 0 && sndcall->addr.buf) {
		memcpy(buf+msg->conn_req.DEST_offset, sndcall->addr.buf,
			sndcall->addr.len);
		msg->conn_req.OPT_offset += sndcall->addr.len;
	}
	msg->conn_req.OPT_length = sndcall->opt.len;
	if (sndcall->opt.len > 0 && sndcall->opt.buf)
		memcpy(buf+msg->conn_req.OPT_offset, sndcall->opt.buf,
			sndcall->opt.len);

	ctl.len = msg->conn_req.OPT_offset + msg->conn_req.OPT_length;
	ctl.buf = buf;
	dat.len = sndcall->udata.len;
	/* SCO cu seems to pass udata.len == 0 but SCO libnsl_s t_open()
	 * seems to force udata.len == -1 (necessary because otherwise
	 * you are trying to send a zero length user data packet and
	 * thus get an error back because user data isn't allowed for
	 * connects on the TCP transport).
	 */
	if (!(dat.buf = sndcall->udata.buf))
		dat.len = -1;
	if (putmsg(fd, &ctl, &dat, 0) < 0) {
		free(buf);
		dbg_write(2, "putmsg: not a transport end point\n", 34);
		t_errno = TBADF;
		return -1;
	}

	ctl.maxlen = bufsize;
	ctl.buf = buf;
	dat.maxlen = -1;
	dat.buf = NULL;
	flags = 0;
	i = getmsg(fd, &ctl, &dat, &flags);
	dbg_write(2, "getmsg 1: returned ", 20);
	dbg_write_int(2, i, 10);
	dbg_write(2, ", errno ", 8);
	dbg_write_int(2, errno, 10);
	dbg_write(2, "\n", 1);
	if (i < 0) {
		free(buf);
		dbg_write(2, "getmsg 1: not a transport end point\n", 36);
		t_errno = TBADF;
		return -1;
	}

	if (msg->type == T_ERROR_ACK) {
		dbg_write(2, "connect failed: t_errno ", 24);
		dbg_write_int(2, msg->error_ack.TLI_error, 10);
		dbg_write(2, ", errno ", 8);
		dbg_write_int(2, msg->error_ack.UNIX_error, 10);
		dbg_write(2, "\n", 1);
		t_errno = msg->error_ack.TLI_error;
		errno = msg->error_ack.UNIX_error;
		free(buf);
		return -1;
	}

	/* FIXME: If O_NDELAY is set we should return here and leave the
	 * connection confirmation to be picked up later.
	 */

	ctl.maxlen = bufsize;
	ctl.buf = buf;
	dat.maxlen = rcvcall ? rcvcall->udata.maxlen : -1;
	dat.buf = rcvcall ? rcvcall->udata.buf : NULL;
	flags = 0;
	i = getmsg(fd, &ctl, &dat, &flags);
	dbg_write(2, "getmsg 2: returned ", 20);
	dbg_write_int(2, i, 10);
	dbg_write(2, ", errno ", 8);
	dbg_write_int(2, errno, 10);
	dbg_write(2, "\n", 1);
	if (i < 0) {
		free(buf);
		dbg_write(2, "getmsg 2: not a transport end point\n", 36);
		t_errno = TBADF;
		return -1;
	}

	if (msg->type == T_CONN_CON && rcvcall) {
		dbg_write(2, "connect confirmed\n", 18);
		if ((rcvcall->addr.maxlen > 0
		&& rcvcall->addr.maxlen < msg->conn_con.RES_length)
		|| (rcvcall->opt.maxlen > 0
		&& rcvcall->opt.maxlen < msg->conn_con.OPT_length)) {
			free(buf);
			dbg_write(2, "rcvcall buffers too small\n", 26);
			t_errno = TBUFOVFLW;
			errno = EINVAL;
			return -1;
		}

		if (rcvcall->addr.maxlen > 0 && rcvcall->addr.buf) {
			memcpy(rcvcall->addr.buf, buf+msg->conn_con.RES_offset,
				msg->conn_con.RES_length);
			rcvcall->addr.len = msg->conn_con.RES_length;
		}
		if (rcvcall->opt.maxlen > 0 && rcvcall->opt.buf) {
			memcpy(rcvcall->opt.buf, buf+msg->conn_con.OPT_offset,
				msg->conn_con.OPT_length);
			rcvcall->opt.len = msg->conn_con.OPT_length;
		}
	}

	free(buf);
	return 0;
}


void
t_error(char *errmsg)
{
	extern char *t_errlist[];

	if (t_errno < 0 || t_errno >= t_nerr)
		t_errno = 0;
	write(2, errmsg, strlen(errmsg));
	write(2, ": ", 2);
	write(2, t_errlist[t_errno], strlen(t_errlist[t_errno]));
	write(2, "\n", 1);
}


int
t_free(char *ptr, int struct_type)
{
	struct netbuf *p_addr = NULL;
	struct netbuf *p_opt = NULL;
	struct netbuf *p_udata = NULL;

	dbg_write(2, "t_free\n", 7);

	if (!ptr)
		return 0;

	switch (struct_type) {
		case T_BIND:
			p_addr = (struct netbuf *)&((struct t_bind *)ptr)->addr;
			break;

		case T_OPTMGMT:
			p_opt = (struct netbuf *)&((struct t_optmgmt *)ptr)->opt;
			break;

		case T_CALL:
			p_addr = (struct netbuf *)&((struct t_call *)ptr)->addr;
			p_opt = (struct netbuf *)&((struct t_call *)ptr)->opt;
			p_udata = (struct netbuf *)&((struct t_call *)ptr)->udata;
			break;

		case T_DIS:
			p_udata = (struct netbuf *)&((struct t_discon *)ptr)->udata;
			break;

		case T_UNITDATA:
			p_addr = (struct netbuf *)&((struct t_unitdata *)ptr)->addr;
			p_opt = (struct netbuf *)&((struct t_unitdata *)ptr)->opt;
			p_udata = (struct netbuf *)&((struct t_unitdata *)ptr)->udata;
			break;

		case T_UDERROR:
			p_addr = (struct netbuf *)&((struct t_uderr *)ptr)->addr;
			p_opt = (struct netbuf *)&((struct t_uderr *)ptr)->opt;
			break;

		case T_INFO:
			break;

		default:
#ifdef XTI
			t_errno = TNOSTRUCTYPE;
#else
			t_errno = TSYSERR;
			errno = EINVAL;
#endif
			return -1;
	}

	if (p_addr  && p_addr->buf)  free(p_addr->buf);
	if (p_opt   && p_opt->buf)   free(p_opt->buf);
	if (p_udata && p_udata->buf) free(p_udata->buf);
	free(ptr);
	return 0;
}


int
t_getinfo(int fd, struct t_info *info)
{
	struct strioctl si;
	union {
		struct T_info_req req;
		struct T_info_ack ack;
	} buf;
	int i;

	dbg_write(2, "t_getinfo\n", 10);

	buf.req.PRIM_type = T_INFO_REQ;
	si.ic_cmd = TI_GETINFO;
	si.ic_timout = 5;
	si.ic_len = sizeof(struct T_info_req);
	si.ic_dp = (char *)&buf;
	if ((i = ioctl(fd, I_STR, &si))) {
		dbg_write(2, "failed\n", 7);
		t_errno = i & 0xff;
		errno = (i >> 8) & 0xff;
		return -1;
	}

	if (info) {
		info->addr = buf.ack.ADDR_size;
		info->options = buf.ack.OPT_size;
		info->tsdu = buf.ack.TSDU_size;
		info->etsdu = buf.ack.ETSDU_size;
		info->connect = buf.ack.CDATA_size;
		info->discon = buf.ack.DDATA_size;
		info->servtype = buf.ack.SERV_type;
	}

	return 0;
}


int
t_getstate(int fd)
{
	struct strioctl si;
	union {
		struct T_info_req req;
		struct T_info_ack ack;
	} buf;
	int i;

	dbg_write(2, "t_getstate\n", 11);

	buf.req.PRIM_type = T_INFO_REQ;
	si.ic_cmd = TI_GETINFO;
	si.ic_timout = 5;
	si.ic_len = sizeof(struct T_info_req);
	si.ic_dp = (char *)&buf;
	if ((i = ioctl(fd, I_STR, &si))) {
		dbg_write(2, "failed\n", 7);
		t_errno = i & 0xff;
		errno = (i >> 8) & 0xff;
		return -1;
	}

	return buf.ack.CURRENT_state;
}


int
t_listen(int fd, struct t_call *call)
{
	/* FIXME */
	dbg_write(2, "t_listen\n", 9);
	t_errno = TNOTSUPPORT;
	return -1;
}


int
t_look(int fd)
{
	/* FIXME */
	dbg_write(2, "t_look\n", 7);
	return 0;
}


int
t_open(char *path, int oflag, struct t_info *info)
{
	int fd;
	int i;

	dbg_write(2, "t_open\n", 7);

#ifdef XTI
	if (oflag != O_RDWR && oflag != (O_RDWR|O_NONBLOCK)) {
		t_errno = TBADFLAG;
		return -1;
	}
#endif

#if 0
	/* Avoid /dev/tcp and /dev/udp. Use /dev/inet/tcp and
	 * /dev/inet/udp instead.
	 */
	if (!strcmp(path, "/dev/tcp"))
		path = "/dev/inet/tcp";
	if (!strcmp(path, "/dev/udp"))
		path = "/dev/inet/udp";
#endif

	if ((fd = open(path, oflag)) < 0) {
#ifdef XTI
		t_errno = T_BADNAME;
#else
		t_errno = TSYSERR;
#endif
		errno = ENOENT;
		return -1;
	}

	i = ioctl(fd, I_FIND, "timod");
	if (i == 0)
		i = ioctl(fd, I_PUSH, "timod");
	if (i < 0) {
		close(fd);
#ifdef XTI
		t_errno = T_BADNAME;
#else
		t_errno = TSYSERR;
#endif
		errno = ENOENT;
		return -1;
	}

	if (info)
		t_getinfo(fd, info);

	return fd;
}


int
t_optmgmt(int fd, struct t_optmgmt *req, struct t_optmgmt *ret)
{
	/* FIXME */
	dbg_write(2, "t_optmgmt\n", 10);
	t_errno = TNOTSUPPORT;
	return -1;
}


int
t_rcv(int fd, char *buf, unsigned int nbytes, int *flags)
{
	struct strbuf ctl, dat;
	int i;

	dbg_write(2, "t_rcv\n", 6);

	ctl.maxlen = -1;
	ctl.buf = NULL;
	dat.maxlen = nbytes;
	dat.buf = buf;
	i = getmsg(fd, &ctl, &dat, flags);
	dbg_write(2, "getmsg: returned ", 18);
	dbg_write_int(2, i, 10);
	dbg_write(2, ", errno ", 8);
	dbg_write_int(2, errno, 10);
	dbg_write(2, "\n", 1);
	if (i < 0) {
		if (errno == EAGAIN || errno == EWOULDBLOCK) {
			dbg_write(2, "getmsg: no data\n", 16);
			t_errno = TNODATA;
		} else {
			dbg_write(2, "getmsg: not a transport end point\n", 34);
			t_errno = TBADF;
		}
		free(buf);
		return -1;
	}
	return dat.len;
}


int
t_rcvconnect(int fd, struct t_call *call)
{
	/* FIXME */
	dbg_write(2, "t_rcvconnect\n", 13);
	t_errno = TNOTSUPPORT;
	return -1;
}


int
t_rcvdis(int fd, struct t_discon *discon)
{
	/* FIXME */
	dbg_write(2, "t_rcvdis\n", 9);
	t_errno = TNOTSUPPORT;
	return -1;
}


int
t_rcvrel(int fd)
{
	/* FIXME */
	dbg_write(2, "t_rcvrel\n", 9);
	t_errno = TNOTSUPPORT;
	return -1;
}


int
t_rcvudata(int fd, struct t_unitdata *unitdata, int *flags)
{
	struct strbuf ctl, dat;
	struct T_unitdata_ind *ind;
	int i;
	char buf[4096];

	dbg_write(2, "t_rcvudata\n", 11);

	if (!unitdata) {
		dbg_write(2, "no unitdata\n", 12);
		t_errno = TSYSERR;
		errno = EINVAL;
		return -1;
	}

	ctl.maxlen = sizeof(buf);
	ctl.buf = buf;
	dat.maxlen = unitdata->udata.maxlen;
	dat.buf = unitdata->udata.buf;
	*flags = 0;
	if ((i = getmsg(fd, &ctl, &dat, flags)) < 0) {
		dbg_write(2, "getmsg 1: returned ", 20);
		dbg_write_int(2, i, 10);
		dbg_write(2, ", errno ", 8);
		dbg_write_int(2, errno, 10);
		dbg_write(2, "\n", 1);
		if (errno == EAGAIN || errno == EWOULDBLOCK) {
			dbg_write(2, "getmsg: no data\n", 16);
			t_errno = TNODATA;
		} else {
			dbg_write(2, "getmsg: not a transport end point\n", 34);
			t_errno = TBADF;
		}
		return -1;
	}

	ind = (struct T_unitdata_ind *)buf;

	unitdata->udata.len = dat.len;

	if (unitdata->addr.maxlen > 0) {
		if (unitdata->addr.maxlen < ind->SRC_length) {
			t_errno = TBUFOVFLW;
			return -1;
		}
		unitdata->addr.len = ind->SRC_length;
		memcpy(unitdata->addr.buf, buf+ind->SRC_offset,
			ind->SRC_length);
	}

	if (unitdata->opt.maxlen > 0) {
		if (unitdata->opt.maxlen < ind->OPT_length) {
			t_errno = TBUFOVFLW;
			return -1;
		}
		unitdata->opt.len = ind->OPT_length;
		memcpy(unitdata->opt.buf, buf+ind->OPT_offset,
			ind->OPT_length);
	}

	return 0;
}


int
t_rcvuderr(int fd, struct t_uderr *uderr)
{
	/* FIXME */
	dbg_write(2, "t_rcvuderr\n", 11);
	t_errno = TNOTSUPPORT;
	return -1;
}


int
t_snd(int fd, char *buf, unsigned int nbytes, int flags)
{
	struct strbuf ctl, dat;
	struct T_data_req req;
	int i;

	dbg_write(2, "t_snd\n", 6);

	/* FIXME: We should loop breaking the data up into TSDU sized
	 * blocks if necessary.
	 */
	req.PRIM_type = T_DATA_REQ;
	req.MORE_flag = 0;

	ctl.len = sizeof(req);
	ctl.buf = (char *)&req;
	dat.len = nbytes;
	dat.buf = buf;
	if ((i = putmsg(fd, &ctl, &dat, flags)) < 0) {
		dbg_write(2, "putmsg returns ", 15);
		dbg_write_int(2, i, 10);
		dbg_write(2, " errno ", 7);
		dbg_write_int(2, errno, 10);
		dbg_write(2, "\n", 1);
		t_errno = TBADF;
		return -1;
	}

	return dat.len;
}


int
t_snddis(int fd, struct t_call *call)
{
	/* FIXME */
	dbg_write(2, "t_snddis\n", 9);
	t_errno = TNOTSUPPORT;
	return -1;
}


int
t_sndrel(int fd)
{
	/* FIXME */
	dbg_write(2, "t_sndrel\n", 9);
	t_errno = TNOTSUPPORT;
	return -1;
}


int
t_sndudata(int fd, struct t_unitdata *unitdata)
{
	struct strbuf ctl, dat;
	struct T_unitdata_req *req;
	int bufsize;
	char *buf;
	int i;

	dbg_write(2, "t_sndudata\n", 11);


	if (!unitdata) {
		dbg_write(2, "no unitdata buffer\n", 23);
		t_errno = TSYSERR;
		errno = EINVAL;
		return -1;
	}

	/* t_sndudata() doesn't send zero length data units. */
	if (unitdata->udata.len == 0)
		return 0;

	bufsize = sizeof(struct T_unitdata_req)
		+ (unitdata->addr.len > 0 ? unitdata->addr.len : 0)
		+ (unitdata->opt.len > 0 ? unitdata->opt.len : 0);
	if (bufsize < unitdata->udata.len)
		bufsize = unitdata->udata.len;

	if (!(buf = (char *)calloc(1, bufsize))) {
		dbg_write(2, "no memory for buffer\n", 21);
		t_errno = TSYSERR;
		errno = ENOMEM;
		return -1;
	}

	req = (struct T_unitdata_req *)buf;
	req->PRIM_type = T_UNITDATA_REQ;
	req->OPT_offset = req->DEST_offset = sizeof(struct T_unitdata_req);
	req->DEST_length = unitdata->addr.len;
	if (unitdata->addr.len > 0 && unitdata->addr.buf) {
		memcpy(buf+req->DEST_offset, unitdata->addr.buf,
			unitdata->addr.len);
		req->OPT_offset += unitdata->addr.len;
	}
	req->OPT_length = unitdata->opt.len;
	if (unitdata->opt.len > 0 && unitdata->opt.buf)
		memcpy(buf+req->OPT_offset, unitdata->opt.buf,
			unitdata->opt.len);

	ctl.len = req->OPT_offset + req->OPT_length;
	ctl.buf = buf;
	dat.len = unitdata->udata.len;
	dat.buf = unitdata->udata.buf;
	if ((i = putmsg(fd, &ctl, &dat, 0)) < 0) {
		dbg_write(2, "putmsg returns ", 15);
		dbg_write_int(2, i, 10);
		dbg_write(2, " errno ", 7);
		dbg_write_int(2, errno, 10);
		dbg_write(2, "\n", 1);
		free(buf);
		t_errno = TBADF;
		return -1;
	}

	return 0;
}


int
t_sync(int fd)
{
	dbg_write(2, "t_sync\n", 7);
	return t_getstate(fd);
}


int
t_unbind(int fd)
{
	struct strioctl si;
	struct T_unbind_req req;
	int i;

	dbg_write(2, "t_unbind\n", 9);

	req.PRIM_type = T_UNBIND_REQ;
	si.ic_cmd = TI_UNBIND;
	si.ic_timout = 5;
	si.ic_len = sizeof(req);
	si.ic_dp = (char *)&req;
	if ((i = ioctl(fd, I_STR, &si))) {
		t_errno = i & 0xff;
		errno = (i >> 8) & 0xff;
		return -1;
	}

	return 0;
}


int
_dummy              ()
{
	dbg_write(2, "_dummy???\n", 10);
	return 0;
}


#if 0
int
__empty_slot()
{
	dbg_write(2, "empty branch table slot!\n", 25);
	return 0;
}
int
_t_checkfd          ()
{
	dbg_write(2, "_t_checkfd         \n", 20);
	return 0;
}
int
_t_aligned_copy     ()
{
	dbg_write(2, "_t_aligned_copy    \n", 20);
	return 0;
}
int
_t_max              ()
{
	dbg_write(2, "_t_max             \n", 20);
	return 0;
}
int
_t_putback          ()
{
	dbg_write(2, "_t_putback         \n", 20);
	return 0;
}
int
_t_is_event         ()
{
	dbg_write(2, "_t_is_event        \n", 20);
	return 0;
}
int
_t_is_ok            ()
{
	dbg_write(2, "_t_is_ok           \n", 20);
	return 0;
}
int
_t_do_ioctl         ()
{
	dbg_write(2, "_t_do_ioctl        \n", 20);
	return 0;
}
int
_t_alloc_bufs       ()
{
	dbg_write(2, "_t_alloc_bufs      \n", 20);
	return 0;
}
int
_t_setsize          ()
{
	dbg_write(2, "_t_setsize         \n", 20);
	return 0;
}
int
_null_tiptr         ()
{
	dbg_write(2, "_null_tiptr        \n", 20);
	return 0;
}
int
_snd_conn_req       ()
{
	dbg_write(2, "_snd_conn_req      \n", 20);
	return 0;
}
int
_rcv_conn_con       ()
{
	dbg_write(2, "_rcv_conn_con      \n", 20);
	return 0;
}
int
_alloc_buf          ()
{
	dbg_write(2, "_alloc_buf         \n", 20);
	return 0;
}
#endif
