/***************************************************************************
 *
 *	Program:	scroll.c
 *	Author:		Marc van Kempen
 *	Desc:		A general purpose scroll-list of a set of names
 *			move with arrow-keys, select with return
 *	Usage:		ScrollList(char **slist, int Nitems, int *StartItem, 
 *				int left, int right, int top, int bottom)
 *
 *
 ***************************************************************************/
 
#include <sys/param.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "mlist.h"
#include "curses.h"

#ifndef min
#define min(a,b)	(a) < (b) ? (a) : (b)
#endif

void
DrawItems(char **slist, int start, int end, int x, int y, int width)
/*
 *	desc:	draw the the items from slist between start and end
 *		at (x,y) with a maximal width of width
 */
{
	char	item[MAXPATHLEN];
	int	i, j;

	for (i=start; i<end; i++) {
		strcpy(item, slist[i]);
		for (j=strlen(item); j<width-1; j++) item[j] = ' ';
		item[width-1]='\0';
		MoveCursor(y+i-start, x);
		Write_to_screen("%s", 1, item);
	}

	return;
} /* DrawItems() */
	
void
DrawFrame(int left, int right, int top, int bottom)
/*
 *	desc:	Draw a frame at coordinates and erase its contents
 */
{
	char 	*line;
	int	i;
	
	line = (char *) malloc( right - left + 2 );
	line[right - left + 1] = 0;		/* !!! terminate the string */
	
	/* draw upper border */
	line[0] = '+';
	for (i=1; i<right-left; i++) { line[i] = '-'; }
	line[right-left] = '+';
	MoveCursor(top, left); Write_to_screen("%s", 1, line);

	/* Draw sides and empty the content of the window */
	line[0] = '|';
	for (i=1; i<right-left; i++) { line[i] = ' '; }
	line[right-left] = '|';
	
	for (i=top+1; i<bottom; i++) { 		
		MoveCursor(i, left);
		Write_to_screen("%s", 1, line); 
	}

	/* Draw lower border */

	line[0] = '+';
	for (i=1; i<right-left; i++) { line[i] = '-'; }
	line[right-left] = '+';
	MoveCursor(bottom, left); Write_to_screen("%s", 1, line);
	
	free(line);
	return;
} /* DrawFrame() */

char *
ScrollList(char **slist, int Nitems, int *StartItem, 
	   int left, int right, int top, int bottom)
/*
 *	desc:	A general purpose scroll-list of a set of names. Move with 
 *		arrow-keys, select with return. Display is constrained to 
 *		the window specified.
 *	pre:	slist is a two-dimensional array containing the selectable
 *		items. Nitems is the # of items. Each item is terminated by 
 *		a 0x0. The list will be displayed starting from <StartItem>.
 *		left < right && 
 *		top < bottom
 *	post:	returns a pointer to the selected item, or NULL if esc or 'q' was 
 *		pressed.
 */
{
	int	ch, i, p, redraw;
	int	maxlen;		/* max. length of a string in the list */
	char	item[255];

	/* Initialize the variables */

	/* get the max. length of a string in the list */
	maxlen = 0;
	for (i=0; i<Nitems; i++) {
		if (strlen(slist[i]) > maxlen) {
			maxlen = strlen(slist[i]);
		}
	}
	/* adjust the size of the window, consider <right>-<left> initially to be the maximum width 
	   of the window */
/*
	right = min(right, left+maxlen+3);
*/

	ch = 0;
	i = *StartItem;
	p = (i>bottom-top-2) ? (i-bottom+top+2) : 0;
	redraw = FALSE;

	/* Draw the initial screen */

	DrawFrame(left, right, top, bottom);
	DrawItems(slist, p, min(p+bottom-top-1, Nitems), 
		  left+2, top+1, right-left-1);
	MoveCursor(top+i+1-p, left+2);
	StartInverse();
	Write_to_screen("%s", 1, strncpy(item, slist[i], right-left-1));
	EndInverse();	

	while ((ch != 27) && (ch != '\n') && (ch != 'q') && (ch != 'Q')) {
		ch = getkey();
		MoveCursor(top+i+1-p, left+2);
		Write_to_screen("%s", 1, strncpy(item, slist[i], right-left-1));
		switch(ch) {
			case UPARROW : /* move up */
			case 'i' :
				if (i > 0) {
					i--;
				}
				if (i < p) {
					p--;
					redraw = TRUE;
				}
				break;	
			case DNARROW : /* move down */
			case 'm' :
				if (i < Nitems-1) {
					i++;
				}
				if (i > p+bottom-top-2) {
					p++;
					redraw = TRUE;
				}
				break;
		}
		if (redraw) {
			DrawItems(slist, p, min(p+bottom-top-1, Nitems), 
					left+2, top+1, right-left-1);
			redraw = FALSE;
		}
		MoveCursor(top+i+1-p, left+2);
		StartInverse();
		Write_to_screen("%s", 1, strncpy(item, slist[i], right-left-1));
		EndInverse();
	}
	*StartItem = i;
	if ((ch == 27) || (ch == 'q') || (ch == 'Q')){
		return(NULL);
	} else {
		return(slist[i]);
	}

	return(NULL);
} /* ScrollList() */


