
/****************************************************************************
 *
 *	Program:	dir.c
 *	Author:		Marc van Kempen
 *	desc:		Directory routines, sorting and reading 
 *
 ****************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/param.h>
#include "mlist.h"

extern	void	*_sort_func;


int 
dir_select(struct dirent *d)
/* 
 *	desc:	allways include a directory entry <d>, except 
 *		for the current directory
 *	pre:	<d> points to a dirent
 *	post:	returns TRUE if d->d_name != "." else FALSE
 */
{
	if (strcmp(d->d_name, ".")==0) {	/* don't include the current directory */
		return(FALSE);
	} else {
		return(TRUE);
	}
}/* dir_select() */

int 
dir_select_root(struct dirent *d)
/* 
 *	desc:	allways include a directory entry <d>, except 
 *		for the current directory
 *	pre:	<d> points to a dirent
 *	post:	returns TRUE if d->d_name[0] != "." else FALSE
 */
{
	if (d->d_name[0] == '.') {	/* don't include the current directory */
		return(FALSE);		/* nor the parent directory */
	} else {
		return(TRUE);
	}
}/* dir_select() */


#ifdef NO_ALPHA_SORT
int
alphasort(const void *d1, const void *d2)
/*
 *	desc:	a replacement for what should be in the library
 */
{
	return(strcmp(((struct dirent *) d1)->d_name,
		      ((struct dirent *) d2)->d_name));
} /* alphasort() */
#endif

int
dir_alphasort(const void *d1, const void *d2)
/*
 *	desc:	compare d1 and d2, but put directories always first
 *
 */
{
	DirList		*f1 = ((DirList *) d1), 
			*f2 = ((DirList *) d2);
	struct stat	*s1 = &(f1->filestatus);
	struct stat	*s2 = &(f2->filestatus);
	
	if ((s1->st_mode & S_IFDIR) && (s2->st_mode & S_IFDIR)) {
		return(strcmp(f1->filename, f2->filename));
	};
	if (s1->st_mode & S_IFDIR) {
		return(-1);
	}
	if (s2->st_mode & S_IFDIR) {
		return(1);
	}
	return(strcmp(f1->filename, f2->filename));
} /* dir_alphasort() */

	
int
dir_sizesort(const void *d1, const void *d2)
/*
 *	desc:	compare d1 and d2, but put directories always first
 *
 */
{
	DirList		*f1 = ((DirList *) d1), 
			*f2 = ((DirList *) d2);
	struct stat	*s1 = &(f1->filestatus);
	struct stat	*s2 = &(f2->filestatus);
	
	if ((s1->st_mode & S_IFDIR) && (s2->st_mode & S_IFDIR)) {
		return(s1->st_size < s2->st_size ? 
			-1 
			: 
			s1->st_size >= s2->st_size);
	};
	if (s1->st_mode & S_IFDIR) {
		return(-1);
	}
	if (s2->st_mode & S_IFDIR) {
		return(1);
	}
	return(s1->st_size < s2->st_size ? 
		-1 
		: 
		s1->st_size >= s2->st_size);
		
} /* dir_sizesort() */

int
dir_datesort(const void *d1, const void *d2)
/*
 *	desc:	compare d1 and d2 on date, but put directories always first
 */
{
	DirList		*f1 = ((DirList *) d1), 
			*f2 = ((DirList *) d2);
	struct stat	*s1 = &(f1->filestatus);
	struct stat	*s2 = &(f2->filestatus);
	
	if ((s1->st_mode & S_IFDIR) && (s2->st_mode & S_IFDIR)) {
		return(s1->st_mtime < s2->st_mtime ? 
			-1 
			: 
			s1->st_mtime >= s2->st_mtime);
	};
	if (s1->st_mode & S_IFDIR) {
		return(-1);
	}
	if (s2->st_mode & S_IFDIR) {
		return(1);
	}
	return(s1->st_mtime < s2->st_mtime ? 
		-1 
		: 
		s1->st_mtime >= s2->st_mtime);

} /* dir_datesort() */
		

int
null_strcmp(char *s1, char *s2)
/*
 *	desc:	compare strings allowing NULL pointers
 */
{
	if ((s1 == NULL) && (s2 == NULL)) {
		return(0);
	}
	if (s1 == NULL) {		
		return(-1);		
	}
	if (s2 == NULL) {
		return(1);
	}
	return(strcmp(s1, s2));
} /* null_strcmp() */
		
		

int
dir_extsort(const void *d1, const void *d2)
/*
 *	desc:	compare d1 and d2 on extension, but put directories always first
 *		extension = "the characters after the last dot in the filename"
 *	pre:	d1 and d2 are pointers to  DirList type records
 *	post:	see code
 */
{
	DirList		*f1 = ((DirList *) d1), 
			*f2 = ((DirList *) d2);
	struct stat	*s1 = &(f1->filestatus);
	struct stat	*s2 = &(f2->filestatus);
	char 		*ext1, *ext2;
	int		extf, ret;
	
	/* find the first extension */
	
	ext1 = f1->filename + strlen(f1->filename);
	extf = FALSE;
	while (!extf && (ext1 > f1->filename)) {
		extf = (*--ext1 == '.');
	}
	if (!extf) {
		ext1 = NULL;
	} else {
		ext1++;
	}
	/* ext1 == NULL if there's no "extension" else ext1 points */
   	/* to the first character of the extension string */
	
	/* find the second extension */
	
	ext2 = f2->filename + strlen(f2->filename);
	extf = FALSE;
	while (!extf && (ext2 > f2->filename)) {
		extf = (*--ext2 == '.');
	} 
	if (!extf) {
		ext2 = NULL;
	} else {
		ext2++;
	}
	/* idem as for ext1 */
	
	if ((s1->st_mode & S_IFDIR) && (s2->st_mode & S_IFDIR)) {
		ret = null_strcmp(ext1, ext2);
		if (ret == 0) {
			return(strcmp(f1->filename, f2->filename));
		} else {
			return(ret);
		}
	};
	if (s1->st_mode & S_IFDIR) {
		return(-1);
	}
	if (s2->st_mode & S_IFDIR) {
		return(1);
	}
        ret = null_strcmp(ext1, ext2);
        if (ret == 0) {
                return(strcmp(f1->filename, f2->filename));
        } else {
                return(ret);
        }
} /* dir_extsort() */

void 
get_dir(char *dirname, DirList **dir, int *n)
/*
 *	desc:	get the files in the current directory
 *	pre:	<dir> == NULL
 *	post:	<dir> contains <n> dir-entries
 */
{
	char		cwd[MAXPATHLEN];
	char		buf[256];
	struct dirent	**dire;
	struct stat	status;
	int		i, nb;
	long		d;

	getcwd(cwd, MAXPATHLEN);
	if (strcmp(cwd, "/") == 0) {	/* we are in the root directory */
		*n = scandir(dirname, &dire, dir_select_root, alphasort);
	} else {
		*n = scandir(dirname, &dire, dir_select, alphasort);
	}

	*dir = (DirList *) malloc( *n * sizeof(DirList) );
	d = 0;
	for (i=0; i<*n; i++) {
		strcpy((*dir)[i].filename, dire[i]->d_name);
		lstat(dire[i]->d_name, &status);
		(*dir)[i].filestatus = status;
		if ((S_IFMT & status.st_mode) == S_IFLNK) {  /* handle links */
			(*dir)[i].link = TRUE;
			stat(dire[i]->d_name, &status);
			nb = readlink(dire[i]->d_name, buf, 256);
			if (nb == -1) {
				printf("Error reading link: %s\n", dire[i]->d_name);
				exit(-1);
			} else {
				(*dir)[i].linkname = malloc(sizeof(char) * nb + 1);
				strncpy((*dir)[i].linkname, buf, nb);
				(*dir)[i].linkname[nb] = 0;
			}
			(*dir)[i].filestatus = status;
		} else {
			(*dir)[i].link = FALSE;
		}
	}
	
	/* sort the directory with the directory names on top */
	
	qsort((*dir)+1, *n - 1, sizeof(DirList), _sort_func);
	

	return;
}/* get_dir() */
	

