/*	Curses.c borrowed from the elm 2.4 pl22 distribution */

/**  This library gives programs the ability to easily access the
     termcap information and write screen oriented and raw input
     programs.  The routines can be called as needed, except that
     to use the cursor / screen routines there must be a call to
     InitScreen() first.  The 'Raw' input routine can be used
     independently, however.

**/

/** NOTE THE ADDITION OF: the #ifndef ELM stuff around routines that
    we don't use.  This is for code size and compile time speed...
**/

#include <stdio.h>

#include <unistd.h>		/* needed for read() */
#include <sys/types.h>		/* idem */
#include <sys/time.h>
#include <sys/uio.h>		/* idem */
#include <string.h>

#include "mlist.h"
#include "curses.h"

#include <sys/fcntl.h>

#ifdef TERMIOS
# include <termios.h>
# ifndef sun
#  include <sys/ioctl.h>	/* for TIOCGWINSZ */
# endif
#else
# ifdef TERMIO
#  include <termio.h>
# else
#  include <sgtty.h>
# endif
#endif

#include <ctype.h>

#ifdef PTEM
#  include <sys/stream.h>
#  include <sys/ptem.h>
#endif

#ifdef BSD
#undef tolower
#endif

#define TTYIN	0

#ifdef SHORTNAMES
# define _clearinverse	_clrinv
# define _cleartoeoln	_clrtoeoln
# define _cleartoeos	_clr2eos
# define _transmit_off	xmit_off
# define _transmit_on	xmit_on
#endif

#ifdef TERMIOS
struct termios  _save_tty[10],
		_raw_tty,
	       _original_tty;
#define	ttgetattr(fd,where)	tcgetattr((fd),(where))
#define	ttsetattr(fd,where)	tcsetattr((fd),TCSADRAIN,(where))
#else	/*TERMIOS*/
# ifdef TERMIO
struct termio 	_save_tty[10],
		_raw_tty, 
              	_original_tty;
#define	ttgetattr(fd,where)	ioctl((fd),TCGETA,(where))
#define	ttsetattr(fd,where)	ioctl((fd),TCSETAW,(where))
# else
struct sgttyb 	_save_tty[10],
		_raw_tty,
	      	_original_tty;
#define	ttgetattr(fd,where)	ioctl((fd),TIOCGETP,(where))
#define	ttsetattr(fd,where)	ioctl((fd),TIOCSETP,(where))
# endif	/*TERMIO*/
#endif	/*TERMIOS*/

int use_tite = 1;           /* flag: use termcap/terminfo ti/te?*/
int COLUMNS=80;
int LINES=24;

static int _inraw = 0;                  /* are we IN rawmode?    */

#define DEFAULT_LINES_ON_TERMINAL	24
#define DEFAULT_COLUMNS_ON_TERMINAL	80

static int _memory_locked = 0;		/* are we IN memlock??   */
static int _line  = -1,			/* initialize to "trash" */
           _col   = -1;

static int _intransmit;			/* are we transmitting keys? */

static
char *_clearscreen, *_moveto, *_up, *_down, *_right, *_left,
     *_setbold, *_clearbold, *_setunderline, *_clearunderline, 
     *_sethalfbright, *_clearhalfbright, *_setinverse, *_clearinverse,
     *_cleartoeoln, *_cleartoeos, *_transmit_on, *_transmit_off,
     *_set_memlock, *_clear_memlock, *_start_termcap, *_end_termcap,
	
	*_kN, *_kP, *_kh, *_kH, *_kd, *_kl, *_kr, *_ku;	/* several keydescriptions */

static int _lines, _columns, _automargin, _eatnewlineglitch;
int tabspacing;

static char _terminal[1024];              /* Storage for terminal entry */
static char _capabilities[1024];           /* String for cursor motion */

static char *ptr = _capabilities;	/* for buffering         */

int    outchar(char c);			/* char output for tputs */
char  *tgetstr(),			/* Get termcap capability */
      *tgoto();				/* and the goto stuff    */

/***********************************************************************
 *
 *	local Prototypes
 *
 ***********************************************************************/

void 	ScreenSize();
int	tgetnum(),
	tgetflag(),
	tputs();

void 	PutLine0(int x, int y, register char *line),
	PutLine1(int x, int y, char *line, char *arg1),
	PutLine2(int x, int y, char *line, char *arg1, char *arg2),
	PutLine3(int x, int y, char *line, char *arg1, char *arg2, char *arg3);
int	Writechar(register int ch);

/***********************************************************************
 *
 *	Utilities
 *
 ***********************************************************************/
 
int
chloc(char *string, register char ch)
{
        /** returns the index of ch in string, or -1 if not in string **/
        register char *s;
                 
        for (s = string; *s; s++)
                if (*s == ch)
                        return(s - string);
        return(-1);
         
} /* chloc() */

/***********************************************************************
 *
 *      Main Section
 *
 ***********************************************************************/
    
int                                                                         
InitScreen(void)
{
	/* Set up all this fun stuff: returns zero if all okay, or;
        -1 indicating no terminal name associated with this shell,
        -2..-n  No termcap for this terminal type known
   	*/

	int  tgetent(),      /* get termcap entry */
	     err;
	char termname[40];
	char *termenv;
	
	SaveState(0);		/* save the terminal settings */
	
	if ((termenv = getenv("TERM")) == NULL) return(-1);

	if (strcpy(termname, termenv) == NULL)
		return(-1);

	if ((err = tgetent(_terminal, termname)) != 1)
		return(err-2);

	ScreenSize(&LINES, &COLUMNS);

	_line  =  0;		/* where are we right now?? */
	_col   =  0;		/* assume zero, zero...     */

	/* load in all those pesky values */
	_clearscreen       = tgetstr("cl", &ptr);
	_moveto            = tgetstr("cm", &ptr);
	_up                = tgetstr("up", &ptr);
	_down              = tgetstr("do", &ptr);
	_right             = tgetstr("nd", &ptr);
	_left              = tgetstr("bc", &ptr);
	_setbold           = tgetstr("so", &ptr);
	_clearbold         = tgetstr("se", &ptr);
	_setunderline      = tgetstr("us", &ptr);
	_clearunderline    = tgetstr("ue", &ptr);
	_setinverse        = tgetstr("so", &ptr);
	_clearinverse      = tgetstr("se", &ptr);
	_sethalfbright     = tgetstr("hs", &ptr);
	_clearhalfbright   = tgetstr("he", &ptr);
	_cleartoeoln       = tgetstr("ce", &ptr);
	_cleartoeos        = tgetstr("cd", &ptr);
	_lines	      	   = tgetnum("li");
	_columns	   = tgetnum("co");
	tabspacing	   = ((tabspacing=tgetnum("it"))==-1 ? 8 : tabspacing);
	_automargin	   = tgetflag("am");
	_eatnewlineglitch   = tgetflag("xn");
	_transmit_on	   = tgetstr("ks", &ptr);
	_transmit_off      = tgetstr("ke", &ptr);
	_set_memlock	   = tgetstr("ml", &ptr);
	_clear_memlock	   = tgetstr("mu", &ptr);
	_start_termcap	   = tgetstr("ti", &ptr);
	_end_termcap	   = tgetstr("te", &ptr);

	_kN		= tgetstr("kN", &ptr);		/* Page Down */
	_kP		= tgetstr("kP", &ptr);		/* Page Up */
	_kh		= tgetstr("k1", &ptr);		/* Home Key, upper left key of keypad */
	_kH		= tgetstr("k4", &ptr);		/* End Key, lower left key of keypad */
	_kd		= tgetstr("kd", &ptr);		/* Down Arrow */
	_kr		= tgetstr("kr", &ptr);		/* Right Arrow */
	_kl		= tgetstr("kl", &ptr);		/* Left Arrow */
	_ku		= tgetstr("ku", &ptr);		/* Up Arrow */

	if (!_left) {
		_left = "\b";
	}

#ifdef USE_TERMCAP
	/* Enable the keypad */
	transmit_functions(ON);
#endif	

	return(0);
} /* InitScreen() */

void
CloseScreen()
/*
 *	desc:	restore the terminal's settings
 */
{
	RestoreState(0);
#ifdef USE_TERMCAP
	transmit_functions(OFF);
#endif
	
	return;	
} /* CloseScreen() */

char *
return_value_of(char *termcap_label)
{
	/** This will return the string kept by termcap for the 
	    specified capability. Modified to ensure that if 
	    tgetstr returns a pointer to a transient address	
	    that we won't bomb out with a later segmentation
	    fault (thanks to Dave@Infopro for this one!)

	    Tweaked to remove padding sequences.
	 **/

	static char escape_sequence[20];
	register int i=0,j=0;
	char buffer[20];
	char *myptr, *tgetstr();     		/* Get termcap capability */

	if (strlen(termcap_label) < 2)
	  return(NULL);

	if (termcap_label[0] == 's' && termcap_label[1] == 'o')
	  {
	  if (_setinverse)
	    strcpy(escape_sequence, _setinverse);
	  else
	    return( (char *) NULL );
	  }
	else if (termcap_label[0] == 's' && termcap_label[1] == 'e')
	  {
	  if (_clearinverse)
	    strcpy(escape_sequence, _clearinverse);
	  else
	    return( (char *) NULL );
	  }
	else if ((myptr = tgetstr(termcap_label, &ptr)) == NULL)
	  return( (char *) NULL );
	else
	  strcpy(escape_sequence, myptr);

	if (chloc(escape_sequence, '$') != -1) {
	  while (escape_sequence[i] != '\0') {
	    while (escape_sequence[i] != '$' && escape_sequence[i] != '\0')
	      buffer[j++] = escape_sequence[i++];
	    if (escape_sequence[i] == '$') {
	      while (escape_sequence[i] != '>') i++;
	      i++;
	    }
	  }
	  buffer[j] = '\0';
	  strcpy(escape_sequence, buffer);
	}

	return( (char *) escape_sequence);
} /* return_value_off() */

void
transmit_functions(int newstate)
{
	/** turn function key transmission to ON | OFF **/

	if (newstate != _intransmit) {
		_intransmit = ! _intransmit;
		if (newstate == ON)
		  tputs(_transmit_on, 1, outchar);
		else 
		  tputs(_transmit_off, 1, outchar);
		fflush(stdout);      /* clear the output buffer */
	}
	return;
} /* transmit_functions() */

/****** now into the 'meat' of the routines...the cursor stuff ******/

void
ScreenSize(int *lines, int *columns)
/*
 *	desc:	returns the number of lines and columns on the display.
 */
{
#ifdef TIOCGWINSZ
	struct winsize w;

	if (ioctl(1,TIOCGWINSZ,&w) != -1) {
		if (w.ws_row > 0)
			_lines = w.ws_row;
		if (w.ws_col > 0)
			_columns = w.ws_col;
	}
#endif

	if (_lines == 0) _lines = DEFAULT_LINES_ON_TERMINAL;
	if (_columns == 0) _columns = DEFAULT_COLUMNS_ON_TERMINAL;
	
	*lines = _lines - 1;		/* assume index from zero */
	*columns = _columns;

	return;	
} /* ScreenSize() */

void
SetXYLocation(int x,int y)
{
	/* declare where the cursor is on the screen - useful after using
	 * a function that moves cursor in predictable fasion but doesn't
	 * set the static x and y variables used in this source file -
	 * e.g. getpass().
	 */

	_line = x;
	_col = y;
	return;
} /* SetXYLocation() */

void 
GetXYLocation(int *x, int *y)
{
	/* return the current cursor location on the screen */

	*x = _line;
	*y = _col;
	return;
} /* GetXYLocation() */

int
ClearScreen()
{
	/* clear the screen: returns -1 if not capable */

	_line = 0;	/* clear leaves us at top... */
	_col  = 0;

	if (!_clearscreen)
		return(-1);

	tputs(_clearscreen, 1, outchar);
	fflush(stdout);      /* clear the output buffer */
	return(0);
} /* ClearScreen() */

static int
CursorUp(int n)
{
	/** move the cursor up 'n' lines **/
	/** Calling function must check that _up is not null before calling **/

	_line = (_line-n > 0? _line - n: 0);	/* up 'n' lines... */

	while (n-- > 0)
		tputs(_up, 1, outchar);

	fflush(stdout);
	return(0);
} /* CursorUp() */


static int
CursorDown(int n)
{
	/** move the cursor down 'n' lines **/
	/** Caller must check that _down is not null before calling **/

	_line = (_line+n <= LINES? _line + n: LINES);    /* down 'n' lines... */

	while (n-- > 0)
		tputs(_down, 1, outchar);

	fflush(stdout);
	return(0);
} /* CursorDown() */


static int
CursorLeft(int n)
{
	/** move the cursor 'n' characters to the left **/
	/** Caller must check that _left is not null before calling **/

	_col = (_col - n> 0? _col - n: 0);	/* left 'n' chars... */

	while (n-- > 0)
		tputs(_left, 1, outchar);

	fflush(stdout);
	return(0);
} /* CursorLeft() */


static int
CursorRight(int n)
{
	/** move the cursor 'n' characters to the right (nondestructive) **/
	/** Caller must check that _right is not null before calling **/

	_col = (_col+n < COLUMNS? _col + n: COLUMNS);	/* right 'n' chars... */

	while (n-- > 0)
		tputs(_right, 1, outchar);

	fflush(stdout);
	return(0);
} /* CursorRight() */

static void
moveabsolute(int col, int row)
{

	char *stuff, *tgoto();

	stuff = tgoto(_moveto, col, row);
	tputs(stuff, 1, outchar);
	fflush(stdout);
	return;
} /* moveabsolute() */

int
MoveCursor(int row, int col)
{
	/** move cursor to the specified row column on the screen.
            0,0 is the top left! **/

	int scrollafter = 0;

	/* we don't want to change "rows" or we'll mangle scrolling... */

	if (col < 0)
	  col = 0;
	if (col >= COLUMNS)
	  col = COLUMNS - 1;
	if (row < 0)
	  row = 0;
	if (row > LINES) {
	  if (col == 0)
	    scrollafter = row - LINES;
	  row = LINES;
	}

	if (!_moveto)
		return(-1);

	if (row == _line) {
	  if (col == _col)
	    return(0);				/* already there! */

	  else if (abs(col - _col) < 5) {	/* within 5 spaces... */
	    if (col > _col && _right)
	      CursorRight(col - _col);
	    else if (col < _col &&  _left)
	      CursorLeft(_col - col);
	    else
	      moveabsolute(col, row);
          }
	  else 		/* move along to the new x,y loc */
	    moveabsolute(col, row);
	}
	else if (_line == row-1 && col == 0) {
	  if (_col != 0)
	    putchar('\r');
	  putchar('\n');
	  fflush(stdout);
	}
	else if (col == _col && abs(row - _line) < 5) {
	  if (row < _line && _up)
	    CursorUp(_line - row);
	  else if (row > _line && _down)
	    CursorDown(row - _line);
	  else
	    moveabsolute(col, row);
	}
	else 
	  moveabsolute(col, row);

	_line = row;	/* to ensure we're really there... */
	_col  = col;

	if (scrollafter) {
	  putchar('\r');
	  while (scrollafter--)
	    putchar('\n');
	}

	return(0);
} /* MoveCursor() */

void
CarriageReturn()
{
	/** move the cursor to the beginning of the current line **/
	Writechar('\r');
	return;
} /* CarriageReturn() */

void
NewLine()
{
	/** move the cursor to the beginning of the next line **/

	Writechar('\r');
	Writechar('\n');
	return;
} /* NewLine() */

int
StartBold()
{
	/** start boldface/standout mode **/

	if (!_setbold)
		return(-1);

	tputs(_setbold, 1, outchar);
	fflush(stdout);
	return(0);
} /* StartBold() */


int
EndBold()
{
	/** compliment of startbold **/

	if (!_clearbold)
		return(-1);

	tputs(_clearbold, 1, outchar);
	fflush(stdout);
	return(0);
} /* EndBold() */

#ifndef ELM

int
StartUnderline()
{
	/** start underline mode **/

	if (!_setunderline)
		return(-1);

	tputs(_setunderline, 1, outchar);
	fflush(stdout);
	return(0);
} /* StartUnderline() */


int
EndUnderline()
{
	/** the compliment of start underline mode **/

	if (!_clearunderline)
		return(-1);

	tputs(_clearunderline, 1, outchar);
	fflush(stdout);
	return(0);
} /* EndUnderline() */


int
StartHalfbright()
{
	/** start half intensity mode **/

	if (!_sethalfbright)
		return(-1);

	tputs(_sethalfbright, 1, outchar);
	fflush(stdout);
	return(0);
} /* StartHalfbright() */

int
EndHalfbright()
{
	/** compliment of starthalfbright **/

	if (!_clearhalfbright)
		return(-1);

	tputs(_clearhalfbright, 1, outchar);
	fflush(stdout);
	return(0);
} /* EndHalfbright() */

int
StartInverse()
{
	/** set inverse video mode **/

	if (!_setinverse)
		return(-1);

	tputs(_setinverse, 1, outchar);
	fflush(stdout);
	return(0);
} /* StartInverse() */


int
EndInverse()
{
	/** compliment of startinverse **/

	if (!_clearinverse)
		return(-1);

	tputs(_clearinverse, 1, outchar);
	fflush(stdout);
	return(0);
} /* EndInverse() */

int
HasMemlock()
{
	/** returns TRUE iff memory locking is available (a terminal
	    feature that allows a specified portion of the screen to
	    be "locked" & not cleared/scrolled... **/

	return ( _set_memlock && _clear_memlock );
} /* HasMemlock() */

static int _old_LINES;

int
StartMemlock()
{
	/** mark the current line as the "last" line of the portion to 
	    be memory locked (always relative to the top line of the
	    screen) Note that this will alter LINES so that it knows
	    the top is locked.  This means that (plus) the program 
	    will scroll nicely but (minus) End memlock MUST be called
	    whenever we leave the locked-memory part of the program! **/

	if (! _set_memlock)
	  return(-1);

	if (! _memory_locked) {

	  _old_LINES = LINES;
	  LINES -= _line;		/* we can't use this for scrolling */

	  tputs(_set_memlock, 1, outchar);
	  fflush(stdout);
	  _memory_locked = TRUE;
	}

	return(0);
} /* StartMemlock() */

int
EndMemlock()
{
	/** Clear the locked memory condition...  **/

	if (! _set_memlock)
	  return(-1);

	if (_memory_locked) {
	  LINES = _old_LINES;		/* back to old setting */
  
	  tputs(_clear_memlock, 1, outchar);
	  fflush(stdout);
	  _memory_locked = FALSE;
	}
	return(0);
} /* EndMemlock() */

#endif /* ndef ELM */

int
Writechar(register int ch)
{
	/** write a character to the current screen location. **/

	static int wrappedlastchar = 0;
	int justwrapped, nt;

	ch &= 0xFF;
	justwrapped = 0;

	/* if return, just go to left column. */
	if(ch == '\r') {
	  if (wrappedlastchar)
	    justwrapped = 1;                /* preserve wrap flag */
	  else {
	    putchar('\r');
	    _col = 0;
	  }
	}

	/* if newline and terminal just did a newline without our asking,
	 * do nothing, else output a newline and increment the line count */
	else if (ch == '\n') {
	  if (!wrappedlastchar) {
	    putchar('\n');
	    if (_line < LINES)
	      ++_line;
	  }
	}

	/* if backspace, move back  one space  if not already in column 0 */
	else if (ch == BACKSPACE) {
	  if(_col != 0) {
	    tputs(_left, 1, outchar);
	    _col--;
	  } /* else BACKSPACE does nothing */
	}

	/* if bell, ring the bell but don't advance the column */
	else if (ch == '\007') {
	  putchar(ch);
	}

	/* if a tab, output it */
	else if (ch == '\t') {
	  putchar(ch);
	  if((nt=next_tab(_col+1)) > prev_tab(COLUMNS))
	    _col = COLUMNS-1;
	  else
	    _col = nt-1;
	}

	else {
	  /* if some kind of non-printable character change to a '?' */
#ifdef ASCII_CTYPE
	  if(!isascii(ch) || !isprint(ch))
#else
	  if(!isprint(ch))
#endif
	    ch = '?';

	  /* if we only have one column left, simulate automargins if
	   * the terminal doesn't have them */
	  if (_col == COLUMNS - 1) {
	    putchar(ch);
	    if (!_automargin || _eatnewlineglitch) {
	      putchar('\r');
	      putchar('\n');
	    }
	    if (_line < LINES)
	      ++_line;
	    _col = 0;
	    justwrapped = 1;
	  }

	  /* if we are here this means we have no interference from the
	   * right margin - just output the character and increment the
	   * column position. */
	  else {
	    putchar(ch);
	    _col++;
	  }
	}

	wrappedlastchar = justwrapped;

	return(0);
} /* WriteChar() */

/*VARARGS2*/

void
Write_to_screen(line, argcount, arg1, arg2, arg3)
char *line;
int argcount;
char *arg1, *arg2, *arg3;
{
	/** This routine writes to the screen at the current location.
  	    when done, it increments lines & columns accordingly by
	    looking for "\n" sequences... **/

	switch (argcount) {
	case 0 :
		PutLine0(_line, _col, line);
		break;
	case 1 :
		PutLine1(_line, _col, line, arg1);
		break;
	case 2 :
		PutLine2(_line, _col, line, arg1, arg2);
		break;
	case 3 :
		PutLine3(_line, _col, line, arg1, arg2, arg3);
		break;
	}
	return;
} /* Write_to_screen() */

void
PutLine0(int x, int y, register char *line)
{
	/** Write a zero argument line at location x,y **/

	MoveCursor(x,y);
	while(*line)
	  Writechar(*line++);
	fflush(stdout);
} /* PutLine0() */

/*VARARGS2*/
void
PutLine1(int x, int y, char *line, char *arg1)
{
	/** write line at location x,y - one argument... **/

	char buffer[VERY_LONG_STRING];

	sprintf(buffer, line, arg1);

	PutLine0(x, y, buffer);
        fflush(stdout);
} /* PutLine1() */

/*VARARGS2*/
void
PutLine2(int x, int y, char *line, char *arg1, char *arg2)
{
	/** write line at location x,y - one argument... **/

	char buffer[VERY_LONG_STRING];

	sprintf(buffer, line, arg1, arg2);

	PutLine0(x, y, buffer);
        fflush(stdout);
} /* PutLine2() */

/*VARARGS2*/
void
PutLine3(int x, int y, char *line, char *arg1, char *arg2, char *arg3)
{
	/** write line at location x,y - one argument... **/

	char buffer[VERY_LONG_STRING];

	sprintf(buffer, line, arg1, arg2, arg3);

	PutLine0(x, y, buffer);
        fflush(stdout);
} /* PutLine3() */

int
CleartoEOLN()
{
	/** clear to end of line **/

	if (!_cleartoeoln)
		return(-1);

	tputs(_cleartoeoln, 1, outchar);
	fflush(stdout);  /* clear the output buffer */
	return(0);
} /* CleartoEOLN() */

int
CleartoEOS()
{
	/** clear to end of screen **/

	if (!_cleartoeos)
		return(-1);

	tputs(_cleartoeos, 1, outchar);
	fflush(stdout);  /* clear the output buffer */
	return(0);
} /* CleartoEOS() */


int
RawState()
{
	/** returns either 1 or 0, for ON or OFF **/

	return( _inraw );
} /* RawState() */

#ifdef ultrix
force_raw()
{
	_inraw = 0;
	Raw(ON);
}
#endif

void
Echo(int state)
/*
 *	desc:	set the terminal to echo if state=ON, to noecho if state=OFF
 */
{
	(void) ttgetattr(TTYIN, &_raw_tty);    /** Read the current terminal settings **/

	switch(state) {
		case OFF :

			#if !defined(TERMIO) && !defined(TERMIOS)
			_raw_tty.sg_flags &= ~(ECHO);		/* echo off 	*/
			#else
			_raw_tty.c_lflag &= ~(ECHO);		/* noecho 	*/
			#endif
			break;

		case ON :

			#if !defined(TERMIO) && !defined(TERMIOS)
			_raw_tty.sg_flags &= (ECHO);		/* echo on 	*/
			#else
			_raw_tty.c_lflag &= (ECHO);		/* echo on	*/
			#endif
			break;
	}
	(void) ttsetattr(TTYIN, &_raw_tty);	/* Write the modified terminal settings */
	return;
} /* Echo() */

void
SaveState(int i)
/*
 *	desc:	Save the tty-settings to tty-var # i
 */
{
	(void) ttgetattr(TTYIN, &_save_tty[i]);
	return;
} /* SaveState() */

void
RestoreState(int i)
/*
 *	desc:	Restore the tty-settings from tty-var # i
 */
{
	(void) ttsetattr(TTYIN, &_save_tty[i]);	
	return;
} /* RestoreState() */

int
Raw(int state)
{
	int do_tite = (state & NO_TITE) == 0;

	state = state & ~NO_TITE;

	/** state is either ON or OFF, as indicated by call **/

	if (state == OFF && _inraw) {
	  if (use_tite && _end_termcap && do_tite) {
	    tputs(_end_termcap, 1, outchar);
	    fflush(stdout);
	  }
	  (void) ttsetattr(TTYIN,&_original_tty);
	  _inraw = 0;
	}
	else if (state == ON && ! _inraw) {

	  (void) ttgetattr(TTYIN, &_original_tty);
	  (void) ttgetattr(TTYIN, &_raw_tty);    /** again! **/

#if !defined(TERMIO) && !defined(TERMIOS)
	  _raw_tty.sg_flags &= ~(ECHO);	/* echo off */
	  _raw_tty.sg_flags |= CBREAK;	/* raw on    */
#else
	  _raw_tty.c_lflag &= ~(ICANON | ECHO);	/* noecho raw mode        */

	  _raw_tty.c_cc[VMIN] = '\0';	/* minimum # of chars to queue    */
	  _raw_tty.c_cc[VTIME] = '\0';	/* minimum time to wait for input */

#endif
	  (void) ttsetattr(TTYIN, &_raw_tty);
	  if (use_tite && _start_termcap && do_tite)
	    tputs(_start_termcap, 1, outchar);
	  _inraw = 1;
	}
	return(0);
} /* Raw() */

int
ReadCh()
{
	/** read a character with Raw mode set! **/

	register int result;
	char ch;
	result = read(0, &ch, 1);
        return((result <= 0 ) ? EOF : ch);
} /* ReadCh() */


int
outchar(char c)
{
	/** output the given character.  From tputs... **/
	/** Note: this CANNOT be a macro!              **/

	putc(c, stdout);
	return(0);
} /* outchar() */

int
getkey(void)
/*
 *	desc:	get a key from the keyboard, check for esc-codes
 *		using the termcapabilities.
 *	pre:	true
 *	post:	returns key VALUE
 *		up 	= UPARROW
 *		down 	= DNARROW
 *		right 	= RIGHTARROW
 *		left 	= LEFTARROW
 *		etc...
 */
{
#ifdef __BORLANDC__
	int	ch;
	
	ch = getch();
	if (ch == 0) {
		ch = getch();
		switch(ch) {
			case 0:
			return(0);
			break;
		}
	}
	
	return(ch);
		
#else !__BORLANDC__
#ifdef USE_TERMCAP

	char		escseq[20];
	fd_set		readfds;
	int		i, arg, timedout;
	struct timeval	t, *tp;		


/*
	escseq[0] = ReadCh();
	arg = fcntl(0, F_GETFL, arg);
	fcntl(0, F_SETFL, O_NDELAY);
*/

	bzero(escseq, 20);			/* clear escsequence string */
	FD_ZERO(&readfds);			/* clear file descriptor bitfield */
	FD_SET(0, &readfds);			/* set stdin in bitfield */
	i = select(1, &readfds, NULL, NULL, NULL);	/* wait for characters from stdin */

/*	usleep(10000);*/
	i = read(0, escseq, 20);	
/*
	i=0;
	while ((escseq[i] = ReadCh()) == -1);
	while (escseq[i] != -1) {
		i++;
		escseq[i] = ReadCh();
	}
	escseq[i] = 0;

	fcntl(0, F_SETFL, arg);
*/
	
/*
        timedout = 0;
        i = 0;
        t.tv_sec = 0;
        t.tv_usec = 1000;
	tp = &t;
        while (!timedout) {
                read(0, escseq+i, 1);
                timedout = ! select(1, &readfds, NULL, NULL, tp);
                i++;
        }
        escseq[i] = 0;
*/	
	/* compare escsequence with terminalcapabilities */

/*
printf("%d\n", strlen(escseq));
for (i=0; i<10; i++) 
	if (escseq[i]>31) fprintf(stderr, "%c (%d) ", escseq[i], escseq[i]);
	else fprintf(stderr, "%d ", escseq[i]);
fprintf(stderr,"\n");
*/

	if (_ku) if (strcmp(escseq, _ku) == 0)	return(UPARROW);
	if (_kd) if (strcmp(escseq, _kd) == 0)	return(DNARROW);
	if (_kr) if (strcmp(escseq, _kr) == 0)	return(RIGHTARROW);
	if (_kl) if (strcmp(escseq, _kl) == 0)	return(LEFTARROW);
	if (_kN) if (strcmp(escseq, _kN) == 0)	return(PGDN);
	if (_kP) if (strcmp(escseq, _kP) == 0)	return(PGUP);
	if (_kh) if (strcmp(escseq, _kh) == 0)	return(HOME);
	if (_kH) if (strcmp(escseq, _kH) == 0)	return(END);

	if (strlen(escseq) > 1) {
		return(escseq[strlen(escseq)-1]);	/* assume it's a not recognized esc-sequence  */
	} else {
		return(escseq[0]);			/* assume it's a regular key */
	}
	
#else !USE_TERMCAP

	int 	ch;

	ch = ReadCh();
	
	if (ch == ESC) {	
		ch = ReadCh();
		if (ch == '[') {
			ch = ReadCh();
			switch(ch) {
				case 'A'	: return(UPARROW); break;
				case 'B'	: return(DNARROW); break;
				case 'C'	: return(RIGHTARROW); break;
				case 'D'	: return(LEFTARROW); break;
				case '5'	: ch = ReadCh(); return(PGUP); break;
				case '6'	: ch = ReadCh(); return(PGDN); break;
				case 'U'	: return(END); break;
				case 0		: return(HOME); break;
			}
		} else {
			if (ch == 'O') {
				ch = ReadCh();
				switch(ch) {
					case 'q'	: return(HOME); break;
					case 'w'	: return(END); break;
				}
			} else {
				return(ch);
			}
		}
	} else {
		if (ch == 127) {	/* ^? (DELETE) */
			return(DEL);
		} else {
			return(ch);
		}
	}
	return(ch);

#endif USE_TERMCAP	
#endif __BORLANDC__
}/* getkey() */


char *
ReadString(char *line, int len, char c)
/*
 *	desc:	Read a string from the keyboard, with a maximal
 *		length of <len>, into <s>. <c> is used to mark 
 *		the entry-box.
 */
{
#define LINELEN		300
	char	lineold[LINELEN];
	int	i, key, k, l, ins,
		x, y;			/* cursor coordinates */

	GetXYLocation(&y, &x);
	for (i=0; i<len; i++) {
		MoveCursor(y, x+i);
		Write_to_screen("%c", 1, c);
	}

	if (strlen(line) > len) line[len] = 0;
	strcpy(lineold, line);

	MoveCursor(y, x);
	StartInverse();
	Write_to_screen("%s", 1, line);	/* display line 	*/
	EndInverse();
	MoveCursor(y, x+strlen(line));
	/*
	   Now wait for a keypress, if it's a displayable character then 
	   erase the word and start with a blank string, else leave the word 
	   and act on the character pressed.
	*/
	key = getkey();
	if ((31< key) && (key < 128)) {
		for (i=0; i<strlen(line); i++) {
			MoveCursor(y, x+i);
			Write_to_screen("%c", 1, c);
		}
		i = 0;
		*line = 0;			/* trim the line */
		MoveCursor(y, x);		/* go to the beginning of the line */
	} else {
		MoveCursor(y, x);
		Write_to_screen("%s", 1, line);	/* display line */
		i = strlen(line);
		MoveCursor(y, x+i);
	}		
	ins = TRUE;
	while ((key != '\n') && (key != ESC)) {
		switch(key) {
			case BS :	/* Backspace */
				if (i>0) {
					i--;
					l = strlen(line);
					k = i;
					while (k < l-1) { line[k] = line[k+1]; k++; }
					line[k] = 0;
					MoveCursor(y, x);
					Write_to_screen("%s%c", 2, line, c);		/* redisplay line */
				}
				break;
			case RIGHTARROW :	/* --> */
				if ((i < strlen(line)) && (i < LINELEN)) i++;
				break;
			case LEFTARROW :	/* <-- */
				if (i > 0) i--;
				break;
			case INS :	/* Insert key */
				ins = 1 - ins;
				break;
			case DEL :	/* Delete key */
			case 4 :	/* Ctrl-D */
				l = strlen(line);
				k = i;
					while (k < l-1) { line[k] = line[k+1]; k++; }
				line[k] = 0;
				MoveCursor(y, x);
				Write_to_screen("%s%c", 2, line, c);	     /* display line */
				break;
			case HOME : 	/* home key */
			case 1:			/* Ctrl-A */
				i = 0;
				break;
			case END :	/* end key */
			case 5:		/* Ctrl-E */
				i = strlen(line);
				break;
			default :	/* assume its a displayable key */
				if ((i < len) && (strlen(line) < LINELEN) && key != '\n') {
					/* move all characters to the right of i, to the right */
					if (ins) {
						l = strlen(line) < len ? strlen(line) : len - 1;
						k = l;
						while (k > i) { line[k] =line[k-1]; k--; }
						line[l+1] = 0;
					}
					/* insert the key at position i */
					line[i] = key;
					MoveCursor(y, x);
					Write_to_screen("%s", 1, line);	     /* display line */
					if ((i+1 < len) && (i+1 < LINELEN)) i++;
				}
				break;
		}
		MoveCursor(y, x+i);
		key = getkey();
	}
	if (key == ESC) {
		strcpy(line, lineold);
		MoveCursor(y, x);
		Write_to_screen("%s", 1, line);	     /* display line */
	}

	return(line);
}
