/* TDir.h  ---  Definition and declaration of class TDir 
   for local or remote directory manipulation 
   
   
   Copyright (C) 1993  A.Matthias
   Part of the ftp code designed and implemented by 
   Rolf Krahl, krahl@namu01.gwdg.de
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   */


/* ! All TDir routines are implemeted here, there is *no* TDir.cc ! */

#ifndef _TDir_h
#define _TDir_h

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/file.h>

#include "globals.h" 

class TDir 
{
private:
#define READ  0            // The read end of the pipe
#define WRITE 1            // The write end of the pipe
#define BUFFERSIZE 4096
#define ENDSIGN "-end-\n"
  
#define FTP_CWD      257
#define FTP_CD_OK    250
#define FTP_XFER_OK  226
  
  
  char  cwd[PATH_MAX];
  char  cl[PATH_MAX];    // Command line temporary storage
  char  dirfile[PATH_MAX];
  bool  imlocal;         // Am I connected now or not?
  int   cftp[2];         // File descriptors for sending input to ftp
  int   ftpc[2];         // File descriptors for recv input from ftp
  int   rbytes, wbytes;  // Bytes read or written
  char  buf[BUFFERSIZE];
  char  hostname[PATH_MAX];
  int   ftppid;
  bool  ftpstarted;      // Is an ftp process already running?
  
  int dgets( int fd, char *buf, int maxsize /* maxsize >= 1 */ )
  {
    char c = '\0';
    int i = 0;
    
    while ( ( c != '\n') && ( i < maxsize - 1 ) )
    {
      read( fd, &c, 1 );
      buf[i] = c;
      i++;
    }
    
    buf[i] = '\0';
    
    return i;
  }
  
  int dputs( int fd, const char *buf )
  {
    int i;
    
    i = write( fd, buf, strlen( buf ) );
    write( fd, "\n", 1 );
    
    return ( i >= 0 ) ? i + 1 : i;
  }
  
  void ftpcmd ( int rfd, int wfd, const char *cmd, char *buf, int maxsize )
  {
    char echocmd[256];
    char *p;
    int i = 0;
    
    sprintf( echocmd, "!echo %s", ENDSIGN );
    dputs( wfd, cmd );
    dputs( wfd, echocmd );
    
    do
    {
      p = &buf[i];
      i += dgets( rfd, p, maxsize - i );
    } while ( (strcmp( p, ENDSIGN )) && (i < maxsize-1 /* ERROR ! */) );
    
    *p = '\0';   /* Removes ENDSIGN from buf */
  }
  
  void ftpopen( void )
  {
    pipe( cftp );
    pipe( ftpc );
    
    signal( SIGPIPE, SIG_IGN );
    ftpstarted=TRUE;
    
    if ( ( ftppid = fork() ) == 0 )  /* ftp */
    {
      close( cftp[WRITE] );
      dup2( cftp[READ], 0 );
      close( cftp[READ] );
      
      close( ftpc[READ] );
      dup2( ftpc[WRITE], 1 );
      dup2( ftpc[WRITE], 2 );
      close( ftpc[WRITE] );
      execlp( "ftp", "ftp", "-", "-v", NULL );
      fprintf( stderr, "ftp died\n" );
      exit( 1 );
    }
    else
    {
      close( cftp[READ] );
      close( ftpc[WRITE] );
    }
    return;
    
  }
  
  
public:
  TDir( char *dfile ) 
  { 
    imlocal = TRUE;
    strcpy( hostname, "local" );
    strcpy( dirfile, dfile ); 
    ftpstarted=FALSE;
    getcwd( cwd );
  }
  
  
  int remote( char *host )
  {
    char tmp[1024];
    if (ftpstarted==FALSE)
    {
      ftpopen();
      ftpcmd( ftpc[READ], cftp[WRITE], "prompt", buf, sizeof( buf ) );
      ftpstarted=TRUE;
    }
    strcpy( hostname, host );
    imlocal = FALSE;
    sprintf( tmp, "open %s", host );
    ftpcmd( ftpc[READ], cftp[WRITE], tmp, buf, sizeof( buf ) );
    
    /* We must check for errors (e.g. unknown host) here */
    /* All errors seem to begin with "ftp: " */
    if ( strstr( buf, "ftp: " ) != NULL )
    {
      /* beep and return to local operation */
      fprintf( stderr, "\a" );
      strcpy( hostname, "local" );
      imlocal = TRUE;
      getcwd( cwd );
    }
    
    ftpcmd( ftpc[READ], cftp[WRITE], "", buf, sizeof( buf ) );
    getcwd( cwd );
    return( 0 );       // Error code = OK
  }
  
  
  int local( void )
  {
    ftpcmd( ftpc[READ], cftp[WRITE], "close", buf, sizeof( buf ) );
    imlocal = TRUE;
    strcpy( hostname, "local" );
    getcwd( cwd );
    return( 0 );       // Error code = OK
  }
  
  void setdirfile( char *dfile )
  { 
    strcpy( dirfile, dfile ); 
    getcwd( cwd ); 
  }
  
  char *getcwd( char *wd )
  {
    char tmp[PATH_MAX];
    int cmd_nr;
    
    if ( imlocal == TRUE )
      getwd( wd );
    else
    {
      ftpcmd( ftpc[READ], cftp[WRITE], "pwd", buf, sizeof( buf ) );
      ftpcmd( ftpc[READ], cftp[WRITE], "pwd", buf, sizeof( buf ) );
      sscanf( buf, "%d %s", &cmd_nr, wd );
      if ( cmd_nr != FTP_CWD )
      {
	endwin();
	fprintf( stderr, "getcwd: Unexpected FTP output\n" );
	exit( 1 );
      }
      else
      {
	// wd is still in quotes. We have to remove them
	wd[strlen( wd )-1] = '\0';
	strcpy( tmp, wd );
	strcpy( wd, &tmp[1] );
	sprintf( tmp, "r0:%s", wd );
	strcpy( wd, tmp );
      }
    }
    
    return( wd );
  }
  
  char *telldir( char *wd ) 
  { if (imlocal==FALSE) getcwd( cwd ); strcpy( wd, cwd ); return( wd ); }
  
  
  char *tellhost( char *host )
  {
    strcpy( host, hostname );
    return( host );
  }
  
  
  bool changedir( char *dir )
  { 
    bool ret;
    char cmd[PATH_MAX];
    
    if ( imlocal == TRUE )
      ret = (bool)!chdir( dir ); 
    else
    {
      sprintf( cmd, "cd %s", dir );
      ftpcmd( ftpc[READ], cftp[WRITE], cmd, buf, sizeof( buf ) );
      ret = TRUE;
    }
    
    getcwd( cwd ); 
    writedir(); 
    return( ret ); 
  }
  
  void activate( void )
  {
    changedir( cwd ); 
  }
  
  bool writedir( void )
  {
    char *p, s[16], cmd[PATH_MAX], fname[PATH_MAX];;
    int cmd_nr, fd;
    bool readable;
    FILE *fp;
    
    if ( imlocal == TRUE )
    {
      // Check if we have the permissions to read this directory
      readable=TRUE;
      if ( (fd = open( ".", O_RDONLY)) == -1 )
	readable=FALSE;
      else
	close( fd );

      if ( readable == FALSE )
      {
	sprintf( fname, "%s%s", localtmpdir, dirfile );
	fp = fopen( fname, "w" );
	fprintf( fp, "total 0\n" );
	fprintf( fp, "d?????????  0 user group       0000 Mon  0 00:00 . \n" );
	fprintf( fp, "d?????????  0 user group       0000 Mon  0 00:00 ..\n" );
	fprintf( fp, "\n" );
	fflush( fp );
	fclose( fp );
	return( FALSE );
      }
      else
      {
	sprintf( cl, "ls -la > %s%s", localtmpdir, dirfile );
	system( cl );
	return( TRUE );
      }
    }
    else
    {
      sprintf( cmd, "dir -a %s%s", localtmpdir, dirfile );
      ftpcmd( ftpc[READ], cftp[WRITE], cmd, buf, sizeof( buf ) );
      sprintf( s, "\n%d", FTP_XFER_OK );
      p = strstr( buf, s );
      sscanf( p, "\n%d", &cmd_nr );
      if ( cmd_nr != FTP_XFER_OK )
      {
	endwin();
	fprintf( stderr, "writedir: Unexpected FTP output\n" );
	exit( 1 );
      }
      return TRUE;
    }
    
  }
  
  
  // Remote Get command
  
  void RemGet( char *remotefile, char *localfile )
  {
    char *p, s[16], cmd[PATH_MAX];
    int cmd_nr;
    
    if ( imlocal == TRUE )
      return;
    
    // Switch into binary mode
    ftpcmd( ftpc[READ], cftp[WRITE], "binary", buf, sizeof( buf ) );
    
    sprintf( cmd, "get %s %s", remotefile, localfile );
    ftpcmd( ftpc[READ], cftp[WRITE], cmd, buf, sizeof( buf ) );
    sprintf( s, "\n%d", FTP_XFER_OK );
    p = strstr( buf, s );
    sscanf( p, "\n%d", &cmd_nr );
    if ( cmd_nr != FTP_XFER_OK )
    {
      endwin();
      fprintf( stderr, "remget(): Unexpected FTP output\n" );
      exit( 1 );
    }
    
    // Switch back to text mode
    ftpcmd( ftpc[READ], cftp[WRITE], "ascii", buf, sizeof( buf ) );
  }
  
  
  // Remote Put command
  
  void RemPut( char *localfile, char *remotefile )
  {
    char *p, s[16], cmd[PATH_MAX];
    int cmd_nr;
    
    if ( imlocal == TRUE )
      return;
    
    // Switch into binary mode
    ftpcmd( ftpc[READ], cftp[WRITE], "binary", buf, sizeof( buf ) );
    
    sprintf( cmd, "put %s %s", localfile, remotefile );
    ftpcmd( ftpc[READ], cftp[WRITE], cmd, buf, sizeof( buf ) );
    sprintf( s, "\n%d", FTP_XFER_OK );
    p = strstr( buf, s );
    sscanf( p, "\n%d", &cmd_nr );
    if ( cmd_nr != FTP_XFER_OK )
    {
      endwin();
      fprintf( stderr, "remput(): Unexpected FTP output\n" );
      exit( 1 );
    }
    
    // Switch back to text mode
    ftpcmd( ftpc[READ], cftp[WRITE], "ascii", buf, sizeof( buf ) );
  }
  
};  /* TDir */

#endif



