#include <ncurses.h>
#include <errno.h>
#include <stdio.h>

/*
 * This program helps you to make the most of your keyboard with
 * GC3. If you run this program without arguments, it will read
 * a file called `keycode.def' and just write it back out again.
 * If the file does not exist, one will be created which matches
 * the default UNIX keycodes for GC3.
 *
 * Each line of the keycode.def file has a number, a description,
 * and a keycode. The number refers to the keycode table index.
 *
 * If you invoke keycode with an argument, it will allow you to
 * modify the keycode tabel by pressing the key you want associated
 * with an entry. Pressing ENTER will leave the entry unmodified. 
 * keycode will first check that there are no other table entries
 * with that keycode before it will make the modification.
 *
 * The command line arguments are:
 *
 *	-M	Edit keycodes for Meta-key entries
 *	-F	Edit keycodes for function key entries
 *	-C	Edit keycodes for cursor keys
 *	-E	Edit keycodes for command line editor
 *	-O	Generate an output file keycode.gc3 with
 *			GC3 keycode statements for the configuration
 *			at exit
 *
 * Arguments can be combined, e.g.: keycode -MEO
 *
 */

#include "unixcfg.h"

FILE *keyfp;

char *(cursorKeyNames[]) = {
	"Up", "Down", "Left", "Right",
	"PgUp", "PgDn", "Home", "End",
	"CtlLeft", "CtlRight", "CtlUp", "CtlDown",
	"CtlPgUp", "CtlPgDn", "CtlHome", "CtlEnd",
	"Ins", "Del", "Bkspc"
};

char *(editorKeyNames[]) = {
	"Insert File Name",
	"Move to start of line",
	"Move to end of line",
	"Move cursor left",
	"MOve cursor right",
	"Delete char under cursor",
	"Delete char left of cursor",
	"Toggle insert/overwrite",
	"Clear line",
	"Delete left of cursor",
	"Delete right of cursor",
	"Delete previous word",
	"Delete next word",
	"Move left by one word",
	"Move right by one word",
	"Recall previous command",
	"Recall next command"
};

int KeyTable[256],
    defaultKeyTable[256] = {
	/* Cursor keys */
	KEY_UP, KEY_DOWN, KEY_LEFT, KEY_RIGHT,
	KEY_PPAGE, KEY_NPAGE, KEY_HOME, KEY_END,
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, '\b',
	/* #19 */
	0,
	/* Function Keys 20-29 */
	KEY_F1, KEY_F2, KEY_F3, KEY_F4, KEY_F5,
	KEY_F6, KEY_F7, KEY_F8, KEY_F9, KEY_F0,
	/* Ctrl-Keys 30-55 */
	1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13,
	14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26,
	/* #56-#59 */
	0, 0, 0, 0,
	/* Meta-0 .. Meta-9 */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
	/* Meta-A .. Meta-Z 70-95 */
	0, 0, 0, 0, 0, 0, 0, 0, 0 ,0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0 ,0, 0, 0, 0,
	/* #96 - #99 */
	0, 0, 0, 0,
	/* Printable ASCII */
	        32, 33, 34, 35, 36, 37, 38, 39,
	40, 41, 42, 43, 44, 45, 46, 47, 48, 49,
	50, 51, 52, 53, 54, 55, 56, 57, 58, 59,
	60, 61, 62, 63, 64, 65, 66, 67, 68, 69,
	70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
	80, 81, 82, 83, 84, 85, 86, 87, 88, 89,
	90, 91, 92, 93, 94, 95, 96, 97, 98, 99,
	100,101,102,103,104,105,106,107,108,109,
	110,111,112,113,114,115,116,117,118,119,
	120,121,122,123,124,125,126,
	/* #195..#239 */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0,
	/* Command line edit keys */
	0, KEY_HOME, KEY_END, KEY_LEFT, KEY_RIGHT,
	0, '\b', 0, 27,
	0, 0, 
	0, 0,
	0, 0,
	0, 0
};

#if __STDC__
char *KeyName(int keynum, int is_keyspec) {
#else
char *KeyName(keynum, is_keyspec) {
#endif
	static char namebuff[40];
	if (keynum<19) strcpy(namebuff,cursorKeyNames[keynum]);
	else if (keynum>=20 && keynum<30)
		sprintf(namebuff,"F%d",keynum-20);
	else if (keynum>=30 && keynum<56)
		sprintf(namebuff,"^%c",keynum-30+'A');
	else if (keynum>=60 && keynum<70)
		sprintf(namebuff,"Meta-%d",keynum-60);
	else if (keynum>=70 && keynum<96)
		sprintf(namebuff,"Meta-%c",keynum-70+'A');
	else if (keynum>=100 && keynum<195)
		sprintf(namebuff,"`%c'",keynum-100+' ');
	else if (keynum>=231 && keynum<239)
		sprintf(namebuff,"ESC-#%d",keynum);
	else if (!is_keyspec && keynum>=239)
		strcpy(namebuff,editorKeyNames[keynum-239]);
	else sprintf(namebuff,"#%d",keynum);
	return namebuff;
}


#if __STDC__
void readKeyFile(char *name) {
#else
void readKeyFile(name)
	char *name;
{
#endif
	FILE *fp = fopen(name,"r");
	char buf[80];
	if (fp) {
		int i = 0;
		buf[79] = '\0';
		while (!feof(fp) && i < 256) {
			int j;
			fgets(buf,79,fp);
			j = strlen(buf)-1;
			while (!isdigit(buf[j]) && j>0) j--;
			while (isdigit(buf[j]) && j>0) j--;
			sscanf(buf+j+1,"%d",&KeyTable[i++]);
		}
		fclose(fp);
	} else {
		int i;
		for (i=0;i<256;i++) KeyTable[i] = defaultKeyTable[i];
	}
}

#if __STDC__
void writeKeyFile(char *name) {
#else
void writeKeyFile(name)
	char *name;
{
#endif
	FILE *fp = fopen(name,"w");
	if (fp) {
		int i;
		for (i=0;i<256;i++) 
			fprintf(fp,"%-3d %s\t%d\n",i,KeyName(i,0), KeyTable[i]);
		fclose(fp);
#ifdef NO_STRERROR
	} else fprintf(stderr,"Cannot write code file! errno=%d\n",errno);
#else
	} else fprintf(stderr,"Cannot write code file: %s!\n",strerror(errno));
#endif
}

#if __STDC__
int unique(int k, int v) {
#else
int unique(k,v) {
#endif
	int i;
	if (k<231) {
		for (i=0;i<231;i++) {
			if (i!=k && KeyTable[i]==v) {
				printw("This is already used for [%s]!\n",KeyName(i,0));
				return 0;
			}
		}
	} else if (k<239) { /* ESC-k sequence */
		for (i=231;i<239;i++) {
			if (i!=k && KeyTable[i]==v) {
				printw("This is already used for [%s]!\n",KeyName(i,0));
				return 0;
			}
		}
	} else { /* Command-line editor */
		for (i=239;i<256;i++) {
			if (i!=k && KeyTable[i]==v) {
				printw("\nThis is already used for [%s]!\n",KeyName(i,0));
				return 0;
			}
		}
		for (i=0;i<231;i++) {
			if (KeyTable[i]==v) {
				printw("\nWarning: This is already used for [%s]!\n",KeyName(i,0));
				break;
			}
		}
	}
	return 1;
}

#if __STDC__
void getKey(int k) {
#else
void getKey(k) {
#endif
	int v;	
	wclear(stdscr);
	for (;;) {
		printw("\nPress the key for [%s] or ENTER: ",KeyName(k,0));
		v = wgetch(stdscr);
		if (v=='\r') break;
		if (unique(k,v)) {
			KeyTable[k] = v;
			break;
		}
	}
}

#if __STDC__
void getKeys(int start, int end) {
#else
void getKeys(start, end) {
#endif
	int i;
	for (i=start;i<=end;i++) getKey(i);
}


#if __STDC__
static void useage(void) {
#else
static void useage() {
#endif
	fprintf(stderr,"Useage: keycode [-C] [-M] [-F] [-E]\n");
	exit(-1);
}

#if __STDC__
void main(int argc, char *argv[]) {
#else
void main(argc, argv)
	int argc;
	char *argv[];
{
#endif
	int i, doOutput = 0, doMeta = 0, doCursor = 0, doEditor = 0, doFunc = 0;
	readKeyFile("keycode.def");
	if (argc>1) {
		char *p;
		i = 1;
		while (i<argc) {
			if (argv[i][0]!= '-') useage();
			else p = &argv[i++][1];
			while (*p) {
				switch (*p) {
				case 'M': doMeta = 1; break;
				case 'C': doCursor = 1; break;
				case 'F': doFunc = 1; break;
				case 'E': doEditor = 1; break;
				case 'O': doOutput = 1; break;
				}
				p++;
			}
		}
	}
	initscr(); nonl(); raw(); noecho();
#ifndef NO_KEYENABLE
	keypad(stdscr,1);
#endif
	if (doMeta) getKeys(60,95);
	if (doEditor) getKeys(239,255);
	if (doCursor) getKeys(0,18);
	if (doFunc) getKeys(20,29);
	endwin();
	writeKeyFile("keycode.def");
	if (doOutput) {
		FILE *fp = fopen("keycode.gc3","w");
		if (fp) {
			int i;
			for (i=0;i<256;i++)
				if (KeyTable[i] != defaultKeyTable[i])
					fprintf(fp,"\tkeycode [%s] = %d\n",
						KeyName(i,1),KeyTable[i]);
			fclose(fp);
		} else fprintf(stderr,"Couldn't open file keycode.gc3 for output\n");
	}
}

