/****************************************************

	GC3 EXPRESSION EVALUATOR

 This code evaluates numeric expressions contained
 in a string. It should be portable across any
 decent C compiler.

 Entry to this code is through the routine:

	char *compute(char *str, int *rtn)
		
 which takes as arguments a pointer to the string
 containing the expression, and a pointer to the
 variable in which the result should be placed.
 If an error is found in the expression, a message
 is returned, but if the expression is acceptable
 then NULL is returned.

 (c) 1993 by Graham Wheeler

*****************************************************/

#include <stdio.h> /* For NULL */
#include <ctype.h>

#ifdef __MSDOS__
#	ifndef __STDC__
#		define __STDC__	1
#	endif
#endif

/*
 * Lexical tokens
 */

#define NUM_TOK		1
#define OR_TOK		2
#define AND_TOK		3
#define EQU_TOK		4
#define NEQ_TOK		5
#define ADD_TOK		6
#define SUB_TOK		7
#define MUL_TOK		8
#define DIV_TOK		9
#define MOD_TOK		10
#define LFT_TOK		11
#define RGT_TOK		12
#define GEQ_TOK		13
#define GTR_TOK		14
#define LEQ_TOK		15
#define LES_TOK		16

/*
 * Local Variables
 */

static char	*ebuf,	/* Address of expression string	*/
		*msg,	/* Return message		*/
		*fail = "Syntax error in expression"; /* Failure message */

static int	epos,	/* Current position in ebuf buffer	*/
		tok,	/* Current lexical token		*/
		tokval;	/* Current token value, if a number	*/

/*
 * Forward declarations
 */

#if __STDC__
static int _cond_expr(void);
#endif

/*
 * Lexical analyser
 */

#if __STDC__
static void nexttok(void) {
#else
static void nexttok() {
#endif
	while (isspace(ebuf[epos])) epos++;
	if (ebuf[epos]) {
		if (isdigit(ebuf[epos])) { /* Integer? */
			tok = NUM_TOK;
			tokval = 0;
			while (isdigit(ebuf[epos])) {
				tokval = tokval * 10 + ebuf[epos++] - '0';
			}
		} else { /* Operator */
			switch(ebuf[epos]) {
			case '|': tok = OR_TOK; break;
			case '&': tok = AND_TOK; break;
			case '=': tok = EQU_TOK; break;
			case '#': tok = NEQ_TOK; break;
			case '+': tok = ADD_TOK; break;
			case '-': tok = SUB_TOK; break;
			case '*': tok = MUL_TOK; break;
			case '/': tok = DIV_TOK; break;
			case '%': tok = MOD_TOK; break;
			case '(': tok = LFT_TOK; break;
			case ')': tok = RGT_TOK; break;
			case '>': if (ebuf[epos+1]=='=') {
					epos++;
					tok = GEQ_TOK;
				} else tok = GTR_TOK;
				break;
			case '<': if (ebuf[epos+1]=='=') {
					epos++;
					tok = LEQ_TOK;
				} else tok = LES_TOK;
				break;
			}
			epos++;
		}
	} else tok=-1;
}

/*
 * PARSER
 */

/*
 * A primitive expression can be either a
 * conditional expression in parentheses,
 * or an integer.
 */

#if __STDC__
static int _prim_expr(void) {
#else
static int _prim_expr() {
#endif
	int sgn = 1, v;
	if (tok==LFT_TOK) {
		nexttok();
		v = _cond_expr();
		if (tok!=RGT_TOK) {
			msg = fail;
			return 0;
		}
	} else {
		if (tok==SUB_TOK) { sgn = -1; nexttok(); }
		if (tok!=NUM_TOK) {
			msg = fail;
			return 0;
		}
		v = sgn * tokval;
	}
	nexttok();
	return v;
}

/*
 * Multiplicative expression
 */

#if __STDC__
static int _mul_expr(void) {
#else
static int _mul_expr() {
#endif
	int v = _prim_expr();
	if (tok==MUL_TOK) {
		nexttok();
		return (v * _mul_expr());
	} else if (tok==DIV_TOK) {
		nexttok();
		return (v / _mul_expr());
	} else if (tok==MOD_TOK) {
		nexttok();
		return (v % _mul_expr());
	} else return v;
}

/*
 * Additive expression
 */

#if __STDC__
static int _add_expr(void) {
#else
static int _add_expr() {
#endif
	int v = _mul_expr();
	if (tok==ADD_TOK) {
		nexttok();
		return (v + _add_expr());
	} else if (tok==SUB_TOK) {
		nexttok();
		return (v - _add_expr());
	} else return v;
}

/*
 * Relational expression
 */

#if __STDC__
static int _rel_expr(void) {
#else
static int _rel_expr() {
#endif
	int v = _add_expr();
	if (tok==GTR_TOK) {
		nexttok();
		return (v > _rel_expr());
	} else if (tok==GEQ_TOK) {
		nexttok();
		return (v >= _rel_expr());
	} else if (tok==LEQ_TOK) {
		nexttok();
		return (v <= _rel_expr());
	} else if (tok==LES_TOK) {
		nexttok();
		return (v < _rel_expr());
	} else return v;
}

/*
 * Equality expression
 */

#if __STDC__
static int _equ_expr(void) {
#else
static int _equ_expr() {
#endif
	int v = _rel_expr();
	if (tok==EQU_TOK) {
		nexttok();
		return (v == _equ_expr());
	} else if (tok==NEQ_TOK) {
		nexttok();
		return (v != _equ_expr());
	} else return v;
}

/*
 * AND conditional expressions
 */

#if __STDC__
static int _cond_expr2(void) {
#else
static int _cond_expr2() {
#endif
	int v = _equ_expr();
	if (tok==AND_TOK) {
		nexttok();
		return (v && _cond_expr2());
	} else return v;
}

/*
 * OR conditional expressions
 */

#if __STDC__
static int _cond_expr(void) {
#else
static int _cond_expr() {
#endif
	int v = _cond_expr2();
	if (tok==OR_TOK) {
		nexttok();
		return (v || _cond_expr());
	} else return v;
}

/*
 * Main entry point
 */

#if __STDC__
char *compute(char *str, int *rtn) {
#else
char *compute(str, rtn)
	char *str;
	int *rtn;
{
#endif
	msg = NULL;
	ebuf = str; epos = 0;
	nexttok();
	*rtn = _cond_expr();
	if (tok!=-1) msg = fail;
	return msg;
}


